#! /usr/bin/python3
import os
import sys
import generator_utils as gen

template = """// @{generatedby}@
/* ///////////////////////// The MPI Bugs Initiative ////////////////////////

  Origin: MBI

  Description: @{shortdesc}@
    @{longdesc}@

  Version of MPI: Conforms to MPI 1.1, does not require MPI 2 implementation

BEGIN_MPI_FEATURES
  P2P!basic: @{p2pfeature}@
  P2P!nonblocking: @{ip2pfeature}@
  P2P!persistent: Lacking
  COLL!basic: Lacking
  COLL!nonblocking: Lacking
  COLL!persistent: Lacking
  COLL!tools: Lacking
  RMA: Lacking
END_MPI_FEATURES

BEGIN_MBI_TESTS
  $ mpirun -np 4 ${EXE}
  | @{outcome}@
  | @{errormsg}@
END_MBI_TESTS
//////////////////////       End of MBI headers        /////////////////// */

#include <mpi.h>
#include <stdio.h>
#include <stdlib.h>

#define N 2

int main(int argc, char **argv) {
  int nprocs = -1;
  int rank = -1;
  int dest, src;
  int i=0;
  int root = 0;
  int stag = 0, rtag = 0;
  int buff_size = 1;

  MPI_Init(&argc, &argv);
  MPI_Comm_size(MPI_COMM_WORLD, &nprocs);
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  printf("Hello from rank %d \\n", rank);

  if (nprocs != 4)
    printf("MBI ERROR: This test needs 4 processes to produce a bug!\\n");

  int dbs = sizeof(int)*nprocs; /* Size of the dynamic buffers for alltoall and friends */
  MPI_Comm newcom = MPI_COMM_WORLD;
  MPI_Datatype type = MPI_INT;
  MPI_Op op = MPI_SUM;

  @{init0a}@
  @{init0b}@
  @{init0c}@
  @{init1a}@
  @{init1b}@
  @{init3a}@
  @{init3b}@
  @{init3c}@
  @{init3d}@

  if (rank == 0) {
    src = MPI_ANY_SOURCE; rtag = @{tag}@;
    for (int i = 0; i < 2 * N; i++) {
      @{operation0a}@ /* MBIERROR1 */
      @{fini0a}@
    }
    src = 3; rtag = 0;
    @{operation0b}@ /* MBIERROR2 */
    @{fini0b}@
    @{operation0c}@
    @{fini0c}@
  } else if (rank == 1 || rank == 2) {
    dest = 0; stag = @{tag}@;
    for (int i = 0; i < N; i++) {
      @{operation1a}@
      @{fini1a}@
    }
    dest = 3; stag = 0;
    @{operation1b}@
    @{fini1b}@
  } else if (rank == 3) {
    dest = 0; src = 1; rtag= 0; stag = 0;
    @{operation3a}@
    @{fini3a}@
    @{operation3b}@ /* MBIERROR3 */
    @{fini3b}@
    src = 2;
    @{operation3c}@
    @{fini3c}@
    @{operation3d}@
    @{fini3d}@
  }

  @{free0a}@
  @{free0b}@
  @{free0c}@
  @{free1a}@
  @{free1b}@
  @{free3a}@
  @{free3b}@
  @{free3c}@
  @{free3d}@

  MPI_Finalize();
  printf("Rank %d finished normally\\n", rank);
  return 0;
}
"""

basedesc = 'We have 4 processes (p0, p1, p2 and p3). p1 and p2 send N messages to p0 and send a last message to p3. Process p0 recv 2*N messages from p1 and p2 using MPI_ANY_SOURCE and wait messages from p3. p3 wait a message from p1 and send message to p0, before doing the same for p2.'

for s in gen.send + gen.isend:
    for r in gen.recv + gen.irecv:
        patterns = {}
        patterns = {'s': s, 'r': r}
        patterns['generatedby'] = f'DO NOT EDIT: this file was generated by {os.path.basename(sys.argv[0])}. DO NOT EDIT.'
        patterns['p2pfeature'] = 'Yes' if s in gen.send or r in gen.recv else 'Lacking'
        patterns['ip2pfeature'] = 'Yes' if s in gen.isend or r in gen.irecv else 'Lacking'
        patterns['s'] = s
        patterns['r'] = r

        patterns['init0a'] = gen.init[r]("0a")
        patterns['init0b'] = gen.init[r]("0b")
        patterns['init0c'] = gen.init[r]("0c")
        patterns['operation0a'] = gen.operation[r]("0a")
        patterns['operation0b'] = gen.operation[r]("0b")
        patterns['operation0c'] = gen.operation[r]("0c")
        patterns['fini0a'] = gen.fini[r]("0a")
        patterns['fini0b'] = gen.fini[r]("0b")
        patterns['fini0c'] = gen.fini[r]("0c")
        patterns['free0a'] = gen.free[r]("0a")
        patterns['free0b'] = gen.free[r]("0b")
        patterns['free0c'] = gen.free[r]("0c")

        patterns['init1a'] = gen.init[s]("1a")
        patterns['init1b'] = gen.init[s]("1b")
        patterns['operation1a'] = gen.operation[s]("1a")
        patterns['operation1b'] = gen.operation[s]("1b")
        patterns['fini1a'] = gen.fini[s]("1a")
        patterns['fini1b'] = gen.fini[s]("1b")
        patterns['free1a'] = gen.free[s]("1a")
        patterns['free1b'] = gen.free[s]("1b")

        patterns['init3a'] = gen.init[r]("3a")
        patterns['init3b'] = gen.init[s]("3b")
        patterns['init3c'] = gen.init[r]("3c")
        patterns['init3d'] = gen.init[s]("3d")
        patterns['operation3a'] = gen.operation[r]("3a")
        patterns['operation3b'] = gen.operation[s]("3b")
        patterns['operation3c'] = gen.operation[r]("3c")
        patterns['operation3d'] = gen.operation[s]("3d")
        patterns['fini3a'] = gen.fini[r]("3a")
        patterns['fini3b'] = gen.fini[s]("3b")
        patterns['fini3c'] = gen.fini[r]("3c")
        patterns['fini3d'] = gen.fini[s]("3d")
        patterns['free3a'] = gen.free[r]("3a")
        patterns['free3b'] = gen.free[s]("3b")
        patterns['free3c'] = gen.free[r]("3c")
        patterns['free3d'] = gen.free[s]("3d")

        patterns['tag'] = '1'

        # Generate the correct matching because of the conditional
        replace = patterns.copy()
        replace['shortdesc'] = 'Message race'
        replace['longdesc'] = basedesc + ' In this file, different message tag are used to avoid involuntary message race.'
        replace['outcome'] = 'OK'
        replace['errormsg'] = 'OK'
        gen.make_file(template, f'MessageRace_Loop_{s}_{r}_ok.c', replace)

        # Generate the incorrect matching because of the conditional
        replace = patterns.copy()
        replace['shortdesc'] = 'Message race'
        replace['longdesc'] = basedesc + ' If the loop of p1 ending before the loop of p2, p0 could recv a message from process p3 into recv loop destined to message from p1 and p2. In this case, the program deadlock cause by message race at recv at line @{line:MBIERROR2}@.'
        replace['outcome'] = 'ERROR: MessageRace'
        replace['errormsg'] = 'Message race. The use of wildcard receive calls @{r}@ at @{filename}@:@{line:MBIERROR1}@ from @{r}@ at @{filename}@:@{line:MBIERROR3}@ and @{r}@ without wildcard at @{filename}@:@{line:MBIERROR2}@) leads to nondeterministic matching.'
        replace['tag'] = '0'
        gen.make_file(template, f'MessageRace_Loop_{s}_{r}_nok.c', replace)
