/*
 * xfce4-hdaps, an XFCE4 panel plugin for the HDAPS system.
 *
 * Copyright Michael Orlitzky
 *
 * http://michael.orlitzky.com/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * http://www.fsf.org/licensing/licenses/gpl.html
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <gtk/gtk.h>
#include <libxfce4ui/libxfce4ui.h>
#include <libxfce4panel/xfce-panel-plugin.h>
#include "xfce4-hdaps-dialogs.h"
#include "hdaps.h"

/* Wild guess. Actually borrowed from some code
   I copy-and-pasted. Doesn't seem too ugly. */
#define DEFAULT_BORDER_WIDTH 8

/* If people can't read the README, well maybe
 *  they can do it /online/.
 */
#define PLUGIN_WEBSITE "http://michael.orlitzky.com/code/xfce4-hdaps.php"


static void hdaps_configure_response(GtkWidget   *dialog,
				     gint         response,
				     HdapsPlugin *hdaps) {

  if (response == GTK_RESPONSE_HELP) {
    /* Launch the user's web browser and direct them to
       the plugin's webpage. */
    gboolean spawn_result = g_spawn_command_line_async("exo-open --launch WebBrowser " PLUGIN_WEBSITE, NULL);
    
    if (G_UNLIKELY(spawn_result == FALSE)) {
      g_warning(_("Unable to open the following url: %s"), PLUGIN_WEBSITE);
    }
    
    return;
  }

  
  if (response == GTK_RESPONSE_OK) {
    /* This corresponds to the "Save" button, so we
       want to save any settings that may have changed. */
    g_object_set_data(G_OBJECT(hdaps->plugin), "dialog", NULL);
    hdaps->poll_frequency = gtk_spin_button_get_value_as_int((GtkSpinButton*)hdaps->sb_poll_frequency);
    hdaps->device_name = gtk_combo_box_get_active_text((GtkComboBox*)hdaps->cb_device_name);
    snprintf(hdaps->sysfs_file, FILENAME_MAX, UNLOAD_HEADS_FMT, hdaps->device_name);
    hdaps_save(hdaps->plugin, hdaps);
    hdaps_reset_timeout(hdaps);
  }

  
  /* Since there is already a "save" button, we should ignore
     any changes that were made if the user presses "cancel"
     instead. */
  g_object_set_data(G_OBJECT(hdaps->plugin), "dialog", NULL);
  xfce_panel_plugin_unblock_menu(hdaps->plugin);
  gtk_widget_destroy(dialog);

  return;
}



void hdaps_configure(XfcePanelPlugin *plugin,
		     HdapsPlugin     *hdaps) {
  
  /* Here comes a bunch of GTK garbage to create the
     settings dialog. */
  
  GtkWidget *dialog;
  GtkWidget *label;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkSizeGroup *sg;
  
  /* Block the plugin menu while the configuration
     dialogue is open. Don't forget to unblock it before
     we close. */
  xfce_panel_plugin_block_menu(plugin);

  /* Create the dialog */
  dialog = xfce_titled_dialog_new_with_buttons(_("Hdaps Plugin"),
					       GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (plugin))),
					       GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
					       GTK_STOCK_HELP, GTK_RESPONSE_HELP,
					       GTK_STOCK_SAVE, GTK_RESPONSE_OK,
					       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					       NULL);

  /* Center the dialog on screen. */
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

  /* Set dialog icon to a generic settings icon. */
  gtk_window_set_icon_name(GTK_WINDOW(dialog), "xfce4-settings");

  /* Link the dialog to the plugin, so we can destroy it in case
   * the plugin is closed while the dialog is still open. */
  g_object_set_data(G_OBJECT(plugin), "dialog", dialog);

  /* Connect the reponse signal to the dialog. */
  g_signal_connect(G_OBJECT (dialog), "response",
		   G_CALLBACK(hdaps_configure_response), hdaps);

  
  vbox = gtk_vbox_new(FALSE, DEFAULT_BORDER_WIDTH);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), DEFAULT_BORDER_WIDTH - 2);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), vbox, TRUE, TRUE, 0);
  
  /* Create size group to keep widgets aligned */
  sg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

  hbox = gtk_hbox_new(FALSE, DEFAULT_BORDER_WIDTH);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new(_("Poll Frequency:"));
  gtk_size_group_add_widget(sg, label);
  gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  hdaps->sb_poll_frequency = gtk_spin_button_new_with_range(100, 5000, 100);
  gtk_spin_button_set_value((GtkSpinButton*)hdaps->sb_poll_frequency, hdaps->poll_frequency);
  
  gtk_box_pack_start(GTK_BOX(hbox),
		     hdaps->sb_poll_frequency,
		     FALSE,
		     FALSE,
		     0);

  /* Create the device name dropdown. */
  hbox = gtk_hbox_new(FALSE, DEFAULT_BORDER_WIDTH);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new(_("Device Name:"));
  gtk_size_group_add_widget(sg, label);
  gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  hdaps->cb_device_name = gtk_combo_box_new_text();

  /* Add the current device name, it should always be available.
     Oh, and it should be the default, too. */
  gtk_combo_box_append_text((GtkComboBox*)hdaps->cb_device_name, hdaps->device_name);
  
  /* This function takes an index, and 0 should be the first
     (only) index at this point. */
  gtk_combo_box_set_active((GtkComboBox*)hdaps->cb_device_name, 0);
  
  /* Now loop through the list of available devices, adding each
     to the list as we go. */
  char hdaps_devices[MAX_HDAPS_DEVICES][FILENAME_MAX];
  int found_devices = get_hdaps_device_list(hdaps_devices);
  int list_idx = 0;
  
  for (list_idx = 0; list_idx < found_devices; list_idx++) {
    /* We don't want to add duplicate entries to the combo box.
       However, at this point, the current device name should be
       the only entry. Therefore, to avoid duplicates, we only
       have to avoid adding the current device name a second time. */
    if (strcmp(hdaps_devices[list_idx], hdaps->device_name) != 0) {
      gtk_combo_box_append_text((GtkComboBox*)hdaps->cb_device_name, hdaps_devices[list_idx]);
    }
  }
  
  gtk_box_pack_start(GTK_BOX(hbox), hdaps->cb_device_name, FALSE, FALSE, 0);
  
  /* Show the dialog and all of its widgets. */
  gtk_widget_show_all(dialog);
}




/* URL handler used on the About dialog. */
static void hdaps_url_handler(GtkAboutDialog *about,
			      const gchar *link,
			      gpointer data) {
    gchar *cmd;

    cmd = g_strdup_printf("%s %s","xdg-open", link);

    /* Stolen from xfce4-power-manager. */
    if (!g_spawn_command_line_async(cmd, NULL)) {
      g_free(cmd);
      cmd = g_strdup_printf("%s %s","xfbrowser4", link);
      g_spawn_command_line_async(cmd, NULL);
    }
    g_free(cmd);
}


/* Email address handler used on the About dialog. */
static void hdaps_mailto_handler(GtkAboutDialog *about,
				 const gchar *link,
				 gpointer data) {
    gchar *cmd = g_strdup_printf( "%s %s", "xdg-email", link);

    g_spawn_command_line_async(cmd, NULL);
    g_free(cmd);
}


void hdaps_about(XfcePanelPlugin *plugin) {
  const gchar *authors[] = { "Michael Orlitzky <michael@orlitzky.com>", NULL };
  const gchar *copyright = "Copyright \302\251 2012 Michael Orlitzky";

  gtk_about_dialog_set_url_hook(hdaps_url_handler, NULL, NULL);
  gtk_about_dialog_set_email_hook(hdaps_mailto_handler, NULL, NULL);

  gtk_show_about_dialog(NULL,
			"authors", authors,
			"copyright", copyright,
			"destroy-with-parent", TRUE,
			"license", LICENSE_GPL3,
			"logo-icon-name", PACKAGE_NAME,
			"icon-name", PACKAGE_NAME,
			"program-name", PACKAGE_NAME,
			"version", PACKAGE_VERSION,
			"website", PLUGIN_WEBSITE,
			"website-label", "xfce4-hdaps homepage",
			NULL);
}
