// SPDX-License-Identifier: GPL-2.0
/*
 * CZ.NIC's Turris Omnia MCU TRNG driver
 *
 * 2024 by Marek Behún <kabel@kernel.org>
 */

#include <linux/bitfield.h>
#include <linux/completion.h>
#include <linux/container_of.h>
#include <linux/errno.h>
#include <linux/gpio/consumer.h>
#include <linux/gpio/driver.h>
#include <linux/hw_random.h>
#include <linux/i2c.h>
#include <linux/interrupt.h>
#include <linux/minmax.h>
#include <linux/string.h>
#include <linux/types.h>

#include <linux/turris-omnia-mcu-interface.h>
#include "turris-omnia-mcu.h"

#define OMNIA_CMD_TRNG_MAX_ENTROPY_LEN	64

static irqreturn_t omnia_trng_irq_handler(int irq, void *dev_id)
{
	struct omnia_mcu *mcu = dev_id;

	complete(&mcu->trng_entropy_ready);

	return IRQ_HANDLED;
}

static int omnia_trng_read(struct hwrng *rng, void *data, size_t max, bool wait)
{
	struct omnia_mcu *mcu = container_of(rng, struct omnia_mcu, trng);
	u8 reply[1 + OMNIA_CMD_TRNG_MAX_ENTROPY_LEN];
	int err, bytes;

	if (!wait && !completion_done(&mcu->trng_entropy_ready))
		return 0;

	do {
		if (wait_for_completion_interruptible(&mcu->trng_entropy_ready))
			return -ERESTARTSYS;

		err = omnia_cmd_read(mcu->client,
				     OMNIA_CMD_TRNG_COLLECT_ENTROPY,
				     reply, sizeof(reply));
		if (err)
			return err;

		bytes = min3(reply[0], max, OMNIA_CMD_TRNG_MAX_ENTROPY_LEN);
	} while (wait && !bytes);

	memcpy(data, &reply[1], bytes);

	return bytes;
}

int omnia_mcu_register_trng(struct omnia_mcu *mcu)
{
	struct device *dev = &mcu->client->dev;
	u8 irq_idx, dummy;
	int irq, err;

	if (!(mcu->features & OMNIA_FEAT_TRNG))
		return 0;

	irq_idx = omnia_int_to_gpio_idx[__bf_shf(OMNIA_INT_TRNG)];
	irq = gpiod_to_irq(gpio_device_get_desc(mcu->gc.gpiodev, irq_idx));
	if (!irq)
		return dev_err_probe(dev, -ENXIO, "Cannot get TRNG IRQ\n");

	/*
	 * If someone else cleared the TRNG interrupt but did not read the
	 * entropy, a new interrupt won't be generated, and entropy collection
	 * will be stuck. Ensure an interrupt will be generated by executing
	 * the collect entropy command (and discarding the result).
	 */
	err = omnia_cmd_read(mcu->client, OMNIA_CMD_TRNG_COLLECT_ENTROPY,
			     &dummy, 1);
	if (err)
		return err;

	init_completion(&mcu->trng_entropy_ready);

	err = devm_request_threaded_irq(dev, irq, NULL, omnia_trng_irq_handler,
					IRQF_ONESHOT, "turris-omnia-mcu-trng",
					mcu);
	if (err)
		return dev_err_probe(dev, err, "Cannot request TRNG IRQ\n");

	mcu->trng.name = "turris-omnia-mcu-trng";
	mcu->trng.read = omnia_trng_read;

	err = devm_hwrng_register(dev, &mcu->trng);
	if (err)
		return dev_err_probe(dev, err, "Cannot register TRNG\n");

	return 0;
}
