/* This file is part of the GNU plotutils package.  Copyright (C) 1995,
   1996, 1997, 1998, 1999, 2000, 2005, 2008, Free Software Foundation, Inc.

   The GNU plotutils package is free software.  You may redistribute it
   and/or modify it under the terms of the GNU General Public License as
   published by the Free Software foundation; either version 2, or (at your
   option) any later version.

   The GNU plotutils package is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with the GNU plotutils package; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin St., Fifth Floor,
   Boston, MA 02110-1301, USA. */

/* This file defines the arrays _pl_g_occidental_hershey_glyphs[] and
   _pl_g_oriental_hershey_glyphs[].  The first array includes the standard
   (`occidental') Hershey glyphs digitized by Dr. Allen V. Hershey, some
   glyphs due to other people, and his Japanese Hiragana and Katakana
   glyphs.  The second array includes his digitizations of Japanese
   ideographic symbols (Kanji glyphs).  It originally included the Hiragana
   and Katakana glyphs as well, but they were moved to the first array
   because some people may not wish to include Kanji support.

   Dr. Hershey digitized the glyphs c. 1967, at what is now the U.S. Naval
   Surface Weapons Center in Dahlgren, Virginia.  For many years he
   distributed copies of the glyphs, and his typographic software, on
   magnetic tape.  Over 120 copies of the tape were distributed.  There
   have been many other distributions of the glyphs.  In the 1970's they
   were incorporated, not always with attribution, in several commercial
   plotting packages.  They were first freely distributed in 1985(?), by
   being posted to Usenet (to vol. 4 of mod.sources) by Pete Holzmann
   <pete@xc.org>, then at Octopus Enterprises.  In the 1980's the glyphs
   were incorporated in at least two freeware plotting programs, Nelson
   Beebe's PLOT79 and Tim Pearson's PGPLOT.  The latter is still available
   (see http://www.astro.caltech.edu/~tjp/pgplot/ ).

   The standard hardcopy reference for the occidental Hershey glyphs, which
   tabulates and displays them, is:

   @TechReport{Wolcott76,
     author =       {Norman M. Wolcott and Joseph Hilsenrath},
     title =        {A Contribution to Computer Typesetting Techniques: 
                     Tables of Coordinates for {Hershey's} Repertory of 
                     Occidental Type Fonts and Graphic Symbols},
     institution =  {U.S. National Bureau of Standards},
     year =         {1976},
     month =        {April},
     type =         {Special Publication},
     number =       {424},
     OPTnote =      {US NTIS stock number PB251845}}
     
   This publication is available from the NTIS (US National Technical
   Information Service, +1 703 487 4650).  The NTIS stock number is given
   above.  See also Dr. Hershey's original TR, which describes the
   alphabets from which the glyphs were taken and gives information on the
   oriental glyphs:

   @TechReport{Hershey67,
     author = 	 {Allen V. Hershey},
     title = 	 {Calligraphy for Computers},
     institution =  {U.S. Naval Weapons Laboratory},
     address =	 {Dahlgren, VA},
     year = 	 {1967},
     type =	 {Report},
     number =	 {TR--2101},
     month =	 {Aug},
     OPTnote =	 {US NTIS stock number AD662398}}

   Additional references are given in the file ./doc/hershey.bib.

   The _pl_g_occidental_hershey_glyphs[] array below was constructed from
   releases of the Hershey glyphs that are later than the one described in
   the 1976 NBS publication above.  For example, Fig. 15 of that work
   portrays 43 additional glyphs digitized by Norman Wolcott of the NBS;
   they are included here.  See the file ./doc/hershey-number for details
   on how the array was assembled.  The original Hershey glyphs, including
   the 43 Wolcott glyphs, are stored in the 0..3999 slots of the array.

   The _pl_g_occidental_hershey_glyphs[] array also includes non-Hershey
   glyphs taken from various sources.  All post-Hershey glyphs have been
   placed in slots 4000..4194.  This includes a few glyphs taken from the
   UGS (the Unified Graphics System, developed by Bob Beach at SLAC; see
   Computer Graphics, Fall 1974, pp. 22-23).  As well, it includes some
   freeware glyphs developed by Thomas Wolff <wolff@inf.fu-berlin.de> and
   distributed as part of the Ghostscript distribution.  The Beach and
   Wolff glyphs are used in the Hershey Symbol fonts.  Locally developed
   glyphs (accented characters, in particular) have been added too.

   The Japanese Hiragana and Katakana glyphs digitized by Dr. Hershey,
   formerly located among the `oriental' glyphs, are now located in slots
   4195..4399 of the _pl_g_occidental_hershey_glyphs[] array.

   The format of the glyphs in the _pl_g_occidental_hershey_glyphs[] and
   _pl_g_oriental_hershey_glyphs[] arrays is the format in which the
   Hershey glyphs were distributed to mod.sources in 1985(?) by Pete
   Holzmann, and is due to Jim Hurt, then at Cognition Inc.  Glyphs are
   encoded as strings consisting of pairs of printable ascii characters in
   the range 0x20 to 0x7e, i.e. SP (ascii space) through ~ (ascii tilde).
   Each character in the range [0x20,0x7e] encodes an integer.  The integer
   may be obtained from the character by subtracting 'R', i.e, 0x52.
   Integers in the range [-0x32,0x2c], i.e., [-50,44], may be represented
   in this way.  Actually, all integers appearing in the occidental Hershey
   glyphs are restricted to the range [-41,41], and those in the oriental
   glyphs are restricted to the range [-12,13].

   (Note: the non-Hershey pointing hand glyphs in slots 4040..4043 of the
   _pl_g_occidental_hershey_glyphs[] array, which are due to Nelson Beebe,
   include integers drawn from the range [-50,50].  So the four
   corresponding strings include characters drawn from outside the
   printable ASCII range, e.g. \204 (i.e. 0x84), which represents 50.)

   The first two characters in each string encode the x range
   (xinit,xfinal) of a glyph.  (The `width' of the glyph, as distinct from
   the x extent of its bounding box, is xfinal - xinit.)  Subsequent pairs
   encode (x1,y1),(x2,y2),(x3,y3),..., i.e., the coordinates of points.
   The glyph is drawn by moving to (x1,y1), stroking a vector to (x2,y2),
   stroking a vector to (x3,y3), etc.  The pair (-50,0), which is encoded
   as the pair of characters " R", if encountered in the string is
   interpreted as a `pen up' command: the vector immediately following
   should be drawn with pen up, i.e., should be interpreted as a motion
   command rather than a drawing command.

   (The pair (-50,0) could conceivably appear in the abovementioned
   pointing hand glyphs as the coordinates of a vector endpoint, but
   fortunately it does not.  So Jim Hurt's encoding scheme works, even for
   the four pointing hands.)  */

#include "sys-defines.h"
#include "extern.h"

#define NUM_OCCIDENTAL_HERSHEY_GLYPHS 4400 /*should agree with val in g_cntrlify.c */
#define NUM_ORIENTAL_HERSHEY_GLYPHS 5500 /* should agree with value in g_cntrlify.c */

const char * const _pl_g_occidental_hershey_glyphs[NUM_OCCIDENTAL_HERSHEY_GLYPHS] =
{
  /******** Hershey Glyphs 0 to 99 ********/   
  "",   "MWRMNV RRMVV RPSTS",   "MWOMOV ROMSMUNUPSQ ROQSQURUUSVOV",   "MXVNTMRMPNOPOSPURVTVVU",   "MWOMOV ROMRMTNUPUSTURVOV",   "MWOMOV ROMUM ROQSQ ROVUV",   "MVOMOV ROMUM ROQSQ",   "MXVNTMRMPNOPOSPURVTVVUVR RSRVR",   "MWOMOV RUMUV ROQUQ",   "PTRMRV",   "NUSMSTRVPVOTOS",   "MWOMOV RUMOS RQQUV",   "MVOMOV ROVUV",   "LXNMNV RNMRV RVMRV RVMVV",   "MWOMOV ROMUV RUMUV",   "MXRMPNOPOSPURVSVUUVSVPUNSMRM",   "MWOMOV ROMSMUNUQSROR",   "MXRMPNOPOSPURVSVUUVSVPUNSMRM RSTVW",   "MWOMOV ROMSMUNUQSROR RRRUV",   "MWUNSMQMONOOPPTRUSUUSVQVOU",   "MWRMRV RNMVM",   "MXOMOSPURVSVUUVSVM",   "MWNMRV RVMRV",   "LXNMPV RRMPV RRMTV RVMTV",   "MWOMUV RUMOV",   "MWNMRQRV RVMRQ",   "MWUMOV ROMUM ROVUV",   "MWRMNV RRMVV RPSTS",   "MWOMOV ROMSMUNUPSQ ROQSQURUUSVOV",   "MVOMOV ROMUM",   "MWRMNV RRMVV RNVVV",   "MWOMOV ROMUM ROQSQ ROVUV",   "MWUMOV ROMUM ROVUV",   "MWOMOV RUMUV ROQUQ",   "MXRMPNOPOSPURVSVUUVSVPUNSMRM RQQTR RTQQR",   "PTRMRV",   "MWOMOV RUMOS RQQUV",   "MWRMNV RRMVV",   "LXNMNV RNMRV RVMRV RVMVV",   "MWOMOV ROMUV RUMUV",   "MWOMUM RPQTR RTQPR ROVUV",   "MXRMPNOPOSPURVSVUUVSVPUNSMRM",   "MWOMOV RUMUV ROMUM",   "MWOMOV ROMSMUNUQSROR",   "MWOMRQOV ROMUM ROVUV",   "MWRMRV RNMVM",   "MWNONNOMPMQNRPRV RVOVNUMTMSNRP",   "LXRMRV RPONPNSPTTTVSVPTOPO",   "MWOMUV RUMOV",   "LXRMRV RNOOPOSQTSTUSUPVO",   "MXOVQVOROPPNRMSMUNVPVRTVVV",   "MWSMMV RSMUV ROSTS",   "MWQMNV RQMTMVNVPSQPQ RSQURUTTURVNV",   "LXVPUNTMRMPNOONQNSOUPVRVTUUT",   "MXQMNV RQMUMVOVQUTTURVNV",   "MVQMNV RQMVM RPQSQ RNVSV",   "MVQMNV RQMVM RPQSQ",   "LXVPUNTMRMPNOONQNSOUPVRVTUUSRS",   "MXQMNV RWMTV RPQUQ",   "PUTMQV",   "OVUMSSRUQVPVOUOT",   "MVQMNV RVMOS RRQTV",   "NVRMOV ROVTV",   "LYPMMV RPMQV RXMQV RXMUV",   "MXQMNV RQMTV RWMTV",   "LXRMPNOONQNSOUPVRVTUUTVRVPUNTMRM",   "MWQMNV RQMUMVNVPUQSRPR",   "LXRMPNOONQNSOUPVRVTUUTVRVPUNTMRM RQVPUPTQSRSSTTVUWVW",   "MWQMNV RQMUMVNVPUQSRPR RQRRUSVTVUU",   "MWVNTMRMPNPPQQTRUSUUSVPVNU",   "MVSMPV RPMVM",   "LXPMNSNUOVRVTUUSWM",   "MWOMQV RWMQV",   "KXNMNV RSMNV RSMSV RXMSV",   "NWQMTV RWMNV",   "NWQMSQQV RWMSQ",   "MWQMWMNVTV",   "",   "",   "",   "",   "",   "",   "LXNMRV RVMRV RNMVM",   "MWNLVX",   "LXRONU RROVU",   "",   "",   "",   "",   "",   "",   "",   "",   "",   "",   "",   "",   "",   "",   
  /******** Hershey Glyphs 100 to 199 ********/
  "",   "NWQPTPUQUV RURQSPTPUQVSVUU",   "MWOMOV ROSPURVTUUSTQRPPQOS",   "MWUQSPRPPQOSPURVSVUU",   "MWUMUV RUSTQRPPQOSPURVTUUS",   "MWOSUSTQRPPQOSPURVTV",   "NVUNTMSMRNRV RPPTP",   "MWUPUVTXRYPY RUSTQRPPQOSPURVTUUS",   "MWOMOV ROSPQRPTQUSUV",   "PTRLQMRNSMRL RRPRV",   "PUSLRMSNTMSL RSPSXRYQYPX",   "NWPMPV RUPPT RRSUV",   "PTRMRV",   "KYMPMV RMSNQOPPPQQRSRV RRSSQTPUPVQWSWV",   "MWOPOV ROSPQRPTQUSUV",   "MWRPPQOSPURVTUUSTQRP",   "MWOPOY ROSPURVTUUSTQRPPQOS",   "MWUPUY RUSTQRPPQOSPURVTUUS",   "NVPPPV RPSQQSPTP",   "NWUQTPQPPQPRQSTSUTUUTVQVPU",   "NVRMRUSVTVUU RPPTP",   "MWUPUV ROPOSPURVTUUS",   "NVOPRV RUPRV",   "LXNPPV RRPPV RRPTV RVPTV",   "MWOPUV RUPOV",   "MWOPRV RUPRVQXPYOY",   "MWOPUPOVUV",   "MXVPUSTURVPUOSPQRPTQUUVV",   "MWOTQVSVTUTSSRPQRQTPUOUNTMRMQNPPOTNY",   "MXNQOPQPRQRSQW RVPURSTQWPY",   "MWTNSMRMQNQORPTQUSTURVPUOSPQRP",   "NWUQSPQPPQPRQS RSSQSPTPUQVSVUU",   "NWTMSNSOTP RUPSPQQPSPUQVSWSXRYQY",   "LXNQOPPPQQQSPV RQSRQTPUPVQVSUVTY",   "LXNQOPPPQQQURVSVTUUSVPVNUMTMSNSPTRUSWT",   "OVRPQSQURVSVTU",   "MWQPOV RUPTPRQPS RPSQUSVTV",   "MWOMPMQNRPUV RRPOV",   "LYPPMY RUPTSSUQVPVOUOS RTSTUUVVVWU",   "MWNPOPOV RUPTSRUOV",   "NWTMSNSOTP RUPSPQQQRRSTS RSSQTPUPVQWSXSYRZQZ",   "MWRPPQOSPURVTUUSTQRP",   "MXOQQPVP RQPQRPV RTPTRUV",   "MWOSPURVTUUSTQRPPQOSNY",   "MXVPRPPQOSPURVTUUSTQRP",   "MXOQQPVP RSPRV",   "KXMQNPOPPQPUQVSVTUUSVP",   "MXPPOQOSPURVSVUUVSVQUPTPSQRSQY",   "MWOPPPQQSXTYUY RUPTRPWOY",   "KYTMRY RMQNPOPPQPUQVTVUUVSWP",   "LXOPNRNTOVQVRTRR RUPVRVTUVSVRT",   "LWTSSQQPOQNSOUQVSUTS RUPTSTUUVVV",   "MWQMOSPURVTUUSTQRPPQOS",   "MWUQSPRPPQOSPURVTV",   "LWTSSQQPOQNSOUQVSUTS RVMTSTUUVVV",   "MWOSTSURUQSPRPPQOSPURVTV",   "OVVMUMTNSPQVPXOYNY RQPUP",   "MXUSTQRPPQOSPURVTUUS RVPTVSXRYPYOX",   "MVQMNV ROSPQQPSPTQTRSTSUTVUV",   "PUSMSNTNTMSM RQPRPSQSRRTRUSVTV",   "OUSMSNTNTMSM RQPRPSQSRRVQXPYOYNX",   "NVRMOV RUPTPRQPS RPSQUSVTV",   "OTSMQSQURVSV",   "JYKPLPMQMSLV RMSNQOPQPRQRSQV RRSSQTPVPWQWRVTVUWVXV",   "MWNPOPPQPSOV RPSQQRPTPUQURTTTUUVVV",   "MWRPPQOSPURVTUUSTQRP",   "MXNPOPPQPSNY RPSQUSVUUVSUQSPQQPS",   "MXUSTQRPPQOSPURVTUUS RVPSY",   "MVOPPPQQQSPV RUQTPSPRQQS",   "NVTQSPQPPQPRQSRSSTSURVPVOU",   "NUSMQSQURVSV RPPTP",   "MWNPOPPQPROTOUPVRVSUTS RUPTSTUUVVV",   "MWNPOPPQPROTOUPVRVTUURUP",   "KYLPMPNQNRMTMUNVPVQURSSP RRSRUSVUVVUWRWP",   "MWOQPPQPRQRUSVTVUU RVQUPTPSQQUPVOVNU",   "MWNPOPPQPROTOUPVRVSUTS RUPSVRXQYOYNX",   "NVUPOV RPQQPSPTQ RPUQVSVTU",   "",   "",   "",   "",   "",   "",   "MWUSTQRPPQOSPURVTUUSUPTNRMQM",   "MWUQSPRPPQOSPURVSVUU ROSSS",   "MWRMQNPPOSOVPWRWSVTTUQUNTMRM RPRTR",   "MWTMQY RRPPQOSPURVSVUUVSUQSPRP",   "MWUQSPQPOQOSPTRUSVSWRXQX",   "",   "",   "KYTPTSUTVTWSWQVOUNSMQMONNOMQMSNUOVQWSWUV RTQSPQPPQPSQTSTTS",   "MWUNORUV",   "MWONUROV",   "OUTKQKQYTY",   "OUPKSKSYPY",   "OUTKSLRNROSQQRSSRURVSXTY",   "OUPKQLRNROQQSRQSRURVQXPY",   "LYPMQNQOPPOPNONNOMPMSNUNWMNV RUSTTTUUVVVWUWTVSUS",   "PT",   "NV",   
  /******** Hershey Glyphs 200 to 299 ********/   
  "MWRMPNOPOSPURVTUUSUPTNRM", "MWPORMRV", "MWONQMSMUNUPTROVUV", "MWONQMSMUNUPSQ RRQSQURUUSVQVOU", "MWSMSV RSMNSVS", "MWPMOQQPRPTQUSTURVQVOU RPMTM", "MWTMRMPNOPOSPURVTUUSTQRPPQOS", "MWUMQV ROMUM", "MWQMONOPQQSQUPUNSMQM RQQOROUQVSVUUURSQ", "MWUPTRRSPROPPNRMTNUPUSTURVPV", "PURURVSVSURU", "PUSVRVRUSUSWRY", "PURPRQSQSPRP RRURVSVSURU", "PURPRQSQSPRP RSVRVRUSUSWRY", "PURMRR RSMSR RRURVSVSURU", "NWPNRMSMUNUPRQRRSRSQUP RRURVSVSURU", "PTRMRQ", "NVPMPQ RTMTQ", "NVQMPNPPQQSQTPTNSMQM", "MWRKRX RUNSMQMONOPQQTRUSUUSVQVOU", "MWVLNX", "OUTKRNQQQSRVTY", "OUPKRNSQSSRVPY", "PTRKRY", "LXNRVR", "LXRNRV RNRVR", "LXNPVP RNTVT", "MWOOUU RUOOU", "MWRORU ROPUT RUPOT", "PURQRRSRSQRQ", "PUSMRORQSQSPRP", "PUSNRNRMSMSORQ", "LXSOVRSU RNRVR", "MXQLQY RTLTY ROQVQ ROTVT", "LXVRURTSSURVOVNUNSORRQSPSNRMPMONOPQSSUUVVV", "LXNNOQOSNV RVNUQUSVV RNNQOSOVN RNVQUSUVV", "LYRQQPOPNQNSOTQTRSSQTPVPWQWSVTTTSSRQ", "", "H\\NRMQLRMSNR RVRWQXRWSVR", "H\\MPLQLRMSNSOROQNPMP RMQMRNRNQMQ RWPVQVRWSXSYRYQXPWP RWQWRXRXQWQ", "I[KRYR", "", "H\\RUJPRTZPRU", "", "", "", "", "", "F^ISJQLPNPPQTTVUXUZT[Q RISJPLONOPPTSVTXTZS[Q RIYJWLVNVPWTZV[X[ZZ[W RIYJVLUNUPVTYVZXZZY[W", "", "F^ISJQLPNPPQTTVUXUZT[Q RISJPLONOPPTSVTXTZS[Q RIW[W RI[[[", "", "CaGO]OXI RL[GU]U", "", "D`F^^^^FFFF^", "", "KYQVOUNSNQOOQNSNUOVQVSUUSVQV RSVVS RQVVQ ROUUO RNSSN RNQQN", "", "H\\IR[R", "H\\IR[R RIQ[Q", "", "LYPFSCSP RRDRP ROPVP RMRXR ROVOWNWNVOUQTTTVUWWVYTZQ[O\\N^Na RTTUUVWUYTZ RN`O_P_S`V`W_ RP_SaVaW_W^", "LYPFSCSP RRDRP ROPVP RMRXR ROVOWNWNVOUQTTTVUWWVYTZ RTTUUVWUYTZ RRZTZV[W]W^V`TaQaO`N_N^O^O_ RTZU[V]V^U`Ta", "LYPFSCSP RRDRP ROPVP RMRXR RVVVWWWWVVUTTRTPUOVNYN^O`QaTaV`W^W\\VZTYQYN[ RRTPVOYO^P`Qa RTaU`V^V\\UZTY", "LYPFSCSP RRDRP ROPVP RMRXR RQTOUNWOYQZTZVYWWVUTTQT RQTPUOWPYQZ RTZUYVWUUTT RQZO[N]N^O`QaTaV`W^W]V[TZ RQZP[O]O^P`Qa RTaU`V^V]U[TZ", "LYOEOFNFNEODQCTCVDWFVHTIQJOKNMNP RTCUDVFUHTI RNOONPNSOVOWN RPNSPVPWNWM RMRXR ROVOWNWNVOUQTTTVUWWVYTZ RTTUUVWUYTZ RRZTZV[W]W^V`TaQaO`N_N^O^O_ RTZU[V]V^U`Ta", "LYOEOFNFNEODQCTCVDWFVHTI RTCUDVFUHTI RRITIVJWLWMVOTPQPOONNNMOMON RTIUJVLVMUOTP RMRXR RQTOUNWOYQZTZVYWWVUTTQT RQTPUOWPYQZ RTZUYVWUUTT RQZO[N]N^O`QaTaV`W^W]V[TZ RQZP[O]O^P`Qa RTaU`V^V]U[TZ", "LYOCNI ROCVC RODSDVC RNIOHQGTGVHWJWMVOTPQPOONNNMOMON RTGUHVJVMUOTP RMRXR RQTOUNWOYQZTZVYWWVUTTQT RQTPUOWPYQZ RTZUYVWUUTT RQZO[N]N^O`QaTaV`W^W]V[TZ RQZP[O]O^P`Qa RTaU`V^V]U[TZ", "LYNCNG RVERLPP RWCTIQP RNEPCRCUE RNEPDRDUEVE RMRXR RQTOUNWOYQZTZVYWWVUTTQT RQTPUOWPYQZ RTZUYVWUUTT RQZO[N]N^O`QaTaV`W^W]V[TZ RQZP[O]O^P`Qa RTaU`V^V]U[TZ", "LYOCNI ROCVC RODSDVC RNIOHQGTGVHWJWMVOTPQPOONNNMOMON RTGUHVJVMUOTP RMRXR RVVVWWWWVVUTTRTPUOVNYN^O`QaTaV`W^W\\VZTYQYN[ RRTPVOYO^P`Qa RTaU`V^V\\UZTY", "LYPFSCSP RRDRP ROPVP RMRXR RSVSa RTTTa RTTM]X] RQaVa", "LYOEOFNFNEODQCTCVDWFVHTI RTCUDVFUHTI RRITIVJWLWMVOTPQPOONNNMOMON RTIUJVLVMUOTP RMRXR RSVSa RTTTa RTTM]X] RQaVa", "F^YXWZU[R[PZMXKWIWHXHZI[K[MZOWPURQTKWGYFZF[G\\H[IZH[G[FZFYFWGVHTLRPPVNZMZ ROPUP", "E^P[MZJXHUGRGOHLJIMGPFTFWGYI[L\\O\\R[UYXVZS[P[ RNJNW ROJOW RLJSJVKWMWNVPSQOQ RSJUKVMVNUPSQ RLWQW RSQTRUVVWWWXV RSQURVVWW", "E^P[MZJXHUGRGOHLJIMGPFTFWGYI[L\\O\\R[UYXVZS[P[ RUKVJVNUKSJPJNKMLLOLRMUNVPWSWUVVT RPJNLMOMRNUPW", "E_IM[M RIR[R RIW[W RK[YI", "CaHQGRHSIRHQ RRQQRRSSRRQ R\\Q[R\\S]R\\Q", "", "E_NWLTIRLPNM RLPJRLT RJRZR RVWXT[RXPVM RXPZRXT", "JZWNTLRIPLMN RPLRJTL RRJRZ RWVTXR[PXMV RPXRZTX", "F^ZJSJOKMLKNJQJSKVMXOYSZZZ RSFS^", "F^JJQJUKWLYNZQZSYVWXUYQZJZ RQFQ^", "F^JJQJUKWLYNZQZSYVWXUYQZJZ RORZR", "", "H\\LBL[ RRBR[ RXBX[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 300 to 399 ********/ 
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 400 to 499 ********/ 
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 500 to 599 ********/ 
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aOdNfLgKfKdLaN^P\\SZWX", "F^MMKLJJJIKGMFNFPGQIQKPONULYJ[H[GZGX R^I^G]F\\FZGXIVLTNROPO RROSQSXTZU[V[XZYY[V", "I\\MRORSQVOXMYKYHXFVFUGTISNRSQVPXNZL[J[IZIXJWLWNXQZT[V[YZ[X", "@aEMCLBJBICGEFFFHGIIIKHPGTE[ RGTJLLHMGOFPFRGSISKRPQTO[ RQTTLVHWGYFZF\\G]I]K\\PZWZZ[[\\[^Z_YaV", "E]JMHLGJGIHGJFKFMGNINKMPLTJ[ RLTOLQHRGTFVFXGYIYKXPVWVZW[X[ZZ[Y]V", "H]TFQGOIMLLNKRKVLYMZO[Q[TZVXXUYSZOZKYHXGVFTFRHRKSNUQWSZU\\V", "F_SHTITLSPRSQUOXMZK[J[IZIWJRKOLMNJPHRGUFZF\\G]H^J^M]O\\PZQWQUPTO", "H^ULTNSOQPOPNNNLOIQGTFWFYGZIZMYPWSSWPYNZK[I[HZHXIWKWMXPZS[V[YZ[X", "F_SHTITLSPRSQUOXMZK[J[IZIWJRKOLMNJPHRGUFYF[G\\H]J]M\\O[PYQVQSPTQUSUXVZX[ZZ[Y]V", "H\\H[JZLXOTQQSMTJTGSFRFQGPIPKQMSOVQXSYUYWXYWZT[P[MZKXJVJT", "H[RLPLNKMINGQFTFXG[G]F RXGVNTTRXPZN[L[JZIXIVJULUNV", "E]JMHLGJGIHGJFKFMGNINKMOLRKVKXLZN[P[RZSYUUXMZF RXMWQVWVZW[X[ZZ[Y]V", "F]KMILHJHIIGKFLFNGOIOKNOMRLVLYM[O[QZTWVTXPYMZIZGYFXFWGVIVKWNYP[Q", "C_HMFLEJEIFGHFIFKGLILLK[ RUFK[ RUFS[ RaF_G\\JYNVTS[", "F^NLLLKKKILGNFPFRGSISLQUQXRZT[V[XZYXYVXUVU R]I]G\\FZFXGVITLPUNXLZJ[H[GZGX", "F]KMILHJHIIGKFLFNGOIOKNOMRLVLXMZN[P[RZTXVUWSYM R[FYMVWT]RbPfNgMfMdNaP^S[VY[V", "H]ULTNSOQPOPNNNLOIQGTFWFYGZIZMYPWTTWPZN[K[JZJXKWNWPXQYR[R^QaPcNfLgKfKdLaN^Q[TYZV", "", "", "", "", "", "", "I[JFR[ RZFR[ RJFZF", "G]IL[b", "E_RJIZ RRJ[Z", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 600 to 699 ********/ 
  "", "I\\XMX[ RXPVNTMQMONMPLSLUMXOZQ[T[VZXX", "H[LFL[ RLPNNPMSMUNWPXSXUWXUZS[P[NZLX", "I[XPVNTMQMONMPLSLUMXOZQ[T[VZXX", "I\\XFX[ RXPVNTMQMONMPLSLUMXOZQ[T[VZXX", "I[LSXSXQWOVNTMQMONMPLSLUMXOZQ[T[VZXX", "MYWFUFSGRJR[ ROMVM", "I\\XMX]W`VaTbQbOa RXPVNTMQMONMPLSLUMXOZQ[T[VZXX", "I\\MFM[ RMQPNRMUMWNXQX[", "NVQFRGSFREQF RRMR[", "MWRFSGTFSERF RSMS^RaPbNb", "IZMFM[ RWMMW RQSX[", "NVRFR[", "CaGMG[ RGQJNLMOMQNRQR[ RRQUNWMZM\\N]Q][", "I\\MMM[ RMQPNRMUMWNXQX[", "I\\QMONMPLSLUMXOZQ[T[VZXXYUYSXPVNTMQM", "H[LMLb RLPNNPMSMUNWPXSXUWXUZS[P[NZLX", "I\\XMXb RXPVNTMQMONMPLSLUMXOZQ[T[VZXX", "KXOMO[ ROSPPRNTMWM", "J[XPWNTMQMNNMPNRPSUTWUXWXXWZT[Q[NZMX", "MYRFRWSZU[W[ ROMVM", "I\\MMMWNZP[S[UZXW RXMX[", "JZLMR[ RXMR[", "G]JMN[ RRMN[ RRMV[ RZMV[", "J[MMX[ RXMM[", "JZLMR[ RXMR[P_NaLbKb", "J[XMM[ RMMXM RM[X[", "H]QMONMPLRKUKXLZN[P[RZUWWTYPZM RQMSMTNUPWXXZY[Z[", "I\\UFSGQIOMNPMTLZKb RUFWFYHYKXMWNUORO RROTPVRWTWWVYUZS[Q[OZNYMV", "I\\JPLNNMOMQNROSRSVR[ RZMYPXRR[P_Ob", "I[TMQMONMPLSLVMYNZP[R[TZVXWUWRVOTMRKQIQGRFTFVGXI", "JZWOVNTMQMONOPPRSS RSSOTMVMXNZP[S[UZWX", "JYTFRGQHQIRJUKXK RXKTMQONRMUMWNYP[S]T_TaSbQbP`", "H\\IQJOLMNMONOPNTL[ RNTPPRNTMVMXOXRWWTb", "G\\HQIOKMMMNNNPMUMXNZO[Q[SZUWVUWRXMXJWGUFSFRHRJSMUPWRZT", "LWRMPTOXOZP[R[TYUW", "I[OMK[ RYNXMWMUNQROSNS RNSPTQUSZT[U[VZ", "JZKFMFOGPHX[ RRML[", "H]OMIb RNQMVMYO[Q[SZUXWT RYMWTVXVZW[Y[[Y\\W", "I[LMOMNSMXL[ RYMXPWRUURXOZL[", "JZTFRGQHQIRJUKXK RUKRLPMOOOQQSTTVT RTTPUNVMXMZO\\S^T_TaRbPb", "J[RMPNNPMSMVNYOZQ[S[UZWXXUXRWOVNTMRM", "G]PML[ RUMVSWXX[ RIPKNNM[M", "I[MSMVNYOZQ[S[UZWXXUXRWOVNTMRMPNNPMSIb", "I][MQMONMPLSLVMYNZP[R[TZVXWUWRVOUNSM", "H\\SMP[ RJPLNOMZM", "H\\IQJOLMNMONOPMVMYO[Q[TZVXXTYPYM", "G]ONMOKQJTJWKYLZN[Q[TZWXYUZRZOXMVMTORSPXMb", "I[KMMMOOU`WbYb RZMYOWRM]K`Jb", "F]VFNb RGQHOJMLMMNMPLULXMZO[Q[TZVXXUZP[M", "F]NMLNJQITIWJZK[M[OZQW RRSQWRZS[U[WZYWZTZQYNXM", "L\\UUTSRRPRNSMTLVLXMZO[Q[SZTXVRUWUZV[W[YZZY\\V", "M[MVOSRNSLTITGSFQGPIOMNTNZO[P[RZTXUUURVVWWYW[V", "MXTTTSSRQROSNTMVMXNZP[S[VYXV", "L\\UUTSRRPRNSMTLVLXMZO[Q[SZTXZF RVRUWUZV[W[YZZY\\V", "NXOYQXRWSUSSRRQROSNUNXOZQ[S[UZVYXV", "OWOVSQUNVLWIWGVFTGSIQQNZKaJdJfKgMfNcOZP[R[TZUYWV", "L[UUTSRRPRNSMTLVLXMZO[Q[SZTY RVRTYPdOfMgLfLdMaP^S\\U[XY[V", "M\\MVOSRNSLTITGSFQGPIOMNSM[ RM[NXOVQSSRURVSVUUXUZV[W[YZZY\\V", "PWSMSNTNTMSM RPVRRPXPZQ[R[TZUYWV", "PWSMSNTNTMSM RPVRRLdKfIgHfHdIaL^O\\Q[TYWV", "M[MVOSRNSLTITGSFQGPIOMNSM[ RM[NXOVQSSRURVSVUTVQV RQVSWTZU[V[XZYY[V", "OWOVQSTNULVIVGUFSGRIQMPTPZQ[R[TZUYWV", "E^EVGSIRJSJTIXH[ RIXJVLSNRPRQSQTPXO[ RPXQVSSURWRXSXUWXWZX[Y[[Z\\Y^V", "J\\JVLSNROSOTNXM[ RNXOVQSSRURVSVUUXUZV[W[YZZY\\V", "LZRRPRNSMTLVLXMZO[Q[SZTYUWUUTSRRQSQURWTXWXYWZV", "KZKVMSNQMUGg RMUNSPRRRTSUUUWTYSZQ[ RMZO[R[UZWYZV", "L[UUTSRRPRNSMTLVLXMZO[Q[SZ RVRUUSZPaOdOfPgRfScS\\U[XY[V", "MZMVOSPQPSSSTTTVSYSZT[U[WZXYZV", "NYNVPSQQQSSVTXTZR[ RNZP[T[VZWYYV", "OXOVQSSO RVFPXPZQ[S[UZVYXV RPNWN", "L[LVNRLXLZM[O[QZSXUU RVRTXTZU[V[XZYY[V", "L[LVNRMWMZN[O[RZTXUUUR RURVVWWYW[V", "I^LRJTIWIYJ[L[NZPX RRRPXPZQ[S[UZWXXUXR RXRYVZW\\W^V", "JZJVLSNRPRQSQZR[U[XYZV RWSVRTRSSOZN[L[KZ", "L[LVNRLXLZM[O[QZSXUU RVRPdOfMgLfLdMaP^S\\U[XY[V", "LZLVNSPRRRTTTVSXQZN[P\\Q^QaPdOfMgLfLdMaP^S\\WYZV", "J\\K[NZQXSVUSWOXKXIWGUFSGRHQJPOPTQXRZT[V[XZYY", "", "", "", "", "", "I[WUWRVOUNSMQMONMPLSLVMYNZP[R[TZVXWUXPXKWHVGTFRFPGNI", "JZWNUMRMPNNPMSMVNYOZQ[T[VZ RMTUT", "J[TFRGPJOLNOMTMXNZO[Q[SZUWVUWRXMXIWGVFTF RNPWP", "H\\VFNb RQMNNLPKSKVLXNZQ[S[VZXXYUYRXPVNSMQM", "I[XOWNTMQMNNMOLQLSMUOWSZT\\T^S_Q_", "", "", "DaWNVLTKQKOLNMMOMRNTOUQVTVVUWS RWKWSXUYV[V\\U]S]O\\L[JYHWGTFQFNGLHJJILHOHRIUJWLYNZQ[T[WZYY", "F^ZIJRZ[", "F^JIZRJ[", "KYOBOb ROBVB RObVb", "KYUBUb RNBUB RNbUb", "KYTBQEPHPJQMSOSPORSTSUQWPZP\\Q_Tb", "KYPBSETHTJSMQOQPURQTQUSWTZT\\S_Pb", "F^[FYGVHSHPGNFLFJGIIIKKMMMOLPJPHNF R[FI[ RYTWTUUTWTYV[X[ZZ[X[VYT", "NV", "JZ", 
  /******** Hershey Glyphs 700 to 799 ********/
  "H\\QFNGLJKOKRLWNZQ[S[VZXWYRYOXJVGSFQF", "H\\NJPISFS[", "H\\LKLJMHNGPFTFVGWHXJXLWNUQK[Y[", "H\\MFXFRNUNWOXPYSYUXXVZS[P[MZLYKW", "H\\UFKTZT RUFU[", "H\\WFMFLOMNPMSMVNXPYSYUXXVZS[P[MZLYKW", "H\\XIWGTFRFOGMJLOLTMXOZR[S[VZXXYUYTXQVOSNRNOOMQLT", "H\\YFO[ RKFYF", "H\\PFMGLILKMMONSOVPXRYTYWXYWZT[P[MZLYKWKTLRNPQOUNWMXKXIWGTFPF", "H\\XMWPURRSQSNRLPKMKLLINGQFRFUGWIXMXRWWUZR[P[MZLX", "MWRYQZR[SZRY", "MWSZR[QZRYSZS\\R^Q_", "MWRMQNROSNRM RRYQZR[SZRY", "MWRMQNROSNRM RSZR[QZRYSZS\\R^Q_", "MWRFRT RRYQZR[SZRY", "I[LKLJMHNGPFTFVGWHXJXLWNVORQRT RRYQZR[SZRY", "NVRFRM", "JZNFNM RVFVM", "KYQFOGNINKOMQNSNUMVKVIUGSFQF", "H\\PBP_ RTBT_ RYIWGTFPFMGKIKKLMMNOOUQWRXSYUYXWZT[P[MZKX", "G][BIb", "KYVBTDRGPKOPOTPYR]T`Vb", "KYNBPDRGTKUPUTTYR]P`Nb", "NVRBRb", "E_IR[R", "E_RIR[ RIR[R", "E_IO[O RIU[U", "G]KKYY RYKKY", "JZRLRX RMOWU RWOMU", "MWRQQRRSSRRQ", "MWSFRGQIQKRLSKRJ", "MWRHQGRFSGSIRKQL", "E_UMXP[RXTUW RIR[R", "G]OFOb RUFUb RJQZQ RJWZW", "E_\\O\\N[MZMYNXPVUTXRZP[L[JZIYHWHUISJRQNRMSKSIRGPFNGMIMKNNPQUXWZY[[[\\Z\\Y", "G]IIJKKOKUJYI[ R[IZKYOYUZY[[ RIIKJOKUKYJ[I RI[KZOYUYYZ[[", "F_\\Q[OYNWNUOTPQTPUNVLVJUISIQJOLNNNPOQPTTUUWVYV[U\\S\\Q", "KYOBO[ RUBU[", "F^RBR[ RI[[[", "F^[BI[[[", "E_RIQJRKSJRI RIYHZI[JZIY R[YZZ[[\\Z[Y", "F^RHNLKPJSJUKWMXOXQWRU RRHVLYPZSZUYWWXUXSWRU RRUQYP\\ RRUSYT\\ RP\\T\\", "F^RNQKPINHMHKIJKJOKRLTNWR\\ RRNSKTIVHWHYIZKZOYRXTVWR\\", "F^RGPJLOIR RRGTJXO[R RIRLUPZR] R[RXUTZR]", "F^RTTWVXXXZW[U[SZQXPVPSQ RSQUOVMVKUISHQHOINKNMOOQQ RQQNPLPJQISIUJWLXNXPWRT RRTQYP\\ RRTSYT\\ RP\\T\\", "F^RRR[Q\\ RRVQ\\ RRIQHOHNINKONRR RRISHUHVIVKUNRR RRRNOLNJNIOIQJR RRRVOXNZN[O[QZR RRRNULVJVIUISJR RRRVUXVZV[U[SZR", "F^ISJSLTMVMXLZ RISIRJQLQMRNTNWMYLZ RRGPIOLOOQUQXPZR\\ RRGTIULUOSUSXTZR\\ R[S[RZQXQWRVTVWWYXZ R[SZSXTWVWXXZ RKVYV", "", "", "", "PSSRRSQSPRPQQPRPSQSSRUQV RQQQRRRRQQQ", "PTQPPQPSQTSTTSTQSPQP RRQQRRSSRRQ", "NVPOTU RTOPU RNRVR", "MWRKQMOPMR RRKSMUPWR RRMOQ RRMUQ RROPQ RROTQ RQQSQ RMRWR", "MWMRMQNOONQMSMUNVOWQWR RPNTN ROOUO RNPVP RNQVQ RMRWR", "LRLFLRRRLF RLIPQ RLLOR RLOMQ", "MWRKQMOPMR RRKSMUPWR", "MWWRWQVOUNSMQMONNOMQMR", "G]]R]P\\MZJWHTGPGMHJJHMGPGR", "MWMRMSNUOVQWSWUVVUWSWR", "LXLPNRQSSSVRXP", "RURUTTURTPRO", "RVRRUPVNVLUKTK", "NRRROPNNNLOKPK", "MWWHVGTFQFOGNHMJMLNNOOUSVTWVWXVZU[S\\P\\N[MZ", "G]IWHVGTGQHOINKMMMONPOTUUVWWYW[V\\U]S]P\\N[M", "G]RRTUUVWWYW[V\\U]S]Q\\O[NYMWMUNTOPUOVMWKWIVHUGSGQHOINKMMMONPORR", "H\\KFK[ RHF[FQP[Z RZV[Y\\[ RZVZY RWYZY RWYZZ\\[", "KYUARBPCNELHKLKRLUNWQXSXVWXUYR RKPLMNKQJSJVKXMYPYVXZV]T_R`Oa", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", ">f>RfR", "D`D``D", "RRR>Rf", "D`DD``", 
  /******** Hershey Glyphs 800 to 899 ********/ 
  "D`DR`R", "F^FY^K", "KYK^YF", "", "KYKFY^", "F^FK^Y", "KYKRYR", "MWMWWM", "", "MWMMWW", "", "", "", "", "D`DOGQKSPTTTYS]Q`O", "PUUDSGQKPPPTQYS]U`", "OTODQGSKTPTTSYQ]O`", "D`DUGSKQPPTPYQ]S`U", "KYRJYNKVRZ", "JZJRNKVYZR", "KYKVKNYVYN", "JZLXJPZTXL", "JZJ]L]O\\Q[TXUVVSVOULTJSIQIPJOLNONSOVPXS[U\\X]Z]", "I]]Z]X\\U[SXPVOSNONLOJPIQISJTLUOVSVVUXT[Q\\O]L]J", "JZZGXGUHSIPLONNQNUOXPZQ[S[TZUXVUVQUNTLQIOHLGJG", "G[GJGLHOIQLTNUQVUVXUZT[S[QZPXOUNQNNOLPISHUGXGZ", "E[EPFRHTJUMVQVUUXSZP[NZLWLSMQNNPLSKVKYL\\M^", "EYETHVKWPWSVVTXQYNYLXKVKSLPNNQMTMYN\\P_", "OUQOOQOSQUSUUSUQSOQO RQPPQPSQTSTTSTQSPQP RRQQRRSSRRQ", "", "D`DRJR RORUR RZR`R", "D`DUDO`O`U", "JZRDJR RRDZR", "D`DR`R RJYZY RP`T`", "D`DR`R RDRRb R`RRb", "", "", "", "", "", "KYQKNLLNKQKSLVNXQYSYVXXVYSYQXNVLSKQK", "LXLLLXXXXLLL", "KYRJKVYVRJ", "LXRHLRR\\XRRH", "JZRIPOJOOSMYRUWYUSZOTORI", "KYRKRY RKRYR", "MWMMWW RWMMW", "MWRLRX RMOWU RWOMU", "", "", "NVQNOONQNSOUQVSVUUVSVQUOSNQN ROQOS RPPPT RQOQU RRORU RSOSU RTPTT RUQUS", "NVNNNVVVVNNN ROOOU RPOPU RQOQU RRORU RSOSU RTOTU RUOUU", "MWRLMUWURL RROOT RROUT RRRQT RRRST", "LULRUWUMLR RORTU RORTO RRRTS RRRTQ", "MWRXWOMORX RRUUP RRUOP RRRSP RRRQP", "OXXROMOWXR RURPO RURPU RRRPQ RRRPS", "LXRLNWXPLPVWRL RRRRL RRRLP RRRNW RRRVW RRRXP", "", "", "", "MWRLRX ROOUO RMUOWQXSXUWWU", "LXRLRX RLQMOWOXQ RPWTW", "KYMNWX RWNMX ROLLOKQ RULXOYQ", "I[NII[ RVI[[ RMM[[ RWMI[ RNIVI RMMWM", "I[RGRV RMJWP RWJMP RIVL\\ R[VX\\ RIV[V RL\\X\\", "G[MJSV RKPSL RG\\[\\[RG\\", "LXPLPPLPLTPTPXTXTTXTXPTPTLPL", "KYYPXNVLSKQKNLLNKQKSLVNXQYSYVXXVYT RYPWNUMSMQNPOOQOSPUQVSWUWWVYT", "KYRJKVYVRJ RRZYNKNRZ", "G]PIPGQFSFTGTI RGZHXJVKTLPLKMJOIUIWJXKXPYTZV\\X]Z RGZ]Z RQZP[Q\\S\\T[SZ", "JZRMRS RRSQ\\ RRSS\\ RQ\\S\\ RRMQJPHNG RQJNG RRMSJTHVG RSJVG RRMNKLKJM RPLLLJM RRMVKXKZM RTLXLZM RRMPNOOOR RRMPOOR RRMTNUOUR RRMTOUR", "JZRIRK RRNRP RRSRU RRYQ\\ RRYS\\ RQ\\S\\ RRGQIPJ RRGSITJ RPJRITJ RRKPNNOMN RRKTNVOWN RNOPORNTOVO RRPPSNTLTKRKSLT RRPTSVTXTYRYSXT RNTPTRSTTVT RRUPXOYMZLZKYJWJYLZ RRUTXUYWZXZYYZWZYXZ RMZOZRYUZWZ", "JZRYQ\\ RRYS\\ RQ\\S\\ RRYUZXZZXZUYTWTYRZOYMWLUMVJUHSGQGOHNJOMMLKMJOKRMTKTJUJXLZOZRY", "JZRYQ\\ RRYS\\ RQ\\S\\ RRYVXVVXUXRZQZLYIXHVHTGPGNHLHKIJLJQLRLUNVNXRY", "I[IPKR RLKNP RRGRO RXKVP R[PYR", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "QSRQQRRSSRRQ", 
  /******** Hershey Glyphs 900 to 999 ********/ 
  "PTQPPQPSQTSTTSTQSPQP", "NVQNOONQNSOUQVSVUUVSVQUOSNQN", "MWQMONNOMQMSNUOVQWSWUVVUWSWQVOUNSMQM", "KYQKNLLNKQKSLVNXQYSYVXXVYSYQXNVLSKQK", "G]PGMHJJHMGPGTHWJZM\\P]T]W\\ZZ\\W]T]P\\MZJWHTGPG", "AcPALBJCGEEGCJBLAPATBXCZE]G_JaLbPcTcXbZa]__]aZbXcTcPbLaJ_G]EZCXBTAPA", "<hP<K=G?DAAD?G=K<P<T=Y?]A`DcGeKgPhThYg]e`cc`e]gYhThPgKeGcD`A]?Y=T<P<", "){O)I*E+@-;073370;-@+E*I)O)U*[+_-d0i3m7q;t@wEyIzO{U{[z_ydwitmqqmtiwdy_z[{U{OzIyEw@t;q7m3i0d-_+[*U)O)", ">fRAPCMDJDGCEA>H@JAMAZB]D_G`M`PaRc RRATCWDZD]C_AfHdJcMcZb]`_]`W`TaRc", "AcRAPCMDJDGCEABGAKAPBTDXG\\L`Rc RRATCWDZD]C_AbGcKcPbT`X]\\X`Rc RBHbH", "H[WPVQWRXQXPVNTMQMNNLPKSKULXNZQ[S[VZXX RQMONMPLSLUMXOZQ[ RLbXF", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 1000 to 1099 ********/ 
  "", "KYRKMX RRNVX RRKWX ROTTT RKXPX RTXYX", "JZNKNX ROKOX RLKSKVLWNVPSQ RSKULVNUPSQ ROQSQVRWTWUVWSXLX RSQURVTVUUWSX", "KYVLWKWOVLTKQKOLNMMPMSNVOWQXTXVWWU RQKOMNPNSOVQX", "JZNKNX ROKOX RLKSKVLWMXPXSWVVWSXLX RSKULVMWPWSVVUWSX", "JYNKNX ROKOX RSOSS RLKVKVOUK ROQSQ RLXVXVTUX", "JXNKNX ROKOX RSOSS RLKVKVOUK ROQSQ RLXQX", "K[VLWKWOVLTKQKOLNMMPMSNVOWQXTXVW RQKOMNPNSOVQX RTXUWVU RVSVX RWSWX RTSYS", "J[NKNX ROKOX RVKVX RWKWX RLKQK RTKYK ROQVQ RLXQX RTXYX", "NWRKRX RSKSX RPKUK RPXUX", "LXSKSURWQX RTKTUSWQXPXNWMUNTOUNV RQKVK", "JZNKNX ROKOX RWKOS RQQVX RRQWX RLKQK RTKYK RLXQX RTXYX", "KXOKOX RPKPX RMKRK RMXWXWTVX", "I\\MKMX RNNRX RNKRU RWKRX RWKWX RXKXX RKKNK RWKZK RKXOX RUXZX", "JZNKNX ROMVX ROKVV RVKVX RLKOK RTKXK RLXPX", "KZQKOLNMMPMSNVOWQXTXVWWVXSXPWMVLTKQK RQKOMNPNSOVQX RTXVVWSWPVMTK", "JYNKNX ROKOX RLKSKVLWNWOVQSROR RSKULVNVOUQSR RLXQX", "KZQKOLNMMPMSNVOWQXTXVWWVXSXPWMVLTKQK RQKOMNPNSOVQX RTXVVWSWPVMTK RPWPUQTSTTUUZV[W[XZ RTUUXVZW[", "JZNKNX ROKOX RLKSKVLWNWOVQSROR RSKULVNVOUQSR RLXQX RSRTSUWVXWXXW RSRUSVWWX", "KZVMWKWOVMULSKQKOLNMNOOPQQTRVSWT RNNOOQPTQVRWSWVVWTXRXPWOVNTNXOV", "KZRKRX RSKSX RNKMOMKXKXOWK RPXUX", "J[NKNUOWQXTXVWWUWK ROKOUPWQX RLKQK RUKYK", "KYMKRX RNKRU RWKRX RKKPK RTKYK", "I[LKOX RMKOT RRKOX RRKUX RSKUT RXKUX RJKOK RVKZK", "KZNKVX ROKWX RWKNX RLKQK RTKYK RLXQX RTXYX", "LYNKRRRX ROKSR RWKSRSX RLKQK RTKYK RPXUX", "LYVKNX RWKOX ROKNONKWK RNXWXWTVX", "KYRKMX RRNVX RRKWX ROTTT RKXPX RTXYX", "JZNKNX ROKOX RLKSKVLWNVPSQ RSKULVNUPSQ ROQSQVRWTWUVWSXLX RSQURVTVUUWSX", "KXOKOX RPKPX RMKWKWOVK RMXRX", "KYRKLX RRMWX RRKXX RMWVW RLXXX", "JYNKNX ROKOX RSOSS RLKVKVOUK ROQSQ RLXVXVTUX", "LYVKNX RWKOX ROKNONKWK RNXWXWTVX", "J[NKNX ROKOX RVKVX RWKWX RLKQK RTKYK ROQVQ RLXQX RTXYX", "KZQKOLNMMPMSNVOWQXTXVWWVXSXPWMVLTKQK RQKOMNPNSOVQX RTXVVWSWPVMTK RQOQT RTOTT RQQTQ RQRTR", "NWRKRX RSKSX RPKUK RPXUX", "JZNKNX ROKOX RWKOS RQQVX RRQWX RLKQK RTKYK RLXQX RTXYX", "KYRKMX RRNVX RRKWX RKXPX RTXYX", "I\\MKMX RNNRX RNKRU RWKRX RWKWX RXKXX RKKNK RWKZK RKXOX RUXZX", "JZNKNX ROMVX ROKVV RVKVX RLKOK RTKXK RLXPX", "JZMJLM RXJWM RPPOS RUPTS RMVLY RXVWY RMKWK RMLWL RPQTQ RPRTR RMWWW RMXWX", "KZQKOLNMMPMSNVOWQXTXVWWVXSXPWMVLTKQK RQKOMNPNSOVQX RTXVVWSWPVMTK", "J[NKNX ROKOX RVKVX RWKWX RLKYK RLXQX RTXYX", "JYNKNX ROKOX RLKSKVLWNWOVQSROR RSKULVNVOUQSR RLXQX", "K[MKRQ RNKSQMX RMKWKXOVK RNWWW RMXWXXTVX", "KZRKRX RSKSX RNKMOMKXKXOWK RPXUX", "KZMONLOKPKQLRORX RXOWLVKUKTLSOSX RMONMOLPLQMRO RXOWMVLULTMSO RPXUX", "KZRKRX RSKSX RQNNOMQMRNTQUTUWTXRXQWOTNQN RQNOONQNROTQU RTUVTWRWQVOTN RPKUK RPXUX", "KZNKVX ROKWX RWKNX RLKQK RTKYK RLXQX RTXYX", "J[RKRX RSKSX RLPMONOOSQU RTUVSWOXOYP RMONROTQUTUVTWRXO RPKUK RPXUX", "KZMVNXQXMRMONMOLQKTKVLWMXOXRTXWXXV ROUNRNOOMQK RTKVMWOWRVU RNWPW RUWWW", "KYTKKX RSMTX RTKUX RNTTT RIXNX RRXWX", "JYPKLX RQKMX RNKUKWLWNVPSQ RUKVLVNUPSQ ROQRQTRUSUUTWQXJX RRQTSTUSWQX", "KXVLWLXKWNVLTKRKPLOMNOMRMUNWPXRXTWUU RRKPMOONRNVPX", "JYPKLX RQKMX RNKTKVLWNWQVTUVTWQXJX RTKULVNVQUTTVSWQX", "JYPKLX RQKMX RSORS RNKXKWNWK ROQRQ RJXTXUUSX", "JXPKLX RQKMX RSORS RNKXKWNWK ROQRQ RJXOX", "KYVLWLXKWNVLTKRKPLOMNOMRMUNWPXRXTWUVVS RRKPMOONRNVPX RRXTVUS RSSXS", "J[PKLX RQKMX RXKTX RYKUX RNKSK RVK[K ROQVQ RJXOX RRXWX", "NWTKPX RUKQX RRKWK RNXSX", "LXUKRUQWPX RVKSURWPXOXMWLUMTNUMV RSKXK", "JZPKLX RQKMX RYKOR RRPTX RSPUX RNKSK RVK[K RJXOX RRXWX", "KXQKMX RRKNX ROKTK RKXUXVUTX", "I\\OKKX ROMPX RPKQV RYKPX RYKUX RZKVX RMKPK RYK\\K RIXMX RSXXX", "JZPKLX RPKTX RQKTU RXKTX RNKQK RVKZK RJXNX", "KYRKPLOMNOMRMUNWPXRXTWUVVTWQWNVLTKRK RRKPMOONRNVPX RRXTVUTVQVMTK", "JYPKLX RQKMX RNKUKWLXMXOWQTROR RUKWMWOVQTR RJXOX", "KYRKPLOMNOMRMUNWPXRXTWUVVTWQWNVLTKRK RRKPMOONRNVPX RRXTVUTVQVMTK ROWOVPUQURVRZS[T[UZ RRVSZT[", "JZPKLX RQKMX RNKUKWLXMXOWQTROR RUKWMWOVQTR RSRTWUXVXWW RSRTSUWVX RJXOX", "KZWLXLYKXNWLUKRKPLOMOOPPUSVT RONPOURVSVVUWSXPXNWMULXMWNW", "KZTKPX RUKQX RPKNNOKZKYNYK RNXSX", "J[PKMUMWOXSXUWVUYK RQKNUNWOX RNKSK RWK[K", "KYOKPX RPKQV RYKPX RMKRK RVK[K", "I[NKMX ROKNV RTKMX RTKSX RUKTV RZKSX RLKQK RXK\\K", "KZPKTX RQKUX RYKLX RNKSK RVK[K RJXOX RRXWX", "LYPKRQPX RQKSQ RYKSQQX RNKSK RVK[K RNXSX", "LYXKLX RYKMX RQKONPKYK RLXUXVUTX", "", "", "", "", "", "", "", "KZMHX\\", "JZRMLW RRMXW", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 1100 to 1199 ********/ 
  "", "LZQOPPPQOQOPQOTOVQVWWXXX RTOUQUWWX RURRSPTOUOWPXSXTWUU RRSPUPWQX", "JYNKNX ROKOX RORPPROTOVPWRWUVWTXRXPWOU RTOUPVRVUUWTX RLKOK", "LXVQUQURVRVQUPSOQOOPNRNUOWQXSXUWVV RQOPPOROUPWQX", "L[VKVX RWKWX RVRUPSOQOOPNRNUOWQXSXUWVU RQOPPOROUPWQX RTKWK RVXYX", "LXOSVSVRUPSOQOOPNRNUOWQXSXUWVV RUSUQSO RQOPPOROUPWQX", "LWTKULUMVMVLTKRKPMPX RRKQMQX RNOSO RNXSX", "LYQOOQOSQUSUUSUQSOQO RQOPQPSQU RSUTSTQSO RTPUOVO RPTOUOXPYTYVZ ROWPXTXVYV[T\\P\\N[NYPX", "J[NKNX ROKOX RORPPROTOVPWRWX RTOUPVRVX RLKOK RLXQX RTXYX", "NWRKRLSLSKRK RRORX RSOSX RPOSO RPXUX", "NWSKSLTLTKSK RSOSZR\\ RTOTZR\\P\\O[OZPZP[O[ RQOTO", "JZNKNX ROKOX RWOOU RRSVX RSSWX RLKOK RTOYO RLXQX RTXYX", "NWRKRX RSKSX RPKSK RPXUX", "F_JOJX RKOKX RKRLPNOPORPSRSX RPOQPRRRX RSRTPVOXOZP[R[X RXOYPZRZX RHOKO RHXMX RPXUX RXX]X", "J[NONX ROOOX RORPPROTOVPWRWX RTOUPVRVX RLOOO RLXQX RTXYX", "LYQOOPNRNUOWQXTXVWWUWRVPTOQO RQOPPOROUPWQX RTXUWVUVRUPTO", "JYNON\\ ROOO\\ RORPPROTOVPWRWUVWTXRXPWOU RTOUPVRVUUWTX RLOOO RL\\Q\\", "KYUOU\\ RVOV\\ RURTPROPONPMRMUNWPXRXTWUU RPOOPNRNUOWPX RS\\X\\", "KXOOOX RPOPX RPRQPSOUOVPVQUQUPVP RMOPO RMXRX", "LYTOUPUQVQVPTOQOOPORQSTTVU ROQQRTSVTVWTXQXOWOVPVPWQX", "LWPKPVRXTXUWUV RQKQVRX RNOTO", "J[NONUOWQXSXUWVU ROOOUPWQX RVOVX RWOWX RLOOO RTOWO RVXYX", "KYNORX ROORV RVORX RLOQO RTOXO", "I[LOOX RMOOU RROOX RROUX RSOUU RXOUX RJOOO RVOZO", "KYNOUX ROOVX RVONX RLOQO RTOXO RLXPX RSXXX", "KYNORX ROORV RVORXP[N\\M\\L[LZMZM[L[ RLOQO RTOXO", "LXUONX RVOOX ROONQNOVO RNXVXVVUX", "K[QOOPNQMSMUNWPXQXSWUUWRXO RQOOQNSNUOWPX RQOSOUPWWXX RSOTPVWXXYX", "KXRKPMOOMUK\\ RQLPNNTL\\ RRKTKVLVNUPRQ RTKULUNTPRQ RRQTRUTUVTWRXQXOWNT RRQSRTTTVRX", "KYLQNOPORPSSSXR\\ RLQNPPPRQSS RWOVRSXQ\\", "KYSOQOOPNQMSMUNWPXRXTWUVVTVRUPRNQLQKRJTJUKVM RQOOQNSNVPX RRXTVUTUQSO RQLRKTKVM", "LXVPTOQOOPOQPRRS RQOPPPQRS RRSOTNUNWPXSXUW RRSPTOUOWPX", "LWRKQLQMSNVNVMSNPOOPNRNTOVPWRXSYS[R\\P\\O[ RSNQOPPOROTPVRX", "IYJRKPLONOOPOQMX RMONPNQLX ROQPPROTOVPVRS\\ RTOUPURR\\", "IYJSKQLPNPOQOVPX RMPNQNUOWPXQXSWTVUTVQVNULTKRKQLQNRPURWS RQXSVTTUQUNTK", "NWROPVPWQXSXUWVU RSOQVQWRX", "KYOOLX RPOMX RUOVPWPVOTORQOR RORPSRWTXVWWU RORQSSWTX", "LXLKNKPLWX RNKOLVX RRPMX RRPNX", "KZOOK\\ RPOL\\ RNUNWOXQXSWTV RVOTVTWUXWXXWYU RWOUVUWVX", "JYNOMX ROONUMX RVRVOWOVRTUQWNXMX RLOOO", "MXRKQLQMSNVN RTNQOPPPRRSUS RTNROQPQRRS RSSPTOUOWQXSYTZT[S\\Q\\ RSSQTPUPWQX", "KXQOOPNQMSMUNWPXRXTWUVVTVRUPSOQO RQOOQNSNVPX RRXTVUTUQSO", "IZPPMX RPPNX RTPSX RTPTX RKQMOXO RKQMPXP", "JXSOQOOPNQMSJ\\ RQOOQNSK\\ RSOUPVRVTUVTWRXPXNWMU RSOUQUTTVRX", "K[YOQOOPNQMSMUNWPXRXTWUVVTVRUPYP RQOOQNSNVPX RRXTVUTUQSO", "KZSPQX RSPRX RMQOOXO RMQOPXP", "JXKRLPMOOOPPPROUOWPX RNOOPORNUNWPXQXSWUUVRVOUOVP", "KZOPNQMSMUNWPXRXUWWUXRXPWOUOTPSRRUO\\ RMUNVPWRWUVWTXR RXQWPUPSR RRUQXP\\", "KXMONOPPS[T\\ RNOOPR[T\\U\\ RVOTRNYL\\", "I[TKQ\\ RUKP\\ RJRKPLONOOPOVPWSWUVWT RMONPNTOWPXSXUWWTXRYO", "JZNPPPPONPMQLSLUMWNXPXQWRUSR RLUNWPWRU RRRRWSXUXWVXTXRWPVOVPWP RRUSWUWWV", "KZVOTVTWUXWXXWYU RWOUVUWVX RUSUQSOQOOPNQMSMUNWPXRXTV RQOOQNSNVPX", "JXOKMR RPKNRNVPX RNROPQOSOUPVRVTUVTWRXPXNWMUMR RSOUQUTTVRX RMKPK", "KXUPUQVQUPSOQOOPNQMSMUNWPXRXTWUV RQOOQNSNVPX", "KZWKTVTWUXWXXWYU RXKUVUWVX RUSUQSOQOOPNQMSMUNWPXRXTV RQOOQNSNVPX RUKXK", "KWNURTTSURUPSOQOOPNQMSMUNWPXRXTWUV RQOOQNSNVPX", "MXWKXLXKVKTLSNPYO[N\\ RVKULTNQYP[N\\L\\L[M\\ RPOVO", "KYVOTVSYR[ RWOUVTYR[P\\M\\L[M[N\\ RUSUQSOQOOPNQMSMUNWPXRXTV RQOOQNSNVPX", "KZPKLX RQKMX ROQPPROTOVPVRUUUWVX RTOUPURTUTWUXWXXWYU RNKQK", "MWSKSLTLTKSK RNROPPOROSPSRRURWSX RQORPRRQUQWRXTXUWVU", "MWTKTLULUKTK RORPPQOSOTPTRRYQ[O\\M\\M[N\\ RROSPSRQYP[O\\", "KXPKLX RQKMX RVPUQVQVPUOTORQPROR RORPSQWRXTXUWVU RORQSRWSX RNKQK", "NVSKPVPWQXSXTWUU RTKQVQWRX RQKTK", "F^GRHPIOKOLPLQJX RJOKPKQIX RLQMPOOQOSPSQQX RQORPRQPX RSQTPVOXOZPZRYUYWZX RXOYPYRXUXWYX[X\\W]U", "J[KRLPMOOOPPPQNX RNOOPOQMX RPQQPSOUOWPWRVUVWWX RUOVPVRUUUWVXXXYWZU", "KXQOOPNQMSMUNWPXRXTWUVVTVRUPSOQO RQOOQNSNVPX RRXTVUTUQSO", "JYKRLPMOOOPPPQM\\ RNOOPOQL\\ RPQROTOVPWRWTVVUWSXQXOVOT RTOVQVTUVSX RJ\\O\\", "KYVOR\\ RWOS\\ RUSUQSOQOOPNQMSMUNWPXRXTV RQOOQNSNVPX RP\\U\\", "LXMRNPOOQORPRQPX RPOQPQQOX RRQSPUOVOWPWQVQWP", "LYVPVQWQVPTOQOOPORQSTTVU ROQQRTSVTVWTXQXOWNVOVOW", "NWSKPVPWQXSXTWUU RTKQVQWRX RPOUO", "IZJRKPLONOOPORNUNWOX RMONPNRMUMWOXQXSWTV RVOTVTWUXWXXWYU RWOUVUWVX", "JXKRLPMOOOPPPROUOWPX RNOOPORNUNWPXQXSWUUVRVOUOVP", "H\\IRJPKOMONPNRMUMWNX RLOMPMRLULWNXOXQWRV RTORVRWTX RUOSVSWTXUXWWYUZRZOYOZP", "JZMRNPPOROSPSR RQORPRRQUPWNXMXLWLVMVLW RXPWQXQXPWOVOTPSRRURWSX RQUQWRXTXVWWU", "IYJRKPLONOOPORNUNWOX RMONPNRMUMWOXQXSWTV RVOTVSYR[ RWOUVTYR[P\\M\\L[M[N\\", "KYWOWPVQNVMWMX RNQOOROUQ ROPRPUQVQ RNVOVRWUW ROVRXUXVV", "H[RKSLSMTMTLRKOKMLLNLX ROKNLMNMX RXKYLYMZMZLXKVKTMTX RVKUMUX RJOWO RJXOX RRXWX", "J[UKVLWLWKQKOLNNNX RQKPLONOX RVOVX RWOWX RLOWO RLXQX RTXYX", "J[WKQKOLNNNX RQKPLONOX RUKVLVX RWKWX RLOVO RLXQX RTXYX", "F_PKQLQMRMRLPKMKKLJNJX RMKLLKNKX RYKZL[L[KUKSLRNRX RUKTLSNSX RZOZX R[O[X RHO[O RHXMX RPXUX RXX]X", "F_PKQLQMRMRLPKMKKLJNJX RMKLLKNKX R[KUKSLRNRX RUKTLSNSX RYKZLZX R[K[X RHOZO RHXMX RPXUX RXX]X", "NWRORX RSOSX RPOSO RPXUX", "", "LXVPTOROPPOQNSNUOWQXSXUW RROPQOSOVQX ROSSS", "LYSKQLPMOONRNUOWPXRXTWUVVTWQWNVLUKSK RSKQMPOOSOVPX RRXTVUTVPVMUK ROQVQ", "KZTKQ\\ RUKP\\ RQONPMRMUNWQXTXWWXUXRWPTOQO RQOOPNRNUOWQX RTXVWWUWRVPTO", "LXUPVRVQUPSOQOOPNRNTOVRX RQOOQOTPVRXSYS[R\\P\\", "", "", "", "I[VKWLXLVKSKQLPMOOLYK[J\\ RSKQMPOMYL[J\\H\\H[I\\ RZK[L[KYKWLVNSYR[Q\\ RYKXLWNTYS[Q\\O\\O[P\\ RLOYO", "IZVKWLXLXKSKQLPMOOLYK[J\\ RSKQMPOMYL[J\\H\\H[I\\ RVOTVTWUXWXXWYU RWOUVUWVX RLOWO", "IZVKWL RXKSKQLPMOOLYK[J\\ RSKQMPOMYL[J\\H\\H[I\\ RWKTVTWUXWXXWYU RXKUVUWVX RLOVO", "F^SKTLTM RULSKPKNLMMLOIYH[G\\ RPKNMMOJYI[G\\E\\E[F\\ RZK[L\\L\\KWKUL RTMSOPYO[N\\ RWKUMTOQYP[N\\L\\L[M\\ RZOXVXWYX[X\\W]U R[OYVYWZX RIO[O", "F^SKTLTM RULSKPKNLMMLOIYH[G\\ RPKNMMOJYI[G\\E\\E[F\\ RZK[L R\\KWKUL RTMSOPYO[N\\ RWKUMTOQYP[N\\L\\L[M\\ R[KXVXWYX[X\\W]U R\\KYVYWZX RIOZO", "MWNROPPOROSPSRRURWSX RQORPRRQUQWRXTXUWVU", "", "OU", "LX", 
  /******** Hershey Glyphs 1200 to 1299 ********/ 
  "LYQKOLNONTOWQXTXVWWTWOVLTKQK RQKPLOOOTPWQX RTXUWVTVOULTK", "LYPNSKSX RRLRX ROXVX", "LYOMONNNNMOLQKTKVLWNVPTQQROSNUNX RTKULVNUPTQ RNWOVPVSWVWWV RPVSXVXWVWU", "LYOMONNNNMOLQKTKVLWNVPTQ RTKULVNUPTQ RRQTQVRWTWUVWTXQXOWNVNUOUOV RTQURVTVUUWTX", "LYSMSX RTKTX RTKMTXT RQXVX", "LYOKNQ ROKVK ROLSLVK RNQOPQOTOVPWRWUVWTXQXOWNVNUOUOV RTOUPVRVUUWTX", "LYVMVNWNWMVLTKRKPLOMNPNUOWQXTXVWWUWSVQTPQPNR RRKPMOPOUPWQX RTXUWVUVSUQTP", "LYNKNO RVMRTPX RWKTQQX RNMPKRKUM RNMPLRLUMVM", "LYQKOLNNOPQQTQVPWNVLTKQK RQKPLONPPQQ RTQUPVNULTK RQQORNTNUOWQXTXVWWUWTVRTQ RQQPROTOUPWQX RTXUWVUVTURTQ", "LYOVOUNUNVOWQXSXUWVVWSWNVLTKQKOLNNNPORQSTSWQ RSXUVVSVNULTK RQKPLONOPPRQS", "NVRVQWRXSWRV", "NVSWRXQWRVSWSYQ[", "NVROQPRQSPRO RRVQWRXSWRV", "NVROQPRQSPRO RSWRXQWRVSWSYQ[", "NVRKQLRSSLRK RRLRO RRVQWRXSWRV", "LYNNONOONONNOLQKTKVLWNWOVQSRRSRTST RTKVMVPUQSR RRWRXSXSWRW", "OVRKRP RSKRP", "LXOKOP RPKOP RUKUP RVKUP", "MWQKPLPNQOSOTNTLSKQK", "MWRJRP ROKUO RUKOO", "KZXHM\\", "MWUHSJQMPPPTQWSZU\\ RSJRLQPQTRXSZ", "MWOHQJSMTPTTSWQZO\\ RQJRLSPSTRXQZ", "MWPHP\\ RQHQ\\ RPHUH RP\\U\\", "MWSHS\\ RTHT\\ ROHTH RO\\T\\", "LWSHRIQKQMRORPPRRTRUQWQYR[S\\ RRIQM RQKRO RRUQY RQWR[", "MXQHRISKSMRORPTRRTRUSWSYR[Q\\ RRISM RSKRO RRUSY RSWR[", "MWTHPRT\\", "MWPHTRP\\", "OURHR\\", "MWPHP\\ RTHT\\", "I[LRXR", "I[RLRX RLRXR", "JZRMRX RMRWR RMXWX", "JZRMRX RMMWM RMRWR", "JZMMWW RWMMW", "NVRQQRRSSRRQ", "I[RLQMRNSMRL RLRXR RRVQWRXSWRV", "I[LPXP RLTXT", "I[WLMX RLPXP RLTXT", "I[LNXN RLRXR RLVXV", "JZWLMRWX", "JZMLWRMX", "JZWKMOWS RMTWT RMXWX", "JZMKWOMS RMTWT RMXWX", "H[YUWUUTTSRPQOONNNLOKQKRLTNUOUQTRSTPUOWNYN", "JZLTLRMPOPUSWSXR RLRMQOQUTWTXRXP", "JZMSRPWS RMSRQWS", "NVSKPO RSKTLPO", "NVQKTO RQKPLTO", "LXNKOMQNSNUMVK RNKONQOSOUNVK", "NVSLRMQLRKSLSNQP", "NVSKQMQORPSORNQO", "NVQLRMSLRKQLQNSP", "NVQKSMSORPQORNSO", "", "JZWMQMONNOMQMSNUOVQWWW", "JZMMMSNUOVQWSWUVVUWSWM", "JZMMSMUNVOWQWSVUUVSWMW", "JZMWMQNOONQMSMUNVOWQWW", "JZWMQMONNOMQMSNUOVQWWW RMRUR", "I[TOUPXRUTTU RUPWRUT RLRWR", "MWRMRX ROPPORLTOUP RPORMTO", "I[POOPLROTPU ROPMROT RMRXR", "MWRLRW ROTPURXTUUT RPURWTU", "KYVSUPSOQOOPNQMSMUNWPXRXTWUVVTWQWNVLTKQKPLQLRK RQOOQNSNVPX RRXTVUTVQVNULTK", "JZLKRX RMKRV RXKRX RLKXK RNLWL", "G[IOLORW RKORX R[FRX", "I[XIXJYJYIXHVHTJSLROQUPYO[ RUITKSORUQXPZN\\L\\K[KZLZL[", "I[XIXJYJYIXHVHTJSLROQUPYO[ RUITKSORUQXPZN\\L\\K[KZLZL[ RQNOONQNSOUQVSVUUVSVQUOSNQN", "H\\ZRYTWUVUTTSSQPPONNMNKOJQJRKTMUNUPTQSSPTOVNWNYOZQZR", "JZXKLX ROKPLPNOOMOLNLLMKOKSLVLXK RUTTUTWUXWXXWXUWTUT", "J[YPXPXQYQYPXOWOVPUTTVSWQXOXMWLVLTMSORRPSNSLRKPKOLONPQUWWXXXYW ROXMVMTOR RONPPVWWX", "J[UPSOQOPQPRQTSTUS RUOUSVTXTYRYQXNVLSKRKOLMNLQLRMUOWRXSXVW", "KZQHQ\\ RTHT\\ RWLVLVMWMWLUKPKNLNNOPVSWT RNNOOVRWTWVVWTXQXOWNVNUOUOVNV", "KYPKP[ RTKT[ RMQWQ RMUWU", "LXTLSLSMTMTLSKQKPLPNQPTRUS RPNQOTQUSUUSW RQPOROTPVSXTY ROTPUSWTYT[S\\Q\\P[PZQZQ[P[", "LXRKQLRMSLRK RRMRQ RRQQSRVSSRQ RRVR\\ RPOONNOOPPOTOUNVOUPTO", "LXRMSLRKQLRMRQQRSURV RRQSRQURVRZQ[R\\S[RZ RPOONNOOPPOTOUNVOUPTO RPXOWNXOYPXTXUWVXUYTX", "LYVKVX RNKVK RQQVQ RNXVX", "", "H\\QKNLLNKQKSLVNXQYSYVXXVYSYQXNVLSKQK RRQQRRSSRRQ", "LYQKPLPMQN RTKULUMTN RRNPOOQORPTRUSUUTVRVQUOSNRN RRURY RSUSY ROWVW", "LYRKPLONOOPQRRSRUQVOVNULSKRK RRRRX RSRSX ROUVU", "H\\QKNLLNKQKSLVNXQYSYVXXVYSYQXNVLSKQK RRKRY RKRYR", "JYRRPQOQMRLTLUMWOXPXRWSUSTRR RWMRR RRMWMWR RRMVNWR", "JZLLMKOKQLRNRPQRPSNT ROKPLQNQQPS RVKUX RWKTX RNTXT", "JYNKNU ROKNR RNROPQOSOUPVQVTTVTXUYVYWX RSOUQUTTV RLKOK", "LYONRKRQ RVNSKSQ RRQPROTOUPWRXSXUWVUVTURSQ RRTRUSUSTRT", "JZRKRY RMKMPNRPSTSVRWPWK RLMMKNM RQMRKSM RVMWKXM ROVUV", "JYNKNX ROKOX RLKSKVLWNWOVQSROR RSKULVNVOUQSR RLXVXVUUX", "LYWKTKQLONNQNSOVQXTYWY RWKTLRNQQQSRVTXWY", "JZRRPQOQMRLTLUMWOXPXRWSUSTRR RSLQQ RWMRR RXQSS", "KYPMTW RTMPW RMPWT RWPMT", "J[OUMULVLXMYOYPXPVNTMRMONMOLQKTKVLWMXOXRWTUVUXVYXYYXYVXUVU RNMPLULWM", "J[OOMOLNLLMKOKPLPNNPMRMUNWOXQYTYVXWWXUXRWPUNULVKXKYLYNXOVO RNWPXUXWW", "", "", "", "", 
  /******** Hershey Glyphs 1300 to 1399 ********/ 
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 1400 to 1499 ********/ 
  "", "F^KHK\\ RLHL\\ RXHX\\ RYHY\\ RHH\\H RH\\O\\ RU\\\\\\", "H]KHRQJ\\ RJHQQ RJHYHZMXH RK[X[ RJ\\Y\\ZWX\\", "KYVBTDRGPKOPOTPYR]T`Vb RTDRHQKPPPTQYR\\T`", "KYNBPDRGTKUPUTTYR]P`Nb RPDRHSKTPTTSYR\\P`", "KYOBOb RPBPb ROBVB RObVb", "KYTBTb RUBUb RNBUB RNbUb", "JYTBQEPHPJQMSOSPORSTSUQWPZP\\Q_Tb RRDQGQKRN RRVQYQ]R`", "KZPBSETHTJSMQOQPURQTQUSWTZT\\S_Pb RRDSGSKRN RRVSYS]R`", "KYU@RCPFOIOLPOSVTYT\\S_Ra RRCQEPHPKQNTUUXU[T^RaOd", "KYO@RCTFUIULTOQVPYP\\Q_Ra RRCSETHTKSNPUOXO[P^RaUd", "AXCRGRR` RGSRa RFSRb RX:Rb", "F^[CZD[E\\D\\C[BYBWCUETGSJRNPZO^N` RVDUFTJRVQZP]O_MaKbIbHaH`I_J`Ia", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 1500 to 1599 ********/ 
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 1600 to 1699 ********/ 
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 1700 to 1799 ********/ 
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 1800 to 1899 ********/ 
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 1900 to 1999 ********/ 
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 2000 to 2099 ********/ 
  "", "H\\RFK[ RRFY[ RRIX[ RMUVU RI[O[ RU[[[", "G]LFL[ RMFM[ RIFUFXGYHZJZLYNXOUP RUFWGXHYJYLXNWOUP RMPUPXQYRZTZWYYXZU[I[ RUPWQXRYTYWXYWZU[", "G\\XIYLYFXIVGSFQFNGLIKKJNJSKVLXNZQ[S[VZXXYV RQFOGMILKKNKSLVMXOZQ[", "G]LFL[ RMFM[ RIFSFVGXIYKZNZSYVXXVZS[I[ RSFUGWIXKYNYSXVWXUZS[", "G\\LFL[ RMFM[ RSLST RIFYFYLXF RMPSP RI[Y[YUX[", "G[LFL[ RMFM[ RSLST RIFYFYLXF RMPSP RI[P[", "G^XIYLYFXIVGSFQFNGLIKKJNJSKVLXNZQ[S[VZXX RQFOGMILKKNKSLVMXOZQ[ RXSX[ RYSY[ RUS\\S", "F^KFK[ RLFL[ RXFX[ RYFY[ RHFOF RUF\\F RLPXP RH[O[ RU[\\[", "MXRFR[ RSFS[ ROFVF RO[V[", "KZUFUWTZR[P[NZMXMVNUOVNW RTFTWSZR[ RQFXF", "F\\KFK[ RLFL[ RYFLS RQOY[ RPOX[ RHFOF RUF[F RH[O[ RU[[[", "I[NFN[ ROFO[ RKFRF RK[Z[ZUY[", "F_KFK[ RLFRX RKFR[ RYFR[ RYFY[ RZFZ[ RHFLF RYF]F RH[N[ RV[][", "G^LFL[ RMFYY RMHY[ RYFY[ RIFMF RVF\\F RI[O[", "G]QFNGLIKKJOJRKVLXNZQ[S[VZXXYVZRZOYKXIVGSFQF RQFOGMILKKOKRLVMXOZQ[ RS[UZWXXVYRYOXKWIUGSF", "G]LFL[ RMFM[ RIFUFXGYHZJZMYOXPUQMQ RUFWGXHYJYMXOWPUQ RI[P[", "G]QFNGLIKKJOJRKVLXNZQ[S[VZXXYVZRZOYKXIVGSFQF RQFOGMILKKOKRLVMXOZQ[ RS[UZWXXVYRYOXKWIUGSF RNYNXOVQURUTVUXV_W`Y`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`NFH[ RNFO[ ROFPY R\\FO[ R\\FV[ R]FW[ RKFOF R\\F`F RE[K[ RS[Z[", "F_OFI[ ROFVX ROIV[ R\\FV[ RLFOF RYF_F RF[L[", "G]SFPGNILLKOJSJVKYLZN[Q[TZVXXUYRZNZKYHXGVFSF RSFQGOIMLLOKSKVLYN[ RQ[SZUXWUXRYNYKXHVF", "F]OFI[ RPFJ[ RLFXF[G\\I\\K[NYPUQMQ RXFZG[I[KZNXPUQ RF[M[", "G]SFPGNILLKOJSJVKYLZN[Q[TZVXXUYRZNZKYHXGVFSF RSFQGOIMLLOKSKVLYN[ RQ[SZUXWUXRYNYKXHVF RLYLXMVOUPURVSXS_T`V`W^W] RSXT^U_V_W^", "F^OFI[ RPFJ[ RLFWFZG[I[KZNYOVPMP RWFYGZIZKYNXOVP RRPTQURVZW[Y[ZYZX RURWYXZYZZY RF[M[", "G^ZH[H\\F[L[JZHYGVFRFOGMIMKNMONVRXT RMKOMVQWRXTXWWYVZS[O[LZKYJWJUI[JYKY", "H]UFO[ RVFP[ ROFLLNF]F\\L\\F RL[S[", "F_NFKQJUJXKZN[R[UZWXXU\\F ROFLQKUKXLZN[ RKFRF RYF_F", "H\\NFO[ ROFPY R\\FO[ RLFRF RXF^F", "E_MFK[ RNFLY RUFK[ RUFS[ RVFTY R]FS[ RJFQF RZF`F", "G]NFU[ ROFV[ R\\FH[ RLFRF RXF^F RF[L[ RR[X[", "H]NFRPO[ ROFSPP[ R]FSP RLFRF RYF_F RL[S[", "G][FH[ R\\FI[ ROFLLNF\\F RH[V[XUU[", "H\\KILKXWYYY[ RLLXX RKIKKLMXYY[ RPPLTKVKXLZK[ RKVMZ RLTLVMXMZK[ RSSXN RVIVLWNYNYLWKVI RVIWLYN", "H\\QIK[ RSIY[ RRIX[ RMUVU RI[O[ RU[[[ RQBOCNENGOIQJSJUIVGVEUCSBQB", "", "", "", "", "", "G]IB[b", "F^RJIZ RRJ[Z", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 2100 to 2199 ********/ 
  "", "I]NONPMPMONNPMTMVNWOXQXXYZZ[ RWOWXXZZ[[[ RWQVRPSMTLVLXMZP[S[UZWX RPSNTMVMXNZP[", "G\\LFL[ RMFM[ RMPONQMSMVNXPYSYUXXVZS[Q[OZMX RSMUNWPXSXUWXUZS[ RIFMF", "H[WPVQWRXQXPVNTMQMNNLPKSKULXNZQ[S[VZXX RQMONMPLSLUMXOZQ[", "H]WFW[ RXFX[ RWPUNSMQMNNLPKSKULXNZQ[S[UZWX RQMONMPLSLUMXOZQ[ RTFXF RW[[[", "H[LSXSXQWOVNTMQMNNLPKSKULXNZQ[S[VZXX RWSWPVN RQMONMPLSLUMXOZQ[", "KXUGTHUIVHVGUFSFQGPIP[ RSFRGQIQ[ RMMUM RM[T[", "I\\QMONNOMQMSNUOVQWSWUVVUWSWQVOUNSMQM RONNPNTOV RUVVTVPUN RVOWNYMYNWN RNUMVLXLYM[P\\U\\X]Y^ RLYMZP[U[X\\Y^Y_XaUbObLaK_K^L\\O[", "G]LFL[ RMFM[ RMPONRMTMWNXPX[ RTMVNWPW[ RIFMF RI[P[ RT[[[", "MXRFQGRHSGRF RRMR[ RSMS[ ROMSM RO[V[", "MXSFRGSHTGSF RTMT_SaQbObNaN`O_P`Oa RSMS_RaQb RPMTM", "G\\LFL[ RMFM[ RWMMW RRSX[ RQSW[ RIFMF RTMZM RI[P[ RT[Z[", "MXRFR[ RSFS[ ROFSF RO[V[", "BcGMG[ RHMH[ RHPJNMMOMRNSPS[ ROMQNRPR[ RSPUNXMZM]N^P^[ RZM\\N]P][ RDMHM RD[K[ RO[V[ RZ[a[", "G]LML[ RMMM[ RMPONRMTMWNXPX[ RTMVNWPW[ RIMMM RI[P[ RT[[[", "H\\QMNNLPKSKULXNZQ[S[VZXXYUYSXPVNSMQM RQMONMPLSLUMXOZQ[ RS[UZWXXUXSWPUNSM", "G\\LMLb RMMMb RMPONQMSMVNXPYSYUXXVZS[Q[OZMX RSMUNWPXSXUWXUZS[ RIMMM RIbPb", "H\\WMWb RXMXb RWPUNSMQMNNLPKSKULXNZQ[S[UZWX RQMONMPLSLUMXOZQ[ RTb[b", "IZNMN[ ROMO[ ROSPPRNTMWMXNXOWPVOWN RKMOM RK[R[", "J[WOXMXQWOVNTMPMNNMOMQNRPSUUWVXW RMPNQPRUTWUXVXYWZU[Q[OZNYMWM[NY", "KZPFPWQZS[U[WZXX RQFQWRZS[ RMMUM", "G]LMLXMZP[R[UZWX RMMMXNZP[ RWMW[ RXMX[ RIMMM RTMXM RW[[[", "I[LMR[ RMMRY RXMR[ RJMPM RTMZM", "F^JMN[ RKMNX RRMN[ RRMV[ RSMVX RZMV[ RGMNM RWM]M", "H\\LMW[ RMMX[ RXML[ RJMPM RTMZM RJ[P[ RT[Z[", "H[LMR[ RMMRY RXMR[P_NaLbKbJaK`La RJMPM RTMZM", "I[WML[ RXMM[ RMMLQLMXM RL[X[XWW[", "G^QMNNLPKRJUJXKZN[P[RZUWWTYPZM RQMONMPLRKUKXLZN[ RQMSMUNVPXXYZZ[ RSMTNUPWXXZZ[[[", "G\\TFQGOIMMLPKTJZIb RTFRGPINMMPLTKZJb RTFVFXGYHYKXMWNTOPO RVFXHXKWMVNTO RPOTPVRWTWWVYUZR[P[NZMYLV RPOSPURVTVWUYTZR[", "H\\IPKNMMOMQNROSRSVRZOb RJOLNPNRO RZMYPXRSYP^Nb RYMXPWRSY", "I\\VNTMRMONMQLTLWMYNZP[R[UZWWXTXQWOSJRHRFSEUEWFYH RRMPNNQMTMXNZ RR[TZVWWTWPVNTKSISGTFVFYH", "I[XPVNTMPMNNNPPRSS RPMONOPQRSS RSSNTLVLXMZP[S[UZWX RSSOTMVMXNZP[", "I[TFRGQHQIRJUKZKZJWKSMPOMRLULWMYP[S]T_TaSbQbPa RULQONRMUMWNYP[", "G]HQIOKMNMONOPNTL[ RMMNNNPMTK[ RNTPPRNTMVMXNYOYRXWUb RVMXOXRWWTb", "F]GQHOJMMMNNNPMUMXNZO[ RLMMNMPLULXMZO[Q[SZUXWUXRYMYIXGVFTFRHRJSMUPWRZT RSZUWVUWRXMXIWGVF", "LXRMPTOXOZP[S[UYVW RSMQTPXPZQ[", "H\\NMJ[ ROMK[ RXMYNZNYMWMUNQROSMS ROSQTSZT[ ROSPTRZS[U[WZYW", "H\\KFMFOGPHQJWXXZY[ RMFOHPJVXWZY[Z[ RRMJ[ RRMK[", "F]MMGb RNMHb RMPLVLYN[P[RZTXVU RXMUXUZV[Y[[Y\\W RYMVXVZW[", "H\\NML[ ROMNSMXL[ RYMXQVU RZMYPXRVUTWQYOZL[ RKMOM", "IZTFRGQHQIRJUKXK RUKQLOMNONQPSSTVT RUKRLPMOOOQQSST RSTOUMVLXLZN\\S^T_TaRbPb RSTPUNVMXMZO\\S^", "I[RMONMQLTLWMYNZP[R[UZWWXTXQWOVNTMRM RRMPNNQMTMXNZ RR[TZVWWTWPVN", "G]PNL[ RPNM[ RVNV[ RVNW[ RIPKNNM[M RIPKONN[N", "H[LVMYNZP[R[UZWWXTXQWOVNTMRMONMQLTHb RR[TZVWWTWPVN RRMPNNQMTIb", "H][MQMNNLQKTKWLYMZO[Q[TZVWWTWQVOUNSM RQMONMQLTLXMZ RQ[SZUWVTVPUN RUN[N", "H\\SNP[ RSNQ[ RJPLNOMZM RJPLOONZN", "H\\IQJOLMOMPNPPNVNYP[ RNMONOPMVMYNZP[Q[TZVXXUYRYOXMWNXOYR RXUYO", "G]ONMOKQJTJWKYLZN[Q[TZWXYUZRZOXMVMTORSPXMb RJWLYNZQZTYWWYU RZOXNVNTPRSPYNb", "I[KMMMONPPU_VaWb RMMNNOPT_UaWbYb RZMYOWRM]K`Jb", "F]UFOb RVFNb RGQHOJMMMNNNPMUMXOZRZTYWVYS RLMMNMPLULXMZO[R[TZVXXUYS[M", "F]JQLOONNMLNJQITIWJZK[M[OZQWRT RIWJYKZMZOYQW RQTQWRZS[U[WZYWZTZQYNXMWNYOZQ RQWRYSZUZWYYW", "H]XMVTUXUZV[Y[[Y\\W RYMWTVXVZW[ RVTVQUNSMQMNNLQKTKWLYMZO[Q[SZUWVT RQMONMQLTLXMZ", "H[PFLSLVMYNZ RQFMS RMSNPPNRMTMVNWOXQXTWWUZR[P[NZMWMS RVNWPWTVWTZR[ RMFQF", "I[WPWQXQXPWNUMRMONMQLTLWMYNZP[R[UZWW RRMPNNQMTMXNZ", "H]ZFVTUXUZV[Y[[Y\\W R[FWTVXVZW[ RVTVQUNSMQMNNLQKTKWLYMZO[Q[SZUWVT RQMONMQLTLXMZ RWF[F", "I[MVQUTTWRXPWNUMRMONMQLTLWMYNZP[R[UZWX RRMPNNQMTMXNZ", "KZZGYHZI[H[GZFXFVGUHTJSMP[O_Na RXFVHUJTNRWQ[P^O`NaLbJbIaI`J_K`Ja ROMYM", "H\\YMU[T^RaObLbJaI`I_J^K_J` RXMT[S^QaOb RVTVQUNSMQMNNLQKTKWLYMZO[Q[SZUWVT RQMONMQLTLXMZ", "H]PFJ[ RQFK[ RMTOPQNSMUMWNXOXQVWVZW[ RUMWOWQUWUZV[Y[[Y\\W RMFQF", "LYUFTGUHVGUF RMQNOPMSMTNTQRWRZS[ RRMSNSQQWQZR[U[WYXW", "LYVFUGVHWGVF RNQOOQMTMUNUQR[Q^P`OaMbKbJaJ`K_L`Ka RSMTNTQQ[P^O`Mb", "H\\PFJ[ RQFK[ RXNWOXPYOYNXMWMUNQROSMS ROSQTSZT[ ROSPTRZS[U[WZYW RMFQF", "MYUFQTPXPZQ[T[VYWW RVFRTQXQZR[ RRFVF", "AbBQCOEMHMINIPHTF[ RGMHNHPGTE[ RHTJPLNNMPMRNSOSQP[ RPMRORQO[ RRTTPVNXMZM\\N]O]Q[W[Z\\[ RZM\\O\\QZWZZ[[^[`YaW", "F]GQHOJMMMNNNPMTK[ RLMMNMPLTJ[ RMTOPQNSMUMWNXOXQVWVZW[ RUMWOWQUWUZV[Y[[Y\\W", "I[RMONMQLTLWMYNZP[R[UZWWXTXQWOVNTMRM RRMPNNQMTMXNZ RR[TZVWWTWPVN", "G\\HQIOKMNMONOPNTJb RMMNNNPMTIb RNTOQQNSMUMWNXOYQYTXWVZS[Q[OZNWNT RWNXPXTWWUZS[ RFbMb", "H\\XMRb RYMSb RVTVQUNSMQMNNLQKTKWLYMZO[Q[SZUWVT RQMONMQLTLXMZ RObVb", "IZJQKOMMPMQNQPPTN[ ROMPNPPOTM[ RPTRPTNVMXMYNYOXPWOXN", "J[XOXPYPYOXNUMRMONNONQORVVWW RNPOQVUWVWYVZS[P[MZLYLXMXMY", "KYTFPTOXOZP[S[UYVW RUFQTPXPZQ[ RNMWM", "F]GQHOJMMMNNNQLWLYN[ RLMMNMQKWKYLZN[P[RZTXVT RXMVTUXUZV[Y[[Y\\W RYMWTVXVZW[", "H\\IQJOLMOMPNPQNWNYP[ RNMONOQMWMYNZP[Q[TZVXXUYQYMXMYO", "C`DQEOGMJMKNKQIWIYK[ RIMJNJQHWHYIZK[M[OZQXRV RTMRVRYSZU[W[YZ[X\\V]R]M\\M]O RUMSVSYU[", "H\\KQMNOMRMSOSR RQMRORRQVPXNZL[K[JZJYKXLYKZ RQVQYR[U[WZYW RYNXOYPZOZNYMXMVNTPSRRVRYS[", "G\\HQIOKMNMONOQMWMYO[ RMMNNNQLWLYMZO[Q[SZUXWT RZMV[U^SaPbMbKaJ`J_K^L_K` RYMU[T^RaPb", "H\\YMXOVQNWLYK[ RLQMOOMRMVO RMOONRNVOXO RLYNYRZUZWY RNYR[U[WYXW", "G^VGUHVIWHWGUFRFOGMILLL[ RRFPGNIMLM[ R\\G[H\\I]H]G\\FZFXGWIW[ RZFYGXIX[ RIM[M RI[P[ RT[[[", "G]WGVHWIXHWGUFRFOGMILLL[ RRFPGNIMLM[ RWMW[ RXMX[ RIMXM RI[P[ RT[[[", "G]VGUHVIWHWGUF RXFRFOGMILLL[ RRFPGNIMLM[ RWHW[ RXFX[ RIMWM RI[P[ RT[[[", "BcRGQHRISHRGPFMFJGHIGLG[ RMFKGIIHLH[ R]G\\H]I^H]G[FXFUGSIRLR[ RXFVGTISLS[ R]M][ R^M^[ RDM^M RD[K[ RO[V[ RZ[a[", "BcRGQHRISHRGPFMFJGHIGLG[ RMFKGIIHLH[ R\\G[H\\I]H]G[F R^FXFUGSIRLR[ RXFVGTISLS[ R]H][ R^F^[ RDM]M RD[K[ RO[V[ RZ[a[", "MXRMR[ RSMS[ ROMSM RO[V[", "", "IZWNUMRMONMPLSLVMYNZQ[T[VZ RRMPNNPMSMVNYOZQ[ RMTUT", "I\\TFQGOJNLMOLTLXMZO[Q[TZVWWUXRYMYIXGVFTF RTFRGPJOLNOMTMXNZO[ RQ[SZUWVUWRXMXIWGVF RNPWP", "G]UFOb RVFNb RQMMNKPJSJVKXMZP[S[WZYXZUZRYPWNTMQM RQMNNLPKSKVLXNZP[ RS[VZXXYUYRXPVNTM", "I[TMVNXPXOWNTMQMNNMOLQLSMUOWSZ RQMONNOMQMSNUSZT\\T^S_Q_", "", "", "G]LMKNJPJRKUOYP[ RJRKTOXP[P]O`MbLbKaJ_J\\KXMTOQRNTMVMYNZPZTYXWZU[T[SZSXTWUXTY RVMXNYPYTXXWZ", "E_YGXHYIZHYGWFTFQGOINKMNLRJ[I_Ha RTFRGPIOKNNLWK[J^I`HaFbDbCaC`D_E`Da R_G^H_I`H`G_F]F[GZHYJXMU[T_Sa R]F[HZJYNWWV[U^T`SaQbObNaN`O_P`Oa RIM^M", "F^[GZH[I\\H[GXFUFRGPIOKNNMRK[J_Ia RUFSGQIPKONMWL[K^J`IaGbEbDaD`E_F`Ea RYMWTVXVZW[Z[\\Y]W RZMXTWXWZX[ RJMZM", "F^YGXHYIZHZGXF R\\FUFRGPIOKNNMRK[J_Ia RUFSGQIPKONMWL[K^J`IaGbEbDaD`E_F`Ea R[FWTVXVZW[Z[\\Y]W R\\FXTWXWZX[ RJMYM", "@cTGSHTIUHTGRFOFLGJIIKHNGRE[D_Ca ROFMGKIJKINGWF[E^D`CaAb?b>a>`?_@`?a R`G_H`IaH`G]FZFWGUITKSNRRP[O_Na RZFXGVIUKTNRWQ[P^O`NaLbJbIaI`J_K`Ja R^M\\T[X[Z\\[_[aYbW R_M]T\\X\\Z][ RDM_M", "@cTGSHTIUHTGRFOFLGJIIKHNGRE[D_Ca ROFMGKIJKINGWF[E^D`CaAb?b>a>`?_@`?a R^G]H^I_H_G]F RaFZFWGUITKSNRRP[O_Na RZFXGVIUKTNRWQ[P^O`NaLbJbIaI`J_K`Ja R`F\\T[X[Z\\[_[aYbW RaF]T\\X\\Z][ RDM^M", "LYMQNOPMSMTNTQRWRZS[ RRMSNSQQWQZR[U[WYXW", "", "NV", "JZ", 
  /******** Hershey Glyphs 2200 to 2299 ********/ 
  "H\\QFNGLJKOKRLWNZQ[S[VZXWYRYOXJVGSFQF RQFOGNHMJLOLRMWNYOZQ[ RS[UZVYWWXRXOWJVHUGSF", "H\\NJPISFS[ RRGR[ RN[W[", "H\\LJMKLLKKKJLHMGPFTFWGXHYJYLXNUPPRNSLUKXK[ RTFVGWHXJXLWNTPPR RKYLXNXSZVZXYYX RNXS[W[XZYXYV", "H\\LJMKLLKKKJLHMGPFTFWGXIXLWNTOQO RTFVGWIWLVNTO RTOVPXRYTYWXYWZT[P[MZLYKWKVLUMVLW RWQXTXWWYVZT[", "H\\THT[ RUFU[ RUFJUZU RQ[X[", "H\\MFKP RKPMNPMSMVNXPYSYUXXVZS[P[MZLYKWKVLUMVLW RSMUNWPXSXUWXUZS[ RMFWF RMGRGWF", "H\\WIVJWKXJXIWGUFRFOGMILKKOKULXNZQ[S[VZXXYUYTXQVOSNRNOOMQLT RRFPGNIMKLOLUMXOZQ[ RS[UZWXXUXTWQUOSN", "H\\KFKL RKJLHNFPFUIWIXHYF RLHNGPGUI RYFYIXLTQSSRVR[ RXLSQRSQVQ[", "H\\PFMGLILLMNPOTOWNXLXIWGTFPF RPFNGMIMLNNPO RTOVNWLWIVGTF RPOMPLQKSKWLYMZP[T[WZXYYWYSXQWPTO RPONPMQLSLWMYNZP[ RT[VZWYXWXSWQVPTO", "H\\XMWPURRSQSNRLPKMKLLINGQFSFVGXIYLYRXVWXUZR[O[MZLXLWMVNWMX RQSORMPLMLLMIOGQF RSFUGWIXLXRWVVXTZR[", "MWRYQZR[SZRY", "MWR[QZRYSZS\\R^Q_", "MWRMQNROSNRM RRYQZR[SZRY", "MWRMQNROSNRM RR[QZRYSZS\\R^Q_", "MWRFQHRTSHRF RRHRN RRYQZR[SZRY", "I[MJNKMLLKLJMHNGPFSFVGWHXJXLWNVORQRT RSFUGVHWJWLVNTP RRYQZR[SZRY", "NVRFQM RSFQM", "JZNFMM ROFMM RVFUM RWFUM", "KYQFOGNINKOMQNSNUMVKVIUGSFQF", "JZRFRR RMIWO RWIMO", "G][BIb", "KYVBTDRGPKOPOTPYR]T`Vb RTDRHQKPPPTQYR\\T`", "KYNBPDRGTKUPUTTYR]P`Nb RPDRHSKTPTTSYR\\P`", "KYOBOb RPBPb ROBVB RObVb", "KYTBTb RUBUb RNBUB RNbUb", "JYTBQEPHPJQMSOSPORSTSUQWPZP\\Q_Tb RRDQGQKRN RRVQYQ]R`", "KZPBSETHTJSMQOQPURQTQUSWTZT\\S_Pb RRDSGSKRN RRVSYS]R`", "KYUBNRUb", "KYOBVROb", "NVRBRb", "KYOBOb RUBUb", "E_IR[R", "E_RIR[ RIR[R", "F^RJR[ RJRZR RJ[Z[", "F^RJR[ RJJZJ RJRZR", "G]KKYY RYKKY", "MWQQQSSSSQQQ RRQRS RQRSR", "E_RIQJRKSJRI RIR[R RRYQZR[SZRY", "E_IO[O RIU[U", "E_YIK[ RIO[O RIU[U", "E_IM[M RIR[R RIW[W", "F^ZIJRZ[", "F^JIZRJ[", "F^ZFJMZT RJVZV RJ[Z[", "F^JFZMJT RJVZV RJ[Z[", "F_[WYWWVUTRPQOONMNKOJQJSKUMVOVQURTUPWNYM[M", "F^IUISJPLONOPPTSVTXTZS[Q RISJQLPNPPQTTVUXUZT[Q[O", "G]JTROZT RJTRPZT", "LXTFOL RTFUGOL", "LXPFUL RPFOGUL", "H\\KFLHNJQKSKVJXHYF RKFLINKQLSLVKXIYF", "MWRHQGRFSGSIRKQL", "MWSFRGQIQKRLSKRJ", "MWRHSGRFQGQIRKSL", "MWQFRGSISKRLQKRJ", "E[HMLMRY RKMR[ R[BR[", "F^ZJSJOKMLKNJQJSKVMXOYSZZZ", "F^JJJQKULWNYQZSZVYXWYUZQZJ", "F^JJQJUKWLYNZQZSYVWXUYQZJZ", "F^JZJSKOLMNKQJSJVKXMYOZSZZ", "F^ZJSJOKMLKNJQJSKVMXOYSZZZ RJRVR", "E_XP[RXT RUMZRUW RIRZR", "JZPLRITL RMORJWO RRJR[", "E_LPIRLT ROMJROW RJR[R", "JZPXR[TX RMURZWU RRIRZ", "I\\XRWOVNTMRMONMQLTLWMYNZP[R[UZWXXUYPYKXHWGUFRFPGOHOIPIPH RRMPNNQMTMXNZ RR[TZVXWUXPXKWHUF", "H\\JFR[ RKFRY RZFR[ RJFZF RKGYG", "AbDMIMRY RHNR[ Rb:R[", "F^[CZD[E\\D\\C[BYBWCUETGSJRNPZO^N` RVDUFTJRVQZP]O_MaKbIbHaH`I_J`Ia", "F^[CZD[E\\D\\C[BYBWCUETGSJRNPZO^N` RVDUFTJRVQZP]O_MaKbIbHaH`I_J`Ia RQKNLLNKQKSLVNXQYSYVXXVYSYQXNVLSKQK", "F_\\S[UYVWVUUTTQPPONNLNJOIQISJULVNVPUQTTPUOWNYN[O\\Q\\S", "F^[FI[ RNFPHPJOLMMKMIKIIJGLFNFPGSHVHYG[F RWTUUTWTYV[X[ZZ[X[VYTWT", "F_[NZO[P\\O\\N[MZMYNXPVUTXRZP[M[JZIXIUJSPORMSKSIRGPFNGMIMKNNPQUXWZZ[[[\\Z\\Y RM[KZJXJUKSMQ RMKNMVXXZZ[", "E`WNVLTKQKOLNMMPMSNUPVSVUUVS RQKOMNPNSOUPV RWKVSVUXVZV\\T]Q]O\\L[JYHWGTFQFNGLHJJILHOHRIUJWLYNZQ[T[WZYYZX RXKWSWUXV", "H\\PBP_ RTBT_ RXIWJXKYJYIWGTFPFMGKIKKLMMNOOUQWRYT RKKMMONUPWQXRYTYXWZT[P[MZKXKWLVMWLX", "G]OFOb RUFUb RJQZQ RJWZW", "JZUITJUKVJVIUGSFQFOGNINKOMQOVR ROMTPVRWTWVVXTZ RPNNPMRMTNVPXU[ RNVSYU[V]V_UaSbQbOaN_N^O]P^O_", "JZRFQHRJSHRF RRFRb RRQQTRbSTRQ RLMNNPMNLLM RLMXM RTMVNXMVLTM", "JZRFQHRJSHRF RRFRT RRPQRSVRXQVSRRP RRTRb RR^Q`RbS`R^ RLMNNPMNLLM RLMXM RTMVNXMVLTM RL[N\\P[NZL[ RL[X[ RT[V\\X[VZT[", "I\\XFX[ RKFXF RPPXP RK[X[", "", "E`QFNGKIILHOHRIUKXNZQ[T[WZZX\\U]R]O\\LZIWGTFQF RROQPQQRRSRTQTPSORO RRPRQSQSPRP", "J[PFNGOIQJ RPFOGOI RUFWGVITJ RUFVGVI RQJOKNLMNMQNSOTQUTUVTWSXQXNWLVKTJQJ RRUR[ RSUS[ RNXWX", "I\\RFOGMILLLMMPORRSSSVRXPYMYLXIVGSFRF RRSR[ RSSS[ RNWWW", "D`PFMGJIHLGOGSHVJYM[P\\T\\W[ZY\\V]S]O\\LZIWGTFPF RRFR\\ RGQ]Q", "G`PMMNKPJSJTKWMYPZQZTYVWWTWSVPTNQMPM R]GWG[HUN R]G]M\\IVO R\\HVN", "F\\IIJGLFOFQGRIRLQOPQNSKU ROFPGQIQMPPNS RVFT[ RWFS[ RKUYU", "I\\MFMU RNFMQ RMQNOONQMTMWNXPXRWTUV RTMVNWPWRTXTZU[W[YY RKFNF", "I\\RNOOMQLTLUMXOZR[S[VZXXYUYTXQVOSNRN RRHNJRFRN RSHWJSFSN RRSQTQURVSVTUTTSSRS RRTRUSUSTRT", "G^QHRFR[ RTHSFS[ RJHKFKMLPNRQSRS RMHLFLNMQ R[HZFZMYPWRTSSS RXHYFYNXQ RNWWW", "G]LFL[ RMFM[ RIFUFXGYHZJZMYOXPUQMQ RUFWGXHYJYMXOWPUQ RI[Y[YVX[", "H[YGUGQHNJLMKPKSLVNYQ[U\\Y\\ RYGVHSJQMPPPSQVSYV[Y\\", "F_OQMQKRJSIUIWJYKZM[O[QZRYSWSURSQROQ RSHPQ RZJRR R\\QST", "H\\OKUY RUKOY RKOYU RYOKU", "F^NVLUKUIVHXHYI[K\\L\\N[OYOXNVKRJOJMKJMHPGTGWHYJZMZOYRVVUXUYV[X\\Y\\[[\\Y\\X[VYUXUVV RJMKKMIPHTHWIYKZM", "F^NMLNKNIMHKHJIHKGLGNHOJOKNMKQJTJVKYM[P\\T\\W[YYZVZTYQVMUKUJVHXGYG[H\\J\\K[MYNXNVM RJVKXMZP[T[WZYXZV", "I[KYYK RQLULYKXOXS RULXLXO", "I[YKKY RLQLUKYOXSX RLULXOX", "I[YYKK RSLOLKKLOLS ROLLLLO", "I[KKYY RQXUXYYXUXQ RUXXXXU",
  /******** Hershey Glyphs 2300 to 2399 ********/ 
  "", "F_JMILIJJHLGNGPHQIRKSP RIJKHMHOIPJQLRPR[ R[M\\L\\J[HYGWGUHTISKRP R\\JZHXHVIUJTLSPS[", "F^IGJKKMMOPPTPWOYMZK[G RIGJJKLMNPOTOWNYLZJ[G RPONPMQLSLVMXOZQ[S[UZWXXVXSWQVPTO RPPNQMSMVNY RVYWVWSVQTP", "F^MJMV RNKNU RVKVU RWJWV RIGKIMJPKTKWJYI[G RIYKWMVPUTUWVYW[Y", "F^[ILIJJILINJPLQNQPPQNQLPJ[J RIMJOKPMQ RQMPKOJMI RIXXXZW[U[SZQXPVPTQSSSUTWIW R[TZRYQWP RSTTVUWWX", "F]OUMTLTJUIWIXJZL[M[OZPXPWOUJPINIKJILHOGSGWHYJZLZOYRVUUWUYV[X[YZZX RMSKPJNJKKILH RSGVHXJYLYOXRVU", "G_HKKHMKMV RJILLLV RMKPHRKRU ROIQLQU RRKUHWKW[ RTIVLV[ RWKZH[J\\M\\P[SZUXWUYP[ RYIZJ[M[PZSYUWWTYP[", "F^ISMSLRKOKMLJNHQGSGVHXJYMYOXRWS[S RITOTMRLOLMMJOHQG RSGUHWJXMXOWRUT[T RKXYX RKYYY", "F_GLJIMLMX RIJLMLX RMLPISLSX ROJRMRX RSLVIYLYW[Y RUJXMXXZZ]W", "G]ZIJY RZIWJQJ RXKUKQJ RZIYLYR RXKXNYR RQRJR RPSMSJR RQRQY RPSPVQY", "F^HOJKOU RJMOWRPWPZO[M[KZIXHWHUITKTMUPVRWUWXUZ RWHVIUKUMWQXTXWWYUZ", "F^IOLLPN RKMOORLUN RQMTOWLYN RVMXO[L RIULRPT RKSOURRUT RQSTUWRYT RVSXU[R", "F^JHNJPLQOQRPUNWJY RJHMIOJQLRO RRRQUOWMXJY RZHWIUJSLRO RRRSUUWWXZY RZHVJTLSOSRTUVWZY RIP[P RIQ[Q", "", "", "", "", "NVQQQSSSSQQQ RQQSS RSQQS", "JZMPQRTTVVWYW[V]U^ RMQST RMRPSTUVWWY", "JZWKVMTOPQMR RSPMS RUFVGWIWKVNTPQRMT", "H\\SMONLPKRKTLVNWQWUVXTYRYPXNVMSM RXNSM RVMQNLP RONKR RLVQW RNWSVXT RUVYR", "H\\SMONLPKRKTLVNWQWUVXTYRYPXNVMSM RXNSM RVMQNLP RONKR RLVQW RNWSVXT RUVYR", "J[SMPNNPMRMTNVPWRWUVWTXRXPWNUMSM ROPUM RNRVN RMTWO RNUXP ROVWR RPWVT", "JZOGO^ RUFU] RMNWL RMOWM RMWWU RMXWV", "JZNFNX RVLV^ RNNVL RNOVM RNWVU RNXVV", "JZNBNW RNNQLTLVMWOWQVSSUQVNW RNNQMTMVN RUMVOVQUSSU", "E_HIHL R\\I\\L RHI\\I RHJ\\J RHK\\K RHL\\L", "JZMNMQ RWNWQ RMNWN RMOWO RMPWP RMQWQ", "JZMLWX RMLONQOTOVNWMWKUKUMTO RONTO RQOWM RVKVN RULWL RWXUVSUPUNVMWMYOYOWPU RUVPU RSUMW RNVNY RMXOX", "JZPOOMOKMKMMNNPOSOUNWL RNKNN RMLOL RMMSO RPOUN RWLWY", "A^GfHfIeIdHcGcFdFfGhIiKiNhPfQdR`RUQ;Q4R/S-U,V,X-Y/Y3X6W8U;P?JCHEFHEJDNDREVGYJ[N\\R\\V[XZZW[T[PZMYKWITHPHMIKKJNJRKUMW RGdGeHeHdGd RU;Q?LCIFGIFKENERFVGXJ[ RR\\U[WZYWZTZPYMXKVITH", "EfNSOUQVSVUUVSVQUOSNQNOONPMSMVNYP[S\\V\\Y[[Y\\W]T]P\\MZJXIUHRHOIMJKLIOHSHXI]KaMcPeTfYf]e`cba RKLJNIRIXJ\\L`NbQdUeYe]d_cba RPOTO ROPUP RNQVQ RNRVR RNSVS ROTUT RPUTU RaLaNcNcLaL RbLbN RaMcM RaVaXcXcVaV RbVbX RaWcW", "D`H@Hd RM@Md RW@Wd R\\@\\d RMMWK RMNWL RMOWM RMWWU RMXWV RMYWW", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "NVQQQSSSSQQQ RQQSS RSQQS", "JZMPQRTTVVWYW[V]U^ RMQST RMRPSTUVWWY", "JZWKVMTOPQMR RSPMS RUFVGWIWKVNTPQRMT", "H\\PMMNLOKQKSLUMVPWTWWVXUYSYQXOWNTMPM RMNLPLSMUNVPW RWVXTXQWOVNTM", "H\\SMONLPKRKTLVNWQWUVXTYRYPXNVMSM RXNSM RVMQNLP RONKR RLVQW RNWSVXT RUVYR", "J[SMPNNPMRMTNVPWRWUVWTXRXPWNUMSM ROPUM RNRVN RMTWO RNUXP ROVWR RPWVT", "JZOGO^ RUFU] RMNWL RMOWM RMWWU RMXWV", "JZNFNX RVLV^ RNNVL RNOVM RNWVU RNXVV", "JZNBNW RNNQLTLVMWOWQVSSUQVNW RNNQMTMVN RUMVOVQUSSU", "E_HIHL R\\I\\L RHI\\I RHJ\\J RHK\\K RHL\\L", "JZMNMQ RWNWQ RMNWN RMOWO RMPWP RMQWQ", "JZQCVMRTRU RULQS RTITKPRRUUY RW\\UYSXQXOYN[N]O_Ra RW\\UZSYOYO]P_Ra RSXPZN]", "JZPOOMOKMKMMNNPOSOUNWL RNKNN RMLOL RMMSO RPOUN RWLSY", "A^GfHfIeIdHcGcFdFfGhIiKiNhPfQdR`RUQ;Q4R/S-U,V,X-Y/Y3X6W8U;P?JCHEFHEJDNDREVGYJ[N\\R\\V[XZZW[T[PZMYKWITHPHMIKKJNJRKUMW RGdGeHeHdGd RU;Q?LCIFGIFKENERFVGXJ[ RR\\U[WZYWZTZPYMXKVITH", "IjNQOOQNSNUOVQVSUUSVQVOUNTMQMNNKPISHWH[I^K`NaRaW`[_]]`ZcVfQiMk RWHZI]K_N`R`W_[^]\\`YcTgQi RPOTO ROPUP RNQVQ RNRVR RNSVS ROTUT RPUTU ReLeNgNgLeL RfLfN ReMgM ReVeXgXgVeV RfVfX ReWgW", "D`H>Hf RI>If RM>Mf RQBSBSDQDQAR?T>W>Y?[A\\D\\I[LYNWOUOSNRLQNOQNROSQVRXSVUUWUYV[X\\[\\`[cYeWfTfReQcQ`S`SbQb RRBRD RQCSC RY?ZA[D[IZLYN RRLRNPQNRPSRVRX RYVZX[[[`ZcYe RR`Rb RQaSa", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 2400 to 2499 ********/ 
  "", "AcHBHb RIBIb R[B[b R\\B\\b RDB`B RDbMb RWb`b", "BaGBQPFb RFBPP REBPQ REB\\B^I[B RGa\\a RFb\\b^[[b", "I[X+U1R8P=OANFMNMVN^OcPgRlUsXy RU1S6Q<P@OFNNNVO^PdQhSnUs", "I[L+O1R8T=UAVFWNWVV^UcTgRlOsLy RO1Q6S<T@UFVNVVU^TdShQnOs", "I[M+MRMy RN+NRNy RM+X+ RMyXy", "I[V+VRVy RW+WRWy RL+W+ RLyWy", "H[V+R1P5O:O>PBTJTLSNROMRRUSVTXTZPbOfOjPoRsVy RT.R2Q5P:P>QCRF RR^QaPfPjQoRrTv", "I\\N+R1T5U:U>TBPJPLQNROWRRUQVPXPZTbUfUjToRsNy RP.R2S5T:T>SCRF RR^SaTfTjSoRrPv", "I[V.S1Q4O8N=NCOIPMSXT\\UbUgTlSoQs RS1Q5P8O=OBPHQLTWU[VaVgUlSpQsNv", "I[N.Q1S4U8V=VCUITMQXP\\ObOgPlQoSs RQ1S5T8U=UBTHSLPWO[NaNgOlQpSsVv", "7Z:RARRo R@RQo R?RRr RZ\"VJRr", "Ca].\\.[/[0\\1]1^0^.],[+Y+W,U.T0S3R:QJQjPsOv R\\/\\0]0]/\\/ RR:Rj RU.T1S:SZRjQqPtOvMxKyIyGxFvFtGsHsItIuHvGv RGtGuHuHtGt", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 2500 to 2599 ********/ 
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`RFNOKUIXGZE[C[BZBXCWDXCY RRFPMOQNVNZP[ RRFQJPOOVOZP[ R[FWORXP[ R[FYMXQWVWZY[Z[\\Z^X R[FZJYOXVXZY[", "G^RFQJOPMULWJZH[F[EZEXFWGXFY RRFRKSVT[ RRFSKTVT[ R`G_H`IaHaG`F^F\\GZJYLWQUWT[", "H]SFQGOIMLLNKRKVLYMZO[Q[TZVXXUYSZOZKYHXGWGUHSJQNPSPV RQGOJMNLRLVMYO[", "F]UGTHSJQOOUNWLZJ[ RTHSKQSPVOXMZJ[H[GZGXHWIXHY ROLNNMOKOJNJLKJMHOGRFVFYGZH[J[MZOYPVQTQRP RVFXGYHZJZMYOXPVQ", "H]UJULTNSOQPOPNNNLOIQGTFWFYGZIZMYPWSSWPYNZK[I[HZHXIWKWMXPZS[V[XZZX RWFXGYIYMXPVSSVOYK[", "F^UGTHSJQOOUNWLZJ[ RTHSKQSPVOXMZJ[H[GZGXHWIXHY ROLNNMOKOJNJLKJMHOGRFWFZG[I[KZMYNVORO RWFYGZIZKYMXNVO RROUPVRWYX[ RROTPURVYX[Y[[Z]X", "H\\NIMKMMNOPPSPVOXN[K\\H\\G[FZFXGWHVJUMSTRWPZN[ RVJUNTUSXQZN[K[IZHXHWIVJWIX", "I[YHXJVOTUSWQZO[ RSLRNPONOMMMKNIPGSF\\FZGYHXKVSUVTXRZO[M[KZJYJXKWLXKY RUFYGZG", "G]HJJGLFMFOHOKNNKVKYL[ RMFNHNKKSJVJYL[N[PZSWUTVR RZFVRUVUYW[X[ZZ\\X R[FWRVVVYW[", "G\\HJJGLFMFOHOKNOLVLYM[ RMFNHNKLRKVKYM[N[QZTWVTXPYMZIZGYFXFWGVIVLWNYP[Q]Q", "F]ILHLGKGIHGJFNFMHLLKUJ[ RLLLUK[ RVFTHRLOUMYK[ RVFUHTLSUR[ RTLTUS[ R`F^G\\IZLWUUYS[", "H\\PKOLMLLKLIMGOFQFSGTITLSPQUOXMZJ[H[GZGXHWIXHY RQFRGSISLRPPUNXLZJ[ R]G\\H]I^H^G]F[FYGWIULSPRURXSZT[U[WZYX", "G]JJLGNFOFQGQIOOORPT ROFPGPINONRPTRTUSWQYNZL R\\FZLWTUX R]F[LYQWUUXSZP[L[JZIXIWJVKWJX", "G\\ZHYJWOVRUTSWQYOZL[ RSLRNPONOMMMKNIPGSF]F[GZHYKXOVUTXQZL[H[GZGXHWJWLXOZQ[T[WZYX RVFZG[G", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 2600 to 2699 ********/ 
  "", "H\\WMW[X[ RWMXMX[ RWPUNSMPMNNLPKSKULXNZP[S[UZWX RWPSNPNNOMPLSLUMXNYPZSZWX", "H\\LFL[M[ RLFMFM[ RMPONQMTMVNXPYSYUXXVZT[Q[OZMX RMPQNTNVOWPXSXUWXVYTZQZMX", "I[XPVNTMQMONMPLSLUMXOZQ[T[VZXX RXPWQVOTNQNOONPMSMUNXOYQZTZVYWWXX", "H\\WFW[X[ RWFXFX[ RWPUNSMPMNNLPKSKULXNZP[S[UZWX RWPSNPNNOMPLSLUMXNYPZSZWX", "I[MTXTXQWOVNTMQMONMPLSLUMXOZQ[T[VZXX RMSWSWQVOTNQNOONPMSMUNXOYQZTZVYWWXX", "LZWFUFSGRJR[S[ RWFWGUGSH RTGSJS[ ROMVMVN ROMONVN", "H\\XMWMW\\V_U`SaQaO`N_L_ RXMX\\W_UaSbPbNaL_ RWPUNSMPMNNLPKSKULXNZP[S[UZWX RWPSNPNNOMPLSLUMXNYPZSZWX", "H\\LFL[M[ RLFMFM[ RMQPNRMUMWNXQX[ RMQPORNTNVOWQW[X[", "NWRFQGQHRISITHTGSFRF RRGRHSHSGRG RRMR[S[ RRMSMS[", "NWRFQGQHRISITHTGSFRF RRGRHSHSGRG RRMRbSb RRMSMSb", "H[LFL[M[ RLFMFM[ RXMWMMW RXMMX RPTV[X[ RQSX[", "NWRFR[S[ RRFSFS[", "CbGMG[H[ RGMHMH[ RHQKNMMPMRNSQS[ RHQKOMNONQORQR[S[ RSQVNXM[M]N^Q^[ RSQVOXNZN\\O]Q][^[", "H\\LML[M[ RLMMMM[ RMQPNRMUMWNXQX[ RMQPORNTNVOWQW[X[", "I\\QMONMPLSLUMXOZQ[T[VZXXYUYSXPVNTMQM RQNOONPMSMUNXOYQZTZVYWXXUXSWPVOTNQN", "H\\LMLbMb RLMMMMb RMPONQMTMVNXPYSYUXXVZT[Q[OZMX RMPQNTNVOWPXSXUWXVYTZQZMX", "H\\WMWbXb RWMXMXb RWPUNSMPMNNLPKSKULXNZP[S[UZWX RWPSNPNNOMPLSLUMXNYPZSZWX", "KYOMO[P[ ROMPMP[ RPSQPSNUMXM RPSQQSOUNXNXM", "J[XPWNTMQMNNMPNRPSUUWV RVUWWWXVZ RWYTZQZNY ROZNXMX RXPWPVN RWOTNQNNO RONNPOR RNQPRUTWUXWXXWZT[Q[NZMX", "MXRFR[S[ RRFSFS[ ROMVMVN ROMONVN", "H\\LMLWMZO[R[TZWW RLMMMMWNYPZRZTYWW RWMW[X[ RWMXMX[", "JZLMR[ RLMMMRY RXMWMRY RXMR[", "F^IMN[ RIMJMNX RRMNX RRPN[ RRPV[ RRMVX R[MZMVX R[MV[", "I[LMW[X[ RLMMMX[ RXMWML[ RXMM[L[", "JZLMR[ RLMMMRY RXMWMRYNb RXMR[ObNb", "I[VNL[ RXMNZ RLMXM RLMLNVN RNZXZX[ RL[X[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "K[UUTSRRPRNSMTLVLXMZO[Q[SZTX RPRNTMVMYO[ RVRTXTZV[XZYY[V RWRUXUZV[", "LZLVNSPO RSFMXMZO[P[RZTXUUURVVWWXWZV RTFNXNZO[", "LXTSSTTTTSSRQROSNTMVMXNZP[S[VYXV RQROTNVNYP[", "K[UUTSRRPRNSMTLVLXMZO[Q[SZTX RPRNTMVMYO[ RZFTXTZV[XZYY[V R[FUXUZV[", "LXOYQXRWSUSSRRQROSNTMVMXNZP[S[VYXV RQROTNVNYP[", "OXRRUOWLXIXGWFUGTIKdKfLgNfOcPZQ[S[UZVYXV RTISNRRO[M`Kd", "K[UUTSRRPRNSMTLVLXMZO[Q[SZTX RPRNTMVMYO[ RVRPd RWRT[R`PdOfMgLfLdMaO_R]V[YY[V", "L[LVNSPO RSFL[ RTFM[ ROUQSSRTRVSVUUXUZV[ RTRUSUUTXTZV[XZYY[V", "NVSLRMSNTMSL RQROXOZQ[SZTYVV RRRPXPZQ[", "NVSLRMSNTMSL RQRKd RRRO[M`KdJfHgGfGdHaJ_M]Q[TYVV", "LZLVNSPO RSFL[ RTFM[ RURUSVSURTRRTOU ROURVSZT[ ROUQVRZT[U[XYZV", "NVNVPSRO RUFOXOZQ[SZTYVV RVFPXPZQ[", "E^EVGSIRKSKUI[ RIRJSJUH[ RKUMSORPRRSRUP[ RPRQSQUO[ RRUTSVRWRYSYUXXXZY[ RWRXSXUWXWZY[[Z\\Y^V", "I[IVKSMROSOUM[ RMRNSNUL[ ROUQSSRTRVSVUUXUZV[ RTRUSUUTXTZV[XZYY[V", "KYRRPRNSMTLVLXMZO[Q[SZTYUWUUTSRRQSQURWTXVXXWYV RPRNTMVMYO[", "L[LVNSPO RQLHg RRLIg ROUQSSRTRVSVUUXUZV[ RTRUSUUTXTZV[XZYY[V", "K[UUTSRRPRNSMTLVLXMZO[Q[SZ RPRNTMVMYO[ RVRPdPfQgSfTcT[V[YY[V RWRT[R`Pd", "LZLVNSPRRSRUP[ RPRQSQUO[ RRUTSVRWRVU RVRVUWWXWZV", "NZNVPSQQQSTUUWUYTZR[ RQSSUTWTYR[ RNZP[U[XYZV", "NVNVPSRO RUFOXOZQ[SZTYVV RVFPXPZQ[ RPNVN", "K[NRLXLZN[O[QZSXUU RORMXMZN[ RVRTXTZV[XZYY[V RWRUXUZV[", "KZNRMTLWLZN[O[RZTXUUUR RORNTMWMZN[ RURVVWWXWZV", "H]LRJTIWIZK[L[NZPX RMRKTJWJZK[ RRRPXPZR[S[UZWXXUXR RSRQXQZR[ RXRYVZW[W]V", "JZJVLSNRPRQSQUPXOZM[L[KZKYLYKZ RWSVTWTWSVRURSSRUQXQZR[U[XYZV RQSRU RSSQU RPXQZ RQXOZ", "K[NRLXLZN[O[QZSXUU RORMXMZN[ RVRPd RWRT[R`PdOfMgLfLdMaO_R]V[YY[V", "LYLVNSPRRRTSTVSXPZN[ RRRSSSVRXPZ RN[P\\Q^QaPdNfLgKfKdLaO^R\\VYYV RN[O\\P^PaOdNf", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "NV", "JZ", 
  /******** Hershey Glyphs 2700 to 2799 ********/ 
  "H\\QFNGLJKOKRLWNZQ[S[VZXWYRYOXJVGSFQF ROGMJLOLRMWOZ RNYQZSZVY RUZWWXRXOWJUG RVHSGQGNH", "H\\NJPISFS[ RNJNKPJRHR[S[", "H\\LKLJMHNGPFTFVGWHXJXLWNUQL[ RLKMKMJNHPGTGVHWJWLVNTQK[ RLZYZY[ RK[Y[", "H\\MFXFQO RMFMGWG RWFPO RQNSNVOXQYTYUXXVZS[P[MZLYKWLW RPOSOVPXS RTOWQXTXUWXTZ RXVVYSZPZMYLW ROZLX", "H\\UIU[V[ RVFV[ RVFKVZV RUILV RLUZUZV", "H\\MFLO RNGMN RMFWFWG RNGWG RMNPMSMVNXPYSYUXXVZS[P[MZLYKWLW RLOMOONSNVOXR RTNWPXSXUWXTZ RXVVYSZPZMYLW ROZLX", "H\\VGWIXIWGTFRFOGMJLOLTMXOZR[S[VZXXYUYTXQVOSNRNOOMQ RWHTGRGOH RPGNJMOMTNXQZ RMVOYRZSZVYXV RTZWXXUXTWQTO RXSVPSOROOPMS RQONQMT", "H\\KFYFO[ RKFKGXG RXFN[O[", "H\\PFMGLILKMMNNPOTPVQWRXTXWWYTZPZMYLWLTMRNQPPTOVNWMXKXIWGTFPF RNGMIMKNMPNTOVPXRYTYWXYWZT[P[MZLYKWKTLRNPPOTNVMWKWIVG RWHTGPGMH RLXOZ RUZXX", "H\\WPURRSQSNRLPKMKLLINGQFRFUGWIXMXRWWUZR[P[MZLXMXNZ RWMVPSR RWNUQRRQRNQLN RPRMPLMLLMIPG RLKNHQGRGUHWK RSGVIWMWRVWTZ RUYRZPZMY", "MXRXQYQZR[S[TZTYSXRX RRYRZSZSYRY", "MXTZS[R[QZQYRXSXTYT\\S^Q_ RRYRZSZSYRY RS[T\\ RTZS^", "MXRMQNQORPSPTOTNSMRM RRNROSOSNRN RRXQYQZR[S[TZTYSXRX RRYRZSZSYRY", "MXRMQNQORPSPTOTNSMRM RRNROSOSNRN RTZS[R[QZQYRXSXTYT\\S^Q_ RRYRZSZSYRY RS[T\\ RTZS^", "MXRFRTST RRFSFST RRXQYQZR[S[TZTYSXRX RRYRZSZSYRY", "I\\LKLJMHNGQFTFWGXHYJYLXNWOUPRQ RLKMKMJNHQGTGWHXJXLWNUORP RMIPG RUGXI RXMTP RRPRTSTSP RRXQYQZR[S[TZTYSXRX RRYRZSZSYRY", "MXTFRGQIQLRMSMTLTKSJRJQK RRKRLSLSKRK RRGQK RQIRJ", "MXTHSIRIQHQGRFSFTGTJSLQM RRGRHSHSGRG RSITJ RTHSL", "F_\\MZMXNWPUVTXSYQZMZKYJWJUKSLRQOSMTKTISGQFPFNGMIMKNNPQUWXZZ[\\[ R\\M\\NZNWP RZMXPVVUXSZQ[M[KZJYIWIUJSLQQNRMSKSIRG RSHQGPGNH ROGNINKONQQVWXYZZ\\Z\\[", "I\\RBR_S_ RRBSBS_ RWIYIWGTFQFNGLILKMMNNVRWSXUXWWYTZQZOYNX RWIVHTGQGNHMIMKNMVQXSYUYWXYWZT[Q[NZLXNX RXXUZ", "G^[BIbJb R[B\\BJb", "KYUBSDQGOKNPNTOYQ]S`UbVb RUBVBTDRGPKOPOTPYR]T`Vb", "KYNBPDRGTKUPUTTYR]P`NbOb RNBOBQDSGUKVPVTUYS]Q`Ob", "JZRFQGSQRR RRFRR RRFSGQQRR RMINIVOWO RMIWO RMIMJWNWO RWIVINOMO RWIMO RWIWJMNMO", "F_JQ[Q[R RJQJR[R", "F_RIRZSZ RRISISZ RJQ[Q[R RJQJR[R", "F_JM[M[N RJMJN[N RJU[U[V RJUJV[V", "NWSFRGRM RSGRM RSFTGRM", "I[NFMGMM RNGMM RNFOGMM RWFVGVM RWGVM RWFXGVM", "KYQFOGNINKOMQNSNUMVKVIUGSFQF RQFNIOMSNVKUGQF RSFOGNKQNUMVISF", "F^ZIJRZ[ RZIZJLRZZZ[", "F^JIZRJ[ RJIJJXRJZJ[", "G^OFObPb ROFPFPb RUFUbVb RUFVFVb RJP[P[Q RJPJQ[Q RJW[W[X RJWJX[X", "F^[FYGVHSHPGNFLFJGIIIKKMMMOLPJPHNF R[FH[I[ R[F\\FI[ RYTWTUUTWTYV[X[ZZ[X[VYT RNFJGIKMMPJNF RLFIIKMOLPHLF RYTUUTYX[[XYT RWTTWV[ZZ[VWT", "E`WMTKQKOLNMMOMRNTOUQVTVWT RWMTLQLOMNONROTQUTUWT RVKVSWUYVZV\\U]S]O\\L[JYHWGTFQFNGLHJJILHOHRIUJWLYNZQ[U[YZ RVKWKWSXUZV RYV[U\\S\\O[LZJYIWHTGQGNHLIKJJLIOIRJUKWLXNYQZUZYYYZ", "E_JPLONOPPSTTUVVXVZU[S[QZOXNVNTOSPPTNULUJT RZPXOVOTPQTPUNVLVJUISIQJOLNNNPOQPTTVUXUZT RKOJQJSKU RYUZSZQYO", "", "", "", "", "", "", "", "", "", "", "", "", "NV", "JZ", "H]TFQGOIMLLOKSKVLYMZO[Q[TZVXXUYRZNZKYHXGVFTF RTFRGPINLMOLSLVMYO[ RQ[SZUXWUXRYNYKXHVF", "H]TJO[ RVFP[ RVFSIPKNL RUIQKNL", "H]OJPKOLNKNJOHPGSFVFYGZIZKYMWOTQPSMUKWI[ RVFXGYIYKXMVOPS RJYKXMXRZUZWYXW RMXR[U[WZXW", "H]OJPKOLNKNJOHPGSFVFYGZIZKYMVOSP RVFXGYIYKXMVO RQPSPVQWRXTXWWYVZS[O[LZKYJWJVKULVKW RSPUQVRWTWWVYUZS[", "H]XGR[ RYFS[ RYFJUZU", "H]QFLP RQF[F RQGVG[F RLPMOPNSNVOWPXRXUWXUZR[O[LZKYJWJVKULVKW RSNUOVPWRWUVXTZR[", "H]YIXJYKZJZIYGWFTFQGOIMLLOKSKWLYMZO[R[UZWXXVXSWQVPTOQOOPMRLT RTFRGPINLMOLSLXMZ RR[TZVXWVWRVP", "H]NFLL R[FZIXLSRQUPWO[ RXLRRPUOWN[ RMIPFRFWI RNHPGRGWIYIZH[F", "H]SFPGOHNJNMOOQPTPXOYNZLZIYGVFSF RSFQGPHOJOMPOQP RTPWOXNYLYIXGVF RQPMQKSJUJXKZN[R[VZWYXWXTWRVQTP RQPNQLSKUKXLZN[ RR[UZVYWWWSVQ", "H]YMXOVQTRQROQNPMNMKNIPGSFVFXGYHZJZNYRXUVXTZQ[N[LZKXKWLVMWLX ROQNONKOIQGSF RXGYIYNXRWUUXSZQ[", "MXPYOZP[QZPY", "MXP[OZPYQZQ[P]N_", "MXSMRNSOTNSM RPYOZP[QZ", "MXSMRNSOTNSM RP[OZPYQZQ[P]N_", "MXUFTGRS RUGRS RUFVGRS RPYOZP[QZPY", "H]OJPKOLNKNJOHPGSFWFZG[I[KZMYNSPQQQSRTTT RWFYGZIZKYMXNVO RPYOZP[QZPY", "MXVFTHSJSKTLUKTJ", "MXUHTGUFVGVHUJSL", "E_\\N[O\\P]O]N\\M[MYNWPRXPZN[K[HZGXGVHTISKRPPROTMUKUITGRFPGOIOLPRQUSXUZW[Y[ZYZX RK[IZHXHVITJSPP ROLPQQTSWUYWZYZZY", "H]TBL_ RYBQ_ RZJYKZL[K[JZHYGVFRFOGMIMKNMONVRXT RMKOMVQWRXTXWWYVZS[O[LZKYJWJVKULVKW", "G]_BEb", "KZZBVESHQKOONTNXO]P`Qb RVESIQMPPOUOZP_Qb", "JYSBTDUGVLVPUUSYQ\\N_Jb RSBTEUJUOTTSWQ[N_", "J[TFTR ROIYO RYIOO", "E_IR[R", "E_RIR[ RIR[R", "E_IO[O RIU[U", "NWUFSM RVFSM", "I[PFNM RQFNM RYFWM RZFWM", "KZSFQGPIPKQMSNUNWMXKXIWGUFSF", "F^ZIJRZ[", "F^JIZRJ[", "H]SFLb RYFRb RLQZQ RKWYW", "E_^F\\GXHUHQGOFMFKGJIJKLMNMPLQJQHOF R^FF[ RXTVTTUSWSYU[W[YZZXZVXT", "E`WNVLTKQKOLNMMPMSNUPVSVUUVS RQKOMNPNSOUPV RWKVSVUXVZV\\T]Q]O\\L[JYHWGTFQFNGLHJJILHOHRIUJWLYNZQ[T[WZYYZX RXKWSWUXV", "F_\\S[UYVWVUUTTQPPONNLNJOIQISJULVNVPUQTTPUOWNYN[O\\Q\\S", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 2800 to 2899 ********/ 
  "", "H\\RFK[ RRFY[ RRIX[ RMUVU RI[O[ RU[[[", "G]LFL[ RMFM[ RIFYFYLXF RMPUPXQYRZTZWYYXZU[I[ RUPWQXRYTYWXYWZU[", "G]LFL[ RMFM[ RIFUFXGYHZJZLYNXOUP RUFWGXHYJYLXNWOUP RMPUPXQYRZTZWYYXZU[I[ RUPWQXRYTYWXYWZU[", "I[NFN[ ROFO[ RKFZFZLYF RK[R[", "F^NFNLMTLXKZJ[ RXFX[ RYFY[ RKF\\F RG[\\[ RG[Gb RH[Gb R[[\\b R\\[\\b", "G\\LFL[ RMFM[ RSLST RIFYFYLXF RMPSP RI[Y[YUX[", "CbRFR[ RSFS[ ROFVF RGGHHGIFHFGGFHFIGJIKMLONPWPYOZM[I\\G]F^F_G_H^I]H^G RNPLQKSJXIZH[ RNPMQLSKXJZI[G[FZEX RWPYQZS[X\\Z][ RWPXQYSZX[Z\\[^[_Z`X RO[V[", "H\\LIKFKLLINGPFTFWGXIXLWNTOQO RTFVGWIWLVNTO RTOVPXRYTYWXYWZT[O[MZLYKWKVLUMVLW RWQXTXWWYVZT[", "F^KFK[ RLFL[ RXFX[ RYFY[ RHFOF RUF\\F RXHLY RH[O[ RU[\\[", "F^KFK[ RLFL[ RXFX[ RYFY[ RHFOF RUF\\F RXHLY RH[O[ RU[\\[ RN@N?M?M@NBPCTCVBW@", "F^KFK[ RLFL[ RHFOF RLPSPUOVMWIXGYFZF[G[HZIYHZG RSPUQVSWXXZY[ RSPTQUSVXWZX[Z[[Z\\X RH[O[", "E^MFMLLTKXJZI[H[GZGYHXIYHZ RXFX[ RYFY[ RJF\\F RU[\\[", "F_KFK[ RLFRX RKFR[ RYFR[ RYFY[ RZFZ[ RHFLF RYF]F RH[N[ RV[][", "F^KFK[ RLFL[ RXFX[ RYFY[ RHFOF RUF\\F RLPXP RH[O[ RU[\\[", "G]QFNGLIKKJOJRKVLXNZQ[S[VZXXYVZRZOYKXIVGSFQF RQFOGMILKKOKRLVMXOZQ[ RS[UZWXXVYRYOXKWIUGSF", "F^KFK[ RLFL[ RXFX[ RYFY[ RHF\\F RH[O[ RU[\\[", "G]LFL[ RMFM[ RIFUFXGYHZJZMYOXPUQMQ RUFWGXHYJYMXOWPUQ RI[P[", "G\\XIYLYFXIVGSFQFNGLIKKJNJSKVLXNZQ[S[VZXXYV RQFOGMILKKNKSLVMXOZQ[", "I\\RFR[ RSFS[ RLFKLKFZFZLYF RO[V[", "H]KFRV RLFSV RZFSVQYPZN[M[LZLYMXNYMZ RIFOF RVF\\F", "F_RFR[ RSFS[ ROFVF RPILJJLIOIRJULWPXUXYW[U\\R\\O[LYJUIPI RPIMJKLJOJRKUMWPX RUXXWZU[R[OZLXJUI RO[V[", "H\\KFX[ RLFY[ RYFK[ RIFOF RUF[F RI[O[ RU[[[", "F^KFK[ RLFL[ RXFX[ RYFY[ RHFOF RUF\\F RH[\\[ R[[\\b R\\[\\b", "F]KFKQLSOTRTUSWQ RLFLQMSOT RWFW[ RXFX[ RHFOF RTF[F RT[[[", "BcGFG[ RHFH[ RRFR[ RSFS[ R]F][ R^F^[ RDFKF ROFVF RZFaF RD[a[", "BcGFG[ RHFH[ RRFR[ RSFS[ R]F][ R^F^[ RDFKF ROFVF RZFaF RD[a[ R`[ab Ra[ab", "F`PFP[ RQFQ[ RIFHLHFTF RQPXP[Q\\R]T]W\\Y[ZX[M[ RXPZQ[R\\T\\W[YZZX[", "CaHFH[ RIFI[ REFLF RIPPPSQTRUTUWTYSZP[E[ RPPRQSRTTTWSYRZP[ R[F[[ R\\F\\[ RXF_F RX[_[", "H]MFM[ RNFN[ RJFQF RNPUPXQYRZTZWYYXZU[J[ RUPWQXRYTYWXYWZU[", "H]LIKFKLLINGQFSFVGXIYKZNZSYVXXVZS[P[MZLYKWKVLUMVLW RSFUGWIXKYNYSXVWXUZS[ RPPYP", "CbHFH[ RIFI[ REFLF RE[L[ RVFSGQIPKOOORPVQXSZV[X[[Z]X^V_R_O^K]I[GXFVF RVFTGRIQKPOPRQVRXTZV[ RX[ZZ\\X]V^R^O]K\\IZGXF RIPOP", "G]WFW[ RXFX[ R[FOFLGKHJJJLKNLOOPWP ROFMGLHKJKLLNMOOP RRPPQORLYKZJZIY RPQOSMZL[J[IYIX RT[[[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 2900 to 2999 ********/ 
  "", "I]NONPMPMONNPMTMVNWOXQXXYZZ[ RWOWXXZZ[[[ RWQVRPSMTLVLXMZP[S[UZWX RPSNTMVMXNZP[", "H\\XFWGQINKLNKQKULXNZQ[S[VZXXYUYSXPVNSMQMNNLPKS RXFWHUIQJNLLN RQMONMPLSLUMXOZQ[ RS[UZWXXUXSWPUNSM", "H\\MMM[ RNMN[ RJMUMXNYPYQXSUT RUMWNXPXQWSUT RNTUTXUYWYXXZU[J[ RUTWUXWXXWZU[", "HZMMM[ RNMN[ RJMXMXRWM RJ[Q[", "F]NMNQMWLZK[ RWMW[ RXMX[ RKM[M RI[H`H[[[[`Z[", "H[LSXSXQWOVNTMQMNNLPKSKULXNZQ[S[VZXX RWSWPVN RQMONMPLSLUMXOZQ[", "E`RMR[ RSMS[ ROMVM RJNIOHNIMJMKNMRNSPTUTWSXRZN[M\\M]N\\O[N RPTNUMVKZJ[ RPTNVLZK[I[HZGX RUTWUXVZZ[[ RUTWVYZZ[\\[]Z^X RO[V[", "I[MOLMLQMONNPMTMWNXPXQWSTT RTMVNWPWQVSTT RQTTTWUXWXXWZT[P[MZLXLWMVNWMX RTTVUWWWXVZT[", "G]LML[ RMMM[ RWMW[ RXMX[ RIMPM RTM[M RI[P[ RT[[[ RWNMZ", "G]LML[ RMMM[ RWMW[ RXMX[ RIMPM RTM[M RI[P[ RT[[[ RWNMZ ROGOFNFNGOIQJSJUIVG", "H\\MMM[ RNMN[ RJMQM RNTPTSSTRVNWMXMYNXOWN RPTSUTVVZW[ RPTRUSVUZV[X[YZZX RJ[Q[", "G]NMNQMWLZK[J[IZJYKZ RWMW[ RXMX[ RKM[M RT[[[", "G^LML[ RLMR[ RMMRY RXMR[ RXMX[ RYMY[ RIMMM RXM\\M RI[O[ RU[\\[", "G]LML[ RMMM[ RWMW[ RXMX[ RIMPM RTM[M RMTWT RI[P[ RT[[[", "H\\QMNNLPKSKULXNZQ[S[VZXXYUYSXPVNSMQM RQMONMPLSLUMXOZQ[ RS[UZWXXUXSWPUNSM", "G]LML[ RMMM[ RWMW[ RXMX[ RIM[M RI[P[ RT[[[", "G\\LMLb RMMMb RMPONQMSMVNXPYSYUXXVZS[Q[OZMX RSMUNWPXSXUWXUZS[ RIMMM RIbPb", "H[WPVQWRXQXPVNTMQMNNLPKSKULXNZQ[S[VZXX RQMONMPLSLUMXOZQ[", "I\\RMR[ RSMS[ RMMLRLMYMYRXM RO[V[", "I[LMR[ RMMRY RXMR[P_NaLbKbJaK`La RJMPM RTMZM", "H]RFRb RSFSb ROFSF RRPQNPMNMLNKQKWLZN[P[QZRX RNMMNLQLWMZN[ RWMXNYQYWXZW[ RSPTNUMWMYNZQZWYZW[U[TZSX RObVb", "H\\LMW[ RMMX[ RXML[ RJMPM RTMZM RJ[P[ RT[Z[", "G]LML[ RMMM[ RWMW[ RXMX[ RIMPM RTM[M RI[[[[`Z[", "G]LMLTMVPWRWUVWT RMMMTNVPW RWMW[ RXMX[ RIMPM RTM[M RT[[[", "CbHMH[ RIMI[ RRMR[ RSMS[ R\\M\\[ R]M][ REMLM ROMVM RYM`M RE[`[", "CbHMH[ RIMI[ RRMR[ RSMS[ R\\M\\[ R]M][ REMLM ROMVM RYM`M RE[`[``_[", "H]QMQ[ RRMR[ RLMKRKMUM RRTVTYUZWZXYZV[N[ RVTXUYWYXXZV[", "E_JMJ[ RKMK[ RGMNM RKTOTRUSWSXRZO[G[ ROTQURWRXQZO[ RYMY[ RZMZ[ RVM]M RV[][", "J[OMO[ RPMP[ RLMSM RPTTTWUXWXXWZT[L[ RTTVUWWWXVZT[", "I\\MOLMLQMONNPMSMVNXPYSYUXXVZS[P[NZLXLWMVNWMX RSMUNWPXSXUWXUZS[ RRTXT", "DaIMI[ RJMJ[ RFMMM RF[M[ RVMSNQPPSPUQXSZV[X[[Z]X^U^S]P[NXMVM RVMTNRPQSQURXTZV[ RX[ZZ\\X]U]S\\PZNXM RJTPT", "G\\VMV[ RWMW[ RZMOMLNKPKQLSOTVT ROMMNLPLQMSOT RTTQUPVNZM[ RTTRUQVOZN[L[KZJX RS[Z[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 3000 to 3099 ********/ 
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`Y`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`MFGZ RMGNYN[ RNFOY ROFPX R[FPXN[ R[FU[ R\\FV[ R]FW[ RJFOF R[F`F RD[J[ RR[Z[ RKFMG RLFMH R^F\\H R_F\\G RGZE[ RGZI[ RVZS[ RVYT[ RWYX[ RVZY[", "F_OFIZ ROFV[ RPFVX RQFWX R\\GWXV[ RLFQF RYF_F RF[L[ RMFPG RNFPH RZF\\G R^F\\G RIZG[ RIZK[", "G]SFPGNILLKOJSJVKYLZN[Q[TZVXXUYRZNZKYHXGVFSF ROIMLLOKSKWLY RUXWUXRYNYJXH RSFQGOJNLMOLSLXMZN[ RQ[SZUWVUWRXNXIWGVF", "F]OFI[ RPFJ[ RQFK[ RLFXF[G\\I\\K[NYPUQMQ RZG[I[KZNXP RXFYGZIZKYNWPUQ RF[N[ RMFPG RNFOH RRFPH RSFPG RJZG[ RJYH[ RKYL[ RJZM[", "G]SFPGNILLKOJSJVKYLZN[Q[TZVXXUYRZNZKYHXGVFSF ROIMLLOKSKWLY RUXWUXRYNYJXH RSFQGOJNLMOLSLXMZN[ RQ[SZUWVUWRXNXIWGVF RLXMVOUPURVSXT]U^V^W] RT^U_V_ RSXS_T`V`W]W\\", "F^OFI[ RPFJ[ RQFK[ RLFWFZG[I[KZNYOVPNP RYGZIZKYNXO RWFXGYIYKXNVP RRPTQURWXXYYYZX RWYXZYZ RURVZW[Y[ZXZW RF[N[ RMFPG RNFOH RRFPH RSFPG RJZG[ RJYH[ RKYL[ RJZM[", "G^ZH[H\\F[L[JZHYGVFRFOGMIMLNNPPVSWUWXVZ RNLONVRWT ROGNINKOMUPWRXTXWWYVZS[O[LZKYJWJUI[JYKY", "G]TFN[ RUFO[ RVFP[ RMFKL R]F\\L RMF]F RK[S[ RNFKL RPFLI RRFMG RYF\\G RZF\\H R[F\\I R\\F\\L ROZL[ ROYM[ RPYQ[ ROZR[", "F_NFKQJUJXKZN[R[UZWXXU\\G ROFLQKUKYLZ RPFMQLULYN[ RKFSF RYF_F RLFOG RMFNH RQFOH RRFOG RZF\\G R^F\\G", "H\\NFNHOYO[ ROGPX RPFQW R[GO[ RLFSF RXF^F RMFNH RQFPH RRFOG RYF[G R]F[G", "E_MFMHKYK[ RNGLX ROFMW RUFMWK[ RUFUHSYS[ RVGTX RWFUW R]GUWS[ RJFRF RUFWF RZF`F RKFNG RLFMH RPFNI RQFNG R[F]G R_F]G", "G]NFT[ ROFU[ RPFV[ R[GIZ RLFSF RXF^F RF[L[ RQ[X[ RMFOH RQFPH RRFPG RYF[G R]F[G RIZG[ RIZK[ RTZR[ RTYS[ RUYW[", "G]MFQPN[ RNFRPO[ ROFSPP[ R\\GSP RKFRF RYF_F RK[S[ RLFNG RPFOH RQFNG RZF\\G R^F\\G ROZL[ ROYM[ RPYQ[ ROZR[", "G]ZFH[ R[FI[ R\\FJ[ R\\FNFLL RH[V[XU ROFLL RPFMI RRFNG RR[VZ RT[WX RU[XU", "", "", "", "", "", "", "H\\JFR[ RKFRX RLFSX RJFZFR[ RLGYG RLHYH", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 3100 to 3199 ********/ 
  "", "I]NPNOOOOQMQMONNPMTMVNWOXQXXYZZ[ RVOWQWXXZ RTMUNVPVXWZZ[[[ RVRUSPTMULWLXMZP[S[UZVX RNUMWMXNZ RUSQTOUNWNXOZP[", "G\\LFL[MZOZ RMGMY RIFNFNZ RNPONQMSMVNXPYSYUXXVZS[Q[OZNX RWPXRXVWX RSMUNVOWRWVVYUZS[ RJFLG RKFLH", "H[WQWPVPVRXRXPVNTMQMNNLPKSKULXNZQ[S[VZXX RMPLRLVMX RQMONNOMRMVNYOZQ[", "H]VFV[[[ RWGWZ RSFXFX[ RVPUNSMQMNNLPKSKULXNZQ[S[UZVX RMPLRLVMX RQMONNOMRMVNYOZQ[ RTFVG RUFVH RXYY[ RXZZ[", "H[MSXSXQWOVNSMQMNNLPKSKULXNZQ[S[VZXX RWRWQVO RMPLRLVMX RVSVPUNSM RQMONNOMRMVNYOZQ[", "KYWHWGVGVIXIXGWFTFRGQHPKP[ RRHQKQZ RTFSGRIR[ RMMVM RM[U[ RPZN[ RPYO[ RRYS[ RRZT[", "I\\XNYOZNYMXMVNUO RQMONNOMQMSNUOVQWSWUVVUWSWQVOUNSMQM ROONQNSOU RUUVSVQUO RQMPNOPOTPVQW RSWTVUTUPTNSM RNUMVLXLYM[N\\Q]U]X^Y_ RN[Q\\U\\X] RLYMZP[U[X\\Y^Y_XaUbObLaK_K^L\\O[ RObMaL_L^M\\O[", "G^LFL[ RMGMZ RIFNFN[ RNQOOPNRMUMWNXOYRY[ RWOXRXZ RUMVNWQW[ RI[Q[ RT[\\[ RJFLG RKFLH RLZJ[ RLYK[ RNYO[ RNZP[ RWZU[ RWYV[ RYYZ[ RYZ[[", "LXQFQHSHSFQF RRFRH RQGSG RQMQ[ RRNRZ RNMSMS[ RN[V[ ROMQN RPMQO RQZO[ RQYP[ RSYT[ RSZU[", "KXRFRHTHTFRF RSFSH RRGTG RRMR^QaPb RSNS]R` ROMTMT]S`RaPbMbLaL_N_NaMaM` RPMRN RQMRO", "G]LFL[ RMGMZ RIFNFN[ RWNNW RRSY[ RRTX[ RQTW[ RTM[M RI[Q[ RT[[[ RJFLG RKFLH RUMWN RZMWN RLZJ[ RLYK[ RNYO[ RNZP[ RWYU[ RVYZ[", "LXQFQ[ RRGRZ RNFSFS[ RN[V[ ROFQG RPFQH RQZO[ RQYP[ RSYT[ RSZU[", "AcFMF[ RGNGZ RCMHMH[ RHQIOJNLMOMQNROSRS[ RQORRRZ ROMPNQQQ[ RSQTOUNWMZM\\N]O^R^[ R\\O]R]Z RZM[N\\Q\\[ RC[K[ RN[V[ RY[a[ RDMFN REMFO RFZD[ RFYE[ RHYI[ RHZJ[ RQZO[ RQYP[ RSYT[ RSZU[ R\\ZZ[ R\\Y[[ R^Y_[ R^Z`[", "G^LML[ RMNMZ RIMNMN[ RNQOOPNRMUMWNXOYRY[ RWOXRXZ RUMVNWQW[ RI[Q[ RT[\\[ RJMLN RKMLO RLZJ[ RLYK[ RNYO[ RNZP[ RWZU[ RWYV[ RYYZ[ RYZ[[", "H\\QMNNLPKSKULXNZQ[S[VZXXYUYSXPVNSMQM RMPLRLVMX RWXXVXRWP RQMONNOMRMVNYOZQ[ RS[UZVYWVWRVOUNSM", "G\\LMLb RMNMa RIMNMNb RNPONQMSMVNXPYSYUXXVZS[Q[OZNX RWPXRXVWX RSMUNVOWRWVVYUZS[ RIbQb RJMLN RKMLO RLaJb RL`Kb RN`Ob RNaPb", "H\\VNVb RWOWa RUNWNXMXb RVPUNSMQMNNLPKSKULXNZQ[S[UZVX RMPLRLVMX RQMONNOMRMVNYOZQ[ RSb[b RVaTb RV`Ub RX`Yb RXaZb", "IZNMN[ RONOZ RKMPMP[ RWOWNVNVPXPXNWMUMSNQPPS RK[S[ RLMNN RMMNO RNZL[ RNYM[ RPYQ[ RPZR[", "J[WOXMXQWOVNTMPMNNMOMQNSPTUUWVXY RNNMQ RNRPSUTWU RXVWZ RMONQPRUSWTXVXYWZU[Q[OZNYMWM[NY", "KZPHPVQYRZT[V[XZYX RQHQWRY RPHRFRWSZT[ RMMVM", "G^LMLVMYNZP[S[UZVYWW RMNMWNY RIMNMNWOZP[ RWMW[\\[ RXNXZ RTMYMY[ RJMLN RKMLO RYYZ[ RYZ[[", "I[LMR[ RMMRY RNMSY RXNSYR[ RJMQM RTMZM RKMNO RPMNN RVMXN RYMXN", "F^JMN[ RKMNX RLMOX RRMOXN[ RRMV[ RSMVX RRMTMWX RZNWXV[ RGMOM RWM]M RHMKN RNMLN RXMZN R\\MZN", "H\\LMV[ RMMW[ RNMX[ RWNMZ RJMQM RTMZM RJ[P[ RS[Z[ RKMMN RPMNN RUMWN RYMWN RMZK[ RMZO[ RVZT[ RWZY[", "H[LMR[ RMMRY RNMSY RXNSYP_NaLbJbIaI_K_KaJaJ` RJMQM RTMZM RKMNO RPMNN RVMXN RYMXN", "I[VML[ RWMM[ RXMN[ RXMLMLQ RL[X[XW RMMLQ RNMLP ROMLO RQMLN RS[XZ RU[XY RV[XX RW[XW", "G^[MZQYTWXUZR[P[MZKXJUJSKPMNPMRMUNVOWQYXZZ[[\\[ RZMYQXTWVUYTZR[ RLXKVKRLP RP[NZMYLVLRMONNPM RRMTNUOVQXXYZ[[", "G\\QFNGMHLJKNKb RNHMJLNLa RQFOGNIMNMb RQFSFVGWHXJXLWNVOSP RPPTPWQXRYTYWXYWZT[Q[OZNYMW RVHWJWLVN RWRXTXWWY RSFUGVIVMUOSP RTPVQWSWXVZT[ RKbMb", "F\\HRINKMMMONPOQRRYSb RIOKNMNOOPP RHRIPKOMOOPPQQTRYRa RXMWPVRTUSWR[Qb RYMWQ RZMYOWRTVSXR[ RXMZM RQbSb", "H\\SMQMNNLPKSKULXNZQ[S[VZXXYUYSXPVNSMPLNKMJMHNGPFSFWH RMPLSLUMX RWXXUXSWP RQMONNOMRMVNYOZQ[ RS[UZVYWVWRVOUNOKNJNIOHQGTGWH", "I[SMUNVOWOVNSMQMMNLOLQMRQS RSSQSMTKVKXMZP[S[VZXXWXVZ RNNMOMQNR RMULVLXMY RQMONNONQORQS RQSNTMVMXNZP[", "I[QHRGRFQFPGPIQJTKXKYKYJXJUKSLPNNPMRLULWMYNZP[S\\U]V_VaUbSbRaR`S`Sa RPOOPNRMUMWNYOZ RUKRMQNOQNTNWOYQ[S\\", "G]JMKNLPL[ RKMLNMPMZ RHPINJMLMMNNPN[ RUMVNWQWb RWOXRXa RNQOOPNRMUMWNXOYRYb RL[N[ RWbYb", "F]IMJNKPKTLWMYNZQ[S[VZWYXWYRYOXJVGTFRFPGOIOKPMSOVP[Q RJMKNLPLTMWNY RVYWWXRXOWJVHTG RGPHNIMKMLNMPMTNXOZQ[ RS[UZVXWSWNVJUHSGQGOI", "KZNMONPPPXQZS[U[WZXX ROMPNQPQXRZ RLPMNNMPMQNRPRXSZT[", "G]JMKNLPL[ RKMLNMPMZ RHPINJMLMMNNPN[ RSOUNWNXOXPZPZNXMVMTNQQOTNW RXNYOYP RPSQSWYYYZX RTWWZYZ RRTUZV[X[YZZX RL[N[", "H\\JGKFMFOGQIXXYZZ[ ROHPIWXXY RMFNGOIVXXZZ[[[ RRMJZJ[K[RM", "G]KMKb RLNLa RMMMb RVMVXWZX[Z[[Z\\X RWNWXXZY[ RXMXXYZZ[ RMXNZP[R[TZUYVW RKMMM RVMXM RKbMb", "G]JMKNLPMTN[ RKMLNMPNTOZ RHPINJMLMMNNPOTPZ RVVWTXQXMYMZNYQXSVVTXQZN[ RXRYOYM", "JZPGSFRFPGOHOIPJSKVLWKVJSKPLNMMOMQNRPSSTVUWTVSSTOUMVLXLZM[O\\S]U^V_VaTbRbOaPaRb ROMNONQOR RNVMXMZN[ RVKSKQLPMOOOQQSST RVTSTPUOVNXNZP\\S]", "H\\QMNNLPKSKULXNZQ[S[VZXXYUYSXPVNSMQM RMPLRLVMX RWXXVXRWP RQMONNOMRMVNYOZQ[ RS[UZVYWVWRVOUNSM", "G]IQJOKNMM[M RKOMNZN RIQJPLO[O ROONZM[LZMWOO RUOVZW[XZWWUO R[M[O ROOMZ RUOWZ", "G\\QMNNLPKTKb RMPLTLa RQMONNOMSMb RMWNYOZQ[S[VZXXYUYSXPVNSMQM RWXXVXRWP RS[UZVYWVWRVOUNSM RKbMb", "G]PMMNKPJSJUKXMZP[R[UZWXXUXSWPUNRM RLPKRKVLX RVXWVWRVP RPMNNMOLRLVMYNZP[ RR[TZUYVVVRUOTNRM RRMZO[N[MPM RRMZN", "H\\JQKOLNNMZM RLONNYN RJQKPMOZO RROQZR[SZRO RZMZO RRORZ", "G\\JMKNLPLUMXOZQ[S[UZWXXVYRYNXMWMXPXSWWUZ RKMLNMPMUNX RWMXNXO RHPINJMLMMNNPNVOYQ[", "G]RQQNPMNMLNKOJRJUKXMZP[T[WZYXZUZRYOXNVMTMSNRQ RLOKRKULX RXXYUYRXO RNMMNLQLVMYNZP[ RT[VZWYXVXQWNVM RRQQb RRQRa RRQSb RQbSb", "H\\LMMNNPT_VaXbZb[a RNOOPU_V` RINJMLMNNPPV_WaXb RVSXPYMZMYOVSN\\K`JbKbL_N\\", "F]HNINJPJUKXMZP[T[VZXXYVZRZNYMXMYPYSXWVZ RJNKPKULX RXMYNYO RGPHNIMJMKNLPLVMYNZP[ RQFSb RRGRa RSFQb RQFSF RQbSb", "F^NMLNJPISIWJYKZM[O[QZRYSWSTRSQTQWRYSZU[W[YZZY[W[SZPXNVM RKPJSJWKY RRTRX RYYZWZSYP RNMLOKRKWLZM[ RW[XZYWYRXOVM", "G]WMUTUXVZW[Y[[Y\\W RXMVTVZ RWMYMWTVX RUTUQTNRMPMMNKQJTJVKYLZN[P[RZSYTWUT RNNLQKTKWLY RPMNOMQLTLWMZN[", "I\\PFNMMSMWNYOZQ[S[VZXWYTYRXOWNUMSMQNPOOQNT RQFOMNQNWOZ RVYWWXTXQWO RMFRFPMNT RS[UYVWWTWQVNUM RNFQG ROFPH", "I[WQWPVPVRXRXPWNUMRMONMQLTLVMYNZP[R[UZWW ROONQMTMWNY RRMPOOQNTNWOZP[", "G]YFVQUUUXVZW[Y[[Y\\W RZFWQVUVZ RVF[FWTVX RUTUQTNRMPMMNKQJTJVKYLZN[P[RZSYTWUT RMOLQKTKWLY RPMNOMQLTLWMZN[ RWFZG RXFYH", "I[MVQUTTWRXPWNUMRMONMQLTLVMYNZP[R[UZWX ROONQMTMWNY RRMPOOQNTNWOZP[", "JZZHZGYGYI[I[GZFXFVGTISKRNQRO[N^M`Kb RTJSMRRP[O^ RXFVHUJTMSRQZP]O_MaKbIbHaH_J_JaIaI` RNMYM", "H]XMT[S^QaOb RYMU[S_ RXMZMV[T_RaObLbJaI`I^K^K`J`J_ RVTVQUNSMQMNNLQKTKVLYMZO[Q[SZTYUWVT RNOMQLTLWMY RQMOONQMTMWNZO[", "G]OFI[K[ RPFJ[ RLFQFK[ RMTOPQNSMUMWNXPXSVX RWNWRVVVZ RWPUUUXVZW[Y[[Y\\W RMFPG RNFOH", "KXTFTHVHVFTF RUFUH RTGVG RLQMOOMQMRNSPSSQX RRNRRQVQZ RRPPUPXQZR[T[VYWW", "KXUFUHWHWFUF RVFVH RUGWG RMQNOPMRMSNTPTSRZQ]P_NaLbJbIaI_K_KaJaJ` RSNSSQZP]O_ RSPRTP[O^N`Lb", "G]OFI[K[ RPFJ[ RLFQFK[ RYOYNXNXPZPZNYMWMUNQROS RMSOSQTRUTYUZWZ RQUSYTZ ROSPTRZS[U[WZYW RMFPG RNFOH", "LXTFQQPUPXQZR[T[VYWW RUFRQQUQZ RQFVFRTQX RRFUG RSFTH", "@cAQBODMFMGNHPHSF[ RGNGSE[ RGPFTD[F[ RHSJPLNNMPMRNSPSSQ[ RRNRSP[ RRPQTO[Q[ RSSUPWNYM[M]N^P^S\\X R]N]R\\V\\Z R]P[U[X\\Z][_[aYbW", "F^GQHOJMLMMNNPNSL[ RMNMSK[ RMPLTJ[L[ RNSPPRNTMVMXNYPYSWX RXNXRWVWZ RXPVUVXWZX[Z[\\Y]W", "H\\QMNNLQKTKVLYMZP[S[VZXWYTYRXOWNTMQM RNOMQLTLWMY RVYWWXTXQWO RQMOONQMTMWNZP[ RS[UYVWWTWQVNTM", "G]HQIOKMMMNNOPOSNWKb RNNNSMWJb RNPMTIb ROTPQQORNTMVMXNYOZRZTYWWZT[R[PZOWOT RXOYQYTXWWY RVMWNXQXTWWVYT[ RFbNb RJaGb RJ`Hb RK`Lb RJaMb", "G\\WMQb RXMRb RWMYMSb RUTUQTNRMPMMNKQJTJVKYLZN[P[RZSYTWUT RMOLQKTKWLY RPMNOMQLTLWMZN[ RNbVb RRaOb RR`Pb RS`Tb RRaUb", "I[JQKOMMOMPNQPQTO[ RPNPTN[ RPPOTM[O[ RYOYNXNXPZPZNYMWMUNSPQT", "J[XPXOWOWQYQYOXNUMRMONNONQOSQTTUVVWX RONNQ RORQSTTVU RWVVZ RNOOQQRTSVTWVWXVZS[P[MZLYLWNWNYMYMX", "KYTFQQPUPXQZR[T[VYWW RUFRQQUQZ RTFVFRTQX RNMXM", "F^GQHOJMLMMNNPNSLX RMNMRLVLZ RMPKUKXLZN[P[RZTXVU RXMVUVXWZX[Z[\\Y]W RYMWUWZ RXMZMXTWX", "H\\IQJOLMNMONPPPSNX RONORNVNZ ROPMUMXNZP[R[TZVXXUYQYMXMXNYP", "CaDQEOGMIMJNKPKSIX RJNJRIVIZ RJPHUHXIZK[M[OZQXRU RTMRURXSZU[W[YZ[X]U^Q^M]M]N^P RUMSUSZ RTMVMTTSX", "G]JQLNNMPMRNSPSR RPMQNQRPVOXMZK[I[HZHXJXJZIZIY RRORRQVQY RZOZNYNYP[P[NZMXMVNTPSRRVRZS[ RPVPXQZS[U[WZYW", "G]HQIOKMMMNNOPOSMX RNNNRMVMZ RNPLULXMZO[Q[SZUXWT RYMU[T^RaPb RZMV[T_ RYM[MW[U_SaPbMbKaJ`J^L^L`K`K_", "H\\YMXOVQNWLYK[ RXOOOMPLR RVORNONNO RVORMOMMOLR RLYUYWXXV RNYRZUZVY RNYR[U[WYXV", "", "", "", "", "", "", "H\\WQVOUNSMQMNNLPKSKULXNZQ[S[VZWYXWYSYNXJWHVGSFQFNGMHNHOGQF RMPLRLVMX RVYWWXSXNWJVH RQMONNOMRMVNYOZQ[ RS[UZVXWTWMVIUGSF", "I[UMWNXOYOXNUMRMONMPLSLUMXOZR[U[XZYYXYWZU[ RNPMSMUNX RRMPNOONRNVOYPZR[ RNTTUUTTSNT RNTTT", "H\\QFNGLJKOKRLWNZQ[S[VZXWYRYOXJVGSFQF RNHMJLNLSMWNY RVYWWXSXNWJVH RQFOGNIMNMSNXOZQ[ RS[UZVXWSWNVIUGSF RLPXQ RLQXP", "G]PMMNKPJSJUKXMZP[T[WZYXZUZSYPWNTMPM RLPKSKULX RXXYUYSXP RPMNNMOLRLVMYNZP[T[VZWYXVXRWOVNTM RQFSb RRGRa RSFQb RQFSF RQbSb", "H\\TMVNXPYPYOWNTMPMMNLOKQKSLUNWPXRYSZT\\T^S_Q_O^P^Q_ RMOLQLSMUOW RPMNNMPMSNURY RYPXO", "", "", "", "", "", "", "", "", "", "", "NV", "JZ", 
  /******** Hershey Glyphs 3200 to 3299 ********/ 
  "H\\QFNGLJKOKRLWNZQ[S[VZXWYRYOXJVGSFQF RNHMJLNLSMWNY RVYWWXSXNWJVH RQFOGNIMNMSNXOZQ[ RS[UZVXWSWNVIUGSF", "H\\QHQ[ RRHRZ RSFS[ RSFPINJ RM[W[ RQZO[ RQYP[ RSYT[ RSZU[", "H\\LJLKMKMJLJ RLIMINJNKMLLLKKKJLHMGPFTFWGXHYJYLXNUPPRNSLUKXK[ RWHXJXLWN RTFVGWJWLVNTPPR RKYLXNXSYWYYX RNXSZWZXY RNXS[W[XZYXYV", "H\\LJLKMKMJLJ RLIMINJNKMLLLKKKJLHMGPFTFWGXIXLWNTO RVGWIWLVN RSFUGVIVLUNSO RQOTOVPXRYTYWXYWZT[P[MZLYKWKVLUMUNVNWMXLX RWRXTXWWY RSOUPVQWTWWVZT[ RLVLWMWMVLV", "H\\SIS[ RTHTZ RUFU[ RUFJUZU RP[X[ RSZQ[ RSYR[ RUYV[ RUZW[", "H\\MFKPMNPMSMVNXPYSYUXXVZS[P[MZLYKWKVLUMUNVNWMXLX RWPXRXVWX RSMUNVOWRWVVYUZS[ RLVLWMWMVLV RMFWF RMGUG RMHQHUGWF", "H\\VIVJWJWIVI RWHVHUIUJVKWKXJXIWGUFRFOGMILKKOKULXNZQ[S[VZXXYUYTXQVOSNQNOONPMR RNIMKLOLUMXNY RWXXVXSWQ RRFPGOHNJMNMUNXOZQ[ RS[UZVYWVWSVPUOSN", "H\\KFKL RYFYIXLTQSSRWR[ RSRRTQWQ[ RXLSQQTPWP[R[ RKJLHNFPFUIWIXHYF RMHNGPGRH RKJLINHPHUI", "H\\PFMGLILLMNPOTOWNXLXIWGTFPF RNGMIMLNN RVNWLWIVG RPFOGNINLONPO RTOUNVLVIUGTF RPOMPLQKSKWLYMZP[T[WZXYYWYSXQWPTO RMQLSLWMY RWYXWXSWQ RPONPMSMWNZP[ RT[VZWWWSVPTO", "H\\MWMXNXNWMW RWOVQURSSQSNRLPKMKLLINGQFSFVGXIYLYRXVWXUZR[O[MZLXLWMVNVOWOXNYMY RMPLNLKMI RVHWIXLXRWVVX RQSORNQMNMKNHOGQF RSFUGVIWLWSVWUYTZR[", "MXRXQYQZR[S[TZTYSXRX RRYRZSZSYRY", "MXTZS[R[QZQYRXSXTYT\\S^Q_ RRYRZSZSYRY RS[T\\ RTZS^", "MXRMQNQORPSPTOTNSMRM RRNROSOSNRN RRXQYQZR[S[TZTYSXRX RRYRZSZSYRY", "MXRMQNQORPSPTOTNSMRM RRNROSOSNRN RTZS[R[QZQYRXSXTYT\\S^Q_ RRYRZSZSYRY RS[T\\ RTZS^", "MXRFQGQIRQ RRFRTST RRFSFST RSFTGTISQ RRXQYQZR[S[TZTYSXRX RRYRZSZSYRY", "I\\MKMJNJNLLLLJMHNGPFTFWGXHYJYLXNWOSQ RWHXIXMWN RTFVGWIWMVOUP RRQRTSTSQRQ RRXQYQZR[S[TZTYSXRX RRYRZSZSYRY", "MXTFRGQIQLRMSMTLTKSJRJQK RRKRLSLSKRK RRGQK RQIRJ", "MXTHSIRIQHQGRFSFTGTJSLQM RRGRHSHSGRG RSITJ RTHSL", "E_[O[NZNZP\\P\\N[MZMYNXPVUTXRZP[L[JZIXIUJSPORMSKSIRGPFNGMIMLNOPRTWWZY[[[\\Y\\X RKZJXJUKSLR RRMSI RSKRG RNGMK RNNPQTVWYYZ RN[LZKXKULSPO RMINMQQUVXYZZ[Z\\Y", "H\\PBP_ RTBT_ RXKXJWJWLYLYJXHWGTFPFMGKIKLLNOPURWSXUXXWZ RLLMNOOUQWRXT RMGLILKMMONUPXRYTYWXYWZT[P[MZLYKWKUMUMWLWLV", "G^[BIbJb R[B\\BJb", "KYUBSDQGOKNPNTOYQ]S`Ub RQHPKOOOUPYQ\\ RSDRFQIPOPUQ[R^S`", "KYOBQDSGUKVPVTUYS]Q`Ob RSHTKUOUUTYS\\ RQDRFSITOTUS[R^Q`", "JZRFQGSQRR RRFRR RRFSGQQRR RMINIVOWO RMIWO RMIMJWNWO RWIVINOMO RWIMO RWIWJMNMO", "F_JQ[Q[R RJQJR[R", "F_RIRZSZ RRISISZ RJQ[Q[R RJQJR[R", "F_JM[M[N RJMJN[N RJU[U[V RJUJV[V", "NWSFRGRM RSGRM RSFTGRM", "I[NFMGMM RNGMM RNFOGMM RWFVGVM RWGVM RWFXGVM", "KYQFOGNINKOMQNSNUMVKVIUGSFQF RQFNIOMSNVKUGQF RSFOGNKQNUMVISF", "F^ZIJRZ[ RZIZJLRZZZ[", "F^JIZRJ[ RJIJJXRJZJ[", "G^OFObPb ROFPFPb RUFUbVb RUFVFVb RJP[P[Q RJPJQ[Q RJW[W[X RJWJX[X", "F^[FYGVHSHPGNFLFJGIIIKKMMMOLPJPHNF R[FH[ R[FI[ R[FJ[ RYTWTUUTWTYV[X[ZZ[X[VYT ROGLFIIJLMMPJOG RNFJGIK RKMOLPH RZUWTTWUZX[[XZU RYTUUTY RV[ZZ[V RH[J[", "E`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bFb R_B`BFb", "JZZBXCUERHPKNOMSMXN\\O_Qb RSHQKOONTN\\ RZBWDTGRJQLPOOSN\\ RNTO]P`Qb", "JZSBUEVHWLWQVUTYR\\O_LaJb RVHVPUUSYQ\\ RSBTDUGVP RVHUQTUSXRZP]M`Jb", "J[TFSGUQTR RTFTR RTFUGSQTR ROIPIXOYO ROIYO ROIOJYNYO RYIXIPOOO RYIOO RYIYJONOO", "F_JQ[Q[R RJQJR[R", "F_RIRZSZ RRISISZ RJQ[Q[R RJQJR[R", "F_JM[M[N RJMJN[N RJU[U[V RJUJV[V", "MWUFTGRM RUGRM RUFVGRM", "H\\PFOGMM RPGMM RPFQGMM RZFYGWM RZGWM RZF[GWM", "KZSFQGPIPKQMSNUNWMXKXIWGUFSF RSFPIQMUNXKWGSF RUFQGPKSNWMXIUF", "F^ZIJRZ[ RZIZJLRZZZ[", "F^JIZRJ[ RJIJJXRJZJ[", "G^SFKbLb RSFTFLb RYFQbRb RYFZFRb RKP\\P\\Q RKPKQ\\Q RIWZWZX RIWIXZX", "E^^F\\GXHUHQGOFMFKGJIJKLMNMPLQJQHOF R^FE[ R^FF[ R^FG[ RXTVTTUSWSYU[W[YZZXZVXT RPGMFJIKLNMQJPG ROFKGJK RLMPLQH RYUVTSWTZW[ZXYU RXTTUSY RU[YZZV RE[G[", "E`UQUNTLRKPKNLMMLPLSMUOVQVSUTTUQ ROLNMMPMSNU RRKPLOMNPNSOUPV RVKUQUSVUXVZV\\U]R]O\\L[JYHWGTFQFNGLHJJILHOHRIUJWLYNZQ[T[WZYYXYWZ RWKVQVSWU RVKXKWQWSXUZV", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 3300 to 3399 ********/ 
  "", "F^IHJIIJHIIGKFMFOGPHQKQOPRNTLUIV ROHPKPPOR RMFNGOJOPNSLU RLVOY RKVOZ RIVN[UV R\\G[H\\H\\G[FYFWGVHUJUYW[[W RWHVJVXXZ RYFXGWJWWYY", "E_GQGRHSJSLRLOKMIJIHKF RKOIK RJSKRKPIMHKHIIGKFNFPGQHRJRRQUOW RPHQJQT RNFOGPJPUOW RRISGUFWFYGZH[J\\K RYHZJ RWFXGYJZK\\K R\\KRP RYM[O\\R\\U[XYZV[S[PZJWIWHX RXNYN[P RVNYO[Q\\S RTZRZLWKW RZYXZUZRYNWKVIVHXHZI[JZIY", "F^RHPFNFLGJJINIRJVLYNZQ[T[WZYY[W RLHKJJMJRKVMYPZ RNFMGLIKMKQLUMWOYRZUZXY[W RUFRHQIPKPLQNTPURUT RQKQLUPUQ RQIQJRLUNVPVRUTSURUPTOR RUFVGXHZH RUGVHWH RTGVIXIZH[G", "E_HLHKIIKGNFRFUGWHYJ[M\\Q\\U[XYZV[S[PZJWIWHX RKHMGRGUHWIYK[N RTZRZLWKW RHKJIMHRHUIWJYL[O\\R RZYXZUZRYNWKVIVHXHZI[JZIY RPHMKLMLONSNU RMNMONQNR RMKMMOQOSNUMVKVJUJT", "F^RHPFNFLGJJINIRJVLYNZQ[T[WZYY[W RLHKJJMJRKVMYPZ RNFMGLIKMKQLUMWOYRZUZXY[W RUFRHQIPKPLQNTPURUT RQKQLUPUQ RQIQJRLUNVPVRUTSURUPTOR RUFVGXHZH RUGVHWH RTGVIXIZH[G RUNYK RYKZL\\L RXLYMZM RWMXNZN\\L", "F^MNKMJKJIKGNFQFTGXI RKHMGRGUH RJKKIMHRHXIZI[H[GZFYF RSHRIQKQMROVSWVWYV\\U]S^ RTPWSXVXYW[ RQMSOVQXSYVYYX[V]S^O^L]K\\JZJWLTLRKQ RL\\K[KWLU RO^M]L[LWMTMRLQJQIRIS RUPYL RYLZM\\M RXMYNZN RWNXOZO\\M", "E_UJTHSGQFNFKGIJHNHRIUJWLYNZQ[T[WZYY[W\\T\\Q[NYL RKHJJIMIRJUKW RZW[U[QZNYM RNFLGKIJMJRKVLXNZ RWZYXZUZQYOWM RUFRHPJOLOMPOSQTSTU RPLPMTQTR RPJPKQMTOUQUSTURVQVOUNS RTOYLZJ R\\FZJ RYG]I R\\F[GYGZHZJ[I]I\\H\\F", "F_RFPGNIMKMMNOPQQSQU RNLNMQQQR RNINKOMQORQRSQUPVNWLWJVIUHSHQIPJQIR RRFTHVHXG RQGSH RPGQHSIUIXG RRPYK RYK[N\\Q\\T[WYYVZR[ RXLZN[Q[UZW RVMWMYOZRZVYXXYVZ RVZTZRYPYNZM\\N^P_R_T^ RSZQZ RR[PZNZ", "F_PPNPLOKNJLJJKHLGOFQFTGWJYK RLHNGRGTHUI RJJKIMHQHTIVJYK[K\\J\\H[GYG RJXKYJZIYIWJVLVNWPYR\\T^ RNXOYQ\\R] RLVMWNYP\\Q]S^V^X]Y\\ZZZWYUWRVPVO RYXYWVRVQ RX]Y[YYXWVTURUPWNYNZOZP", "F_PPNPLOKNJLJJKHLGOFQFTGWJYK RLHNGRGTHUI RJJKIMHQHTIVJYK[K\\J\\H[GYG RJXKYJZIYIWJVLVNWPYR\\T^ RNXOYQ\\R] RLVMWNYP\\Q]S^V^X]Y\\ZZZWYUWRVPVO RYXYWVRVQ RX]Y[YYXWVTURUPWNYNZOZP", "E_[KZIXGUFRFOGMILKLNMQPWPYN[ RMNMOPUPV RNHMJMMNOPSQVQXPZN[L[JZ RHVJZ RGYKW RHVHXGYIYJZJXKWIWHV RNONMOKQJTJVKXMYM RUKWM RRJTKULVN RYMPQ RUOYXZY[Y RTPXXZZ RSPWYY[\\X", "G^ZSYTVTUSUQVOXLYJYH RVQVPYLYK RWTVSVRWPYNZLZJYHXGUFPFMGLHKJKLLNNQOSOTNV RLKLLOQOR RLHLJMLOOPQPSOUMWJY RMWOWRYUZXZZY RNXOXSZTZ RJYLXMXQZT[V[YZZY[W", "BbEQERFSHSJRJOIMGJGHIF RIOGK RHSIRIPGMFKFIGGIFKFMGOIPLPROUNWLYI[HZGZ RNIOLORNUMW RJZIYHY RKFMHNKNRMVLXKYJXIXF[ RNGPFRFTGVIWLWRVUUWSYQ[PZOZ RUIVLVRUV RRZQYPY RRFTHUKUSTWSYRXQXN[ RUHVGXFZF\\G]H^J_K R\\H]J RZF[G\\J]K_K R_K\\M[NZQZT[X][`X R\\N[P[T\\W^Z R_K]M\\O\\S]W_Y", "D`GQGRHSJSLRLOKMIJIHKF RKOIK RJSKRKPIMHKHIIGKFNFPGRISLSRRUQWOYL[KZIZG[ RQIRKRRQUPWOX RMZKYIY RNFPHQKQRPVNYLXJXG[ RRHSGUFWFYGZH[J\\K RYHZJ RWFXGYJZK\\K R\\KYMXNWQWTXXZ[]X RYNXPXTYW[Z R\\KZMYOYSZW\\Y", "D`PFNGLIKKKMMQMS RLLLMMOMP RLILKNONQMSLTJTISIR RPFQGWIZK[M\\P\\S[VZXXZU[R[OZIWHWGX RPGQHWJYKZL RPFPHQIWKYL[N\\P RSZQZKWJW RYYWZTZQYMWJVHVGXGZH[IZHY", "E`HQHRISKSMRMOLMJJJHLF RLOJK RKSLRLPJMIKIIJGLFOFQGRHSJSU RSWS\\R^P_M_L^L\\M[N\\M] RQHRJR\\Q^ ROFPGQJQU RQWQ\\P^O_ RSJXF RXFZI[K\\O\\R[UYXV[ RWGZK[N[O RVHXJZM[P[SZVYX RWYUVSU RQUOVMX RWZUWSVPV RV[TXSW RQWOWMX", "D`PFNGLIKKKMMQMS RLLLMMOMP RLILKNONQMSLTJTISIR RPFQGWIZK[M\\P\\S[VZX RXZU[R[OZIWHWGX RPGQHWJYKZL RPFPHQIWKYL[N\\P RSZQZKWJW RXZTZQYMWJVHVGXGZH[IZHY RTXVVXV\\Z]Z RWWXW[Z RUWVWZ[\\[^Y", "D`GQGRHSJSLRLOKMIJIHKF RKOIK RJSKRKPIMHKHIIGKFNFPGQHRJRVQXOZM[K[IZ RPHQJQVPX RNFOGPJPVOYM[ RGVIZ RFYJW RGVGXFYHYIZIXJWHWGV RRISGUFWFYGZH[J\\K RYHZJ RWFXGYJZK\\K R\\KRP RTOXYZ[]X RUOYX[Z RVNZX[Y\\Y", "E`\\H[G\\F]G]I\\KZKVISHOHKIIK RYJVHSGOGLH R]I\\JZJVGSFOFLGJIIKHNHRIUJWLYNZQ[U[XZZY\\W]T]Q\\OZNWNUOSRQSOS RLXNYQZUZYY RIUKWMXPYUYYX[W\\V]T RXOWOSSRS R]Q[OYOWPUSSTQTOSNQNOOMQL", "F_LNJMIKIIJGMFRFUGYJ[J\\I RJHLGRGUHXJ RIKJILHRHUIYK[K\\I\\G[FZG[H RUIRLQNQPSTSV RRORPSRSS RRLRNTRTTSVRWPWOVOT RJYKZJ[IZIXJVLVOWSYVZYZ[Y RLWMWSZUZ RIXJWKWMXQZT[W[ZZ\\X", "G]JHKHLILWJX RKGMHMXPZ RIILFNHNWPYRY RJXKXMYO[RYVV RTHUHVIVYX[[X RUGWHWYYZ RSIVFYHXIXXYYZY", "D`GQGRHSJSLRLOKMIJIHKF RKOIK RJSKRKPIMHKHIIGKFNFPGQHRJRRQUOW RPHQJQT RNFOGPJPUOW RRISGUFWFYG[J\\K RYHZJ RWFXGYJZK\\K RZKXKWLWNXP[R\\T RXO[Q RWMXN[P\\R\\V[XYZW[S[PZJWIWHX RTZRZLWKW RZYXZUZRYNWKVIVHXHZI[JZIY", "BcEQERFSHSJRJOIMGJGHIF RIOGK RHSIRIPGMFKFIGGIFLFNGOHPJPNOQMTKV RNHOJOONR RLFMGNJNOMSKV RNGPFSFUG RWFTGSISMTPVSWUWWVY RTMTNWSWT RWFUGTITLUNWQXTXVWXUZS[O[MZKXIWGWFX RNZKWJW RQ[OZLWJVGVFXFZG[HZGY RWFZF\\G^J_K R\\H]J RZF[G\\J]K_K R]K[KZLZN[P^R_T R[O^Q RZM[N^P_R_W^Y]Z[[X[UZ RYZXZVY R^Y\\ZZZXYWX", "F^KHMHOIPJQMQO RQQQUPXM[KZI[ RNZLYKY ROYNYLXI[ RMGPHQIRLRUSWUYWZ RIINFPGRISLSO RSQSTTWUXWYYY RQURXTZV[[X RSLTIWFYG[F RVGXHYH RUHVHXI[F RKSMOQO RSOWOYM RMPWP RKSMQQQ RSQWQYM", "E_HQHRISKSMRMOLMJJJHLF RLOJK RKSLRLPJMIKIIJGLFOFQGRHSJSORRQTQUSWTW RQHRJRPQSPUSX ROFPGQJQPPTOVRYUV RSJ[F RYGYZX] RZGZXY[ R[F[VZZY\\W^T_P_M^K\\JZKYLZK[", "F^NIOGQFTFVGWHXJXMWOVPTQ RQQOPNN RVHWIWNVO RTFUGVIVNUPTQ RMUNSORQQTQWRYTZVZZY\\W^T_P_N^KZJY RXTYVYZX\\ RTQWSXUX[W]V^T_ RO^N]LZKY RR_P^O]MZLYIYHZH\\I]J]", "", "", "", "F^IHJIIJHIIGKFMFOGPHQKQOPRNTLUIV ROHPKPPOR RMFNGOJOPNSLU RLVOY RKVOZ RIVN[UV R\\G[H\\H\\G[FYFWGVHUJUYW[[W RWHVJVXXZ RYFXGWJWWYY RP@OAOE RPAOE RP@QAOE RW@VAVE RWAVE RW@XAVE", "D`PFNGLIKKKMMQMS RLLLMMOMP RLILKNONQMSLTJTISIR RPFQGWIZK[M\\P\\S[VZXXZU[R[OZIWHWGX RPGQHWJYKZL RPFPHQIWKYL[N\\P RSZQZKWJW RYYWZTZQYMWJVHVGXGZH[IZHY RP@OAOE RPAOE RP@QAOE RW@VAVE RWAVE RW@XAVE", "G]JHKHLILWJX RKGMHMXPZ RIILFNHNWPYRY RJXKXMYO[RYVV RTHUHVIVYX[[X RUGWHWYYZ RSIVFYHXIXXYYZY RO@NANE ROANE RO@PANE RV@UAUE RVAUE RV@WAUE", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 3400 to 3499 ********/ 
  "", "J[TMQNOONPMSMVNYO[UX RNVOYPZ RQNOPNSNUOXQZ RRNSOUPUYW[ZX RSNVPVXXZ RTMUNWOXO RWPXO RWPWXXYYY", "J[LHMINK RTFQGOINKNXMY RPIOKOXRZ RTFRGQHPKPXRYSZ RMYNYPZQ[TZ RPPVMWOXRXUWXVYTZ RUNVOWQ RTNVPWSWUVXTZ", "KXRNTPVOTMRNOPNRNWOYQ[UY RSNUO RPPOROWPYQZ RQOPQPVQXSZ", "J[QFNINKOLSNVPWRWUVXTZ ROJOKSMVOWP ROHOIPJUMWOXRXUWXTZQ[ RRNNPNXMY ROPOXRZ RPOPXRYSZ RMYNYPZQ[", "KXPUVQSMOPNRNWOYQ[UY RUQRN RPPOROWPYQZ RTRROQOPQPVQXSZ", "LYXFWGUGSFQFPHPMOONP RVHTHRGQG RXFWHVITIRHQHPI RPKQMRNTOVOVP RNPPP RRPVP RPPPTQ` RSOPOQNQ[ RRPRTQ`", "J[TMQNOONPMSMVNYO[UX RNWOYPZ RQNOPNSNUOXQZ RRNSOUPUXV[V]U_ RSNVPVZ RTMUNWOXO RWPXO RWPW\\V^U_S`P`N_M^M]N]N^", "J[LHMINK RTFQGOINKNXMY RPIOKOYPZ RTFRGQHPKPXQYRY RMYOZP[SX RPPVMWOXSXWWZV\\T^Q` RUNVOWR RTNVQWTWWV[T^", "MWRFQGQHRISHSGRF RQGSH RQHSG ROOPOQPQYS[VX RPNRORXTZ RNPQMRNTO RSPTO RSPSXTYUY", "MWRFQGQHRISHSGRF RQGSH RQHSG ROOPOQPQ[P^O_M` RPNROR[Q] RNPQMRNTO RSPTO RSPS[R]P_M` RS[T]U^", "KYNHOIPK RUFSGQIPKPMOONP RPPPXOY RRIQKQM RQOPOQMQXSZ RUFSHRKRO RRPRXSYTY ROYQZR[UX RRLVIWJWLUNSO RUJVKVLUN RROWOWP RNPPP RRPWP", "MWOHPIQK RWFTGRIQKQXPY RSIRKRYTZ RWFUGTHSKSXTYUY RPYRZS[VX", "E_GOHOIPIXHYJ[ RHNJPJXIYJZKYJX RFPIMKOKXLYJ[ RNNPOQQQXPYR[ RPNQORQRXQYRZSYRX RKPNNPMRNSPSXTYR[ RVNWOYPYY[[^X RWNZPZX\\Z RSPVNXMYN[O\\O R[P\\O R[P[X\\Y]Y", "I[KOLOMPMXLYN[ RLNNPNXMYNZOYNX RJPMMOOOXPYN[ RRNSOUPUYW[ZX RSNVPVXXZ ROPRNTMUNWOXO RWPXO RWPWXXYYY", "J[NPNXMY ROPOXRZ RQOPPPXRYSZ RMYNYPZQ[TZ RNPQOVMWOXRXUWXVYTZ RUNVOWQ RTNVPWSWUVXTZ", "J[OJMLMNNQNXLZ RNYO` RNMNNOQO[ RNKNLONPQPXQXSYTZ RPYO` RSZQY RTZR[PY RNYLZ RPPVMWOXRXUWXVYTZ RUNVOWQ RTNVPWSWUVXTZ", "J[TMQNOONPMSMVNYO[UX RNWOYPZ RQNOPNSNUOXQZ RRNSOUPUXV` RSNVPV[ RTMUNWOXO RWPXO RWPWXV`", "KYNOOOPPPXOY RONQPQYSZ RMPPMRORXSYTY ROYQZR[UX RTNUPWOVMRO RUNVO", "LWXFWGUGSFQFPHPMOONP RVHTHRGQG RXFWHVITIRHQHPI RPKRP RPPPTQ` RQOPOQNQ[ RRPRTQ` RNPPP", "LXSIRLQNPONP RSISOVOVP RNPQP RSPVP RQPQXPY RROQORMRXTZ RSPSXTYUY RPYRZS[VX", "I[KOLOMPMXLY RLNNPNXPZ RJPMMOOOXQYRZ RLYMYOZP[RZUX RVMTOUPUYW[ZX RVPWOVNUOVPVXXZ RVMXOWPWXXYYY", "J[OKMMMONRNXMY RNNNOOROXRZ RNLNMOOPRPXRYSZ RMYNYPZQ[TZ RPPVMWOXRXUWXVYTZ RUNVOWQ RTNVPWSWUVXTZ", "F_KKIMIOJRJXIYK[ RJNJOKRKXJYKZLYKX RJLJMKOLRLXMYK[ RONQORQRXQY RQNROSQSXVZ RLPONQMSNTPTXVYWZ RQYRYTZU[XZ RTPZM[O\\R\\T[XZYXZ RYNZO[Q RXNZP[S[UZXXZ", "KZOOPOQPQXPXNYM[M]N_P`S`V_V^U^U_ RPNRPRXUZ RNPQMSOSXUYVZ RXYT[SZQYOYM[ RUNVPXOWMSO RVNWO", "J[OKMMMONRNXMY RNNNOOROYQZ RNLNMOOPRPXQYRY RMYOZP[SX RPPVMWOXSXWWZV\\T^Q` RUNVOWR RTNVQWTWWV[T^", "KYNPSMUNVPVRUTQV RSNUO RRNTOUQURTTSU RSUUWVYV]U_S`Q`O_N]N[OYQXWV RRVTWUY RQVTXUZU]T_S`", "JZRMPNMPMRNU RNPNROT RPNOOORPT RPNROTOVNWMWKVJTJ RQNSN RRMTNVN RNUVRWUWWVYR[ RUSVUVXUY RTSUUUXTZ RTZRYOYL[ RSZQZ RR[PZNZL[", "J[VFUGSGQFOFNHNMMOLP RTHRHPGOG RVFUHTIRIPHOHNI RNKPP RNPNTO` ROONOONO[ RPPPTO` RLPNP RPPUMWNXPXRWTSV RUNWO RTNVOWQWRVTUU RUUWVXXX[W]U_R` RUVWW RSVTVVWWYW\\V^", "J[PIOLNNMOKP RPIPXQYO[ ROONOONOXNYOZPYOX RKPNPNXMYO[ RPPUMWNXPXRWTSV RUNWO RTNVOWQWRVTUU RUUWVXXX[W]U_R` RUVWW RSVTVVWWYW\\V^", "J[TMQNOONPMSMVNYO[UX RNVOYPZ RQNOPNSNUOXQZ RRNSOUPUYW[ZX RSNVPVXXZ RTMUNWOXO RWPXO RWPWXXYYY RPGOHOL RPHOL RPGQHOL RWGVHVL RWHVL RWGXHVL", "J[NPNXMY ROPOXRZ RQOPPPXRYSZ RMYNYPZQ[TZ RNPQOVMWOXRXUWXVYTZ RUNVOWQ RTNVPWSWUVXTZ RPGOHOL RPHOL RPGQHOL RWGVHVL RWHVL RWGXHVL", "I[KOLOMPMXLY RLNNPNXPZ RJPMMOOOXQYRZ RLYMYOZP[RZUX RVMTOUPUYW[ZX RVPWOVNUOVPVXXZ RVMXOWPWXXYYY ROGNHNL ROHNL ROGPHNL RVGUHUL RVHUL RVGWHUL", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 3500 to 3599 ********/ 
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`LJKKJMJOHOGPGRHQJQJU RKLKS RHPKP RLJLRKTJU RE[GYIXKXMYNYOX RHYKYMZ RE[GZJZL[M[NZOX RLJPFTJTWUYVY RPGSJSXRYSZTYSX RPPSP RNHOHRKROOO ROQRQRXQYS[VYWX RTJXF\\J\\W]Y^Y RXG[J[X]Z RXP[P RVHWHZKZOWO RWQZQZY\\[^Y ROHOX RWHWX", "E^GIIGKFMFOGQJVUXXYY RMGOIPKVWYZ RIGKGMHOKTVVYWZY[ RVHXIZI\\H]F RWGYH[H RVHXFZG\\G]F RKOIOHPHRIQKQ RIPKP RG[IYKXNXPY RJYMYOZ RG[JZMZN[PY RKGKX RYIY[ RRLSMUNWNYM RKTMSQSST", "E_NFLGJIIKHNHRIUJWLYNZQ[S[VZXYZW[U\\R\\N[KZIXGVFUGRIOJ RJJIMISJV RNFLHKJJMJSKVLXNZ RZV[S[MYIXH RVZXXYVZSZMYKWHUG ROJOW RPJPU RQJQTPVOW RUGUZ RUMWNXNZM RUSWRXRZS", "H^KFLGMIMOKOJPJRKQMQMYJ[MZMbO` RMHNJN` RKPNP RKFMGNHOJO` ROKRIVFZJZX RVGYJYX RTHUHXKXY RRXUXXY RSYUYWZ RRZTZV[XYZX RRIR_ RRMTNVNXM RRSTRVRXS", "E_NFLGJIIKHNHRIUJWLYNZP[T[VZXYZW[U\\R\\N[KZIXGVFUGRIOJ RJJIMISJV RNFLHKJJMJSKVLXNZ RZV[S[MYIXH RVZXXYVZSZMYKWHUG ROJOW RPJPU RQJQTPVOW RUGUZ RUMWNXNZM RUSWRXRZS RP[QZRZT[X`Za[a RT\\V_XaYa RRZS[VaXbZb[a", "F^HHJFMFOGQF RKGNG RHHJGLHOHQF RMKLLKNKOIOHPHRIQKQKW RLMLU RIPLP RMKMTLVKW RJ[MYPXRXUY RLZNYRYTZ RJ[NZQZS[UY RRIQJPLPU RQKQS RRIRRQTPU RRIUGWFYGZIZLYNXOTQRR RWGXGYIYMXN RUGWHXJXMWOTQ RTQVRWSZX[Y\\Y RWTYX[Z RTQVSXYZ[\\Y", "G^UITHRGOF RVHTG RWGSFOFLGKHJJKLLMONWNYOZPZRYU RKKLLOMXMZN[O[QZS RKHKJLKOLYL[M\\O\\QYUU[ RIOJPLQUQVRVSUU RJQLRTRUS RIOIPJRLSSSUTUU RI[LYPXSXVY RKZNYRYUZ RI[MZRZU[ RWGUISL RRNPQ ROSMUKVJVJUKV", "E]JJILHOHSIVKYMZP[S[VZXYZW[U RISJVLXNYQZTZWY RJJIMIQJTLWNXQYTYWXYW[U RHIIGKFOFUGYG[F RPGTHXH RHIIHKGNGTIWIYH[F RSIRJPKPV RQKQT RRJRSQUPV RWIWX", "F^HHJFLFOGQF RKGNG RHHJGMHOHQF RKJJLIOISJVKXMZP[S[VZXYZ[\\Y RJSKVNYQZTZ RKJJNJQKTLVNXQYUYXX RUIQJPLPV RQKQT RRJRSQUPV RUIWHYFZG\\HZIZW[Y\\Y RYIZHYGXHYIYX[Z RWHXIXX RUIUY RUNXN RURXR", "G^JFKGLILOJOIPIRJQLQLXJY RLHMJMX RJPMP RNYQYSZ RJFLGMHNJNXRXUY RJYMYPZR[UYXXZX RRJUIWHYFZG\\HZIZX RYIZHYGXHYIYW RWHXIXX RRJRX RRMTNVNXM RRSTRVRXS", "E`HFIGJIJOHOGPGRHQJQJXHY RJHKJKX RHPKP RLYNYPZ RHFJGKHLJLXOXQY RHYKYNZO[QYTXVYW[YY\\X ROHRFTHTXWXYY RRGSHSX ROHQHRIRXQY RWYXZ RWHZF\\H\\X RZG[H[X RWHYHZIZXYY ROHOX RWHWX RONRN RORRR RWNZN RWRZR", "G]HIJGLFNFOGWYXZZZ RMGNHVYWZ RJGLGMHUZV[X[ZZ\\X RWFYG[G\\F RWGXHZH RVHWIYI[H\\F RH[IYKXMXNY RJYLYMZ RH[IZKZM[ RWFSO RQRM[ RLPPP RSPXP", "G^JFKGLILOJOIPIRJQLQLXJY RLHMJMX RJPMP RNYQYSZ RJFLGMHNJNXRXUY RJYMYPZR[UYXX RRJUIWHYFZG\\HZIZ^Y`WbUaQ`L` RYIZHYGXHYIYY RWHXIXXZ[ RXaV`S` RY`V_P_L` RRJRX RRMTNVNXM RRSTRVRXS", "H\\XGWIROOSMWJ[ RVKNV RZFWJUNRRMXLZ RJHLFOGUGZF RKGOHSHWG RJHNIRIVHXG RLZNYRXVXZY RMZQYUYYZ RJ[OZUZX[ZY RMPQP RTPXP", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 3600 to 3699 ********/ 
  "", "J[PRNTMVMXNZP[RYUX RMVNXOYQZ RNTNVOXQYRY RNPPPSOUNVMXOWPWXXYYY RONNOQO RTOWOVNVYWZ RMOOMPNROUPUYW[YY RMORT", "I[LHMJMXKY RNJMHNGNXQZ RLHOFOXQYRZ RKYMYOZP[RZUYWY ROPROTNUMVNXOYOWPWY RTNVOVX RROSOUPUY", "JXNONXLYMYOZP[ ROOOYQZ RPOPXRYSYQZP[ RNORNTMUNWOXO RSNTOVO RPORNTPVPXO", "IZRMPNMOMXKY RNONXQZ RRMOOOXQYRZ RKYMYOZP[RZUYWY RMHPFQIWOWY RPINHOGPIVOVX RMHUPUY", "JXNONXLYMYOZP[ ROOOYQZ RPOPXRYSYQZP[ RNORNTMWQURPU RSNVQ RPORNUR", "JWNHNXLYMYOZP[ ROHOYQZ RPHPXRYSYQZP[ RNHQGSFTGVHWH RRGSHUH RPHQGSIUIWH RKMNM RPMTM", "I[MOMXKYLYNZO[PZRYUX RNPNYPZ ROOOXQYRY RMOOORNTMUNWOYOWPW\\V_TaRbQaO`M` RSNVPV\\ RSaQ`P` RRNSOUPUZV]V_ RTaS`Q_O_M`", "I[LHMJMXKYLYNZO[ RNJMHNGNYPZ RLHOFOXQYO[ ROPROTNUMVNXOYOWPWYU[T] RTNVOVYU[ RROSOUPUYT]T`UbVbT`", "MWRFPHRITHRF RRGQHSHRG RRMQNOOQPQYS[UY RRPSORNQORPRYSZ RRMSNUOSPSXTYUY", "MWRFPHRITHRF RRGQHSHRG RRMQNOOQPQYS[T] RRPSORNQORPRYS[ RRMSNUOSPSYT]T`RbPbPaRb", "IZLHMJMXKYLYNZO[ RNJMHNGNYPZ RLHOFOXQYO[ ROPRNTMVPSROU RSNUP RRNTQ RSRTSVXWYXY RSSTTUYVZ RRSSTTYV[XY", "MWPHQJQXOYPYRZS[ RRJQHRGRYTZ RPHSFSXUYVYTZS[", "E_GOHOIPIXGYHYJZK[ RINJOJYLZ RGOIMKOKXMYK[ RKPNOPNQMSOSXUYS[ RPNRORYTZ RNOOOQPQXPYRZS[ RSPVOXNYMZN\\O]O[P[X\\Y]Y RXNZOZY[Z RVOWOYPYY[[]Y", "I[KOLOMPMXKYLYNZO[ RMNNONYPZ RKOMMOOOXQYO[ ROPROTNUMVNXOYOWPWXXYYY RTNVOVYWZ RROSOUPUYW[YY", "I[MOMXKY RNPNXQZ ROOOXQYRZ RKYMYOZP[RZUYWY RMOOORNTMUNWOYOWPWY RSNVPVX RRNSOUPUY", "I[LMMOMXKYMYMb RMNNONaO`N^ RNYOYQZ RLMNNOOOXQYRZ ROZP[RZUYWY ROZO^P`Mb ROPROTNUMVNXOYOWPWY RTNVOVX RROSOUPUY", "I[MOMXKY RNPNYPZ ROOOXQYRY RKYLYNZO[PZRYUX RMOOORNTMUNWOYOWPWb RSNVPVaU`V^ RRNSOUPU^T`Wb", "JXLOMONPNXLYMYOZP[ RMNOOOYQZ RLONMPOPXRYSYQZP[ RPOTMUNWOXO RSNTOVO RRNTPVPXO", "JZMOMSOTUTWUWY RNONS RVUVY RPNOOOSQT RSTUUUYTZ RMOPNRMTNVNWM RQNSN RPNROTOVN RWYTZR[PZNZL[ RSZQZ RTZRYOYL[ RWMVOTROWL[", "MWPHQJQXOYPYRZS[ RRJQHRGRYTZ RPHSFSXUYVYTZS[ RNMQM RSMVM", "I[KOLOMPMXKY RLNNONYPZ RKOMMOOOXQYRY RKYLYNZO[PZRYUX RUMVNXOYOWPWXXYYY RTNVOVYWZ RUMSOUPUYW[YY", "I[LMMOMXP[RYUXWX RMNNONXQZ RLMNNOOOWPXRY RUMVNXOYOWPWX RTNVOVW RUMSOUPUX", "E_HMIOIXL[NYQX RINJOJXMZ RHMJNKOKWLXNY RQMOOQPQXT[VYYX[X RPNRORXUZ RQMRNTOSPSWTXVY RYMZN\\O]O[P[X RXNZOZW RYMWOYPYX", "H[KOLONPOQSYTZV[XY RMNOOTYVZ RKOMMONPOTWUXWYXY RRSUMVNXNYM RUNVOWO RTOVPXOYM RQUN[MZKZJ[ RNZMYLY ROYMXKYJ[ RMTPT RSTVT", "I[KOLOMPMXKY RLNNONYPZ RKOMMOOOXQYRY RKYLYNZO[PZRYUX RUMVNXOYOWPW\\V_TaRbQaO`M` RTNVOV\\ RSaQ`P` RUMSOUPUZV]V_ RTaS`Q_O_M`", "I[XML[ RLONPQPTOXM RMNOOSO RLONMPNTNXM RL[PYSXVXXY RQYUYWZ RL[PZTZV[XY RNTVT", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "NV", "JZ", 
  /******** Hershey Glyphs 3700 to 3799 ********/ 
  "H\\LHLXJY RMIMXPZ RNHNXPYQZ RLHNHSGUF RSGTHVIVY RTGWIWX RUFVGXHZHXIXY RJYLYNZO[QZVYXY", "H\\OHPIQKQXOY RQIPHQGRIRYTZ ROHRFSHSXUYVY ROYPYRZS[TZVY", "H\\LHNHPGQFSGVHXH RPHRG RLHNIPIRHSG RVHVP RWIWO RXHXP RXPVPK[ RK[OYSXVXZY RNZQYVYYZ RK[PZUZX[ZY", "H\\LHMHOGPFRGVHXH ROHQG RLHNIPIRG RVHVO RWIWN RXHXOVOSPQQ RQPSQVRXRXY RWSWX RVRVY RKYMXOXQYRZ ROYQZ RKYMYOZP[RZVYXY", "H\\UFKPKUTU RVUZU[V[TZU RLPLT RMNMU RTGTXRY RUJVHUGUYWZ RUFWHVJVXXYYY RRYSYUZV[WZYY", "H\\LFLO RLFXF RMGVG RLHUHWGXF RVLUMSNOOLO RSNTNVOVY RUMWNWX RVLWMYNZNXOXY RKYMXOXQYRZ ROYQZ RKYMYOZP[RZVYXY", "H\\LHLXJY RMIMXPZ RNHNXPYQZ RLHNHRGTFUGWHXH RSGUH RRGTIVIXH RNPOPSOUNVM RSOTOVPVY RUNWPWX RVMWNYOZOXPXY RJYLYNZO[QZVYXY", "H\\KHMFPGUGZF RLGOHTHWG RKHOIRIVHZF RZFYHWKSOQRPUPXQ[ RRQQTQWRZ RUMSPRSRVSYQ[", "H\\LILO RMJMN RNINO RLINISHUGVF RSHTHVIVO RUGWHWN RVFWGYHZHXIXO RLONOVRXR RXOVONRLR RLRLXJY RMSMXPZ RNRNXPYQZ RVRVY RWSWX RXRXY RJYLYNZO[QZVYXY", "H\\LHLQJR RMIMROS RNHNQPRQR RLHNHSGUF RSGTHVIVY RTGWIWX RUFVGXHZHXIXY RJRKRMSNTOSQRUQVQ RKYMXOXQYRZ ROYQZ RKYMYOZP[RZVYXY", "LXRXPZR[TZRX RRYQZSZRY", "LXR^R\\PZRXSZS\\R^P_ RRYQZR[RY", "LXRMPORPTORM RRNQOSORN RRXPZR[TZRX RRYQZSZRY", "LXRMPORPTORM RRNQOSORN RR^R\\PZRXSZS\\R^P_ RRYQZR[RY", "LXRFQGOHQIRT RRISHRGQHRIRT RRFSGUHSIRT RRXPZR[TZRX RRYQZSZRY", "I[KIMIOHQFRHVJXJ ROIQHQGOI RKIMJOJQIRH RVJVP RWKWO RXJXP RXPTPQQPRRT RSQQRRSSQ RVPTQSRRT RRXPZR[TZRX RRYQZSZRY", "LXTFRGQIQKRMTKRIRG RRJRLSKRJ", "LXRLRJPHRFSHSJRLPM RRGQHRIRG", "F_MIOIQHRFSGUHWH RMIMN RNJNM ROION RQHSISN RRGTITM RWHUIUN RMNONUQWQZN RZN[O\\O]N RYOZP[P RXPYQZQ\\P]N RUNSNMQKQ RKQKXIY RLRLXNZOZ RMQMXPZ RUQUX RVRVW RWQWX RIYKYMZN[PZSZUXWXZ[ RXYYXZX\\Y][ RYZZY[Y RZ[[Z\\Z][", "G]XGUFOFLGKHJJJLKNLONPVPXQYSYTXW RWHUG RXGVITGRF RKLLNNOVOXPYQ RLGKIKKLMNNVNXOYPZRZTXWV[ RWYTXQXNYK[ RVZSYPYMZ RV[SZNZK[ RPBPF RPFPN RPPPX RPZP_ RTBTF RTGTN RTPTX RT[T_", "G^[BIbJb R[B\\BJb", "KYUBSDQGOKNPNTOYQ]S`Ub RQHPKOOOUPYQ\\ RSDRFQIPOPUQ[R^S`", "KYOBQDSGUKVPVTUYS]Q`Ob RSHTKUOUUTYS\\ RQDRFSITOTUS[R^Q`", "JZRFQGSQRR RRFRR RRFSGQQRR RMINIVOWO RMIWO RMIMJWNWO RWIVINOMO RWIMO RWIWJMNMO", "F_JQ[Q[R RJQJR[R", "F_RIRZSZ RRISISZ RJQ[Q[R RJQJR[R", "F_JM[M[N RJMJN[N RJU[U[V RJUJV[V", "NWSFRGRM RSGRM RSFTGRM", "I[NFMGMM RNGMM RNFOGMM RWFVGVM RWGVM RWFXGVM", "KYNGOGQFSFUGVG RNGNL ROGOL RUGUL RVGVL RNLOLQMSMULVL", "F^ZIJRZ[ RZIZJLRZZZ[", "F^JIZRJ[ RJIJJXRJZJ[", "G^OFObPb ROFPFPb RUFUbVb RUFVFVb RJP[P[Q RJPJQ[Q RJW[W[X RJWJX[X", "F^JGKGMFOGPG RJGJK RKGKK ROGOK RPGPK RJKKKMLOKPK RZFI[J[ RZF[FJ[ RTVUVWUYVZV RTVTZ RUVUZ RYVYZ RZVZZ RTZUZW[YZZZ", "E`NMPMSLUKVLXL RNMNTMU RONOTPUQU RPMPTRU RSLUMUT RTLULVMVTWU RXLWMWTXV RMUOUPVRUUTVUXVZV\\U]R]O\\L[JYHWGTFQFNGLHJJILHOHRIUJWLYNZQ[T[WZYYXYWZ", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Hershey Glyphs 3800 to 3899 ********/ 
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
  /******** Hershey Glyphs 3900 to 3999 ********/ 
  "", "J[PQMTMXP[TY RNTNXPZ ROROWRZ RRSMNNMONNO RONSNUMWOWXXY RUNVOVXUYVZWYVX RSNUPUXTYV[XY", "IZNHLFMJMXP[UYWX RNHNXPZ RNHPFOJOWRZ ROOTMWPWX RTNVPVX RRNUQUY", "KWNPNYP[RY ROPOYPZ RPOPXQYRY RNPTMVOTPRN RSNUO", "JZRMMPMXP[RZUYWY RNPNXPZ ROOOWRZ RPIPFQIWPWY RPIVPVX RPIMIPJUPUY", "KXNPNYP[RY ROPOYPZ RPOPXQYRY RNPTMWQPU RSNVQ RRNUR", "KWOIOXNYP[ RPIPXOYPZQYPX RQHQXRYP[ ROIUFWHUISG RTGVH RLMOM RQMUM", "J[MPMXP[UY RNPNXPZ ROOOWRZ RMPOOTMWPW]V_U`SaQaO`MaObQa RTNVPV]U_ RPaNa RRNUQU^T`Sa", "I[NHLFMJMXLYN[ RNHNXMYNZOYNX RNHPFOJOXPYN[ ROORNTMWPWYT]T`UbVbT` RTNVPVYU[ RRNUQUZT]", "MWRFPHRJTHRF RRGQHRISHRG RRMPOQPQXPYR[ RRPSORNQORPRXQYRZSYRX RRMTOSPSXTYR[", "MWRFPHRJTHRF RRGQHRISHRG RRMPOQPQYT] RRPSORNQORPRYS[ RRMTOSPSZT]T`RbPaPbRb", "IZNHLFMJMXLYN[ RNHNXMYNZOYNX RNHPFOJOXPYN[ ROPRNTMVPSROU RSNUP RRNTQ RRSSTTYV[XY RSSTUUYVZ RSRTSVXWYXY", "MWRHPFQJQXPYR[ RRHRXQYRZSYRX RRHTFSJSXTYR[", "E_GOHOIPIXHYJ[ RINJOJXIYJZKYJX RGOIMKOKXLYJ[ RKONNPMSOSXTYR[ RPNRORXQYRZSYRX RNNQPQXPYR[ RSOVNXM[O[X\\YZ[ RXNZOZXYYZZ[YZX RVNYPYXXYZ[", "I[KOLOMPMXLYN[ RMNNONXMYNZOYNX RKOMMOOOXPYN[ ROORNTMWOWXXYV[ RTNVOVXUYVZWYVX RRNUPUXTYV[", "JZMPMXP[UYWX RNPNXPZ ROOOWRZ RMPOOTMWPWX RTNVPVX RRNUQUY", "IZLMMOMXKYMYM_LbN` RNON` RLMNNOOOXQYRZ RNYOYQZ ROZP[UYWX ROZO_PbN` ROORNTMWPWX RTNVPVX RRNUQUY", "J[MPMXP[UY RNPNXPZ ROOOWRZ RMPOOTMWPW_XbV` RTNVPV` RRNUQU_TbV`", "KXMONOOPOXNYP[ RONPOPXOYPZQYPX RMOOMQOQXRYP[ RQOUMWOUPSN RTNVO", "JZMPMSOUURWTWX RNPNSOT ROOOSPT RUSVTVX RTSUTUY RMPSMVNTOQN RRNUN RWXQ[MYOXSZ ROYQZ", "MWRHPFQJQXPYR[ RRHRXQYRZSYRX RRHTFSJSXTYR[ RNMQM RSMVM", "I[KOLOMPMYP[UY RMNNONYPZ RKOMMOOOXRZ RVMXOWPWXXYYY RVPWOVNUOVPVYWZ RVMTOUPUYW[YY", "I[LMMOMXQ[SYWW RMNNONXQZ RLMNNOOOWRYSY RVMXOWPWW RVPWOVNUOVPVW RVMTOUPUX", "E_HMIOIXM[OYQX RINJOJXMZ RHMJNKOKWNYOY RRMPOQPQXU[WY[W RRPSORNQORPRXUZ RRMTOSPSWVYWY RZM\\O[P[W RZP[OZNYOZPZW RZMXOYPYX", "I[LONPUZV[XY RMNOOUYWZ RLONMONVXXY RXMVMVOXOXMVOSS RQUNYL[N[NYLYL[ RNTQT RSTVT", "I[KOLOMPMYP[UY RMNNONYPZ RKOMMOOOXRZ RVMXOWPW]V_U`SaQaO`MaObQa RVPWOVNUOVPV^U_ RPaNa RVMTOUPU^T`Sa", "L[RNOPOORNTMWOWSRU RTNVOVS RRNUPUSTT RRUWWW]V_U`SaQaO`MaObQa RVWV^U_ RPaNa RTVUWU^T`Sa", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 

  /******** Glyphs 4000 to 4099 ********/
  "",
  /* BEGINNING OF NON-HERSHEY GLYPHS */

  /* UGS characters designed by Bob Beach at SLAC.  Used in ISO-Latin-1,
     Hershey Symbol fonts (plus over/underbar). */
  
  /* arrowboth, prob. less good than Wolff [4001] */
  "CaKPHRKT RNMIRNW RIR[R RVM[RVW RYP\\RYT",
  /* backslash (reversed version of 720 or 2220) [4002] */
  "G]IB[b",
  /* circlemultiply [4003] */
  "H\\QJNKLMKPKRLUNWQXSXVWXUYRYPXMVKSJQJ RVMNU RNMVU",
  /* circleplus [4004] */
  "H\\QJNKLMKPKRLUNWQXSXVWXUYRYPXMVKSJQJ RRLRV RMQWQ",
  /* congruent, i.e. ~ over = [4005] */
  "E_INJLLKNKQLSMVNXNZM[K RIR[R RIV[V",
  /* 1 UNUSED SLOT [4006] */
  "",
  /* notelement [4007] */
  "F^ZJSJOKMLKNJQJSKVMXOYSZZZ RJRVR RUHP\\",
  /* overbar (width 22, no space on either side) [4008] */
  "G]G:]:",
  /* radicalex (width 22, no space on either side, displaced) [4009] */
  "E[f:~:",
  /* reflexsubset [4010] */
  "F^ZJOJMKKMJOJQKSMUOVZV RJZZZ",
  /* reflexsuperset [4011] */
  "F^JJUJWKYMZOZQYSWUUVJV RJZZZ",
  /* underline (width 22, no space on either side) [4012] */
  "G]Ga]a",
  /* underscore (total width 22, two units space on either side of bar of width 18) [4013] */
  "G]I][]",
  /* universal [4014] */
  "I[KFR[YF RMLWL",
  /* 4 UNUSED SLOTS [4015--4018] */
  "", "", "", "",

  /* UGS characters designed by Bob Beach at SLAC.  Not placed in our
     ISO-Latin-1 or Symbol fonts, but accessible via escape sequences */
  
  /* congruent, i.e., ~ over - [4019] */
  "E_IPJNLMNMQNSOVPXPZO[M RIT[T",
  /* hbar [4020] */
  "G^MFM[ RNFN[ RNPPNSMUMXNYPY[ RUMWNXPX[ RJFNF RJ[Q[ RU[\\[ RWGIO",
  /* interbang [4021] */
  "I[MLNMMNLMLLMJNIPHSHVIWJXLXNWPVQRS RSHUIVJWLWNVPTR RREQGRVSGRE RRGRO RRYQZR[SZRY",
  /* lambdabar [4022] */
  "H\\KFMFOGPHQJWXXZY[ RMFOHPJVXWZY[Z[ RRMJ[ RRMK[ RUGKP",
  /* undefined [4023] */
  /* NOTE: position of this glyph is #defined as `UNDE' in extern.h */
  "F^J[Z[ RZWJW RJSZS RZOJO RJKZK RZGJG RJCZC",
  
  /* 16 UNUSED SLOTS [4024--4039] */
  "", "", "", "", "",  "", "", "", "", "",
  "", "", "", "", "",  "",

  /* Glyphs designed by Nelson Beebe <beebe@math.utah.edu>, and distributed
     as part of his PLOT79 utility */

  /* large pointing hands [4040--4043] */
  " \204.l'm*n.l/i/a-c*[+L-G.H0H-E)K(\\*f-k.l RKgMiPiRh RR[TWNW R\\MaTaZ]` Refdead]e\\i`lckefddabT]NWOUYL\\MeKmKpJxJ}H\200D\177B RvBvDzE\177C\177BvBmCVCQB8D,J+L+[/a7eIgOgRhVi`l", " \2048v7}6z8v;uCuAwIzXy]w\\v\\t_wY{H|>z9w8v R=Y;W;T<R RIRMPMV RWHPCJCDG R>??@@C?G;H8D9A>?@@BCGPMVOUXKWHY?Y7Z4Z,\\'`$b% Rb.`._*a%b%b.a7aNbS`lZxXyIyCu?m=[=U<R;N8D", " \204?f@eCdGeHiDlAk?f@dCbP]VWUUKLHM?K7K4J,J'H$D%B R.B.D*E%C%B.B7CNCSBlDxJyLy[uame[gUgRhNiDl RG`CZCTHM RR[PWVW RYgWiTiRh Rvl}mznvluiuawcz[yLwGvHtHwE{K|\\zfwkvl", " \2048`;V<R=O=I?7C/I+X+Z,`8bQaVambvb\177a\177_z`vbv Rb\177`\200\\}ZxZpYmYeW\\XYOOMNGTBa@d>e9c8`;\\?]@a?d>e RD]JaPaW\\ RIRMTMN R=K;M;P<R R8.7'6*8.;/C/A-I*X+]-\\.\\0_-Y)H(>*9-8.",

  /* Wolff symbol glyphs (freeware, designed by Thomas Wolff
     <wolff@inf.fu-berlin.de>), used in Hershey Symbol fonts */
  
  /* arrowvertex [4044] */
  "JZR3Rk",
  /* arrowboth, prob. better than UGS [4045] */
  "E_XP[RXT RUMZRUW RIR[R RLPIRLT ROMJROW",
  /* arrowdblboth [4046] */
  "E_XP[RXT RUMZRUW RLPXP RLTXT RLPIRLT ROMJROW",
  /* arrowdbldown [4047] */
  "JZPXR[TX RMURZWU RPIPX RTITX",
  /* arrowdblleft [4048] */
  "E_LPIRLT ROMJROW RLP[P RLT[T",
  /* arrowdblright [4049] */
  "E_XP[RXT RUMZRUW RIPXP RITXT",
  /* arrowdblup [4050] */
  "JZPLRITL RMORJWO RPLP[ RTLT[",
  /* arrowhorizex [4051] */
  "E_ARcR",
  /* 13 UNUSED SLOTS [4052--4064] */
  "", "", "", "", "", "", "", "", "", "", "", "", "",
  /* carriagereturn [4065] */
  "E_LPIRLT ROMJROW RJRYRYE",
  /* 4 UNUSED SLOTS [4066-4069] */
  "", "", "", "",
  /* ellipsis [4070] */
  "G]KYJZK[LZKY RRYQZR[SZRY RYYXZY[ZZYY",
  /* emptyset [4071] */
  "D`PFMGJIHLGOGSHVJYM[P\\T\\W[ZY\\V]S]O\\LZIWGTFPF RE^^E",
  /* 6 UNUSED SLOTS [4072--4077] */
  "", "", "", "", "", "",
  /* logicaland [4078] */
  "F^J[RMZ[",
  /* 1 UNUSED SLOT [4079] */
  "",
  /* logicalnot [4080] */
  "E_JOZOZW",
  /* logicalor [4081] */
  "F^JMR[ZM",
  /* notsubset [4082] */
  "F^ZJSJOKMLKNJQJSKVMXOYSZZZ RUHP\\",
  /* omega1 [4083] */
  "F]JQLOONNMLNJQITIWJZK[M[OZQWRT RIWJYKZMZOYQW RQTQWRZS[U[WZYWZTZQYNXMWNYOZQ RQWRYSZUZWYYW RIPKNNM[M RIPKONN[N",
  /* 13 UNUSED SLOTS [4084-4096] */
  "", "", "", "", "", "", "", "", "", "", "", "", "",
  /* trademarksans [4097] */
  "H^MGMP RIGQG RUGUP RUGYP R]GYP R]G]P",
  /* trademarkserif [4098] */
  "G_MGMP RLPNP RIHIGQGQH RTGUGUP RTPVP RUGYP R]GYP R^G]G]P R\\P^P",
  /* 1 UNUSED SLOT [4099] */
  "",

  /******** Non-Hershey Glyphs 4100 to 4199 ********/

  /* Locally developed glyphs (Robert Maier) */

  /* 4 UNUSED SLOTS [4100..4103] */
  "", "", "", "",

  /* hyphen, length 10, 3 units space on either side [4104] */
  "I[MSWS",
  /* bold hyphen, same dimensions [4105] */
  "I[MRWR RMSWS",
  /* broken bar [4106] */
  "NVRBRN RRVRb",
  /* bold broken bar [4107] */
  "NWRBRN RRVRb RSBSN RSVSb",
  /* bold pipe [4108] */
  "NWRBRb RSBSb",
  /* small and large circles superimposed (H. glyphs 903, 904) [4109] */
  "G]PGMHJJHMGPGTHWJZM\\P]T]W\\ZZ\\W]T]P\\MZJWHTGPG RQKNLLNKQKSLVNXQYSYVXXVYSYQXNVLSKQK", 

  /* The inverted exclamation point and question mark are obtained from the
     non-inverted glyphs by reflection through (0,0), and shifting downward
     by 4 units. */

  /* caret (modification of Hershey glyph 2262) [4110] */
  "JZPLRITL RMORJWO",
  /* exclamdown, simplex (modification of Hershey glyph 714) [4111] */
  "MWRbRT RROSNRMQNRO",
  /* questiondown, simplex (modification of Hershey glyph 715) [4112] */
  "I[X]X^W`VaTbPbNaM`L^L\\MZNYRWRT RROSNRMQNRO",
  /* exclamdown, complex (modification of Hershey glyph 2214) [4113] */
  "MWRbS`RTQ`Rb RR`RZ RROSNRMQNRO",
  /* questiondown, complex (modification of Hershey glyph 2215) [4114] */
  "I[W^V]W\\X]X^W`VaTbQbNaM`L^L\\MZNYRWRT RQbOaN`M^M\\NZPX RROSNRMQNRO",
  /* exclamdown, duplex (modification of Hershey glyph 2714) [4115] */
  "LWRbRTQT RRbQbQT RRPSOSNRMQMPNPOQPRP RRORNQNQORO",
  /* questiondown, duplex (modification of Hershey glyph 2715) [4116] */
  "H[X]X^W`VaSbPbMaL`K^K\\LZMYOXRW RX]W]W^V`SaPaM`L^L\\MZOYRX RW_Ta ROaL_ RL[PX RRXRTQTQX RRPSOSNRMQMPNPOQPRP RRORNQNQORO",
  /* exclamdown, complex script (modification of Hershey glyph 2764) [4117] */
  "LWObPaRU ROaRU RObNaRU RTOUNTMSNTO",
  /* questiondown, complex script (modification of Hershey glyph 2765) [4118] */
  "G\\U^T]U\\V]V^U`TaQbMbJaI_I]J[KZQXSWSURTPT RMbKaJ_J]K[LZNY RTOUNTMSNTO",
  /* exclamdown, triplex (modification of Hershey glyph 3214) [4119] */
  "LWRbSaS_RW RRbRTQT RRbQbQT RQbPaP_QW RRPSOSNRMQMPNPOQPRP RRORNQNQORO",
  /* questiondown, triplex (modification of Hershey glyph 3215) [4120] */
  "H[W]W^V^V\\X\\X^W`VaTbPbMaL`K^K\\LZMYQW RM`L_L[MZ RPbNaM_M[NYOX RRWRTQTQWRW RRPSOSNRMQMPNPOQPRP RRORNQNQORO",
  /* exclamdown, triplex italic (modification of Hershey glyph 3264) [4121] */
  "LWNbObPaRT RNaOaRT RNaN`RT RNbMaM`RT RTPUOUNTMSMRNROSPTP RTOTNSNSOTO",
  /* questiondown, triplex italic (modification of Hershey glyph 3265) [4122] */
  "G\\U]U^T^T\\V\\V^U`TaQbMbJaI_I]J[KZMYQXSWSUQTPT RObJa RKaJ_J]K[LZNY RMbLaK_K]L[MZQXRWRUQT RTPUOUNTMSMRNROSPTP RTOTNSNSOTO",
  /* exclamdown, gothic english (modification of Hershey glyph 3714) [4123] */
  "LXRbSaU`S_RT RR_Q`RaS`R_RT RRbQaO`Q_RT RRPTNRMPNRP RROSNQNRO",
  /* questiondown, gothic english (modification of Hershey glyph 3715) [4124] */
  "I[X^W`VaSbQbNaM`L^L\\MZOXQW RW^V` RN`M_M[NZ RX^V]V_UaSb RQbOaN_N[OYQW RRWRTQWSWRT RRPTNRMPNRP RROSNQNRO",

  /* yen, complex (modification of Hershey glyph 2025) [4125] */
  "H]KFRQR[ RLFSQS[ RZFSQ RIFOF RVF\\F RO[V[ RLQYQ RLUYU",
  /* yen, triplex (modification of Hershey glyph 3025) [4126] */
  "G]JFQQQ[ RKFRQRZ RLFSQS[ RYGSQ RHFOF RVF\\F RN[V[ RIFKG RNFLG RWFYG R[FYG RQZO[ RQYP[ RSYT[ RSZU[ RKQYQ RKUYU",
  /* yen, simplex (modification of Hershey glyph 525) [4127] */
  "I[JFRPR[ RZFRP RLPXP RLTXT",
  /* yen, duplex (modification of Hershey glyph 2525) [4128] */
  "I\\KFRPR[S[ RKFLFSP RZFYFRP RZFSPS[ RLPYP RLTYT",
  /* yen, complex italic (modification of Hershey glyph 2075) [4129] */
  "H]NFRPO[ ROFSPP[ R]FSP RLFRF RYF_F RL[S[ RLPYP RKTXT",
  /* yen, triplex italic (modification of Hershey glyph 3075) [4130] */
  "G]MFQPN[ RNFRPO[ ROFSPP[ R\\GSP RKFRF RYF_F RK[S[ RLFNG RPFOH RQFNG RZF\\G R^F\\G ROZL[ ROYM[ RPYQ[ ROZR[ RLPYP RKTXT",

  /* bold centered dot [4131] */
  "MXRQQRQSRTSTTSTRSQRQ RRRRSSSSRRR",

  /* inverted therefore (inversion of glyph 740) [4132] */
  "E_R[SZRYQZR[ R[K\\J[IZJ[K RIKJJIIHJIK",

  /* inverted glaze (inversion of glyph 765) [4133] */
  "G]IWHVGTGQHOINKMMMONPOTUUVWWYW[V\\U]S]P\\N[M",

  /* 26 UNUSED SLOTS [4134--4159] */
  "",   "", "", "", "", "",
  "", "", "", "", "",   "", "", "", "", "",
  "", "", "", "", "",   "", "", "", "", "",

  /* versions of `dotless i', not included among original Hershey glyphs */

  /* dotless i, triplex [4160] */
  "LXQMQ[ RRNRZ RNMSMS[ RN[V[ ROMQN RPMQO RQZO[ RQYP[ RSYT[ RSZU[",
  /* dotless i, triplex italic [4161] */
  "KXLQMOOMQMRNSPSSQX RRNRRQVQZ RRPPUPXQZR[T[VYWW",
  /* dotless i, simplex [4162] */
  "NVRMR[",
  /* dotless i, duplex [4163] */
  "NWRMR[S[ RRMSMS[",
  /* dotless i, gothic German [4164] */
  "MWOOPOQPQYS[VX RPNRORXTZ RNPQMRNTO RSPTO RSPSXTYUY",
  /* dotless i, gothic English [4165] */
  "MWRMQNOOQPQYS[UY RRPSORNQORPRYSZ RRMSNUOSPSXTYUY",
  /* dotless i, gothic Italian [4166] */
  "MWRMPOQPQXPYR[ RRPSORNQORPRXQYRZSYRX RRMTOSPSXTYR[",

  /* 3 UNUSED SLOTS [4167--4169] */
  "", "", "",

  /* Floor/ceiling brackets */

  /* simplex left floor bracket [4170] */
  "KYPBPb RPbVb",
  /* simplex right floor bracket [4171] */
  "KYTBTb RNbTb",
  /* simplex left ceiling bracket [4172] */
  "KYPBPb RPBVB",
  /* simplex right ceiling bracket [4173] */
  "KYTBTb RNBTB",
  /* duplex left floor bracket [4174] */
  "KYOBOb RPBPb RObVb", 
  /* duplex right floor bracket [4175] */
  "KYTBTb RUBUb RNbUb",
  /* duplex left ceiling bracket [4176] */
  "KYOBOb RPBPb ROBVB", 
  /* duplex right ceiling bracket [4177] */
  "KYTBTb RUBUb RNBUB",

  /* Misc. extensions */

  /* bold backslash (reversed 2720) [4178] */
  "G^JB\\b[b RJBIB[b",
  /* Gothic-German `ch' ligature [4179] */
  "G_NNPPROPMNNKPJRJWKYM[QY RONQO RLPKRKWLYMZ RMOLQLVMXOZ RPHQIRK RXFUGSIRKRXQY RTISKSYTZ RXFVGUHTKTXUYVY RQYSZT[WX RTPZM[O\\S\\W[ZZ\\X^U` RYNZO[R RXNZQ[T[WZ[X^",

  /* Accents, positioned at heights of lower-case letters (for upper-case,
     displace additionally upward by 7 units, also by 2 additional units
     rightward for upper-case italic).  These are suggested by 2246--2252,
     the only accents in the original set of glyphs.  Modifications mostly
     involved vertical displacements.  The ring accent is a vertically
     displaced version of #900. */

  /* acute accent [4180] */
  "LXTEOK RTEUFOK",
  /* grave accent [4181] */
  "LXPEUK RPEOFUK",
  /* dieresis [4182] */
  "I[OHNGMHNIOH RUHVGWHVIUH",
  /* circumflex accent [4183] */
  "G]JKRFZK RJKRGZK",
  /* small circumflex accent [4184] */
  "I[LKRFXK RLKRGXK",
  /* tilde accent [4185] */
  "JZLJLHMFOFUIWIXH RLHMGOGUJWJXHXF",
  /* "F^IJIHJELDNDPETHVIXIZH[F RIHJFLENEPFTIVJXJZI[F[D", */
  /* ring accent for `a' [4186] */
  "PTQFPGPIQJSJTITGSFQF",
  /* macron (width 14, no space on either side) [4187] */
  "KYKIYI",
  /* acute accent, simplex [4188] */
  "LXTEOK", 
  /* grave accent, simplex [4189] */
  "LXPEUK",
  /* circumflex accent, simplex [4190] */
  "G]JKRFZK", 
  /* small circumflex accent, simplex [4191] */
  "I[LKRFXK", 

  /* 3 UNUSED SLOTS [4192--4194] */
  "", "", "",

  /* The following section is occupied by Japanese Hiragana and Katakana.
     It was moved to this array from the array of Hershey Oriental glyphs,
     since that array now consists entirely of Kanji, and is optional (it
     can be omitted by a compile-time option -DNO_KANJI). */

  /* Japanese period, comma, overbar, two unuseds [4195--4199] (first
     three were originally Oriental 6200--6202) */
  "D_HYGZH[IZHY", "D_H[GZHYIZI\\H^G_", "D_HOJQZP\\Q RIP\\Q", "", "",

  /******** Hershey Glyphs 4200 to 4299 (orig. Oriental 6000 to 6099) ********/
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
  /******** Hershey Glyphs 4300 to 4399 (orig. Oriental 6100 to 6199) ********/
  /* Note: 4399 is an empty space of the same width as any of the glyphs */
  /* Note: 4398 is a locally added glyph (Katakana 'vu') */
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
/* Katakana 'vu' glyph (modified version of 4302) [4398] */
"D_PFSGSIRK RPFRGRK RJIKKKPLQ RLKLQ RLKYK RJIKJOK RUKXJZL RZLWRUURXOZM[ RYKXNVRSVPYM[ R[H]J R]F_H",
/* empty Kanji glyph (space) [4399] */
"D_", 
};

#ifndef NO_KANJI
const char * const _pl_g_oriental_hershey_glyphs[NUM_ORIENTAL_HERSHEY_GLYPHS] =
{
/******** Oriental Hershey Glyphs 0 to 99 ********/
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
  /******** Oriental Hershey Glyphs 100 to 199 ********/
  "", "D_SFQFPI RSFPI RJIHHH[ RIII[ RII[I RZI[H]I\\K R[I[WZYYZ R[X[YZZ R\\I\\Y[[Z[YZWY RNMNV RNMONOV RONUN RUNUV RTNUMVNVV ROUUU", "", "D_OHPGNFN[ ROGO[ RUHVGTFT[ RUGU[ RJKHJH\\ RIKI\\ RIKZK RYKZJ\\K[M RZKZ\\ R[K[\\ RISZS RI[Z[", "", "", "", "D_QGQ\\ RRGR\\ RLGJFJP RKGKP RKGXG RWGXFZGYI RXGXP RYGYP RKKXK RKOXO RGS]S[RZS RPSNVKYH[ RQTMXJZH[ RSSTVVYXZ[[]Z RTVVXXY]Z", "D_RGSFQFQP RRGRP RIH[HYGXH RJLZLYKXL RHP\\PZOYP RQPNSJVGX RQPRQNTKVGX RRPSSTUVXXZ[\\][ RYZ[[ RTUVWYY][ RMTMZ RNTNZ RI[UW RI[K\\UW RXRUU RXRYSUU", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_NFPGQHRHRGPFNF RKLQL RPLQKSLRN RQLQXPZO[ RQYQZP[ RRLRZQ\\P\\O[MZ RSLTOVSXV[Y]X RYV[X RSLUPXTZV]X RGPOP RLPMOOPMR RMOLRKTIWGY RNPMSKVGY RZLVP RZL[MVP", "D_JHLJMLNLNKMJJH RHONO RLOMNOPMQ RMNLRJVGZ RNPLTJWGZ RRISGQFQXPZO[ RQYQZP[ RRGRZQ\\P\\O[MZ RSGSITNUQVSXV[Y]X RYV[X RSITMVQXTZV]X RZKVP RZK[LVP", "D_RHSGQFQ] RRGR] RJHKIMLNLNKMJJH RXHUM RXHYIUM RHO\\OZMXO RXO[N RGU]U[SYU RYU\\T", "", "", "", "", "", "D_IFJGKILILHKGIF RRFPFOIMNLPIUGX RQFOK RRFOLLQJTGX RGJUJ RTJUIWJVL RUJUN RVJVN RNNXN RWNXMZNYP RXNXS RYNYS RKS[S RZS[R]S[V R[SZZY[ R\\S[ZZ\\Y[WZ RKVJXIYIZJZKXKV RNVOZPZPYNV RRVSYTYTXRV RUUVVWXXXXWWVUU", "D_JFKGLJMJMILGJF ROFPGQJRJRIQGOF RWFUK RWFXGUK RLLJKJU RKLKU RKLXL RWLXKZLYN RXLXU RYLYU RQLQ] RRLR] RKPXP RKTXT RGX]X[WZX", "", "", "", "", "", "", "D_NIOHMGMRLVJYG\\ RNHNRMVKYG\\ RGNTM RSMTLVMUO RTMTZU[\\[]Z\\X R[Z\\Z\\Y RUMUYVZZZ\\X]U", "", "", "", "", "", "", "", "D_OGOLNQMTLVJY RPGPLOQNTLWJYG[ RPLQPSTUWXZ[\\][ RVWXY[[ RPLQORQTTXX[Z][ RIGXG RUGVFXGWH RVGTN RWHUN RTN\\N RYNZM\\NZP RZMXRVUSXPZL\\ R[NYRWUTXQZO[L\\", "", "D_IJOITHXGZI RRIYH RIJZI RQIQ\\ RRIR\\ RGQ]Q[OYQ RYQ\\P", "", "", "", "", "", "D_QIQ] RRIR] RLEILHN RLEMFLHHN RKI[IYGWIZH RGQ]Q[OYQ\\P", "", "", "", "D_RHSGQFQPPROS RQQQRPS RRGRRQTPTOSMR RLIIPHR RLIMJLLHR RVIXKZN[N[MYKVI RXPWRUURXOZM[ RXPYQVUSXPZM[I\\", "", "", "", "", "", "", "D_OFPENENRMVKYI[ ROFOSNVMXKZG\\ RUFVETET\\ RUFU\\ RIHYH RXHYG[HZJ RYHYN RZHZN RKMILHT RJMIT RJMYM RIR[R RZR[Q]R[U R[RZXYY R\\R[XZZYYWX", "", "", "", "", "", "D_RGSFQFQ\\ RRGR\\ RIJZJXHVJYI RGP]P[NYP\\O RQPPSNVKYH[ RQSOVLYH[ RRPTTVWXY[[]Z RTTWWZY]Z", "", "", "", "", "", "", "", "", "D_LFJKGQ RLFMGLIINGQ RLI\\IZGXI RXI[H RRIR\\ RSIS\\ RNOLNLU RMOMU RMO[OYMWO RWOZN RGU]U[SYU RYU\\T", "", "", "", "", "", "", "D_JHWFYG RJHYG RPGML RRGML RKKMMNOOOONNMKK RVJQQ RVJWKQQ RHRYQ RHRJSYQ RXOZQ[S\\S\\R[QXO RQRQ\\ RRRR\\ RLUKWIZ RLUMVIZ RVUXWYYZYZXYWVU", "D_RGSFQFQ\\ RRGR\\ RGI]I[GYI RYI\\H RLLJKJS RKLKS RKLXL RWLXKZLYN RXLXS RYLYS RKRXR RPRNUJY RQSOUJYG[ RSRTTVWYY\\Z]Y RTTVVYX]Y", "", "", "D_HHHW RHHIIIV ROJOSNWLZJ\\ ROJPJPSOWMZJ\\ RIIPFQG RIIQG RKNLQMQMPKN RGWLU RGWHXLU RTHT\\ RTHUIU\\ RUI[I R[I[XZY R[I[H\\I\\X[ZZYXX RWOXRYRYQWO", 
  /******** Oriental Hershey Glyphs 200 to 299 ********/
  "", "", "D_RHSGQFQ\\ RRGR\\ RIJ[JYHWJ RWJZI RKKLLMNNNNMMLKK RWKTP RWKXLTP RGP]P[NYP RYP\\O RQPPRNULWHZ RQROUMWJYHZ RSPTSVVXX[Z]Y RTSWVZX]Y", "", "", "", "", "", "", "", "", "D_IHQGXFZG RIHZG RQGQ[ RRGR[ RHL\\LZKYL RLLLV RMLMV RVLVV RWLWV RGQ]Q[OYQ RYQ\\P RHV\\VZUYV RG[][[YY[ RY[\\Z", "", "D_RGSFQFQ\\ RRGR\\ RGH]H[GZH RLLJKJU RKLKU RKLXL RWLXKZLYN RXLXU RYLYU RKPXP RKTXT RQTNWKYG[ RQVNXLYG[ RRTUXXZ[[]Z RRTUWYY]Z", "", "", "", "", "", "", "", "", "", "D_IHQGWFYG RIHYG RQGQ\\ RRGR\\ RHK\\K[JZK RLKLS RMKMS RVKVS RWKWS RGO]O[NZO RHS\\S[RZS RPSMWJYHZ RQTNWKYHZ RSSTUVWYY\\Z]Y RTUWWYX]Y", "D_QGQ[ RRGR[ RIHQGXFZG RIHZG RHK\\KZJYK RJMJT RJMKNKT RKNXN RXNXT RWNXMYNYT RKQXQ RJTYT RHW\\WZVYW RG[][[ZZ[", "", "", "", "", "", "D_SEQEPG RSEPG RMGKFKT RLGLT RLGWG RVGWFYGXI RWGWM RXGXM RLJWJ RLMXM RLP[PYOXP RLS[S RZS[R]S[U R[SZZY[ R\\S[ZZ\\Y[WZ RNUNZ RNUOUOZ RHVH[ RHVIVI[ RIZTZ RTVT[ RTVUVU[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IGUG RRGTFVH RTFSHJTIVIXJZL[[[\\Z[X RUGSI RZZ[Z[Y RVHTISJKTJVJXKYMZYZ[X\\U", "D_MINHLGLZM[Y[ZZYX RXZYZYY RMHMYNZWZYXZU RHQNPSO[MXLWN RWNYM", "", "", "", "", "", "", "", "", "", "", "D_RFSEQEQXPZO[ RQYQZP[ RRFRZQ\\P\\O[MZ RGH]H[GZH RJJJN RJJKKKN RKKXK RXKXN RWKXJYKYN RJNYN RHQXQ RXQXX RWQXPYQYX RGT]T[SZT RHWXW", "D_JJZJXHVJ RVJYI RGX]X[VYX RYX\\W", "", "D_IH[HYFWH RWHZG RGN]N[LYN RYN\\M RNNNRMVLXJZH[ RONORNVMXKZH[G[ RTNTZU[\\[]Z\\X R[Z\\Z\\Y RUNUYVZZZ\\X]U", "", "", "", "", "", "", "", "D_RISHQGQM RRHRM RGM]M[KYM RYM\\L RMQLTJXH[ RNRMT RMQORMUJYH[ RTQVSYWZY[Z\\Y[WXTTQ RYV[Y", "D_RIRGSFQFQI RGI]I[HZI RRLSKQKQ\\ RRLR\\ RLOJNJ[ RKOK[ RKOXO RWOXNZOYQ RXOXYWZ RYOYYX[WZUY", "D_OEPFQIRIRHQFOE RHJ\\JZHXJ[I RQJQ[ RRJR[ RIR[RYPWRZQ RG[][[YY[\\Z", "", "", "", "", "D_RHRFSEQEQH RGH]H[GZH RMJKLGO RMJNKKMGO RUJXL[O\\O\\NYLUJ RUPWOUNTQRTOWLYJZG[ RVOUQSTPWMYKZG[ RNNKPKR RNNKR RNNOQQTTWWY[[]Z RQTUWYY]Z", "", "", "", "", "D_RGRFSEQEQG RGG]G[EYG RYG\\F RLKJJJR RKKKR RKKXK RWKXJZKYM RXKXR RYKYR RKQXQ RQQQXPZO[ RQYQZP[ RRQRZQ\\P\\O[MZ RLTGY RLTMUGY RVTXVZY[Y[XZWVT", "", "", "D_RHRFSEQEQH RGH]H[GZH RLIKLIQGT RLIMJKNJPGT RJPJ\\ RKNK\\ RSIRLPPNS RSITJSLQONS RSL[L RZLYOWSUVRYO[ RYLZK[LZOXSVVTXQZO[L\\ RRNSOURVRVQUPRN RPPQSSVUXXZ[[]Z RSVVXYY]Z", "", 
  /******** Oriental Hershey Glyphs 300 to 399 ********/
  "", "", "", "", "", "", "D_RHRFSEQEQH RGH]H[GZH ROHNKLNJP RPHOKMNJPHQ RSHSNRO RTHTNSPROPN RJJGN RJJKKGN RWJXKZN[N[MZLWJ ROPKUGY ROPPQMTGY RNSXS RWRUURXOZM[ RVSWRXSTWPZM[I\\ RNSOUQXTZW[[\\][ ROUQWTYWZ][", "", "", "", "", "", "", "", "", "", "", "", "", "D_RHRFSEQEQH RHH\\HZGYH RQHOM RQHRIOM RMLNMOOPOPNOMML RUKQQ RUKVLQQ RLRUQ RLRMSUQ RUPVQWSXSXRWQUP RHJIKJMKMKLJKHJ RZJXM RZJ[KXM RKOHRIS RKOIS RXOYP[S\\S\\R[QXO RRUSTQSQ\\ RRTR\\ RGV]V[UZV", "", "D_RGRFSEQEQG RGG]G[FZG RMHNIOKPKPJOIMH RUHTK RUHVITK RJLHKH[ RILI[ RILZL RYLZK\\L[N RZLZYYZ R[L[YZ[YZWY ROLNOMQ ROLPMOOMQKR RSLSPTQWQXPWO RTLTOUPVPWOWN RNSNZ RNSOTOZ ROTTT RTTTZ RSTTSUTUZ ROYTY", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_QFQLPQNULXJZG\\ RRGRK RQFSGRNQROVLYI[G\\ RSGSKTQUTVVXY[\\][ RYY[[ RSKTPVTXWZY][", "", "", "", "", "", "", "", "", "", "", "D_MFKKIOGS RMFNGMIIPGS RKLK[ RLKL[ RTHUGSFSZT[\\[]Z\\X R[Z\\Z\\Y RTGTYUZZZ\\X]U R[KTQ R[K\\LTQ", "", "D_SFQFPHNKKNHP RRFPI RSFPJMMJOHP RSGTIVLXN[P]O RYN[O RTIWLZN]O RMOVOUNTO RLTVT RUTVSXTVW RVTU\\ RWTV\\", "", "", "", "", "", "", "", "", "D_LFKIINGR RLFMGKKGR RJMJ\\ RKKK\\ RUHVGTFTW RUGUW RPLQKOJOZP[\\[]Z\\X R[Z\\Z\\Y RPKPYQZZZ\\X]U RLRXKYK RXKYJ[KZM RYKYRXS RZKZRYTXSVR", "D_MFIOGS RMFNGLKIPGS RKMK\\ RLKL\\ RVHWGUFU[ RVGV[ RON]N[LYN\\M RP[\\[ZYX[[Z", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RSGTFRFRKSQTTUVWYY[[\\\\\\][\\Z RXYYZ[[\\[ RSGSKTQUTWWYY[Z\\Z]W RVFWGXIYIYHXGVF RKL]K[JZK", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_SFQFPHNKKNHP RRFPI RSFPJMMJOHP RSGTIVLXN[P]O RYN[O RTIWLZN]O RMPWPVOUP RNTLSL\\ RMTM\\ RMTWT RVTWSYTXV RWTW\\ RXTX\\ RM[W[", "D_SFQFPHNKKNHP RRFPI RSFPJMMJOHP RSGTIVLXN[P]O RYN[O RTIWLZN]O RLPXPVOUP RQPQ[ RRPR[ RKUYUWTVU RG[][[YY[ RY[\\Z", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 400 to 499 ********/
  "", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RUHVGTFTM RUGUM RNM\\MZKXM RXM[L ROPPRQVQXRXRVQSOP RYPVZ RYPZQVZ RN[][[YY[ RY[\\Z", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RVHWGUFU\\ RVGV\\ RQKOJOV RPKPV RPK[K RZK[J]K\\M R[K[V R\\K\\V RPP[P RPU[U", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RUHVGTFT\\ RUGU\\ RNL]L[KZL RTLSORQPTNV RTOSQQTNVLW RULVOWQYT\\W]V RVOXRZT]V RQWXWWVVW", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RQHRHPGPX RQHQW RNXWU RNXOYWU RRHWGZF\\G RRH\\G RVHVMWSXVYX[[\\\\]\\][\\Y R[Z\\[ RWGWMXSYVZX[Y\\Y]V RQO]O[NZO RQ[X[WZV[", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RRFQJOONQ RRFSGRJPNNQ RUKU\\ RVKV\\ RQK]K[IYK\\J RVQ]Q[OYQ\\P RVW]W[UYW\\V", "D_SFQFPHNKKNHP RRFPI RSFPJMMJOHP RSGTIVLXN[P]O RYN[O RTIWLZN]O RMOVOUNTO RQOQZP[ RRORZQ\\P[NZ RIS[SYRXS RMVKXHZ RMVNWHZ RTVWXYZZZZYYXWWTV", "D_KFILGQ RKFLGJLGQ RIMI\\ RJLJ\\ RMH]H[FYH RYH\\G RZHZXYZX[ RZYZZY[ R[H[ZZ\\Y\\X[VZ RNLNV RNLOMOV ROMUM RUMUU RTMULVMVU ROTUT", "", "", "", "", "", "", "", "", "", "", "", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ ROG]G[FZG RSGS[ RTGT[ RWGW[ RXGX[ RQOONO\\ RPOP\\ RPO[O RZO[N]O\\Q R[O[\\ R\\O\\\\ RP[[[", "", "", "", "", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ[ RKLK[ RMGVGUFTG RPGOLNPMR RQGPLOOMR RPLUL RTKSQRTQVOY RSLTKULTQSTQWOYL[ ROPPQQSRSRRQQOP RWIWU RWIXIXU R[F[YZZ R[F\\F\\Y[[ZZXY", "", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RSGTFRFRT RSGST RXGYFWFWT RXGXT RNL]L[KZL RNT]T[SZT RRVOZM\\ RRVSWQYM\\ RWVYX[\\\\\\\\[[YYWWV", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RPHVGZF\\G RPH\\G RUHSL RUHVISL RPKQLRNSNSMRLPK RYKUQ RYKZLUQ ROR[Q RORPS[Q RZO[P\\R]R]Q\\PZO RURU\\ RVRV\\ RQUPWNZ RQURVNZ RYU[W\\Y]Y]X\\WYU", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RXEVEVK RWEVK RXEVK ROG]G[FZG RRJRX RRJSKSX RSKZK RZKZX RYKZJ[K[X RSOZO RSSZS RSWZW ROMPLNLN\\ ROMO\\ RO[][[ZZ[", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 500 to 599 ********/
  "", "", "", "", "", "", "", "", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ[ RKLK[ ROFOV ROFPGPV RPGSG RSGSV RRGSFTGTV RPLSL RPQSQ ROVTV ROWM[ ROWPXM[ RSWT[U[UZSW RWHWU RWHXHXU R[F[YZZ R[F\\F\\Y[[ZZXY", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RMG]G[FZG ROIOSNXMZ ROIPJPSOWMZL[ RPJZJ RZJZN RYJZI[J[N RPN[N RRPR\\ RRPSQS\\ RSQ[Q R[Q[ZZ[ RZQ[P\\Q\\Z[\\Z[YZ RUQU[ RVQV[ RXQX[ RYQY[ RSU[U", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_LFKIINGR RLFMGJMGR RJMJ\\ RKKK\\ ROKPJNJNUOVRVSURT ROKOTPUQURTRR RRLOO RRLSMOO RSG]G\\F[G RXGWK RYGWK RUJUW RUJVKVW RVK[K R[K[W RZK[J\\K\\W RVO[O RVS[S RUW\\W RWZVYS[Q\\ RWZQ\\ RZY[Z\\\\]\\][\\ZZY", "", "", "", "", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ\\ RKLK\\ RTFRFPIMM RTFMM RRHYH RWHXGYHWI RXGWIUK RPJPO RPJQKQO RQKZK RZKZO RYKZJ[K[O RUKUO RVKVO RPO[O RUOPRNS RVOUPSQNS RSQTTUXUZT[ RSQUUVXVZU\\T[RZ RTSQUNV RTTRUNV RUVQXNY RUWRXNY RVPWTXVZY\\[]Z RXVZX]Z R[RXU R[R\\SXU", "", "", "", "", "", "", "", "", "", "", "D_LFKJIOGS RLFMGLJJNGS RJNJ[ RKLK[ RUHUGVFTFTH ROH[HZGYH RQHRKSKSJQH RWHVK RXHVK RMK]K\\J[K ROMOU ROMPNPU RPNYN RYNYU RXNYMZNZU RPQYQ RPTYT RQVQZR[Y[ZZYY RQVRVRYSZXZYYYW ROWMYMZNZOW RTVUXVXVWTV R[W\\Y]Y]X[W", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_SGTFRFRP RSGSP RMGLJKLIO RMGNHMJKMIO RLK[KYIWK RWKZJ RGP]P[NYP RYP\\O RPPNVLYJ[H\\ RPPQQPTOVMYK[H\\ RTPT[U\\\\\\][\\Y R[[\\[\\Z RUPUZV[Z[\\Y]V", "", "", "D_RLQOOTMWJZG\\ RRLSMRPPTNWLYI[G\\ RLGUG RRGSFUGSI RSGSLTQUTVVXY[\\][ RYY[[ RSLTPVTXWZY][", "", "", "D_MKMNLRJVGY RNLNNMQ RMKOLNPMSKVGY RUKVPWSYW[Y]X RYV[X RUKVOWRYU[W]X", "D_MGLIJLGO RMGNHLKJMGO RUFVHYL\\O]N RVHYK]N RJOWO RVOWOYOXQ RWOVWUYTZ RVXVYUZ RXOWYV[U[TZRY ROONTLXJZH[ ROOPPOTMXKZH[", "D_MHLKJOHR RMHNILMIQGS RUHVKWMYP[R\\Q RVKXNZP\\Q RQONY RQORPNY RIZWY RIZK[WY RUVWXY[Z[ZZXXUV", "", "D_NGOFMFMT RNGNT RVGWFUFUT RVGVT RHL\\LZJXL RXL[K RGT]T[RYT RYT\\S RNVLXG\\ RNVOWLYG\\ RTVWXZ[[[[ZZYTV", "", "", "", "", "", "", "D_PGQFOFOV RPGPV RUGVFTFTV RUGUV RLJJIJV RKJKV RKJYJ RXJYI[JZL RYJYV RZJZV RKPYP RGV]V\\U[V ROYMXI[G\\ ROYMZG\\ RUXWYZ[[\\\\\\\\[[ZYYUX", "D_MFNGOJPJPIOGMF RVFTK RVFWGTK RHK\\KZIXK RXK[J ROKO[ RPKP[ RTKT[ RUKU[ RJOKRLWMWMULRJO RZOWX RZO[PWX RG[][[YY[ RY[\\Z", "D_NGOFMFMV RNGNV RVGWFUFUV RVGVV RHJ\\JZIYJ RNNUN RNRUR RGV]V[UZV RNXKZG\\ RNXOYJ[G\\ RUXXZZ\\[\\[[ZZXYUX", "", "", "", "", "D_MFNGOIPIPHOGMF RVFTJ RVFWGTJ RGJ]J[HYJ RYJ\\I RIMI\\ RIMJNJ\\ RJNON RONOZN[ RNNOMPNPZO\\N[LZ RJROR RJVOV RTOTX RTOUOUX RZMZZY[ RZM[M[ZZ\\Y[WZ", "", "", "", "", 
  /******** Oriental Hershey Glyphs 600 to 699 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGIFI[ RJGJ[ RJGYG RXGYF[GZI RYGYWXYWZ RYXYYXZ RZGZYY[X[WZUY RQGQP RRGRP RJPYP", "", "D_JGHFH\\ RIGI\\ RIG[G RZG[F]G\\I R[G[XZZY[ R[Y[ZZ[ R\\G\\Z[\\Z\\Y[WZ RLKXKWJVK RNNNW RNNOOOW ROOUO RUOUW RTOUNVOVW ROVUV", "", "", "D_LGJFJTIYH[ RKGKTJXIZG\\ RKG[G RZG[F]G\\I R[G[ZZ[ R\\G\\Z[\\Z[XZ RSJTIRIRP RSJSP ROLWLVKUL RNPXPWOVP ROROY RORPSPY RPSUS RUSUY RTSURVSVY RPXUX", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HGIHJJKJKIJHHG RLMIUGUIVIYHZI[J[KZJYJULM RIZJZ RTFRFQIONNP RSFQJ RTFNP RPL[L RZL[K]M[N R[L[N R\\L[NYP RTLTPSTRVPY RULUPTTRWPYM[ RUPVTWVYY\\[]Z RWVYX]Z", "", "", "", "D_HGIHJJKJKIJHHG RLMIUGUIVIYHZI[J[KZJYJULM RIZJZ RTFRFQIOMMP RSFQJ RTFRJONMP RTFUIVKXN[P]O RYN[O RTFUHWKYM]O RQOVOUNTO RNSYS RXSYR[SZU RYSYYXZ RZSZYY[XZVY RSSS\\ RTST\\", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HGZG RYGZF\\GZK RZGYVXYWZ RYXYYXZ R[GZYY[X[WZUY RQGQIPPOTMWKYH[ RRGRKQQPTOVMXJZH[", "", "D_KGLFJFJUKVOVPUOT RKGKTLUNUOTOR RGNQKOJOK ROGZG RYGZF\\GZJ RZGYXXZW[ RYYYZX[ R[GZZY\\X\\W[UZ RTGTMSSRVPYN[ RUGUMTSSVQYN[L\\", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 700 to 799 ********/
  "", "", "", "D_NHNFOEMEMH RIGHIGJGKHKIIIG RIHRH RQHRGTIRJ RRHRJ RSHRJQK RMJMR RMJNJNR RJLRLQKPL RJOROQNPO RHRTRSQRR RITI[ RITJUJ[ RJUQU RQUQ[ RPUQTRUR[ RJZQZ RVHVU RVHWHWU R[F[YZZ R[F\\F\\Y[[ZZXY", "", "", "", "", "", "", "", "", "", "", "", "D_RHSGQFQJPQOTMWKYH[ RRGRKQQPTOVMXJZH[ RHLZL RYLZK\\L[N RZLYVXYWZ RYXYYXZ R[L[NZYY[X[WZUY", "D_LGMFKFKNJTIWG[ RLGLNKSJVIXG[ RGKPK RPKPTOYNZ ROKPJQKQTPYO[NZLY RTITZ RTIUJUZ RUJ[J R[J[Z RZJ[I\\J\\Z RUY[Y", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HHPFQG RHHQG RLHL[ RMGMZ RGKRKQJPK RHMHT RHMINIT RINPN RPNPT RONPMQNQT RIQPQ RHTQT RGWRWQVPW RG[QZ RG[H\\QZ RWHXGVFVSUXTZ RWGWSVWTZR\\ RSM[M R[M[VZZY[ RZM[L\\M\\V[ZZ\\Y[WZ", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_OGPFNFN\\ ROGO\\ RGNNN RGXNU RGXHYNU RUHVGTFTZU[\\[]Z\\X R[Z\\Z\\Y RUGUYVZZZ\\X]U R[KUP R[K\\LUP", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_RHSGQFQ\\ RRGR\\ RGO]O[MYO RYO\\N", "", "D_RHSGQFQR RRGRR RHL\\LZJXL RXL[K RLRJQJ\\ RKRK\\ RKRXR RWRXQZRYT RXRX\\ RYRY\\ RK[X[", "", "", "", "", "D_TEREQK RSERH RTEQK RGG]G[FZG RPKNJNX ROKOX ROKYK RXKYJ[KZM RYKYX RZKZX ROOYO ROSYS ROWYW RJNKMIMI\\ RJNJ\\ RJ[][[ZZ[", "", "", "D_RGSFQFQL RRGRL RGH]H[GZH RJLHKH\\ RILI\\ RILZL RYLZK\\L[N RZLZZY[ R[L[ZZ\\Y[WZ RNLOPPPPONL RULSQ RULVMSQ RLQXQWPVQ RQQQ[ RRQR[ RLUXUWTVU", "", "", "", "", "D_SEQEPJ RREPJ RSEPJ RGG]G[FZG RMJKIKT RLJLT RLJWJ RVJWIYJXL RWJWT RXJXT RLMWM RLPWP RLSWS RGV]V[UZV RMXI[G\\ RMXNYLZG\\ RUXWYZ\\[\\[[ZZXYUX", "", "", "", "", "", "", "D_LGMFKFKM RLGLM RGIQIPHOI RGLGT RGLHMHT RHMOM ROMOT RNMOLPMPT RHPOP RHSOS RKSK\\ RLSL\\ RGWQWPVOW RXFWFVIUK RXFWHUKSM RXFYIZK\\M]L RYI[K]L RTP\\P[OZP RWPW\\ RXPX\\ RSU]U\\T[U", "D_IFJGKILILHKGIF RGIHJILJLJKIJGI RNIJOHOJPJRISJTKTLSKRKONI RJSKS RRFPJNM RRFSGNM RWFUJ RWFXGUJ RUJUS RVJVS RPJPS RQJQS RPJ\\JZIYJ RQM[MZLYM RQP[PZOYP RPS\\SZRYS RRVSUQUQ\\ RRVR\\ RGW]W[VZW", "", "", "", "", "", "", "D_RHSGQFQ[ RRGR[ RRP[PYNWP RWPZO RG[][[YY[ RY[\\Z", "", 
  /******** Oriental Hershey Glyphs 800 to 899 ********/
  "", "", "", "", "D_RGSFQFQN RRGRN RRJ\\JZIYJ RLNJMJV RKNKV RKNXN RWNXMZNYP RXNXV RYNYV RKUXU RIXHZG[G\\H\\IZIX RMXN\\O\\O[MX RSXT\\U\\U[SX RXXZZ[\\\\\\\\[ZYXX", "", "", "", "", "", "", "", "", "", "", "", "", "D_MGKFKOJUIXHZ RLGLOKTJWHZG[ RLG]G[FZG RLMYM RWMXLZNXO RXMWPVRTUPYM[ RYNWRUURXOZM[J\\ RPMPOQRRTTWVYY[[\\][ RQRSUVXZZ][", "D_LGJFJRIWHZG\\ RKGKSJWIYG\\ RKG]G[EYG RYG\\F RTJUISIS[ RTJT[ RMP[PYNWP RWPZO RK[][[YY[ RY[\\Z", "", "", "", "", "", "", "D_LGJFJUIYG\\ RKGKUJXIZG\\ RKG]G[FZG RTGSK RTGUHSK RQKOJOS RPKPS RPKYK RXKYJ[KZM RYKYS RZKZS RPOYO ROSZS RTSTZS[ RUSUZT\\S[QZ RPVMZ RPVQWMZ RXVZX[Z\\Z\\Y[XXV", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IHXH RWHYG[IXK RXHVNTRRUPWMYG[ RYHWM RZHXMVQTTQWNYKZG[ RLKMMOPRTVXYZ[[]Z RLKNNQRUVXXZY]Z", "", "", "", "D_HGQG RNGOFQGOJ ROFNMMRLUJY RPGOMNQMTLVJYH[ RIKLONRPVQVQUPSMOIK RQG\\G RYGZF\\GZJ RZFYMXRWUUY R[GZMYQXTWVUYS[Q\\ RTGTJUPVSWUYX\\[]Z RTJUOVRWTYW]Z", "", "", "", "", "", "", "", "", "D_LJJIJZ RKJKZ RKXYX RKJYJ RXJYI[JZL RYJYZ RZJZZ", "", "", "", "", "", "", "", "", "", "D_RITGRFPKNOLRIV RSGQLOOLSGX RGJ]J[HYJ RYJ\\I RORMQM\\ RNRN\\ RNRYR RXRYQ[RZT RYRY\\ RZRZ\\ RN[Y[", "", "", "", "D_NGLFLM RMGMM RMGWG RVGWFYGXI RWGWM RXGXM RMLWL RGP]P[NYP RYP\\O RMPKW RNPLW RMTXT RWTXSZTXW RXTWXVZT[ RWZV[U[ RYTXYW[V\\U\\T[RZ", "", "", "D_GIGU RGIHJHU RHJLJ RLJLT RKJLIMJMT RHSLS RQGQMPROUNWLZ RRGRMQRPUNXLZI\\ RRMSQUUWX[\\][ RXX[[ RRMSPTRVUZY][ RNGYG RWGUN RVGWFYGXHVN RUN[N RZN[M]N[P R[NYRWUTXQZM\\ R\\NZRXUUXRZP[M\\", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 900 to 999 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HGHU RHGIHIU RIHLH RLHLT RKHLGMHMT RISLS RVGWFUFU\\ RVGV\\ RPK\\KZJYK ROO]O[NZO RTORROVLY RUPSSPVLYJZ RVOWRXTZW\\Y]X RXT[W]X", "", "", "", "", "", "", "", "", "", "D_NGLFLO RMGMO RMGVG RUGVFXGWI RVGVO RWGWO RMNVN RJSHRH[ RISI[ RISNS RMSNRPSOU RNSN[ ROSO[ RIZNZ RVSTRT[ RUSU[ RUSZS RYSZR\\S[U RZSZ[ R[S[[ RUZZZ", "", "", "", "", "", "", "", "D_LGMFKFKPJQ RLGLPKRIP RGIOINHMI RGMOL RGMHNOL RTHRGRMQQPS RSHSMRPPS RTHYF[G RSHTH[G RSL]L[KZL RWLWR RXLXR RMUKTK\\ RLUL\\ RLUWU RVUWTYUXW RWUW\\ RXUX\\ RL[W[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HGHU RHGIHIU RIHLH RLHLT RKHLGMHMT RISLS RWEUETH RWETH RRHPGPQ RQHQQ RQHYH RXHYG[HZJ RYHYQ RZHZQ RQKYK RQNYN RPQZQ RTQSVRXPZL\\ RTQURTVSXQZO[L\\ RXRYTZTZSXR ROV]V\\U[V RVVWYY[[\\][ RVVWXXYZZ][", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IFIM RIFJGJM RJGOG ROGOM RNGOFPGPM RJLOL RSFSM RSFTGTM RTGYG RYGYM RXGYFZGZM RTLYL RQMOPLSIUGV RQMRNOQMSJUGV RGP]P[OZP RSPVSYU[V]U RTQWSYT]U RJUJ\\ RJUKVK\\ RKVOV ROVO\\ RNVOUPVP\\ RK[O[ RSUS\\ RSUTVT\\ RTVXV RXVX\\ RWVXUYVY\\ RT[X[", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 1000 to 1099 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_JIHHHZ RIIIZ RII[I RZI[H]I\\K R[I[Z R\\I\\Z ROIONNRMT RPIPNOQMTKV RSISSTTXTYSXR RTITRUSWSXRXP RIY[Y", "D_JGHFH\\ RIGI\\ RIG[G RZG[F]G\\I R[G[\\ R\\G\\\\ RRJSIQIQOPSNVLX RRJROQSOVLX RKMYMXLWM RTSVUWWXWXVWUTSRR RI[[[", "", "D_JGHFH[ RIGI[ RIG[G RZG[F]G\\I R[G[[ R\\G\\[ RNKNV RNKOLOV ROLUL RULUV RTLUKVLVV ROUUU RIZ[Z", "", "", "", "", "", "D_JGHFH\\ RIGI\\ RIGZG RYGZF\\G[I RZGZ\\ R[G[\\ RI[Z[ RLKMNNNNMLK RPIQLRLRKPI RVJSPQSNW RVJWKVMTPQTNWLY RLQORRTVXWXWWUURSPRLQ", "", "D_JGHFH\\ RIGI\\ RIGZG RYGZF\\G[I RZGZ\\ R[G[\\ RRJSIQIQQ RRJRQ RLLXLWKVL RMPMX RMPNQNX RNQUQ RUQUX RTQUPVQVX RNWUW RI[Z[", "D_JGHFH\\ RIGI\\ RIG[G RZG[F]G\\I R[G[\\ R\\G\\\\ RKKYKXJWK RQKQW RRKRW RLQXQWPVQ RKWYWXVWW RTSUUVUVTTS RI[[[", "", "", "", "", "", "", "", "D_JFHEH\\ RIFI\\ RIF[F RZF[E]F\\H R[F[\\ R\\F\\\\ RQHPKMQKT RQHRIPMNPKT RLHMKNKNJLH RUHTK RUHVITK RLLXLWKVL RSLTOVRXTYS RTOVQYS RKOYOXNWO RNRSR RSRSU RRRSQTRTU RNTNXOYVYWXVW RNTOUOWPXUXVWVV ROUTU RI[[[", "", "", "", "", "D_RHSGQFQ[ RRGR[ RIP[PYNWP RWPZO RG[][[YY[ RY[\\Z", "D_RGSFQFQR RRGRR RIL[LYJWL RWLZK RGR]R[PYR RYR\\Q RQRL[ RQRRSL[ RI[XZ RI[K\\XZ RVVXXZ\\[\\[[ZYXWVV", "", "", "", "D_RFPFOIMMKPGU RQFOJ RRFPJNMKQGU RGI]I[GYI\\H RLPMOKNK\\ RLOL\\ RUNVMTLT[ RUMU[ RNR\\RZQYR RL[][[YY[\\Z", "D_KGLFJFJW RKGKV RGMOMNLMM RGWNV RGWHXNV RRISHQGQZR[\\[]Z\\X R[Z\\Z\\Y RRHRYSZZZ\\X]V RWHXGVFVW RWGWW RNPZK[K RZK[J]K\\M R[K[RZS R\\K\\R[TZSXR", "", "", "", "", "", "", "", "", "D_KGLFJFJV RKGKU RGLNLMKLL RGVNU RGVHWNU RSFQKOO RSFTGSIOO RRJ[J RZJ[I]J[N R[JZWYYXZ RZXZYYZ R\\J[YZ[Y[XZVY RROXOWNVO RPUXS RPUQVXS", "", "", "", "", "", "", "", "", "", "", "", "D_GGSGRFQG RJGJKIPHR RKGKKJOIQGS RNGNR ROGOR RJLSLRKQL RVGVO RVGWGWO R[F[QZR R[F\\F\\Q[SZRXQ RRSSRQRQ[ RRSR[ RIV[VYUXV RG[][[ZZ[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_NFOEMEMQ RNFNQ RVFWEUEUQ RVFVQ RHH\\HZGYH RNKUK RNNUN RGQ]Q[PZQ RMQLSJVGZ RMQNRGZ RUQVSXVZX\\Y]X RVSXU[W]X RRTSSQSQ[ RRTR[ RNWVWUVTW RI[[[YZX[", "", 
  /******** Oriental Hershey Glyphs 1100 to 1199 ********/
  "", "", "", "", "", "", "", "", "", "D_KGLFJFJV RKGKU RGLNLMKLL RGVMU RGVHWMU RSGTFRFRK RSGSK RYGZFXFXK RYGYK RNI]I\\H[I RULRPOS RULVMTOOS RULWOYQ\\S]R RWOZQ]R RSSXSWRVS RQUQ\\ RQURVR\\ RRVYV RYVY\\ RXVYUZVZ\\ RR[Y[", "", "", "", "D_KGLFJFJV RKGKU RGLNLMKLL RGVMU RGVHWMU RQFQN RQFRGRN RRGYG RYGYN RXGYFZGZN RRJYJ RRMYM ROP]P\\O[P RQPPTOV RQPRQQSOV RQS\\S R\\S[ZZ[ R[S\\R]S\\Z[\\Z[XZ RTSSVQXOY RUSTVRXOY RXSWVVXTZR[O\\ RYSXVWXUZS[O\\", "", "", "", "", "", "", "", "", "", "", "", "D_KGLFJFJV RKGKU RGLNLMKLL RGVNU RGVHWNU RSFQKON RSFTGRJON RSI\\I[HZI RQLQR RQLRMRR RRMYM RYMYR RXMYLZMZR RRQYQ ROTO[ ROTPUP[ RPU[U R[U[[ RZU[T\\U\\[ RSUS[ RTUT[ RWUW[ RXUX[ RN[][", "", "", "", "", "", "", "", "", "", "D_KGLFJFJU RKGKT RGLNLMKLL RGUNT RGUHVNT RVHVGWFUFUH ROH]H[GZH RRISLTLTKRI RXIWM RXIYJWM RPM\\M[LZM RPOPV RPOQPQV RQPYP RYPYV RXPYOZPZV RQSYS RPVZV RSVRXPZL\\ RTVSXQZO[L\\ RVVV[W\\\\\\][\\Z RWVWZX[[[\\Z]X", "", "D_KGLFJFJV RKGKV RGLNLMKLL RGVNU RGVHWNU RQFRGSITITHSGQF RZFXFWI RZFWI RUJUR RVJVR RQJOIOR RPJPR RPJ[J RZJ[I]J\\L R[J[R R\\J\\R RPN[N ROR\\R RQTQ\\ RQTRUR\\ RRUYU RYUY\\ RXUYTZUZ\\ RRXYX RR[Y[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_RHSGQFQ[ RRGR[ RGN]N[LYN RYN\\M RI[[[YYW[ RW[ZZ", "D_QFOFMJKMGR RPFNI RQFOIKNGR ROIXI RWHUKROORKUHW RVIWHXIUMQQMTHW RNJPMSPWS[U]T RWR[T RPMTPXR]T RQTRUSWTWTVSUQT RMYQZT[V\\W\\W[SZMY", "D_KIMGKFJKIOHR RLGKKJNGT RKKQK RPJOONRLVJY ROKPJQKPOORNTLWH[ RJNJQKULWNYPZT[\\[]Z RJQKTLVNXPYTZ]Z RVGTFTQSTRV RUGUQTTQW RUGYG RXGYF[GZI RYGYUZV]V]T]R RZGZT[U\\U]T", "D_RFPFMJJM RRFNJJMGO ROIXI RWHTLRNNQJSGT RVIWHXISNOQKSGT ROIPKRNUQXS[T]S RRNUPYR]S RLTL\\ RLTMUM\\ RMUVU RVUV\\ RUUVTWUW\\ RM[V[", "", "", "", "D_PIRGPFOJMNKQHT RQGPJNNLQHT RPJ[J RYJZI\\KYM RZIXOVSSWOZM[ R[JYPXRVUSXPZM[I\\ ROMPNRQSSTSTRSPQNOM", "D_MFLJKMJOHR RMFNGMKLMJPGS RMKTK RQKRJTKSL RRKQOPROTMWJZG\\ RSLRPQSOVLYI[G\\ RKOMQNSOSORNQKO RXHYGWFW\\ RXGX\\ RTKUNWQYS\\U]T RTKUMXQ[S]T", "D_SFQFNIJL RSFOILKHM RPHXH RWHXGZIXJ RXHTLPOLQ RYIVLSNOPLQHR RNKOLPNQNQMPLNK RUPSPPSMUKVHW RUPSRPTLVHW RSR[R RZR[Q]S[T R[RWVTXPZM[H\\ R\\SYVVXRZN[H\\ RQURVSXTXTWSVQU", "D_QGRFPFOHMKJNGP RQGOJLMIOGP RPHYH RWHXGZIWK RXHVKSOPRLUIW RYIUNQRMUGX RNLOMPOQOQNPMNL RRSPRP\\ RQSQ\\ RQS[S RZS[R]S\\U R[S[\\ R\\S\\\\ RQ[[[", "D_QFQLPQNULXJZG\\ RRGRK RQFSGRNQROVLYI[G\\ RGM]M[KYM RYM\\L RSMTQUTVVXY[\\][ RYY[[ RSMTPVTXWZY][", "D_SGQFQLPQNULXJZG\\ RRGRK RSGRNQROVLYI[G\\ RGM]M[KYM RYM\\L RSMTQUTVVXY[\\][ RYY[[ RSMTPVTXWZY][ RPVQWRZSZSYRWPV", "", "", "", "", "", "", "", "", "", "", "", "", "D_PFOJKV RQGPJ RPFRGQIONLV RGL]L[JYL RYL\\K RVLUPSTQWOYL[H\\ RWLVO RXLVQUSSVPYM[H\\ RNTRUTVWX[[\\[\\Z[YXWVVSUNTLT", "", "", "", "D_LFKKIRHU RLFMGLKJRIU RGKPK RPKOPNSMUKXG\\ ROKPJQKPPOSNULXJZG\\ RLUOXPZQZQYOWLUJT RTITZ RTIUJUZ RUJ[J R[J[Z RZJ[I\\J\\Z RUY[Y", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 1200 to 1299 ********/
  "", "", "", "", "", "", "", "", "D_KFHT RKFLFIT RGKNK RNKMPLSJWG[ RMKNJOKNPMSLUJXG[ RKUMXN[O[OZMWKUIS RVFSN RVFWGSN RQN[M RQNRO[M RYK[M\\O]O]N\\MYK RTSRRR[ RSSS[ RSSZS RYSZR\\S[U RZSZ[ R[S[[ RSZZZ", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_JGXG RWGXFZHWI RXGWI RYGWITK RSLTKRKRWQZP[NZ RRYRZQ[ RSLSZR\\Q\\P[ RGQ]Q[OYQ RYQ\\P", "", "", "D_GH]H[FYH\\G ROIRGPFNKKQHV RQGNMLPHV RLPL\\ RMNM\\ RNLZL RYLZK\\MYN RZLYN R[LYNVP RUPUZT[ RUPVPVZU\\T[RZ ROT]T[RYT\\S", "", "", "", "D_KFLGMININHMGKF RPFQGRISISHRGPF RYFWFVI RYFVI RIIHKGLGMHMIKII RIJ[J RZJ[I]KZL R[JZL R\\JZLYM RLNVN RUNVMXOUP RVNUP RWNUPRR RQRQXPZO[ RQYQZP[ RQRRRRZQ\\P\\O[MZ RGT]T[RYT RYT\\S", "", "", "", "", "", "", "", "", "D_RHRGSFQFQH RIHHJGKGLHLIJIG RIH[H RZH[G]IZJ R[HZJ R\\HZJYK RKMYMWKUMXL RQMQXPZO[ RQYQZP[ RRMRZQ\\P\\O[MZ RGS]S[QYS\\R", "D_RHRGSFQFQH RIHHJGKGLHLIJIG RIH[H RZH[G]IZJ R[HZJ R\\HZJYK RKLWL RVLWKYMVN RWLVN RXLVNSP RRQSPQPQXPZO[ RQYQZP[ RRQRZQ\\P\\O[MZ RGT]T[RYT RYT\\S", "", "", "", "", "", "", "", "", "", "D_RHRGSFQFQH RIHHJGKGLHLIJIG RIH[H RZH[G]IZJ R[HZJ R\\HZJYK RRKSJQJQ[ RRKR[ RLOJNJ\\ RKOK\\ RKOXO RWOXNZOYQ RXOX\\ RYOY\\ RKUXU RK[X[", "", "", "", "", "D_RHRGSFQFQH RIHHJGKGLHLIJIG RIH[H RZH[G]I[J R[H[J R\\H[JZK RIMZMXLWM RQMQZ RRMRZ RRSZSXRWS RMOKTIXG[ RMONPLTJWG[ RKULWOZS[\\[]Z RLWOYSZ]Z", "D_RHRGSFQFQH RIGHIGJGKHKIIIG RIH[H RZH[G]I[J R[H[J R\\H[JZK RQJQQPUOWMYJ[G\\ RRJRRQUPWNYK[G\\ RSTUXWZY[\\\\][ RSTUWWYYZ][ RJLZLYKXL RJPZPYOXP RGT]T[SZT", "", "", 
  /******** Oriental Hershey Glyphs 1300 to 1399 ********/
  "D_RHRGSFQFQH RIGHIGJGKHKIIIG RIH\\H RZH[G]I[J R[G[J R\\H[JZK RJLZLYKXL RQLOP RQLRMOP RHQXP RHQJRXP RUNWOZR[R[QZPXOUN RRTSSQSQ[ RRTR[ RJVZVYUXV RG[][[ZZ[", "", "", "", "", "", "", "", "", "", "", "D_RHRGSFQFQH RIHHJGKGLHLIJIG RIH[H RZH[G]I[J R[H[J R\\H[JZK RJLZLYKXL RQLONKQ RSLOOKQHR ROOQRRVRXQZP[ RRYRZQ[ ROOQQRSSVSZR\\Q\\P[NZ RQQKU RQRKUHV RRUOWKY RRVPWKYHZ RSMTQUTVVYY[Z]Y RTQUSWVZX]Y RZOVS RZO[PVS", "", "", "", "", "D_RHRGSFQFQH RIGHIGJGKHKIIIG RIH[H RZH[G]I[J R[G[J R\\H[JZK RMLMRNSWSXRWQ RMLNLNQORVRWQWO RIMHOGPGQHQIOIM RPIQLRLRKPI RZN[O\\Q]Q]P\\OZN RVKSNOQKS RVKWLSOPQKSGT RQUQ[ RQURUR[ RIVI\\ RIVJVJ\\ RJ[Y[ RYVY\\ RYVZVZ\\", "", "", "", "", "", "D_RHRGSFQFQH RIGHIGJGKHKIIIG RIH[H RZH[G]I[J R[H[J R\\H[JZK ROKPJNJNR ROKOR RUKVJTJTR RUKUR RJLZLYKXL RJOZOYNXO RGR]R[QZR RNRJWGZ RNROSLVGZ RURVUWWYY\\Z]Y RVUXWZX]Y RQUSWTWTVQU ROYRZU\\V\\V[TZOY", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IEJFKHLHLGKFIE RGLKL RJLKKMLLN RKLKQGTHULQ RLLLQ RLQNTPU\\U]T RLQNSPT]T RRESHTHTGRE RZEXEWH RZEWH ROH]H\\G[H RUHTK RUHVITK RPJPR RPJQKQR RQKZK RZKZR RYKZJ[K[R RQMZM RQOZO RQQZQ RVUV[U\\ RWUW[V]U\\S[ RGX]X[WZX RLYMZN\\O\\O[NZLY", "D_SHTGRFRWQZP[ RRYRZQ[ RSGSZR\\Q\\P[NZ RMLLPJTHW RMLNMMPLRJUHW RVMXOZR[V\\V\\T[RYOVM", "", "", "D_RHSGQFQP RRGRP RJILKMNNNNMMKJI RXITO RXIYJTO RGP]P[NYP RYP\\O ROPORNVLYJ[H\\ RPPPROVMYK[H\\ RSPS[T\\\\\\][\\Y R[[\\[\\Z RTPTZU[Z[\\Y]V", "D_KGLIMLNLNKMIKG RRHSGQFQN RRGRN RXGTM RXGYHTM RINYN RXNYM[NZP RYNY[ RZNZ[ RITYT RIZYZ", "", "", "", "", "D_RGSFQFQJ RRGRJ RKFMININHKF RWFTJ RWFXGTJ RIIHKGLGMHMIKII RIJ[J RZJ[I]K[L R[I[L R\\J[LZM RKLKQ RKLLMLQ RLMWM RWMWQ RVMWLXMXQ RKQXQ RQQQ\\ RRQR\\ RJSJ[ RJSKTK[ RKTXT RXTXYWZ RWTXSYTYYX[WZUY", "", "", "", "", "", "", "", "", "", "", "", "", "D_MGKFKTJXIZ RLGLTKWIZG\\ RLGWG RVGWFYGXI RWGWN RXGXN RLMWM RQNRQSSUVXY[[]Z RRQTTWWZY]Z", "", "", "", "", "", "D_JFJSIXHZ RJFKGKSJWHZG[ RKGZG RZGZK RYGZF[G[K RKK[K RMPTNWMYN RMPYN RRORZS[\\[]Z\\Y RSOSYTZ[Z\\Y]W RMTZRYQXR RLX[VZUYV", "", "", "D_LGJFJTIYH[ RKGKTJXH[G\\ RKGZG RYGZF\\G[I RZGZM R[G[M RKLZL RTNT[ RTNUNU[ ROOOT ROOPOPT RPSYS RYOYT RYOZOZT RNVN\\ RNVOVO\\ RO[Z[ RZVZ\\ RZV[V[\\", "D_LGJFJTIYH[ RKGKTJXH[G\\ RKGZG RYGZF\\G[I RZGZM R[G[M RKLZL RTNUMSLSU RTMTU RKP]P[OZP RPUNTN\\ ROUO\\ ROUYU RXUYT[UZW RYUY\\ RZUZ\\ RO[Y[", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 1400 to 1499 ********/
  "", "", "D_IFIVHZG\\ RIFJGJVIYG\\ RJG[G R[G[J RZG[F\\G\\J RJJ\\J RQJRMSMSLQJ RWJUM RWJXKUM RTMTS RUMUS RMLMS RMLNMNS RNM[M R[M[S RZM[L\\M\\S RNP[P RMS\\S ROUO\\ ROUPVP\\ RPVYV RYVY\\ RXVYUZVZ\\ RPYYY RO\\Z\\", "", "", "", "", "D_RHSGQFQY RRGRY RJLKKIJI[ RJKJ[ RJYYY RZL[KYJY[ RZKZ[", "", "", "", "", "", "", "", "", "", "", "D_RERI RRESESI RJFJI RJFKFKI RJI[I RZFZI RZF[F[I RJLJUIYG\\ RJLKMKVJYG\\ RKM\\MZLYM RSOSRRVQXOZK\\ RSOTOTRSVRXPZN[K\\ RTRUUWXZ[\\\\][ RWXZZ][ RPQNTNUOUPQ RZRYQUU RZRUU", "", "", "", "", "", "", "", "", "", "", "", "", "D_QEQI RQERERI RIFII RIFJFJI RYFYI RYFZFZI RIIZI RKLIKIWHZG\\ RJLJWIZG\\ RJLYL RXLYK[LZN RYLYUZY[[\\\\]\\]Z]X R[Z\\[][ RZLZU[Y\\Z]Z RMOTNVO RMOVO RQOQY RRORY RMQMV RMQNRNV RNRUR RURUV RTRUQVRVV RMVVV RLZUY RLZN[UY RTXUYV[W[WZVYTX", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KHLGJFJUIYG\\ RKGKVJYG\\ RSITHRGRY RSHSY R[H\\GZFZ\\ R[G[\\", "", "", "", "D_IH[HYFWH RWHZG RQHQZ RRHRZ RGZ]Z[XYZ RYZ\\Y", "", "", "", "D_QISGQFOLMQKUG[ RRGQJOOMSKVG[ RGJ]J[HYJ RYJ\\I RMR[RYPWR RWRZQ RSRS[ RTRT[ RK[][[YY[ RY[\\Z", "", "", "", "D_GIOINHMI RKIKU RLILT RGUOT RGUHVOT RPG]G[FZG RVGUK RVGWHUK RSKQJQW RRKRW RRKZK RYKZJ\\K[M RZKZW R[K[W RROZO RRSZS RQW[W RSXO[M\\ RSXTYRZM\\ RXX\\\\]\\][\\ZXX", "", "", "", "", "", "", "D_JFKGLIMIMHLGJF RQHSGQFOKMOKR RRGPKNNKRHU RWGVFTJ RWGTJ RHJ\\JZIYJ RTKUMWPYR\\T]S RUMXP[R]S RGN]N[MZN RNRUR RURUW RTRUQVRVW RNVUV RMUMZN[Z[[ZZY RMUNVNYOZYZZYZW", "", "D_PFNJKOHS RPFQGMMJQGT RFJ]J[HYJ RYJ\\I RSMTLRLR\\ RSMS\\ RMPKOKZ RLPLZ RLPYP RXPYO[PZR RYPYVXXWY RYWYXXY RZPZXYZXZWYUX", "D_LHMGKFK\\ RLGL\\ RGJGW RGJHKHW RHKOK ROKOUNV RNKOJPKPUOWMU RSFSTRXQZO\\ RSFTGTTSXRZO\\ RTGYG RYGYZZ[][]Y]W RXGYFZGZY[Z\\Z]Y RUNWRXRXQUN", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_LHMGKFK\\ RLGL\\ RGJGW RGJHKHW RHKOK ROKOUNV RNKOJPKPUOWMU RRG]G\\F[G RTJTP RTJUKUP RUKZK RZKZP RYKZJ[K[P RUOZO RWSW[ RXSX[ RSRS\\ RSRTST\\ RTS[S R[S[\\ RZS[R\\S\\\\ RTW[W RT[[[", "", "", "", "", "", "", "", "D_IG[GYEWG RWGZF RQGQ] RRGR] RGP]P[NYP RYP\\O", "", "", "", "D_RGSFQFQORTTXW[Y\\\\\\][\\Z RRGROSTUXWZY[[[\\Z]W RKFJJ RKFLGJJ RHIJLKLKKHI RNIJO RNIOJJO RGONO RGOHPNO RNMOPPPPONM RWFVJ RWFXGVJ RTIVLWLWKTI RZIVO RZI[JVO RSOZO RSOTPZO RZM[P\\P\\OZM RVPWRXRXQVP RGS]S[RZS RKSKVJYH\\ RLSLVKYH\\ RLVMWNYOYOXNWLV RXUUXRZN\\ RXUYVXWUYQ[N\\", "", "", "", 
  /******** Oriental Hershey Glyphs 1500 to 1599 ********/
  "", "", "", "", "D_SISGTFRFRI RKIIHITHYG[ RJIJTIXG[ RJI]I[GYI RYI\\H RRPSOQNQZR[X[YZXY RRORYSZWZXYXW RNTMVLWLXMXNVNT RTLUMVOWOWNVMTL RZT[U\\W]W]V\\UZT", "", "", "", "D_SHSGTFRFRH RKHIGITHYG[ RJHJTIXG[ RJH]H[GZH RPLNKNX ROLOX RPLYJ[K RNLOL[K RUKUQVUXXZZ\\[]Z\\Y RZY\\Z RVKVQWUYX[Y\\Y]W ROQ]Q[PZQ RKYTV RKYLZTV RO[W[VZU[", "", "", "D_SHSGTFRFRH RIGIUHYG[ RIGJHJUIXG[ RJH]H[GZH ROIOQ ROIPIPQ RUIUQ RUIVIVQ RJL]L[KZL RPPUP RMSYS ROSQVSXVZX[\\\\][ RSXUYXZ][ RXRTVPYM[ RXRYSTWQYM[J\\", "", "", "D_SHSGTFRFRH RIGIUHYG[ RIGJHJUIXG[ RJH]H[GZH RJKPK ROJMR RNKOJPKNR RMRRR RQQPUOWMZ RPRQQRRQUOXMZJ\\ RMTLULW RMTLW RMTNWPZS[\\[]Z RNWPYSZ]Z RSLWKZJ\\K RSL\\K RWKWW RXKXW RSQ]Q\\P[Q RSW]W\\V[W", "D_SHSGTFRFRH RHGHUGYF[ RHGIHIUHXF[ RIH]H[GZH RSKTJRJR[ RSKS[ RNKMNLPJS RNKOLNNLQJS RMPNQOSPSPROQMP RXKWNVPTS RXKYLXNVQTS RXPYQZS[S[RZQXP RKV[VZUYV RI[][[ZZ[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_SGTFRFRKSQTTUVWYY[[\\\\\\][\\Z RXYYZ[[\\[ RSGSKTQUTWWYY[Z\\Z]W RVFWGXIYIYHXGVF RGK]K[JZK RGPPPOONP RKPKY RLPLY RGZRW RGZH[RW", "", "", "", "D_IGXG RWGXFZGYI RXGXN RYGYN RKMXM RLMJLIT RKMJT RJSZS RYSZR\\SZU RZSXYWZU[ RXZW[V[ R[SYYX[W\\V\\U[RZ", "", "D_HGPG ROGPFRGQI RPGPM RQGQM RJLPL RIKHR RIKJKIR RIQQQ RPQQPSQQT RQQOYNZL[ ROZN[M[ RRQPYO[N\\M\\L[JZ RZH[GYFY\\ RZGZ\\", "", "", "", "", "D_GGLG RLGLM RKGLFMGMM RGKGS RGKHLHS RHLLL RHRLR RLRKZJ[ RKRLQMRLZK\\J[HZ RPHPUOZN\\ RPHQIQUPYN\\ RQIZF\\G RQI\\G RZGZT[Z\\\\][ RZT[X\\Z][ RUJUZ RUJVJVY RRZXY RRZS[XY RXWY[Z[ZZXW", "D_GGLG RLGLM RKGLFMGMM RGKGS RGKHLHS RHLLL RHRLR RLRKZJ[ RKRLQMRLZK\\J[HZ RVGWFUFUJ RVGVJ RPJ]J[IZJ RUJRQ RUJVKRQ RPPRRSTTTTSSRPP RYOS[ RYOZPS[ RO[[Z RO[Q\\[Z RYX[Z\\\\]\\][\\ZYX", "", "", "D_HGLG RLGLM RKGLFMGMM RILLL RHKHS RHKILIS RIRLR RLRKZJ[ RKRLQMRLZK\\J[HZ RUFSL RUFVGSL ROL[K ROLQM[K RYI[K\\M]M]L\\KYI RULU[ RVLVZ RRPPOPV RQPQV RQPZP RYPZO\\P[R RZPZV R[P[V RQUZU RO[[Z RO[Q\\[Z RYX[Z\\\\]\\][\\ZYX", "", "", "", "D_HGMG RMGMM RLGMFNGNM RILML RHKHS RHKILIS RIRMR RMRLZK[ RLRMQNRMZL\\K[IZ RQFRISISHQF RUFVIWIWHUF R[FXK R[F\\GXK RQKQU RQKRLRU RRL[L R[L[U RZL[K\\L\\U RVLV\\ RWLW\\ RRP[P RRT[T RPX]X[WZX", "", "", "", "", "", "", "D_MHNGLFLTKXJZH\\ RMGMULXKZH\\ RINHPGQGRHRIPIM RQGYG RYGYM RXGYFZGZM RQJYJ RQMZM RQOPQOROSPSQQQO RQP[P RZP[O]Q[R R[P[R R\\P[RZS RUPU\\ RVPV\\ RQTQ[ RQTRUR[ RRUYU RYUYYXZ RXUYTZUZYY[WY", "", "", "", "", "", "", "D_KGKSJWIYG\\ RLGLSKWIZG\\ RPGPZQ[ RQGQ[ RHGUGTFSG RHOUOTNSO R\\FYIVK R\\F]G\\HYJVK R\\MYPVRTS R\\M]N\\OYQWRTS R\\TYWUZ R\\T]U\\VYXUZR[", "", "", "", "", "", "", "", "", "D_LFHJ RLFMGHJ RLKIPGS RLKMLJPGS RJPJ\\ RKOK\\ RRGPFPKONNO RQGQKPNNO RQGVG RUGVFXGWI RVGVMWN\\N]M\\L RWGWLXM[M\\L]J RNQWQ RWQVSTVQYN[ RXQVT RVQWPYQVUSXPZL\\ RPQQSSVUXXZ[[]Z RQSTVWXYY]Z", "", 
  /******** Oriental Hershey Glyphs 1600 to 1699 ********/
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
  /******** Oriental Hershey Glyphs 1700 to 1799 ********/
  "", "", "", "", "", "", "", "", "", "", "D_SFTERERJSNUQWSYT\\U]T\\S RXS[T\\T RSFSJTNVQWRZS\\S]Q RVEWGXGXFVE RGI]I[HZI RXLVOSRPT RXLYMVPRSPT RIKIP RIKJLJP RJLNL RNLNP RMLNKOLOP RJONO RGSOR RGSHTOR RMVM[N\\V\\W[VZ RMVNVNZO[U[VZVX RJWIYHZH[I[JYJW RQVRXSXSWQV RYWZX[Z\\Z\\Y[XYW", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_MGNFLFLT RMGMT RHJQJPIOJ RLJKMJOHR RLMJPHR RONPPQPQOON RTFTT RTFUGUT RUG[G R[G[T RZG[F\\G\\T RUK[K RUO[O RUS[S RMVM[N\\V\\W[VZ RMVNVNZO[U[VZVX RQURXSXSWQU RJWIYHZH[I[JYJW RZW[X\\Z]Z]Y\\XZW", "", "", "D_TFUESESJTNVQYT\\U]T\\S RYS[T\\T RTFTJUNWQXRZS\\S]Q RWEXGYGYFWE RKIIHIOHSGU RJIJOIRGU RJI]I[HZI RZMXPUSRU RZM[NXQTTRU RLLQLPKOL RLNLS RLNMOMS RMOPO RPOPS ROOPNQOQS RLSQS RMVM[N\\V\\W[VZ RMVNVNZO[U[VZVX RJWIYHZH[I[JYJW RQVRXSXSWQV RYWZX[Z\\Z\\Y[XYW", "", "", "", "", "", "", "", "", "", "", "", "D_MFKFJH RMFJH RGHPH RGHHIPH ROFPGQIRIRHQGOF RIJIT RIJJKJT RJKOK ROKOSNT RNKOJPKPSOUNTLS RJNON RJQOQ RUGVFTFTKUL\\L]K\\J RUGUJVK[K\\J]H RUIYGZH RUIZH RUOVNTNTSUT\\T]S\\R RUOURVS[S\\R]P RUQYOZP RUQZP RMWM[N\\W\\X[WZ RMWNWNZO[V[WZWX RJWIXGYGZHZIYJW RQWSYTYTXQW RZW[X\\Z]Z]Y\\XZW", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGLFJFJ\\ RKGK\\ RHLGNFOFPGPHNHL RLKMLNNONOMNLLK RUGUM RVGVM RQFQM RQFRGRM RRGYG RYGYM RXGYFZGZM ROJ]J\\I[J RQMZM RSPQOQY RRPRY RRPYP RXPYO[PZR RYPYY RZPZY RRSYS RRVYV RQYZY RT[RZO\\M] RT[Q\\M] RXZZ[\\]]]]\\[[XZ", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_RGSFQFQKRQSTTVVYX[Z\\[\\\\[[Z RWYXZZ[[[ RRGRKSQTTVWXYZZ[Z\\W RUFVIWIWHUF RHK\\KZIXK RXK[J RZNXNWQVSTVQYN[ RYNWR RZNXRVUSXPZL\\", "", "", "", "", "D_KKIJITHYG[ RJKJTIXG[ RJK]K[JZK RJONO RNONVMW RMONNOOOVNXMWKV RSGTFRFRKSQTTUVWYY[[\\\\\\][\\Z RXYYZ[[\\[ RSGSKTQUTWWYY[Z\\Z]W RVFWGXIYIYHXGVF RYQ[OYNXQWSUVRYO[ RZOYRWUTXQZM\\", 
  /******** Oriental Hershey Glyphs 1800 to 1899 ********/
  "", "", "D_SGTFRFRKSQTTUVWYY[[\\\\\\][\\Z RXYYZ[[\\[ RSGSKTQUTWWYY[Z\\Z]W RVFWGXIYIYHXGVF RGK]K[JZK RIMIU RIMJNJU RJNNN RNNNT RMNNMONOT RJSNS RGZPW RGZH[PW R[NYNXQWSUVRYO[ RZNXR R[NYRWUTXQZM\\", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_GG]G[EYG RYG\\F RMLKKKSJWIYG\\ RLLLSKWIZG\\ RLLXL RWLXKZLYN RXLXS RYLYS RLRXR", "", "", "", "", "", "D_GG]G[FZG RIIIUHYG[ RIIJJJUIXG[ RJJ[J R[J[N RZJ[I\\J\\N RJN\\N RJQRQ RRQRZQ[ RQQRPSQSZR\\Q[OZ RLSMUNUNTLS RLXQV RLXMYQV RSQ[Q R[Q[ZZ[ RZQ[P\\Q\\Z[\\Z[XZ RUSVUWUWTUS RUXZV RUXVYZV", "", "", "", "D_JIRHVGXFZH RTHYG RJIZH RRHRXQZP[ RRYRZQ[ RSHSZR\\Q\\P[MZ RIN[NYLWN RWNZM RHS]S[QYS RYS\\R", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGLFJFJZI[ RKGKZJ\\I[GZ RGKOKNJMK RGROP RGRHSOP RTHRGRRQVPXN[ RSHSRRVPYN[L\\ RTHXGZF\\G RSHTH\\G RSP]P[OZP RXPX\\ RYPY\\", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_LGMFKFKZJ[ RLGLZK\\J[HZ RGKPKOJNK RGQOO RGQHROO RQFQV RQFRGRV RRG[G R[G[V RZG[F\\G\\V RVGV\\ RWGW\\ RRN[N RRU[U", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 1900 to 1999 ********/
  "", "", "", "D_KGLFJFJZI[ RKGKZJ\\I[GZ RGKOKNJMK RGROP RGRHSOP RVGWFUFUP RVGVP RQK[KZJYK ROP]P[OZP RXPXZW[ RYPYZX\\W[UZ RNT]T[SZT RPVQWRYSYSXRWPV", "D_LGMFKFKZJ[ RLGLZK\\J[HZ RHKOKNJMK RHQOO RHQIROO RRGSFQFQNRO\\O]N\\M RRGRMSN[N\\M]K RRJYGZH RRJZH RQRQ\\ RQRRSR\\ RRSZS RZSZ\\ RYSZR[S[\\ RRWZW RR[Z[", "", "", "", "", "", "", "", "", "", "D_KGLFJFJZI[ RKGKZJ\\I[GZ RGKNKMJLK RGQNO RGQHRNO RNGTG RSFPO RRGSFTGQO RPOTO RTOSSQWOZ RSOTNUOTSSUQXM\\ RORNSNUOR RORPVQXSZV[\\[]Z RORPUQWSYVZ]Z RUIXH[F]G RUI]G RXHXV RYHYV RUO]O\\N[O RUV]V\\U[V", "", "", "", "", "", "D_KGLFJFJYIZ RKGKYJ[IZGY RGKNKMJLK RGQNO RGQHRNO RRGPFPROVNXL[ RQGQRPVNYL[ RQG]G[FZG RQKYKXJWK RQO]O[NZO RTOTY RUOUX RRYYW RRYSZYW RXOXRYVZX\\Z]Y RXRYU[X]Y R\\RZU R\\R]SZU", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGLFJFJYIZ RKGKYJ[IZGY RGKOKNJMK RGQNO RGQHRNO RQGQRPVOXM[ RQGRHRRQVOYM[ RRHWG[F]G RRH]G RRK[K R[K[O RZK[J\\K\\O RRO\\O RVOVTUWSZ RWOWTVWSZQ[ RXPYRZRZQXP RRT]T\\S[T RWTXWZZ\\[]Z RXWZY]Z", "", "", "", "", "", "", "", "", "D_KGLFJFJZI[ RKGKZJ\\I[GZ RGKNKMJLK RGQNO RGQHRNO RVGWFUFUI RVGVI RPI\\I[HZI RRJSNTNTMRJ RXJWO RXJYKWO RNO]O\\N[O RTPPW RTPUPQW RNS]S[RZS RXSWUUXR[ RXSYTWWUYR[O\\ RSWVXYZ[\\\\\\\\[YYWXSWPW", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGLFJFJZI[ RKGKZJ\\I[GZ RGKNKMJLK RGRMP RGRHSMP RVGWFUFUI RVGVI ROI]I[HZI RRJSMTMTLRJ RXJWN RXJYKWN RQNOMO\\ RPNP\\ RPN[N RZN[M]N\\P R[N[ZZ[ R\\N\\Z[\\Z[XZ RUNUT RVNVT RRQYQXPWQ RSTSX RTTTX RSTXT RWTWX RXTXX RSXXX", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 2000 to 2099 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_RGSFQFQN RRGRN RGI]I[GYI RYI\\H RJNWN RVNWMYOVQ RWNVQTTQWNYJ[ RXOWQUTRWOYMZJ[G\\ RMNMONRPUSXVZX[[\\][ RPUTXVYYZ][", "", "", "", "", "D_LGMFKFKQ RLGLQ RGKQKPJOK RHPHY RHPIQIY RIQNQ RNQNY RMQNPOQOY RIXNX RUFTISKQN RUFVGUISLQN RSK\\K[JZK RYKXOWRUVSY RYKZLYPWTTXRZO\\ RSLSOTSUUWXYZ[[]Z RUUWWZY]Z", "", "", "", "", "", "", "", "D_LGMFKFKM RLGLM RIIOINHMI RGMRMQLPM RQFOJLOIS RQFRGQIOLKQGU RHPPP ROPPORQOR RPPOR RQPORMS RLSLZK[ RLSMSMZL\\K[IZ RIVQU RIVJWQU RVFTKRO RVFWGVIRO RTM]M\\L[M RZMYRXUVXTZQ\\ R[MZRYUWXUZQ\\ RTMUQVTWVYY\\\\][ RVTXWZY][", "", "", "", "D_KGLFJFJN RKGKN ROGPFNFNN ROGON RHIRIQHPI RGNSNRMQN RIPI\\ RIPJQJ\\ RJQOQ ROQOZN[ RNQOPPQPZO\\N[LZ RJTOT RJWOW RWFVITNSP RWFXGVKSP RTN]N\\M[N RZNYRXUWWUZ R[NZRYUWXUZR\\ RUOVTWWXYZ[\\\\][ RVTXXZZ][", "D_MGNFLFLQ RMGMQ RHGJJKJKIHG RPGNJ RPGQHNJ RGKSKRJQK RKKJMHP RLLHP RNMONPPQPQOPNNM RLSIX RLSMSJX RGUSURTQU RPUOWMYJ[H\\ RQUPWNYK[H\\ RKWNYP[Q[QZPYNXKW RUFTKSN RUFVGUJRP RUK]K\\J[K R[KZPYSXUVXTZQ\\ R\\K[PZSYUWXUZQ\\ RTLUQVTWVYY\\\\][ RVTXWZY][", "", "", "", "", "", "", "D_RGSFQFQJ RRGRJ RGJ]J[HYJ RYJ\\I RWLXKVJUNTQSSQVNYK[H\\ RWKVN RWLVOTSRVOYL[H\\G\\ RMJMMNPORQUTXWZ[\\][ RXZ[[ RNPPSSVVXXY][", "", "", "D_MGNFLFLJ RMGMJ RGJSJRIQJ RPKNPKUIX RPKQLOPMSIXGZ RHLLQNTPXQXQWPUMQHL RZG[FYFYXXZW[ RYYYZX[ RZGZZY\\X\\W[UZ RRM]M\\L[M RSPTQUSVSVRUQSP", "", "", "", "", "", "", "D_NFLFJIFN RNFFN RMGOHRKSKSJRIPHMG RIMQMPLOM RLMLZK[ RMMMZL\\K[IZ RGQRQQPPQ RISGX RISJTGX ROTPUQWRWRVQUOT RZG[FYFY\\ RZGZ\\ RTHUIVKWKWJVITH RSNTOUQVQVPUOSN RRV]S\\R[S", "", "", "", "", "", "D_MHMGNFLFLH RGHSHRGQH RIIJMKMKLII ROINM ROIPJNM RGNSNRMQN RLNL\\ RMNM\\ RHRRRQQPR RLQKTJVHY RLTJWHY RNTOUPWQWQVPUNT RUGUSTWRZP\\ RUGVHVSUWSZP\\ RVH[F]G RVH]G RVO]O\\N[O RZOZ\\ R[O[\\", "", "D_RGSFQFQK RRGRK RGK]K[IYK RYK\\J RQKPPOSNULXJZG\\ RQKRLQPPSNWLYI[G\\ RQP\\P RYPZO\\PZS RZPYVXYWZU[ RXZW[V[ R[PZVYYX[W\\V\\U[RZ", "D_LHMGKFKNJUIXG\\ RLGLOKTJWIYG\\ RGKPKOJNK RLOOO ROOOVNZM[ RNOONPOPVOZN\\M[KZ RWFVKUNTP RWFXGWKVMTPRR RXIYL[P\\R]Q RYL[O]Q RVRWSXUYUYTXSVR RUXWYZ\\[\\[[ZZXYUX", "D_LHMGKFKNJUIXG\\ RLGLOKTJWIYG\\ RGKPKOJNK RLOOO ROOOVNZM[ RNOONPOPVOZN\\M[KZ RVFTFRKQM RUFTH RVFTIQM RSK]K[JZK RYKXPWTUX RZKYPXSWUUXSZQ[ RTKTMURVUXXZZ\\[]Z RURVTXWZY]Z", "", "", "", "", "", "", "", "", "", "", "", "", "D_MHKGK\\ RLHL\\ RLZXZ RLHXH RWHXGZHYJ RXHX\\ RYHY\\ RLQXQ", "", "", 
  /******** Oriental Hershey Glyphs 2100 to 2199 ********/
  "D_LGJFJR RKGKR RKGXG RWGXFZGYI RXGXR RYGYR RKLXL RKQXQ RQQQ] RRQR] RGV]V[TYV\\U", "", "", "", "", "", "", "D_MGKFKP RLGLP RLGWG RVGWFYGXI RWGWP RXGXP RLKWK RLOWO RNOLRIU RNOOPLSIUGV RLSYS ROSMVKX ROSPTNVKXHY RTSSUPYM[J\\ RTSUTSWQYN[J\\H\\ RXSYR[SYU RYSWYVZT[ RWZV[U[ RZSXYW[V\\U\\T[RZ", "D_OGPFNFNO ROGOO RUGVFTFTO RUGUO RIJ[JYIXJ RGO]O[NZO RMSKRK\\ RLSL\\ RLSWS RVSWRYSXU RWSW\\ RXSX\\ RLWWW RL[W[", "", "D_HFHU RHFIGIU RIGNG RNGNT RMGNFOGOT RIMNM RISNS RTFTSSWRYO\\ RTFUGUSTWRZO\\ RUG[G R[G[ZZ[ RZG[F\\G\\Z[\\Z[XZ RUM[M RUS[S", "", "", "", "", "", "", "", "", "D_HFHX RHFIGIX RIGMG RMGMX RLGMFNGNX RIOMO RIWMW RRFPLON RRFSGRION RQJ]J[HYJ\\I RUJU\\ RVJV\\ RVP]P[NYP\\O RVV]V[TYV\\U", "", "D_MGKFKN RLGLN RLGXG RWGXFZGYI RXGXN RYGYN RLJXJ RLMXM RROR[ RROSOS[ RLPKSJUHX RLPMQLSJVHX RLRZRYQXR RLVZVYUXV RG[][[ZZ[", "D_RFPFOIMNJSGW RQFOJ RRFOLLQIUGW RHH[HYGXH RIKYKXJWK RSLTNVQXS[U]T RYS[T RTNVPYR]T RGN]N[MZN RLRL\\ RLRMSM\\ RMSVS RVSV\\ RUSVRWSW\\ RMWVW RM[V[", "", "", "", "D_HFHW RHFIGIW RIGMG RMGMV RLGMFNGNV RINMN RIUMU RUGVFTFTN RUGUN RPJ[JYIXJ RNN]N[MZN RWNWZV[ RXNXZW\\V[TZ RNR]R[QZR RQTRUSWTWTVSUQT", "", "", "", "", "", "", "", "", "", "", "D_NGLFLP RMGMP RMGWG RVGWFYGXI RWGWP RXGXP RMKWK RMOWO RKSIRI\\ RJSJ\\ RJSOS RNSORQSPU ROSO\\ RPSP\\ RJWOW RJ[O[ RVSTRT\\ RUSU\\ RUSZS RYSZR\\S[U RZSZ\\ R[S[\\ RUWZW RU[Z[", "D_KEKL RKELFLL RLFWF RWFWL RVFWEXFXL RLIWI RKLXL RPLPR RQLQR RIOTOSNRO RGR]R[QZR RXMSQNTJV RXMYNVPQSMUJVGW RLTL\\ RLTMUM\\ RMUWU RWUW\\ RVUWTXUX\\ RMXWX RM[W[", "", "", "D_KEKL RKELFLL RLFWF RWFWL RVFWEXFXL RLIWI RKLXL RHN\\NZMYN RQPQ[ RRPR[ RKPKV RLPLV RKPXP RWPWV RXPXV RLSWS RKVXV RIX[XZWYX RG[][[ZZ[", "", "D_HFHW RHFIGIW RIGMG RMGMV RLGMFNGNV RINMN RIUMU RWFXEVEVN RWFWN RQH\\H[GZH RRK[KZJYK RPN]N\\M[N RRPR\\ RRPSQS\\ RSQZQ RZQZZY[ RYQZP[Q[ZZ\\Y[WZ RSTZT RSWZW", "", "", "D_LELL RLEMFML RMFVF RVFVL RUFVEWFWL RMIVI RLLWL RGO]O[NZO RJOJY RKOKY RGZOX RGZH[OX RPOP\\ RQOQ\\ RKRPR RKUPU RSR\\R R[QZTXWUZR\\ RZR[Q\\R[TYWWYT[R\\ RTSUVWYY[[\\][ RUVWXZZ][", "", "", "", "", "", "", "", "D_HFHX RHFIGIX RIGMG RMGMX RLGMFNGNX RIOMO RIWMW RVIVGWFUFUI RPI\\I[HZI RRJSMTMTLRJ RXIWN RXIYJWN ROO]O[NZO RSSQRQ\\ RRSR\\ RRSYS RXSYR[SZU RYSY\\ RZSZ\\ RRWYW RR[Y[", "", "", "", "", "", "D_KEKJ RKELFLJ RLFWF RWFWJ RVFWEXFXJ RLHWH RKJXJ RKLYLXKWL RQLQR RRLRR RIMHOGPGQHQIOIM RIN[N RZN[M]O[P R[M[P R\\N[PZQ RKPNP RUPXP RKRNR RURXR RITZTYSXT RGW]W[VZW ROWN[ ROWPXN[ RI[XZ RI[K\\XZ RWYXZY\\Z\\Z[YZWY", "", "", "", "D_IFIM RIFJGJM RJGOG ROGOM RNGOFPGPM RJJOJ RIMPM RGPRPQOPP RLPLX RMPMY RMTQTPSOT RISIUHYG[ RISJSJUIXG[ RJVKXMZP[\\[]Z RKXMYPZ]Z RSG]G\\F[G RWGVK RWGXHVK RTJTT RTJUKUT RUKZK RZKZT RYKZJ[K[T RUNZN RUQZQ RTT[T RUURX RUUVVRX RYU\\X]X]WYU", "", "", "", "", "D_OGMFMNLTKWIZG\\ RNGNNMTLWJZG\\ RNGXG RWGXFZGYI RXGXXWZV[ RXYXZW[ RYGYZX\\W\\V[TZ RNMXM RMSXS", "D_RHSGQFQ\\ RRGR\\ RGL]L[JYL RYL\\K RPLMRJVGY RQMOQMTKVGY RSMTPURXV[Y]X RYV[X RSMTOWSZV]X", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGLFJFJ\\ RKGK\\ RGKOKNJMK RJLIPHSGU RJPIRGU RLPMQNSOSORNQLP RTHRGRRQVPXN[ RSHSRRVPYN[L\\ RTHXGZF\\G RSHTH\\G RSP]P[OZP RXPX\\ RYPY\\", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 2200 to 2299 ********/
  "", "", "", "", "", "", "", "", "", "", "D_MHNGLFL\\ RMGM\\ RHLQLPKOL RLLKPJRGV RLPJSGV RNPOQPSQSQRPQNP RWHXGVFV\\ RWGW\\ RQL]L[KZL RVLUPSUQX RVPTTRWPY RWMXQYTZV\\Y]X RWMXPZT\\W]X", "D_KGLFJFJ\\ RKGK\\ RGKOKNJMK RJLIPHSGU RJPIRGU RLOMPNROROQNPLO RVGWFUFUO RVGVO ROK]K[JZK RPOZO RZOYRWUTXQZ R[OZQ RYOZN\\O[QXUVWSYQZN[ RQPRSTVVXYZ[[]Z RTVWXYY]Z", "D_KGLFJFJ[ RKGK[ RGKOKNJMK RJLIPHSGU RJPIRGU RLOMPNROROQNPLO RTGRMQO RTGUHSLQO RXFYJZL\\O]N RXFYI[L]N RVORZ RVOWPRZ ROZZX ROZP[ZX RZV[X\\[][]Z\\XZV", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGLFJFJ\\ RKGK\\ RGKNKMJLK RJLIPHSGU RJPIRGU RKOLPMRNRNQMPKO ROGOV ROGPHPV RPHTFUG RPHUG RSISVRYP[N\\ RSITITVSYQ[N\\ RNVST RNVOWST RWHW\\ RWHXIX\\ RXI[I R[I[VZW RZI[H\\I\\V[XZWXV", "", "", "D_KGLFJFJ\\ RKGK\\ RGKOKNJMK RJLIPHSGU RJPIRGU RLOMPNROROQNPLO RSFTGUIVIVHUGSF RPK]K[JZK RUKU[ RVKV[ RQS[SZRYS RO[][[ZZ[", "", "", "", "", "D_LHMGKFK\\ RLGL\\ RGLPLOKNL RKNJQISGV RKQITGV RMPNQOSPSPROQMP RSGS\\ RSGTHT\\ RTH[H R[H[\\ RZH[G\\H\\\\ RTN[N RTT[T RTZ[Z", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGLFJFJ\\ RKGK\\ RGKOKNJMK RJLIPHSGU RJPIRGU RLPMQNSOSORNQLP RWGXFVFVJ RWGWJ RQJ]J\\I[J RUJSO RUJVKSO RQNROSQTQTPSOQN RYNWQTTQV RYNZOVSSUOW R\\RZUWXTZ R\\R]SYWVYR[O\\ RYX[Z\\\\]\\][\\ZYX", "", "D_KGLFJFJ\\ RKGK\\ RGKOKNJMK RJLIPHSGU RJPIRGU RLOMPNROROQNPLO ROFPGQJRJRIQGOF RTFUGVJWJWIVGTF R\\FZK R\\F]GZK RTMQW RTMUMRW RPQ]Q\\P[Q RYQXTWVUYS[Q\\ RYQZRXVVYT[Q\\P\\ RTVWWZY\\[][]ZZXXWTVRV", "", "", "", "", "D_KGLFJFJ\\ RKGK\\ RGKOKNJMK RJLIPHSGU RJPIRGU RLPMQNSOSORNQLP RQFQY RQFRGRX RRGYG RYGYO RXGYFZGZO RRKYK RROZO RUOVRWTYW\\Z]Y RVRXUZW]Y RNYXW RNYOZXW R[QXT R[Q\\RXT", "", "", "D_JGKFIFI\\ RJGJ\\ RGKMKLJKK RIOHSGU RIRGU RJPKQLSMSMRLQJP RXGYFWFWNXUYXZZ[[\\[]Z\\Y RZY[Z\\Z RXGXNYUZX[Y\\Y]W RZF[H\\H\\GZF RMK]K[JZK R[NZRYUWY R[N\\O[RYVWYU[S\\ ROMOTNXMZ ROMPMPTOWMZL[ RRMRY RRMSMSY RMRVRUQTR", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 2300 to 2399 ********/
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
  /******** Oriental Hershey Glyphs 2400 to 2499 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "D_OFMFLIINGQ RNFMH ROFKLGQ RKK[K RYKZJ\\LYM RZJYM R[KYMXN RQKPPOSNULXJZG\\ RRKQPPSOUMXKZG\\ RSKTPUSWWYZ[\\][ RYY[[ RUSWVZY][", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_SHTGRFR[ RSGS[ RMONNLML[ RMNM[ RSP[PYNWP RWPZO RG[][[YY[ RY[\\Z", "D_NGOFMFMX RNGNX RJLKKIKIY RJLJY RNNRNQMPN RGYSW RGYHZSW RUGVFTFTZU[\\[]Z\\X R[Z\\Z\\Y RUGUYVZZZ\\X]U R[KUQ R[K\\LUQ", "", "", "", "", "D_HKHY RHKIKIY RKFKY RKFLFLX RLNNNMMLN RFYOX RFYGZOX ROFOYPZSZSW ROFPFPXQYRYSX RRKPN RRKSLPN RWFVITNSP RWFXGWISP R[FYK R[F\\GYK RULU\\ RVLV\\ RUL]L\\K[L RYLY[ RZLZ[ RVQ]Q\\P[Q RVV]V\\U[V RV[][\\Z[[", "D_MFNELELS RMFMS RGGRGQFPG RHIHM RHIIJIM RIJPJ RPJPM ROJPIQJQM RHMQM RKMIPGR RLNGR RNOOPPRQRQQPPNO RUFTJSL RUFVGUISL RUI]I\\H[I RZIYLXNVQ RZI[JZLXOVQTR RUIVLWNYP\\R]Q RWNZP]Q RHU\\UZTYU RRUR[ RSUS[ RSXYXXWWX RLXL[ RLXMXM[ RG[][[ZZ[", "", "D_GGRGQFPG RLGKLJOIQ RMGLLKNIQGS RLLPL RPLOQNTLWJYG[ ROLPKQLPQOTNVLXIZG[ RJPLRMTNTNSMRJP RUIUU RUIVIVU R[F[YZZ R[F\\F\\Y[[ZZXY", "D_GG]G[EYG RYG\\F RLGKLJOIQ RMGLLKNIQGS RLLPL RPLOQNTLWJYG[ ROLPKQLPQOTNVLXIZG[ RJPLRMTNTNSMRJP RTGTZU[\\[]Z\\X R[Z\\Z\\Y RUGUYVZZZ\\X]U RZMUQ RZM[NUQ", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_LGJFIY RKGJY RKGXG RWGXFZGYI RXGWXVZU[ RWYWZV[ RYGXZW\\V\\U[SZ RPIQJRLSLSKRJPI RGO]O[NZO ROQPRQTRTRSQROQ RJW]W[VZW", "D_NFLFJJGN RNFKJGN RLH]H[GZH RMLKKJY RLLKY RLLYL RXLYK[LYO RYLXXWZV[ RXYXZW[ RZLYZX\\W\\V[TZ RRLQX RSLRX RGR]R\\Q[R RKX]X\\W[X", "", "", "D_KHLGJFJY RKGKY RKOPOONNO RGZQW RIZLY RGZI[QW RTHUGSFSZT[\\[]Z\\X R[Z\\Z\\Y RTGTYUZZZ\\X]U RTPWNYL[M RXNYM RTP[M", "", "", "D_GINHTGXFZG RGIZG RPHPZQ[\\[]Z\\X R[Z\\Z\\Y RQHQYRZZZ\\X]U RHN[LYKXL RGT]R[QZR", "", "", "", "", "D_LHJGJY RKHKX RLHRGWFYG RKHLHYG RGYSV RGYIZSV RRGRPSUTXW[Z\\\\\\][\\Z RSGSPTUUXWZZ[[[\\Z]W RKO\\OZMXO RXO[N", "", "D_OFMFLHJKGO ROFKKGO RLI\\IZHYI RLLZLXKWL RJOXO RWOXNZOYQ RXOXTYX[[\\\\]\\][\\Y R[Z\\[ RYOYTZX[Y\\Y]V RTQOVKYG[ RTQURRUNXJZG[ RIRLTPWSZTZTYRWMTIR", "", "D_RISGQFQWPZO[MZ RQYQZP[ RRGRZQ\\P\\O[ RHLML RLLMKOLNN RMLMMLQKTJVGZ RNLNNMRLTJWGZ RSGSITNUQVSXV[Y]X RYV[X RSITMVQXTZV]X RZKVP RZK[LVP", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 2500 to 2599 ********/
  "", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RRGPFPTOXNZ RQGQTPWNZM[ RQGYG RXGYF[GZI RYGYO RZGZO RQNYN RUNUQVUWWYY\\[]Z RUQVTWVYX]Z", "", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RRFPKOM RRFSGRIOM RRI\\I[HZI RPLZLYKXL ROOXO RWOXNZOYQ RXOXTYX[[\\\\]\\][\\Y R[Z\\[ RYOYTZX[Y\\Y]V", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RUGVFTFTQSURWPYM[ RUGUQTUSWQYM[ ROJYJ RYJYQ RXJYIZJZQ RNQ]Q\\P[Q RUQVUWWYY\\[]Z RUQVTWVYX]Z", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RVGWFUFUP RVGVP RRJPIPSOWNYL\\ RQJQSPWNZL\\ RQJ[J RZJ[I]K[L R[J[L R\\J[LZN RQPYP RYPXSVVSYP[ RZPYR RXPYO[PZRXUUXRZN\\ RSPTTVXY[[\\][ RTTVWXYZZ][", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RMH]H[FYH\\G RZHZXYZX[ RZYZZY[ R[H[ZZ\\Y\\X[VZ ROLOU ROLPMPU RPMUM RUMUU RTMULVMVU RPTUT", "", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RVGWFUFUZ RVGVZ RQLOKO[ RPLP[ RPL[L RZL[K]L\\N R[L[[ R\\L\\[ RPS[S RPZ[Z", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RUHVGTFTR RUGUR ROL[LZKYL RNR]R[QZR RTSQZ RTSUTQZ RMZYY RMZO[YY RXWYX[[\\[\\Z[YXW", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RRFPJNM RRFSGNM RQI]I[HZI RRMPLOX RQMPX RQMZM RYMZL\\MZP RZMYZX[ R[MZZY\\X[VZ RUMTW RVMUW RNR]R\\Q[R RPW]W\\V[W", "", "", "", "", "", "", "", "", "", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ ROGZG RYGZF\\HYI RZGYI R[GYIVK RRITKUMVLTJPH RUMU\\ RVMV\\ RQMOLO\\ RPMP\\ RPM[M RZM[L]M\\O R[M[ZZ[ R\\M\\Z[\\Z[XZ RPQ[Q RPU[U", "", "", "", "", "", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RNG]G[FZG RSGSNRQPS RTGTNSQPS RWGWQXR[R RXGXPYQ[Q RQLOKO[ RPLP[ RPL[L RZL[K]L\\N R[L[[ R\\L\\[ RPV[V RPZ[Z", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RUIUGVFTFTI RNI]I[HZI RTJRO RTJUKRO ROO[N ROOPP[N RYL[N\\P]P]O\\NYL RPRPVOYM[ RPRQRQVPYM[ RTRT[ RTRURU[ RXRXZY[][]W RXRYRYYZZ\\Z]Y", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RVGWFUFUI RVGVI ROI]I\\H[I RQJOOMS RQJRKQMMS ROPO[ RPNP[ RWIVLTPRS RWIXJWMUPRS RVN\\N R[MZQXUUY R[M\\N[QZSXVUYR[ RVPWQXSYSYRXQVP RTRUUWXZZ\\[]Z RUUWWZY]Z", 
  /******** Oriental Hershey Glyphs 2600 to 2699 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RQFQP RQFRGRP RRGYG RYGYP RXGYFZGZP RUKUP RUKVLVP RVLYL ROOO\\ ROOPPP\\ RPP[P R[P[ZZ[ RZP[O\\P\\Z[\\Z[XZ RSSSX RSSTTTX RTTWT RWTWX RVTWSXTXX RSXXX", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RRGPFPP RQGQP RQGZG RYGZF\\G[I RZGZP R[G[P RQKZK RQOZO RSRS[ RSRTRT[ RWRW[ RWRXRX[ ROUPYQYQXOU R[UYY R[U\\VYY RN[][[ZZ[", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ ROFOV ROFPGPV RPGSG RSGSV RRGSFTGTV RPLSL RPQSQ ROVTV ROWM[ ROWPXM[ RSWT[U[UZSW RWHWU RWHXHXU R[F[YZZ R[F\\F\\Y[[ZZXY", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RRGPFPO RQGQO RQGZG RYGZF\\G[I RZGZO R[G[O RQKZK RPO[O RORO[ RORPSP[ RPS[S R[S[[ RZS[R\\S\\[ RSSS[ RTST[ RWSW[ RXSX[ RN[][", "", "", "D_GFHGIIJIJHIGGF RFLGMHOIOINHMFL RLNHVFVHWHYGZH[I[JZIYIVLN RHZIZ RMJMTLYK[ RMJNKNTMXK[ RNK]K[JZK RXGYFWFWOXUYXZZ\\\\]\\]X RXGXOYUZX[Z][ R[F\\H]H]G[F RPOUOTNSO RPRPX RPRQSQX RQSTS RTSTW RSSTRUSUW RQWUW R\\N[RZTXWVYS[ R\\N]O\\R[TYWWYS[P\\", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RUHUGVFTFTH ROH[HZGYH RQHRLSLSKQH RWHVM RWHXIVM RNM]M[LZM ROOOW ROOPPPW RPPYP RYPYW RXPYOZPZW RPSYS RPVYV RTPTZU[\\[]Z\\Y RUPUYVZ[Z\\Y]W", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RNENRMWLY RNEOFORNVLY ROF]F[EZF RVFUJ RWFUJ RRIRS RRISJSS RSJZJ RZJZS RYJZI[J[S RSNZN RSRZR RVRVYUZ RWRWYV[UZSY RRUOZ RRUSVOZ RZU\\Y]Y]XZU", "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RVHVGWFUFUH RPGOINJNKOKPIPG RPH[H RZH[G]I[J R[H[J R\\H[JZK RRLOO RRLSMOO RXLYMZO[O[NZMXL RUOSRQT RUOVPTRQTOU RVPYS\\U]T RWQZS]T RQUQ\\ RQURVR\\ RRVYV RYVY\\ RXVYUZVZ\\ RR[Y[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 2700 to 2799 ********/
  "", "", "D_HFIGJIKIKHJGHF RGLHMIOJOJNIMGL RMNIVGVIWIYHZI[J[KZJYJVMN RIZJZ RRGSFQFQM RRGRM RNJUJTISJ RNLNS RNLOMOS ROMTM RTMTS RSMTLUMUS ROPTP RNSUS RQSQ[ RRSR[ RNVUVTUSV RWFWUVYU[ RWFXGXUWXU[ RXG[G R[G[YZZ RZG[F\\G\\Y[[ZZYY RXM[M RXS[S", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_QFQLPQNULXJZG\\ RRGRK RQFSGRNQROVLYI[G\\ RSGSKTQUTVVXY[\\][ RYY[[ RSKTPVTXWZY][ RLLKOIQIRJRKQLNLL RZLUQ RZL[MUQ", "", "D_MGNFLFLRKVIYG[ RMGMRLVJYG[ RILHNGOGPHPINIK RPJMN RPJQKMN RMTNUPXQXQWPVMT RRH]H[FYH RYH\\G RWHWWVYUZ RWXWYVZ RXHXYW[V[UZSY", "", "", "", "", "D_LGMFKFKSJWIY RLGLSKVIYG[ RILHNGOGPHPINIL ROKMN ROKPLMN RLUMVNXOXOWNVLU RQG]G[FZG RTLRKRSQXPZ RSLSSRWPZN\\ RSLZL RYLZK\\L[N RZLZS R[L[S RSRZR", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KFJJIMGQ RKFLGKJINGQ RKJPJ RPJOMMQJU ROJPIQJPMOOMRJUGW RKLMNNNNMKL RIOJPKRLRLQKPIO RVGWFUFUMTQSSQUOV RVGVNUQTSRUOV RXHYJZJZIXH RPM]M[LZM RVNWQYT\\V]U RWQYS]U RIXHZG[G\\H\\IZIX RMXN\\O\\O[MX RSXT\\U\\U[SX RXXZZ[\\\\\\\\[ZYXX", "", "D_KGLFJFJSIXHZ RKGKSJWHZG[ RHLGNFOFPGPHNHL RNJKN RNJOKKN RLUMVNXOXOWNVLU RVGWFUFUN RVGVN RPI\\I[HZI RON]N\\M[N RQKQS RQKRKRS RYKYS RYKZKZS ROS]S\\R[S RSSSURXQZ RTSTUSXQZO[ RWSWZX[\\[]Z\\Y RXSXYYZ[Z\\Y]W", "D_LFJFIIGM RLFJIGM RLHJGJV RKHKV RKH]H\\G[H ROHOV RPHPV RTHTV RUHUV RYHYV RZHZV RGO]O\\N[O RGV]V\\U[V RIXHZG[G\\H\\IZIX RMXN\\O\\O[MX RSXT\\U\\U[SX RXXZZ[\\\\\\\\[ZYXX", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_MGNFLFLL RMGML RHIQIPHOI RGLRLQKPL RJLJMIOGQ RKLKMJOGQ RNLNOOPRPRN ROLOOQORN RLQLV RLQMQMV RHSQSPROS RGVQV RGVHWQV RVGWFUFUQTTRV RVGVQUTRV RRIYI RYIYUZV]V]T]R RXIYHZIZU\\U]T RTOVQWSXRVPTO RIXHZG[G\\H\\IZIX RMXN\\O\\O[MX RSXT\\U\\U[SX RXXZZ[\\\\\\\\[ZYXX", "", "", 
  /******** Oriental Hershey Glyphs 2800 to 2899 ********/
  "", "", "", "", "", "", "", "D_KGLFJFJSIXHZ RKGKSJWHZG[ RHLGNFOFPGPHNHL RMJKM RMJNKKM RKVLYMYMXKV RVGWFUFUQ RVGVQ RPGQHRJSJSIRHPG RZGXJ RZG[HXJ ROK]K\\J[K RTKRMOO RULOO RVKYN[O\\N RWLYM\\N RPQOTNV RPQQRNV RPTTT RSSRVQXO[ RRTSSTTSVQYN\\ ROVPXQXQWOV RYRY\\ RYRZRZ\\ RTT]T\\S[T RVTVX RWTWX RRX]X\\W[X", "D_KGLFJFJSIXHZ RKGKSJWHZG[ RHLGNFOFPGPHNHL RMJKM RMJNKKM RKVLYMYMXKV RQFPKON RQFRGQJNP RQIUI RTHSMQS RSITHUISOQSOV RQKRMSMSLQK RONPQQQQPON RXGYFWFWNVRUTSV RXGXNWRVTSV RZH[J\\J\\IZH RTM]M[LZM RXNYRZT\\V]U RYR[T]U RPXO[O\\P\\PX RSXS\\T\\T[SX RVXW\\X\\X[VX RYX[\\\\\\\\[YX", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_JGRFXEZF RJGZF RKHLIMKNKNJMIKH RPHQKRKRJPH RVGUK RVGWHUK RIJHLGMGNHNILIJ RIK[K RZK[J]L[M R[K[M R\\K[MZN RMMMQNRVRWQVP RMMNMNPOQUQVPVN RPMROSOSNPM RKNIPIQJQKN RYNZQ[Q[PYN RNSJWGY RNSOTLVGY RMUXU RWTUVQYM[ RWTXUTXPZM[I\\ RMTOWPXSZV[[\\][ RPXRYVZ][", "", "", "D_NGLIHL RNGOHHL RUGXI[L\\L\\KZIXHUG RNMJP RNMKQJP RNMOQQTTWWY[[]Z RNMOPQSSUVWZY]Z RVNVMXNWO RVNUQSTPWMYKZH[ RWNVP RWOVQTTQWNYLZH[", "", "", "", "", "", "", "D_LGMFKFK[ RLGL[ RGIHLILIKGI RGSKQ RGSHTKQ RUGVFTFTMSRRUQWOZ RUGUMTRSUQXOZM[ RXFZH[H[GXF RNL]L[KZL RUMVQWTXVZY\\[]Z RVQXUZX]Z", "", "", "D_KHLGJFJTIYH[ RKGKTJXH[G\\ RSHTGRFRL RSGSL RKL]L[JYL RYL\\K RKRTR RSRTQVRUT RTRT\\ RURU\\", "", "", "", "", "", "D_IG\\GZEXG[F RTGTWSYRZ RTXTYSZ RUGUYT[S[RZPY RMGMO RNGNO RMO[OYMWOZN RSOQRNUKW RTPPTMVIX", "", "", "", "D_RHSGQFQ] RRGR] RKGJJHOGQ RKGLHJLGQ RJLZLXJVL RVLYK RGS]S[QYS RYS\\R", "", "", "", "", "D_LGMFKFK[ RLGL[ RHIGQ RHIIJGQ RHLOLNKML RGUNQ RGUHVNQ RSFPMOO RSFTGSIOO RQL[L R[L[TZYYZ RZL[K\\L\\T[YZ[YZWY RTLSORQPT RULTORRPT RXLWQUUSX RYLXQVUSXPZ", "", "", "D_MGNFLFL\\ RMGM\\ RIHGP RIHJIGP RILQLPKOL RGTQQ RGTHUQQ RWGXFVFVO RWGWO RRJ\\J[IZJ RQO]O[NZO RXOXZW[ RYOYZX\\W[UZ RQS]S[RZS RRUSVTXUXUWTVRU", "", "", "", "", "", "", "", "D_SGQFQLPQNULXJZG\\ RRGRK RSGRNQROVLYI[G\\ RVGWHXJYJYIXHVG RGM]M[KYM RYM\\L RSMTQUTVVXY[\\][ RYY[[ RSMTPVTXWZY][", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 2900 to 2999 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HH\\HZFXH[G RQHQ[ RRHR[ RIQ[QYOWQZP RG[][[YY[\\Z", "D_HH\\HZFXH RXH[G RQHQ[ RRHR[ RIQ[QYOWQ RWQZP RG[][[YY[ RY[\\Z RUTVUWWXWXVWUUT", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_GGPGOFNG RKGKW RLGLV RHNONNMMN RGWPV RGWIXPV RWGXFVEVO RWFWO RRI\\I[HZI RQO]O[NZO RWRXQVPV[ RWQW[ RRU\\U[TZU RQ[][[ZZ[", "", "", "", "D_GGPGOFNG RKGKW RLGLV RHNONNMMN RGWPV RGWHXPV RWGXFVFVZU[ RWGWZV\\U[SZ RZF[H\\H\\GZF RPK]K\\J[K RQNROSQTQTPSOQN RWKXQYUZW\\Z]Y RXQYT[W]Y RPVVT RPVQWVT R[NYR R[N\\OYR", "D_GGPGOFNG RKGKW RLGLV RHNONNMMN RGWPV RGWIXPV RVGV[ RWGW[ RQFQR RQFRGRR RRG[G R[G[R RZG[F\\G\\R RRL[L RRQ[Q RQV\\V[UZV RP[][[ZZ[", "D_GGPGOFNG RKGKV RLGLU RHNONNMMN RGVPU RGVIWPU RTGRFRS RSGSS RSGZG RYGZF\\G[I RZGZS R[G[S RSKZK RSOZO RRS[S RTSSVQYN[ RUSTVRYN[ RWSWZX[\\[]Z\\Y RXSXYYZ[Z\\Y]W", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_MIKHKPJUIXG[ RLILPKUJXG[ RMISHWGYFZG RLIMIUHZG RWHWLXRYVZX\\[]Z RXRYU[X]Z RQIQX RRIRX RLYUX RLYMZUX RTUUWVZWZWYVWTU", "", "", "", "D_GH]H[FYH RYH\\G RMHJZ RNHKZ RMMVM RUMVLXMWO RVMVZW[\\[]Z\\Y RWMWYXZ[Z\\Y]V RNQPSQURURTQSNQLP RGZTY RGZH[TY", "", "", "", "", "", "", "", "", "", "", "D_MGNFLFL\\ RMGM\\ RWGXFVFV\\ RWGW\\ RGL]L[JYL RYL\\K RMSVS RM[V[", "", "", "D_SHTGRFR[ RSHS[ RLHKLJOIQGU RLHMILLJPGU RLL[LYJWL RWLZK RKSZSXQVS RVSYR RG[][[YY[ RY[\\Z", "", "D_LGJFJTIYH[ RKGKTJXIZG\\ RKGZG RYGZF\\G[I RZGZXYZX[ RZYZZY[ R[G[ZZ\\Y\\X[VZ RRGR[ RSGS[ RKMZM RKSZS", "D_QIQY RRIRY RKIIHI[ RJIJ[ RJIYI RXIYH[IZK RYIY[ RZIZ[ RJQYQ RJYYY", "", "D_RGRO RSGSO RMGKFKP RLGLP RLGYG RXGYF[GZI RYGYP RZGZP RLKYK RLOYO RSRTQRPRQQUOXMZK[H\\ RSRRUPXNZL[H\\ RGT[T RZT[S]T[V R[TZXYZX[ RZYZZY[ R\\T[ZZ\\Y\\X[VZ", "", "D_QGQO RRGRO RLGJFJP RKGKP RKGXG RWGXFZGYI RXGXP RYGYP RKKXK RKOXO RQOORKUGW RQORPOSLUJVGW RRPVTYV[W]V RRPVSZU]V RNSNVMYK[I\\ RNSOSOVNYL[I\\ RTST\\ RTSUSU\\", "", 
  /******** Oriental Hershey Glyphs 3000 to 3099 ********/
  "", "D_QGQQ RRGRQ RLGJFJR RKGKR RKGXG RWGXFZGYI RXGXR RYGYR RKLXL RKQXQ RMUNTLTL[M\\W\\X[WZ RMUMZN[V[WZWX RQTRUSWTWTVSUQT RIVHXGYGZHZIXIV RZV[W\\Y]Y]X\\WZV", "", "", "", "", "", "", "D_KFKP RKFLGLP RLGWG RWGWP RVGWFXGXP RQGQO RRGRO RLKWK RLOWO RNPNW RNPOPOW RTPTW RTPUPUW RISZSYRXS RGW]W[VZW RNYMZJ\\H] RNYOZM[H] RTYVZY\\Z][][\\Z[XZTY", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_THTGUFSFSH RJGJWIZG\\ RJGKHKWJZG\\ RKH]H[GZH RGIHLILIKGI RGSGQJP RGSJP RML]L[KZL RTLTQSTRV RULUQTTRVPW RPPNON\\ ROPO\\ ROPZP RYPZO\\P[R RZPZZY[ R[P[ZZ\\Y[WZ RURVTWWXWXVWTUR", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IFQF RQFPHMLJNGO RQFRGPJNLKNGO RRGTJVLYN[O]N RRGUJXLZM]N RIHJIKKLKLJKIIH RWFUI RWFXGUI RZIXL RZI[JXL RLOWOVNUO ROOORNVMXKZI[ RPOPROVNXLZI[ RSOSZT[\\[]Z\\Y RTOTYUZ[Z\\Y]W RGT\\TZSYT", "", "", "D_TFRFQJ RSFRH RTFQJ RLJJIJ\\ RKJK\\ RKJYJ RXJYI[JZL RYJY\\ RZJZ\\ RKRYR RKZYZ", "", "D_KFJJ RKFLGJJ RHIH\\ RHIIJI\\ RIJNJ RNJN[ RMJNIOJO[ RIRNR RIZNZ RUFSKRM RUFVGUIRM RTJ[J RZJ[I]J\\L R[J[XZZY[ R[Y[ZZ[ R\\J\\Z[\\Z\\Y[WZ RTPUQVSWSWRVQTP", "", "", 
  /******** Oriental Hershey Glyphs 3100 to 3199 ********/
  "", "", "", "", "", "", "", "", "", "D_RGSFQFQP RRGRP RKJIIIUHYG[ RJJJUIXG[ RJJ[J RYJZI\\KYL RZIYL R[JYLXM RJPWP RVPWOYQVS RWPTUQXM[ RXQVTRXOZM[J\\ RNQOSQVTYW[[\\][ RQVUYXZ][", "", "", "", "D_LHJGJZ RKHKZ RKHYH RXHYG[HZJ RYHYZ RZHZZ ROHOZ RPHPZ RTHTZ RUHUZ RHZ]Z\\Y[Z", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_MHKGK\\ RLHL\\ RLHXH RWHXGZHYJ RXHX\\ RYHY\\ RLNXN RLTXT RLZXZ", "D_NGLFLT RMGMT RMGVG RUGVFXGWI RVGVT RWGWT RMKVK RMOVO RMSVS RGV]V[UZV RMYJ[G\\ RMYNZK[G\\ RUYWZZ\\[\\[[YZUY", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_JGWG RVGWFYHVI RWGVI RXGVISL RMJOKRNSNSMRLPKMJ RGN[N RZN[M]OZP R[NZP R\\NZPXR RRNRXQZP[ RRYRZQ[ RSNSZR\\Q\\P[NZ RQNORLVJXGZ RQNROQQOTLWIYGZ", "", "", "", "D_OFMFKJIM RNFLI ROFLJIM RLJ[JYHWJ RWJZI RQJQOPSNWLYI[G\\ RRJROQSOWMYJ[G\\ RGP]P[NYP RYP\\O RSPTTVXXZ[\\][ RYZ[[ RSPTSVVYY][", "D_JFIJHMGO RJFKGJJILGO RJJQJPIOJ RLJLSKWIZG\\ RMJMSLWJZG\\ RGPRPQOPP RNUOWPZQZQYPWNU RTITZ RTIUJUZ RUJ[J R[J[Z RZJ[I\\J\\Z RUY[Y", "", "", "D_IFHJGMFO RIFJGIJHLFO RHKPKOJNK RKKKRJVIXG[ RLKLRKVIYG[ RGQPQOPNQ RLUNWOYPYPXNVLU RQG]G[FZG RRKRS RRKSLSS RSLZL RZLZS RYLZK[L[S RSRZR RSTTYUYUXST RZTX[ RZT[UX[ RQ[][\\Z[[", "", "", "", "D_GH]H[FYH RYH\\G RQHOMMQKTGY RRHPL RSHQLNQKUGY RPQNPN\\ ROQO\\ ROQYQ RXQYP[QZS RYQY\\ RZQZ\\ ROZYZ", "", "", "", "D_GGOGNFMG RJGINHR RKGJMIPGT RINIZ RINJOJZ RJOMO RMOMY RLOMNNONY RJXMX RSGSRRVQXO[ RTGTRSVQYO[M\\ RWGW\\ RXGX\\ RPG]G[FZG RPQ]Q[PZQ", "", "", "", "", "", "", "", "", "", "", "D_FGNGMFLG RIGHNGR RJGIMHPFT RHNHZ RHNIOIZ RIOLO RLOLY RKOLNMOMY RIXLX RUGVFTFTI RUGUI RNI]I[HZI RTJRO RTJUKRO ROO[N ROOPP[N RYL[N\\P]P]O\\NYL RPRPVOYN[ RPRQRQVPYN[ RTRT[ RTRURU[ RXRXZY[][]W RXRYRYYZZ\\Z]Y", "D_GGOGNFMG RJGINHR RKGJMIPGT RIOI[ RIOJPJ[ RJPMP RMPMZ RLPMONPNZ RJYMY RWGXFVFVN RWGWN RRHSKTKTJRH R[HYK R[H\\IYK RRMR\\ RRMSNS\\ RSNZN RZNZZY[ RYNZM[N[ZZ\\Y[WZ RSRZR RSVZV", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 3200 to 3299 ********/
  "D_FGMGLFKG RHGGNFR RIGHLFR RGNGZ RGNHOHZ RHOKO RKOKY RJOKNLOLY RHXKX ROFOTNYM[ ROFPGPTOXM[ RPGSG RSGSYRZ RRGSFTGTYS[QY RPMSM RPSSS RWFWTVYU[ RWFXGXTWXU[ RXG[G R[G[YZZ RZG[F\\G\\Y[[YY RXM[M RXS[S", "", "", "", "", "", "", "", "", "D_GGOGNFMG RJGINHR RKGJMIPGT RIOI[ RIOJPJ[ RJPMP RMPMZ RLPMONPNZ RJYMY RRFSJTJTIRF RYFWK RYFZGWK ROK]K\\J[K RSKQR RSKTLQR ROQPRQTRTRSQROQ RUPQZ RUPVQQZ RPZUZ RPZQ[UZ RTXU[V[VZTX RZKXR RZK[LXR RVQWRXTYTYSXRVQ R\\PXZ R\\P]QXZ RWZ\\Z RWZX[\\Z R[X\\[][]Z[X", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_JHYHWFUH RUHXG RGN]N[LYN RYN\\M RQNQXPZO[ RQYQZP[ RRNRZQ\\P\\O[MZ RLQIWGZ RLQMRLTGZ RVRXTZW[Y\\Y\\X[VYTVR", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HHLGOFPG RHHPG RLGL\\ RMGM\\ RGLRLQKPL RLMKPITGW RLPJSGW RNQORPTQTQSPRNQ RUIUU RUIVIVU R[F[YZZ R[F\\F\\Y[[ZZXY", "D_HHLGOFPG RHHPG RLGL\\ RMGM\\ RGLRLQKPL RLMKPITGW RLPJSGW RNQORPTQTQSPRNQ RXFVFSY RWFVK RXFSY RPYZW RPYRZZW RXTZV\\Y]Y]X[VXT", "", "", "D_HHLGOFPG RHHPG RLGL\\ RMGM\\ RGLRLQKPL RLMKPITGW RLPJSGW RNQORPTQTQSPRNQ RTITZ RTIUJUZ RUJ[J R[J[Z RZJ[I\\J\\Z RUY[Y", "", "", "D_GHMFNG RGHNG RJGJ\\ RKGK\\ RGLOLNKML RJOIRGV RJRGV RLQMRNTOTOSNRLQ RWGXFVFVQUR RWGWQVSURSQ RRKPO RRKSLPO RZK[L\\N]N]M\\LZK R[RXVUYR[ R[R\\SWXTZR[O\\", "D_GHMFNG RGHNG RJGJ\\ RKGK\\ RGLOLNKML RJOIRGV RJRGV RLQMRNTOTOSNRLQ RRHSITKUKUJTIRH RQNROSQTQTPSOQN RYGZFXFX\\ RYGY\\ RPV]T\\S[T", "D_GHMFNG RGHNG RJGJ\\ RKGK\\ RGLOLNKML RJOIRGV RJRGV RLQMRNTOTOSNRLQ RVHWGUFUNTSSVQYN\\ RVGVPUTTVRYN\\ RVNWRXUZY\\\\][ RWRYV[Y][ RRMQOPPPQQQRORM R[LWQ R[L\\MWQ", "", "D_GHMFNG RGHNG RJGJ\\ RKGK\\ RGLOLNKML RJOIRGV RJRGV RLQMRNTOTOSNRLQ RRG\\G[FZG RVGV\\ RWGW\\ RSJRMQO RSJTKQO RYJZL[O\\O\\N[LYJ RQR]R[QZR", "", "", "", "", "D_GHMFNG RGHNG RJGJ\\ RKGK\\ RGLOLNKML RJOIRGV RJRGV RLQMRNTOTOSNRLQ RSFQKOO RSFTGSIOO RQL]L[KZL RVLVZU[ RWLWZV\\U[SZ RSQRTPX RSQTRPX RYQ[T\\W]W]V[SYQ", "", "", "", "", "D_GHMFNG RGHNG RJGJ\\ RKGK\\ RGLOLNKML RJOIRGV RJRGV RLQMRNTOTOSNRLQ RRFRN RRFSGSN RSGZG RZGZN RYGZF[G[N RSMZM RQQ]Q[PZQ RVQV[ RWQW[ RQV\\V[UZV RP[][[ZZ[", "", "", "", "", "", "", "", "", "D_GHMFNG RGHNG RJGJ\\ RKGK\\ RGLOLNKML RJOIRGV RJRGV RLQMRNTOTOSNRLQ RPIUH[F]G RPIXH]G RQKRNSNSMQK RVKWNXNXMVK R\\J[N R\\J]K[N RQPQ\\ RQPRPR\\ RUPU\\ RUPVQV\\ RVQ[Q R[Q[\\ RZQ[P\\Q\\\\ RVV[V RV[[[", "D_GHMFNG RGHNG RJGJ\\ RKGK\\ RGLOLNKML RJOIRGV RJRGV RLQMRNTOTOSNRLQ RRIVHZF[G RRI[G RVHV[ RWHW[ RQK]K\\J[K RQMQT RQMRNRT RRN[N R[N[T RZN[M\\N\\T RRQ[Q RQT\\T RQW\\W[VZW RP[][\\Z[[", "", "", "", "", 
  /******** Oriental Hershey Glyphs 3300 to 3399 ********/
  "", "", "", "", "", "", "D_GHMFNG RGHNG RJGJ\\ RKGK\\ RGLOLNKML RJOIRGV RJRGV RLQMRNTOTOSNRLQ RVGWFUFUN RVGVN RPH]H[GZH RQK[KZJYK RPN]N[MZN RQPQZ RQPRQRZ RRQYQ RYQYZ RXQYPZQZZ RRTYT RRWYW RQZZZ RT[SZN] RT[N] RXZ[\\\\]]]]\\[[XZ", "", "", "", "", "", "", "D_RIRGSFQFQI RIHHJGKGLHLIJIH RII\\I RZI[H]JZK R[HZK R\\IZKYL RMPLTKVIY RNQLU RMPOQNSKWG[ RTNTPUTVVXY[[]Z RXX[Z RTPUSWVYX]Z", "D_RHRGSFQFQH RIGHIGJGKHKIIIG RIH\\H RZH[G]IZJ R[GZJ R\\HZJYK RNHNJMMKOIP ROHOJNMLOIPGP RTHTNUO\\O]N\\M RUHUMVN[N\\M]K RPQQPOPOSNWMYK[I\\ RPQPSOWNYL[I\\G\\ RISTS RSSTRVSUU RTSTZU[\\[]Z\\Y RUSUYVZ[Z\\Y]W", "", "", "D_RHRGSFQFQH RIGHIGJGKHKIIIG RIH\\H RZH[G]IZJ R[GZJ R\\HZJYK RNHNJMMKOIP ROHOJNMLOIPGP RTHTNUO\\O]N\\M RUHUMVN[N\\M]K RHT\\TZRXT RXT[S RQTQ[ RRTR[ RG[][[YY[ RY[\\Z", "", "", "", "", "", "", "", "D_RGRFSEQEQG RIFHHGIGJHJIHIF RIG\\G RZG[F]HZI R[FZI R\\GZIYJ RNGMJLL RNGOHNJLLJMHM RTGTLUM\\M]L\\K RUGUKVL[L\\K\\J RGP]P[OZP ROPNT RPPOR RQPNT RITYT RITKUYT RVRXSZU[U[TYSVR RQTQ[ RRTR[ RIX[XZWYX RG[][[ZZ[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_RGSFQFQL RRGRL RHL\\LZJXL RXL[K RKOLRMXNWNUMRKO RWNTZ RWNXPTZ RG[][[YY[ RY[\\Z", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KFJJIMGR RLGKI RKFMGKKGR RJLRLPJNL RNLQK RLLL\\ RMLM\\ RUFTJSMQR RVGUI RUFWGUKQR RTL]L[JYL RYL\\K RWLWXVZU[ RWYWZV[ RXLXZW\\V\\U[SZ", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_LEJEIGGJ RLEGJ RJGQGPFOG RLGMJNJNILG RWEUETGRJ RWERJ RUG]G[FZG RWGXJYJYIWG RQLQ\\ RRLR\\ RILYL RYLYQ RXLYKZLZQ RIOHU RIOJPIU RJPYP RITZT RYTZS\\TZV RZTYYXZ R[TZYY[XZVY RPTMWIZ RQUNWIZG[", "", "", "", "", "", "", "", "", "", "", "D_LEJEIGGJ RLEGJ RJGQGPFOG RLGMJNJNILG RWEUETGRJ RWERJ RUG]G[FZG RWGXJYJYIWG RRLSKQKQQ RRLRQ RJNZNYMXN RGQ]Q[PZQ RUQUZT[ RVQVZU\\T[RZ RHT\\TZSYT RLVMWNYOYOXNWLV", "D_LEJEIGGJ RLEGJ RJGQGPFOG RLGMJNJNILG RWEUETGRJ RWERJ RUG]G[FZG RWGXJYJYIWG RRLSKQKQ\\ RRLR\\ RJMXM RXMXS RWMXLYMYS RGP]P[OZP RJSYS RIVZVYUXV RGY]Y[XZY", "", "", 
  /******** Oriental Hershey Glyphs 3400 to 3499 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_LEJEIGGJ RLEGJ RJGQGPFOG RLGMJNJNILG RWEUETGRJ RWERJ RUG]G[FZG RWGXJYJYIWG RKKKU RKKLLLU RLLXL RXLXU RWLXKYLYU RLOXO RLRXR RKUYU ROUOWNZL\\J] RPUPWOZM\\J] RTUT] RUUU] RGX]X[WZX", "D_LEJEIGGJ RLEGJ RJGQGPFOG RLGMJNJNILG RWEUETGRJ RWERJ RUG]G[FZG RWGXJYJYIWG RRLRKSJQJQL RIKHMGNGOHOIMIK RIL\\L RZL[K]MZN R[KZN R\\LZNYO RKOK\\ RKOLPL\\ RLPVP RVPVT RUPVOWPWT RLTWT RLWXW RXWX\\ RWWXVYWY\\ RL[X[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_LEJEIGGJ RLEGJ RJGQGPFOG RLGMJNJNILG RWEUETGRJ RWERJ RUG]G[FZG RWGXJYJYIWG RHMQMPLOM RMMMLLLLM RIMJPKPKOIM RNMMP RNMONMP RGPRPQOPP RIRI\\ RIRJSJ\\ RJSOS ROSOZN[ RNSORPSPZO\\N[LZ RJUOU RJWOW RTLTO RTLULUO RUM]M\\L[M RTO[O RZOZR R[O[R RUR[R RTQT[U\\]\\]Z]X RTQURUZV[\\[]Z RUT[TZSYT RUV[VZUYV RUX[XZWYX", "", "", "D_RHSGQFQ\\ RRGR\\ RJILKMMNMNLMKJI RXHTO RXHYITO RGO]O[MYO RYO\\N RQOPRNULWHZ RQROUMWJYHZ RSPTSVVXX[Z]Y RTSWVZX]Y", "", "", "", "", "", "", "D_LGMFKFK\\ RLGL\\ RGHHLILIKGH ROHML ROHPIML RFNQNPMON RKOISGV RKQGV RMQNROTPTPSORMQ RYGZFXFX\\ RYGY\\ RSHTIUKVKVJUISH RRNSOTQUQUPTORN RQV]T\\S[T", "", "", "D_LGMFKFK\\ RLGL\\ RGHHLILIKGH ROHML ROHPIML RFNQNPMON RKOISGV RKQGV RMQNROTPTPSORMQ RWGXFVFVL RWGWL RRL\\L[KZL RSNTVTYUYUWTRSN RZNXY RZN[OXY RP[][\\Z[[", "D_LGMFKFK\\ RLGL\\ RGHHLILIKGH ROHML ROHPIML RFNQNPMON RKOISGV RKQGV RMQNROTPTPSORMQ RWGXFVFVS RWGWS RWL\\L[KZL RRRR\\ RRRSSS\\ RSSZS RZSZ\\ RYSZR[S[\\ RS[Z[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_QFNK RQFRGNK RKILJNMOMOLNKKI RVIQP RVIWJQP RHQZP RHQJRZP RYMZO[R\\R\\Q[OYM RQQQ\\ RRQR\\ RMTJXHZ RMTNULWHZ RUTWVYYZYZXYWUT", "", "", "", "D_KFIK RKFLGIK RGJHKIMJMJLIKGJ RNJJQ RNJOKJQ RGQNQ RGQHRNQ RNOORPRPQNO RKQK\\ RLQL\\ RITGX RITJUGX RMTNUOWPWPVOUMT RSGSTRXQZ RTGTTSXQZO[ RTKUQVTWVYY\\[]Z RWVYX]Z ROGYG RYGWO RXGYFZGXO RXN]N R\\M[QZTYVWY R[N\\M]N\\R[TYWWYT[R\\", "", "", "", 
  /******** Oriental Hershey Glyphs 3500 to 3599 ********/
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
  /******** Oriental Hershey Glyphs 3600 to 3699 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HEHI RHEIFII RIFZF RZFZI RYFZE[F[I RNFNI ROFOI RTFTI RUFUI RHI[I RQIPO RRIQL RSIPO RGL]L[KZL RMNMY RMNNONY RNOWO RWOWY RVOWNXOXY RNRWR RNUWU RNXWX RHOH\\ RHOIOI\\ RI[][[ZZ[", "", "", "", "", "", "", "", "", "", "", "", "D_LFMGNJOJOINGLF RWFUFTK RVFUH RWFTK RQKQ] RRKR] RGK]K[IYK RYK\\J RIP[PYNWP RWPZO RGV]V[TYV RYV\\U", "", "D_MENHOHOGME RUETH RUEVFTH RGH]H[GZH RQHQP RRHRP RJLZLYKXL RGP]P[OZP RQQPUOWMYJ[G\\ RQQRRQUPWNYK[G\\ RGT]T[SZT RSTUXWZY[\\\\][ RSTUWWYYZ][", "", "", "", "D_MFNIOIOHMF RUFTI RUFVGTI RQIQQ RRIRQ RGI]I[HZI RJMZMYLXM RGQ]Q[PZQ RNQMTLVJYH[ ROQNTMVKYH[ RSUS[ RTUT[ RNUZUYTXU RL[][[ZZ[", "", "", "D_LFMGNIOIOHNGLF RWFUFTI RWFTI RHI\\IZHYI RQIQO RRIRO RILYLXKWL RGO]O[NZO RNOLSJVGZ RNOOPMSJWGZ RORMQM\\ RNRN\\ RNRWR RVRWQYRXT RWRW\\ RXRX\\ RNUWU RNXWX RN[W[", "", "", "", "", "", "", "", "D_GGOG RNGOFQGPI ROGOXNZM[ ROYOZN[ RPGPZO\\N\\M[KZ RHKILJNKNKMJLHK RGTNQ RGTHUNQ RQG[G RZG[F]G\\I R[G[XZZY[ R[Y[ZZ[ R\\G\\Z[\\Z\\Y[WZ RTKULVNWNWMVLTK RRTZQ RRTSUZQ", "", "", "D_GGOG ROGOM RNGOFPGPM RLGJIGK RNGGK RMJIMGN RMJNKGN RSG[G R[G[M RZG[F\\G\\M RZGVISK RZGSK RYJUMSN RYJZKSN RRQRPSOQOQQ RGQ]Q[PZQ RMQKTIV ROQMSIV RMSNTOVPVPUOTMS RWQUTSV RYQWSSV RWSXTYVZVZUYTWS RRXSWQWQ\\ RRXR\\ RGY]Y[XZY", "", "", "", "D_GFOF ROFOK RNFOEPFPK RIGLILHIG RGJNJ RGJHKNJ RSF[F R[F[K RZF[E\\F\\K RUGXIXHUG RSJZJ RSJTKZJ RQMQS RRMRS RKLKS RKLLMLS RLMWM RWMWS RVMWLXMXS RLPWP RKSXS RNSNY ROSOY RTSTY RUSUY RIV[VZUYV RGY]Y[XZY RO[MZJ\\H] RO[L\\H] RUZW[Z][][\\Y[UZ", "", "", "D_QGRFPFPN RQGQN RIIVIUHTI RGN]N[MZN RYGWJTNPRLUIW RYGZHVMRQNTIWGX RPSQROQOZP[\\[]Z\\X R[Z\\Z\\Y RPRPYQZZZ\\X]U RPVSUWSYRZS RPVUUXTZS", "D_PGQFOFOM RPGPM RJIUITHSI RGM]M[LZM RWGUJRNNRJUGW RWGXHTMORLTGW RPRNQMW RORNW RPRXPYQ RORPRYQ RNUXU RWUXTZUXW RXUWYVZT[ RWZV[U[ RYUXYW[V\\U\\T[RZ", "D_QGRFPFPM RQGQM RIIUITHSI RGM]M[LZM RXGUKRNNQIT RXGYHTMPPMRITGU ROQMPM\\ RNQN\\ RNQXQ RWQXPZQYS RXQX\\ RYQY\\ RNVXV RN[X[", "", "", "", "", "", "", "", "", "", "", "", "D_HH\\HZFXH RXH[G RLHLW RMHMW RWHW\\ RXHX\\ RMMWM RMRWR RGX]V[TYV RYV\\U", "", "D_GGTGRFQG RJGJX RKGKW RPGP\\ RQGQ\\ RKLPL RKQPQ RGXPV RGXHYPV RQJ\\J R[IZNYQWVUY RZJ[I\\J[NZQXUUYR\\ RTKUPVSWUYX\\[]Z RUPWTYW]Z", 
  /******** Oriental Hershey Glyphs 3700 to 3799 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_RFSEQEQS RRFRS RJGWG RVGWFYGXI RWGWM RXGXM RGJ]J[IZJ RJMXM RJPYPXOWP RGS]S[RZS RMVKUK\\ RLVL\\ RLVWV RVVWUYVXX RWVW\\ RXVX\\ RLYWY RK\\X\\", "", "", "", "", "D_JIHHH\\ RIII\\ RIIZI RYIZH\\I[K RZIZXYZX[ RZYZZY[ R[I[ZZ\\Y\\X[VZ RRGSFQFPJNNLQ RRGQJPLNOLQJR RQKSLVNWOXOXNWMULQK RRPSOQOPROTLX RRPQROULXJY RQSSTVVWWXWXVWUUTQS", "", "", "D_RFPFOIMMKPHT RQFOJ RRFQHNMKQHT RGI]I[GYI RYI\\H RMNM\\ RNMN\\ RNMXM RWMXLZMYO RXMXWWZV[ RXYXZW[ RYMYZX\\W\\V[TZ RNQXQ RNUXU", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IFIUHYG[ RIFJGJUIXG[ RJGMG RMGMYLZ RLGMFNGNYM[KY RJMMM RJSMS RTFRFQKPN RTFPN RSHZH RYHZG\\H[J RZHZUYV R[H[UZWYVWU RQMVM RVMVS RUMVLWMWS RQQQZR[\\[]Z\\Y RQQRRRYSZ[Z\\Y]W RRRVR", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGLFJFJU RKGKU RPGQFOFOU RPGPU RGISIRHQI RKMOM RKQOQ RGUSURTQU RJWIYG\\ RJWKXG\\ ROWPXQZRZRYQXOW RUFUTTYS[ RUFVGVTUXTZR\\ RVG[G R[G[ZZ[ RZG[F\\G\\Z[\\Z[XZ RVM[M RVS[S", "", "", "D_MGNFLFLL RMGML RGIRIQHPI RHKHS RHKILIS RILPL RPLPS ROLPKQLQS RIOPO RIRPR RLRL\\ RMRM\\ RGVRVQUPV RUFUSTWRZP\\ RUFVGVSUWSZP\\ RVG[G R[G[ZZ[ RZG[F\\G\\Z[\\Z[XZ RVL[L RVQ[Q", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 3800 to 3899 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGIFI\\ RJGJ\\ RJG\\GZEXG RXG[F RQGQM RRGRM RJMYM RXMYL[MZO RYMYU RZMZU RJTYT RQTQ[ RRTR[ RJ[][[YY[ RY[\\Z", "", "", "", "D_REPEOH RREOH RLHJGJ\\ RKHK\\ RKHYH RXHYG[HZJ RYHY\\ RZHZ\\ RKNYN RKTYT RKZYZ", "", "", "", "D_GG]G[EYG RYG\\F RQGMO RQGRHMO RIOXN RIOKPXN RULXNZP[P[OZNXMUL RRQSPQOQ[ RRPR[ RIU[UYSWU RWUZT RG[][[YY[ RY[\\Z", "", "", "", "", "", "", "", "", "", "D_HIOHTGWFYH RQHXG RHIYH RQHQS RRHRS RGM]M[KYM RYM\\L RMSKRK\\ RLSL\\ RLSWS RVSWRYSXU RWSW\\ RXSX\\ RL[W[", "D_GILHOGQFSG RGINHSG RMHMS RNHNS RGMTMSLRM RIRI[ RIRJSJ[ RJSQS RQSQ[ RPSQRRSR[ RJZQZ RWHXGVFVZW[\\[]Z\\X R[Z\\Z\\Y RWGWYXZZZ\\X]U", "", "", "", "D_GHKGNFPG RGHPG RKGKS RLGLS RGMPMOLNM RHRH[ RHRISI[ RISNS RNSN[ RMSNROSO[ RIZNZ RWGXFVFVJ RWGWJ RRJ\\J[IZJ RSKTOUOUNSK RYKXO RYKZLXO RQP]P\\O[P RVPV\\ RWPW\\ RRU\\U[TZU", "", "", "D_RFPFOI RRFOI RMIKHKSJWIYG\\ RLILSKWIZG\\ RLIWI RVIWHYIXK RWIWXVZU[ RWYWZV[ RXIXZW\\V\\U[SZ RPKQLRNSNSMRLPK RGP]P[NYP RYP\\O RPRQSRUSUSTRSPR", "", "D_MFLI RMFNGLI RIHIUHYG[ RIHJIJUIXG[ RJIPI RPIPZO[ ROIPHQIQZP\\O[MZ RLKMNNNNMLK RGQSO RGQHRSO RLSMVNVNULS RUFUJTMSO RUFVGVJUMSO RVGYG RYGYMZN]N]J RXGYFZGZL[M\\M]L RSQZQ RZQYTXVUZ RYQZP[QZTXWUZR\\ RTRUUWXZZ\\[]Z RUUWWZY]Z", "", "", "", "", "", "", "", "D_MFLI RMFNGLI RIHIVHZG\\ RIHJIJVIYG\\ RJIPI RPIPZO[ ROIPHQIQZP\\O[MZ RLKMNNNNMLK RGQSO RGQHRSO RLSMVNVNULS RVGUKTNSP RVGWHVKUMSP RYGZK[N\\P]O RZK\\N]O RTRT\\ RTRUSU\\ RUS[S R[S[\\ RZS[R\\S\\\\ RU[[[", "", "", "", "", "", "", "", "", "", "", "", "D_RGRFSEQEQG RMGKFKZ RLGLZ RLGWG RVGWFYGXI RWGWR RXGXR RLLWL RLQWQ RPQQTSWUYX[[\\][ RPQQSSVTWWYYZ][ RG[SX RG[H\\SX RXSTW RXSYTTW", "", "", "", "D_OFMFKJIM ROFKKGO RMIUI RTIUHWJTK RUITK RVITKRM RRMRR RSMSR RMMKLKZL[\\[]Z\\X R[Z\\Z\\Y RLMLYMZZZ\\X]U RLMYM RXMYL[MZO RYMYS RZMZS RLRYR", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 3900 to 3999 ********/
  "", "", "", "", "", "", "", "", "", "D_OFPENENI ROFOI RUFVETETI RUFUI RGG]G[FZG RNLLPJSGW RNLOMMPJTGW RLQL[ RMPM[ RUNVMTLTZU[\\[]Z\\X R[Z\\Z\\Y RUMUYVZZZ\\X]U RZOUS RZO[PUS", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_OFPENENJ ROFOJ RUFVETETJ RUFUJ RGG]G[FZG RQKOOLSGX RQKRLQNNRKUGX RGO]O[NZO RLSL\\ RLSMTM\\ RMTWT RWTW\\ RVTWSXTX\\ RM[W[", "", "", "", "", "", "", "", "", "", "", "", "", "D_OFPENENI ROFOI RUFVETETI RUFUI RGG]G[FZG RLKJJJT RKKKT RKKXK RWKXJZKYM RXKXT RYKYT RKOXO RKSXS RQSQ\\ RRSR\\ RGW]W[VZW", "D_OFPENENI ROFOI RUFVETETI RUFUI RGG]G[FZG RQJPLNOKR RQJRKPNNPKRGT RRKTNVPYR\\S]R RTNWPYQ]R RRRSQQQQ\\ RRRR\\ RLTXTWSVT RMVLXJZ RMVNWMXJZH[ RUVWWZZ[Z[YZXXWUV", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_OFPENENI ROFOI RUFVETETI RUFUI RGG]G[FZG RKJJMHRGT RKJLKKMGT RJOJ\\ RKMK\\ RML]L[KZL RYLYZX[ RZLZZY\\X[VZ ROOOW ROOPPPW RPPTP RTPTW RSPTOUPUW RPVTV", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_OFPENENI ROFOI RUFVETETI RUFUI RGG]G[FZG RMJIOGQ RMJNKLMGQ RMLZL RYLZK\\L[N RZLZ[Y\\ R[L[[Z]Y\\W[ RROSNQNQ[ RROR[ RLOMPNROROQNPLO RVOTS RVOWPTS RJTXTWSVT RPTKXHZ RQUNWHZ RTVUWVYWYWXVWTV", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 4000 to 4099 ********/
  "", "D_OFPENENI ROFOI RUFVETETI RUFUI RGG]G[FZG RQJQP RQJRJRP RROWO RWJWP RWJXJXP RHL\\L[KZL RKKKS RKKLKLS RLR[RZQYR RQRQ\\ RRRR\\ RGU]U[TZU RPUMXJZ RQVNXJZG[ RRUUXXZ[[]Z RSVVXYY]Z", "D_OFPENENI ROFOI RUFVETETI RUFUI RGG]G[FZG RJKYK RVKWJYKVL RWJVL RVLSN RROSNQNQSPT RRORSQUPTNS RSNUQXT[V]U RUQXSZT]U RGONO RMNLPJSGV RLOMNNOLRJTGV RZNVQ RZN[OVQ RJWXWWVVW RIYG[G\\H\\IY RMYN\\O\\O[MY RSYT\\U\\U[SY RYYZZ[\\\\\\\\[[ZYY", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_OFPENENI ROFOI RUFVETETI RUFUI RGG]G[FZG RQIPL RQIRJPL RNKNS RNKOLOS ROLTL RTLTS RSLTKULUS ROOTO RORTR RHKILJNKNKMJLHK RZKVO RZK[LVO RGQMO RGQHRMO RWOYP[R\\R\\QZPWO RQRQ\\ RRRR\\ RGU]U[TZU RPUNWKYG[ RQVNXJZG[ RSUUXXZ[[]Z RSUUWYY]Z", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 4100 to 4199 ********/
  "", "", "", "", "", "", "", "", "", "D_RFSEQEQPRQ\\Q]P\\O RRFROSP[P\\O\\N RRGZGYFXG RIIIUHYG[ RIIJJJUIXG[ RJJ\\J RZJ[I]K[L R[I[L R\\J[LZM RJNXMWLVM RMVNYOYOXMV RRUSTQTQ[ RRUR[ RVUWTUTU[ RVUV[ RZVWZ RZV[WWZ RK[][\\Z[[", "", "", "", "", "", "D_RGSFQFQZ RRGRZ RLKJJJT RKKKT RKKXK RWKXJZKYM RXKXT RYKYT RKSXS RG[YY RG[I\\YY RWVYX\\\\]\\][[YWV", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 4200 to 4299 ********/
  "", "", "", "", "", "D_SEQEOI RSEOI RLIJHJZ RKIKZ RKIYI RXIYH[IZK RYIYZ RZIZZ ROIOZ RPIPZ RTITZ RUIUZ RGZ]Z\\Y[Z", "", "", "", "", "", "", "", "D_MFKHGK RMFNGGK RMKKNGS RMKNLGS RKOK\\ RLNL\\ ROG[GYEWG RWGZF ROM]M[KYM RYM\\L RXMXXWZV[ RXYXZW[ RYMYZX\\W\\V[TZ", "D_RGSFQFQK RRGRK RHK\\KZIXK RXK[J RQKPMNPKTGX RQKRLPOMSKUGX RNQNZ ROPOY RJ[UW RJ[K\\UW RSLTPUSVUXX[[]Z RYX[Z RTPURWUZX]Z RZOVT RZO[PVT", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_MGNFLFLO RMGMO RGGHHIJJJJIIHGG RGMKL RGMHNKL RVGWFUFUN RVGVN ROJ]J[IZJ RPN[NZMYN RRRRQSPQPQR RGR]R[QZR RPRNTJWGY RRRNUKWGY RRRSUUXWZ[\\][ RSUVXYZ][ RMVMZ RNUNZ RJ[TX RJ[K\\TX RYTVW RYTZUVW", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KGLFJFJJ RKGKJ RGJNJ RMILKJNGR RLJMINJKNGR RJOJ\\ RKNK\\ RNOKR RNOOPKR RLRMSNUOUOTNSLR RUFSFQJ RUFQJ RSH]H[GZH RRJRQ RRJSKSQ RSKZK RZKZQ RYKZJ[K[Q RSNZN RRQ[Q RTRPX RTRUSPX RTT[T RZSYUWXUZS[P\\ RYTZS[TYWWYT[P\\ RTTUWWZY[\\\\][ RUWWYYZ][", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_GG]G[EYG RYG\\F ROGOPNTMV RPGPPOSMVKX RSGSTTUWUXTWS RTGTSUTVTWSWQ RJMHLH\\ RIMI\\ RIMZM RYMZL\\M[O RZMZ\\ R[M[\\ RIZZZ", "D_GG]G[EYG RYG\\F ROGOO RPGPO RTGTO RUGUO RJJJO RJJKKKO RKKYK RYKYO RXKYJZKZO RJOZO RGS]S[QYS RYS\\R RQOLV RQORPMVLV RVSTVQYN[K\\ RVSWTRYO[K\\I\\ RMVQWVYY[Z\\[\\[[ZZVXRWMV", "", "", "", "", "", "", "", "", "", "D_MGKFKT RLGLT RLGWG RVGWFYGXI RWGWT RXGXT RLKWK RLOWO RLSWS RNSMWLYJ[H\\ ROSNWMYK[H\\G\\ RSSS[T\\\\\\][\\Y R[[\\[\\Z RTSTZU[Z[\\Y]V", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 4300 to 4399 ********/
  "", "D_OFMFKJIM ROFKKGO RMHUH RTHUGWITJ RUHTJ RVHTJRL RMLKKKUJYI[ RLLLVKYI[G\\ RLLYL RXLYK[LZN RYLYXXZW[ RYYYZX[ RZLZZY\\X\\W[UZ RRLRV RSLSV RLQYQ RLVYV", "", "", "", "", "D_JEHKGM RJEKFJHGM RJHOH RNGMK RMHNGOHMK RIJITHYG[ RIJJKJTIXG[ RJKQK RQKQZP[ RPKQJRKRZQ\\P[NZ RMKMS RNKNS RJOQO RJSQS RRG\\G R\\G[KZL R[G\\F]G\\K[MYK RWGVKUM RXGWJUM RXNX\\ RXNYNY\\ RUOTT RUOVPTT RUQ\\Q[PZQ RUV]V\\U[V", "", "", "D_KGYGWFVG RHJ\\JZIYJ RKMYMWLVM RKPYPWOVP RMTKSK\\ RLTL\\ RLTWT RVTWSYTXV RWTW\\ RXTX\\ RL[W[", "", "", "D_HGPGOFNG RGJQJPIOJ RHMPMOLNM RHPPPOONP RHSH\\ RHSITI\\ RITOT ROTO\\ RNTOSPTP\\ RI[O[ RXHYGWFW\\ RXGX\\ RSN]N\\M[N", "", "", "", "", "", "D_HGNGMFLG RGJOJNIMJ RHMNMMLLM RHPNPMOLP RHSH[ RHSITI[ RITMT RMTM[ RLTMSNTN[ RIZMZ RRHZH RYHZG\\H[J RZHZP R[H[P RTORNRZS[\\[]Z\\Y RSOSYTZ[Z\\Y]W RSOZO", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HGMGLFKG RGJNJMILJ RHMMMLLKM RHPMPLOKP RHSH\\ RHSITI\\ RITLT RLTL[ RKTLSMTM[ RIZLZ ROG]G[EYG RYG\\F RVGV[ RWGW[ RSPTORNR[ RSOS[ RWP\\PZNXP RXP[O RO[][[YY[ RY[\\Z", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HGNGMFLG RGJOJNIMJ RHMNMMLLM RHPNPMOLP RHSH[ RHSITI[ RITMT RMTM[ RLTMSNTN[ RIZMZ RQIUHXGZF\\G RQIWH\\G RVHVS RWHWS RQM]M[LZM RRRR[ RRRSSS[ RSSZS RZSZ[ RYSZR[S[[ RSZZZ", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HGNGMFLG RGJOJNIMJ RHMNMMLLM RHPNPMOLP RHSH[ RHSITI[ RITMT RMTM[ RLTMSNTN[ RIZMZ RPG]G[FZG RUGSP RVGTP RQKXK RXKWP RWKXJYKXP RPP]P[OZP RTTRSR[ RSTS[ RSTYT RXTYS[TZV RYTY[ RZTZ[ RSZYZ", "D_HGNGMFLG RGJOJNIMJ RHMNMMLLM RHPNPMOLP RHSH[ RHSITI[ RITLT RLTL[ RKTLSMTM[ RIZLZ RVGWFUFUM RVGVM RPI]I[HZI RQM[MYLXM RQOPQOROSPSQQQO RQP[P RZP[O]Q[R R[P[R R\\P[RZS RSSSURYQ[ RSSTSTUSXRZP\\ RWSW[X\\\\\\][\\Z RWSXSXZY[[[\\Z]Y", "", "", "", "", "", "", "", "", "D_HGNGMFLG RGJOJNIMJ RHMNMMLLM RHPNPMOLP RHSH\\ RHSITI[ RITMT RMTM\\ RLTMSNTN\\ RI[M[ RTFSIQN RTFUGQN RYFWL RYFZGWL RRLR\\ RSLS\\ RRL]L[KZL RWLW[ RXLX[ RSQ\\Q[PZQ RSV\\V[UZV RS[][[ZZ[", "", "", "", "", "", "", "D_GGLGKFJG RFJMJLIKJ RGMLMKLJM RGPLPKOJP RGSG\\ RGSHTH\\ RHTKT RKTK[ RJTKSLTL[ RHZKZ RUFSJPM RUFVGTJRLNN RVHXKZM\\N]M RVHXJ[L]M RSNXNWMVN ROPO\\ ROPPQP\\ RPQ[Q R[Q[ZZ[ RZQ[P\\Q\\Z[\\YZ RSQS[ RTQT[ RWQW[ RXQX[ RPU[U", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 4400 to 4499 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_NGLFKHIKGM RMGKI RNGKJGM RUFWGYIZK[K\\J[IXGUF RZI[J RQJPLMPJRGS RQJRKPNNPKRGS RRKTNVPYR[S]R RRKUNXPZQ]R ROUMTM\\ RNUN\\ RNUVU RUUVTXUWW RVUV\\ RWUW\\ RN[V[", "", "", "", "", "", "", "D_IG[GYEWG RWGZF RMLKKKS RLLLS RLLWL RVLWKYLXN RWLWS RXLXS RLRWR RLTMVNYOYOXNVLT RWTTZ RWTXUTZ RG[][[YY[ RY[\\Z", "", "", "", "", "", "", "D_PENELHJJ RPENGJJGL RNGVG RUFTHRK RTGUFVGRK RRKRO RSKSO RKJKO RKJLKLO RLKYK RYKYO RXKYJZKZO RKOZO RQONQLR RSOOQLRHS RPQRTSWS[R]Q\\O[ RPQQSRWR[Q\\ RQSNULV RQTLVHW RRWLZ RRXLZH[ RSOTSVWXY[[]Z RTSVVXXZY]Z RZQVU RZQ[RVU", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_MGKFKW RLGLW RLGXG RWGXFZGYI RXGXW RYGYW RLLXL RLQXQ RLVXV RNXI[G\\ RNXOYMZG\\ RUXYZ\\\\]\\][\\ZYYUX", "", "D_PFNFLIHN RPFHN RNHVH RUHVGXIUJ RVHUJ RWHUJSL RNLLKLY RMLMY RMLYL RXLYK[LZN RYLYY RZLZY RMPYP RMTYT RMXYX RP[NZK\\H] RP[M\\H] RVZX[[]\\]\\\\Z[VZ", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 4500 to 4599 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IFIKHMGN RIFJGJKIMGN RJGPFQG RJGQG RJJRJQIPJ RMJMN RNJNN RTFTKSMRN RTFUGUKTMRN RUG[F\\G RUG\\G RUJ]J\\I[J RXJXN RYJYN RKOKY RKOLPLY RLPXP RXPXY RWPXOYPYY RLSXS RLVXV RKYYY ROZH] ROZP[H] RUZ[]\\]\\\\Y[UZ", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_RGSFQFQO RRGRO RIJ[JYHWJ RWJZI RGO]O[MYO RYO\\N RNONRMVLXJZG\\ ROOORNVMXKZG\\ RTOTXSZR[ RTZS[ RUOUZT\\S\\R[PZ RIRHUFX RIRJSHVFX RWRYS[V\\X]X]W\\UZSWR", "", "", "", "", "D_RGSFQFQZ RRGRZ RHJ[JYHWJ RWJZI RGO]O[MYO RYO\\N RRTZTXRVT RVTYS RLRKUIYG\\ RLRMSLUJXG\\ RKULWOZS[\\[]Z RLWOYSZ]Z", "", "", "", "D_MGNFLFLX RMGMY RIIQIPHOI RGNRNQMPN RMSQSPROS RIQIUHYG[ RIQJQJUIXG[ RJVKXMZP[\\[]Z RKXMYPZ]Z RRG\\G R\\G[LZM R[G\\F]G\\L[NYL RVGULTN RWGVKUMSO RSPSX RSPTQTX RTQ[Q R[Q[X RZQ[P\\Q\\X RTW[W", "", "", "D_NGLFLO RMGMO RMGXG RWGXFZGYI RXGXO RYGYO RMNXN RRNR[ RSNS[ RST\\TZRXT RXT[S RLRKUIYG\\ RLRMSLUJXG\\ RKULWOZS[\\[]Z RLWOYSZ]Z", "", "D_HFHN RHFIGIN RIGNG RNGNN RMGNFOGON RIMNM RKMKY RLMLY RHQHY RHQIQIY RLRPROQNR RGYPX RGYHZPX RRFR[ RRFSGS[ RSG]G[FZG RSLZL RZLZU RYLZK[L[U RSTZT RSZ]Z[YZZ", "", "", "", "", "", "", "", "", "", "", "", "", "D_HFHN RHFIGIN RIGNG RNGNN RMGNFOGON RIMNM RKMKY RLMLY RHQHY RHQIQIY RLRPROQNR RGYPX RGYHZPX RWFUFTISKQN RWFQN RTI\\I R[HZJXMUQ RZI[H\\IYMUQRS RTIUKWNZQ\\R]Q RWNZP]Q RSSS\\ RSSTTT\\ RTTZT RZTZ\\ RYTZS[T[\\ RT[Z[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 4600 to 4699 ********/
  "", "D_SFQFPH RSFPH ROHMGMU RNHNT RNHXH RWHXGZHYJ RXHXXWZV[ RXYXZW[ RYHYZX\\W\\V[TZ RNLXL RNPXP RHUXS RHUIVXS R\\OXSSWNZ R\\O]P[RWUTWNZK[G\\", "", "D_NFLFKI RNFKI RIHIR RIHJIJR RJIPI RPIPZO[ ROIPHQIQZP\\O[MZ RJLPL RJOPO RGRPR RGRHSPR RPQMUJX RPSNUJXGZ RZG[FYFYXXZW[ RYYYZX[ RZGZZY\\X\\W[UZ RQL]L\\K[L RSOTPURVRVQUPSO", "", "", "", "", "D_RGSFQFQ\\ RRGR\\ RHI\\IZGXI RXI[H RLLJKJU RKLKU RKLXL RWLXKZLYN RXLXU RYLYU RKPXP RKTXT RGX]X[VYX RYX\\W", "", "D_MGNFLFL\\ RMGM\\ RGIRIQHPI RHLHT RHLIMIT RIMPM RPMPT ROMPLQMQT RIPPP RISPS RGWRWQVPW RVGWFUFURTWSY RVGVRUVSYQ[ RRMYL RYLYZZ[][]W RXLYKZLZY[Z\\Z]Y", "", "", "", "", "D_MGNFLFL\\ RMGM\\ RGIRIQHPI RHLHT RHLIMIT RIMPM RPMPT ROMPLQMQT RIPPP RISPS RGWRWQVPW RSI\\I[HZI RRP]P[OZP RWPUZ RWPXQUZ RRZ[Y RRZS[[Y RZV[X\\[][]Z\\XZV", "", "", "", "D_LGMFKFK\\ RLGL\\ RGIPIOHNI RGKGS RGKHLHS RHLOL ROLOS RNLOKPLPS RHOOO RHROR RGVPVOUNV RXGYFWFWZ RXGXZ RSKS\\ RSKTLT\\ RTL[L R[L[\\ RZL[K\\L\\\\ RTS[S RTZ[Z", "D_MGNFLFL\\ RMGM\\ RGIRIQHPI RHLHT RHLIMIT RIMPM RPMPT ROMPLQMQT RIPPP RISPS RGWRWQVPW RSGZG RZGYJXLUP RYGZF[GZJXMUPRR RTHUKWNZP\\Q]P RUKWMZO]P RWSXRVRV[ RWSW[ RSV[VZUYV RQ[][\\Z[[", "", "", "D_MGNFLFL\\ RMGM\\ RGIRIQHPI RHLHT RHLIMIT RIMPM RPMPT ROMPLQMQT RIPPP RISPS RGWRWQVPW RXIXGYFWFWI RSI]I\\H[I RUKSN RUKVLSN RZK[L\\N]N]M\\LZK RZPYSWVTYQ[O\\ RZP[QZSXVUYR[O\\ RSRVP RSRTSVP RVPWTYX\\\\][ RVPWSXUZX][", "", "", "", "", "", "", "D_LGMFKFK\\ RLGL\\ RGIPIOHNI RGKGS RGKHLHS RHLOL ROLOS RNLOKPLPS RHOOO RHROR RGVPVOUNV RWFVHTKQN RWFXGVJTLQN RWGXIZL\\N]M RXIZK]M RTNYNXMWN RRPR\\ RRPSQS\\ RSQ[Q R[Q[ZZ[ RZQ[P\\Q\\Z[\\Z[YZ RUQU[ RVQV[ RXQX[ RYQY[ RSV[V", "", "", "D_LGMFKFK\\ RLGL\\ RGIPIOHNI RGKGS RGKHLHS RHLOL ROLOS RNLOKPLPS RHOOO RHROR RGVPVOUNV RRG]G\\F[G RTJTP RTJUKUP RUKZK RZKZP RYKZJ[K[P RUOZO RSRS\\ RSRTST\\ RTS[S R[S[\\ RZS[R\\S\\\\ RWSW[ RXSX[ RTW[W RT[[[", "", "", "", "", "", "", "", "", "", "", "", "", "D_RGSFQFQI RRGRI RHI\\IZGXI RXI[H RLJMLNPOPONNLLJ RVJTP RVJWKTP RGQ]Q[OYQ RYQ\\P RQQQ] RRQR] RIV[VYTWV RWVZU", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKWGZ RLPLW RLWH[GZ RLWNZQ[\\[]Z RLWNYQZ]Z RPGTG RSGTFVGUI RTGTJSMQQNU RUGUJTMSOQRNU RUJVMWOYR\\U]T RVMXPZR]T", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKWGZH[LW RLPLW RLWNZQ[\\[]Z RLWNYQZ]Z ROH[H RZH[G]H\\J R[HZRYTXU RZSZTYU R\\H\\J[TZVYVXUVT RUHTLSORQPT RVHULSPPTNV", "", "", "", "", "", "", "", "", "", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKWGZH[LW RLPLW RLWNZQ[\\[]Z RLWNYQZ]Z RRHPGPPOTNV RQHQPPSNV RRHVGYF[G RQHRH[G RQM]M[LZM RWMWW RXMXW", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKWGZH[LW RLPLW RLWNZQ[\\[]Z RLWNYQZ]Z RRFSITITHRF RZFXFWJ RZFWJ ROJ]J\\I[J RUJUTTVSW RVJVTUVSWQW RPMPS RPMQMQS RQRZR RZMZS RZM[M[S", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 4700 to 4799 ********/
  "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKW RLPLW RKWGZ RLWH[GZ RLWNZQ[\\[]Z RLWNYQZ]Z RVGWFUFUX RVGVX ROI]I[HZI RPLPQ RPKQLQQ RQLZL RZLZQ RYLZK[L[Q RPQ[Q RUQSTQV RUSTTQVOW RVSXT[W\\W\\V[UYTVS", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKWGZH[LW RLPLW RLWNZQ[\\[]Z RLWNYQZ]Z RVGWFUFUM RVGVM RSGQGPK RSGPK RQI[IZHYI ROM]M[LZM RQPQW RQPRQRW RRQYQ RYQYW RXQYPZQZW RRVYV", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKWGZH[LW RLPLW RLWNZQ[\\[]Z RLWNYQZ]Z RVGWFUFUW RVGVW ROH]H\\G[H RPJPQ RPJQKQQ RQKZK RZKZQ RYKZJ[K[Q RQNZN RPQ[Q ROT]T\\S[T", "D_HGIHJJKJKIJHHG RGPJP RIPJOLPKR RJPJWGZH[KW RKPKW RKWMZP[\\[]Z RKWMYPZ]Z ROG[G RYGZF\\HYH RZFYH R[GYHWI RSHTIUKVKVJUISH RQKOJOW RPKPW RPK[K RZK[J]K\\M R[K[UZV R\\K\\U[WZVXU RUKUW RVKVW RPO[O RPS[S", "", "", "", "", "", "D_HGIHJJKJKIJHHG RGPJP RIPJOLPKR RJPJWGZH[KW RKPKW RKWMZP[\\[]Z RKWMYPZ]Z RSFQFPINM RRFQH RSFNM RYFWFVJ RXFWH RYFVJ RPIPW RPIQJQW RQJ]J[IZJ RVJVV RWJWV RQN\\N[MZN RQR\\R[QZR RQV]V[UZV", "", "", "", "", "", "", "", "", "", "", "", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKWGZH[LW RLPLW RLWNZQ[\\[]Z RLWNYQZ]Z RVFWEUEUK RVFVK RQH[HZGYH ROK]K[JZK RRKSOTOTNRK RYKWO RYKZLWO ROO]O[NZO RUOUX RVOVX RRRZRYQXR ROU]U[TZU", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKW RLPLW RKWGZ RLWH[GZ RLWNZQ[\\[]Z RLWNYQZ]Z RPFPQOVNX RPFQGQQPUNX RQG\\G R\\G\\J R[G\\F]G]J RQJ]J RSJTNUNUMSJ RYJXN RYJZKXN RVNVX RWNWX RQN]N\\M[N RRQ\\Q[PZQ RQT]T\\S[T", "D_HGIHJJKJKIJHHG RGPJP RIPJOLPKR RJPJWGZH[KW RKPKW RKWMZP[\\[]Z RKWMYPZ]Z RQFQN RQFRGRN RRGYG RYGYN RXGYFZGZN RUIUN RUIVJVN RVJYJ ROMOW ROMPNPW RPN[N R[N[VZW RZN[M\\N\\V[XZWXV RSPSU RSPTQTU RTQWQ RWQWU RVQWPXQXU RSUXU", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKWGZH[LW RLPLW RLWNZQ[\\[]Z RLWNYQZ]Z RPFQGRISISHRGPF RXFWJ RXFYGWJ RNJ]J[IZJ RUJTM RWJTM RRMPLPW RQMQW RQMYM RXMYL[MZO RYMYW RZMZW RQPYP RQSYS RQVYV", "D_HGIHJJKJKIJHHG RGPJP RIPJOLPKR RJPJWGZH[KW RKPKW RKWMZP[\\[]Z RKWMYPZ]Z RPFNHNIOIPF RPG[G RZG[F]H[H R[F[H R\\G[HZI RVIWHUHUX RVIVX RPJ[JZIYJ RPLPS RPLQMQS RQMZM RZMZS RYMZL[M[S RQPZP RPS[S RNV]V[UZV", "", "", "", "", "", "", "", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKWGZH[LW RLPLW RLWNZQ[\\[]Z RLWNYQZ]Z RVFWEUEUK RVFVK RQH[HZGYH ROK]K[JZK RQMQQ RQMRNRQ RRNYN RYNYQ RXNYMZNZQ RQQZQ RUQUX RVQVX RUQRTOV RVRQUOV RVSXT[W\\W\\V[UYTVS R[RYT R[R\\SYT", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IGJHKJLJLIKHIG RGPKP RJPKOMPLR RKPKWGZH[LW RLPLW RLWNZQ[\\[]Z RLWNYQZ]Z RSFSI RTFTI RWFWI RXFXI ROEOI ROEPFPI RPF[F R[F[I RZF[E\\F\\I ROI\\I RNL]L[KZL RPNPR RPNQOQR RQOZO RZOZR RYOZN[O[R RPR[R RURUX RVRVX RURRUOW RVSQVOW RWTYU[W\\W\\VZUWT R[SYU R[S\\TYU", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_GGSGRFQG RKGKNJR RLGLNKQIS RNGNQORQR ROGOPPQQQ RHKH[ RHKILI[ RILQL RQLQ[ RPLQKRLR[ RIVQV RIZQZ RWFVJ RWFXGVJ RSJ[J RSJTK[J R[H\\K]K]J[H RVJVMUOTP RWJWMVOTPSP RYJYOZP]P]M RZJZO\\O]N RVQUTTV RVQWRTV RVS\\S R[RZTXWUZ R[R\\SZVXXUZS[ RVQWUYYZZ\\[]Z RXWZY]Z", "", "", "", "", "", "", "", "", "D_GGQGPFOG RJGJNIR RKGKNJQHS RMGMQNRPR RNGNPOQPQ RGKG[ RGKHLH[ RHLPL RPLP[ ROLPKQLQ[ RHVPV RHZPZ RWEVH RWEXFVH RSGSP RSGTHTP RTH[H R[H[P RZH[G\\H\\P RWHWP RXHXP RTL[L RSP\\P RTRTUSYR[ RTRURUUTXR[ RWQWZX[][]X RWQXQXYYZ\\Z]Y R[RZRYU R[RYU RXU\\U RXUYV\\U R\\T\\V]V\\T", "", 
  /******** Oriental Hershey Glyphs 4800 to 4899 ********/
  "", "", "", "", "", "", "", "", "", "D_GHKGNFPG RGHPG RKGK\\ RLGL\\ RHJIMJMJLHJ ROIMM ROIPJMM RFNQNPMON RKOISGV RKQGV RMQNROTPTPSORMQ RSFSSRXQZ RSFTGTSSWQZO\\ RTG[G R[G[O RZG[F\\G\\O RTN[N RWNXSYVZX\\[]Z RWNXRYU[X]Z", "", "D_IHQGWFYG RIHYG RQGQS RRGRS RLIMLNLNKLI RWIUL RWIXJUL RGM]M[LZM RPMNOJRGT RQNNPISGT RSMTOWR[T]S RTOWQYR]S RQUQ[ RRUR[ RMUKTK\\ RLUL\\ RLUWU RVUWTYUXW RWUW\\ RXUX\\ RLXWX RL[W[", "", "D_LGJFJR RKGKR RKGXG RWGXFZGYI RXGXR RYGYR RQGQ[ RRGR[ RKLXL RKQXQ RIV[VYTWV RWVZU RG[][[YY[ RY[\\Z", "", "D_SFQFPHNKKNHP RRFPI RSFPJMMJOHP RSGTIVLXN[P]O RYN[O RTIWLZN]O RNOVOUNTO RQOQ[ RROR[ RISZSYRXS RKULWMZNZNYMWKU RXUUZ RXUYVUZ RG[][[YY[ RY[\\Z", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_KEIJGN RKELFKHGN RMGNHOJPJPIOHMG RIMOMNLMM RKMKZ RLMLZ RGQPQOPNQ RGSHWIWIVGS ROSMX ROSPTMX RGZPY RGZH[PY RWGXFVFVL RWGWL RRKRTQXPZ RRKSLSTRWPZO[ RSL]L[KZL RXNUY RXNYOUY RSZ[X RSZT[[X RZV[X\\[][]Z\\XZV", "D_KEIJGN RKELFKHGN RMGNHOJPJPIOHMG RIMOMNLMM RKMKZ RLMLZ RGQPQOPNQ RGSHWIWIVGS ROSMX ROSPTMX RGZPY RGZH[PY RWGXFVFVRUVSYQ[ RWGWRVVTYQ[N\\ RSHRLQN RSHTISKQN RSK\\K[JZK RPQ]Q[PZQ RWRXVZZ\\\\][ RXVZY][", "", "", "", "", "", "", "", "", "D_KEIJGN RKELFKHGN RMGNHOJPJPIOHMG RIMOMNLMM RKMKZ RLMLZ RGQPQOPNQ RGSHWIWIVGS ROSMX ROSPTMX RGZPY RGZH[PY RRFR\\ RRFSGS\\ RSG[G R[G[ZZ[ RZG[F\\G\\Z[\\Z[XZ RUKYKXJWK RUNUX RUNVOVX RVOXO RXOXW RWOXNYOYW RVVXV", "", "D_KEIJGN RKELFKHGN RMGNHOJPJPIOHMG RIMOMNLMM RKMKZ RLMLZ RGQPQOPNQ RGSHWIWIVGS ROSMX ROSPTMX RGZPY RGZH[PY RRFR[ RRFSGSZ RSGZG RZGZO RYGZF[G[O RSKZK RSO[O RWPWRXVZY\\[]Z RWRXUZX]Z R\\QYU R\\Q]RYU RP[XX RP[Q\\XX", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_IEFK RIEJFFK RJGLJMJMIJG RGMMMLLKM RIMIZ RJMJZ RFQNQMPLQ RFSGWHWHVFS RMSKW RMSNTKW RGZMY RGZH[MY RQGOFO\\ RPGP\\ RPG[G RZG[F]G\\I R[G[ZZ[ R\\G\\Z[\\Z[XZ RRHSLTLTKRH RXHWL RXHYIWL RQMZMYLXM RUMUW RVMVW RRPRX RRPSPSX RSWXW RXPXX RXPYPYX", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 4900 to 4999 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_MGKFKZ RLGLZ RLG[GYFXG RLJXJWIVJ RLMXMWLVM RGP]P[OZP RPQQTSWUYX[[\\][ RPQQSSVTWWYYZ][ RXRUV RXRYSUV RG[SX RG[H\\SX", "", "D_HFH\\ RHFIGI\\ RIGOG ROGOP RNGOFPGPP RIKOK RIOOO RTFTP RTFUGUP RUG[G R[G[XZZY[ R[Y[ZZ[ RZG[F\\G\\Z[\\Z\\Y[WZ RUK[K RUO[O", "", "", "", "D_HFH\\ RHFIGI\\ RIGOG ROGOP RNGOFPGPP RIKOK RIOOO RTFTP RTFUGUP RUG[G R[G[ZZ[ RZG[F\\G\\Z[\\Z[XZ RUK[K RUO[O RNRNZ RNROSOZ ROSUS RUSUZ RTSURVSVZ ROYUY", "D_HFH\\ RHFIGI\\ RIGOG ROGOM RNGOFPGPM RIJOJ RIMPM RTFTM RTFUGUM RUG[G R[G[ZZ[ RZG[F\\G\\Z[\\Z[XZ RUJ[J RTM[M RRORZQ[ RROSOSZR\\Q[OZ RKRXRWQVR RQRMWKY RRSPUKY", "", "", "", "D_HFH\\ RHFIGI\\ RIGOG ROGOM RNGOFPGPM RIJOJ RIMPM RTFTM RTFUGUM RUG[G R[G[ZZ[ RZG[F\\G\\Z[\\Z[XZ RUJ[J RTM[M RNPNZ RNPOQOZ ROQUQ RUQUZ RTQUPVQVZ ROUUU ROYUY", "D_HFH\\ RHFIGI\\ RIGOG ROGOM RNGOFPGPM RIJOJ RIMPM RTFTM RTFUGUM RUG[G R[G[ZZ[ RZG[F\\G\\Z[\\Z[XZ RUJ[J RTM[M RMPXPWOVP ROPOUNXMZ RPPPUOXMZK[ RTPT[ RUPU[ RLTXTWSVT", "", "", "", "", "", "", "", "D_HFH\\ RHFIGI\\ RIGOG ROGOM RNGOFPGPM RIJOJ RIMPM RTFTM RTFUGUM RUG[G R[G[ZZ[ RZG[F\\G\\Z[\\Z[XZ RUJ[J RTM[M RMONPORPRPQOPMO RUOSS RUOVPSS RLSXSWRVS RQSQUPXOZ RRSRUQXOZM[ RRTSWTYV[WZ RSWUYWZ RKVYVXUWV", "D_HFH\\ RHFIGI\\ RIGOG ROGOM RNGOFPGPM RIJOJ RIMPM RTFTM RTFUGUM RUG[G R[G[ZZ[ RZG[F\\G\\Z[\\Z[XZ RUJ[J RTM[M RLPXPWOVP RNPNZ ROPOZ RUPU[ RVPV[ ROSUS ROVUV RLZXX RLZM[XX", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HFH[ RHFIGI[ RIGNG RMFKM RLGMFNGKM RKMLQLTKUJU RKMMQMTLVKVJUHT RQFQY RQFRGRX RRGYG RYGYO RXGYFZGZO RRKYK RROZO RUOVRWTYW\\Z]Y RVRXUZW]Y RNYXW RNYOZXW R[QXT R[Q\\RXT", "", "", "", "", "", "D_HFH[ RHFIGI[ RIGNG RMFKM RLGMFNGKM RKMLQLTKUJU RKMMQMTLVKVJUHT RUFSIPL RUFVGSJPLNM RUFWIXJZL\\M]L RXJZK]L RRNYNXMWN RUNUYTZ RVNVYU[TZRY ROR\\R[QZR RQUPWNZ RQURVNZ RYUZW[Z\\Z\\Y[WYU", "D_HFH[ RHFIGI[ RIGNG RMFKM RLGMFNGKM RKMLQLTKUJU RKMMQMTLVKVJUHT RUFSFRIPL RUFQKOM RRI[I RZHXKUNRP RYIZH[IXLTOPQ RTIUKWNZP\\Q]P RUKWMZO]P RVQV\\ RVQWQW\\ RRRRW RRRSSSW RSS[SZRYS RNW]W[VZW", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 5000 to 5099 ********/
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
  /******** Oriental Hershey Glyphs 5100 to 5199 ********/
  "", "", "", "", "", "", "", "", "", "", "D_RHRGSFQFQH RHH[HYGXH RLHMJNMOMOLNJLH RVHTM RVHWITM RGM]M[LZM RMQKPK\\ RLQL\\ RLQWQ RVQWPYQXS RWQW\\ RXQX\\ RLVWV RL[W[", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_MGNFLFLQ RMGMQ RHGJJKJKIHG RPGNJ RPGQHNJ RGKRKQJPK RKKJMHP RLLHP RNMONPPQPQOPNNM RLRKWJY RLRMRLVKXIZG[ RGURUQTPU RNWOXPZQZQYPXNW RSG]G\\F[G RXGWK RXGYHWK RTJTW RTJUKUW RUK[K R[K[W RZK[J\\K\\W RUO[O RUS[S RTW\\W RUYR\\ RUYVZR\\ RZY[Z\\\\]\\][\\ZZY", "", "", "", "", "", "", "", "", "", "D_KGIFIUHZG\\ RJGJUIYG\\ RJGYG RXGYF[GZI RYGYUZY[[\\\\]\\]Z]X R[Z\\[][ RZGZU[Y\\Z]Z RLKUIVJ RLKVJ RQKQZ RRJRY RMMMU RMMNNNU RNNUN RUNUU RTNUMVNVU RNTUT RLZVY RLZM[VY RUWW[X[XZUW", "", "", "", "D_GGTG RSGTFVGUI RTGTIULWOZQ\\Q]P\\O RUGUIVLXOZP[P\\O]M RYGVK RYGZHVK RWKYMZMZLWK RGMLKML RGMML RJLJTIYH[ RKLKTJXH[G\\ RPJQIOIO\\ RPJP\\ RGRTR RSRTQVRUT RTRTTUWWZZ\\\\\\][\\Z RURUTVWXZZ[[[\\Z]X RYRVV RYRZSVV RWVYXZXZWWV", "", "D_QFPHMLJNGO RQFRGPJNLKNGO RRGTJVLYN[O]N RRGUJXLZM]N RRORNSMQMQO RLNL\\ RLNMOM[ RMOVO RVOVU RUOVNWOWU RMRVR RMUWU RQURWTZW\\[]]\\ RRWTYX[]\\ RI\\SZ RI\\J]SZ RXVUY RXVYWUY", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_MFNGOIPIPHOGMF RUFTJ RUFVGTJ RGJ]J[HYJ RYJ\\I RRJPN RRJSKPN RMNKMK\\ RLNL\\ RLNXN RWNXMZNYP RXNX\\ RYNY\\ RLRXR RLVXV RLZXZ", "", "D_JHQGVFXG RJHXG RQGQS RRGRS RGK]K[IYK RYK\\J RPKMNIQGR RQLMOJQGR RRKTMVOYQ[R]Q RTMWOYP]Q RNSLRL\\ RMSM\\ RMSVS RUSVRXSWU RVSV\\ RWSW\\ RMWVW RM[V[", "", "", "D_IFIT RIFJGJT RJG[GYFXG RQGQS RRGRS RJKZKYJXK RJOZOYNXO RJS[S RZS[R]S[U R[SZZY[ R\\S[ZZ\\Y[WZ RIWHYGZG[H[IYIW RLWM[N[NZLW RQVRZSZSYQV RUUVVWXXXXWWVUU", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 5200 to 5299 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_HFHQ RHFIGIQ RIGQGPFOG RLGLP RMGMP RIJQJPIOJ RIMQMPLOM RIPPP RPPPVOZN[ ROPPOQPQVPZO\\N[LZ RHUGXHXHU RJUJXKWJU RLTMWNVLT RMROTOSMR RWFTJRL RWFXGVIRL RXGZJ\\L]K RXGZI]K RUMZMYLXM RWMWTVXUZT[R\\ RXMXTWXVZU[R\\ RSOSU RSOTPTU RTP[P R[P[U RZP[O\\P\\U RTT[T RXUYXZZ\\\\][ RYX[Z][", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_NGLFLN RMGMN RMGWG RVGWFYGXI RWGWN RXGXN RQIQN RQIRJRN RRJWJ RIMHOGPGQHQIOIM RIN\\N RZN[M]O[P R[M[P R\\N[PZQ RNQLPL\\ RMQM\\ RMQWQ RVQWPYQXS RWQWXVZU[ RWYWZV[ RXQXZW\\V\\U[SZ RMTWT RMWWW", "", "", "", "", "", "", "", "", "", "", "", "D_RHRFSEQEQH RGH]H[GZH RLJLN RLJMKMN RMKVK RVKVN RUKVJWKWN RLNWN RHPH\\ RHPIQI\\ RIQZQ RZQZZY[ RYQZP[Q[ZZ\\Y[WZ RMSMX RMSNTNX RNTUT RUTUX RTTUSVTVX RMXVX", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_REPEOH RREOH RLHJGJQ RKHKQ RKHXH RWHXGZHYJ RXHXQ RYHYQ RQHQP RRHRP RKLXL RKPXP RNQMVLXJZH[ RNQORNVMXKZH[G[ RRQRZS[\\[]Z\\Y RRQSQSYTZ[Z\\Y]W RXRWRVU RXRVU RUUYU RUUVVYU RYTZV[V[UYT", "", "", "", "", "D_PENELIJLGP RPELJGP RNHVH RTHUGWITJ RUGTJ RVHTJRL RMLKKKV RLLLV RLLYL RXLYK[LZN RYLYV RZLZV RRLRV RSLSV RLQYQ RKVZV RIXHZG[G\\H\\IZIX RMXN\\O\\O[MX RSXT\\U\\U[SX RXXYY[\\\\\\\\[ZYXX", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", 
  /******** Oriental Hershey Glyphs 5300 to 5399 ********/
  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_SEQEPG RSEPG RMGKFKT RLGLT RLGWG RVGWFYGXI RWGWM RXGXM RLJWJ RLMXM RLP[PYOXP RLS[S RZS[R]S[U R[SZZY[ R\\S[ZZ\\Y[WZ RIWHYGZG[H[IYIW RLWM[N[NZLW RQVRZSZSYQV RUUVVWXXXXWWVUU", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_SHSGTFRFRH RIGIUHZG\\ RIGJHJUIYG\\ RJH]H[GZH ROHOP RPHPP RUHUP RVHVP RJL[L R[L[P RZL[K\\L\\P RJP\\P ROQOZ ROQPQPZ RPUTUSTRU RK[UX RK[L\\UX RUQU[V\\\\\\][\\Z RUQVQVZW[[[\\Z]X RVV[S\\T RVV\\T", "", "", "", "", "", "", "", "", "", "D_RGSFQFQO RRGRO RHI\\IZHYI RILZLYKXL RGO]O[NZO RPPNRJUGW RPPQQNSKUGW RNSXS RWRSVOYK[H\\ RWRXSSWPYL[H\\ RNSOUQXTZW[[\\][ ROUQWTYWZ][", "", "", "", "", "D_SHSGTFRFRH RIGIVHZG\\ RIGJHJVIYG\\ RJH]H[GZH ROKPJNJN\\ ROKO\\ RJNTNSMRN RNPMTLWKY RNTMVKY RPQQRRTSTSSRRPQ RXKYJWJW\\ RXKX\\ RTN]N[MZN RWPVTUVSY RWTVVSY RXQYTZW\\Z]Y RYT[W]Y", "", "", "", "", "", "", "", "", "D_OGPFNFNM ROGOM RUGVFTFTM RUGUM RHI\\IZHYI RGM]M[LZM RQMQX RRMRX RLPJOJX RKPKX RKPXP RWPXOZPYR RXPXX RYPYX RKTXT RJXYX RNYG\\ RNYOZG\\ RUYXZ\\\\]\\][ZZUY", 
  /******** Oriental Hershey Glyphs 5400 to 5499 ********/
  "", "", "", "D_QGQV RRGRV RLGJFJP RKGKP RKGXG RWGXFZGYI RXGXP RYGYP RKKXK RKOXO RISZSYRXS RGV]V[UZV RIXHZG[G\\H\\IZIX RMXN\\O\\O[MX RSXT\\U\\U[SX RXXZZ[\\\\\\\\[ZYXX", "D_QFQR RRFRR RJEJL RJEKFKL RKFXF RXFXL RWFXEYFYL RKIXI RJLYL RIOZOYNXO RGR]R[QZR RITGUGVHVIT RMTNVOVOUMT RTTUVVVVUTT RZT[V\\V\\UZT RQVQ[ RQVRVR[ RJXZXYWXX RG[][[ZZ[", "", "", "", "", "", "", "", "", "", "", "D_MGNFLFLM RMGMM RGISIQHPI RHMRMQLPM RIOIT RIOJPJT RJPOP ROPOT RNPOOPPPT RITPT RIUJXKXKWIU ROUNX ROUPVNX RGZRX RGZH[RX RWGXFVFVQ RWGWQ RRK]K[JZK RRQZQ RZQYTWWUYR[ RYQZP[QZTXWVYR[O\\ RSRTUVXYZ\\[]Z RTUVWYY]Z", "", "", "", "", "", "D_SEQEPG RSEPG RKFKM RKFLGLM RLGWG RWGWM RVGWFXGXM RLIWI RLKWK RKMXM RQOQU RRORU RJOJU RKOKU RJOYO RXOXU RYOYU RKRXR RJUYU RNUNZM\\L] ROUOZN\\L]J] RTUT] RUUU] RGX]X[WZX", "", "", "", "", "", "", "D_RFSEQEQK RRFRK RLHMGKGKK RLHLK RRHXHWGVH RGK]K[JZK RRNSMQMQY RRNRY RLMMNNPOPOONNLM RVMTQ RVMWNTQ RJRYRXQWR RPRMVKX RQSOUKX RSTTUVXWXWWVVST RHMH\\ RHMIMI\\ RI[Z[ RZMZ\\ RZM[M[\\", "", "", "", "", "", "", "", "", "", "", "", "D_RHRGSFQFQH RHH\\HZGYH RLHNLOLOKLH RVHTL RVHWITL RGL]L[KZL RLOJNJX RKOKX RKOXO RWOXNZOYQ RXOXX RYOYX RKSXS RKWXW RQOQ[R\\\\\\][\\Y R[[\\[\\Z RRORZS[Z[\\Y]V", "", "", "", "", "D_NELEJHGL RNEGL RMFSF RRFSEUGRH RSFRH RTFRHQI RJHJO RJHKIKO RKIXI RXIXO RWIXHYIYO RKLXL RJOYO RQIQ[R\\\\\\][\\Z RRIRZS[[[\\Z]X RIQIY RIQJRJY RJRYR RYRYY RXRYQZRZY RJUYU RJXYX", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "D_", 
};
#else  /* NO_KANJI */
const char * const _pl_g_oriental_hershey_glyphs[NUM_ORIENTAL_HERSHEY_GLYPHS] = { };
#endif /* KANJI */
