/*
 *  Copyright 2008-2013 NVIDIA Corporation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


/*! \file unique.inl
 *  \brief Inline file for unique.h.
 */

#pragma once

#include <thrust/detail/config.h>
#include <thrust/system/detail/generic/unique.h>
#include <thrust/iterator/iterator_traits.h>
#include <thrust/transform.h>
#include <thrust/unique.h>
#include <thrust/detail/temporary_array.h>
#include <thrust/detail/internal_functional.h>
#include <thrust/detail/copy_if.h>
#include <thrust/distance.h>
#include <thrust/functional.h>

namespace thrust
{
namespace system
{
namespace detail
{
namespace generic
{


template<typename DerivedPolicy,
         typename ForwardIterator>
__host__ __device__
  ForwardIterator unique(thrust::execution_policy<DerivedPolicy> &exec,
                         ForwardIterator first,
                         ForwardIterator last)
{
  typedef typename thrust::iterator_traits<ForwardIterator>::value_type InputType;

  return thrust::unique(exec, first, last, thrust::equal_to<InputType>());
} // end unique()


template<typename DerivedPolicy,
         typename ForwardIterator,
         typename BinaryPredicate>
__host__ __device__
  ForwardIterator unique(thrust::execution_policy<DerivedPolicy> &exec,
                         ForwardIterator first,
                         ForwardIterator last,
                         BinaryPredicate binary_pred)
{
  typedef typename thrust::iterator_traits<ForwardIterator>::value_type InputType;
  
  thrust::detail::temporary_array<InputType,DerivedPolicy> input(exec, first, last);
  
  return thrust::unique_copy(exec, input.begin(), input.end(), first, binary_pred);
} // end unique()


template<typename DerivedPolicy,
         typename InputIterator,
         typename OutputIterator>
__host__ __device__
  OutputIterator unique_copy(thrust::execution_policy<DerivedPolicy> &exec,
                             InputIterator first,
                             InputIterator last,
                             OutputIterator output)
{
  typedef typename thrust::iterator_value<InputIterator>::type value_type;
  return thrust::unique_copy(exec, first,last,output,thrust::equal_to<value_type>());
} // end unique_copy()


template<typename DerivedPolicy,
         typename InputIterator,
         typename OutputIterator,
         typename BinaryPredicate>
__host__ __device__
  OutputIterator unique_copy(thrust::execution_policy<DerivedPolicy> &exec,
                             InputIterator first,
                             InputIterator last,
                             OutputIterator output,
                             BinaryPredicate binary_pred)
{
  // empty sequence
  if(first == last)
    return output;
  
  thrust::detail::temporary_array<int,DerivedPolicy> stencil(exec, thrust::distance(first, last));
  
  // mark first element in each group
  stencil[0] = 1; 
  thrust::transform(exec, first, last - 1, first + 1, stencil.begin() + 1, thrust::detail::not2(binary_pred)); 
  
  return thrust::copy_if(exec, first, last, stencil.begin(), output, thrust::identity<int>());
} // end unique_copy()


} // end namespace generic
} // end namespace detail
} // end namespace system
} // end namespace thrust

