#include <asm/types.h>
#include "mport.h"
#include "common.h"

#include <sys/ioctl.h>
#include <sys/mman.h>		/*** mmap */
#include <string.h>		/*** memcmp */
#include <sys/io.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/param.h>		/** MIN */

#include <sched.h>

/*** sigaction */
#include <signal.h>

#include "sysdep.h"
#include "log.h"
#include "core.h"

#include "watch.h"
#include "../martian.h"

#define MSIGVI (SIGRTMIN + 1)

extern int pin_thread (void);
extern int smp_setup (int irq, int dev_fd);

void *serve_interrupts (void *arg); 
extern int setup_virtual_irq (mport_t *port);
extern int link_to_driver (char *program, struct martian_common *common); 
extern void print_core_vars_common (struct martian_common *common); 
extern void ask_driver_vars (int dev_fd); 
extern void print_irq_state (int dev_fd);

#define MSR_DDCD	0x008
#define MSR_DCD		0x080

/*** module vars ***/
int mfd;
/**************************/

static void init_martian_core (struct martian_conf *mconf) {

	BaseAddress 	= mconf->BaseAddress;
	ComAddress 	= mconf->CommAddress;
	Irq 		= mconf->irq;

	if (mconf->flags & MARTIAN_DEV_TOSHIBA)
		ToshibaFlag = 1;

	{
		extern char eeprom[];
		eeprom[0] = config.country_id;
	}

	lucent_init_modem();
}

static int vuart_tid;

// timer tick handler; call core's function
static void softirq_handler (int sig_num, siginfo_t *info, void *ctx) {
	extern struct _actionwatch pulse_latency;
	mport_t *port = info->si_value.sival_ptr;

	if (sig_num == MSIGVI) 
		logdebugadd (5, "t");
	else if (sig_num == SIGALRM) 
		logdebugadd (5, "a");

	if (port->in_pulse) {
		logdebugadd (2, "e"); /* re-enter */
	}
	if (port->pulse) logdebugadd (2, "r"); /* repeated */

	if (vuart_tid != mgettid()) { /* different thread */
		logdebugadd (2, ".");
		/* vuart_tid = gettid(); */
	}

	port->pulsesAll++;
	port->pulse = 1;

	if (Debugged (Timings)) 
		watch_start (&pulse_latency);
}

static void inline do_pulse_core (mport_t *port) {
	extern struct _actionwatch pulse_latency;

	port->pulse = 0;
	port->in_pulse = 1;

	if (Debugged (Timings)) 
		watch_stop (&pulse_latency);

	mport_lock (port);
	timertick_function (0);
	mport_unlock (port);

	port->in_pulse = 0;
}

mtimer_t vitimer;

/* virtual interrupt in session & we already have exlusive access */
extern void virtual_rs_interrupt (int );

static int setup_vi_timer (mport_t *port)
{
	struct sigevent vievent;

	/* set up event */
	memset (&vievent, 0, sizeof vievent);

	// ptyevent.sigev_value.sival_int = pty; 
	vievent.sigev_signo = MSIGVI;

	/* works with syscalls but fails with glibc implementation */
	vievent.sigev_notify = SIGEV_THREAD_ID;
	vievent.sigev_notify_thread_id = mgettid();
	vievent.sigev_value.sival_ptr = port;

	int res = mtimer_create (CLOCK_REALTIME, &vievent, &vitimer);
	if (res >= 0) {
		LOGDEBUG (2, "set timer for thread\n");
		return 0;
	}

	LOGWARN ("setting timer 0x%x by thread %d failed\n", vitimer.id, mgettid());
	LOGSYSERR (Debug, "timer");
	
	/*** second try ***/
	memset (&vievent, 0, sizeof vievent);

	vievent.sigev_signo = MSIGVI;
	vievent.sigev_notify = SIGEV_SIGNAL;
	/***/
	res = mtimer_create (CLOCK_REALTIME, &vievent, &vitimer);


	if (res < 0) {
		LOGERR ("cannot setup timer 0x%x, won't work\n", vitimer.id);
		return res;
	}

	LOGDEBUG (2, "virtirq: set timer\n");
	return 0;
}

void (*lt_rs_interrupt) (int );
int setup_virtual_irq (mport_t *port) {
	struct sigaction timer_action;

	if (setup_vi_timer (port) != 0) 
		return -1;

	//timer_action.sa_handler = softirq_handler;
	timer_action.sa_sigaction = softirq_handler;
	timer_action.sa_flags = SA_SIGINFO;
	sigemptyset (&timer_action.sa_mask);
	sigaction (SIGALRM, & timer_action, NULL);
	sigaction (MSIGVI, & timer_action, NULL);

	vuart_tid = mgettid();
	lt_rs_interrupt = virtual_rs_interrupt;
	LOGDEBUG (2, "virtirq: set handler\n");

	return 0;
}


static inline int mport_has_data (mport_t *port) {
	return port->available > 0;
}


static inline int mport_ready (void ) {
	return  io_app_tx_count <= 0x7be;
}

static inline int mport_available (void ) {
	return UART_CommReadCount();
}

static size_t page_align(size_t size) {
	size_t npagemask = sysconf(_SC_PAGESIZE) - 1;
	return (size + npagemask) & ~npagemask;
}

/*** port interface ***/

port_status_t mport_init (mport_t *port)
{
	port->lock = mutex_create();
	if (! port->lock) {
		LOGERR ("Cannot create mutex\n");
		return 5;
	}

	port->mctrl = 0;
	port->carrier = COk;

	if (system ("grep -q '/proc ' /proc/mounts 2> /dev/null") != 0) {
		LOGERR ("procfs on /proc is not found\n");
		return 1;
	}

	char *device_path = "/proc/driver/mars/0"; // dev_name

	int dev_fd = open (device_path, O_RDWR);
	if (dev_fd < 0) {
		LOGSYSERR (Error, "open");
		if (errno == EACCES) 
			return 1;
		else if (errno == EBUSY) {
			/* for multiple device this should iterate next */
			LOGINFO ("martian is already running.\n");
			return 0;
		}
		else if (errno == ENOENT) {
		/* check driver loaded */
			if (system ("grep -q '^martian_dev' /proc/modules 2> /dev/null") != 0) {
				LOGINFO ("kernel part is not loaded i.e.\n");
				LOGINFO ("\tmodprobe martian_dev\n");
			}
			else {
				LOGINFO ("Mars DSP164x device is not detected\n");
			}
		}

		return 1;
	}
	mfd = dev_fd;
	
	// get parameters
	struct martian_conf mconf;
	
	int result = ioctl (dev_fd, MARTIAN_CONF_GET, &mconf);
	if (result != 0) {
		LOGERR ("Failed to get device information.. quitting\n");
		return 1;
	}

	smp_setup (mconf.irq, dev_fd);

	if (Debugged (Note)) {
		result = ioctl (dev_fd, MARTIAN_KDEBUG);
		if (result != 0)
			LOGWARN ("Failed to enable debug in kernel module\n");
	}

	/* mmap common block */
	struct martian_common *common;
	common = (void *) mmap (
		NULL, 
		page_align(sizeof *common), 
		PROT_WRITE | PROT_READ,
		MAP_SHARED,
		dev_fd, 
		0x0
		);

	if (common == MAP_FAILED) {
		LOGSYSERR (Error, "mmap");
		LOGERR ("Failed to map device common data block\n");
		/* flee */
		return 2;
	}

	/*** check signature ***/
	result = memcmp (
		common->signature, 
		MARTIAN_SIGNATURE, 
		sizeof MARTIAN_SIGNATURE - 1
		// MIN (sizeof MARTIAN_SIGNATURE, sizeof (struct martian_common))
		);

	if (result != 0) {
		LOGERR ("Common area unrecognized\n");
		return 3;
	}
	
	result = memcmp (
		common->tsignature, 
		MARTIAN_TSIGNATURE, 
		sizeof MARTIAN_TSIGNATURE - 1
		);

	if (result != 0) {
		LOGERR ("Common area unrecognized, no terminating signature\n");
		return 3;
	}

	LOGDEBUG (Note, "Kernel module stamp `%s'\n", common->kstamp);
	
	/*** find the slot ***/
	//common += (mconf.index + 1);
	
	
	result = link_to_driver ("/proc/self/exe", common); /* not argv[0] */
	if (! result) {
		LOGERR ("Failed to link to driver\n");
		return 3;
	}
	Mcb = common;

	/* core statistics in coresubst */
	memset (&core, 0, sizeof core);

	/*** pass to the core and initialize it ***/
	init_martian_core (&mconf);

#ifdef MARTIAN_CORE_DEBUG
	print_core_vars_common (common);
	print_core_vars();
#endif

	// print_irq_state (dev_fd);


	Mcb->write_check = 'm';

	LOGDEBUG (Note, "Notifying kernel READY\n");
	if (ioctl (dev_fd, MARTIAN_MODEM_READY) != 0) {
		LOGERR("common area failure\n");
		return 4;
	}

	/*** start isr thread ***/
	/*launch_isr_thread (dev_fd);*/
	if (mthread_create (serve_interrupts, (void *) dev_fd) == -1) {
		LOGERR ("Cannot create isr thread\n");
	}
	else 
		LOGDEBUG (Note, "isr thread launched\n");

	port->do_pulse = do_pulse_core;

	return PortOk;	
}

void mport_tinit (mport_t *port)
{
	port->tid = mgettid();
	pin_thread();

/*
	if (config.realtime)
		realtime_setup();*/

	/* setup timer and handler */
	setup_virtual_irq (port);

}

void mport_dump_summary (mport_t *port) {
	/* interrupts */
	print_irq_state (mfd);
	if (Mcb->RepeatedIrqs) 
		LOGQ (Debug, "Irq: %d repeated\n", Mcb->RepeatedIrqs);
	if (Mcb->OverlappedIrqs) 
		LOGQ (Debug, "Irq: %d overlapped\n", Mcb->OverlappedIrqs);
}

void mport_mctrl_set (mport_t *port, unsigned mctrl) {
	mport_mctrl_flipflop (port, mctrl ^ port->mctrl);
}

void mport_mctrl_flipflop (mport_t *port, unsigned mctrl) {
	port->mctrl ^= mctrl;

	mport_lock (port);
	if (mctrl & TIOCM_DTR) {
		if (port->mctrl & TIOCM_DTR) {
			UART_dtr_on();
			LOGDEBUG (Event, "dtr on\n");
		}
		else {
			
			UART_dtr_off();
			LOGDEBUG (Event, "dtr off\n");
		}
	}
	if (mctrl & TIOCM_RTS) {
		if (port->mctrl & TIOCM_RTS) {
			UART_rts_on();
			LOGDEBUG (Event, "rts on\n");
		}
		else {
			UART_rts_off();
			LOGDEBUG (Event, "rts off\n");
		}
	}
	if (mctrl & TIOCM_CTS) {
		if (port->mctrl & TIOCM_CTS) 
			UART_cts_on();
		else
			UART_cts_off();
	}
	mport_unlock (port);
}

int mport_write (mport_t *port, char *buff, int count) {
	int written = V16550_Write_THR_buffer (buff, count);
	port->ready = mport_ready();
	return written;
}

int mport_read (mport_t *port, char *buf, int bytes)
{
	V16550_Read_RBR_buffer (buf, bytes); 
	port->available -= bytes;
	ASSERT (port->available == UART_CommReadCount(), );
	port->has_data = mport_has_data (port);

	return bytes;
}

/*** virtual interrupt ***/
void mport_vi (mport_t *port) {
	port->available	= UART_CommReadCount();
	port->has_data 	= mport_has_data (port);
	port->ready 	= mport_ready();

	/* check carrier */
	if ((io_uart_msr & MSR_DDCD) && !(io_uart_msr & MSR_DCD)) {
		port->carrier = CLost;
		io_uart_msr &= ~MSR_DDCD;
	}
}

void mport_open (mport_t *port) {
	port->carrier = COk;
	port->mctrl = 0;

	port->ready 	= 0;
	port->has_data 	= 0;

	port->room = 64;
	port->available = 0;

	port->pulsesAll = 0;

	ASSERT (! port->pulse, port->pulse = 0);
	
	vxdPortOpen();

	UART_dtr_on();
	UART_rts_on();
	port->mctrl = TIOCM_DTR | TIOCM_RTS;
}

void mport_close (mport_t *port) {
	/* clearing core fifos should be added */
	/* flush */
	port->carrier = COk;
	port->mctrl = 0;
	vxdPortClose();
	port->pulse = 0;

	/*UART_dtr_on();
	UART_rts_on();*/
}

