
/*
  Title: GP2X minimal library v0.C by rlyeh, 2005-2008.

  GP2X minimal library by rlyeh, 2005-2008. emulnation.info@rlyeh (swap it!):

  GP2X minimal library is a general purpose C multimedia library done for GP2X handheld, 
  featuring efficent code and a very exhaustive API covering almost all known hardware.
 
  I wrote the library originally in 2005 as a fun exercise trying to support video (8,15 & 16), 
  joystick, sound and timers in less than 128 lines of code (hence the original 'minimal' name :-).
  
  I certainly had a fun at coding whe whole library in less than 128 lines, but I knew 
  I had to switch to a bigger library at some point in order to provide a full GP2X support 
  to the recent community of upcoming GP2X developers.

  Nowadays the library is around 7,000 lines of code, including snippets from other developers. 
  I'm lucky that I have not to change the library name to 'maximal' at this point though, because 
  it stills being 'minimal' in some aspects (setup is only minimal.c and minimal.h files! :-).
  
  Enjoy the coding!
  
  
  Features:
  =========
  
  + GP2X 920t/940t CPUs library with a FIFO message system.
  + GP2X 32 bits YUV video library using double buffering.
  + GP2X 15/16 bits RGB video library using triple buffering.
  + GP2X 8 bits RGB video library using sextuple buffering.
  + GP2X sound library using octuple buffering.
  + GP2X TV out library using CX24785 video encoder chipset directly.
  + GP2X MMU library by providing own /dev/minimal kernel module.
  + GP2X joystick library with USB support.
  + GP2X profiler library.
  + GP2X blitter library.
  + GP2X image library.
  + GP2X timer library.
  + GP2X font library.


  Setting up:
  ===========
  
  Just add 'minimal.c' and 'minimal.h' files to your project directory. A piece of cake.

  
  License:
  ========

    Copyright (c) 2005-2008, rlyeh.

    Permission is hereby granted, free of charge, to any person
    obtaining a copy of this software and associated documentation
    files (the "Software"), to deal in the Software without
    restriction, including without limitation the rights to use,
    copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following
    conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
    OTHER DEALINGS IN THE SOFTWARE.

  
  Credits:
  ========
  
  I would like to thank Squidge, RobBrown, snaff, Reesy, NK, foft, K-teto, sasq, whizzbang,
  paeryn, god_at_hell, craigix, Emu&Co, Puck2099 and GnoStiC for the help & previous work! :-)
  
  I would like to thank Anarchy, Craigix and EvilDragon for providing me free hardware
  to develop nicely this library, my Reminiscence port and all the emulators I am coding. Thanks guys! :-)
  
  I would like to "thank" GamePark Holdings for constructing such a "quality" device and 
  for ignoring developers and their requests actively.
  
  
  Contact:
  ========
  
  You can contact me at emulnation.info @ rlyeh (swap it!) or rlyeh at #gp2xdev @ EFnet (IRC :-)
  
  
  Known projects using the library or parts from it:
  ==================================================

  + AlexKidd2X (Puck2099)
  + Atari800 GP2X (foft)
  + Criticalbeep 0.1.0 (Critical)
  + D3DGP2XCross (FredrikH)
  + DrMDx (Reesy)
  + DrPocketSnes (Reesy)
  + Exult GP2X (Puck2099)
  + Fenix GP2X (Puck2099)
  + File Frontend (kounch)
  + Fuse GP2X (ben_ost)
  + GNUBoy2x (K-teto)
  + GP2Xengine (Hermes/PS2Reality, Metalbrain)
  + GPFCE (zzhu8192, notaz)
  + GP LadyKiller (Puck2099)
  + Handy2X (Tharos)
  + Lotty's Lines (Baggers)
  + MAME GP2X (Franxis)
  + NeoPOP-GP2x (madelman)
  + Noiz2sa (notaz)
  + Openjazz GP2X (Puck2099)
  + Payback (jrd)
  + Phamicom-GP2X (madelman)
  + Picodrive GP2X (Notaz)
  + REminiscence (rlyeh)
  + Rise of the Triad (Puck2099)
  + SquidgeSNES (Squidge, Reesy, Notaz)
  + Samurai (Puck2099)
  + UAE4ALL GP2X (Critical)
  + VICE2X (Snaff)
  + WS2X (^MiSaTo^, Puck2099)
  + XUMP (kedo)


  What's delayed for final release:
  =================================

  - OSD layer support
  - fine scaling (post scaling) support (holy grail? YUV only?)
  - support for DaveC's pad configurations (model 0,I,II,III,IV)
  - support for scaling and rotating rects
  - touchpad support
  - usb mouse support
  - usb keyboard support
  - custom tv modes
  - custom framebuffer size
  - single buffering support (una-i's request)
  - eeprom reading support


  What's missing for this milestone:
  ==================================

  - open2x and minimal.o bug.
  - fix a bug in RGB windows (joanvr's bug).
  - test gp2x_file handling (& file selector) & write proper api.
  - do zip/lzw loader.
  - test PSD,PNG image handling.
  - test hardware blitting & write proper api.
  - test yuv scaling fix.
  - test tv video output (ntsc & pal) & test all layers.
  - test I2C support.
  - test uninstallhack.
  - do WAV/RAW loader.
  - do sw mixer.
  
  - fix touchscreen input values.
  
  - tested tv rgb scaling fix.
  - added gp2x_timer_reset.
  - gp2x_counter_* write proper doc.


  What's new:
  ===========

  (start code)

  0.C: added full 64mb support finally! :)
       added optimized RAM timings (done with UPLL prescaler; thanks god_at_hell, Emu&Co and craigix)
       added built-in MMU hack (thanks Squidge & NK)
       added built-in high resolution profiler (thanks GnoStiC for suggestions :-)
       added 16 high resolution software timers
       added overclocking support (thanks RobBrown, sasq & whizzbang)
       added preliminary gp2x hardware blitter support (thanks Paeryn, Reesy & Squidge)
       added /dev/minimal module driver (thanks oyzzo, GnoStiC and Gary Miller for compiling my sources)
       added hardcore homebrew (HH) programs launching support (thanks RobBrown!)
       added TV out support by using CX24785 encoder directly (thanks THB & vimacs for testing)
       added USB joysticks support (thanks GnoStiC for testing)
       added I2C (CX24785) support
       added BMP file image handling
       added TGA file image handling
       added flushing cache support
       added rebooting support
       added GP2X-F200 touch screen support (thanks ********************************)
       added GP2X-F100 leds support (thanks K-teto & RobBrown)
       added GP2X-F100 battery check support (thanks Squidge)
       added LCD backlight support (thanks RobBrown)
       added stereo/mono sound support
       added 3D DAC boost support
       added 6db pre-DAC attenuation support
       added bass/treble intensity support
       added headphones detection support
       added memory card detection support
       added levels based debug macro (produces compilable code only when levels match)
       added API exceptions in touch screen functions for GP2X-F100 units
       added API exceptions in battery checkout, battery LED, LCD backlight, custom sound rates, and volume level for GP2X-F200 units
       added UP+DOWN+LEFT+RIGHT as PUSH in GP2X-F200 models.
       fixed white LCD bug when using TV out (thanks THB for the report)
       fixed hsync/vsync polarity for firmware 2.0.0 (thanks K-teto)
       fixed YUV aspect ratio (thanks Puck2099)
       fixed dualcore support again (broken in 0.B)
       fixed two stupid typos in sound.c tutorial (thanks Nawa for the report)
       fixed intensive cpu usage bug when not playing sound
       fixed gp2x_video_setdithering function
       changed video buffering for 16bits mode (triple now)
       changed video buffering for 8bits mode (sextuple now!)
       changed YUV/RGB/dualcore/cursor memory mappings
       changed whole memory mapping to a better layout featuring a free, big and contiguous memory block sized up to 31 Mb.
       changed dualcore initialization method
       changed dualcore memory mapping method
       changed dualcore shared area usage through gp2x_dualcore_data() macro (easier to use now)
       changed dualcore stack area to a bigger and safer one (using 64kb now)
       changed dualcore 940t icache/dcache settings (should be faster now)
       changed dualcore 940t binary limit from 10Mb to 31Mb
       changed RGB region initialization method
       changed joystick read handling. You can check also for absolute values now.
       changed license for commercial projects
       changed volume level when using GP2X-F200 model
       restored clock setting when initializing your program (200 MHz default)
       restored leds when initializing your program
       restored low battery led when initializing your program

       ... 
       and I'm pretty sure I've forgotten a few things. I'm sorry for that.

       added functions:
       - gp2x_blitter_enable()
       - gp2x_blitter_send()
       - gp2x_blitter_wait()
       - gp2x_dualcore_launch_HH()
       - gp2x_dualcore_clock()
       - gp2x_i2c_read()
       - gp2x_i2c_write()
       - gp2x_image_bmp() 
       - gp2x_image_tga() 
       - gp2x_image_free()
       - gp2x_joystick_name()
       - gp2x_joystick_scan()
       - gp2x_joystick_wait()
       - gp2x_misc_flushcache()
       - gp2x_misc_led()
       - gp2x_misc_lcd()
       - gp2x_misc_battery()
       - gp2x_misc_headphones()
       - gp2x_misc_memcard()
       - gp2x_profile_analyze()
       - gp2x_reboot()
       - gp2x_sound_stereo()
       - gp2x_sound_3Dboost()
       - gp2x_sound_attenuation()
       - gp2x_sound_setintensity()
       - gp2x_timer_read_raw()
       - gp2x_timer_raw_to_ticks()
       - gp2x_timer_raw_one_second()
       - gp2x_timer_init()
       - gp2x_timer_start()
       - gp2x_timer_read()
       - gp2x_timer_stop()
       - gp2x_tv_setmode()
       - gp2x_tv_adjust()

       added macros:
       - gp2x_debug
       - gp2x_profile

       modified variables:
       - *gp2x_dualcore_ram

       changed functions:
       - gp2x_timer_read() is now gp2x_timer_read_ticks()
       
       modified functions:
       - gp2x_joystick_read()
       - gp2x_video_setdithering()

       modified structs:
       - deleted { *screen8, *screen16, *screen32 } pointers in gp2x_video_YUV[0,1,2,3].
       - deleted { *screen8, *screen16, *screen32 } pointers in gp2x_video_RGB[0].
       - added { void *screen } pointer into gp2x_video_YUV[0,1,2,3]
       - added { void *screen } pointer into gp2x_video_RGB[0]
       - added gp2x_rect.type enum  { RECT_RGB8=1, RECT_RGB16=2, RECT_YUV=4 };

       deprecated headers:
       - removed minimal_940t.h. Use '#define MINIMAL_940T' before '#include minimal.h' and you're ready now.

       documentation:
       - added dualcore introduction
       - added new documentation
       - improved old documentation

       examples:
       - added yuv example
       - added reboot example

  0.B: preliminary documentation. Created with NaturalDocs (thanks synkro for the tip!).
       preliminary Cursor layer support.
       preliminary RGB windows support.
       preliminary RGB alpha blending support.
       preliminary RGB key colouring support.
       preliminary RGB coarse scaling support.
       preliminary YUV layer support.
       preliminary YUV regions support.
       preliminary YUV splitting screen support.
       preliminary YUV mirroring support.
       preliminary YUV coarse scaling support.
       preliminary Gamma correction support.
       preliminary dithering support (untested).
       preliminary font support. Using public domain 6x10 console font.
       using octuple buffering for sound now.
       new 16 bits RGB video mode.

       improved functions:
        - gp2x_dualcore_exec()  ; it queues up to 1000 commands (dualcore should work better now)
  
       fixed functions:
        - gp2x_blitter_rect()   ; fixed stupid typo (thanks slygamer for the bug report)

       added structs:
        - gp2x_video_YUV[0,1,2,3] which contain {*screen8,*screen16,*screen32} pointers
        - gp2x_video_RGB[0] which contains {*screen8,*screen16,*screen32} pointers

       added functions:
        - gp2x_video_YUV_color()
        - gp2x_video_YUV_flip()
        - gp2x_video_YUV_setregions()
        - gp2x_video_YUV_setscaling()
        - gp2x_video_RGB_color16()
        - gp2x_video_RGB_setwindows()
        - gp2x_video_RGB_setscaling()
        - gp2x_video_RGB_setcolorkey()
        - gp2x_video_setluminance()
        - gp2x_video_setgammacorrection()
        - gp2x_video_waitvsync()
        - gp2x_video_waithsync()
        - gp2x_printf_init()
        - gp2x_printf()

       deprecated variables:
        - *gp2x_screen15
        - *gp2x_screen8

       changed functions:
        - gp2x_video_flip()           to  gp2x_video_RGB_flip()
        - gp2x_video_setpalette()     to  gp2x_video_RGB_setpalette()
        - gp2x_video_color15(R,G,B,A) to  gp2x_video_RGB_color15(R,G,B,A)
        - gp2x_video_color8(C,R,G,B)  to  gp2x_video_RGB_color8(R,G,B,C)  ; notice new parameters order 
        - gp2x_init()                 to  gp2x_init()                     ; got an extra parameter now
  
  0.A: 940t disabled all time unless required
       sound is paused by default now, use gp2x_sound_pause() to unlock it
  
       fixed functions:
        - gp2x_sound_play()                        ; increased the number of sampling buffers
        - gp2x_timer_read()                        ; it should work again. broken at some point before
        - gp2x_dualcore_launch_program_from_disk() ; it missed the code uploading call
  
       added functions:
        - gp2x_sound_pause()
        - gp2x_timer_delay()
        - gp2x_dualcore_pause()
  
  0.9: initial FIFO message system for dual cpu cores.
       initial 48 Mb support.
       initial quadruple buffering in 8bbp mode.              
  
       added functions:
        - gp2x_dualcore_exec() ; initial FIFO message system for dual cpu cores. 
        - gp2x_dualcore_sync() ; initial FIFO message system for dual cpu cores.
  
       improved functions:
        - gp2x_video_flip()    ; initial quadruple buffering in 8bbp mode.              
  
  0.8: initial dual cores cpu support.
       very basic blit functions by popular demand ;-)
       vsync & hsync waiting code (thanks Reesy)
  
       added functions:
        - gp2x_launch_program()            ; initial dual cores cpu support.
        - gp2x_launch_program_from_disk()  ; initial dual cores cpu support.
        - gp2x_launch_subprogram()         ; initial dual cores cpu support.
        - gp2x_blitter_rect15()            ; very basic blit function by popular demand ;-)
        - gp2x_blitter_rect8()             ; very basic blit function by popular demand ;-)
        - gp2x_video_hsync()               ; hsync waiting code (thanks Reesy)
        - gp2x_video_vsync()               ; vsync waiting code (thanks Reesy)
  
       fixed functions:
        - gp2x_video_color8()              ; bugfixed a stupid typo (thanks Franxis for the bug report)
  
  0.7: added functions:
        - gp2x_sound_volume()
  
       fixed functions:
        - gp2x_deinit()           ; fixed menu launch code when exiting.
  
       improved functions:
        - gp2x_timer_read()       ; rewritten timer. it should be more accurate now.
        - gp2x_init()             ; faster init code.
  
  0.6: added functions:
        - gp2x_timer_read()
        - gp2x_sound_pause()
  
       fixed functions:
        - gp2x_video_setpalette() ; palette handling was incorrect. fixed.
  
  0.5: improved functions:
        - gp2x_init()             ; better and cleaner initalizing code.
        - gp2x_init()             ; sound patched for real stereo output (by using NK's solution)
  
  0.4: lots of cleanups.
       sound is threaded and double buffered now.
       8 bpp video support.
  
       fixed functions:
        - gp2x_deinit()           ; better and cleaner exiting code.
  
  0.3: shorter library.
  
       fixed functions:
        - gp2x_joystick_read()    ; improved joystick diagonal detection.
  
  0.2: better code layout.
       public release.
  
  0.1: beta release.
*/

#include "minimal.h"

struct termios initial_settings, new_settings;
unsigned char keybuffer[64];

/* Typedef: gp2x_rect
   This type definition is used to blit bitmaps and/or sprites into screen.
   It contains the following public fields { x,y,w,h,solid,*data; }

   Note:
   - Set and use the fields manually.
   - <gp2x_blitter_rect> expects the gp2x_rect to be filled ok.

   Fields:
   x - Horizontal coordinate in pixels.
   y - Vertical coordinate in pixels.
   w - Width size in pixels.
   h - Height size in pixels.
   type - RECT_RGB8, RECT_RGB16, RECT_YUV.
   solid - 0 for transparent sprites (black is transparent). 1 for solid rects.
   data - pointer to your sprite/rect data.

   See also:
   <gp2x_blitter_rect> */


/* Typedef: gp2x_font
   This type definition is used to create a custom font.
   It contains the following public fields { x,y,w,h,fg,bg,solid,*data; }

   Note:
   - Use <gp2x_printf_init> to set the variables into a gp2x_font struct. Do not set them manually.

   Fields:
   w - Width of each character, in pixels.
   h - Height of each character, in pixels.
   fg - Foreground font color.
   bg - Background font color (when solid is not 0).
   solid - When 1, background font color is used.
           When 0, transparent background font is used.
   data - Pointer to your font bitmap.

   See also:
   <gp2x_printf>, <gp2x_printf_init> */


/* Enumeration: GP2X joystick values
   This enumeration contains all the possible values which <gp2x_joystick_read> may return.

   Constants:
   GP2X_UP        - joystick up
   GP2X_DOWN      - joystick down
   GP2X_LEFT      - joystick left
   GP2X_RIGHT     - joystick right
   GP2X_PUSH      - joystick push
   GP2X_L         - button L
   GP2X_R         - button R
   GP2X_A         - button A
   GP2X_B         - button B
   GP2X_X         - button X
   GP2X_Y         - button Y
   GP2X_START     - button START
   GP2X_SELECT    - button SELECT
   GP2X_VOL_DOWN  - button volume down (-)
   GP2X_VOL_UP    - button volume up (+)

   See also:
   <gp2x_joystick_read> */


/* Macro: gp2x_debug(level, function)
   This macro calls to given function *only* if given level && <GP2X_DEBUG_LEVEL> expression is true.

   Note:
   - <GP2X_DEBUG_LEVEL> is 0 (off) by default
   - level has to be a pow of 2 constant

   Parameters:
   level - pow of 2 constant which defines a debug level
   function - given function to call if GP2X_DEBUG_LEVEL ANDS* logically with given level argument

   Example:
   > #define DEBUG_SOUND      1
   > #define DEBUG_GRAPHICS   2
   > #define DEBUG_3D         4
   >
   > #define GP2X_DEBUG_LEVEL (DEBUG_SOUND|DEBUG_3D)
   >
   > #include "minimal.h"
   >
   > void main() {
   > gp2x_debug(DEBUG_SOUND,    printf("im     printing this line because 5 & 1 is true");
   > gp2x_debug(DEBUG_GRAPHICS, printf("im not printing this line because 5 & 2 is false");
   > gp2x_debug(DEBUG_3D,       printf("im     printing this line because 5 & 4 is true");
   > }

   Credits:
   rlyeh  */


/* Macro: gp2x_profile(function)
   This function calls to given function and profiles its execution time by default.
   Parsed results can be obtained by calling <gp2x_profile_analyze> later.

   You can turn profiling off by defining GP2X_PROFILE_DISABLE. In this specific
   case function call is done but no profiling code is generated.

   Note:
   - For optimal results program should run from a few seconds to up 7 minutes.
   - To get profiling results you will have to call <gp2x_profile_analyze> later.

   Example:
   > while(condition) {
   > gp2x_profile( draw() );
   > gp2x_profile( input() );
   > gp2x_profile( network() );
   } }
   >
   > gp2x_printf(NULL, 0,0, "%s", gp2x_profile_analyze()); 

   See also:
   <gp2x_profile_analyze>

   Credits:
   rlyeh  */

   
/* Macro: gp2x_dualcore_data(address)

   This macro retrieves or sets a value into a shared address; it works for both 920T and 940T cores.

   Memory map for our shared area for the current library:

    range 0x00000000-0x00000FFF (4  kb) is reserved                             

    range 0x00001000-0x00007FFF (8  kb) is free, and non-cacheable

    range 0x00008000-0x00F00000 (15 Mb) is free, and cacheable

   More info about it in tutorials\12.dualcore\using-subprograms sample.

   See also:
   <gp2x_dualcore_pause>, <gp2x_dualcore_sync>, <gp2x_dualcore_exec> */

   
/* Macro: gp2x_dualcore_declare_subprogram(name)
   This macro declares a 940t subprogram in 920t source, in order to launch it with <gp2x_dualcore_launch_subprogram>.
   
   See also:
   <gp2x_dualcore_declare_subprogram>
   
   Credits:
   rlyeh */

   
/* Macro: gp2x_dualcore_launch_subprogram(name)
   This macro launches a 940t subprogram from a 920t source.
   Read more about it in tutorials\12.dualcore\using-subprograms sample.
   
   Note:
   - You have to declare the subprogram prior to launch it. See 
   
   See also:
   <gp2x_dualcore_declare_subprogram>
   
   Credits:
   rlyeh */

   
/* Variable: gp2x_dualcore_ram
   This private array holds the shared RAM for both 920T and 940T processors.
   If you want to access this RAM consider using <gp2x_dualcore_data> macro. */

   
/* Variable: gp2x_frames
   This variable holds the current rendered frames you've done so far. */

   
/* Variable: gp2x_usbjoys
   This variable holds the maximum number of usb joysticks found in last scan.

   Note:
   - Do not modify this variable directly. Use <gp2x_joystick_scan> instead.

   See also:
   <gp2x_joystick_scan> */

   
/* Variable: gp2x_ticks
   This internal variable holds the current conversion between desired ticks and GP2X ticks.

   Note:
   - Do not modify this variable directly. Use <gp2x_init> instead.

   See also:
   <gp2x_timer_read>, <gp2x_timer_delay> */


/* Variable: gp2x_video_YUV
   This struct allows you to work directly with YUV video layer.
   It contains three pointers which are updated in each <gp2x_video_YUV_flip> call.
   There are four YUV regions, each one with its own framebuffer. Each framebuffer orientation is top-left (common orientation).

   Contains:
   screen - void pointer to the hidden video surface. 

   Note:
   - Use <gp2x_video_YUV_flip> to flip video surfaces and to update the screen pointer.
   - Use <gp2x_video_YUV_color> to calculate colors when you are working in Y/Cb/Cr video mode.

   Example:
   > long *ptr = (long *) gp2x_video_YUV[1].screen;
   > ptr[160+120*320]=gp2x_video_YUV_color(255,255,255);  //write white at (160,120)
   > gp2x_video_YUV_flip(1); */

   gp2x_video_layer gp2x_video_YUV[4];

   
/* Variable: gp2x_video_RGB
   This struct allows you to work directly with RGB video layer.
   It contains three pointers which are updated in each <gp2x_video_RGB_flip> call.
   There is a single RGB region, with a single framebuffer. The framebuffer orientation is top-left (common orientation).

   Contains:
   screen - void pointer to the hidden video surface.

   Note:
   - Use <gp2x_video_RGB_flip> to flip video surfaces and to update the screen pointer.
   - Use <gp2x_video_RGB_color15> to calculate colors when you are working in hicolor video mode.
   - Use <gp2x_video_RGB_color8> to set colors when working in palettized video mode.
   - Use <gp2x_video_RGB_setpalette> to flush any change in the palette.

   Example:
   > short *ptr = (short *)gp2x_video_RGB[0].screen;
   > ptr[160+120*320]=gp2x_video_RGB_color15(255,255,255, 0); //write white at (160,120)
   > gp2x_video_RGB_flip(0);
   >
   > or...
   >
   > unsigned char *ptr8 = (unsigned char *)gp2x_video_RGB[0].screen;
   > gp2x_video_RGB_color8(255,255,255, 1);
   > gp2x_video_RGB_setpalette();
   > ptr8[160+120*320]=1;                                       //write white at (160,120)
   > gp2x_video_RGB_flip(0); */

   gp2x_video_layer gp2x_video_RGB[1];

static          pthread_t       gp2x_sound_thread, gp2x_video_logo_thread;
static          unsigned long   gp2x_dev[8]={0,0,0,0,0,0,0,0}, gp2x_ticks_per_second;
static          unsigned short *gp2x_sound_buffer;
static volatile unsigned short  gp2x_video_RGB_palette[512], gp2x_video_OSD_palette[512];
       volatile unsigned short *gp2x_memregs;
       volatile unsigned long  *gp2x_memregl, *gp2x_dualcore_ram, *gp2x_blitter;
static volatile unsigned long   gp2x_sound_pausei, gp2x_exit, gp2x_ticks;
                gp2x_font       gp2x_default_font;
static          int             gp2x_usbjoy[4];
                unsigned long   gp2x_usbjoys, gp2x_frames, gp2x_f200;
static          unsigned long long int gp2x_timer[16], gp2x_timer_counter[16];
static          unsigned long   gp2x_timer_active[16], gp2x_timer_overflow[16];
static          unsigned long   gp2x_profiles_calibration;
static          gp2x_profiles  *gp2x_profiles_area = NULL;
static          unsigned long   gp2x_profiles_done = 0, gp2x_profiles_roomspace = 0;
static          char           *gp2x_profiles_string = NULL;

static          struct        { unsigned short x,y,pressed;                                     } gp2x_input;

static          int             gp2x_touchscreen_calibration[7] = {6203, 0, -1501397, 0, -4200, 16132680, 65536};

extern  int fcloseall(void);
static void gp2x_deinit_(void);







void gp2x_touchscreen_read(int *rx, int *ry) 
{
    static int old_press = 0, okx = 0, oky = 0;
    int x, y, press, up, down;
    static struct { unsigned short pressure, x, y, padding; struct timeval stamp; } gp2x_touchscreen;

	if (!gp2x_f200) 
        return;

	read(gp2x_dev[7], &gp2x_touchscreen, sizeof(gp2x_touchscreen));

    x = (gp2x_touchscreen_calibration[2] + gp2x_touchscreen_calibration[0] * gp2x_touchscreen.x) / gp2x_touchscreen_calibration[6];
    y = (gp2x_touchscreen_calibration[5] + gp2x_touchscreen_calibration[4] * gp2x_touchscreen.y) / gp2x_touchscreen_calibration[6];
    if(x < 0 ) x = 0; else if (x > 319) x = 319;
    if(y < 0 ) y = 0; else if (y > 239) y = 239;
    x &= 0xffe;
    y &= 0xffe;
    
    press = (gp2x_touchscreen.pressure > 0);
   
    down = ((!old_press) && ( press)); 
    up   = (( old_press) && (!press));

    old_press = press;

    press = (press && (!down) && (!up));
    
    
    static int recordx = 0, recordy = 0, lastx = 0, lasty = 0, diffx = 0, diffy = 0, goodx =0, goody = 0, lastdiffx = 0, lastdiffy = 0;
    
    #define abs(a)   ( (a) < 0 ? -(a) : (a) )
    #define max(a,b) ( abs(a) > abs(b) ? (a) : (b) )
    
    /* 
    
    From my tests:
    - down,press,up events are fine.
    - x,y coords are always fine when stylus touches the screen the first time.
    - x,y coords may be wrong when stylus is pressing the screen continuosly 
    
    (case 1)
    if last_diff > 4 && diff <= 4
        last is invalid 
        return current

    (case 2)
    if last_diff > 4 && diff > 4
        last is valid
        return last
   
    (case 0)
    return last
    
    case 0    
    d1 p0 u0 135 -127 diff( 0, 0) ok
    d0 p1 u0 134 -127 diff(-1, 0) ok <-- valid
    d0 p1 u0 134 -127 diff(-1, 0) ok <-- im here, return last
        
    case 1
    d1 p0 u0 135 -127 diff( 0, 0) ok
    d0 p1 u0 134 -127 diff(-1, 0) ok
    d0 p1 u0 129 -127 diff(-5, 1) ** <-- invalid
    d0 p1 u0 134 -127 diff(-1, 0) ok <-- im here. return last = current 

    case 2
    d1 p0 u0 135 -127 diff( 0, 0) ok
    d0 p1 u0 134 -127 diff(-1, 0) ok
    d0 p1 u0 129 -127 diff(-5, 1) ok --> new goodx, new goody
    d0 p1 u0 129 -127 diff(-5, 1) ok <-- im here. return last
    
    */
    
    if(down)
    goodx = x, goody = y, lastx = x, lasty = y, lastdiffx = 0, lastdiffy = 0, diffx = diffy = 0/*, recordx = recordy = 0*/;

    lastdiffx = diffx;
    lastdiffy = diffy;
    
    diffx = x-lastx;
    diffy = y-lasty;
    
    #define SENSIBILITY 2
 
    //apply good values when invalid
    int invalid1 = (abs(diffx) <= SENSIBILITY && abs(lastdiffx) > SENSIBILITY) || (abs(diffy) <= SENSIBILITY && abs(lastdiffy) > SENSIBILITY);
    int invalid2 = (abs(diffx)  > SENSIBILITY && abs(lastdiffx) > SENSIBILITY) || (abs(diffy)  > SENSIBILITY && abs(lastdiffy) > SENSIBILITY);
    int Case ;
    
    if(invalid1)
    {
        lastx = x, lasty = y;
        okx = lastx, oky = lasty;
        
        Case = 1;
    }
    else
    {
        if(invalid2)
        {
            okx = lastx, oky = lasty;
            goodx = x, goody = y, lastx = x, lasty = y, lastdiffx = lastdiffy = diffx = diffy = 0;
            
            Case = 2;
        }
        else //case 0
        {        
            lastx = ( lastx + x ) >> 1;
            lasty = ( lasty + y ) >> 1;
            
            okx = lastx, oky = lasty;

            Case = 0;
        }
    }
    
    //recordx = max(recordx,diffx), recordy = max(recordy,diffy) ;

    *rx = okx;
    *ry = oky;
 
    gp2x_printf(NULL, 0, 60, "d%d p%d u%d %3d %3d diff(%3d, %3d) %d ****\n", 
                                down,press,up,okx,oky,
                                diffx, diffy,
                                Case);
                                
    /*                          
    static FILE *out = NULL;
    if(!out)
        out = fopen("log.txt","wb");
        
    fprintf(out, "d%d p%d u%d %3d %3d diff(%3d, %3d) (%d)\n", 
                                down,press,up,okx,oky,
                                diffx, diffy,
                                Case);
    */
}

    
    
    



/*
   ,--------------------.
   |                    |X
   |    GP2X - VIDEO    |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/


/* Function: gp2x_video_waitvsync
   This function halts the program until a vertical sync is done.

   See also:
   <gp2x_video_waithsync> 

   Credits:
   rlyeh (original code)
   K-teto (fixed this function for firmware 2.0.0) */

void gp2x_video_waitvsync(void)
{
    while(  gp2x_memregs[0x1182>>1]&(1<<4))
    {
        asm volatile ("nop");
        asm volatile ("nop");
        asm volatile ("nop");
        asm volatile ("nop");

        //asm volatile ("" ::: "memory");
    }

    while(!(gp2x_memregs[0x1182>>1]&(1<<4)))
    {
        asm volatile ("nop");
        asm volatile ("nop");
        asm volatile ("nop");
        asm volatile ("nop");

        //asm volatile ("" ::: "memory");
    }
}

/* Function: gp2x_video_waithsync
   This function halts the program until an horizontal sync is done.

   See also:
   <gp2x_video_waitvsync>

   Credits:
   rlyeh */

void gp2x_video_waithsync(void)
{
  while(  gp2x_memregs[0x1182>>1]&(1<<5));
  while(!(gp2x_memregs[0x1182>>1]&(1<<5))); /*needed? if not, FIXME*/
}

/* Function: gp2x_video_setgammacorrection
   This function sets the Gamma correction value for the whole display.

   Parameters:
   gamma (0.00 .. 2.00) - from 0.00 (darkest) to 2.00 (brightest).

   Default:
   - gamma is set to 1.00

   Credits:
   rlyeh (original code)
   RobBrown (fixed my original code which was wrong at first)*/

void gp2x_video_setgammacorrection(float gamma)
{
  int i;

  gp2x_memregs[0x2880>>1]&=~(1<<12);                       //enable gamma

  gp2x_memregs[0x295C>>1]=0;
  for(i=0,gamma=2.0-gamma;i<256;i++)
  {
   unsigned char g=(unsigned char)(0.5+255.0*pow(i/255.0,gamma));
   gp2x_memregs[0x295E>>1]=(g<<8)|g; gp2x_memregs[0x295E>>1]=g;
  }
}

/* Function: gp2x_video_setdithering
   This function sets the dithering values for all the video layers.

   Note:
   - This function only works when merging layers of different bits per pixel.
   
   Parameters:
   mode (0..3) - set mode to 0 for 2x2 dithering off, 1 for 4x4 dithering off, 2 for 2x2 dithering on, and 3 for 4x4 dithering on.

   Default:
   - mode is set to 0 (2x2 dithering off).

   Credits:
   rlyeh */

void gp2x_video_setdithering(int mode)
{
  gp2x_memregs[0x2946>>1]=mode;
}

/* Function: gp2x_video_setluminance
   This function sets the contrast/brigness values for the YUV/OSD layers.

   Parameters:
   contrast (0..7) - adjusts contrast where as 0=1.00, 1=1.125... 7=1.875 are contrast values.
   brightness (0..255) - adjusts brightness.

   Default:
   - contrast is set to 0
   - brightness is set to 0 */

void gp2x_video_setluminance(int contrast, int brightness)
{
  gp2x_memregs[0x2934>>1]=((contrast&0x07)<<8)|brightness;
}





/*
	
   ,--------------------.
   |                    |X
   |     GP2X - I2C     |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
	
*/


typedef struct { unsigned char id,addr,data;} i2cw;
typedef struct { unsigned char id,addr,*pdata;} i2cr;

/* Function: gp2x_i2c_write
   This function writes a byte into an I2C address.
   
   Parameters:
   id (0..) - i2c ID
   addr - address to write data to.
   data (0..255) - data to be written.

   See also:
   <gp2x_i2c_read>
   
   Credits:
   rlyeh */

void gp2x_i2c_write(unsigned char id, unsigned char addr, unsigned char data)
{
 i2cw a;
 a.id = id, a.addr = addr, a.data = data;
 ioctl(gp2x_dev[0], _IOW('v', 0x00, i2cw), &a); //CX25874_I2C_WRITE_BYTE
 //ioctl(gp2x_dev[6], MINIMAL_I2C_WRITE, &a);
}

/* Function: gp2x_i2c_read
   This function reads a byte from an I2C address.
   
   Parameters:
   id (0..) - i2c ID
   addr - address to write data to.
   
   See also:
   <gp2x_i2c_write>
   
   Credits:
   rlyeh */

unsigned char gp2x_i2c_read(unsigned char id, unsigned char addr)
{
 unsigned char temp;
 i2cr a;
 a.id = id, a.addr = addr, a.pdata = &temp ;
 ioctl(gp2x_dev[0], _IOW('v', 0x01, i2cr), &a); //CX25874_I2C_READ_BYTE
 //ioctl(gp2x_dev[6], MINIMAL_I2C_READ, &a);

 return (*a.pdata);
}

#define gp2x_cx25784_read(a)    gp2x_i2c_read(0x8A,(a))
#define gp2x_cx25784_write(a,v) gp2x_i2c_write(0x8A,(a),(v))


/*
	
   ,--------------------.
   |                    |X
   |   GP2X - TV OUT    |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
	
*/

/* WRITEME */

static int gp2x_tv_lastmode = LCD;

int gp2x_tv_getmode(void)
{
 return gp2x_tv_lastmode;
}

                   
/* Function: gp2x_tv_setmode
   This function set TV out mode on or off.
   
   Parameters:
   mode - set mode to <LCD>, <PAL> or <NTSC>. 
   addr - address to write data to.
   
   See also:
   <gp2x_tv_adjust>
   
   Credits:
   rlyeh */

void gp2x_tv_setmode(unsigned char mode)
{
 if(mode != LCD && mode != PAL && mode != NTSC) return;
 
 if(0)
 {
    int n, offset = 0;
    
    if(mode == 4 && gp2x_tv_lastmode != 4) offset = 20*320;
    if(mode != 4 && gp2x_tv_lastmode == 4) offset = -20*320;
    
    for(n = 0; n <= 6; n++)
        gp2x_video_RGB[0].p[n] = &(((short *)gp2x_video_RGB[0].p[n]))[offset];

    gp2x_video_RGB[0].screen = gp2x_video_RGB[0].p[gp2x_video_RGB[0].i[7]];
 }

 gp2x_tv_lastmode = mode;
  
 if(!gp2x_dev[0]) gp2x_dev[0] = open("/dev/cx25874",O_RDWR);

 gp2x_misc_lcd(1);
 
 if(mode == LCD)
 {
  ioctl(gp2x_dev[0], _IOW('v', 0x02, unsigned char), 0);
  close(gp2x_dev[0]);
  gp2x_dev[0] = 0;
   
  return;
 }
 
 ioctl(gp2x_dev[0], _IOW('v', 0x02, unsigned char), mode);
  
 //gp2x_video_RGB_setwindows(0x11,-1,-1,-1,319,239);              
 gp2x_video_RGB_setscaling(320,240); 
 //gp2x_video_YUV_setparts(-1,-1,-1,-1,319,239);
 gp2x_video_YUV_setscaling(0,320,240);
 gp2x_video_YUV_setscaling(1,320,240);
 gp2x_video_YUV_setscaling(2,320,240);
 gp2x_video_YUV_setscaling(3,320,240);
  
 gp2x_misc_lcd(0);

 if(mode == NTSC)
 {
    gp2x_tv_adjust(25, -7);
 }
 else
 {
    gp2x_tv_adjust(25,  20);
 }
}


/* Function: gp2x_tv_adjust
   This function adjusts and centers screen to TV.
   
   Parameters:
   horizontal (-50..0..35) - number of pixels to move the image horizontally.
   vertical (-15..0..45) - number of pixels to move the image vertically.
   
   Note:
   - horizontal and vertical are both signed.
   - default setting for NTSC is (16, -7) 
   - default setting for PAL  is (16, 19)
   
   See also:
   <gp2x_tv_setmode>
   
   Credits:
   rlyeh */

void gp2x_tv_adjust(signed char horizontal, signed char vertical)
{
    int lines, syncs_start, syncs_end;

    //horizontal adjustment
    gp2x_video_waitvsync();
    gp2x_cx25784_write(0x8c, (unsigned char) (50 - 11 + 3 - horizontal) );
    
    //vertical adjustment
    if(gp2x_tv_getmode() == PAL) 
        lines = 288, syncs_start = 1, syncs_end = 24; 
    else 
        lines = 240, syncs_start = 1, syncs_end = 22;
    
    lines -= vertical, syncs_end += vertical;
       
    gp2x_video_waitvsync();
    gp2x_memregs[0x2818 >> 1]  =  lines - 1;
    gp2x_memregs[0x2820 >> 1] &= (0xFF00);
    gp2x_memregs[0x2820 >> 1] |= (syncs_start << 8);
    gp2x_memregs[0x2822 >> 1] &= ~(0x1FF);
    gp2x_memregs[0x2822 >> 1] |=  syncs_end; // syncs_end = verBackPorch+verFontPorch - 1

    if(gp2x_tv_getmode() == PAL)
    {
     //bottom screen image cut off (PAL 320x288 full -> PAL 320x240 centered w/ black borders)
     int real_lines = 288, 
         wanted_lines = 240, 
         top_spacing = (real_lines - wanted_lines) / 2,
         active_lines = wanted_lines + top_spacing;
     
     gp2x_video_waitvsync();
    
     active_lines += -top_spacing +5 + vertical;
     
     gp2x_cx25784_write(0x84, active_lines & 0xFF); //reduce overscan, VACTIVE_0
     gp2x_cx25784_write(0x86, 0x26 | ((active_lines & 0x100)>>1) );
      
     gp2x_cx25784_write(0x94, active_lines & 0xFF); //reduce overscan, VACTIVE_1
     gp2x_cx25784_write(0x96, 0x31 | ((active_lines & 0x300)>>8) );
    }
}



/*
   ,-----------------------.
   |                       |X
   |  GP2X - VIDEO - OSD   |X
   |                       |X
   `-----------------------'X
    XXXXXXXXXXXXXXXXXXXXXXXXX 
*/

#if 0
//Prototypes following...
//This function returns a VUYA (A/Y/Cb0/Cr0) 32bits color pixel from a given R,G,B triad and alpha value.
unsigned long gp2x_video_OSD_color(int R, int G, int B, int A)
{
  int Y = ( (  66 * R + 129 * G +  25 * B + 128) >> 8) + 16,
      U = ( ( -38 * R -  74 * G + 112 * B + 128) >> 8) + 128,
      V = ( ( 112 * R -  94 * G -  18 * B + 128) >> 8) + 128;

  return ((A&0xFF)<<24) | ((Y&0xFF)<<16) | ((U&0xFF)<<8) | ((V&0xFF)<<0);
}

void gp2x_video_OSD_setpalette(void)
{
  unsigned short *g=(unsigned short *)gp2x_video_OSD_palette; int i=512;
  gp2x_memregs[0x2954>>1]=0;                                                     
  while(i--) gp2x_memregs[0x2956>>1]=*g++; 
}

void gp2x_video_OSD_flip(void)
{
  unsigned long address=0;

  asm volatile (".word 0xef9f0002"); // flush cache

  gp2x_memregs[0x2916>>1]=(unsigned short)(address & 0xFFFF);
  gp2x_memregs[0x2918>>1]=(unsigned short)(address >> 16);
  gp2x_memregs[0x291A>>1]=(unsigned short)(address & 0xFFFF);
  gp2x_memregs[0x291C>>1]=(unsigned short)(address >> 16);
}

void gp2x_video_OSD_write_header(void)
{
 unsigned long *top_header;
 unsigned long *region_header;

 global_mix = 0..1
 alpha_value = 0..15

 transparent = 0..1
 transparent color = 0..255

 top = 11 bits
 bottom = 11 bits
 x = 11 bits
 y = 11 bits

 if(global_mix)
 {
  alpha_value,transparent,transparent_color = valid;
 }
 else
 {
  alpha_value = from palette
 }

 //OSD enable
 memregs[0x2880>>1]&=~(1<<7);
 memregs[0x2880>>1]|=(1<<7);
 memregs[0x2880>>1]&=~(1<<7);
 memregs[0x2880>>1]|=(1<<7);

 *top_header=&region_header;

 //set no next region address
 *(region_header+0)=0;

 //set last region header, uncompressed data, 8bpp, 1 pel
 *(region_header+1)=(1<<30)|(0<<29)|(2<<28)|(0<<27)
 //set global/palettized region alpha value
                   |(global_mix<<26)|(alpha_value<<22)
 //set top/bottom region
                   |(top<<11)|(bottom<<0);
 
 *(region_header+2)=(transparent<<30)|(transparent_color<<22)|(x<<11)|(y<<0);
}
#endif



/*
   ,-----------------------.
   |                       |X
   | GP2X - VIDEO - CURSOR |X
   |                       |X
   `-----------------------'X
    XXXXXXXXXXXXXXXXXXXXXXXXX 
*/

static void                *gp2x_video_cursor_virt;
static unsigned long        gp2x_video_cursor_phys;
static const unsigned char  gp2x_video_cursor_default[] = {
1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
1,2,1,0,0,0,0,0,0,0,0,0,0,0,0,0,
1,2,2,1,0,0,0,0,0,0,0,0,0,0,0,0,
1,2,2,2,1,0,0,0,0,0,0,0,0,0,0,0,
1,2,2,2,2,1,0,0,0,0,0,0,0,0,0,0,
1,2,2,2,2,2,1,0,0,0,0,0,0,0,0,0,
1,2,2,2,2,2,2,1,0,0,0,0,0,0,0,0,
1,2,2,2,2,2,2,2,1,0,0,0,0,0,0,0,
1,2,2,2,2,2,2,2,2,1,0,0,0,0,0,0,
1,2,2,2,2,1,1,1,1,1,1,0,0,0,0,0,
1,2,2,2,1,0,0,0,0,0,0,0,0,0,0,0,
1,2,2,1,0,0,0,0,0,0,0,0,0,0,0,0,
1,2,1,0,0,0,0,0,0,0,0,0,0,0,0,0,
1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};


/* Function: gp2x_video_cursor_show
   This function shows or hides the cursor.

   Parameters:
   yes (0,1) - set 1 to show cursor, and 0 to hide it.

   Default:
   - yes is set to 0 (hide).

   See also:
   <gp2x_video_cursor_move>, <gp2x_video_cursor_setalpha>, <gp2x_video_cursor_setup>

   Credits:
   rlyeh */

void gp2x_video_cursor_show(int yes)
{
  gp2x_memregs[0x2880>>1] &=  ~(1<<9);
  gp2x_memregs[0x2880>>1] |= (yes<<9);
}

/* Function: gp2x_video_cursor_move
   This function updates the cursor coordinates.

   Parameters:
   x (-32767..32768) - x cursor position.
   y (-32767..32768) - y cursor position.

   Default:
   - (x,y) are set to center of screen.

   See also:
   <gp2x_video_cursor_show>, <gp2x_video_cursor_setalpha>, <gp2x_video_cursor_setup>
   
   Credits:
   rlyeh */

void gp2x_video_cursor_move(int x, int y)
{
  gp2x_memregs[0x2920>>1] = x;
  gp2x_memregs[0x2922>>1] = y; 
}

/* Function: gp2x_video_cursor_setalpha
   This function sets the alpha blending of both background and foreground cursor colors.

   Parameters:
   FG_A (0..15) - Foreground alpha blending value.
   BG_A (0..15) - Background alpha blending value.

   Default:
   - FG_A and BG_A both are set to 15 (100% solid, no transparency).

   See also:
   <gp2x_video_cursor_show>, <gp2x_video_cursor_move>, <gp2x_video_cursor_setup>
   
   Credits:
   rlyeh */

void gp2x_video_cursor_setalpha(int FG_A, int BG_A)
{
  gp2x_memregs[0x291e>>1] &= ~((0xF<<12)|(0xF<<8));
  gp2x_memregs[0x291e>>1] |=  (FG_A<<12)|(BG_A<<8);
}

/* Function: gp2x_video_cursor_setup
   This function sets the shape, size, colour and alpha blending of the cursor.

   Parameters:
   *src - pointer to a data area containing the cursor shape.
          each byte in the area corresponds to a pixel in the cursor shape.
          you can define your own foreground (FG), background (BG) and transparent (any other else) colors.
   size (8,16,32,64) - square size of the cursor shape.
   FG (0..255) - byte which identifies foreground pixels in your *src area. Usually set to 1.
   FG_R (0..255) - red component for your foreground shape color
   FG_B (0..255) - green component for your foreground shape color
   FG_G (0..255) - blue component for your foreground shape color
   FG_A (0..15) - alpha blending for foreground shape color 
   BG (0..255) - byte which identifies foreground pixels in your *src area. Usually set to 0 or 2.
   BG_R (0..255) - red component for background shape color
   BG_B (0..255) - green component for background shape color
   BG_G (0..255) - blue component for background shape color
   BG_A (0..15) - alpha blending for background shape color 

   Default:
   - The default cursor holds an arrow sprite of 16 pixels.
     Its foreground color defaults to white, and background one to black. 
     There is no transparency set for the default cursor.

   See also:
   <gp2x_video_cursor_show>, <gp2x_video_cursor_move>, <gp2x_video_cursor_setalpha>
   
   Credits:
   rlyeh */

void gp2x_video_cursor_setup(unsigned char *src, int size, unsigned char FG, int FG_R, int FG_G, int FG_B, int FG_A, unsigned char BG, int BG_R, int BG_G, int BG_B, int BG_A)
{
  unsigned char *dst=(unsigned char *)gp2x_video_cursor_virt;
  int x, c;

  //convert shape
  for(*dst=x=c=0;x<size*size; )
  {
   *dst|=(*src==FG ? 0 : (*src==BG ? 1 : 2) ) << c; c+=2;

   src++;

   if(!((++x)%4)) *(++dst)=c=0;
  }

  //set fg/bg alpha and cursor dimensions
  gp2x_memregs[0x291e>>1] =  (FG_A<<12)|(BG_A<<8) | size; 

  //set cursor position at center of screen
  gp2x_video_cursor_move((320-size)/2,(240-size)/2); 

  //set cursor colors
  gp2x_memregs[0x2924>>1] = (FG_G<<8)|FG_R;
  gp2x_memregs[0x2926>>1] = FG_B;
  gp2x_memregs[0x2928>>1] = (BG_G<<8)|BG_R;
  gp2x_memregs[0x292a>>1] = BG_B;

  //set cursor bitmap
  gp2x_memregs[0x292c>>1]=(unsigned short)(gp2x_video_cursor_phys & 0xFFFF);
  gp2x_memregs[0x292e>>1]=(unsigned short)(gp2x_video_cursor_phys >> 16);
  gp2x_memregs[0x2930>>1]=(unsigned short)(gp2x_video_cursor_phys & 0xFFFF);
  gp2x_memregs[0x2932>>1]=(unsigned short)(gp2x_video_cursor_phys >> 16);
}




/*
   ,---------------------.
   |                     |X
   | GP2X - VIDEO - LOGO |X
   |                     |X
   `---------------------'X
    XXXXXXXXXXXXXXXXXXXXXXX 
*/

static const unsigned char gp2x_video_logo_default[] = {
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,
0,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,0,0,
0,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,0,
0,1,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,1,1,2,2,2,1,0,
0,1,2,2,2,2,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,2,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,1,1,1,1,1,1,1,1,2,1,1,1,1,1,1,1,1,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,2,2,2,2,1,2,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,1,1,2,2,2,2,2,1,1,1,2,2,2,2,2,1,1,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,0,
0,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,0,
0,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,0,
0,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,0,
0,1,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,1,1,2,2,2,1,0,
0,1,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,2,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,1,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,2,1,1,2,2,2,1,0,
0,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,0,
0,0,1,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,0,
0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0 };

static void *gp2x_video_logo_anim(void *blah)
{
  int i; struct timespec ts; ts.tv_sec=3, ts.tv_nsec=25000000; //0.025 seconds delay

  nanosleep(&ts, NULL);

  ts.tv_sec=0;

  for(i=0xF;i>=0x0 && !gp2x_exit;i--) { gp2x_video_cursor_setalpha(((i*0xE)/0xF)+1,((i*0xD)/0xF)+2);
                                        nanosleep(&ts, NULL); }

  return NULL;
  //pthread_exit( NULL );
}

/* Function: gp2x_video_logo_enable
   This function displays the minimal logo watermarked at bottom right of screen.
   Adding this logo to your application makes rlyeh (me) happier and glad with you :-)

   Note:
   - Note that this function uses the cursor layer to work, so you cannot display a cursor and the minimal logo at same time.

   Parameters:
   yes (0,1) - 1 enables logo. 0 disables it.

   Default:
   - yes is set to 0. Logo is disabled by default.
   
   Credits:
   rlyeh */

void gp2x_video_logo_enable(int yes)
{
 gp2x_video_cursor_show(0);

 if(yes) { gp2x_video_cursor_setup((unsigned char *)gp2x_video_logo_default, 32, 2,32,32,255,0xF, 1,255,255,255,0xF);
           gp2x_video_cursor_move(320-32,240-32);
           gp2x_video_cursor_show(1);
           pthread_create( &gp2x_video_logo_thread, NULL, gp2x_video_logo_anim, NULL); }
}




/*
   ,--------------------.
   |                    |X
   | GP2X - VIDEO - RGB |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/



/* Function: gp2x_video_RGB_color8
   This function sets the palettized color entry C to a given triad of R,G,B components.
   All color changes will get updated at next <gp2x_video_RGB_setpalette> call.

   Note:
   - Use <gp2x_video_RGB_setpalette> to flush all palette changes.

   Parameters:
   C (0..255) - Palettized color entry to change
   R (0..255) - Red component
   G (0..255) - Green component
   B (0..255) - Blue component 

   See also:
   <gp2x_video_RGB_setpalette>
   
   Credits:
   rlyeh */ 

void gp2x_video_RGB_color8 (int R, int G, int B, int C)
{
  gp2x_video_RGB_palette[((C)<<1)+0]=((G)<<8)|(B),gp2x_video_RGB_palette[((C)<<1)+1]=(R); 
}


/* Function: gp2x_video_RGB_color15
   This function returns the 16bits color for a given triad of R,G,B components.
   It returns a short in 5515:RGAB format.

   Parameters:
   R (0..255) - Red component
   G (0..255) - Green component 
   B (0..255) - Blue component 
   A (0,1) - General purpose boolean value. Usually sets to 0.
   
   Credits:
   rlyeh */

unsigned short gp2x_video_RGB_color15(int R, int G, int B, int A)
{
  return ((((R)&0xF8)<<8)|(((G)&0xF8)<<3)|(((B)&0xF8)>>3)|((A)<<5));
}

/* Function: gp2x_video_RGB_color16
   This function returns the 16bits color for a given triad of R,G,B components.
   It returns a short in 565:RGB format.

   Parameters:
   R (0..255) - Red component
   G (0..255) - Green component 
   B (0..255) - Blue component
   
   Credits:
   rlyeh */

unsigned short gp2x_video_RGB_color16(int R, int G, int B)
{
  return ((((R)&0xF8)<<8)|(((G)&0xFC)<<3)|(((B)&0xF8)>>3));
}


/* Function: gp2x_video_RGB_setpalette
   This function updates the whole 8 bits palette into screen.
   To change each palette color use <gp2x_video_RGB_color8>

   See also:
   <gp2x_video_RGB_color8>
   
   Credits:
   rlyeh */

void gp2x_video_RGB_setpalette(void)
{
  unsigned short *g=(unsigned short *)gp2x_video_RGB_palette; int i=512;
  gp2x_memregs[0x2958>>1]=0;                                                     
  while(i--) gp2x_memregs[0x295A>>1]=*g++; 
}


/* Function: gp2x_video_RGB_setcolorkey
   This function sets the colorkey transparent value for all RGB colorkeyed regions.

   Notes:
   - This R,G,B triad uses 0..15 units instead of 0..255.
   - All windows marked as 0x10 will transparent those pixels which match the given R,G,B triad.
  
   Parameters:
   R (0..15) - red component
   G (0..15) - green component
   B (0..15) - blue component 

   Default:
   - R,G,B are set to 0.

   See also:
   <gp2x_video_RGB_setwindows>
   
   Credits:
   rlyeh */

void gp2x_video_RGB_setcolorkey(int R, int G, int B)
{
 gp2x_memregs[0x2902>>1]=((G&0xF)<<4)+(B&0xF), gp2x_memregs[0x2904>>1]=R&0xF;
}


/* Function: gp2x_video_RGB_setscaling
   This function adjusts a given resolution to fit the whole display (320,240).

   Notes:
   - Draw at (0,0) coordinates always, whatever resolution you are working with.
   - Call this function once. Do not call this function in every frame.
  
   Parameters:
   W (1..) - virtual width in pixels to scale to 320.
   H (1..) - virtual height in pixels to scale to 240.

   Default:
   - (W,H) are set to (320,240).
   
   Credits:
   rlyeh */

void gp2x_video_RGB_setscaling(int W, int H)
{
  int bpp=(gp2x_memregs[0x28DA>>1]>>9)&0x3;

  float mul = (gp2x_memregs[0x2800>>1] & 0x100 ? 512.0 : 1024.0);
  
  // scale horizontal
  gp2x_memregs[0x2906>>1]=(unsigned short)((float)mul *(W/320.0));
  // scale vertical
  gp2x_memregl[0x2908>>2]=(unsigned  long)((float)320.0*bpp *(H/240.0));
}


/* Function: gp2x_video_RGB_flip
   This function flips video buffers. It updates pointers located at <gp2x_video_RGB> struct too.
   The current pointers will point to the hidden display until a new gp2x_video_RGB_flip() call is done.

   Note:
   - This function does not wait for the vertical/horizontal retraces. You should use <gp2x_video_waitvsync> or <gp2x_video_waithsync> if needed.
   - RGB has one layer, one region and four windows. Everything works merged in the same layer.
   - It is a mistake confusing 'layer', 'region', 'part' and 'window'. They are not the same.

   Parameters:
   layer (0) - RGB layer to flip. Note that currently there is support for RGB layer 0 only.

   See also:
   <gp2x_video_YUV_flip>
   
   Credits:
   rlyeh */

void gp2x_video_RGB_flip(int layer)
{
  unsigned long address=gp2x_video_RGB[layer].i[gp2x_video_RGB[layer].i[7]];

  gp2x_frames++;

  asm volatile (".word 0xef9f0002"); // flush cache

  //if(gp2x_video_RGB[layer].i[6]&1) address+=320*240*2;
  //if(gp2x_video_RGB[layer].i[6]&2) address+=320*2;

  if(++gp2x_video_RGB[layer].i[7]==3) gp2x_video_RGB[layer].i[7]=0;
  gp2x_video_RGB[layer].screen=(void  *)gp2x_video_RGB[layer].p[gp2x_video_RGB[layer].i[7]];

  gp2x_memregs[0x290E>>1]=(unsigned short)(address & 0xFFFF);
  gp2x_memregs[0x2910>>1]=(unsigned short)(address >> 16);

  address += 640; 
 
  gp2x_memregs[0x2912>>1]=(unsigned short)(address & 0xFFFF);
  gp2x_memregs[0x2914>>1]=(unsigned short)(address >> 16);
  
}


/* Function: gp2x_video_RGB_setwindows
   This function enables, disables and sets screen up to four RGB windows.

   Notes:
   - You can disable a window by setting a negative value.
   - Any positive value will cause the window to be alpha blended (0..0x0F), to act as a sprite (0x10), or to be solid (0x11).
   - Alpha values are in 0..15 range. Think about it like a 0% to 100% blending in steps of 6.25%.
   - <gp2x_video_RGB_setcolorkey> sets the transparent pixel color for *all* regions.
  
   Parameters:
   window0 (-1..17) - -1 to disable window0, 0..0xF for alpha blending window0, 0x10 for transparent window0, 0x11 for solid window0
   window1 (-1..17) - -1 to disable window1, 0..0xF for alpha blending window1, 0x10 for transparent window1, 0x11 for solid window1
   window2 (-1..17) - -1 to disable window2, 0..0xF for alpha blending window2, 0x10 for transparent window2, 0x11 for solid window2
   window3 (-1..17) - -1 to disable window3, 0..0xF for alpha blending window3, 0x10 for transparent window3, 0x11 for solid window3
   x,y ((0,0)..(320,240)) - Screen coordinate which breaks resolution into 4 regions. Check diagrams below for a few examples.

   >
   > +-------+-------+ (x,y) = (160,120)  +-----------+---+ (x,y) = (240,180)  +---------------+ (x,y) = (319,239)
   > |       |       |                    |           |   |                    |               |
   > |   0   |   1   |                    |           |   |                    |               |
   > |       |       |                    |     0     | 1 |                    |               |
   > +-------P-------+                    |           |   |                    |       0       |
   > |       |       |                    |           |   |                    |               |
   > |   2   |   3   |                    +-----------P---+                    |               |
   > |       |       |                    |     2     | 3 |                    |               |
   > +-------+-------+                    +-----------+---+                    +---------------P
   >
   >
   > +-------+-------+ (x,y) = (160,239)  +---------------+ (x,y) = (0,239)    +---------------+ (x,y) = (0,120)
   > |       |       |                    |               |                    |               |
   > |       |       |                    |               |                    |       1       |
   > |       |       |                    |               |                    |               |
   > |   0   |   1   |                    |       1       |                    P---------------+
   > |       |       |                    |               |                    |               |
   > |       |       |                    |               |                    |       3       |
   > |       |       |                    |               |                    |               |
   > +-------P-------+                    P---------------+                    +---------------+
   >
   >
   > P---------------+ (x,y) = (0,0)      +---+-----------+ (x,y) = (80,60)    +-------P-------+ (x,y) = (160,0)
   > |               |                    | 0 |     1     |                    |       |       |
   > |               |                    +---P-----------+                    |       |       |
   > |               |                    |   |           |                    |       |       |
   > |       3       |                    |   |           |                    |   2   |   3   |
   > |               |                    | 2 |     3     |                    |       |       |
   > |               |                    |   |           |                    |       |       |
   > |               |                    |   |           |                    |       |       |
   > +---------------+                    +---+-----------+                    +-------+-------+
   >
   >
   > +---------------P (x,y) = (319,0)    +---------------+ (x,y) = (319,120)    
   > |               |                    |               |                    
   > |               |                    |       0       |                    
   > |               |                    |               |                    
   > |       2       |                    +---------------P                    
   > |               |                    |               |                    
   > |               |                    |       2       |                    
   > |               |                    |               |                     
   > +---------------+                    +---------------+                     etc...
   >

   Default:
   - (window0,window1,window2,window3) are set to (0x11,-1,-1,-1).
   - (x,y) are set to (319,239).

   See also:
   <gp2x_video_RGB_setcolorkey>
   
   Credits:
   rlyeh */

void gp2x_video_RGB_setwindows(int window0, int window1, int window2, int window3, int x, int y)
{
  int window,mode,mode2,x1,y1,x2,y2;
  int xmax=gp2x_memregs[0x2816>>1], ymax=gp2x_memregs[0x2818>>1];

  //Escalamos las coordenadas segun las dimensiones de la pantalla
  x=(x * (xmax+1)) / 320;
  y=(y * (ymax+1)) / 240;

  //enable all RGB windows
  gp2x_memregs[0x2880>>1]|=(1<<6)|(1<<5)|(1<<4)|(1<<3)|(1<<2);

  for(window=0;window<4;window++) //windows 0..3
  {
   if(window==0) x1=0, y1=0, x2=x,    y2=y   , mode=window0;
   if(window==1) x1=x, y1=0, x2=xmax, y2=y   , mode=window1;
   if(window==2) x1=0, y1=y, x2=x,    y2=ymax, mode=window2;
   if(window==3) x1=x, y1=y, x2=xmax, y2=ymax, mode=window3;
      
   if(mode<0) { gp2x_memregs[0x28da>>1]&=~(1<<(window<<1)); }
   else {
          mode2=(mode>0x0F?0xF:mode);

          //set alpha 0..0xE / colorkey,solid 0xF value
          if(window<3)
          {
           gp2x_memregs[0x28de>>1]&=~(mode2<<(window<<2)); 
           gp2x_memregs[0x28de>>1]|= (mode2<<(window<<2));
          }
          else
          {
           gp2x_memregs[0x28e0>>1]&=~(mode2<<((window-3)<<2)); 
           gp2x_memregs[0x28e0>>1]|= (mode2<<((window-3)<<2)); 
          }

          //set window as blended (2), transparent/colorkey (1), solid (0)          
          gp2x_memregs[0x28dc>>1]&=~(3                                      <<(window<<1));   
          gp2x_memregs[0x28dc>>1]|= ((mode==0x11 ? 0 : (mode==0x10 ? 1 : 2))<<(window<<1)); 
          
          //window coordinates
          gp2x_memregs[(0x28e2+window*8)>>1]=x1;     
          gp2x_memregs[(0x28e4+window*8)>>1]=x2;
          gp2x_memregs[(0x28e6+window*8)>>1]=y1;
          gp2x_memregs[(0x28e8+window*8)>>1]=y2;

          //enable window
          gp2x_memregs[0x28da>>1]|=(3<<(window<<1)); 
        }
  }
}






/*
   ,--------------------.            
   |                    |X
   | GP2X - VIDEO - YUV |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/


/* Function: gp2x_video_YUV_color
   This function returns a YUY2 (Cr0/Y1/Cb0/Y0) 32bits color pixel from a given R,G,B triad.

   Parameters:
   R (0..255) - red component
   G (0..255) - green component
   B (0..255) - blue component

   Credits:
   rlyeh (original code)
   RobBrown (optimized code) */

unsigned long gp2x_video_YUV_color(int R, int G, int B)
{
#if 0
   int Y = ( (  66 * R + 129 * G +  25 * B + 128) >> 8) + 16,
       U = ( ( -38 * R -  74 * G + 112 * B + 128) >> 8) + 128,
       V = ( ( 112 * R -  94 * G -  18 * B + 128) >> 8) + 128;

   return ((V&0xFF)<<24) | ((Y&0xFF)<<16) | ((U&0xFF)<<8) | ((Y&0xFF)<<0);
#else
   int Y = ( (R<<9) + (G<<8) + (B<<8) + (R<<14) + (G<<15) + (B<<11) + (B<<12) + 0x108000 ) & 0xFF0000,
       U = ( (B<<7) - (B<<4) - (R<<5) - (R<<2) - (R<<1) - (G<<6) - (G<<3) - (G<<1) + 0x8080 ) & 0xFF00,
       V = ( (R<<23) - (R<<20) - (G<<21) - (G<<22) + (G<<17) - (B<<20) - (B<<17) + 0x80800000 ) & 0xFF000000;
 
   return (V | Y | U | (Y>>16));
#endif
}

/* Function: gp2x_video_YUV_setscaling
   This function adjusts a given resolution to fit the whole display (320,240).

   Notes:
   - Draw at (0,0) coordinates of each framebuffer always, whatever resolution you are working with.
   - Call this function once. Do not call this function in every frame.
  
   Parameters:
   region (0..3) - YUV region (0..3)
   W (1..) - virtual width in pixels to scale to 320.
   H (1..) - virtual height in pixels to scale to 240.

   Default:
   - (W,H) are set to (320,240) for each region.
   
   Credits:
   rlyeh */

void gp2x_video_YUV_setscaling(int region, int W, int H)
{
  int reg=(region&1 ? 0x28B0:0x2886), bottom=(region&2);

  gp2x_memregs[(reg+0x0+bottom)>>1]       = (unsigned short)((float)1024.0  *(W/320.0));
  gp2x_memregs[(reg+0x4+(bottom<<=1))>>1] = (unsigned short)((float)320.0*2 *(H/240.0)); 
  gp2x_memregs[(reg+0x6+bottom)>>1]       = 0;
}

#if 0
//probar: esto solo funciona con "YUV B top", q creo q es la region 1
void gp2x_video_FDC_setscaling(int W, int H, int W2, int H2)
{
    int region=1;
    int reg=(region&1 ? 0x28B0:0x2886), bottom=(region&2);

    gp2x_memregs[(reg+0x0+bottom)>>1]=(unsigned short)((float)1024.0 *(W/((float)W2)));// scale horizontal
    gp2x_memregs[(reg+0x4+(bottom<<=1))>>1]=(unsigned short)((float) 512.0 *(H/((float)H2)));// scale vertical
    gp2x_memregs[(reg+0x6+bottom)>>1]       = 0;
    yuvb_top_pixelw=W
}
#endif


/* Function: gp2x_video_YUV_flip
   This function flips video buffers. It updates pointers located at <gp2x_video_YUV> struct too.
   The current pointers will point to the hidden display until a new gp2x_video_YUV_flip() call is done.

   Note:
   - This function does not wait for the vertical/horizontal retraces. You should use <gp2x_video_waitvsync> or <gp2x_video_waithsync> if needed.
   - YUV has 2 layers (A/B), 4 regions and no windows.
   - It is a mistake confusing 'window', 'region', 'part' and 'layer'. They are not the same.

   Parameters:
   region (0..3) - YUV layer to flip 

   See also:
   <gp2x_video_RGB_flip>
   
   Credits:
   rlyeh */

void gp2x_video_YUV_flip(int region)
{
  int regions[4]={0x28A0,0x28CA,0x28A8,0x28D2};
  unsigned long address=gp2x_video_YUV[region].i[gp2x_video_YUV[region].i[7]];

  asm volatile (".word 0xef9f0002"); // flush cache

  if(gp2x_video_YUV[region].i[6]&1) address+=320*240*2;
  if(gp2x_video_YUV[region].i[6]&2) address+=320*2;

  if(++gp2x_video_YUV[region].i[7]==2) gp2x_video_YUV[region].i[7]=0;
  gp2x_video_YUV[region].screen=(void *)gp2x_video_YUV[region].p[gp2x_video_YUV[region].i[7]]; 
  
  gp2x_memregs[(regions[region]+0)>>1]=(unsigned short)(address & 0xFFFF);  
  gp2x_memregs[(regions[region]+2)>>1]=(unsigned short)(address >> 16);
  gp2x_memregs[(regions[region]+4)>>1]=(unsigned short)(address & 0xFFFF);
  gp2x_memregs[(regions[region]+6)>>1]=(unsigned short)(address >> 16);
}


/* Function: gp2x_video_YUV_setparts
   This function enables, disables and sets up screen to four YUV parts.

   Notes:
   - You can disable a region by setting a negative value.
   - Any positive value will active the region, and force mirroring or not.
  
   Parameters:
   part0 (-1..3) - -1 disables part0, 0 sets no mirroring, 1 sets vertical mirroring, 2 sets horizontal mirroring, 3 sets both mirrors
   part1 (-1..3) - -1 disables part1, 0 sets no mirroring, 1 sets vertical mirroring, 2 sets horizontal mirroring, 3 sets both mirrors
   part2 (-1..3) - -1 disables part2, 0 sets no mirroring, 1 sets vertical mirroring, 2 sets horizontal mirroring, 3 sets both mirrors
   part3 (-1..3) - -1 disables part3, 0 sets no mirroring, 1 sets vertical mirroring, 2 sets horizontal mirroring, 3 sets both mirrors
   x,y ((0,0)..(320,240)) - Screen coordinate which breaks resolution into 4 parts. Check diagrams below for a few examples.

   >
   > +-------+-------+ (x,y) = (160,120)  +-----------+---+ (x,y) = (240,180)  +---------------+ (x,y) = (319,239)
   > |       |       |                    |           |   |                    |               |
   > |   0   |   1   |                    |           |   |                    |               |
   > |       |       |                    |     0     | 1 |                    |               |
   > +-------P-------+                    |           |   |                    |       0       |
   > |       |       |                    |           |   |                    |               |
   > |   2   |   3   |                    +-----------P---+                    |               |
   > |       |       |                    |     2     | 3 |                    |               |
   > +-------+-------+                    +-----------+---+                    +---------------P
   >
   >
   > +-------+-------+ (x,y) = (160,239)  +---------------+ (x,y) = (0,239)    +---------------+ (x,y) = (0,120)
   > |       |       |                    |               |                    |               |
   > |       |       |                    |               |                    |       1       |
   > |       |       |                    |               |                    |               |
   > |   0   |   1   |                    |       1       |                    P---------------+
   > |       |       |                    |               |                    |               |
   > |       |       |                    |               |                    |       3       |
   > |       |       |                    |               |                    |               |
   > +-------P-------+                    P---------------+                    +---------------+
   >
   >
   > P---------------+ (x,y) = (0,0)      +---+-----------+ (x,y) = (80,60)    +-------P-------+ (x,y) = (160,0)
   > |               |                    | 0 |     1     |                    |       |       |
   > |               |                    +---P-----------+                    |       |       |
   > |               |                    |   |           |                    |       |       |
   > |       3       |                    |   |           |                    |   2   |   3   |
   > |               |                    | 2 |     3     |                    |       |       |
   > |               |                    |   |           |                    |       |       |
   > |               |                    |   |           |                    |       |       |
   > +---------------+                    +---+-----------+                    +-------+-------+
   >
   >
   > +---------------P (x,y) = (319,0)    +---------------+ (x,y) = (319,120)    
   > |               |                    |               |                    
   > |               |                    |       0       |                    
   > |               |                    |               |                    
   > |       2       |                    +---------------P                    
   > |               |                    |               |                    
   > |               |                    |       2       |                    
   > |               |                    |               |                     
   > +---------------+                    +---------------+                     etc...
   >

   Default:
   - (part0,part1,part2,part3) are set to (-1,-1,-1,-1).
   - (x,y) are set to (319,239).
   
   Credits:
   rlyeh */

void gp2x_video_YUV_setparts(int part0, int part1, int part2, int part3, int x, int y)
{
  int e0=part0>=0, e1=part1>=0, e2=part2>=0, e3=part3>=0;

  if(!e0) part0=0; gp2x_video_YUV[0].i[6]=part0; 
  if(!e1) part1=0; gp2x_video_YUV[1].i[6]=part1; 
  if(!e2) part2=0; gp2x_video_YUV[2].i[6]=part2; 
  if(!e3) part3=0; gp2x_video_YUV[3].i[6]=part3; 

  //enable top regions A (0), B (1)
  gp2x_memregs[0x2880>>1]&=~((1<<1)|(1<<0));
  gp2x_memregs[0x2880>>1]|= (e1<<1)|(e0<<0);

  //enable bottom regions A (4), B (9)
  gp2x_memregs[0x2882>>1]&=~((1<<9)|(1<<4));   
  gp2x_memregs[0x2882>>1]|= (e3<<9)|(e2<<4);   

  //set vertical mirroring and/or horizontal mirroring for each region
  gp2x_memregs[0x2882>>1]&=~((3<<7)|(3<<5)|(3<<2)|(3<<0));
  gp2x_memregs[0x2882>>1]|= (part1<<7)|(part3<<5)|(part0<<2)|(part2<<0);

  //set coordinates for A top, A bottom, B top and B bottom regions
  gp2x_memregs[0x2896>>1]=0;     
  gp2x_memregs[0x2898>>1]=x;   
  gp2x_memregs[0x289A>>1]=0;     
  gp2x_memregs[0x289C>>1]=y;   
  gp2x_memregs[0x289E>>1]=239;   
  gp2x_memregs[0x28C0>>1]=x;     
  gp2x_memregs[0x28C2>>1]=319;   
  gp2x_memregs[0x28C4>>1]=0;     
  gp2x_memregs[0x28C6>>1]=y;   
  gp2x_memregs[0x28C8>>1]=239;

  //flush mirroring changes (if any) into current framebuffers
  gp2x_video_YUV_flip(0); gp2x_video_YUV_flip(1); gp2x_video_YUV_flip(2); gp2x_video_YUV_flip(3);
  gp2x_video_YUV_flip(0); gp2x_video_YUV_flip(1); gp2x_video_YUV_flip(2); gp2x_video_YUV_flip(3);
}




/*
   ,--------------------.
   |                    |X
   |   GP2X - BLITTER   |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/


static void gp2x_blitter_rect16(gp2x_rect *r)
{
  int x, y; unsigned short *data=(unsigned short *)r->data, *offset=&((unsigned short *)gp2x_video_RGB[0].screen)[r->x+r->y*320];

  x=r->w, y=r->h; if(r->solid)
                  while(y--) { while(x--) *offset++=*data++; offset+=320-(x=r->w); }
                  else
                  while(y--) { while(x--) { if(*data) *offset=*data; offset++, data++; }
                               offset+=320-(x=r->w); }
}

static void gp2x_blitter_rect8(gp2x_rect *r)
{
  int x, y; unsigned char *data=(unsigned char *)r->data,   *offset=&((unsigned char *)gp2x_video_RGB[0].screen)[r->x+r->y*320]; 

  x=r->w, y=r->h; if(r->solid)
                  while(y--) { while(x--) *offset++=*data++; offset+=320-(x=r->w); }
                  else
                  while(y--) { while(x--) { if(*data) *offset=*data; offset++, data++; }
                               offset+=320-(x=r->w); }
}


/* Function: gp2x_blitter_rect
   This function blits a <gp2x_rect> into screen.

   Note:
   This function does not perform any clipping.
  
   Parameters:
   *r - Pointer to a <gp2x_rect> struct
   
   Credits:
   rlyeh */

void gp2x_blitter_rect(gp2x_rect *r)
{
   if(r->type==RECT_RGB8)  gp2x_blitter_rect8(r);
   if(r->type==RECT_RGB16) gp2x_blitter_rect16(r);
 //if(r->type==RECT_YUV)   gp2x_blitter_rectYUV(r);
}


/* Function: gp2x_blitter_enable
   This function enables or disables the hardware blitter.

   Parameters:
   on - Set 1 for enabling blitter, 0 to disable it.
   
   Note:
   Notice that if blitter is enabled, batteries will drain faster.
   
   See also:
   <gp2x_blitter_wait>, <gp2x_blitter_send>
   
   Credits:
   rlyeh */

void gp2x_blitter_enable(int on)
{
 // Enable/disable FASTIO @ hw blitter 
 if(on) gp2x_memregs[0x904 >> 1] |= (1<<10); else gp2x_memregs[0x904 >> 1] &= ~(1<<10); 

 // Reset clock timings for all devices
 gp2x_memregs[0x90a >> 1] = 0xffff;
}


/* Function: gp2x_blitter_wait
   This function waits for the blitter to be ready.

   See also:
   <gp2x_blitter_enable>, <gp2x_blitter_send>
   
   Credits:
   rlyeh */

void gp2x_blitter_wait(void)
{
 while (gp2x_blitter[0x0034 >> 2] & 1)
 {
 	asm volatile ("nop");
 	asm volatile ("nop");
 	asm volatile ("nop");
	asm volatile ("nop");
 }
}


/* Function: gp2x_blitter_wait
   This function waits for the blitter to be ready.

   See also:
   <gp2x_blitter_enable>, <gp2x_blitter_send>
   
   Credits:
   rlyeh */

void gp2x_blitter_send(gp2x_rect *r)
{
 //unsigned long screen16phys = 0x3101000;

 #if 0

 BLITTER_DST_BPP( , dst_bpp)
 BLITTER_DST_PTR(&screen16phys[x+y*320*videomode.bpp]);
 BLITTER_DST_BPL(320 * videomode.bpp);
 
 BLITTER_SRC_BPP( , src_bpp)
 BLITTER_SRC_PTR(&rect_data_phys[x+y*320*rect.bpp]);
 BLITTER_SRC_BPL(320 * rect.bpp);

 BLITTER_DIM(w, h);

 if(pattern_mask)
 {
  BLITTER_PATTERN(1, 1, 2, 0); //mono,on,1bpp,no offset

  BLITTER_PATTERN_FG_COLOR(pattern_color565);

  BLITTER_PATTERN_SET(pattern_mask);
 }
 
 if(rect.transp)
 BLITTER_ROP(rect.color,1, fifoclear, mirror, rop); //   transparent color / w fifoclear / orientation / D = rop
 else
 BLITTER_ROP(         0,0, fifoclear, mirror, rop); //no transparent color / w fifoclear / orientation / D = rop

 
 BLITTER_GO();
 #endif
 
 
 //if bpp 16, dst_bpp bit 0 = ( x % 2 ) * 16
 //if bpp  8, dst_bpp bit 0 = ( x % 4 ) * 8

 BLITTER_DST_BPP(0,0); //8 bpp
//BLITTER_DST_PTR(&screen16phys[x+y*320]);
 BLITTER_DST_BPL(320);

 BLITTER_SRC_BPP(0,0); //8 bpp
//BLITTER_SRC_PTR(&kk_phys[offset]);
 BLITTER_SRC_BPL(320);

 BLITTER_DIM(320,240);

 //BLITTER_PATTERN(0, 1, 2, 0); 
 //BLITTER_COLORS(0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF); 

//ROPs: 0xAA, 0xF0 (CLEAR?), 0xCC (COPY TRANS?)
BLITTER_ROP(0,0, 0, NO_MIRROR, 4 );           // Fill from top left to bottom right

//BLITTER_ROP( 0,0, 0, NO_MIRROR, 0xaa ); 
//BLITTER_ROP( 0,0, 0, NO_MIRROR, 0xf0 );     // clear? 
//BLITTER_ROP( 0,1, 0, NO_MIRROR, 0xcc );     // copy trans?



/*

 hw fill : should work

 //if bpp 16, dst_bpp bit 0 = ( x % 2 ) * 16
 //if bpp  8, dst_bpp bit 0 = ( x % 4 ) * 8

 BLITTER_DST_BPP(0,0); //8 bpp
 BLITTER_DST_PTR(&screen16phys[x+y*320]);
 BLITTER_DST_BPL(320);

 BLITTER_SRC_BPP(0,0);

 BLITTER_PATTERN(1, 1, 2, 0); //mono,on,1bpp,no offset

 BLITTER_PATTERN_FG_COLOR(0x0000);

 BLITTER_PATTERN_SET(0xFFFFFFFF);

 BLITTER_DIM(w, h);

 BLITTER_ROP(0,0, 0, MIRROR_W|MIRROR_H, ROP_P); //no transparent color / no fifoclear / orientation / D = P (ROPcopy)

 BLITTER_GO();
 
*/



/*

 hw blit from sw : should work

 //if bpp 16, dst_bpp bit 0 = ( x % 2 ) * 16
 //if bpp  8, dst_bpp bit 0 = ( x % 4 ) * 8

 BLITTER_DST_BPP(0,0); //8 bpp
 BLITTER_DST_PTR(&screen16phys[x+y*320]);
 BLITTER_DST_BPL(320);

 //if bpp 16, dst_bpp bit 0 = ( x % 2 ) * 16
 //if bpp  8, dst_bpp bit 0 = ( x % 4 ) * 8

 BLITTER_SRC_BPP(1,0);  //ojo que es 1,0 pa 8bpp, y 1,1 pa 16bpp
 BLITTER_SRC_PTR(0);    //0 ?
 BLITTER_SRC_BPL(320);

 BLITTER_PATTERN(0, 0, 0, 0); //off

 BLITTER_DIM(w, h);

 if(rect.transp)
 BLITTER_ROP(rect.color,1, 1, MIRROR_W|MIRROR_H, ROP_S); //   transparent color / w fifoclear / orientation / D = S (SrcCopy)
 else
 BLITTER_ROP(         0,0, 1, MIRROR_W|MIRROR_H, ROP_S); //no transparent color / w fifoclear / orientation / D = S (SrcCopy)

 BLITTER_GO();

 //fifo ahora (?)

*/


/*

 hw blit from hw : should work

 //if bpp 16, dst_bpp bit 0 = ( rect.x % 2 ) * 16
 //if bpp  8, dst_bpp bit 0 = ( rect.x % 4 ) * 8

 BLITTER_DST_BPP(0,0); //8 bpp
 BLITTER_DST_PTR(&screen16phys[(rect.x+rect.y*320)*rect.bpp]); //320=layer_width
 BLITTER_DST_BPL(320 * rect.bpp);

 //if bpp 16, dst_bpp bit 0 = ( x % 2 ) * 16
 //if bpp  8, dst_bpp bit 0 = ( x % 4 ) * 8

 BLITTER_SRC_BPP(1,0);  //ojo que es 1,0 pa 8bpp, y 1,1 pa 16bpp
 BLITTER_SRC_PTR(&screenphys);
 BLITTER_SRC_BPL(320*1);

 BLITTER_PATTERN(1, 1, 2, 0); //mono, on, 1bpp, no offset

 BLITTER_PATTERN_FG_COLOR(0xFFFF)

 BLITTER_PATTERN_SET(0xFFFFFFFF)

 BLITTER_DIM(w, h);

 if(rect.transp)
 BLITTER_ROP(rect.color,1, 1, MIRROR_W|MIRROR_H, ROP_S); //   transparent color / w fifoclear / orientation / D = S (SrcCopy)
 else
 BLITTER_ROP(         0,0, 1, MIRROR_W|MIRROR_H, ROP_S); //no transparent color / w fifoclear / orientation / D = S (SrcCopy)

 BLITTER_GO();

 //fifo ahora (?)

*/

 __asm__ __volatile__ ("" : : :"memory");

 BLITTER_GO(); 
}






/*
   ,--------------------.
   |                    |X
   |   GP2X - JOYSTICK  |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/

/* Internal function which inits usb joystick variables.

   Credits:
   rlyeh */

static void gp2x_joystick_init (void)
{
 char device[32]; int i=-1; gp2x_usbjoys = 0;

 while(++i<4) { sprintf (device, "/dev/input/js%d", i); 
                if((gp2x_usbjoy[i] = open(device, O_RDONLY|O_NONBLOCK, 0)) >0) gp2x_usbjoys++; }
}


/* Internal function which deinits usb joystick variables.

   Credits:
   rlyeh */

static void gp2x_joystick_deinit(void)
{
 int i; for(gp2x_usbjoys=i=0;i<4;i++) if(gp2x_usbjoy[i] > 0) close(gp2x_usbjoy[i]);
}


/* Function: gp2x_joystick_scan
   This function forces a new scan looking for recently plugged joysticks.

   Note:
   - Notice that Minimal Library scans already for plugged USB joysticks at start of program.
   - It would be wise to check, after calling this function, new and old joysticks both together with <gp2x_joystick_name>
   
   See also:
   <gp2x_joystick_name>, <gp2x_usbjoys>

   Credits:
   rlyeh */

void gp2x_joystick_scan(void)
{
 gp2x_joystick_deinit();
 gp2x_joystick_init();
}

/* Function: gp2x_joystick_read
   This function returns the active <GP2X joystick values>.

   Usage:
   Call this function once per frame to keep your joystick values updated.

   Parameters:
   joystick - 0 for GP2X pad, 1 for first USB joystick, 2 for second USB joystick, etc...

   Note:
   In order to detect simultaneous buttons you will have to mask the value.

   See also:
   <gp2x_joystick_wait>, <GP2X joystick values>, <gp2x_usbjoys>

   Example:
   > unsigned long pad=gp2x_joystick_read(0);
   >
   > if(pad==GP2X_A) ...               //check that only A is pressed.
   > if(pad&GP2X_A)  ...               //check that A is pressed, despite the other buttons.
   > if(pad&GP2X_R) if(pad&GP2X_L) ... //check that both L and R are pressed, despite the other buttons.

   Credits:
   Puck2099 and GnoStiC (original code)
   rlyeh */

unsigned long gp2x_joystick_read(int joystick)
{
 unsigned long value;
 
 if(!joystick)
 {
  value=(gp2x_memregs[0x1198>>1] & 0x00FF);

  if(value==0xFD) value=0xFA;
  if(value==0xF7) value=0xEB;
  if(value==0xDF) value=0xAF;
  if(value==0x7F) value=0xBE;
  
  value = (~((gp2x_memregs[0x1184>>1] & 0xFF00) | value | (gp2x_memregs[0x1186>>1] << 16))) & (~0xc0040000);
  
  if(gp2x_f200)
    value |= ((value & (GP2X_LEFT|GP2X_RIGHT|GP2X_UP|GP2X_DOWN)) == (GP2X_LEFT|GP2X_RIGHT|GP2X_UP|GP2X_DOWN) ? GP2X_PUSH : 0);
  
  return value;
 }
 else
 { 
  struct JS_DATA_TYPE js; int i;

  if(read(gp2x_usbjoy[joystick-1], &js, JS_RETURN) != JS_RETURN) return 0;

  i = js.buttons;
  value  = ((i &    1)>> 0) * (GP2X_Y); /* these values should be optimized at compilation time */
  value |= ((i &    2)>> 1) * (GP2X_B);
  value |= ((i &    4)>> 2) * (GP2X_X);
  value |= ((i &    8)>> 3) * (GP2X_A);
  value |= ((i &   16)>> 4) * (GP2X_L);
  value |= ((i &   32)>> 5) * (GP2X_R);
  value |= ((i &   64)>> 6) * (GP2X_L);
  value |= ((i &  128)>> 7) * (GP2X_R);
  value |= ((i &  256)>> 8) * (GP2X_SELECT);
  value |= ((i &  512)>> 9) * (GP2X_START);
  value |= ((i & 1024)>>10) * (GP2X_PUSH);
  value |= ((i & 2048)>>11) * (GP2X_PUSH);

  if(js.x & 0x7F) value |= (js.x & 0x80 ? GP2X_RIGHT : GP2X_LEFT);
  if(js.y & 0x7F) value |= (js.y & 0x80 ? GP2X_UP    : GP2X_DOWN);
 }
 return value;
} 

/* Function: gp2x_joystick_name
   This function returns name for joystick.

   Note:
   - Function returns an empty string if joystick not found.

   Parameters:
   joystick - 0 for GP2X pad, 1 for first USB joystick, 2 for second USB joystick, etc...

   Example:
   > printf(gp2x_joystick_name(0)); // printfs "GP2X gamepad"
   > printf(gp2x_joystick_name(1)); // printfs 1st usb joystick name if plugged

   See also:
   <gp2x_joystick_scan>, <gp2x_usbjoys>

   Credits:
   rlyeh */

char *gp2x_joystick_name(int joystick)
{
 static char name[128];
 
 sprintf(name, !joystick ? "GP2X gamepad" : "");
 if(joystick>0) if(gp2x_usbjoy[joystick-1] > 0) ioctl(gp2x_usbjoy[joystick-1], JSIOCGNAME(sizeof(name)), name); 
 
 return name;
}

/* Function: gp2x_joystick_wait
   This function waits for joystick to press *and* release a combination.

   Parameters:
   joystick - 0 for GP2X pad, 1 for first USB joystick, 2 for second USB joystick, etc...
   combination - a combination of or'ed <GP2X joystick values> values

   Example:
   > gp2x_joystick_wait(1, GP2X_A | GP2X_START); // waits until A & START are simultaneously pressed and released on usb joystick #1

   See also:
   <gp2x_joystick_read>, <GP2X joystick values>, <gp2x_usbjoys>

   Credits:
   rlyeh */

void gp2x_joystick_wait(int joystick, unsigned long combination)
{
 while(gp2x_joystick_read(joystick) != combination);
 while(gp2x_joystick_read(joystick) == combination);
}



/* F */

/*
void gp2x_touchscreen_update(void)
{
    if(!gp2x_f200)
    {
        //GP2X_TOUCH = 0;
        GP2X_TOUCH_X = GP2X_TOUCH_Y = 0;
    }
    else
    {
        //GP2X_TOUCH = 1?;
        GP2X_TOUCH_X = 0;
        GP2X_TOUCH_Y = 0;
    }
}
*/


/*
   ,--------------------.
   |                    |X
   |    GP2X - TIMER    |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/

/* Function: gp2x_timer_delay
   This function halts the program for a given delay in <gp2x_ticks> units.

   Parameters:
   ticks (1..) - delay in <gp2x_ticks> units.

   See also:
   <gp2x_timer_read>, <gp2x_ticks>

   Credits:
   rlyeh */

void gp2x_timer_delay(unsigned long ticks)
{
  unsigned long target=gp2x_memregl[0x0A00>>2]+ticks*gp2x_ticks_per_second;
  while(gp2x_memregl[0x0A00>>2]<target);
}

/* Function: gp2x_timer_read_ticks
   This function returns elapsed ticks in <gp2x_ticks> units since last <gp2x_init> call.

   Note:
   - There is no way to reset <gp2x_ticks> value to 0 currently.

   See also:
   <gp2x_timer_read_raw>, <gp2x_timer_read>, <gp2x_ticks> 

   Credits:
   rlyeh */

unsigned long gp2x_timer_read(void)
{
  return gp2x_memregl[0x0A00>>2]/gp2x_ticks_per_second;
}

/* Function: gp2x_timer_read_raw
   This function returns the current GP2X timer value. That is 7372800 ticks per second.

   Returns:
   Current GP2X timer value.
   
   See also:
   <gp2x_timer_read>, <gp2x_timer_read_ticks>, <gp2x_timer_raw_to_ticks> 
   
   Credits:
   rlyeh */

unsigned long gp2x_timer_read_raw(void)
{
 return gp2x_memregl[0x0A00>>2];
}

/* WRITEME */
void gp2x_timer_reset(void)
{
 gp2x_memregl[0x0A00>>2] = 0;
 gp2x_memregl[0x0A00>>2] = 0;
}

/* Function: gp2x_timer_raw_to_ticks
   This function returns given raw ticks into <gp2x_ticks> ticks.

   Parameters:
   ticks - raw ticks to convert to <gp2x_ticks>.
   
   Returns:
   <gp2x_ticks> converted ticks.
   
   See also:
   <gp2x_timer_raw>, <gp2x_ticks>
   
   Credits:
   rlyeh */

unsigned long gp2x_timer_raw_to_ticks(unsigned long ticks)
{
 return ticks/gp2x_ticks_per_second;
}

/* Function: gp2x_timer_one_second
   This function returns the number of raw ticks per second.
   
   Returns:
   7372800.
   
   See also:
   <gp2x_ticks>
   
   Credits:
   rlyeh */

unsigned long gp2x_timer_raw_one_second(void)
{
 return 7372800;
}

/* Function: gp2x_counter_init
   This function iniates a timer.
   This timer works in <gp2x_tick> units.
   
   Parameters:
   timerN (0..15) - timer to init.
   
   See also:
   <gp2x_counter_start>, <gp2x_counter_pause>
   
   Credits:
   rlyeh */

void gp2x_counter_init(int timerN)
{
 gp2x_timer_reset();
 gp2x_timer[timerN] = gp2x_timer_counter[timerN] = gp2x_timer_active[timerN] = gp2x_timer_overflow[timerN] = 0;
}

/* Function: gp2x_timer_start
   This function starts a timer.
   
   Parameters:
   timerN (0..15) - timer to start.
   
   Note:
   - You can check the current timer counter by checking gp2x_timer[timerN] variable.
   
   See also:
   <gp2x_counter_init>, <gp2x_counter_pause>
   
   Credits:
   rlyeh */

void gp2x_counter_start(int timerN)
{
  if(gp2x_timer_active[timerN]) return;
  
  gp2x_timer_active[timerN] = 1;

  
  gp2x_timer[timerN] = gp2x_timer_overflow[timerN] = gp2x_memregl[0x0A00>>2];
}

/* Function: gp2x_timer_pause
   This function pauses a timer.
   
   Parameters:
   timerN (0..15) - timer to stop.
   
   See also:
   <gp2x_timer_init>, <gp2x_timer_start>
   
   Credits:
   rlyeh */

void gp2x_counter_pause(int timerN)
{
 unsigned long long int timer;
 
 if(!gp2x_timer_active[timerN]) return;

 gp2x_timer_active[timerN] = 0;

 
 timer = gp2x_memregl[(0x0A00)>>2]; 

 if(timer >= gp2x_timer[timerN])
  gp2x_timer[timerN] = timer - gp2x_timer[timerN];
 else
  gp2x_timer[timerN] = timer + (0xFFFFFFFF - gp2x_timer[timerN]); //fix 9.7 minutes overflow

  
 gp2x_timer_counter[timerN] += gp2x_timer[timerN];
 
 gp2x_timer[timerN] = 0;
}


/* Function: gp2x_counter_read
   This function reads a timer counter.
   
   Parameters:
   timerN (0..15) - timer to read from.
   
   Note:
   - <gp2x_> polls 32bits hardware register at a frequency of 7372800 ticks per second. This causes the system to overflow the 32bit register each 9.7 minutes whatever <gp2x_tick> you decide to use.
   - <gp2x_timer_init>, <gp2x_timer_start>, <gp2x_timer_read>, <gp2x_timer_pause> are 64bit software timers based. These timing functions are slighty slower than raw timing ones, but they will overflow in 79337 years. We will be dead at that time probably.
   - Choose the method that fits better for your needs. Up to you.

WRITEME   
   
   See also:
   <gp2x_timer_read_raw>, <gp2x_timer_read_ticks>, <gp2x_timer_raw_to_ticks>, <gp2x_ticks> 
   
   Credits:
   rlyeh */

unsigned long long int gp2x_counter_read(int timerN)
{
 signed long long int timer = 0;
 
 if(gp2x_timer_active[timerN])
 { 
    timer = gp2x_memregl[(0x0A00)>>2] - gp2x_timer[timerN];
    
    if(timer < 0)
        timer += (4294967296LL);
 }
 
 return timer + gp2x_timer_counter[timerN];
}



/*
   ,--------------------.
   |                    |X
   |  GP2X - PROFILER   |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/

/* Private: gp2x_profile_deinit
   Function which deinits gp2x_profiles_string in case it has been allocated before (by gp2x_profile_analyze)

   Credits:
   rlyeh */

static void gp2x_profile_deinit(void)
{
 if(gp2x_profiles_string!=NULL) free(gp2x_profiles_string);
}

/* Private: gp2x_profile_register
   Function which controls profiling variables.

   Note:
   - You should not call this function directly. Use <gp2x_profile> macro instead.

   See also:
   <gp2x_profile>

   Credits:
   rlyeh */

gp2x_profiles *gp2x_profile_register(void)
{
 if(gp2x_profiles_done == gp2x_profiles_roomspace)
  gp2x_profiles_area = (gp2x_profiles *)realloc(gp2x_profiles_area, sizeof(gp2x_profiles) * (gp2x_profiles_roomspace += 1024));

 return &gp2x_profiles_area[gp2x_profiles_done++];
}
       
/* Function: gp2x_profile_analyze
   This function analyzes all information gathered through <gp2x_profile> macro,
   and returns formatted information.

   Example:
   >
   > while(condition) {
   > gp2x_profile( draw() );
   > gp2x_profile( input() );
   > gp2x_profile( network() );
   > } }
   > gp2x_printf(NULL, 0,0, "%s", gp2x_profile_analyze());
   >

   See also:
   <gp2x_profile>

   Credits:
   rlyeh */

char *gp2x_profile_analyze(void)
{
 gp2x_profiles *p;
 static char *s, *t, *u; unsigned long i, j, profiled=0, *orderlist, *valuelist;
 signed long nonprofiled = 0;
 unsigned long executed = ( gp2x_memregl[0x0A00>>2] - gp2x_profiles_calibration);

 if(gp2x_profiles_string==NULL) { memset((gp2x_profiles_string = s = (char *)malloc(16*1024)),0,16*1024); t=&s[15*1024];
                                  atexit(gp2x_profile_deinit); }

 if(gp2x_profiles_done) { /*prepare sorting values*/
                          orderlist = (unsigned long *)malloc(gp2x_profiles_done * sizeof(unsigned long));
                          valuelist = (unsigned long *)malloc(gp2x_profiles_done * sizeof(unsigned long));

                          for(i=0;i<gp2x_profiles_done;i++) { p = &gp2x_profiles_area[i];
                                                              orderlist[i] = i;
                                                              profiled += ( valuelist[i] = p->time ); }

                          /* method A) fix any precision error */
                          nonprofiled = executed - profiled;
                          if(nonprofiled<0) { executed += abs(nonprofiled); nonprofiled = 0; } 

                          /* method B) fix any precision error */
                          //executed = abs((signed)executed);
                          //nonprofiled = executed - profiled;

                          /*sort list*/
                          for(i=0;i<gp2x_profiles_done-1;i++)
                           for(j=i+1;j<gp2x_profiles_done;j++)
                            if(valuelist[i] < valuelist[j])
                            {
                             unsigned long swap;
                             swap = orderlist[j]; orderlist[j] = orderlist[i]; orderlist[i] = swap;
                             swap = valuelist[j]; valuelist[j] = valuelist[i]; valuelist[i] = swap;
                            }

                          /*make string*/
                          for(i=0;i<gp2x_profiles_done;i++) { p = &gp2x_profiles_area[orderlist[i]];
                                                              if((u = strrchr (p->fname, '/')) == NULL) if((u = strrchr (p->fname, '\\')) == NULL) u = p->fname;
                                                              sprintf(t,"+%3lu%% %10lu ticks per call; %lu calls at %s%s\n", (p->time / (executed/100)), p->time / p->calls, p->calls, u != p->fname ? u+1 : u, u != p->fname ? "()" : "" );
                                                              strcat(s,t); }


                          sprintf(t,"+%3lu%% %10lu ticks in non profiled code\n", nonprofiled/(executed/100), nonprofiled);
                          strcat(s,t);

                          sprintf(t,"-----\n~100%% (Total execution time %lu seconds or %lu ticks aproximately)\n", executed / 7372800, executed);
                          strcat(s,t);

                          free(orderlist);
                          free(valuelist); }

 else sprintf(s,"Check you're not defining GP2X_PROFILE_DISABLE and you're using gp2x_profile()\n");

 return s;
}





/*
   ,--------------------.
   |                    |X
   |    GP2X - SOUND    |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/


/* Function: gp2x_sound_frame
   This function is automatically called by the library, and expects a sound buffer to be filled in.

   Notes:
   - This function is not included in the library itself.
   - Supply this function in your own program.
   - Do not call this function directly.

   Parameters:
   *blah - unused in this release.
   *buffer - pointer to store the sound data.
   samples - number of *samples*, *NOT* bytes.

   Example:
   Check tutorials section.

   Credits:
   you :-) */

extern void gp2x_sound_frame(void *blah, void *buffer, int samples);

/* Function: gp2x_sound_volume
   This function sets the sound volume.

   Parameters:
   L (0..100) - volume percentage for the left channel
   R (0..100) - volume percentage for the right channel

   Note:
   - A higher percentage than 100 will distort your sound.

   Default:
   - L is set to 100.
   - R is set to 100.

   See also:
   <gp2x_sound_pause>

   Credits:
   snaff */

void gp2x_sound_volume(int L /*0..100*/, int R /*0..100*/)
{
 if(gp2x_f200)
    L >>= 1, R >>= 1;
    
 L=(((L*0x50)/100)<<8)|((R*0x50)/100);          
 ioctl(gp2x_dev[4], SOUND_MIXER_WRITE_PCM, &L); 
}

/* Function: gp2x_sound_pause
   This function pauses and unpauses the sound.

   Parameters:
   yes (0,1) - set it to 1 to enable pause (mute sound), and 0 to disable pause (play sound).

   See also:
   <gp2x_sound_volume>

   Credits:
   rlyeh */

void gp2x_sound_pause(int yes)
{
 gp2x_sound_pausei=yes;
}

/* Function: gp2x_sound_stereo
   This function enables stereo.
   There is no way currently to revert changes done.
   
   Parameters:
   on - unused. To be fixed.  
   
   See also:
   <gp2x_sound_3Dboost>, <gp2x_sound_attenuation>, <gp2x_sound_setintensity>
   
   Credits:
   NK */

void gp2x_sound_stereo(int on)
{
 if(on); //FIXME
 gp2x_memregs[(0xEF0+0x26)>>1] = 0x830a; usleep(100000); 
 gp2x_memregs[(0xEF0+0x68)>>1] = 0x100c; usleep(100000);
}

/* Function: gp2x_sound_3Dboost
   This function enables 3D DAC and lower+upper cut-off at 100% depth.
   There is no way currently to revert changes done.
   
   Parameters:
   on - unused. To be fixed.
   
   See also:
   <gp2x_sound_stereo>, <gp2x_sound_attenuation>, <gp2x_sound_setintensity>

   Credits:
   rlyeh */

void gp2x_sound_3Dboost(int on)
{
 if(on); //FIXME

 gp2x_memregs[(0xEF0+0x20)>>1] |= (1<<13);             //enable 3D DAC

 gp2x_memregs[(0xEF0+0x22)>>1] &= ~0x3f;
 gp2x_memregs[(0xEF0+0x22)>>1] |= (0<<5)|(0<<4)|(0xf); //lower+upper cut-off, 100% depth
}


/* Function: gp2x_sound_attenuation
   This function enables enable 6db pre-dac attenuation.
   There is no way currently to revert changes done.
   
   Parameters:
   on - unused. To be fixed.
   
   See also:
   <gp2x_sound_stereo>, <gp2x_sound_3Dboost>, <gp2x_sound_setintensity>

   Credits:
   rlyeh */

void gp2x_sound_attenuation(int on)
{
 gp2x_memregs[(0xEF0+0x08)>>1] |= (1<<6);
}


/* Function: gp2x_sound_setintensity
   This function enables 3D DAC and lower+upper cut-off at 100% depth.
   There is no way currently to revert changes done.
   
   Parameters:
   bass_boost (0..1) - 0 for normal operation, 1 for bass boost. 
   bass (0..15) - 0 (9dB) to 14 (-6dB), or 15 to disable bass intensity.
   treble (0..15) - 0 (9dB) to 14 (-6dB), or 15 to disable treble intensity.
   
   See also:
   <gp2x_sound_stereo>, <gp2x_sound_attenuation>, <gp2x_sound_3Dboost>

   Credits:
   rlyeh */

void gp2x_sound_setintensity(int boost, int bass, int treble)
{
 gp2x_memregs[(0xEF0+0x08)>>1] &= ~((boost<<15)|(bass<<8)|(treble<<0));
}




static void *gp2x_sound_play(void *blah)
{
  int flip1=0, flip2=gp2x_sound_buffer[1];

  while(!gp2x_exit)       if(!gp2x_sound_pausei) { gp2x_sound_frame(blah, (void *)(&gp2x_sound_buffer[4+flip1]), gp2x_sound_buffer[0]);
                                                   write(gp2x_dev[3],     (void *)(&gp2x_sound_buffer[4+flip2]), gp2x_sound_buffer[1]);

                                                   flip1+=gp2x_sound_buffer[1]; if(flip1==gp2x_sound_buffer[1]*8) flip1=0;
                                                   flip2+=gp2x_sound_buffer[1]; if(flip2==gp2x_sound_buffer[1]*8) flip2=0;
                                           
                                                   } else usleep(100000); 
  return NULL;
  //pthread_exit( NULL );
}





/*
   ,--------------------.
   |                    |X
   |   GP2X - DUALCORE  |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/

static void gp2x_initqueue(gp2x_queue *q, unsigned long queue_items, unsigned long *position920t, unsigned long *position940t)
{
  q->head = q->tail = q->items = 0;
  q->max_items = queue_items;
  if(position920t) q->place920t=position920t; else q->place920t=(unsigned long *)malloc(sizeof(unsigned long) * queue_items);
  if(position940t) q->place940t=position940t;
  memset(q->place920t, 0, sizeof(unsigned long) * queue_items);
}

static void gp2x_enqueue(gp2x_queue *q, unsigned long data)
{	
  while(q->items==q->max_items); /*waiting for tail to decrease...*/
  q->place920t[q->head = (q->head < q->max_items ? q->head+1 : 0)] = data;
  q->items++;
}


#if 0 // UNUSED
static unsigned long gp2x_dequeue(gp2x_queue *q)
{
  while(!q->items); //waiting for head to increase...
  q->items--;
  return q->place920t[q->tail = (q->tail < q->max_items ? q->tail+1 : 0)];
}
#endif 


static void gp2x_save_registers(int mode)  //0=save, 1=restore, 2=restore & exit 
{                 
 static unsigned short *reg=NULL; unsigned long ints, i,c;
 unsigned short savereg[]={     //0x0904>>1,0x0904>>1, //Clocks & power management
                                //0x0912>>1,0x0912>>1, //Clocks & power management
                                //0x091C>>1,0x091C>>1, //Clocks & power management
                                //0x3B40>>1,0x3B48>>1, //Dual CPU
                                0x2880>>1,0x28AE>>1, //YUV A
                                0x28B0>>1,0x28D8>>1, //YUV B
                                0x28DA>>1,0x28E8>>1, //RGB
                                0x290C>>1,0x290C>>1, //RGB - line width
                                0x291E>>1,0x2932>>1, //Cursor
//                              0x1800>>1,0x1836>>1, //Scale processor
//                              0x2800>>1,0x2848>>1, //Display controller
//                              0x2880>>1,0x2980>>1, //Multilayer controller
//                              0x3000>>1,0x30F2>>1, //Image signal processor
                                0,0};

 if(reg==NULL) reg=(unsigned short *)malloc(0x10000);

 ints=gp2x_memregl[0x0808>>2];
 gp2x_memregl[0x0808>>2] = 0xFF8FFFE7;  //mask interrupts

 if(mode) {for(c=0;savereg[c];c+=2) for(i=savereg[c];i<=savereg[c+1];i++) gp2x_memregl[i]=reg[i];}
     else {for(c=0;savereg[c];c+=2) for(i=savereg[c];i<=savereg[c+1];i++) reg[i]=gp2x_memregl[i];}
     
 if(mode==2) { free(reg); reg=NULL; }

 gp2x_memregl[0x0808>>2]=ints;
}



/* Function: gp2x_dualcore_clock
   This function sets the clock at ARM 920T coprocessor and RAM tweaks.

   Note:
   - Call this function only from your 920T program.

   Parameters:
   s (50,75,100,125,150,175,200,225,250) - Speed value in MHz. Default is 200.

   Credits:
   RobBrown, sasq and whizzbang (original clock setting code)
   
   god_at_hell (original RAM tweaking code)
   
   craigix, Emu&Co (tweaked RAM timing values) */

void gp2x_dualcore_clock(int s)
{
 unsigned long interrupt_flags = gp2x_memregl[0x808>>2],
 //tRC=7-1 if mhz > 266
 //CAS=0,tRC=6-1,tRAS=3-1,tWR=1-1,tMRD=1-1,tRFC=1-1,tRP=1-1,tRCD=2-1; //emu&co's timings
   CAS=0,tRC=6-1,tRAS=4-1,tWR=1-1,tMRD=1-1,tRFC=1-1,tRP=2-1,tRCD=2-1; //craigix's timings

 //hardcoded values:
 s=(s>=250?0x5D04:s>=225?0x5304:s>=200?0x4904:s>=175?0x3F04:s>=125?0x3c01:s>=100?0x6502:s>=75?0x4902:s>=50?0x6503:0);

 if(s)
 {
  gp2x_memregl[0x0808>>2] = 0xFF8FFFE7;        // Mask interrupts
  
  gp2x_memregs[0x0910>>1] = s;                 // Set clock and wait 

  while(gp2x_memregs[0x0902>>1] & 1);

  gp2x_memregs[0x3802>>1] = ((tMRD<<12)|(tRFC<<8)|(tRP<<4)|(tRCD)); //set RAM tweaks
  gp2x_memregs[0x3804>>1] = ((CAS<<12)|(tRC<<8)|(tRAS<<4)|(tWR));

  gp2x_memregs[0x0924>>1] = 0x8900 + ((1)<<8); // Set upll timing prescaler to 1 (0x5A00 for fw 1.0.x)

  gp2x_memregl[0x0808>>2] = interrupt_flags;   // Turn on interrupts
 }
}


/* Function: gp2x_dualcore_pause
   This function pauses the secondary ARM 940T coprocessor.

   Note:
   - Call this function only from your 920T program.

   Parameters:
   yes (0,1) - set it to 1 to enable pause (halt 940T), and 0 to disable pause (run 940T).

   See also:
   <gp2x_dualcore_sync>, <gp2x_dualcore_exec>, <gp2x_dualcore_data(address)> */

       void gp2x_dualcore_pause(int yes) { if(yes) gp2x_memregs[0x0904>>1] &= 0xFFFE; else gp2x_memregs[0x0904>>1] |= 1; }
static void gp2x_940t_reset(int yes)     { gp2x_memregs[0x3B48>>1] = ((yes&1) << 7) | (0x02); } //2 = 0x2000000 = 940BANK
static void gp2x_940t_pause(int yes)     { gp2x_dualcore_pause(yes); }


/* Function: gp2x_dualcore_sync
   This function halts the 920T code until the 940T has finished all its tasks.

   Note:
   - Call this function only from your 920T program.

   See also:
   <gp2x_dualcore_pause>, <gp2x_dualcore_exec>, <gp2x_dualcore_data(address)>

   Credits:
   rlyeh */

void gp2x_dualcore_sync(void)
{
  gp2x_queue *q=(gp2x_queue *)gp2x_1stcore_data_ptr(GP2X_QUEUE_STRUCT_PTR);
  while(q->items);
}

/* Function: gp2x_dualcore_exec
   This function executes a 940T command.

   Note:
   - Call this function only from your 920T program.

   See also:
   <gp2x_dualcore_pause>, <gp2x_dualcore_sync>, <gp2x_dualcore_data(address)>

   Credits:
   rlyeh */

void gp2x_dualcore_exec(unsigned long command) { gp2x_enqueue((gp2x_queue *)gp2x_1stcore_data_ptr(GP2X_QUEUE_STRUCT_PTR),command); }


/* Function: gp2x_dualcore_launch_program
   This function sets the 940T program, and launchs it.

   Note:
   - Call this function only from your 920T program.

   Parameters:
   *area - pointer to the 940T raw binary.
   size (1..) - size of 940T raw binary, in bytes.

   See also:
   <gp2x_dualcore_launch_program_from_disk>, <gp2x_dualcore_launch_subprogram(name)>, <gp2x_dualcore_launch_HH>

   Credits:
   rlyeh */

void gp2x_dualcore_launch_program(unsigned long *area, unsigned long size)
{
  unsigned long i=0, *arm940t_ram=(unsigned long *)&gp2x_dualcore_ram[0];
  
  gp2x_940t_reset(1);
  gp2x_940t_pause(1);            

  while(i < size) *arm940t_ram++=area[i++];

  gp2x_memregs[0x3B40>>1] = 0;                               //disable interrupts
  gp2x_memregs[0x3B42>>1] = 0;
  gp2x_memregs[0x3B44>>1] = 0xffff; //needed?
  gp2x_memregs[0x3B46>>1] = 0xffff; //needed?

  gp2x_initqueue(
  (gp2x_queue *)gp2x_1stcore_data_ptr(GP2X_QUEUE_STRUCT_PTR),
   GP2X_QUEUE_MAX_ITEMS, (unsigned long *)gp2x_1stcore_data_ptr(GP2X_QUEUE_DATA_PTR), (unsigned long *)gp2x_2ndcore_data_ptr(GP2X_QUEUE_DATA_PTR));

  gp2x_940t_reset(0);
  gp2x_940t_pause(0);
}

/* Function: gp2x_dualcore_launch_program_from_disk
   This function sets the 940T program, and launchs it.

   Note:
   - Call this function only from your 920T program.

   Parameters:
   *file - string to the file which contains the 940T raw binary.
   offset (0..) - amount of bytes to skip at the beginning of the file.
   size (1..) - size of 940T raw binary, in bytes.

   See also:
   <gp2x_dualcore_launch_program>, <gp2x_dualcore_launch_subprogram(name)>, <gp2x_dualcore_launch_HH>

   Credits:
   rlyeh */

void gp2x_dualcore_launch_program_from_disk(char *file, unsigned long offset, unsigned long size)
{
 FILE *in; void *data;

 if((in=fopen(file, "rb"))==NULL) return;
 if((data=malloc(size))==NULL) { fclose(in); return; }
 fseek(in, 0L, offset);
 fread(data, 1, size, in);
 gp2x_dualcore_launch_program((unsigned long *)data, size);
 free(data);
 fclose(in);
}


/* Function: gp2x_dualcore_launch_HH
   This function exits Linux and starts supplied hardcore homebrew (HH) program.

   Parameters:
   *file - string for a valid raw binary HH program (usually coded with RobBrown's sdk2x)

   See also:
   <gp2x_dualcore_launch_program>, <gp2x_dualcore_launch_program_from_disk>, <gp2x_dualcore_launch_subprogram(name)>

   Credits:
   RobBrown (original trampoline code & sdk2x code)
   rlyeh */

void gp2x_dualcore_launch_HH(char *file)
{
 const unsigned char prog[] = {
 0x01,0x00,0x00,0xea,0x0c,0x2a,0x80,0x03,0xd4,0x2a,0x80,0x03,0x00,0x00,0x0f,0xe1,
 0x1f,0x00,0xc0,0xe3,0xd3,0x00,0x80,0xe3,0x00,0xf0,0x29,0xe1,0x98,0x00,0x9f,0xe5,
 0x00,0x10,0xa0,0xe3,0x00,0x10,0x80,0xe5,0x00,0x10,0xe0,0xe3,0x8c,0x00,0x9f,0xe5,
 0x00,0x10,0x80,0xe5,0x88,0x10,0x9f,0xe5,0x00,0x00,0x00,0xe2,0x07,0x0c,0x80,0xe2,
 0xf0,0x00,0x80,0xe2,0x08,0x00,0x80,0xe2,0xb0,0x00,0xc1,0xe1,0x00,0x00,0xa0,0xe3,
 0x17,0x0f,0x07,0xee,0x17,0x0f,0x08,0xee,0x10,0x0f,0x11,0xee,0x23,0x0c,0xc0,0xe3,
 0x87,0x00,0xc0,0xe3,0x02,0x00,0x80,0xe3,0x01,0x0a,0x80,0xe3,0x10,0x0f,0x01,0xee,
 0x50,0x00,0x9f,0xe5,0x02,0x08,0x80,0xe2,0x0c,0xd0,0x40,0xe2,0x80,0x00,0x1f,0xe5,
 0x80,0x10,0x1f,0xe5,0x00,0x20,0xa0,0xe3,0x01,0x00,0x50,0xe1,0x01,0x00,0x00,0x0a,
 0x01,0x20,0xc0,0xe4,0xfb,0xff,0xff,0xea,0x8e,0x02,0x00,0xeb,0x00,0x00,0xa0,0xe3,
 0x27,0x13,0xa0,0xe3,0x02,0x2c,0xa0,0xe3,0x01,0x30,0xd1,0xe4,0x01,0x30,0xc0,0xe4,
 0x02,0x00,0x50,0xe1,0xfb,0xff,0xff,0x1a,0x00,0xf0,0xa0,0xe3,0x14,0x0a,0x00,0xc0,
 0x08,0x08,0x00,0xc0,0x3c,0x3a,0x00,0xc0,0xdc,0xd0,0xf9,0x03,0x00,0x00,0xa0,0xe1,
 0x70,0x40,0x2d,0xe9,0x9f,0x01,0x00,0xeb,0x02,0x4b,0xa0,0xe3,0x07,0x36,0xe0,0xe3,
 0x23,0x41,0x84,0xe2,0x18,0x30,0x43,0xe2,0x00,0x30,0x84,0xe5,0x09,0xec,0xa0,0xe3,
 0x13,0xe1,0x8e,0xe2,0xb0,0x30,0xde,0xe1,0x80,0x30,0xc3,0xe3,0x03,0x38,0xa0,0xe1,
 0xa1,0x0e,0xa0,0xe3,0x23,0x38,0xa0,0xe1,0xc1,0x1e,0xa0,0xe3,0xb0,0x30,0xce,0xe1,
 0x00,0xc0,0xa0,0xe1,0x23,0x11,0x81,0xe2,0x04,0x30,0xa0,0xe3,0xb0,0x30,0xc1,0xe1,
 0x1b,0x01,0x80,0xe2,0x08,0x30,0xa0,0xe3,0x23,0xc1,0x8c,0xe2,0xc1,0x2e,0xa0,0xe3,
 0xb0,0x30,0xc0,0xe1,0x03,0x51,0xa0,0xe3,0xb0,0x30,0xcc,0xe1,0x20,0x60,0xa0,0xe3,
 0x13,0x21,0x82,0xe2,0x01,0x3c,0xa0,0xe3,0x00,0x68,0x85,0xe5,0x10,0x68,0x85,0xe5,
 0xb0,0x30,0xc2,0xe1,0x00,0x30,0x94,0xe5,0x20,0x30,0xc3,0xe3,0x00,0x30,0x84,0xe5,
 0x00,0x2a,0x95,0xe5,0xa1,0x1e,0xa0,0xe3,0x03,0x30,0xa0,0xe3,0x32,0x20,0x82,0xe2,
 0x13,0x11,0x81,0xe2,0x10,0x2a,0x85,0xe5,0xb0,0x30,0xc1,0xe1,0xb0,0x30,0xde,0xe1,
 0x80,0x30,0x83,0xe3,0xb0,0x30,0xce,0xe1,0xfe,0xff,0xff,0xea,0x04,0xe0,0x2d,0xe5,
 0x00,0x08,0xa0,0xe1,0x02,0x3b,0xa0,0xe3,0x07,0x26,0xe0,0xe3,0x91,0x1e,0xa0,0xe3,
 0x23,0x31,0x83,0xe2,0x18,0x20,0x42,0xe2,0x20,0x08,0xa0,0xe1,0x23,0x11,0x81,0xe2,
 0x00,0xc0,0x93,0xe5,0x00,0x20,0x83,0xe5,0x09,0x2c,0xa0,0xe3,0xb0,0x00,0xc1,0xe1,
 0x0b,0x21,0x82,0xe2,0xb0,0x30,0xd2,0xe1,0x04,0x00,0x13,0xe3,0xfc,0xff,0xff,0x1a,
 0x26,0x27,0xa0,0xe3,0x96,0x2c,0x82,0xe2,0x02,0x3b,0xa0,0xe3,0x91,0x1e,0xa0,0xe3,
 0x23,0x31,0x83,0xe2,0x80,0x20,0x82,0xe2,0x2b,0x11,0x81,0xe2,0x00,0xc0,0x83,0xe5,
 0x01,0x00,0x00,0xea,0x01,0x20,0x52,0xe2,0x02,0x00,0x00,0x0a,0xb0,0x30,0xd1,0xe1,
 0x00,0x00,0x53,0xe1,0xfa,0xff,0xff,0x1a,0x91,0x3e,0xa0,0xe3,0x2b,0x31,0x83,0xe2,
 0xb0,0x10,0xd3,0xe1,0x01,0x18,0xa0,0xe1,0x21,0x0c,0xa0,0xe1,0x21,0x39,0xa0,0xe1,
 0x00,0x02,0x60,0xe0,0x00,0x02,0x60,0xe0,0x3f,0x30,0x03,0xe2,0x21,0x18,0xa0,0xe1,
 0x02,0x30,0x83,0xe2,0x80,0x07,0xa0,0xe1,0x03,0x10,0x01,0xe2,0x13,0x11,0xa0,0xe1,
 0xe1,0x07,0x80,0xe2,0x3a,0x02,0x00,0xeb,0x04,0xf0,0x9d,0xe4,0x00,0xc0,0x90,0xe5,
 0x00,0x00,0x5c,0xe3,0x1e,0xff,0x2f,0x01,0x0c,0x10,0x80,0xe2,0x01,0x20,0xa0,0xe1,
 0x0b,0x00,0x00,0xea,0x04,0x00,0x53,0xe3,0x04,0x30,0x90,0x05,0x00,0x30,0x8c,0x05,
 0x02,0x00,0x00,0x0a,0x01,0x00,0x53,0xe3,0x04,0x30,0x90,0x05,0x00,0x30,0xcc,0x05,
 0x0c,0xc0,0x91,0xe4,0x00,0x00,0x5c,0xe3,0x02,0x00,0xa0,0xe1,0x0c,0x20,0x82,0xe2,
 0x09,0x00,0x00,0x0a,0x08,0x30,0x90,0xe5,0x02,0x00,0x53,0xe3,0xf0,0xff,0xff,0x1a,
 0xb4,0x00,0xd0,0xe1,0xb0,0x00,0xcc,0xe1,0x0c,0xc0,0x91,0xe4,0x00,0x00,0x5c,0xe3,
 0x02,0x00,0xa0,0xe1,0x0c,0x20,0x82,0xe2,0xf5,0xff,0xff,0x1a,0x1e,0xff,0x2f,0xe1,
 0x04,0xe0,0x2d,0xe5,0x91,0x3e,0xa0,0xe3,0x0b,0x31,0x83,0xe2,0xb0,0x10,0xd3,0xe1,
 0x01,0x18,0xa0,0xe1,0x21,0x0c,0xa0,0xe1,0x21,0x39,0xa0,0xe1,0x00,0x02,0x60,0xe0,
 0x00,0x02,0x60,0xe0,0x3f,0x30,0x03,0xe2,0x21,0x18,0xa0,0xe1,0x02,0x30,0x83,0xe2,
 0x80,0x07,0xa0,0xe1,0x03,0x10,0x01,0xe2,0x13,0x11,0xa0,0xe1,0xe1,0x07,0x80,0xe2,
 0x0b,0x02,0x00,0xeb,0x04,0xf0,0x9d,0xe4,0x04,0xe0,0x2d,0xe5,0x91,0x3e,0xa0,0xe3,
 0x1b,0x31,0x83,0xe2,0xb0,0x10,0xd3,0xe1,0x01,0x18,0xa0,0xe1,0x21,0x0c,0xa0,0xe1,
 0x21,0x39,0xa0,0xe1,0x00,0x02,0x60,0xe0,0x00,0x02,0x60,0xe0,0x3f,0x30,0x03,0xe2,
 0x21,0x18,0xa0,0xe1,0x02,0x30,0x83,0xe2,0x80,0x07,0xa0,0xe1,0x03,0x10,0x01,0xe2,
 0x13,0x11,0xa0,0xe1,0xe1,0x07,0x80,0xe2,0xf9,0x01,0x00,0xeb,0x04,0xf0,0x9d,0xe4,
 0x04,0xe0,0x2d,0xe5,0x91,0x3e,0xa0,0xe3,0x2b,0x31,0x83,0xe2,0xb0,0x10,0xd3,0xe1,
 0x01,0x18,0xa0,0xe1,0x21,0x0c,0xa0,0xe1,0x21,0x39,0xa0,0xe1,0x00,0x02,0x60,0xe0,
 0x00,0x02,0x60,0xe0,0x3f,0x30,0x03,0xe2,0x21,0x18,0xa0,0xe1,0x02,0x30,0x83,0xe2,
 0x80,0x07,0xa0,0xe1,0x03,0x10,0x01,0xe2,0x13,0x11,0xa0,0xe1,0xe1,0x07,0x80,0xe2,
 0xe7,0x01,0x00,0xeb,0x04,0xf0,0x9d,0xe4,0x04,0xe0,0x2d,0xe5,0x00,0x08,0xa0,0xe1,
 0x02,0x3b,0xa0,0xe3,0x07,0x26,0xe0,0xe3,0x03,0x11,0xa0,0xe3,0x23,0x31,0x83,0xe2,
 0x18,0x20,0x42,0xe2,0x20,0x08,0xa0,0xe1,0x91,0x1e,0x81,0xe2,0x00,0xc0,0x93,0xe5,
 0x00,0x20,0x83,0xe5,0x09,0x2c,0xa0,0xe3,0xb0,0x00,0xc1,0xe1,0x0b,0x21,0x82,0xe2,
 0xb0,0x30,0xd2,0xe1,0x01,0x00,0x13,0xe3,0xfc,0xff,0xff,0x1a,0x26,0x27,0xa0,0xe3,
 0x96,0x2c,0x82,0xe2,0x02,0x3b,0xa0,0xe3,0x91,0x1e,0xa0,0xe3,0x23,0x31,0x83,0xe2,
 0x80,0x20,0x82,0xe2,0x0b,0x11,0x81,0xe2,0x00,0xc0,0x83,0xe5,0x01,0x00,0x00,0xea,
 0x01,0x20,0x52,0xe2,0x02,0x00,0x00,0x0a,0xb0,0x30,0xd1,0xe1,0x00,0x00,0x53,0xe1,
 0xfa,0xff,0xff,0x1a,0x91,0x3e,0xa0,0xe3,0x0b,0x31,0x83,0xe2,0xb0,0x10,0xd3,0xe1,
 0x01,0x18,0xa0,0xe1,0x21,0x0c,0xa0,0xe1,0x21,0x39,0xa0,0xe1,0x00,0x02,0x60,0xe0,
 0x00,0x02,0x60,0xe0,0x3f,0x30,0x03,0xe2,0x21,0x18,0xa0,0xe1,0x02,0x30,0x83,0xe2,
 0x80,0x07,0xa0,0xe1,0x03,0x10,0x01,0xe2,0x13,0x11,0xa0,0xe1,0xe1,0x07,0x80,0xe2,
 0xb7,0x01,0x00,0xeb,0x04,0xf0,0x9d,0xe4,0x04,0xe0,0x2d,0xe5,0x00,0x08,0xa0,0xe1,
 0x02,0x3b,0xa0,0xe3,0x07,0x26,0xe0,0xe3,0x91,0x1e,0xa0,0xe3,0x23,0x31,0x83,0xe2,
 0x18,0x20,0x42,0xe2,0x20,0x08,0xa0,0xe1,0x13,0x11,0x81,0xe2,0x00,0xc0,0x93,0xe5,
 0x00,0x20,0x83,0xe5,0x09,0x2c,0xa0,0xe3,0xb0,0x00,0xc1,0xe1,0x0b,0x21,0x82,0xe2,
 0xb0,0x30,0xd2,0xe1,0x02,0x00,0x13,0xe3,0xfc,0xff,0xff,0x1a,0x26,0x27,0xa0,0xe3,
 0x96,0x2c,0x82,0xe2,0x02,0x3b,0xa0,0xe3,0x91,0x1e,0xa0,0xe3,0x23,0x31,0x83,0xe2,
 0x80,0x20,0x82,0xe2,0x1b,0x11,0x81,0xe2,0x00,0xc0,0x83,0xe5,0x01,0x00,0x00,0xea,
 0x01,0x20,0x52,0xe2,0x02,0x00,0x00,0x0a,0xb0,0x30,0xd1,0xe1,0x00,0x00,0x53,0xe1,
 0xfa,0xff,0xff,0x1a,0x91,0x3e,0xa0,0xe3,0x1b,0x31,0x83,0xe2,0xb0,0x10,0xd3,0xe1,
 0x01,0x18,0xa0,0xe1,0x21,0x0c,0xa0,0xe1,0x21,0x39,0xa0,0xe1,0x00,0x02,0x60,0xe0,
 0x00,0x02,0x60,0xe0,0x3f,0x30,0x03,0xe2,0x21,0x18,0xa0,0xe1,0x02,0x30,0x83,0xe2,
 0x80,0x07,0xa0,0xe1,0x03,0x10,0x01,0xe2,0x13,0x11,0xa0,0xe1,0xe1,0x07,0x80,0xe2,
 0x87,0x01,0x00,0xeb,0x04,0xf0,0x9d,0xe4,0x34,0x02,0x9f,0xe5,0x00,0x10,0x90,0xe5,
 0x02,0x2b,0xa0,0xe3,0x07,0x36,0xe0,0xe3,0x23,0x21,0x82,0xe2,0x18,0x30,0x43,0xe2,
 0x00,0x00,0x51,0xe3,0x10,0x40,0x2d,0xe9,0x00,0x30,0x82,0xe5,0x16,0x00,0x00,0x0a,
 0x00,0x20,0xa0,0xe1,0x0c,0x00,0x80,0xe2,0x0a,0x00,0x00,0xea,0x04,0x00,0x53,0xe3,
 0x04,0x30,0x92,0x05,0x00,0x30,0x81,0x05,0x02,0x00,0x00,0x0a,0x01,0x00,0x53,0xe3,
 0x04,0x30,0x92,0x05,0x00,0x30,0xc1,0x05,0x0c,0x10,0x90,0xe4,0x00,0x00,0x51,0xe3,
 0x0c,0x20,0x82,0xe2,0x08,0x00,0x00,0x0a,0x08,0x30,0x92,0xe5,0x02,0x00,0x53,0xe3,
 0xf1,0xff,0xff,0x1a,0xb4,0x30,0xd2,0xe1,0xb0,0x30,0xc1,0xe1,0x0c,0x10,0x90,0xe4,
 0x00,0x00,0x51,0xe3,0x0c,0x20,0x82,0xe2,0xf6,0xff,0xff,0x1a,0x02,0x2b,0xa0,0xe3,
 0x07,0x16,0xe0,0xe3,0x03,0x01,0xa0,0xe3,0x49,0x3c,0xa0,0xe3,0x23,0x21,0x82,0xe2,
 0x18,0x10,0x41,0xe2,0x91,0x0e,0x80,0xe2,0x04,0x30,0x83,0xe2,0x00,0xc0,0x92,0xe5,
 0x00,0x10,0x82,0xe5,0x09,0x2c,0xa0,0xe3,0xb0,0x30,0xc0,0xe1,0x0b,0x21,0x82,0xe2,
 0xb0,0x30,0xd2,0xe1,0x01,0x00,0x13,0xe3,0xfc,0xff,0xff,0x1a,0x26,0x27,0xa0,0xe3,
 0x96,0x2c,0x82,0xe2,0x02,0x3b,0xa0,0xe3,0x91,0x0e,0xa0,0xe3,0x49,0x1c,0xa0,0xe3,
 0x23,0x31,0x83,0xe2,0x80,0x20,0x82,0xe2,0x0b,0x01,0x80,0xe2,0x04,0x10,0x81,0xe2,
 0x00,0xc0,0x83,0xe5,0x01,0x00,0x00,0xea,0x01,0x20,0x52,0xe2,0x02,0x00,0x00,0x0a,
 0xb0,0x30,0xd0,0xe1,0x01,0x00,0x53,0xe1,0xfa,0xff,0xff,0x1a,0x91,0x2e,0xa0,0xe3,
 0x02,0xc0,0xa0,0xe1,0x02,0x3b,0xa0,0xe3,0x07,0x06,0xe0,0xe3,0x06,0x1a,0xa0,0xe3,
 0x23,0x31,0x83,0xe2,0x0b,0x21,0x82,0xe2,0x18,0x00,0x40,0xe2,0x13,0xc1,0x8c,0xe2,
 0x02,0x10,0x81,0xe2,0xb0,0xe0,0xd2,0xe1,0x00,0x40,0x93,0xe5,0x4b,0x21,0x42,0xe2,
 0x00,0x00,0x83,0xe5,0xb0,0x10,0xcc,0xe1,0x0b,0x21,0x82,0xe2,0xb0,0x30,0xd2,0xe1,
 0x02,0xe0,0x13,0xe2,0xfc,0xff,0xff,0x1a,0x26,0xc7,0xa0,0xe3,0x96,0xcc,0x8c,0xe2,
 0x02,0x3b,0xa0,0xe3,0x91,0x0e,0xa0,0xe3,0x06,0x1a,0xa0,0xe3,0x23,0x31,0x83,0xe2,
 0x0e,0x20,0xa0,0xe1,0x1b,0x01,0x80,0xe2,0x02,0x10,0x81,0xe2,0x80,0xc0,0x8c,0xe2,
 0x00,0x40,0x83,0xe5,0x01,0x00,0x00,0xea,0x0c,0x00,0x52,0xe1,0x03,0x00,0x00,0x0a,
 0xb0,0x30,0xd0,0xe1,0x01,0x00,0x53,0xe1,0x01,0x20,0x82,0xe2,0xf9,0xff,0xff,0x1a,
 0x91,0x2e,0xa0,0xe3,0x02,0xc0,0xa0,0xe1,0x02,0x3b,0xa0,0xe3,0x07,0x06,0xe0,0xe3,
 0x67,0x1c,0xe0,0xe3,0x23,0x31,0x83,0xe2,0x1b,0x21,0x82,0xe2,0x18,0x00,0x40,0xe2,
 0x23,0xc1,0x8c,0xe2,0xfd,0x10,0x41,0xe2,0xb0,0xe0,0xd2,0xe1,0x00,0x40,0x93,0xe5,
 0x5b,0x21,0x42,0xe2,0x00,0x00,0x83,0xe5,0xb0,0x10,0xcc,0xe1,0x0b,0x21,0x82,0xe2,
 0xb0,0x30,0xd2,0xe1,0x04,0xe0,0x13,0xe2,0xfc,0xff,0xff,0x1a,0x26,0xc7,0xa0,0xe3,
 0x96,0xcc,0x8c,0xe2,0x02,0x3b,0xa0,0xe3,0x91,0x0e,0xa0,0xe3,0x26,0x1b,0xa0,0xe3,
 0x23,0x31,0x83,0xe2,0x0e,0x20,0xa0,0xe1,0x2b,0x01,0x80,0xe2,0x02,0x10,0x81,0xe2,
 0x80,0xc0,0x8c,0xe2,0x00,0x40,0x83,0xe5,0x01,0x00,0x00,0xea,0x0c,0x00,0x52,0xe1,
 0x03,0x00,0x00,0x0a,0xb0,0x30,0xd0,0xe1,0x01,0x00,0x53,0xe1,0x01,0x20,0x82,0xe2,
 0xf9,0xff,0xff,0x1a,0x91,0x3e,0xa0,0xe3,0x2b,0x31,0x83,0xe2,0xb0,0x20,0xd3,0xe1,
 0x10,0x80,0xbd,0xe8,0x48,0x0c,0x80,0x03,0x0a,0x2b,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0x2b,0x31,0x82,0xe2,0xb0,0x10,0xc3,0xe1,0x0b,0xc1,0x82,0xe2,0x33,0x01,0x82,0xe2,
 0x03,0x31,0xa0,0xe3,0x3b,0x21,0x82,0xe2,0xb0,0x10,0xc0,0xe1,0x0a,0x3b,0x83,0xe2,
 0xb0,0x10,0xc2,0xe1,0x04,0x20,0xa0,0xe3,0xb0,0x20,0xc3,0xe1,0x01,0x30,0xa0,0xe3,
 0xb0,0x30,0xcc,0xe1,0x1e,0xff,0x2f,0xe1,0x04,0xe0,0x2d,0xe5,0x41,0x2d,0xa0,0xe3,
 0x8b,0x21,0x82,0xe2,0xb0,0x30,0xd2,0xe1,0x0c,0x30,0xc3,0xe3,0x03,0x38,0xa0,0xe1,
 0x23,0x38,0xa0,0xe1,0xb0,0x30,0xc2,0xe1,0x41,0x1d,0xa0,0xe3,0x9b,0x11,0x81,0xe2,
 0xb0,0x30,0xd1,0xe1,0x10,0x30,0xc3,0xe3,0x03,0x38,0xa0,0xe1,0x23,0x38,0xa0,0xe1,
 0xb0,0x30,0xc1,0xe1,0x41,0x0d,0xa0,0xe3,0xbb,0x01,0x80,0xe2,0xb0,0x30,0xd0,0xe1,
 0x06,0x30,0xc3,0xe3,0x03,0x38,0xa0,0xe1,0x23,0x38,0xa0,0xe1,0xb0,0x30,0xc0,0xe1,
 0xb0,0x20,0xd0,0xe1,0x0a,0x3b,0xa0,0xe3,0x08,0x20,0x82,0xe3,0xb0,0x20,0xc0,0xe1,
 0x03,0x10,0xa0,0xe1,0x00,0x00,0xe0,0xe3,0x03,0xc0,0xa0,0xe1,0x03,0xe0,0xa0,0xe1,
 0x2b,0x31,0x83,0xe2,0xb0,0x00,0xc3,0xe1,0x33,0x11,0x81,0xe2,0x00,0x30,0xa0,0xe3,
 0x3b,0xc1,0x8c,0xe2,0x0b,0xe1,0x8e,0xe2,0xb0,0x00,0xc1,0xe1,0x03,0x21,0xa0,0xe3,
 0xb0,0x00,0xcc,0xe1,0xb0,0x30,0xce,0xe1,0x0a,0x2b,0x82,0xe2,0xb0,0x30,0xd2,0xe1,
 0x01,0x30,0x83,0xe3,0xb0,0x30,0xc2,0xe1,0x3d,0x29,0xa0,0xe3,0x09,0x2d,0x82,0xe2,
 0x00,0x30,0xa0,0xe3,0x01,0x30,0x83,0xe2,0x02,0x00,0x53,0xe1,0xfc,0xff,0xff,0x1a,
 0x41,0x2d,0xa0,0xe3,0x8b,0x21,0x82,0xe2,0xb0,0x30,0xd2,0xe1,0x0c,0x30,0x83,0xe3,
 0xb0,0x30,0xc2,0xe1,0x41,0x1d,0xa0,0xe3,0x9b,0x11,0x81,0xe2,0xb0,0x30,0xd1,0xe1,
 0x10,0x30,0x83,0xe3,0xb0,0x30,0xc1,0xe1,0x41,0x0d,0xa0,0xe3,0xbb,0x01,0x80,0xe2,
 0xb0,0x30,0xd0,0xe1,0x08,0x30,0xc3,0xe3,0x03,0x38,0xa0,0xe1,0x23,0x38,0xa0,0xe1,
 0xb0,0x30,0xc0,0xe1,0xb0,0x20,0xd0,0xe1,0x06,0x20,0x82,0xe3,0xb0,0x20,0xc0,0xe1,
 0x04,0xf0,0x9d,0xe4,0x04,0xe0,0x2d,0xe5,0x29,0x3c,0xa0,0xe3,0x00,0x28,0xa0,0xe1,
 0x22,0x28,0xa0,0xe1,0x20,0x08,0xa0,0xe1,0x53,0xe1,0x83,0xe2,0x3b,0x11,0x83,0xe2,
 0x43,0xc1,0x83,0xe2,0x4b,0x31,0x83,0xe2,0xb0,0x20,0xc1,0xe1,0xb0,0x00,0xcc,0xe1,
 0xb0,0x20,0xc3,0xe1,0xb0,0x00,0xce,0xe1,0x04,0xf0,0x9d,0xe4,0xa5,0x3d,0xa0,0xe3,
 0x63,0x31,0x83,0xe2,0x00,0x20,0xa0,0xe3,0x04,0xe0,0x2d,0xe5,0xa5,0xed,0xa0,0xe3,
 0xb0,0x20,0xc3,0xe1,0x6b,0xe1,0x8e,0xe2,0x00,0xc0,0xa0,0xe3,0xb2,0x30,0xd0,0xe0,
 0x03,0x38,0xa0,0xe1,0xa3,0x16,0xa0,0xe1,0xa3,0x25,0xa0,0xe1,0x3f,0x2b,0x02,0xe2,
 0xf8,0x10,0x01,0xe2,0x01,0x20,0x82,0xe0,0x02,0x28,0xa0,0xe1,0x01,0xc0,0x8c,0xe2,
 0xa3,0x3d,0xa0,0xe1,0x83,0x31,0xa0,0xe1,0x22,0x28,0xa0,0xe1,0x01,0x0c,0x5c,0xe3,
 0xb0,0x20,0xce,0xe1,0xb0,0x30,0xce,0xe1,0xef,0xff,0xff,0x1a,0x04,0xf0,0x9d,0xe4,
 0xf0,0x45,0x2d,0xe9,0x0a,0x6b,0xa0,0xe3,0x06,0x70,0xa0,0xe1,0x06,0x80,0xa0,0xe1,
 0x03,0x51,0xa0,0xe3,0x06,0x50,0x85,0xe0,0x00,0x30,0xa0,0xe3,0x06,0xa0,0xa0,0xe1,
 0x33,0x71,0x87,0xe2,0x2b,0x61,0x86,0xe2,0x3b,0x81,0x88,0xe2,0xb0,0x30,0xc6,0xe1,
 0xb0,0x30,0xc7,0xe1,0xb0,0x30,0xc8,0xe1,0x04,0x30,0xa0,0xe3,0xb0,0x30,0xc5,0xe1,
 0x0b,0xa1,0x8a,0xe2,0x01,0x30,0xa0,0xe3,0x00,0x40,0xa0,0xe1,0xb0,0x30,0xca,0xe1,
 0x2c,0x01,0x9f,0xe5,0x28,0xfe,0xff,0xeb,0x29,0x3c,0xa0,0xe3,0x04,0xc8,0xa0,0xe1,
 0x03,0x20,0xa0,0xe1,0x03,0x10,0xa0,0xe1,0x03,0x00,0xa0,0xe1,0x2c,0xc8,0xa0,0xe1,
 0x24,0x48,0xa0,0xe1,0x3b,0x31,0x83,0xe2,0x43,0x21,0x82,0xe2,0x4b,0x11,0x81,0xe2,
 0x53,0x01,0x80,0xe2,0xb0,0xc0,0xc3,0xe1,0xb0,0x40,0xc2,0xe1,0xb0,0xc0,0xc1,0xe1,
 0x41,0x2d,0xa0,0xe3,0xb0,0x40,0xc0,0xe1,0x8b,0x21,0x82,0xe2,0xb0,0x30,0xd2,0xe1,
 0x0c,0x30,0xc3,0xe3,0x03,0x38,0xa0,0xe1,0x23,0x38,0xa0,0xe1,0xb0,0x30,0xc2,0xe1,
 0x41,0x1d,0xa0,0xe3,0x9b,0x11,0x81,0xe2,0xb0,0x30,0xd1,0xe1,0x10,0x30,0xc3,0xe3,
 0x03,0x38,0xa0,0xe1,0x23,0x38,0xa0,0xe1,0xb0,0x30,0xc1,0xe1,0x41,0x0d,0xa0,0xe3,
 0xbb,0x01,0x80,0xe2,0xb0,0x30,0xd0,0xe1,0x06,0x30,0xc3,0xe3,0x03,0x38,0xa0,0xe1,
 0x23,0x38,0xa0,0xe1,0xb0,0x30,0xc0,0xe1,0xb0,0x20,0xd0,0xe1,0x00,0x30,0xe0,0xe3,
 0x08,0x20,0x82,0xe3,0xb0,0x20,0xc0,0xe1,0xb0,0x30,0xc6,0xe1,0xb0,0x30,0xc7,0xe1,
 0xb0,0x30,0xc8,0xe1,0x00,0x30,0xa0,0xe3,0xb0,0x30,0xca,0xe1,0xb0,0x30,0xd5,0xe1,
 0x3d,0x29,0xa0,0xe3,0x01,0x30,0x83,0xe3,0xb0,0x30,0xc5,0xe1,0x09,0x2d,0x82,0xe2,
 0x00,0x30,0xa0,0xe3,0x01,0x30,0x83,0xe2,0x02,0x00,0x53,0xe1,0xfc,0xff,0xff,0x1a,
 0x41,0x2d,0xa0,0xe3,0x8b,0x21,0x82,0xe2,0xb0,0x30,0xd2,0xe1,0x0c,0x30,0x83,0xe3,
 0xb0,0x30,0xc2,0xe1,0x41,0x1d,0xa0,0xe3,0x9b,0x11,0x81,0xe2,0xb0,0x30,0xd1,0xe1,
 0x10,0x30,0x83,0xe3,0xb0,0x30,0xc1,0xe1,0x41,0x0d,0xa0,0xe3,0xbb,0x01,0x80,0xe2,
 0xb0,0x30,0xd0,0xe1,0x08,0x30,0xc3,0xe3,0x03,0x38,0xa0,0xe1,0x23,0x38,0xa0,0xe1,
 0xb0,0x30,0xc0,0xe1,0xb0,0x20,0xd0,0xe1,0x06,0x20,0x82,0xe3,0xb0,0x20,0xc0,0xe1,
 0xf0,0x85,0xbd,0xe8,0x4c,0x26,0x80,0x03,0x04,0xe0,0x2d,0xe5,0x8d,0xfe,0xff,0xeb,
 0x0e,0x35,0xa0,0xe3,0xff,0x3d,0x83,0xe2,0x3c,0xc0,0x93,0xe5,0x00,0x00,0x5c,0xe3,
 0x0b,0x00,0x00,0x0a,0x3f,0x03,0xa0,0xe3,0x02,0x05,0x80,0xe2,0x0e,0x15,0xa0,0xe3,
 0x01,0x19,0x81,0xe2,0x01,0x09,0x80,0xe2,0x00,0x20,0xa0,0xe3,0x00,0x30,0x91,0xe5,
 0x01,0x20,0x82,0xe2,0x02,0x00,0x5c,0xe1,0x00,0x30,0x81,0xe7,0x04,0x10,0x81,0xe2,
 0xf9,0xff,0xff,0x1a,0x02,0x39,0xa0,0xe3,0x0f,0xe0,0xa0,0xe1,0x13,0xff,0x2f,0xe1,
 0x04,0xf0,0x9d,0xe4,0x01,0x20,0x51,0xe2,0x1e,0xff,0x2f,0x01,0x36,0x00,0x00,0x3a,
 0x01,0x00,0x50,0xe1,0x22,0x00,0x00,0x9a,0x02,0x00,0x11,0xe1,0x23,0x00,0x00,0x0a,
 0x0e,0x02,0x11,0xe3,0x81,0x11,0xa0,0x01,0x08,0x30,0xa0,0x03,0x01,0x30,0xa0,0x13,
 0x01,0x02,0x51,0xe3,0x00,0x00,0x51,0x31,0x01,0x12,0xa0,0x31,0x03,0x32,0xa0,0x31,
 0xfa,0xff,0xff,0x3a,0x02,0x01,0x51,0xe3,0x00,0x00,0x51,0x31,0x81,0x10,0xa0,0x31,
 0x83,0x30,0xa0,0x31,0xfa,0xff,0xff,0x3a,0x00,0x20,0xa0,0xe3,0x01,0x00,0x50,0xe1,
 0x01,0x00,0x40,0x20,0x03,0x20,0x82,0x21,0xa1,0x00,0x50,0xe1,0xa1,0x00,0x40,0x20,
 0xa3,0x20,0x82,0x21,0x21,0x01,0x50,0xe1,0x21,0x01,0x40,0x20,0x23,0x21,0x82,0x21,
 0xa1,0x01,0x50,0xe1,0xa1,0x01,0x40,0x20,0xa3,0x21,0x82,0x21,0x00,0x00,0x50,0xe3,
 0x23,0x32,0xb0,0x11,0x21,0x12,0xa0,0x11,0xef,0xff,0xff,0x1a,0x02,0x00,0xa0,0xe1,
 0x1e,0xff,0x2f,0xe1,0x01,0x00,0xa0,0x03,0x00,0x00,0xa0,0x13,0x1e,0xff,0x2f,0xe1,
 0x01,0x08,0x51,0xe3,0x21,0x18,0xa0,0x21,0x10,0x20,0xa0,0x23,0x00,0x20,0xa0,0x33,
 0x01,0x0c,0x51,0xe3,0x21,0x14,0xa0,0x21,0x08,0x20,0x82,0x22,0x10,0x00,0x51,0xe3,
 0x21,0x12,0xa0,0x21,0x04,0x20,0x82,0x22,0x04,0x00,0x51,0xe3,0x03,0x20,0x82,0x82,
 0xa1,0x20,0x82,0x90,0x30,0x02,0xa0,0xe1,0x1e,0xff,0x2f,0xe1,0x04,0xe0,0x2d,0xe5,
 0x07,0x00,0x00,0xeb,0x00,0x00,0xa0,0xe3,0x04,0xf0,0x9d,0xe4,0x03,0x40,0x2d,0xe9,
 0xbf,0xff,0xff,0xeb,0x06,0x40,0xbd,0xe8,0x92,0x00,0x03,0xe0,0x03,0x10,0x41,0xe0,
 0x1e,0xff,0x2f,0xe1,0x1e,0xff,0x2f,0xe1,0x00,0x3a,0x00,0xc0,0x7f,0x1c,0x00,0x00,
 0x02,0x00,0x00,0x00,0x02,0x3a,0x00,0xc0,0xdf,0xff,0x00,0x00,0x02,0x00,0x00,0x00,
 0x04,0x3a,0x00,0xc0,0xdf,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x3a,0x00,0xc0,
 0xdf,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0x08,0x3a,0x00,0xc0,0xdf,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0x0a,0x3a,0x00,0xc0,0xff,0xf1,0x00,0x00,0x02,0x00,0x00,0x00,
 0x0e,0x3a,0x00,0xc0,0x55,0x55,0x00,0x00,0x02,0x00,0x00,0x00,0x10,0x3a,0x00,0xc0,
 0x40,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x12,0x3a,0x00,0xc0,0x42,0x08,0x00,0x00,
 0x02,0x00,0x00,0x00,0x14,0x3a,0x00,0xc0,0x63,0x0c,0x00,0x00,0x02,0x00,0x00,0x00,
 0x16,0x3a,0x00,0xc0,0x84,0x10,0x00,0x00,0x02,0x00,0x00,0x00,0x18,0x3a,0x00,0xc0,
 0xa5,0x14,0x00,0x00,0x02,0x00,0x00,0x00,0x1a,0x3a,0x00,0xc0,0x85,0x0c,0x00,0x00,
 0x02,0x00,0x00,0x00,0x1c,0x3a,0x00,0xc0,0x02,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x20,0x3a,0x00,0xc0,0x08,0x40,0x00,0x00,0x02,0x00,0x00,0x00,0x22,0x3a,0x00,0xc0,
 0xe7,0x1f,0x00,0x00,0x02,0x00,0x00,0x00,0x24,0x3a,0x00,0xc0,0x01,0xf0,0x00,0x00,
 0x02,0x00,0x00,0x00,0x26,0x3a,0x00,0xc0,0x84,0x10,0x00,0x00,0x02,0x00,0x00,0x00,
 0x28,0x3a,0x00,0xc0,0x84,0x10,0x00,0x00,0x02,0x00,0x00,0x00,0x2a,0x3a,0x00,0xc0,
 0x84,0x10,0x00,0x00,0x02,0x00,0x00,0x00,0x2c,0x3a,0x00,0xc0,0x84,0x10,0x00,0x00,
 0x02,0x00,0x00,0x00,0x2e,0x3a,0x00,0xc0,0x84,0x10,0x00,0x00,0x02,0x00,0x00,0x00,
 0x30,0x3a,0x00,0xc0,0x84,0x10,0x00,0x00,0x02,0x00,0x00,0x00,0x32,0x3a,0x00,0xc0,
 0x84,0x10,0x00,0x00,0x02,0x00,0x00,0x00,0x34,0x3a,0x00,0xc0,0x22,0x42,0x00,0x00,
 0x02,0x00,0x00,0x00,0x36,0x3a,0x00,0xc0,0x22,0x02,0x00,0x00,0x02,0x00,0x00,0x00,
 0x38,0x3a,0x00,0xc0,0x22,0x22,0x00,0x00,0x02,0x00,0x00,0x00,0x3a,0x3a,0x00,0xc0,
 0x83,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x3c,0x3a,0x00,0xc0,0xf8,0x07,0x00,0x00,
 0x02,0x00,0x00,0x00,0x3e,0x3a,0x00,0xc0,0xcc,0xff,0x00,0x00,0x02,0x00,0x00,0x00,
 0x40,0x3a,0x00,0xc0,0xf0,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0x42,0x3a,0x00,0xc0,
 0xcf,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0x44,0x3a,0x00,0xc0,0x10,0x02,0x00,0x00,
 0x02,0x00,0x00,0x00,0x00,0x38,0x00,0xc0,0x1a,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x02,0x38,0x00,0xc0,0x77,0x77,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x38,0x00,0xc0,
 0xf2,0x07,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x38,0x00,0xc0,0xff,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x08,0x38,0x00,0xc0,0x00,0x01,0x00,0x00,0x02,0x00,0x00,0x00,
 0x00,0x09,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x09,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x09,0x00,0xc0,0xfe,0xab,0x00,0x00,
 0x02,0x00,0x00,0x00,0x08,0x09,0x00,0xc0,0x14,0x01,0x00,0x00,0x02,0x00,0x00,0x00,
 0x0a,0x09,0x00,0xc0,0x01,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0c,0x09,0x00,0xc0,
 0xf1,0x0d,0x00,0x00,0x02,0x00,0x00,0x00,0x1c,0x09,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x1e,0x09,0x00,0xc0,0x70,0x41,0x00,0x00,0x02,0x00,0x00,0x00,
 0x20,0x09,0x00,0xc0,0x02,0x05,0x00,0x00,0x02,0x00,0x00,0x00,0x22,0x09,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x24,0x09,0x00,0xc0,0x00,0x4a,0x00,0x00,
 0x02,0x00,0x00,0x00,0x26,0x09,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x28,0x09,0x00,0xc0,0xc9,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x2a,0x09,0x00,0xc0,
 0x0f,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x2c,0x09,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x2e,0x09,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x08,0x0c,0x00,0xc0,0xfe,0x03,0x00,0x00,0x02,0x00,0x00,0x00,0x0a,0x0c,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0c,0x0c,0x00,0xc0,0xff,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0x10,0x0c,0x00,0xc0,0x03,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x12,0x0c,0x00,0xc0,0x03,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x14,0x0c,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x16,0x0c,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x18,0x0c,0x00,0xc0,0x04,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x1a,0x0c,0x00,0xc0,0x64,0xa0,0x00,0x00,0x02,0x00,0x00,0x00,0x1c,0x0c,0x00,0xc0,
 0xff,0x0f,0x00,0x00,0x02,0x00,0x00,0x00,0x1e,0x0c,0x00,0xc0,0x31,0x01,0x00,0x00,
 0x02,0x00,0x00,0x00,0x20,0x0c,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x22,0x0c,0x00,0xc0,0x00,0x80,0x00,0x00,0x02,0x00,0x00,0x00,0x24,0x0c,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0c,0x08,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x04,0x00,0x00,0x00,0x04,0x0a,0x00,0xc0,0x00,0x00,0x8e,0x7a,0x04,0x00,0x00,0x00,
 0x08,0x0a,0x00,0xc0,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x0c,0x0a,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x10,0x0a,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x04,0x00,0x00,0x00,0x14,0x0a,0x00,0xc0,0x01,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x16,0x0a,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x18,0x0a,0x00,0xc0,
 0x01,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x12,0x00,0xc0,0x03,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x02,0x12,0x00,0xc0,0x85,0x03,0x00,0x00,0x02,0x00,0x00,0x00,
 0x04,0x12,0x00,0xc0,0x01,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x12,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x08,0x12,0x00,0xc0,0x06,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x0a,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x0c,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0e,0x12,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x14,0x12,0x00,0xc0,0x07,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x16,0x12,0x00,0xc0,0x1f,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x20,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x22,0x12,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x24,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x26,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x40,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x42,0x12,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x44,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x46,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x60,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x62,0x12,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x64,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x66,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x80,0x12,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x82,0x12,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x40,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,
 0x02,0x00,0x00,0x00,0x20,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,
 0x42,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x22,0x10,0x00,0xc0,
 0x50,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x44,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x24,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x46,0x10,0x00,0xc0,0x2a,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x26,0x10,0x00,0xc0,
 0x8a,0x09,0x00,0x00,0x02,0x00,0x00,0x00,0x48,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,
 0x02,0x00,0x00,0x00,0x28,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,
 0x4a,0x10,0x00,0xc0,0x0a,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x2a,0x10,0x00,0xc0,
 0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x4c,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,
 0x02,0x00,0x00,0x00,0x2c,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,
 0x4e,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x2e,0x10,0x00,0xc0,
 0x55,0x09,0x00,0x00,0x02,0x00,0x00,0x00,0x50,0x10,0x00,0xc0,0xaa,0x8a,0x00,0x00,
 0x02,0x00,0x00,0x00,0x30,0x10,0x00,0xc0,0x0a,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,
 0x52,0x10,0x00,0xc0,0xaa,0x06,0x00,0x00,0x02,0x00,0x00,0x00,0x32,0x10,0x00,0xc0,
 0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x54,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x34,0x10,0x00,0xc0,0x08,0x08,0x00,0x00,0x02,0x00,0x00,0x00,
 0x56,0x10,0x00,0xc0,0x2a,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x36,0x10,0x00,0xc0,
 0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x58,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x38,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x5a,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x3a,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x5c,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x3c,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x60,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x62,0x10,0x00,0xc0,
 0x0c,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x64,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x66,0x10,0x00,0xc0,0x10,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x68,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x6a,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x6c,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x6e,0x10,0x00,0xc0,0x0e,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x70,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x72,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x74,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x76,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x78,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x7a,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x7c,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x80,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x82,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x84,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x86,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x88,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x8a,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x8c,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x8e,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x90,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x92,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x94,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x96,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x98,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x9a,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x9c,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xa0,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xa2,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xa4,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xa6,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xa8,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xaa,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xac,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xae,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xb0,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xb2,0x10,0x00,0xc0,0x00,0x30,0x00,0x00,0x02,0x00,0x00,0x00,
 0xb4,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xb6,0x10,0x00,0xc0,
 0x80,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0xb8,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xba,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xbc,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xc0,0x10,0x00,0xc0,
 0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0xc2,0x10,0x00,0xc0,0x0f,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0xc4,0x10,0x00,0xc0,0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,
 0xc6,0x10,0x00,0xc0,0xf8,0x3f,0x00,0x00,0x02,0x00,0x00,0x00,0xc8,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xca,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xcc,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xce,0x10,0x00,0xc0,0x7e,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xd0,0x10,0x00,0xc0,
 0x00,0x42,0x00,0x00,0x02,0x00,0x00,0x00,0xd2,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xd4,0x10,0x00,0xc0,0xff,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xd6,0x10,0x00,0xc0,0xff,0x7f,0x00,0x00,0x02,0x00,0x00,0x00,0xd8,0x10,0x00,0xc0,
 0xff,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0xda,0x10,0x00,0xc0,0xff,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xdc,0x10,0x00,0xc0,0x3f,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xe0,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xe2,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xe4,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xe6,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xe8,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xea,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xec,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xee,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xf0,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xf2,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xf4,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xf6,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xf8,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xfa,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xfc,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x00,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x02,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x11,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x08,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x0a,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0c,0x11,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0e,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x10,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x12,0x11,0x00,0xc0,0x37,0x0e,0x00,0x00,0x02,0x00,0x00,0x00,0x14,0x11,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x16,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x18,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x1a,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x1c,0x11,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x20,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x40,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xa0,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xa2,0x11,0x00,0xc0,
 0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0xa4,0x11,0x00,0xc0,0xff,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0xa6,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xa8,0x11,0x00,0xc0,0xff,0x03,0x00,0x00,0x02,0x00,0x00,0x00,0xaa,0x11,0x00,0xc0,
 0xff,0x03,0x00,0x00,0x02,0x00,0x00,0x00,0xac,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xae,0x11,0x00,0xc0,0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,
 0xb0,0x11,0x00,0xc0,0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0xb2,0x11,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xb4,0x11,0x00,0xc0,0xff,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0xb6,0x11,0x00,0xc0,0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,
 0xb8,0x11,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xba,0x11,0x00,0xc0,
 0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0xbc,0x11,0x00,0xc0,0xff,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0x00,0x18,0x00,0xc0,0x00,0x01,0x00,0x00,0x02,0x00,0x00,0x00,
 0x02,0x18,0x00,0xc0,0x04,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x18,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x18,0x00,0xc0,0x00,0x04,0x00,0x00,
 0x02,0x00,0x00,0x00,0x08,0x18,0x00,0xc0,0x00,0x04,0x00,0x00,0x02,0x00,0x00,0x00,
 0x0a,0x18,0x00,0xc0,0x00,0x40,0x00,0x00,0x02,0x00,0x00,0x00,0x0c,0x18,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0e,0x18,0x00,0xc0,0x00,0x40,0x00,0x00,
 0x02,0x00,0x00,0x00,0x10,0x18,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x12,0x18,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x14,0x18,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x16,0x18,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x18,0x18,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x1a,0x18,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x1c,0x18,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x1e,0x18,0x00,0xc0,0xf0,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x20,0x18,0x00,0xc0,0x40,0x01,0x00,0x00,0x02,0x00,0x00,0x00,
 0x22,0x18,0x00,0xc0,0x40,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x24,0x18,0x00,0xc0,
 0xf0,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x26,0x18,0x00,0xc0,0x40,0x01,0x00,0x00,
 0x02,0x00,0x00,0x00,0x28,0x18,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x28,0x18,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x2a,0x18,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x2c,0x18,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x2e,0x18,0x00,0xc0,0x73,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x30,0x18,0x00,0xc0,0x10,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x34,0x18,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x36,0x18,0x00,0xc0,0x30,0x23,0x00,0x00,
 0x02,0x00,0x00,0x00,0x80,0x28,0x00,0xc0,0x04,0x10,0x00,0x00,0x02,0x00,0x00,0x00,
 0x82,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x84,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x86,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x88,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x8a,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x8c,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x8e,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x90,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x92,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x94,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x96,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x98,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x9a,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x9c,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x9e,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xa0,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xa2,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xa4,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xa6,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xa8,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xaa,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xac,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xae,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xb0,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xb2,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xb4,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xb6,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xb8,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xba,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xbc,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xbe,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xc0,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xc2,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xc4,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xc6,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xc8,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xca,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xcc,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xce,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xd0,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xd2,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xd4,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xd6,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xd8,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xda,0x28,0x00,0xc0,0xab,0x02,0x00,0x00,0x02,0x00,0x00,0x00,0xdc,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xde,0x28,0x00,0xc0,0x08,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xde,0x28,0x00,0xc0,0x08,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xe0,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xe2,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xe4,0x28,0x00,0xc0,0x3f,0x01,0x00,0x00,
 0x02,0x00,0x00,0x00,0xe6,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xe8,0x28,0x00,0xc0,0xef,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xea,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xec,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xee,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xf0,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xf2,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xf4,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xf6,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xf8,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xfa,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xfc,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xfe,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x00,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x29,0x00,0xc0,
 0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x29,0x00,0xc0,0xff,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x06,0x29,0x00,0xc0,0x00,0x04,0x00,0x00,0x02,0x00,0x00,0x00,
 0x08,0x29,0x00,0xc0,0x40,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x0a,0x29,0x00,0xc0,
 0x40,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x0c,0x29,0x00,0xc0,0x40,0x01,0x00,0x00,
 0x02,0x00,0x00,0x00,0x0e,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x10,0x29,0x00,0xc0,0x00,0x03,0x00,0x00,0x02,0x00,0x00,0x00,0x12,0x29,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x14,0x29,0x00,0xc0,0x00,0x03,0x00,0x00,
 0x02,0x00,0x00,0x00,0x16,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x18,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x1a,0x29,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x1c,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x1e,0x29,0x00,0xc0,0x00,0xff,0x00,0x00,0x02,0x00,0x00,0x00,
 0x20,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x22,0x29,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x24,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x26,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x28,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x2a,0x29,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x2c,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x2e,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x30,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x32,0x29,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x34,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x36,0x29,0x00,0xc0,0x00,0x40,0x00,0x00,0x02,0x00,0x00,0x00,
 0x38,0x29,0x00,0xc0,0x40,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x3a,0x29,0x00,0xc0,
 0x00,0x40,0x00,0x00,0x02,0x00,0x00,0x00,0x3c,0x29,0x00,0xc0,0x40,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x3e,0x29,0x00,0xc0,0x00,0x40,0x00,0x00,0x02,0x00,0x00,0x00,
 0x40,0x29,0x00,0xc0,0x40,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x42,0x29,0x00,0xc0,
 0x00,0x40,0x00,0x00,0x02,0x00,0x00,0x00,0x44,0x29,0x00,0xc0,0x40,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x46,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x74,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x76,0x29,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x78,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x7a,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x7c,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x7e,0x29,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x80,0x29,0x00,0xc0,0x7d,0x77,0x00,0x00,
 0x02,0x00,0x00,0x00,0x00,0x28,0x00,0xc0,0x05,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x02,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x28,0x00,0xc0,
 0x10,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x28,0x00,0xc0,0xff,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0x08,0x28,0x00,0xc0,0xff,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x0a,0x28,0x00,0xc0,0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0x0c,0x28,0x00,0xc0,
 0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0x0e,0x28,0x00,0xc0,0xff,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x16,0x28,0x00,0xc0,0x3f,0x01,0x00,0x00,0x02,0x00,0x00,0x00,
 0x18,0x28,0x00,0xc0,0xef,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x1a,0x28,0x00,0xc0,
 0x1d,0x04,0x00,0x00,0x02,0x00,0x00,0x00,0x1c,0x28,0x00,0xc0,0x09,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x1e,0x28,0x00,0xc0,0x09,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x20,0x28,0x00,0xc0,0x03,0x04,0x00,0x00,0x02,0x00,0x00,0x00,0x22,0x28,0x00,0xc0,
 0x16,0x08,0x00,0x00,0x02,0x00,0x00,0x00,0x24,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x26,0x28,0x00,0xc0,0x50,0x02,0x00,0x00,0x02,0x00,0x00,0x00,
 0x28,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x2a,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x2c,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x2e,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x30,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x32,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x34,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x36,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x38,0x28,0x00,0xc0,0x64,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x3a,0x28,0x00,0xc0,
 0xfc,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x3c,0x28,0x00,0xc0,0x00,0x02,0x00,0x00,
 0x02,0x00,0x00,0x00,0x3e,0x28,0x00,0xc0,0xec,0x02,0x00,0x00,0x02,0x00,0x00,0x00,
 0x40,0x28,0x00,0xc0,0xec,0x02,0x00,0x00,0x02,0x00,0x00,0x00,0x42,0x28,0x00,0xc0,
 0x10,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x44,0x28,0x00,0xc0,0x14,0x01,0x00,0x00,
 0x02,0x00,0x00,0x00,0x46,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x48,0x28,0x00,0xc0,0x10,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x30,0x00,0xc0,0x00,0x01,0x00,0x00,
 0x02,0x00,0x00,0x00,0x04,0x30,0x00,0xc0,0x00,0x44,0x00,0x00,0x02,0x00,0x00,0x00,
 0x06,0x30,0x00,0xc0,0x06,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x08,0x30,0x00,0xc0,
 0x0a,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0a,0x30,0x00,0xc0,0x87,0x01,0x00,0x00,
 0x02,0x00,0x00,0x00,0x0c,0x30,0x00,0xc0,0x32,0x20,0x00,0x00,0x02,0x00,0x00,0x00,
 0x0e,0x30,0x00,0xc0,0x0b,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x10,0x30,0x00,0xc0,
 0x4b,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x12,0x30,0x00,0xc0,0x0a,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x14,0x30,0x00,0xc0,0xfa,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x16,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x18,0x30,0x00,0xc0,
 0x05,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x1a,0x30,0x00,0xc0,0xff,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x1c,0x30,0x00,0xc0,0x05,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x1e,0x30,0x00,0xc0,0x4f,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x20,0x30,0x00,0xc0,
 0x0a,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x22,0x30,0x00,0xc0,0xfa,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x24,0x30,0x00,0xc0,0x0a,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x26,0x30,0x00,0xc0,0x4a,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x28,0x30,0x00,0xc0,
 0x00,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0x2a,0x30,0x00,0xc0,0x3c,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x2c,0x30,0x00,0xc0,0x50,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x2e,0x30,0x00,0xc0,0x0a,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x30,0x30,0x00,0xc0,
 0xfa,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x32,0x30,0x00,0xc0,0x0a,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x34,0x30,0x00,0xc0,0x4a,0x01,0x00,0x00,0x02,0x00,0x00,0x00,
 0x36,0x30,0x00,0xc0,0x03,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x38,0x30,0x00,0xc0,
 0x00,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0x3a,0x30,0x00,0xc0,0x00,0x80,0x00,0x00,
 0x02,0x00,0x00,0x00,0x40,0x30,0x00,0xc0,0x00,0x40,0x00,0x00,0x02,0x00,0x00,0x00,
 0x42,0x30,0x00,0xc0,0x40,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x44,0x30,0x00,0xc0,
 0x00,0x40,0x00,0x00,0x02,0x00,0x00,0x00,0x46,0x30,0x00,0xc0,0x40,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x48,0x30,0x00,0xc0,0x00,0x40,0x00,0x00,0x02,0x00,0x00,0x00,
 0x4a,0x30,0x00,0xc0,0x40,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x4c,0x30,0x00,0xc0,
 0x00,0x40,0x00,0x00,0x02,0x00,0x00,0x00,0x4e,0x30,0x00,0xc0,0x40,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x54,0x30,0x00,0xc0,0x13,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x56,0x30,0x00,0xc0,0x02,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x58,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x5a,0x30,0x00,0xc0,0xff,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0x5c,0x30,0x00,0xc0,0x04,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x5e,0x30,0x00,0xc0,0x04,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x60,0x30,0x00,0xc0,
 0x08,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x62,0x30,0x00,0xc0,0x00,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0x64,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x66,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x68,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x6a,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x6c,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x6e,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x70,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x72,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x76,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x78,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x7a,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x7c,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x7e,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x80,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x82,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x84,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x86,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x88,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x8a,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x8c,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x8e,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x90,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x92,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x94,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x96,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x98,0x30,0x00,0xc0,0x80,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x9a,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x9c,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x9e,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xdc,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xde,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xe0,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xe2,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xe4,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xe6,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xe8,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xea,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xec,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xee,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xf0,0x30,0x00,0xc0,0x80,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xee,0x30,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xf0,0x30,0x00,0xc0,0x80,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xf2,0x30,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x0b,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x02,0x0b,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x04,0x0b,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x0b,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x08,0x0b,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x10,0x0b,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x12,0x0b,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x14,0x0b,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x16,0x0b,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x18,0x0b,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x00,0x3e,0x00,0xc0,0x00,0x40,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x3e,0x00,0xc0,
 0x07,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x3e,0x00,0xc0,0xfb,0xcf,0x00,0x00,
 0x02,0x00,0x00,0x00,0x08,0x3e,0x00,0xc0,0x2a,0x98,0x00,0x00,0x02,0x00,0x00,0x00,
 0x00,0x14,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x14,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x0d,0x00,0xc0,0x60,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x02,0x0d,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x04,0x0d,0x00,0xc0,0xe2,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x0d,0x00,0xc0,
 0x5b,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x42,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x02,0x42,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x04,0x42,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x42,0x00,0xc0,
 0x01,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x0e,0x00,0xc0,0x04,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x02,0x0e,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x04,0x0e,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x0e,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x08,0x0e,0x00,0xc0,0x02,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x0c,0x0e,0x00,0xc0,0x02,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x0e,0x0e,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x10,0x0e,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x44,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x02,0x44,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x04,0x44,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x06,0x44,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x3f,0x00,0xc0,0xe2,0x40,0x00,0x00,
 0x02,0x00,0x00,0x00,0x02,0x3f,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x04,0x3f,0x00,0xc0,0x09,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x06,0x3f,0x00,0xc0,
 0xc5,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x3c,0x00,0xc0,0x03,0x40,0x00,0x00,
 0x02,0x00,0x00,0x00,0x00,0x16,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x06,0x16,0x00,0xc0,0x00,0x0c,0x00,0x00,0x02,0x00,0x00,0x00,0x08,0x16,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0a,0x16,0x00,0xc0,0x31,0x58,0x00,0x00,
 0x02,0x00,0x00,0x00,0x0c,0x16,0x00,0xc0,0xfc,0x5f,0x00,0x00,0x02,0x00,0x00,0x00,
 0x0e,0x16,0x00,0xc0,0xd0,0x90,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x47,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x47,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x04,0x47,0x00,0xc0,0x0e,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x06,0x47,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x08,0x47,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x0a,0x47,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x0c,0x47,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x00,0x17,0x00,0xc0,0x00,0x50,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x17,0x00,0xc0,
 0xb4,0x4c,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x17,0x00,0xc0,0x00,0x8e,0x00,0x00,
 0x02,0x00,0x00,0x00,0x06,0x17,0x00,0xc0,0x03,0xc2,0x00,0x00,0x02,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x24,0x09,0x00,0xc0,
 0x00,0x59,0x00,0x00,0x02,0x00,0x00,0x00,0x46,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x48,0x28,0x00,0xc0,0x10,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x16,0x28,0x00,0xc0,0x3f,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x18,0x28,0x00,0xc0,
 0xef,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x1a,0x28,0x00,0xc0,0x1d,0x04,0x00,0x00,
 0x02,0x00,0x00,0x00,0x1e,0x28,0x00,0xc0,0x09,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x1c,0x28,0x00,0xc0,0x09,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x26,0x28,0x00,0xc0,
 0x50,0x02,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x28,0x00,0xc0,0x10,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x20,0x28,0x00,0xc0,0x03,0x0f,0x00,0x00,0x02,0x00,0x00,0x00,
 0x22,0x28,0x00,0xc0,0x16,0x08,0x00,0x00,0x02,0x00,0x00,0x00,0x80,0x28,0x00,0xc0,
 0x04,0x10,0x00,0x00,0x02,0x00,0x00,0x00,0xda,0x28,0x00,0xc0,0xab,0x02,0x00,0x00,
 0x02,0x00,0x00,0x00,0xdc,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xde,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xe0,0x28,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xe2,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xe4,0x28,0x00,0xc0,0x3f,0x01,0x00,0x00,0x02,0x00,0x00,0x00,
 0xe6,0x28,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xe8,0x28,0x00,0xc0,
 0xef,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x29,0x00,0xc0,0xff,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0x04,0x29,0x00,0xc0,0xff,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x06,0x29,0x00,0xc0,0x00,0x04,0x00,0x00,0x02,0x00,0x00,0x00,0x08,0x29,0x00,0xc0,
 0x40,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0x0a,0x29,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x0c,0x29,0x00,0xc0,0x40,0x01,0x00,0x00,0x02,0x00,0x00,0x00,
 0x20,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x22,0x10,0x00,0xc0,
 0x50,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x24,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x26,0x10,0x00,0xc0,0xaa,0x0a,0x00,0x00,0x02,0x00,0x00,0x00,
 0x28,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x2a,0x10,0x00,0xc0,
 0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x2c,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,
 0x02,0x00,0x00,0x00,0x2e,0x10,0x00,0xc0,0x55,0x09,0x00,0x00,0x02,0x00,0x00,0x00,
 0x30,0x10,0x00,0xc0,0x0a,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x32,0x10,0x00,0xc0,
 0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x34,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x36,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,
 0x38,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x3a,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x3c,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x40,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,
 0x42,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x44,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x46,0x10,0x00,0xc0,0x2a,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x48,0x10,0x00,0xc0,0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,
 0x4a,0x10,0x00,0xc0,0x0a,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x4c,0x10,0x00,0xc0,
 0xaa,0xaa,0x00,0x00,0x02,0x00,0x00,0x00,0x4e,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x50,0x10,0x00,0xc0,0xaa,0x8a,0x00,0x00,0x02,0x00,0x00,0x00,
 0x52,0x10,0x00,0xc0,0xaa,0x16,0x00,0x00,0x02,0x00,0x00,0x00,0x54,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x56,0x10,0x00,0xc0,0x2a,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x58,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x5a,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x5c,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x09,0x00,0xc0,0xff,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0x0a,0x09,0x00,0xc0,0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,
 0x1c,0x09,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x1e,0x09,0x00,0xc0,
 0x70,0x41,0x00,0x00,0x02,0x00,0x00,0x00,0x26,0x09,0x00,0xc0,0x00,0x5e,0x00,0x00,
 0x02,0x00,0x00,0x00,0x06,0x28,0x00,0xc0,0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,
 0x08,0x28,0x00,0xc0,0xff,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xc0,0x10,0x00,0xc0,
 0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,0xc2,0x10,0x00,0xc0,0xff,0xff,0x00,0x00,
 0x02,0x00,0x00,0x00,0xc4,0x10,0x00,0xc0,0xff,0xff,0x00,0x00,0x02,0x00,0x00,0x00,
 0xc6,0x10,0x00,0xc0,0xfc,0x3f,0x00,0x00,0x02,0x00,0x00,0x00,0xc8,0x10,0x00,0xc0,
 0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xca,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xcc,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xce,0x10,0x00,0xc0,0x7e,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0xd0,0x10,0x00,0xc0,
 0x00,0x42,0x00,0x00,0x02,0x00,0x00,0x00,0xd2,0x10,0x00,0xc0,0x00,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xd4,0x10,0x00,0xc0,0xff,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0xd6,0x10,0x00,0xc0,0xff,0x7f,0x00,0x00,0x02,0x00,0x00,0x00,0xd8,0x10,0x00,0xc0,
 0xff,0x01,0x00,0x00,0x02,0x00,0x00,0x00,0xda,0x10,0x00,0xc0,0xef,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0xdc,0x10,0x00,0xc0,0x3f,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 };

 gp2x_deinit_();

 gp2x_dev[2] = open("/dev/mem", O_RDWR);
 gp2x_memregl=(unsigned long  *)mmap(0, 0x10000,                            PROT_READ|PROT_WRITE, MAP_SHARED, gp2x_dev[2], 0xc0000000);
 gp2x_memregs=(unsigned short *)gp2x_memregl;

 if(file)
 {
  FILE *in;
  unsigned int i, j, *resetvector = (unsigned int *)mmap(0,         64, PROT_READ | PROT_WRITE, MAP_SHARED, gp2x_dev[2], 0x00000000),
                     *loader      = (unsigned int *)mmap(0, 0x02000000, PROT_READ | PROT_WRITE, MAP_SHARED, gp2x_dev[2], 0x02000000);

  for (i=0;i<((sizeof(prog)>>2)+1);i++) loader[i]=prog[i]; 

  if((in = fopen(file,"rb")))
  {
   for(i=0;!feof(in);loader[0x1000 + (i++)] = j) fread(&j, 1, 4, in); 
   fclose(in);

   loader[0xfff] = i;

   resetvector[0] = 0xE59FF034, resetvector[15] = 0x02000000;
  }
 }

 gp2x_memregl[0x0808>>2] = 0xFF8FFFE7;                      //mask interrupts

 gp2x_memregs[0x0904>>1] &= (~0x80);                        //disable clock

 gp2x_memregs[0x0C18>>1] = (file ? 7 : 4);                  //set reboot mode

 gp2x_memregs[0x0A16>>1] = gp2x_memregs[0x0A18>>1] = 8;     //clear pending interrupts &
 gp2x_memregs[0x0800>>1] = gp2x_memregs[0x0810>>1] = 32;    //enable timer interrupt too
 gp2x_memregl[0x808>>2] &= (~0x20);             
 
 gp2x_memregs[0x0C14>>1] = 0x100;                           //watchdog reset 
 gp2x_memregl[0x0A10>>2] = gp2x_memregl[0x0A00>>2] + 50000; //watchdog to fire in 50,000 counts 
 gp2x_memregs[0x0A14>>1] = 3;                               //set match 3 to be watchdog reset rather than an interrupt 
 
 gp2x_memregs[0x0904>>1] |= 0x80;                           //enable clock

 while(1);
}





/*
   ,--------------------.
   |                    |X
   |    GP2X - FONT     |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/


/* Credits:
   Font generated from Xorg 6x10-L1.bdf */

const unsigned char gp2x_fontf[256][10] =
{ 
 { 0x00>>2, 0xA8>>2, 0x00>>2, 0x88>>2, 0x00>>2, 0x88>>2, 0x00>>2, 0xA8>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x20>>2, 0x70>>2, 0xF8>>2, 0x70>>2, 0x20>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0xA8>>2, 0x54>>2, 0xA8>>2, 0x54>>2, 0xA8>>2, 0x54>>2, 0xA8>>2, 0x54>>2, 0xA8>>2, 0x54>>2, },
 { 0x00>>2, 0x90>>2, 0x90>>2, 0xF0>>2, 0x90>>2, 0x90>>2, 0x78>>2, 0x10>>2, 0x10>>2, 0x10>>2, },
 { 0x00>>2, 0xE0>>2, 0x80>>2, 0xC0>>2, 0x80>>2, 0xB8>>2, 0x20>>2, 0x30>>2, 0x20>>2, 0x20>>2, },
 { 0x00>>2, 0x70>>2, 0x80>>2, 0x80>>2, 0x70>>2, 0x70>>2, 0x48>>2, 0x70>>2, 0x48>>2, 0x48>>2, },
 { 0x00>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0xF0>>2, 0x78>>2, 0x40>>2, 0x70>>2, 0x40>>2, 0x40>>2, },
 { 0x00>>2, 0x20>>2, 0x50>>2, 0x20>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x20>>2, 0x20>>2, 0xF8>>2, 0x20>>2, 0x20>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x90>>2, 0xD0>>2, 0xD0>>2, 0xB0>>2, 0x90>>2, 0x40>>2, 0x40>>2, 0x40>>2, 0x78>>2, },
 { 0x00>>2, 0x90>>2, 0x90>>2, 0x60>>2, 0x40>>2, 0x78>>2, 0x10>>2, 0x10>>2, 0x10>>2, 0x10>>2, },
 { 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0xE0>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0xE0>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x3C>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, },
 { 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x3C>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0xFC>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, },
 { 0xFC>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0xFC>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0xFF>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0xFC>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0xFC>>2, },
 { 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x3C>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, },
 { 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0xE0>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, },
 { 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0xFC>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0xFC>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, },
 { 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, },
 { 0x00>>2, 0x18>>2, 0x60>>2, 0x80>>2, 0x60>>2, 0x18>>2, 0x00>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xC0>>2, 0x30>>2, 0x08>>2, 0x30>>2, 0xC0>>2, 0x00>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0xF8>>2, 0x50>>2, 0x50>>2, 0x50>>2, 0x50>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x08>>2, 0x10>>2, 0xF8>>2, 0x20>>2, 0xF8>>2, 0x40>>2, 0x80>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x30>>2, 0x48>>2, 0x40>>2, 0xE0>>2, 0x40>>2, 0x48>>2, 0xB0>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x20>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x50>>2, 0x50>>2, 0x50>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x50>>2, 0x50>>2, 0xF8>>2, 0x50>>2, 0xF8>>2, 0x50>>2, 0x50>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x70>>2, 0xA0>>2, 0x70>>2, 0x28>>2, 0x70>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x48>>2, 0xA8>>2, 0x50>>2, 0x20>>2, 0x50>>2, 0xA8>>2, 0x90>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x40>>2, 0xA0>>2, 0xA0>>2, 0x40>>2, 0xA8>>2, 0x90>>2, 0x68>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x10>>2, 0x20>>2, 0x40>>2, 0x40>>2, 0x40>>2, 0x20>>2, 0x10>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x40>>2, 0x20>>2, 0x10>>2, 0x10>>2, 0x10>>2, 0x20>>2, 0x40>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x88>>2, 0x50>>2, 0xF8>>2, 0x50>>2, 0x88>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x20>>2, 0x20>>2, 0xF8>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x30>>2, 0x20>>2, 0x40>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0xF8>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x20>>2, 0x70>>2, 0x20>>2, 0x00>>2, },
 { 0x00>>2, 0x08>>2, 0x08>>2, 0x10>>2, 0x20>>2, 0x40>>2, 0x80>>2, 0x80>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x50>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x50>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x60>>2, 0xA0>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x08>>2, 0x30>>2, 0x40>>2, 0x80>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF8>>2, 0x08>>2, 0x10>>2, 0x30>>2, 0x08>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x10>>2, 0x30>>2, 0x50>>2, 0x90>>2, 0xF8>>2, 0x10>>2, 0x10>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF8>>2, 0x80>>2, 0xB0>>2, 0xC8>>2, 0x08>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x30>>2, 0x40>>2, 0x80>>2, 0xB0>>2, 0xC8>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF8>>2, 0x08>>2, 0x10>>2, 0x10>>2, 0x20>>2, 0x40>>2, 0x40>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x08>>2, 0x10>>2, 0x60>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x20>>2, 0x70>>2, 0x20>>2, 0x00>>2, 0x20>>2, 0x70>>2, 0x20>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x20>>2, 0x70>>2, 0x20>>2, 0x00>>2, 0x30>>2, 0x20>>2, 0x40>>2, 0x00>>2, },
 { 0x00>>2, 0x08>>2, 0x10>>2, 0x20>>2, 0x40>>2, 0x20>>2, 0x10>>2, 0x08>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0xF8>>2, 0x00>>2, 0xF8>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x40>>2, 0x20>>2, 0x10>>2, 0x08>>2, 0x10>>2, 0x20>>2, 0x40>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x10>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x98>>2, 0xA8>>2, 0xB0>>2, 0x80>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x50>>2, 0x88>>2, 0x88>>2, 0xF8>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF0>>2, 0x48>>2, 0x48>>2, 0x70>>2, 0x48>>2, 0x48>>2, 0xF0>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF0>>2, 0x48>>2, 0x48>>2, 0x48>>2, 0x48>>2, 0x48>>2, 0xF0>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF8>>2, 0x80>>2, 0x80>>2, 0xF0>>2, 0x80>>2, 0x80>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF8>>2, 0x80>>2, 0x80>>2, 0xF0>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x80>>2, 0x80>>2, 0x98>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0xF8>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x38>>2, 0x10>>2, 0x10>>2, 0x10>>2, 0x10>>2, 0x90>>2, 0x60>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x88>>2, 0x90>>2, 0xA0>>2, 0xC0>>2, 0xA0>>2, 0x90>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x88>>2, 0x88>>2, 0xD8>>2, 0xA8>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x88>>2, 0x88>>2, 0xC8>>2, 0xA8>>2, 0x98>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF0>>2, 0x88>>2, 0x88>>2, 0xF0>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0xA8>>2, 0x70>>2, 0x08>>2, 0x00>>2, },
 { 0x00>>2, 0xF0>>2, 0x88>>2, 0x88>>2, 0xF0>>2, 0xA0>>2, 0x90>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x80>>2, 0x70>>2, 0x08>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF8>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x50>>2, 0x50>>2, 0x50>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0xA8>>2, 0xA8>>2, 0xD8>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x88>>2, 0x88>>2, 0x50>>2, 0x20>>2, 0x50>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x88>>2, 0x88>>2, 0x50>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF8>>2, 0x08>>2, 0x10>>2, 0x20>>2, 0x40>>2, 0x80>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x40>>2, 0x40>>2, 0x40>>2, 0x40>>2, 0x40>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x80>>2, 0x80>>2, 0x40>>2, 0x20>>2, 0x10>>2, 0x08>>2, 0x08>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x10>>2, 0x10>>2, 0x10>>2, 0x10>>2, 0x10>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x50>>2, 0x88>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0xF8>>2, 0x00>>2, },
 { 0x20>>2, 0x10>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x70>>2, 0x08>>2, 0x78>>2, 0x88>>2, 0x78>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x80>>2, 0x80>>2, 0xB0>>2, 0xC8>>2, 0x88>>2, 0xC8>>2, 0xB0>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0x80>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x08>>2, 0x08>>2, 0x68>>2, 0x98>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0xF8>>2, 0x80>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x30>>2, 0x48>>2, 0x40>>2, 0xF0>>2, 0x40>>2, 0x40>>2, 0x40>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x78>>2, 0x88>>2, 0x88>>2, 0x78>>2, 0x08>>2, 0x88>>2, 0x70>>2, },
 { 0x00>>2, 0x80>>2, 0x80>>2, 0xB0>>2, 0xC8>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x00>>2, 0x60>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x08>>2, 0x00>>2, 0x18>>2, 0x08>>2, 0x08>>2, 0x08>>2, 0x48>>2, 0x48>>2, 0x30>>2, },
 { 0x00>>2, 0x80>>2, 0x80>>2, 0x88>>2, 0x90>>2, 0xE0>>2, 0x90>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x60>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0xD0>>2, 0xA8>>2, 0xA8>>2, 0xA8>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0xB0>>2, 0xC8>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0xB0>>2, 0xC8>>2, 0x88>>2, 0xC8>>2, 0xB0>>2, 0x80>>2, 0x80>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x68>>2, 0x98>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x08>>2, 0x08>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0xB0>>2, 0xC8>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x70>>2, 0x80>>2, 0x70>>2, 0x08>>2, 0xF0>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x40>>2, 0x40>>2, 0xF0>>2, 0x40>>2, 0x40>>2, 0x48>>2, 0x30>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x50>>2, 0x50>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0xA8>>2, 0xA8>>2, 0x50>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x88>>2, 0x50>>2, 0x20>>2, 0x50>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x08>>2, 0x88>>2, 0x70>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0xF8>>2, 0x10>>2, 0x20>>2, 0x40>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x18>>2, 0x20>>2, 0x10>>2, 0x60>>2, 0x10>>2, 0x20>>2, 0x18>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x60>>2, 0x10>>2, 0x20>>2, 0x18>>2, 0x20>>2, 0x10>>2, 0x60>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x48>>2, 0xA8>>2, 0x90>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x00>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x20>>2, 0x78>>2, 0xA0>>2, 0xA0>>2, 0xA0>>2, 0x78>>2, 0x20>>2, 0x00>>2, },
 { 0x00>>2, 0x30>>2, 0x48>>2, 0x40>>2, 0xE0>>2, 0x40>>2, 0x48>>2, 0xB0>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x88>>2, 0x70>>2, 0x50>>2, 0x70>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x88>>2, 0x88>>2, 0x50>>2, 0x20>>2, 0xF8>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x80>>2, 0xE0>>2, 0x90>>2, 0x48>>2, 0x38>>2, 0x08>>2, 0x70>>2, 0x00>>2, },
 { 0x50>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0xA8>>2, 0xC8>>2, 0xA8>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x38>>2, 0x48>>2, 0x58>>2, 0x28>>2, 0x00>>2, 0x78>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x24>>2, 0x48>>2, 0x90>>2, 0x48>>2, 0x24>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x78>>2, 0x08>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x78>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0xE8>>2, 0xC8>>2, 0xC8>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0xF8>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x50>>2, 0x20>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x20>>2, 0x20>>2, 0xF8>>2, 0x20>>2, 0x20>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x30>>2, 0x48>>2, 0x10>>2, 0x20>>2, 0x78>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x70>>2, 0x08>>2, 0x30>>2, 0x08>>2, 0x70>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0x20>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0xC8>>2, 0xB0>>2, 0x80>>2, 0x00>>2, },
 { 0x00>>2, 0x78>>2, 0xE8>>2, 0xE8>>2, 0x68>>2, 0x28>>2, 0x28>>2, 0x28>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x20>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x10>>2, 0x20>>2, },
 { 0x20>>2, 0x60>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x30>>2, 0x48>>2, 0x48>>2, 0x30>>2, 0x00>>2, 0x78>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x90>>2, 0x48>>2, 0x24>>2, 0x48>>2, 0x90>>2, 0x00>>2, 0x00>>2, },
 { 0x40>>2, 0xC0>>2, 0x40>>2, 0x40>>2, 0xE4>>2, 0x0C>>2, 0x14>>2, 0x3C>>2, 0x04>>2, 0x00>>2, },
 { 0x40>>2, 0xC0>>2, 0x40>>2, 0x40>>2, 0xE8>>2, 0x14>>2, 0x04>>2, 0x08>>2, 0x1C>>2, 0x00>>2, },
 { 0xC0>>2, 0x20>>2, 0x40>>2, 0x20>>2, 0xC8>>2, 0x18>>2, 0x28>>2, 0x78>>2, 0x08>>2, 0x00>>2, },
 { 0x00>>2, 0x20>>2, 0x00>>2, 0x20>>2, 0x20>>2, 0x40>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x40>>2, 0x20>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0xF8>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0x20>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0xF8>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0xF8>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x48>>2, 0xB0>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0xF8>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x50>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0xF8>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0xF8>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x3C>>2, 0x50>>2, 0x90>>2, 0x9C>>2, 0xF0>>2, 0x90>>2, 0x9C>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0x88>>2, 0x70>>2, 0x20>>2, 0x40>>2, },
 { 0x40>>2, 0xF8>>2, 0x80>>2, 0x80>>2, 0xF0>>2, 0x80>>2, 0x80>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0xF8>>2, 0x80>>2, 0x80>>2, 0xF0>>2, 0x80>>2, 0x80>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0xF8>>2, 0x80>>2, 0x80>>2, 0xF0>>2, 0x80>>2, 0x80>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x50>>2, 0xF8>>2, 0x80>>2, 0x80>>2, 0xF0>>2, 0x80>>2, 0x80>>2, 0xF8>>2, 0x00>>2, 0x00>>2, },
 { 0x40>>2, 0x20>>2, 0x70>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0x20>>2, 0x70>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x70>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x50>>2, 0x00>>2, 0x70>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xF0>>2, 0x48>>2, 0x48>>2, 0xE8>>2, 0x48>>2, 0x48>>2, 0xF0>>2, 0x00>>2, 0x00>>2, },
 { 0x28>>2, 0x50>>2, 0x88>>2, 0xC8>>2, 0xA8>>2, 0x98>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x40>>2, 0x20>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0x20>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x28>>2, 0x50>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x50>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x88>>2, 0x50>>2, 0x20>>2, 0x50>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x98>>2, 0x98>>2, 0xA8>>2, 0xC8>>2, 0xC8>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x40>>2, 0x20>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0x20>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x50>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0x20>>2, 0x88>>2, 0x88>>2, 0x50>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x80>>2, 0xF0>>2, 0x88>>2, 0xF0>>2, 0x80>>2, 0x80>>2, 0x80>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x70>>2, 0x88>>2, 0x90>>2, 0xA0>>2, 0x90>>2, 0x88>>2, 0xB0>>2, 0x00>>2, 0x00>>2, },
 { 0x40>>2, 0x20>>2, 0x00>>2, 0x70>>2, 0x08>>2, 0x78>>2, 0x88>>2, 0x78>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0x20>>2, 0x00>>2, 0x70>>2, 0x08>>2, 0x78>>2, 0x88>>2, 0x78>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x00>>2, 0x70>>2, 0x08>>2, 0x78>>2, 0x88>>2, 0x78>>2, 0x00>>2, 0x00>>2, },
 { 0x28>>2, 0x50>>2, 0x00>>2, 0x70>>2, 0x08>>2, 0x78>>2, 0x88>>2, 0x78>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x50>>2, 0x00>>2, 0x70>>2, 0x08>>2, 0x78>>2, 0x88>>2, 0x78>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x20>>2, 0x70>>2, 0x08>>2, 0x78>>2, 0x88>>2, 0x78>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x78>>2, 0x14>>2, 0x7C>>2, 0x90>>2, 0x7C>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0x80>>2, 0x88>>2, 0x70>>2, 0x20>>2, 0x40>>2, },
 { 0x40>>2, 0x20>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0xF8>>2, 0x80>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0x20>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0xF8>>2, 0x80>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0xF8>>2, 0x80>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x50>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0xF8>>2, 0x80>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x40>>2, 0x20>>2, 0x00>>2, 0x60>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x40>>2, 0x00>>2, 0x60>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x00>>2, 0x60>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x50>>2, 0x00>>2, 0x60>>2, 0x20>>2, 0x20>>2, 0x20>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0xC0>>2, 0x30>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x28>>2, 0x50>>2, 0x00>>2, 0xB0>>2, 0xC8>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x00>>2, 0x00>>2, },
 { 0x40>>2, 0x20>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0x20>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x28>>2, 0x50>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x50>>2, 0x00>>2, 0x70>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x70>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x20>>2, 0x00>>2, 0xF8>>2, 0x00>>2, 0x20>>2, 0x00>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x00>>2, 0x00>>2, 0x78>>2, 0x98>>2, 0xA8>>2, 0xC8>>2, 0xF0>>2, 0x00>>2, 0x00>>2, },
 { 0x40>>2, 0x20>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x00>>2, 0x00>>2, },
 { 0x10>>2, 0x20>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x00>>2, 0x00>>2, },
 { 0x20>>2, 0x50>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x50>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x00>>2, 0x00>>2, },
 { 0x00>>2, 0x10>>2, 0x20>>2, 0x88>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x08>>2, 0x88>>2, 0x70>>2, },
 { 0x00>>2, 0x00>>2, 0x80>>2, 0xF0>>2, 0x88>>2, 0x88>>2, 0x88>>2, 0xF0>>2, 0x80>>2, 0x80>>2, },
 { 0x00>>2, 0x50>>2, 0x00>>2, 0x88>>2, 0x88>>2, 0x98>>2, 0x68>>2, 0x08>>2, 0x88>>2, 0x70>>2, },
};

static void (*gp2x_printfchar)(gp2x_font *f, unsigned char c);

/* Credits:
   rlyeh */

static void gp2x_printfchar15(gp2x_font *f, unsigned char c)
{
  unsigned short *dst=&((unsigned short *)gp2x_video_RGB[0].screen)[f->x+f->y*320],w,h=f->h;
  unsigned char  *src=&f->data[c*10];

 if(f->solid)
         while(h--)
         {
          w=f->wmask;
          while(w)
          {
           if( *src & w ) *dst++=f->fg; else *dst++=f->bg;
           w>>=1;
          }
          src++;    

          dst+=320-(f->w);
         }
 else
         while(h--)
         {
          w=f->wmask;
          while(w)
          {
           if( *src & w ) *dst=f->fg;
           dst++;
           w>>=1;
          }
          src++;

          dst+=320-(f->w);
         }
}

static void gp2x_printfchar8(gp2x_font *f, unsigned char c)
{
  unsigned char *dst=&((unsigned char *)gp2x_video_RGB[0].screen)[f->x+f->y*320],w,h=f->h;
  unsigned char *src=&f->data[c*10];

 if(f->solid)
         while(h--)
         {
          w=f->wmask;
          while(w)
          {
           if( *src & w ) *dst++=f->fg; else *dst++=f->bg;
           w>>=1;
          }
          src++;    

          dst+=320-(f->w);
         }
 else
         while(h--)
         {
          w=f->wmask;
          while(w)
          {
           if( *src & w ) *dst=f->fg;
           dst++;
           w>>=1;
          }
          src++;    

          dst+=320-(f->w);
         }
}

/* Function: gp2x_printf
   This function printfs a string into the RGB layer.

   Note:
   - This function does not perform any clipping.

   Parameters:
   *f - <gp2x_font> or NULL. NULL will use the default 6x10 font.
   x (0..320) - Horizontal coordinate for the given string
       If *x* is negative, gp2x_printf() will continue on last known horizontal position.
   y (0..240) - Vertical coordinate for the given string 
       If *y* is negative, gp2x_printf() will continue on last known vertical position.
   format - C formatted standard string.

   See also:
   <gp2x_printf_init>

   Example:
   > a=1;
   > gp2x_printf(NULL, 0, 0,"Hello %dst world\n",a);
   > gp2x_printf(NULL,-1,-1,"Hello %dnd world\n",++a);
   >
   > ...it outputs...
   >
   > Hello 1st world                (at top-left of screen)
   > Hello 2nd world                (under the previous line) */
 
void gp2x_printf(gp2x_font *f, int x, int y, const char *format, ...)
{
 char buffer[4096]; int c; gp2x_font *g=&gp2x_default_font;
 va_list  args;

 va_start(args, format);
 vsprintf(buffer, format, args);
 va_end(args);

 if(f!=NULL) g=f;

 if(x<0) x=g->x; else g->x=x; 
 if(y<0) y=g->y; else g->y=y;

 for(c=0;buffer[c];c++)
 {
  switch(buffer[c])
  {
   case '\b': g->x=x;g->y=y; break;

   case '\n': g->y+=g->h;
   case '\r': g->x=x;
              break;

   default:   gp2x_printfchar(g, (unsigned char)buffer[c]);
              g->x+=g->w;
              break;
  }
 }
}

/* Function: gp2x_printf_init
   This function allows you to set your own fixed <gp2x_font>.

   Parameters:
   *f - Pointer to a <gp2x_font> struct.
   w (1..) - Width of each character, in pixels.
   h (1..) - Height of each character, in pixels.
   *data - Pointer to your font bitmap.
   fg - Foreground font color.
   bg - Background font color (when solid is not 0).
   solid (0,1) - When 1, background font color is used.
           When 0, transparent background font is used.
  
   See also:
   <gp2x_printf> */

void gp2x_printf_init(gp2x_font *f, int w, int h, void *data, int fg, int bg, int solid)
{
 f->x=f->y=0;
 f->wmask=1<<(w-1);
 f->w=w;
 f->h=h;
 f->data=(unsigned char *)data;
 f->fg=fg;
 f->bg=bg;
 f->solid=solid;
}






/*
   ,--------------------.
   |                    |X
   |   GP2X - MODULES   |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/

/* Private: gp2x_module_deinit
   Function which deinits modules used by library.

   Credits:
   rlyeh */

static void gp2x_module_deinit(void)
{
 system("/sbin/rmmod mousedev &> /dev/null");
 system("/sbin/rmmod keybdev  &> /dev/null");
 system("/sbin/rmmod joydev   &> /dev/null");
 system("/sbin/rmmod input    &> /dev/null");
 system("/sbin/rmmod minimal  &> /dev/null");
 remove("/mnt/minimal.o");
}

/* Private: gp2x_module_init
   Function which inits modules used by library.

   Credits:
   rlyeh */

static void gp2x_module_init(void)
{
 const unsigned char minimal_o[]={ //v0.21
 0x7f,0x45,0x4c,0x46,0x01,0x01,0x01,0x61,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x01,0x00,0x28,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0xf8,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x00,0x28,0x00,
 0x0c,0x00,0x09,0x00,0x0d,0xc0,0xa0,0xe1,0x00,0xd8,0x2d,0xe9,0x04,0xb0,0x4c,0xe2,
 0x00,0x30,0xa0,0xe3,0x0c,0xd0,0x4d,0xe2,0x12,0x00,0x00,0xea,0x00,0x30,0xa0,0xe3,
 0x0c,0x00,0x00,0xea,0x10,0x30,0x1b,0xe5,0x14,0x20,0x1b,0xe5,0x02,0x30,0x83,0xe0,
 0x18,0x30,0x0b,0xe5,0x14,0x30,0x1b,0xe5,0x01,0x30,0x83,0xe2,0x14,0x30,0x0b,0xe5,
 0x14,0x30,0x1b,0xe5,0x09,0x00,0x53,0xe3,0x06,0x00,0x00,0xda,0x10,0x30,0x1b,0xe5,
 0x01,0x30,0x83,0xe2,0x10,0x30,0x0b,0xe5,0x10,0x30,0x1b,0xe5,0x13,0x00,0x53,0xe3,
 0x04,0x00,0x00,0xda,0x00,0xa8,0x1b,0xe9,0x0d,0xc0,0xa0,0xe1,0x00,0xd8,0x2d,0xe9,
 0x04,0xb0,0x4c,0xe2,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x00,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,
 0x00,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x00,0x00,0xa0,0xe3,
 0x00,0xa8,0x1b,0xe9,0x0d,0xc0,0xa0,0xe1,0xf0,0xd8,0x2d,0xe9,0x04,0xb0,0x4c,0xe2,
 0x00,0xc0,0xa0,0xe1,0x10,0x00,0xa0,0xe3,0x01,0x30,0xa0,0xe1,0x01,0x10,0xa0,0xe3,
 0xff,0x50,0x0c,0xe2,0xff,0x60,0x03,0xe2,0xff,0x70,0x02,0xe2,0xfe,0xff,0xff,0xeb,
 0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,
 0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x07,0x40,0xa0,0xe3,0x55,0x34,0xa0,0xe1,0x01,0x10,0x13,0xe2,0x69,0x00,0x00,0x0a,
 0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x6b,0x00,0x00,0xea,
 0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x01,0x40,0x54,0xe2,0x62,0x00,0x00,0x5a,
 0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x00,0x00,0x50,0xe3,0x82,0x00,0x00,0x0a,
 0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x07,0x40,0xa0,0xe3,0x56,0x34,0xa0,0xe1,0x01,0x10,0x13,0xe2,
 0x92,0x00,0x00,0x0a,0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x94,0x00,0x00,0xea,0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x01,0x40,0x54,0xe2,
 0x8b,0x00,0x00,0x5a,0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x00,0x00,0x50,0xe3,
 0xab,0x00,0x00,0x0a,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,
 0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x07,0x40,0xa0,0xe3,0x57,0x34,0xa0,0xe1,
 0x01,0x10,0x13,0xe2,0xbb,0x00,0x00,0x0a,0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0xbd,0x00,0x00,0xea,0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x01,0x40,0x54,0xe2,0xb4,0x00,0x00,0x5a,0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x00,0x00,0x50,0xe3,0xd4,0x00,0x00,0x0a,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,
 0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x01,0x00,0xa0,0xe3,0xf0,0xa8,0x1b,0xe9,
 0x0d,0xc0,0xa0,0xe1,0xf0,0xd8,0x2d,0xe9,0x04,0xb0,0x4c,0xe2,0xff,0x50,0x00,0xe2,
 0x10,0x00,0xa0,0xe3,0x01,0x30,0xa0,0xe1,0x01,0x10,0xa0,0xe3,0x02,0x70,0xa0,0xe1,
 0xff,0x60,0x03,0xe2,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,
 0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x07,0x40,0xa0,0xe3,0x55,0x34,0xa0,0xe1,
 0x01,0x10,0x13,0xe2,0x23,0x01,0x00,0x0a,0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x25,0x01,0x00,0xea,0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x01,0x40,0x54,0xe2,0x1c,0x01,0x00,0x5a,0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x00,0x00,0x50,0xe3,0x3c,0x01,0x00,0x0a,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x07,0x40,0xa0,0xe3,
 0x56,0x34,0xa0,0xe1,0x01,0x10,0x13,0xe2,0x4c,0x01,0x00,0x0a,0x10,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x4e,0x01,0x00,0xea,0x10,0x00,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x01,0x40,0x54,0xe2,0x45,0x01,0x00,0x5a,0x10,0x00,0xa0,0xe3,
 0x00,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x00,0x00,0x50,0xe3,0x65,0x01,0x00,0x0a,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x01,0x50,0x85,0xe3,0x07,0x40,0xa0,0xe3,0x55,0x34,0xa0,0xe1,
 0x01,0x10,0x13,0xe2,0x8b,0x01,0x00,0x0a,0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x8d,0x01,0x00,0xea,0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x01,0x40,0x54,0xe2,0x84,0x01,0x00,0x5a,0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0x02,0x20,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,
 0x01,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x00,0x00,0x50,0xe3,0xa4,0x01,0x00,0x0a,0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,
 0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x00,0x50,0xa0,0xe3,
 0x07,0x40,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0xfe,0xff,0xff,0xeb,0x00,0x00,0x50,0xe3,
 0x01,0x30,0xa0,0x13,0x13,0x34,0x85,0x11,0xff,0x50,0x03,0x12,0xff,0x50,0x05,0x02,
 0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x01,0x40,0x54,0xe2,0xaa,0x01,0x00,0x5a,0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,0x02,0x20,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x01,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0xfe,0xff,0xff,0xeb,0x11,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,
 0xfe,0xff,0xff,0xeb,0x10,0x00,0xa0,0xe3,0x00,0x10,0xa0,0xe3,0xfe,0xff,0xff,0xeb,
 0x01,0x00,0xa0,0xe3,0x00,0x50,0xc7,0xe5,0xf0,0xa8,0x1b,0xe9,0x0d,0xc0,0xa0,0xe1,
 0x00,0xd8,0x2d,0xe9,0x04,0xb0,0x4c,0xe2,0x00,0x00,0xa0,0xe3,0x00,0xa8,0x1b,0xe9,
 0x0d,0xc0,0xa0,0xe1,0x00,0xd8,0x2d,0xe9,0x04,0xb0,0x4c,0xe2,0x00,0x00,0xa0,0xe3,
 0x00,0xa8,0x1b,0xe9,0x0d,0xc0,0xa0,0xe1,0x30,0xd8,0x2d,0xe9,0x94,0x13,0x9f,0xe5,
 0x04,0xb0,0x4c,0xe2,0x01,0x00,0x52,0xe1,0x03,0x10,0xa0,0xe1,0x10,0xd0,0x4d,0xe2,
 0xf6,0x01,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,
 0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,
 0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0xef,0x01,0x00,0x1a,0x0c,0x20,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0xf1,0x01,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x9a,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,
 0xcb,0x05,0x00,0xea,0x30,0x33,0x9f,0xe5,0x03,0x00,0x52,0xe1,0x0d,0x02,0x00,0x1a,
 0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,
 0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,
 0x1c,0x00,0x4b,0xe2,0x06,0x02,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x08,0x02,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,
 0x03,0x00,0xa0,0xe1,0x15,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,
 0xd8,0x32,0x9f,0xe5,0x03,0x00,0x52,0xe1,0x24,0x02,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,
 0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,
 0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,
 0x1d,0x02,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x1f,0x02,0x00,0xea,
 0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,
 0x16,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x80,0x32,0x9f,0xe5,
 0x03,0x00,0x52,0xe1,0x3b,0x02,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,
 0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,
 0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0x34,0x02,0x00,0x1a,
 0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x36,0x02,0x00,0xea,0x0c,0x10,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x17,0x0f,0x07,0xee,
 0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x11,0x0e,0x52,0xe3,0x51,0x02,0x00,0x1a,
 0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,
 0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,
 0x1c,0x00,0x4b,0xe2,0x4a,0x02,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x4c,0x02,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,
 0x03,0x00,0xa0,0xe1,0x35,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,
 0xd0,0x31,0x9f,0xe5,0x03,0x00,0x52,0xe1,0x68,0x02,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,
 0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,
 0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,
 0x61,0x02,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x63,0x02,0x00,0xea,
 0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,
 0x3d,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x78,0x31,0x9f,0xe5,
 0x03,0x00,0x52,0xe1,0x7f,0x02,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,
 0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,
 0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0x78,0x02,0x00,0x1a,
 0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x7a,0x02,0x00,0xea,0x0c,0x10,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x36,0x0f,0x07,0xee,
 0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x20,0x31,0x9f,0xe5,0x03,0x00,0x52,0xe1,
 0x96,0x02,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,
 0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,
 0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0x8f,0x02,0x00,0x1a,0x0c,0x20,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x91,0x02,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x3a,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,
 0xcb,0x05,0x00,0xea,0x45,0x0f,0x52,0xe3,0xac,0x02,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,
 0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,
 0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,
 0xa5,0x02,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0xa7,0x02,0x00,0xea,
 0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,
 0x3e,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x70,0x30,0x9f,0xe5,
 0x03,0x00,0x52,0xe1,0xcb,0x02,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,
 0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,
 0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0xbc,0x02,0x00,0x1a,
 0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0xbe,0x02,0x00,0xea,0x0c,0x10,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x5a,0x0f,0x07,0xee,
 0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x16,0x01,0x00,0x00,0x0d,0x01,0x00,0x00,
 0x0e,0x01,0x00,0x00,0x0f,0x01,0x00,0x00,0x11,0x01,0x00,0x00,0x12,0x01,0x00,0x00,
 0x13,0x01,0x00,0x00,0x15,0x01,0x00,0x00,0x88,0x33,0x9f,0xe5,0x03,0x00,0x52,0xe1,
 0xe2,0x02,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,
 0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,
 0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0xdb,0x02,0x00,0x1a,0x0c,0x20,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0xdd,0x02,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x5e,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,
 0xcb,0x05,0x00,0xea,0x46,0x0f,0x52,0xe3,0xf8,0x02,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,
 0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,
 0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,
 0xf1,0x02,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0xf3,0x02,0x00,0xea,
 0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,
 0x90,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x01,0x0c,0x52,0xe3,
 0xff,0x02,0x00,0x1a,0x01,0x00,0x2d,0xe9,0x30,0x0f,0x10,0xee,0x00,0x00,0xa0,0xe1,
 0x01,0x00,0xbd,0xe8,0xd1,0x05,0x00,0xea,0xbc,0x32,0x9f,0xe5,0x03,0x00,0x52,0xe1,
 0x07,0x03,0x00,0x1a,0x01,0x00,0x2d,0xe9,0x10,0x0f,0x11,0xee,0x00,0x00,0xa0,0xe1,
 0x01,0x00,0xbd,0xe8,0xd1,0x05,0x00,0xea,0xa0,0x32,0x9f,0xe5,0x03,0x00,0x52,0xe1,
 0x1e,0x03,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,
 0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,
 0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0x17,0x03,0x00,0x1a,0x0c,0x20,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x19,0x03,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x10,0x0f,0x01,0xee,0x01,0x00,0xbd,0xe8,
 0xcb,0x05,0x00,0xea,0x48,0x32,0x9f,0xe5,0x03,0x00,0x52,0xe1,0x26,0x03,0x00,0x1a,
 0x01,0x00,0x2d,0xe9,0x10,0x0f,0x12,0xee,0x00,0x00,0xa0,0xe1,0x01,0x00,0xbd,0xe8,
 0xd1,0x05,0x00,0xea,0x41,0x0f,0x52,0xe3,0x3c,0x03,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,
 0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,
 0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,
 0x35,0x03,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x37,0x03,0x00,0xea,
 0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,
 0x10,0x0f,0x02,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0xd4,0x31,0x9f,0xe5,
 0x03,0x00,0x52,0xe1,0x44,0x03,0x00,0x1a,0x01,0x00,0x2d,0xe9,0x10,0x0f,0x13,0xee,
 0x00,0x00,0xa0,0xe1,0x01,0x00,0xbd,0xe8,0xd1,0x05,0x00,0xea,0xb8,0x31,0x9f,0xe5,
 0x03,0x00,0x52,0xe1,0x5b,0x03,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,
 0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,
 0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0x54,0x03,0x00,0x1a,
 0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x56,0x03,0x00,0xea,0x0c,0x10,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x10,0x0f,0x03,0xee,
 0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x60,0x31,0x9f,0xe5,0x03,0x00,0x52,0xe1,
 0x63,0x03,0x00,0x1a,0x01,0x00,0x2d,0xe9,0x10,0x0f,0x15,0xee,0x00,0x00,0xa0,0xe1,
 0x01,0x00,0xbd,0xe8,0xd1,0x05,0x00,0xea,0x42,0x0f,0x52,0xe3,0x79,0x03,0x00,0x1a,
 0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,
 0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,
 0x1c,0x00,0x4b,0xe2,0x72,0x03,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x74,0x03,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,
 0x03,0x00,0xa0,0xe1,0x10,0x0f,0x05,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,
 0xec,0x30,0x9f,0xe5,0x03,0x00,0x52,0xe1,0x81,0x03,0x00,0x1a,0x01,0x00,0x2d,0xe9,
 0x30,0x0f,0x15,0xee,0x00,0x00,0xa0,0xe1,0x01,0x00,0xbd,0xe8,0xd1,0x05,0x00,0xea,
 0xd0,0x30,0x9f,0xe5,0x03,0x00,0x52,0xe1,0x98,0x03,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,
 0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,
 0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,
 0x91,0x03,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x93,0x03,0x00,0xea,
 0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,
 0x30,0x0f,0x05,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x78,0x30,0x9f,0xe5,
 0x03,0x00,0x52,0xe1,0xa0,0x03,0x00,0x1a,0x01,0x00,0x2d,0xe9,0x10,0x0f,0x16,0xee,
 0x00,0x00,0xa0,0xe1,0x01,0x00,0xbd,0xe8,0xd1,0x05,0x00,0xea,0x43,0x0f,0x52,0xe3,
 0xc0,0x03,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,
 0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,
 0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0xb9,0x03,0x00,0x1a,0x0c,0x20,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0xbb,0x03,0x00,0xea,0x17,0x01,0x00,0x00,0x01,0x01,0x00,0x00,
 0x02,0x01,0x00,0x00,0x03,0x01,0x00,0x00,0x05,0x01,0x00,0x00,0x06,0x01,0x00,0x00,
 0x07,0x01,0x00,0x00,0x09,0x01,0x00,0x00,0x0a,0x01,0x00,0x00,0x0b,0x01,0x00,0x00,
 0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,
 0x10,0x0f,0x06,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0xa8,0x33,0x9f,0xe5,
 0x03,0x00,0x52,0xe1,0xd7,0x03,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,
 0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,
 0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0xd0,0x03,0x00,0x1a,
 0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0xd2,0x03,0x00,0xea,0x0c,0x10,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x17,0x0f,0x08,0xee,
 0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x50,0x33,0x9f,0xe5,0x03,0x00,0x52,0xe1,
 0xee,0x03,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,
 0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,
 0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0xe7,0x03,0x00,0x1a,0x0c,0x20,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0xe9,0x03,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x15,0x0f,0x08,0xee,0x01,0x00,0xbd,0xe8,
 0xcb,0x05,0x00,0xea,0xf8,0x32,0x9f,0xe5,0x03,0x00,0x52,0xe1,0x05,0x04,0x00,0x1a,
 0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,
 0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,
 0x1c,0x00,0x4b,0xe2,0xfe,0x03,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x00,0x04,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,
 0x03,0x00,0xa0,0xe1,0x35,0x0f,0x08,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,
 0x47,0x0f,0x52,0xe3,0x1b,0x04,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,
 0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,
 0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0x14,0x04,0x00,0x1a,
 0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x16,0x04,0x00,0xea,0x0c,0x10,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x16,0x0f,0x08,0xee,
 0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x48,0x32,0x9f,0xe5,0x03,0x00,0x52,0xe1,
 0x32,0x04,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,
 0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,
 0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0x2b,0x04,0x00,0x1a,0x0c,0x20,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x2d,0x04,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x36,0x0f,0x08,0xee,0x01,0x00,0xbd,0xe8,
 0xcb,0x05,0x00,0xea,0xf0,0x31,0x9f,0xe5,0x03,0x00,0x52,0xe1,0x3a,0x04,0x00,0x1a,
 0x01,0x00,0x2d,0xe9,0x10,0x0f,0x19,0xee,0x00,0x00,0xa0,0xe1,0x01,0x00,0xbd,0xe8,
 0xd1,0x05,0x00,0xea,0xd4,0x31,0x9f,0xe5,0x03,0x00,0x52,0xe1,0x51,0x04,0x00,0x1a,
 0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,
 0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,
 0x1c,0x00,0x4b,0xe2,0x4a,0x04,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x4c,0x04,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,
 0x03,0x00,0xa0,0xe1,0x10,0x0f,0x09,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,
 0x12,0x0e,0x52,0xe3,0x58,0x04,0x00,0x1a,0x01,0x00,0x2d,0xe9,0x30,0x0f,0x19,0xee,
 0x00,0x00,0xa0,0xe1,0x01,0x00,0xbd,0xe8,0xd1,0x05,0x00,0xea,0x60,0x31,0x9f,0xe5,
 0x03,0x00,0x52,0xe1,0x6f,0x04,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,
 0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,
 0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0x68,0x04,0x00,0x1a,
 0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x6a,0x04,0x00,0xea,0x0c,0x10,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x30,0x0f,0x09,0xee,
 0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x08,0x31,0x9f,0xe5,0x03,0x00,0x52,0xe1,
 0x77,0x04,0x00,0x1a,0x01,0x00,0x2d,0xe9,0x10,0x0f,0x1a,0xee,0x00,0x00,0xa0,0xe1,
 0x01,0x00,0xbd,0xe8,0xd1,0x05,0x00,0xea,0xec,0x30,0x9f,0xe5,0x03,0x00,0x52,0xe1,
 0x8e,0x04,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,
 0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,
 0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,0x87,0x04,0x00,0x1a,0x0c,0x20,0xa0,0xe1,
 0xfe,0xff,0xff,0xeb,0x89,0x04,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x10,0x0f,0x0a,0xee,0x01,0x00,0xbd,0xe8,
 0xcb,0x05,0x00,0xea,0x49,0x0f,0x52,0xe3,0x95,0x04,0x00,0x1a,0x01,0x00,0x2d,0xe9,
 0x30,0x0f,0x1a,0xee,0x00,0x00,0xa0,0xe1,0x01,0x00,0xbd,0xe8,0xd1,0x05,0x00,0xea,
 0x78,0x30,0x9f,0xe5,0x03,0x00,0x52,0xe1,0xb6,0x04,0x00,0x1a,0x7f,0x3d,0xcd,0xe3,
 0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,0x0c,0x20,0x91,0xe0,
 0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0x1c,0x00,0x4b,0xe2,
 0xa5,0x04,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0xa7,0x04,0x00,0xea,
 0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,
 0x30,0x0f,0x0a,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x19,0x01,0x00,0x00,
 0x1a,0x01,0x00,0x00,0x1b,0x01,0x00,0x00,0x1d,0x01,0x00,0x00,0x1e,0x01,0x00,0x00,
 0x1f,0x01,0x00,0x00,0x21,0x01,0x00,0x00,0x22,0x01,0x00,0x00,0x23,0x01,0x00,0x00,
 0x25,0x01,0x00,0x00,0x8c,0x33,0x9f,0xe5,0x03,0x00,0x52,0xe1,0xbe,0x04,0x00,0x1a,
 0x01,0x00,0x2d,0xe9,0x10,0x0f,0x1d,0xee,0x00,0x00,0xa0,0xe1,0x01,0x00,0xbd,0xe8,
 0xd1,0x05,0x00,0xea,0x70,0x33,0x9f,0xe5,0x03,0x00,0x52,0xe1,0xd5,0x04,0x00,0x1a,
 0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x04,0xc0,0xa0,0xe3,0x0c,0x30,0x93,0xe5,
 0x0c,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,
 0x1c,0x00,0x4b,0xe2,0xce,0x04,0x00,0x1a,0x0c,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0xd0,0x04,0x00,0xea,0x0c,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x01,0x00,0x2d,0xe9,
 0x03,0x00,0xa0,0xe1,0x10,0x0f,0x0d,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,
 0x02,0x00,0x52,0xe3,0xda,0x04,0x00,0x1a,0x02,0x00,0x9f,0xef,0x01,0x00,0xa0,0xe3,
 0xd1,0x05,0x00,0xea,0x00,0x00,0x52,0xe3,0xf8,0x04,0x00,0x1a,0x03,0x40,0xa0,0xe3,
 0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,0x0c,0x30,0x93,0xe5,0x04,0x20,0x91,0xe0,
 0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,0x00,0x00,0x53,0xe3,0xea,0x04,0x00,0x1a,
 0x18,0x00,0x4b,0xe2,0x04,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x00,0x40,0xa0,0xe1,
 0xed,0x04,0x00,0xea,0x18,0x00,0x4b,0xe2,0x04,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,
 0x00,0x00,0x54,0xe3,0x0d,0x00,0xe0,0xe3,0xd1,0x05,0x00,0x1a,0xac,0x02,0x9f,0xe5,
 0xfe,0xff,0xff,0xeb,0x18,0x00,0x5b,0xe5,0x17,0x10,0x5b,0xe5,0x16,0x20,0x5b,0xe5,
 0xfe,0xff,0xff,0xeb,0x01,0x00,0xa0,0xe3,0xd1,0x05,0x00,0xea,0x01,0x00,0x52,0xe3,
 0x16,0x05,0x00,0x1a,0x08,0x40,0xa0,0xe3,0x7f,0x3d,0xcd,0xe3,0x3f,0x30,0xc3,0xe3,
 0x0c,0x30,0x93,0xe5,0x04,0x20,0x91,0xe0,0x03,0x20,0xd2,0x30,0x00,0x30,0xa0,0x33,
 0x00,0x00,0x53,0xe3,0x24,0x50,0x4b,0xe2,0x09,0x05,0x00,0x1a,0x05,0x00,0xa0,0xe1,
 0x04,0x20,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x00,0x40,0xa0,0xe1,0x0c,0x05,0x00,0xea,
 0x05,0x00,0xa0,0xe1,0x04,0x10,0xa0,0xe1,0xfe,0xff,0xff,0xeb,0x00,0x00,0x54,0xe3,
 0x0d,0x00,0xe0,0x13,0xd1,0x05,0x00,0x1a,0x01,0x40,0xa0,0xe3,0x24,0x00,0x5b,0xe5,
 0x04,0x10,0xd5,0xe7,0x04,0x20,0x95,0xe5,0xfe,0xff,0xff,0xeb,0x01,0x00,0xa0,0xe3,
 0xd1,0x05,0x00,0xea,0x04,0x00,0x52,0xe3,0x3f,0x05,0x00,0x1a,0x01,0x00,0x2d,0xe9,
 0x10,0x0f,0x12,0xee,0x00,0x30,0xa0,0xe1,0x01,0x00,0xbd,0xe8,0x03,0x41,0x83,0xe2,
 0x00,0xe0,0xa0,0xe3,0x0e,0x21,0x94,0xe7,0x03,0x30,0x02,0xe2,0x01,0x00,0x53,0xe3,
 0x31,0x05,0x00,0x1a,0xff,0x3f,0xc2,0xe3,0x03,0x30,0xc3,0xe3,0x03,0xc1,0x83,0xe2,
 0x00,0x00,0xa0,0xe3,0xd8,0x31,0x9f,0xe5,0x00,0x11,0x9c,0xe7,0xd4,0x21,0x9f,0xe5,
 0x03,0x30,0x01,0xe0,0x02,0x00,0x53,0xe1,0xff,0x3e,0x81,0x03,0x0c,0x30,0x83,0x03,
 0x00,0x31,0x8c,0x07,0x01,0x00,0x80,0xe2,0xff,0x00,0x50,0xe3,0x26,0x05,0x00,0xda,
 0x01,0xe0,0x8e,0xe2,0x01,0x0a,0x5e,0xe3,0x1e,0x05,0x00,0xba,0x00,0x30,0xa0,0xe3,
 0x1c,0x30,0x0b,0xe5,0x01,0x00,0x2d,0xe9,0x02,0x00,0xa0,0xe1,0x9a,0x0f,0x07,0xee,
 0x01,0x00,0xbd,0xe8,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x17,0x0f,0x08,0xee,
 0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x05,0x00,0x52,0xe3,0x6d,0x05,0x00,0x1a,
 0x01,0x00,0x2d,0xe9,0x10,0x0f,0x12,0xee,0x00,0x30,0xa0,0xe1,0x01,0x00,0xbd,0xe8,
 0x03,0x41,0x83,0xe2,0x00,0xc0,0xa0,0xe3,0x0c,0x21,0x94,0xe7,0x03,0x30,0x02,0xe2,
 0x01,0x00,0x53,0xe3,0x5f,0x05,0x00,0x1a,0xff,0x3f,0xc2,0xe3,0x03,0x30,0xc3,0xe3,
 0x03,0xe1,0x83,0xe2,0x00,0x00,0xa0,0xe3,0x34,0x31,0x9f,0xe5,0x00,0x11,0x9e,0xe7,
 0x34,0x21,0x9f,0xe5,0x03,0x30,0x01,0xe0,0x02,0x00,0x53,0xe1,0x5c,0x05,0x00,0x1a,
 0x21,0x38,0xa0,0xe1,0x24,0x21,0x9f,0xe5,0x03,0x38,0xa0,0xe1,0x02,0x00,0x53,0xe1,
 0xff,0x3e,0x81,0x13,0x0c,0x30,0x83,0x13,0x00,0x31,0x8e,0x17,0x01,0x00,0x80,0xe2,
 0xff,0x00,0x50,0xe3,0x4f,0x05,0x00,0xda,0x01,0xc0,0x8c,0xe2,0x01,0x0a,0x5c,0xe3,
 0x47,0x05,0x00,0xba,0x00,0x30,0xa0,0xe3,0x1c,0x30,0x0b,0xe5,0x01,0x00,0x2d,0xe9,
 0x02,0x00,0xa0,0xe1,0x9a,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,0x01,0x00,0x2d,0xe9,
 0x03,0x00,0xa0,0xe1,0x17,0x0f,0x08,0xee,0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,
 0x03,0x00,0x52,0xe3,0xa3,0x05,0x00,0x1a,0x01,0x00,0x2d,0xe9,0x10,0x0f,0x12,0xee,
 0x00,0x30,0xa0,0xe1,0x01,0x00,0xbd,0xe8,0x03,0x41,0x83,0xe2,0x00,0xc0,0xa0,0xe3,
 0x0c,0x21,0x94,0xe7,0x03,0x30,0x02,0xe2,0x01,0x00,0x53,0xe3,0x8d,0x05,0x00,0x1a,
 0xff,0x3f,0xc2,0xe3,0x03,0x30,0xc3,0xe3,0x03,0xe1,0x83,0xe2,0x00,0x00,0xa0,0xe3,
 0x8c,0x30,0x9f,0xe5,0x00,0x11,0x9e,0xe7,0x78,0x20,0x9f,0xe5,0x03,0x30,0x01,0xe0,
 0x02,0x00,0x53,0xe1,0x8a,0x05,0x00,0x1a,0x21,0x38,0xa0,0xe1,0x6c,0x20,0x9f,0xe5,
 0x03,0x38,0xa0,0xe1,0x02,0x00,0x53,0xe1,0xff,0x3e,0x81,0x13,0x0c,0x30,0x83,0x13,
 0x00,0x31,0x8e,0x17,0x01,0x00,0x80,0xe2,0xff,0x00,0x50,0xe3,0x7d,0x05,0x00,0xda,
 0x01,0xc0,0x8c,0xe2,0x01,0x0a,0x5c,0xe3,0x75,0x05,0x00,0xba,0x00,0x30,0xa0,0xe3,
 0x1c,0x30,0x0b,0xe5,0x01,0x00,0x2d,0xe9,0x02,0x00,0xa0,0xe1,0x9a,0x0f,0x07,0xee,
 0x01,0x00,0xbd,0xe8,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,0x17,0x0f,0x08,0xee,
 0x01,0x00,0xbd,0xe8,0xcb,0x05,0x00,0xea,0x26,0x01,0x00,0x00,0x27,0x01,0x00,0x00,
 0x78,0x97,0x99,0x01,0x0f,0x00,0x00,0xff,0x02,0x00,0x00,0x02,0x02,0x00,0x00,0x03,
 0x00,0x00,0xff,0x03,0x0f,0x00,0x00,0xfe,0x06,0x00,0x52,0xe3,0xd0,0x05,0x00,0x1a,
 0x01,0x00,0x2d,0xe9,0x10,0x0f,0x12,0xee,0x00,0x30,0xa0,0xe1,0x01,0x00,0xbd,0xe8,
 0x03,0x41,0x83,0xe2,0x00,0xe0,0xa0,0xe3,0x0e,0x21,0x94,0xe7,0x03,0x30,0x02,0xe2,
 0x01,0x00,0x53,0xe3,0xbe,0x05,0x00,0x1a,0xff,0x3f,0xc2,0xe3,0x03,0x30,0xc3,0xe3,
 0x03,0xc1,0x83,0xe2,0x00,0x00,0xa0,0xe3,0x64,0x30,0x9f,0xe5,0x00,0x11,0x9c,0xe7,
 0x60,0x20,0x9f,0xe5,0x03,0x30,0x01,0xe0,0x02,0x00,0x53,0xe1,0xff,0x3e,0xc1,0x03,
 0x0c,0x30,0xc3,0x03,0x00,0x31,0x8c,0x07,0x01,0x00,0x80,0xe2,0xff,0x00,0x50,0xe3,
 0xb3,0x05,0x00,0xda,0x01,0xe0,0x8e,0xe2,0x01,0x0a,0x5e,0xe3,0xab,0x05,0x00,0xba,
 0x00,0x30,0xa0,0xe3,0x1c,0x30,0x0b,0xe5,0x01,0x00,0x2d,0xe9,0x02,0x00,0xa0,0xe1,
 0x9a,0x0f,0x07,0xee,0x01,0x00,0xbd,0xe8,0x01,0x00,0x2d,0xe9,0x03,0x00,0xa0,0xe1,
 0x17,0x0f,0x08,0xee,0x01,0x00,0xbd,0xe8,0x01,0x00,0xa0,0xe3,0x1c,0x30,0x0b,0xe5,
 0xd1,0x05,0x00,0xea,0x0f,0x00,0x00,0xfe,0x0e,0x00,0x00,0x02,0x15,0x00,0xe0,0xe3,
 0x30,0xa8,0x1b,0xe9,0x0d,0xc0,0xa0,0xe1,0x00,0xd8,0x2d,0xe9,0x0c,0x00,0x9f,0xe5,
 0x04,0xb0,0x4c,0xe2,0xfe,0xff,0xff,0xeb,0x00,0x00,0xa0,0xe3,0x00,0xa8,0x1b,0xe9,
 0x48,0x00,0x00,0x00,0x0d,0xc0,0xa0,0xe1,0x00,0xd8,0x2d,0xe9,0x08,0x00,0x9f,0xe5,
 0x04,0xb0,0x4c,0xe2,0xfe,0xff,0xff,0xeb,0x00,0xa8,0x1b,0xe9,0x48,0x00,0x00,0x00,
 0x6d,0x69,0x6e,0x69,0x6d,0x61,0x6c,0x00,0x6b,0x65,0x72,0x6e,0x65,0x6c,0x5f,0x76,
 0x65,0x72,0x73,0x69,0x6f,0x6e,0x3d,0x32,0x2e,0x34,0x2e,0x32,0x35,0x00,0x00,0x00,
 0x75,0x73,0x69,0x6e,0x67,0x5f,0x63,0x68,0x65,0x63,0x6b,0x73,0x75,0x6d,0x73,0x3d,
 0x31,0x00,0x00,0x00,0x61,0x75,0x74,0x68,0x6f,0x72,0x3d,0x72,0x6c,0x79,0x65,0x68,
 0x20,0x2f,0x20,0x77,0x77,0x77,0x2e,0x72,0x65,0x74,0x72,0x6f,0x64,0x65,0x76,0x2e,
 0x69,0x6e,0x66,0x6f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x70,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x07,0x00,0x00,0x00,0x00,0x00,0x00,
 0x5c,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0xff,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x47,0x43,0x43,0x3a,0x20,0x28,0x47,
 0x4e,0x55,0x29,0x20,0x32,0x2e,0x39,0x35,0x2e,0x33,0x20,0x32,0x30,0x30,0x31,0x30,
 0x33,0x31,0x35,0x20,0x28,0x72,0x65,0x6c,0x65,0x61,0x73,0x65,0x29,0x00,0x00,0x2e,
 0x73,0x79,0x6d,0x74,0x61,0x62,0x00,0x2e,0x73,0x74,0x72,0x74,0x61,0x62,0x00,0x2e,
 0x73,0x68,0x73,0x74,0x72,0x74,0x61,0x62,0x00,0x2e,0x74,0x65,0x78,0x74,0x00,0x2e,
 0x72,0x65,0x6c,0x2e,0x74,0x65,0x78,0x74,0x00,0x2e,0x72,0x6f,0x64,0x61,0x74,0x61,
 0x00,0x2e,0x6d,0x6f,0x64,0x69,0x6e,0x66,0x6f,0x00,0x2e,0x64,0x61,0x74,0x61,0x00,
 0x2e,0x72,0x65,0x6c,0x2e,0x64,0x61,0x74,0x61,0x00,0x2e,0x62,0x73,0x73,0x00,0x2e,
 0x63,0x6f,0x6d,0x6d,0x65,0x6e,0x74,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x1b,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x34,0x00,0x00,0x00,0x8c,0x17,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x21,0x00,0x00,0x00,0x09,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xd8,0x1a,0x00,0x00,0x20,0x0d,0x00,0x00,
 0x0a,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x08,0x00,0x00,0x00,
 0x2b,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0xc0,0x17,0x00,0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x33,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
 0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xc8,0x17,0x00,0x00,0x50,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x3c,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x18,0x18,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x42,0x00,0x00,0x00,0x09,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xf8,0x27,0x00,0x00,0x30,0x00,0x00,0x00,
 0x0a,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x08,0x00,0x00,0x00,
 0x4c,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x78,0x18,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x51,0x00,0x00,0x00,0x01,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x18,0x00,0x00,0x26,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x11,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x9e,0x18,0x00,0x00,0x5a,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x02,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x28,0x28,0x00,0x00,0x80,0x02,0x00,0x00,
 0x0b,0x00,0x00,0x00,0x17,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x10,0x00,0x00,0x00,
 0x09,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0xa8,0x2a,0x00,0x00,0x83,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x14,0x00,0x00,0x00,0x01,0x01,0x00,0x00,
 0x1c,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0x44,0x00,0x00,0x00,0x01,0x01,0x00,0x00,
 0x5c,0x00,0x00,0x00,0x01,0x01,0x00,0x00,0x78,0x00,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x84,0x00,0x00,0x00,0x01,0x1b,0x00,0x00,0x88,0x00,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x94,0x00,0x00,0x00,0x01,0x1b,0x00,0x00,0x98,0x00,0x00,0x00,0x01,0x1c,0x00,0x00,
 0xa4,0x00,0x00,0x00,0x01,0x1b,0x00,0x00,0xb4,0x00,0x00,0x00,0x01,0x20,0x00,0x00,
 0xc4,0x00,0x00,0x00,0x01,0x20,0x00,0x00,0xf8,0x00,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x04,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,0x14,0x01,0x00,0x00,0x01,0x20,0x00,0x00,
 0x24,0x01,0x00,0x00,0x01,0x20,0x00,0x00,0x28,0x01,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x34,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,0x40,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x44,0x01,0x00,0x00,0x01,0x1c,0x00,0x00,0x50,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x54,0x01,0x00,0x00,0x01,0x1c,0x00,0x00,0x60,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x64,0x01,0x00,0x00,0x01,0x1c,0x00,0x00,0x68,0x01,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x6c,0x01,0x00,0x00,0x01,0x1c,0x00,0x00,0x78,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x7c,0x01,0x00,0x00,0x01,0x1c,0x00,0x00,0x88,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x98,0x01,0x00,0x00,0x01,0x01,0x00,0x00,0xa4,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xa8,0x01,0x00,0x00,0x01,0x01,0x00,0x00,0xb0,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xb4,0x01,0x00,0x00,0x01,0x1c,0x00,0x00,0xc0,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xc4,0x01,0x00,0x00,0x01,0x1c,0x00,0x00,0xd0,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xd8,0x01,0x00,0x00,0x01,0x01,0x00,0x00,0xe8,0x01,0x00,0x00,0x01,0x20,0x00,0x00,
 0xec,0x01,0x00,0x00,0x01,0x1c,0x00,0x00,0xf8,0x01,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x00,0x02,0x00,0x00,0x01,0x26,0x00,0x00,0x08,0x02,0x00,0x00,0x01,0x01,0x00,0x00,
 0x0c,0x02,0x00,0x00,0x01,0x1d,0x00,0x00,0x10,0x02,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x1c,0x02,0x00,0x00,0x01,0x1b,0x00,0x00,0x2c,0x02,0x00,0x00,0x01,0x20,0x00,0x00,
 0x3c,0x02,0x00,0x00,0x01,0x01,0x00,0x00,0x48,0x02,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x4c,0x02,0x00,0x00,0x01,0x01,0x00,0x00,0x54,0x02,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x58,0x02,0x00,0x00,0x01,0x1c,0x00,0x00,0x64,0x02,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x68,0x02,0x00,0x00,0x01,0x1c,0x00,0x00,0x74,0x02,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x7c,0x02,0x00,0x00,0x01,0x01,0x00,0x00,0x8c,0x02,0x00,0x00,0x01,0x20,0x00,0x00,
 0x90,0x02,0x00,0x00,0x01,0x1c,0x00,0x00,0x9c,0x02,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xa4,0x02,0x00,0x00,0x01,0x26,0x00,0x00,0xac,0x02,0x00,0x00,0x01,0x01,0x00,0x00,
 0xb0,0x02,0x00,0x00,0x01,0x1d,0x00,0x00,0xb4,0x02,0x00,0x00,0x01,0x1c,0x00,0x00,
 0xc0,0x02,0x00,0x00,0x01,0x1b,0x00,0x00,0xd0,0x02,0x00,0x00,0x01,0x20,0x00,0x00,
 0xe0,0x02,0x00,0x00,0x01,0x01,0x00,0x00,0xec,0x02,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xf0,0x02,0x00,0x00,0x01,0x01,0x00,0x00,0xf8,0x02,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xfc,0x02,0x00,0x00,0x01,0x1c,0x00,0x00,0x08,0x03,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x0c,0x03,0x00,0x00,0x01,0x1c,0x00,0x00,0x18,0x03,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x20,0x03,0x00,0x00,0x01,0x01,0x00,0x00,0x30,0x03,0x00,0x00,0x01,0x20,0x00,0x00,
 0x34,0x03,0x00,0x00,0x01,0x1c,0x00,0x00,0x40,0x03,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x48,0x03,0x00,0x00,0x01,0x26,0x00,0x00,0x50,0x03,0x00,0x00,0x01,0x01,0x00,0x00,
 0x54,0x03,0x00,0x00,0x01,0x1d,0x00,0x00,0x58,0x03,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x64,0x03,0x00,0x00,0x01,0x1b,0x00,0x00,0x70,0x03,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x74,0x03,0x00,0x00,0x01,0x1c,0x00,0x00,0x80,0x03,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x84,0x03,0x00,0x00,0x01,0x1c,0x00,0x00,0x90,0x03,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xa0,0x03,0x00,0x00,0x01,0x20,0x00,0x00,0xb0,0x03,0x00,0x00,0x01,0x20,0x00,0x00,
 0xe0,0x03,0x00,0x00,0x01,0x1b,0x00,0x00,0xec,0x03,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xfc,0x03,0x00,0x00,0x01,0x20,0x00,0x00,0x0c,0x04,0x00,0x00,0x01,0x20,0x00,0x00,
 0x10,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,0x1c,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x28,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,0x2c,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x38,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,0x3c,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x48,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,0x4c,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x50,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,0x54,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x60,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,0x64,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x70,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,0x80,0x04,0x00,0x00,0x01,0x01,0x00,0x00,
 0x8c,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,0x90,0x04,0x00,0x00,0x01,0x01,0x00,0x00,
 0x98,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,0x9c,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,
 0xa8,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,0xac,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,
 0xb8,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,0xc0,0x04,0x00,0x00,0x01,0x01,0x00,0x00,
 0xd0,0x04,0x00,0x00,0x01,0x20,0x00,0x00,0xd4,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,
 0xe0,0x04,0x00,0x00,0x01,0x1b,0x00,0x00,0xe8,0x04,0x00,0x00,0x01,0x26,0x00,0x00,
 0xf0,0x04,0x00,0x00,0x01,0x01,0x00,0x00,0xf4,0x04,0x00,0x00,0x01,0x1d,0x00,0x00,
 0xf8,0x04,0x00,0x00,0x01,0x1c,0x00,0x00,0x04,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x14,0x05,0x00,0x00,0x01,0x20,0x00,0x00,0x24,0x05,0x00,0x00,0x01,0x01,0x00,0x00,
 0x30,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,0x34,0x05,0x00,0x00,0x01,0x01,0x00,0x00,
 0x3c,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,0x40,0x05,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x4c,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,0x50,0x05,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x5c,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,0x64,0x05,0x00,0x00,0x01,0x01,0x00,0x00,
 0x74,0x05,0x00,0x00,0x01,0x20,0x00,0x00,0x78,0x05,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x84,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,0x8c,0x05,0x00,0x00,0x01,0x26,0x00,0x00,
 0x94,0x05,0x00,0x00,0x01,0x01,0x00,0x00,0x98,0x05,0x00,0x00,0x01,0x1d,0x00,0x00,
 0x9c,0x05,0x00,0x00,0x01,0x1c,0x00,0x00,0xa8,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xb4,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,0xc4,0x05,0x00,0x00,0x01,0x20,0x00,0x00,
 0xc8,0x05,0x00,0x00,0x01,0x1c,0x00,0x00,0xd4,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xd8,0x05,0x00,0x00,0x01,0x1c,0x00,0x00,0xe4,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xe8,0x05,0x00,0x00,0x01,0x1c,0x00,0x00,0xec,0x05,0x00,0x00,0x01,0x1c,0x00,0x00,
 0xf0,0x05,0x00,0x00,0x01,0x1c,0x00,0x00,0xfc,0x05,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x00,0x06,0x00,0x00,0x01,0x1c,0x00,0x00,0x0c,0x06,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x20,0x06,0x00,0x00,0x01,0x01,0x00,0x00,0x2c,0x06,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x30,0x06,0x00,0x00,0x01,0x01,0x00,0x00,0x38,0x06,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x3c,0x06,0x00,0x00,0x01,0x1c,0x00,0x00,0x48,0x06,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x4c,0x06,0x00,0x00,0x01,0x1c,0x00,0x00,0x58,0x06,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x60,0x06,0x00,0x00,0x01,0x01,0x00,0x00,0x70,0x06,0x00,0x00,0x01,0x20,0x00,0x00,
 0x74,0x06,0x00,0x00,0x01,0x1c,0x00,0x00,0x80,0x06,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x88,0x06,0x00,0x00,0x01,0x26,0x00,0x00,0x90,0x06,0x00,0x00,0x01,0x01,0x00,0x00,
 0x94,0x06,0x00,0x00,0x01,0x1d,0x00,0x00,0x98,0x06,0x00,0x00,0x01,0x1c,0x00,0x00,
 0xa4,0x06,0x00,0x00,0x01,0x1b,0x00,0x00,0xb0,0x06,0x00,0x00,0x01,0x1c,0x00,0x00,
 0xbc,0x06,0x00,0x00,0x01,0x1b,0x00,0x00,0xc4,0x06,0x00,0x00,0x01,0x26,0x00,0x00,
 0xdc,0x06,0x00,0x00,0x01,0x1c,0x00,0x00,0xe8,0x06,0x00,0x00,0x01,0x1b,0x00,0x00,
 0xf0,0x06,0x00,0x00,0x01,0x01,0x00,0x00,0xfc,0x06,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x0c,0x07,0x00,0x00,0x01,0x20,0x00,0x00,0x10,0x07,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x1c,0x07,0x00,0x00,0x01,0x1b,0x00,0x00,0x20,0x07,0x00,0x00,0x01,0x1c,0x00,0x00,
 0x2c,0x07,0x00,0x00,0x01,0x1b,0x00,0x00,0x38,0x07,0x00,0x00,0x01,0x1b,0x00,0x00,
 0x8c,0x07,0x00,0x00,0x01,0x01,0x00,0x00,0xb4,0x07,0x00,0x00,0x01,0x01,0x00,0x00,
 0xbc,0x07,0x00,0x00,0x01,0x24,0x00,0x00,0xc0,0x07,0x00,0x00,0x01,0x01,0x00,0x00,
 0xc8,0x07,0x00,0x00,0x01,0x22,0x00,0x00,0xdc,0x07,0x00,0x00,0x01,0x01,0x00,0x00,
 0xe8,0x07,0x00,0x00,0x01,0x01,0x00,0x00,0x10,0x08,0x00,0x00,0x01,0x01,0x00,0x00,
 0x18,0x08,0x00,0x00,0x01,0x24,0x00,0x00,0x1c,0x08,0x00,0x00,0x01,0x01,0x00,0x00,
 0x24,0x08,0x00,0x00,0x01,0x22,0x00,0x00,0x38,0x08,0x00,0x00,0x01,0x01,0x00,0x00,
 0x44,0x08,0x00,0x00,0x01,0x01,0x00,0x00,0x6c,0x08,0x00,0x00,0x01,0x01,0x00,0x00,
 0x74,0x08,0x00,0x00,0x01,0x24,0x00,0x00,0x78,0x08,0x00,0x00,0x01,0x01,0x00,0x00,
 0x80,0x08,0x00,0x00,0x01,0x22,0x00,0x00,0x94,0x08,0x00,0x00,0x01,0x01,0x00,0x00,
 0xa0,0x08,0x00,0x00,0x01,0x01,0x00,0x00,0xc8,0x08,0x00,0x00,0x01,0x01,0x00,0x00,
 0xd0,0x08,0x00,0x00,0x01,0x24,0x00,0x00,0xd4,0x08,0x00,0x00,0x01,0x01,0x00,0x00,
 0xdc,0x08,0x00,0x00,0x01,0x22,0x00,0x00,0xf0,0x08,0x00,0x00,0x01,0x01,0x00,0x00,
 0xf8,0x08,0x00,0x00,0x01,0x01,0x00,0x00,0x20,0x09,0x00,0x00,0x01,0x01,0x00,0x00,
 0x28,0x09,0x00,0x00,0x01,0x24,0x00,0x00,0x2c,0x09,0x00,0x00,0x01,0x01,0x00,0x00,
 0x34,0x09,0x00,0x00,0x01,0x22,0x00,0x00,0x48,0x09,0x00,0x00,0x01,0x01,0x00,0x00,
 0x54,0x09,0x00,0x00,0x01,0x01,0x00,0x00,0x7c,0x09,0x00,0x00,0x01,0x01,0x00,0x00,
 0x84,0x09,0x00,0x00,0x01,0x24,0x00,0x00,0x88,0x09,0x00,0x00,0x01,0x01,0x00,0x00,
 0x90,0x09,0x00,0x00,0x01,0x22,0x00,0x00,0xa4,0x09,0x00,0x00,0x01,0x01,0x00,0x00,
 0xb0,0x09,0x00,0x00,0x01,0x01,0x00,0x00,0xd8,0x09,0x00,0x00,0x01,0x01,0x00,0x00,
 0xe0,0x09,0x00,0x00,0x01,0x24,0x00,0x00,0xe4,0x09,0x00,0x00,0x01,0x01,0x00,0x00,
 0xec,0x09,0x00,0x00,0x01,0x22,0x00,0x00,0x00,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,
 0x0c,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,0x34,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,
 0x3c,0x0a,0x00,0x00,0x01,0x24,0x00,0x00,0x40,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,
 0x48,0x0a,0x00,0x00,0x01,0x22,0x00,0x00,0x5c,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,
 0x64,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,0x8c,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,
 0x94,0x0a,0x00,0x00,0x01,0x24,0x00,0x00,0x98,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,
 0xa0,0x0a,0x00,0x00,0x01,0x22,0x00,0x00,0xb4,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,
 0xc0,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,0xe8,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,
 0xf0,0x0a,0x00,0x00,0x01,0x24,0x00,0x00,0xf4,0x0a,0x00,0x00,0x01,0x01,0x00,0x00,
 0xfc,0x0a,0x00,0x00,0x01,0x22,0x00,0x00,0x10,0x0b,0x00,0x00,0x01,0x01,0x00,0x00,
 0x3c,0x0b,0x00,0x00,0x01,0x01,0x00,0x00,0x64,0x0b,0x00,0x00,0x01,0x01,0x00,0x00,
 0x6c,0x0b,0x00,0x00,0x01,0x24,0x00,0x00,0x70,0x0b,0x00,0x00,0x01,0x01,0x00,0x00,
 0x78,0x0b,0x00,0x00,0x01,0x22,0x00,0x00,0x8c,0x0b,0x00,0x00,0x01,0x01,0x00,0x00,
 0x94,0x0b,0x00,0x00,0x01,0x01,0x00,0x00,0xbc,0x0b,0x00,0x00,0x01,0x01,0x00,0x00,
 0xc4,0x0b,0x00,0x00,0x01,0x24,0x00,0x00,0xc8,0x0b,0x00,0x00,0x01,0x01,0x00,0x00,
 0xd0,0x0b,0x00,0x00,0x01,0x22,0x00,0x00,0xe4,0x0b,0x00,0x00,0x01,0x01,0x00,0x00,
 0xec,0x0b,0x00,0x00,0x01,0x01,0x00,0x00,0x00,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,
 0x0c,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,0x20,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,
 0x2c,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,0x54,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,
 0x5c,0x0c,0x00,0x00,0x01,0x24,0x00,0x00,0x60,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,
 0x68,0x0c,0x00,0x00,0x01,0x22,0x00,0x00,0x7c,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,
 0x88,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,0x9c,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,
 0xa4,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,0xcc,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,
 0xd4,0x0c,0x00,0x00,0x01,0x24,0x00,0x00,0xd8,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,
 0xe0,0x0c,0x00,0x00,0x01,0x22,0x00,0x00,0xf4,0x0c,0x00,0x00,0x01,0x01,0x00,0x00,
 0x00,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,0x14,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,
 0x20,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,0x48,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,
 0x50,0x0d,0x00,0x00,0x01,0x24,0x00,0x00,0x54,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,
 0x5c,0x0d,0x00,0x00,0x01,0x22,0x00,0x00,0x70,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,
 0x7c,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,0x90,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,
 0x98,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,0xc0,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,
 0xc8,0x0d,0x00,0x00,0x01,0x24,0x00,0x00,0xcc,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,
 0xd4,0x0d,0x00,0x00,0x01,0x22,0x00,0x00,0xe8,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,
 0xf4,0x0d,0x00,0x00,0x01,0x01,0x00,0x00,0x08,0x0e,0x00,0x00,0x01,0x01,0x00,0x00,
 0x14,0x0e,0x00,0x00,0x01,0x01,0x00,0x00,0x3c,0x0e,0x00,0x00,0x01,0x01,0x00,0x00,
 0x44,0x0e,0x00,0x00,0x01,0x24,0x00,0x00,0x48,0x0e,0x00,0x00,0x01,0x01,0x00,0x00,
 0x50,0x0e,0x00,0x00,0x01,0x22,0x00,0x00,0x64,0x0e,0x00,0x00,0x01,0x01,0x00,0x00,
 0x70,0x0e,0x00,0x00,0x01,0x01,0x00,0x00,0x84,0x0e,0x00,0x00,0x01,0x01,0x00,0x00,
 0x8c,0x0e,0x00,0x00,0x01,0x01,0x00,0x00,0xb4,0x0e,0x00,0x00,0x01,0x01,0x00,0x00,
 0xbc,0x0e,0x00,0x00,0x01,0x24,0x00,0x00,0xc0,0x0e,0x00,0x00,0x01,0x01,0x00,0x00,
 0xf0,0x0e,0x00,0x00,0x01,0x22,0x00,0x00,0x04,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,
 0x10,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,0x38,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,
 0x40,0x0f,0x00,0x00,0x01,0x24,0x00,0x00,0x44,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,
 0x4c,0x0f,0x00,0x00,0x01,0x22,0x00,0x00,0x60,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,
 0x6c,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,0x94,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,
 0x9c,0x0f,0x00,0x00,0x01,0x24,0x00,0x00,0xa0,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,
 0xa8,0x0f,0x00,0x00,0x01,0x22,0x00,0x00,0xbc,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,
 0xc8,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,0xf0,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,
 0xf8,0x0f,0x00,0x00,0x01,0x24,0x00,0x00,0xfc,0x0f,0x00,0x00,0x01,0x01,0x00,0x00,
 0x04,0x10,0x00,0x00,0x01,0x22,0x00,0x00,0x18,0x10,0x00,0x00,0x01,0x01,0x00,0x00,
 0x20,0x10,0x00,0x00,0x01,0x01,0x00,0x00,0x48,0x10,0x00,0x00,0x01,0x01,0x00,0x00,
 0x50,0x10,0x00,0x00,0x01,0x24,0x00,0x00,0x54,0x10,0x00,0x00,0x01,0x01,0x00,0x00,
 0x5c,0x10,0x00,0x00,0x01,0x22,0x00,0x00,0x70,0x10,0x00,0x00,0x01,0x01,0x00,0x00,
 0x7c,0x10,0x00,0x00,0x01,0x01,0x00,0x00,0xa4,0x10,0x00,0x00,0x01,0x01,0x00,0x00,
 0xac,0x10,0x00,0x00,0x01,0x24,0x00,0x00,0xb0,0x10,0x00,0x00,0x01,0x01,0x00,0x00,
 0xb8,0x10,0x00,0x00,0x01,0x22,0x00,0x00,0xcc,0x10,0x00,0x00,0x01,0x01,0x00,0x00,
 0xd8,0x10,0x00,0x00,0x01,0x01,0x00,0x00,0xec,0x10,0x00,0x00,0x01,0x01,0x00,0x00,
 0xf8,0x10,0x00,0x00,0x01,0x01,0x00,0x00,0x20,0x11,0x00,0x00,0x01,0x01,0x00,0x00,
 0x28,0x11,0x00,0x00,0x01,0x24,0x00,0x00,0x2c,0x11,0x00,0x00,0x01,0x01,0x00,0x00,
 0x34,0x11,0x00,0x00,0x01,0x22,0x00,0x00,0x48,0x11,0x00,0x00,0x01,0x01,0x00,0x00,
 0x50,0x11,0x00,0x00,0x01,0x01,0x00,0x00,0x64,0x11,0x00,0x00,0x01,0x01,0x00,0x00,
 0x70,0x11,0x00,0x00,0x01,0x01,0x00,0x00,0x98,0x11,0x00,0x00,0x01,0x01,0x00,0x00,
 0xa0,0x11,0x00,0x00,0x01,0x24,0x00,0x00,0xa4,0x11,0x00,0x00,0x01,0x01,0x00,0x00,
 0xac,0x11,0x00,0x00,0x01,0x22,0x00,0x00,0xc0,0x11,0x00,0x00,0x01,0x01,0x00,0x00,
 0xcc,0x11,0x00,0x00,0x01,0x01,0x00,0x00,0xe0,0x11,0x00,0x00,0x01,0x01,0x00,0x00,
 0xec,0x11,0x00,0x00,0x01,0x01,0x00,0x00,0x14,0x12,0x00,0x00,0x01,0x01,0x00,0x00,
 0x1c,0x12,0x00,0x00,0x01,0x24,0x00,0x00,0x20,0x12,0x00,0x00,0x01,0x01,0x00,0x00,
 0x28,0x12,0x00,0x00,0x01,0x22,0x00,0x00,0x3c,0x12,0x00,0x00,0x01,0x01,0x00,0x00,
 0x44,0x12,0x00,0x00,0x01,0x01,0x00,0x00,0x58,0x12,0x00,0x00,0x01,0x01,0x00,0x00,
 0x64,0x12,0x00,0x00,0x01,0x01,0x00,0x00,0x8c,0x12,0x00,0x00,0x01,0x01,0x00,0x00,
 0x94,0x12,0x00,0x00,0x01,0x24,0x00,0x00,0x98,0x12,0x00,0x00,0x01,0x01,0x00,0x00,
 0xa0,0x12,0x00,0x00,0x01,0x22,0x00,0x00,0xb4,0x12,0x00,0x00,0x01,0x01,0x00,0x00,
 0xe8,0x12,0x00,0x00,0x01,0x01,0x00,0x00,0xfc,0x12,0x00,0x00,0x01,0x01,0x00,0x00,
 0x08,0x13,0x00,0x00,0x01,0x01,0x00,0x00,0x30,0x13,0x00,0x00,0x01,0x01,0x00,0x00,
 0x38,0x13,0x00,0x00,0x01,0x24,0x00,0x00,0x3c,0x13,0x00,0x00,0x01,0x01,0x00,0x00,
 0x44,0x13,0x00,0x00,0x01,0x22,0x00,0x00,0x58,0x13,0x00,0x00,0x01,0x01,0x00,0x00,
 0x60,0x13,0x00,0x00,0x01,0x01,0x00,0x00,0x6c,0x13,0x00,0x00,0x01,0x01,0x00,0x00,
 0x74,0x13,0x00,0x00,0x01,0x01,0x00,0x00,0x98,0x13,0x00,0x00,0x01,0x01,0x00,0x00,
 0xa4,0x13,0x00,0x00,0x01,0x24,0x00,0x00,0xac,0x13,0x00,0x00,0x01,0x01,0x00,0x00,
 0xb8,0x13,0x00,0x00,0x01,0x22,0x00,0x00,0xc4,0x13,0x00,0x00,0x01,0x01,0x00,0x00,
 0xcc,0x13,0x00,0x00,0x01,0x27,0x00,0x00,0xdc,0x13,0x00,0x00,0x01,0x21,0x00,0x00,
 0xe4,0x13,0x00,0x00,0x01,0x01,0x00,0x00,0xec,0x13,0x00,0x00,0x01,0x01,0x00,0x00,
 0x14,0x14,0x00,0x00,0x01,0x01,0x00,0x00,0x20,0x14,0x00,0x00,0x01,0x24,0x00,0x00,
 0x28,0x14,0x00,0x00,0x01,0x01,0x00,0x00,0x34,0x14,0x00,0x00,0x01,0x22,0x00,0x00,
 0x40,0x14,0x00,0x00,0x01,0x01,0x00,0x00,0x54,0x14,0x00,0x00,0x01,0x25,0x00,0x00,
 0x5c,0x14,0x00,0x00,0x01,0x01,0x00,0x00,0x64,0x14,0x00,0x00,0x01,0x01,0x00,0x00,
 0x8c,0x14,0x00,0x00,0x01,0x01,0x00,0x00,0xc8,0x14,0x00,0x00,0x01,0x01,0x00,0x00,
 0xd4,0x14,0x00,0x00,0x01,0x01,0x00,0x00,0x00,0x15,0x00,0x00,0x01,0x01,0x00,0x00,
 0x08,0x15,0x00,0x00,0x01,0x01,0x00,0x00,0x30,0x15,0x00,0x00,0x01,0x01,0x00,0x00,
 0x58,0x15,0x00,0x00,0x01,0x01,0x00,0x00,0x80,0x15,0x00,0x00,0x01,0x01,0x00,0x00,
 0x8c,0x15,0x00,0x00,0x01,0x01,0x00,0x00,0xb8,0x15,0x00,0x00,0x01,0x01,0x00,0x00,
 0xc0,0x15,0x00,0x00,0x01,0x01,0x00,0x00,0xe8,0x15,0x00,0x00,0x01,0x01,0x00,0x00,
 0x10,0x16,0x00,0x00,0x01,0x01,0x00,0x00,0x38,0x16,0x00,0x00,0x01,0x01,0x00,0x00,
 0x44,0x16,0x00,0x00,0x01,0x01,0x00,0x00,0x70,0x16,0x00,0x00,0x01,0x01,0x00,0x00,
 0x98,0x16,0x00,0x00,0x01,0x01,0x00,0x00,0xc0,0x16,0x00,0x00,0x01,0x01,0x00,0x00,
 0xfc,0x16,0x00,0x00,0x01,0x01,0x00,0x00,0x08,0x17,0x00,0x00,0x01,0x01,0x00,0x00,
 0x3c,0x17,0x00,0x00,0x01,0x01,0x00,0x00,0x60,0x17,0x00,0x00,0x01,0x23,0x00,0x00,
 0x6c,0x17,0x00,0x00,0x02,0x05,0x00,0x00,0x80,0x17,0x00,0x00,0x01,0x17,0x00,0x00,
 0x88,0x17,0x00,0x00,0x02,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x02,0x18,0x00,0x00,
 0x18,0x00,0x00,0x00,0x02,0x01,0x00,0x00,0x20,0x00,0x00,0x00,0x02,0x01,0x00,0x00,
 0x28,0x00,0x00,0x00,0x02,0x01,0x00,0x00,0x4c,0x00,0x00,0x00,0x02,0x03,0x00,0x00,
 0x50,0x00,0x00,0x00,0x02,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x09,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x0a,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x03,0x00,0x0b,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x04,0x00,0xf1,0xff,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x17,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x16,0x00,0x00,0x00,0x01,0x00,0x04,0x00,0x2f,0x00,0x00,0x00,0x18,0x00,0x00,0x00,
 0x12,0x00,0x00,0x00,0x01,0x00,0x04,0x00,0x48,0x00,0x00,0x00,0x48,0x07,0x00,0x00,
 0x14,0x00,0x00,0x00,0x02,0x00,0x01,0x00,0x55,0x00,0x00,0x00,0x5c,0x07,0x00,0x00,
 0x14,0x00,0x00,0x00,0x02,0x00,0x01,0x00,0x65,0x00,0x00,0x00,0x70,0x07,0x00,0x00,
 0xe0,0x0f,0x00,0x00,0x02,0x00,0x01,0x00,0x73,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x48,0x00,0x00,0x00,0x01,0x00,0x05,0x00,0x80,0x00,0x00,0x00,0x48,0x00,0x00,0x00,
 0x18,0x00,0x00,0x00,0x01,0x00,0x05,0x00,0x88,0x00,0x00,0x00,0x50,0x17,0x00,0x00,
 0x20,0x00,0x00,0x00,0x02,0x00,0x01,0x00,0x95,0x00,0x00,0x00,0x70,0x17,0x00,0x00,
 0x1c,0x00,0x00,0x00,0x02,0x00,0x01,0x00,0xa2,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0xb2,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0xc0,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,
 0x21,0x00,0x00,0x00,0x11,0x00,0x04,0x00,0xd0,0x00,0x00,0x00,0x70,0x17,0x00,0x00,
 0x1c,0x00,0x00,0x00,0x12,0x00,0x01,0x00,0xdf,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0xee,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
 0x64,0x00,0x00,0x00,0x12,0x00,0x01,0x00,0xf8,0x00,0x00,0x00,0x64,0x00,0x00,0x00,
 0x6c,0x00,0x00,0x00,0x12,0x00,0x01,0x00,0x03,0x01,0x00,0x00,0x50,0x17,0x00,0x00,
 0x20,0x00,0x00,0x00,0x12,0x00,0x01,0x00,0x0f,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
 0x04,0x00,0x00,0x00,0x11,0x00,0x07,0x00,0x17,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x25,0x01,0x00,0x00,0xd0,0x00,0x00,0x00,
 0xec,0x02,0x00,0x00,0x12,0x00,0x01,0x00,0x2f,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x39,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x47,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x5d,0x01,0x00,0x00,0xbc,0x03,0x00,0x00,
 0x8c,0x03,0x00,0x00,0x12,0x00,0x01,0x00,0x67,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x75,0x01,0x00,0x00,0x00,0x00,0x00,0x00,
 0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x74,0x65,0x73,0x74,0x2e,0x63,0x00,
 0x67,0x63,0x63,0x32,0x5f,0x63,0x6f,0x6d,0x70,0x69,0x6c,0x65,0x64,0x2e,0x00,0x5f,
 0x5f,0x6d,0x6f,0x64,0x75,0x6c,0x65,0x5f,0x6b,0x65,0x72,0x6e,0x65,0x6c,0x5f,0x76,
 0x65,0x72,0x73,0x69,0x6f,0x6e,0x00,0x5f,0x5f,0x6d,0x6f,0x64,0x75,0x6c,0x65,0x5f,
 0x75,0x73,0x69,0x6e,0x67,0x5f,0x63,0x68,0x65,0x63,0x6b,0x73,0x75,0x6d,0x73,0x00,
 0x6d,0x69,0x6e,0x69,0x6d,0x61,0x6c,0x5f,0x6f,0x70,0x65,0x6e,0x00,0x6d,0x69,0x6e,
 0x69,0x6d,0x61,0x6c,0x5f,0x72,0x65,0x6c,0x65,0x61,0x73,0x65,0x00,0x6d,0x69,0x6e,
 0x69,0x6d,0x61,0x6c,0x5f,0x69,0x6f,0x63,0x74,0x6c,0x00,0x6d,0x69,0x6e,0x69,0x6d,
 0x61,0x6c,0x5f,0x66,0x6f,0x70,0x73,0x00,0x6d,0x69,0x6e,0x69,0x6d,0x61,0x6c,0x00,
 0x6d,0x69,0x6e,0x69,0x6d,0x61,0x6c,0x5f,0x69,0x6e,0x69,0x74,0x00,0x6d,0x69,0x6e,
 0x69,0x6d,0x61,0x6c,0x5f,0x65,0x78,0x69,0x74,0x00,0x6d,0x69,0x73,0x63,0x5f,0x64,
 0x65,0x72,0x65,0x67,0x69,0x73,0x74,0x65,0x72,0x00,0x5f,0x5f,0x74,0x68,0x69,0x73,
 0x5f,0x6d,0x6f,0x64,0x75,0x6c,0x65,0x00,0x5f,0x5f,0x6d,0x6f,0x64,0x75,0x6c,0x65,
 0x5f,0x61,0x75,0x74,0x68,0x6f,0x72,0x00,0x63,0x6c,0x65,0x61,0x6e,0x75,0x70,0x5f,
 0x6d,0x6f,0x64,0x75,0x6c,0x65,0x00,0x77,0x72,0x69,0x74,0x65,0x5f,0x67,0x70,0x69,
 0x6f,0x5f,0x62,0x69,0x74,0x00,0x49,0x32,0x43,0x5f,0x64,0x65,0x6c,0x61,0x79,0x00,
 0x49,0x32,0x43,0x5f,0x66,0x61,0x69,0x6c,0x65,0x64,0x00,0x69,0x6e,0x69,0x74,0x5f,
 0x6d,0x6f,0x64,0x75,0x6c,0x65,0x00,0x69,0x32,0x63,0x5f,0x73,0x65,0x74,0x00,0x73,
 0x65,0x74,0x5f,0x67,0x70,0x69,0x6f,0x5f,0x63,0x74,0x72,0x6c,0x00,0x49,0x32,0x43,
 0x5f,0x77,0x62,0x79,0x74,0x65,0x00,0x5f,0x5f,0x6d,0x65,0x6d,0x7a,0x65,0x72,0x6f,
 0x00,0x6d,0x69,0x73,0x63,0x5f,0x72,0x65,0x67,0x69,0x73,0x74,0x65,0x72,0x00,0x5f,
 0x5f,0x61,0x72,0x63,0x68,0x5f,0x63,0x6f,0x70,0x79,0x5f,0x66,0x72,0x6f,0x6d,0x5f,
 0x75,0x73,0x65,0x72,0x00,0x49,0x32,0x43,0x5f,0x72,0x62,0x79,0x74,0x65,0x00,0x72,
 0x65,0x61,0x64,0x5f,0x67,0x70,0x69,0x6f,0x5f,0x62,0x69,0x74,0x00,0x5f,0x5f,0x63,
 0x6f,0x6e,0x73,0x74,0x5f,0x64,0x65,0x6c,0x61,0x79,0x00 };

 FILE *fp;

 gp2x_module_deinit(); //rmmod modules in case they've been insmod'ed already

 fp = fopen("/mnt/minimal.o","wb");
 fwrite(minimal_o,sizeof(minimal_o),1,fp); 
 fclose(fp);

 system("/sbin/insmod /mnt/minimal.o &> /dev/null");
 system("/sbin/insmod input    &> /dev/null");
 system("/sbin/insmod joydev   &> /dev/null");
 system("/sbin/insmod keybdev  &> /dev/null");
 system("/sbin/insmod mousedev &> /dev/null");
}





/*
   ,--------------------.
   |                    |X
   |   GP2X - LIBRARY   |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/

/* Private: gp2x_mem_allocupper
   Function which allocates fixed memory blocks used by library.

   Credits:
   rlyeh */

static void gp2x_mem_allocupper(unsigned long *phys, void **virt, int length)
{
   static int top = 0x2000000;
   
   *phys = 0, *virt = (void **)0;
   
   if(top - length >= 0) 
   {
       top -= length;
     *phys  = 0x2000000 + top;
     *virt  = (void **)&gp2x_dualcore_ram[top>>2];

     memset(*virt, 0, length);
   }
}


/* Function: gp2x_init
   This function sets up the whole library.

   Notes:
   - *You have to call this function before any other function*.
   - If you are merging both SDL and Minimal Library in your project,
   you will have to call gp2x_init after any SDL_Init() / SDL_SetVideoMode() / SDL_CreateRGBSurface() call.

   Parameters:                     
   ticks_per_second (1..7372800) - sets internal <gp2x_ticks>. Eg, set it to 1000 to work in milliseconds.
   bpp (8,15,16) - sets the bits per pixel video mode
   rate (11025,22050,44100) - sets the sound frequency rate
   bits (16) - sets the sound bits. GP2X supports 16 bits only.
   stereo (0,1) - sets the stereo mode. 1 for stereo, 0 for mono.
   Hz (unused) - sets sound poll frequency, in hertzs. Unused in this release.
   solid_font (0,1) - sets default 6x10 font to have solid background or not.

   See also:
   <gp2x_deinit>

   Credits:
   rlyeh */
  
void gp2x_init(int ticks_per_second, int bpp, int rate, int bits, int stereo, int Hz, int solid_font)
{
  static int first=1;

  gp2x_printf_init(&gp2x_default_font,6,10,(unsigned char *)gp2x_fontf,bpp==8?0xFF:0xFFFF,0x0000,solid_font);

  if(bits != 16)
  {
    printf("ERROR: only 16 bits per sample are supported.\n");
    chdir("/usr/gp2x");
    execl("gp2xmenu", "gp2xmenu", NULL);
  }
  
  if(gp2x_f200)
  if(rate != 11025 || rate != 22050 || rate != 32000 || rate != 44100)
  {
    printf("Warning: custom sound rate (%d Hz) might not work on GP2X-F200 models\n", rate);
  }
  
  //open devices + init misc variables
  if(first) { 
              gp2x_sound_pausei=1, gp2x_sound_thread=0, gp2x_exit=0, gp2x_ticks=0, gp2x_ticks_per_second=7372800/ticks_per_second;

              if((gp2x_dev[0] = open("/dev/cx25874",O_RDWR))==(unsigned)-1) 
              {  printf("ERROR: Please disable TV-out before running me.\n");
                 chdir("/usr/gp2x"); execl("gp2xmenu","gp2xmenu",NULL); }
                 
              gp2x_module_init(); 


#if 0
              gp2x_dev[1] = open("/dev/console",         O_RDONLY);

              //gp2x_dev[7] = open("/dev/input/keybdev",         O_RDONLY);
              //gp2x_dev[7] = open("/dev/input/mice",         O_RDONLY);
              int data = 0;
              ioctl(gp2x_dev[1], KDGKBTYPE, &data);
              if(data==KB_84||data==KB_101) printf("!!\n");
              printf("kb %d\n",data);

              data = 0;
              ioctl(0, KDGKBTYPE, &data);
              if(data==KB_84||data==KB_101) printf("!!\n");
              printf("kb %d\n",data);

              data = 0;
              ioctl(1, KDGKBTYPE, &data);
              if(data==KB_84||data==KB_101) printf("!!\n");
              printf("kb %d\n",data);

              data = 0;
              ioctl(2, KDGKBTYPE, &data);
              if(data==KB_84||data==KB_101) printf("!!\n");
              printf("kb %d\n",data);

              printf("device is %d\n",gp2x_dev[1]);
#endif

              gp2x_dev[2] = open("/dev/mem",                O_RDWR); 
              gp2x_dev[3] = open("/dev/dsp",                O_WRONLY);
              gp2x_dev[4] = open("/dev/mixer",              O_RDWR);
              gp2x_dev[5] = open("/dev/batt",               O_RDONLY);
              gp2x_dev[6] = open("/dev/minimal",            O_RDWR);
              gp2x_dev[7] = open("/dev/touchscreen/wm97xx", O_RDONLY | O_NOCTTY);

              if (!gp2x_dev[8]) {
                  gp2x_dev[8] = -1;
                  gp2x_dev[8] = open("/dev/tty0", O_RDWR | O_NDELAY, 0);
                  if (gp2x_dev[8] < 0) {
                      gp2x_dev[8] = open("/dev/vc/0", O_RDWR | O_NDELAY, 0);
                  }

                  if (gp2x_dev[8] >= 0) {
                      tcgetattr(gp2x_dev[8], &initial_settings);
                      new_settings = initial_settings;
                      new_settings.c_lflag &= ~(ICANON | ECHO | ISIG);
                      new_settings.c_iflag &= ~(ISTRIP | IGNCR | ICRNL | INLCR | IXOFF | IXON);
                      new_settings.c_cc[VMIN] = 0;
                      new_settings.c_cc[VTIME] = 0;
                      tcsetattr(gp2x_dev[8], TCSAFLUSH, &new_settings);

                      ioctl(gp2x_dev[8], KDSKBMODE, K_MEDIUMRAW); /* 7bits, 8th bit press/release */
                      ioctl(gp2x_dev[8], KDSETMODE, KD_GRAPHICS);
                      ioctl(gp2x_dev[8], KDSETLED, 0);
                      printf("keyboard enabled\n");
                }
            }
              
              gp2x_f200 = (gp2x_dev[7] >= 0);

              if(gp2x_f200)
              {
            	FILE *pointercal = fopen("/etc/pointercal", "r");

            	if (pointercal) 
                {
            		fscanf(pointercal, "%d %d %d %d %d %d %d", gp2x_touchscreen_calibration+0, gp2x_touchscreen_calibration+1, gp2x_touchscreen_calibration+2, gp2x_touchscreen_calibration+3, gp2x_touchscreen_calibration+4, gp2x_touchscreen_calibration+5, gp2x_touchscreen_calibration+6);
                    
            		fclose(pointercal);
            	}
              }
              
              gp2x_joystick_init(); /* open joystick devices */
              
              //map memory for GP2X hw registers, blitter and dualcore usage
              gp2x_memregl=(unsigned long  *)mmap(0, 0x10000, PROT_READ|PROT_WRITE, MAP_SHARED, gp2x_dev[2], 0xc0000000);
              gp2x_memregs=(unsigned short *)gp2x_memregl;
              gp2x_blitter=(unsigned long  *)mmap(0, 0x100, PROT_READ|PROT_WRITE, MAP_SHARED, gp2x_dev[2], 0xe0020000); 
              gp2x_dualcore_ram = (unsigned long  *)mmap(0, 0x2000000, PROT_READ|PROT_WRITE, MAP_SHARED, gp2x_dev[2], 0x02000000);

              //install mmuhack after last mmap()
              //mmap'ing() new areas after this hack may cause non-optimal speed results
              ioctl(gp2x_dev[6], MINIMAL_MMU_INSTALLHACK, NULL);


              gp2x_tv_setmode(LCD);

              memset((void *)gp2x_dualcore_ram, 0, 0x2000000);

              //reserve 64k memory for 940t + dualcore + FIFO message system + reserved area
              gp2x_mem_allocupper(&gp2x_video_cursor_phys, &gp2x_video_cursor_virt, 0x10000);

              //set our memory layout
              gp2x_mem_allocupper(&gp2x_video_cursor_phys, &gp2x_video_cursor_virt,  0x4000);

              int i;

              for(i=0;i<3;i++)
               gp2x_mem_allocupper(&gp2x_video_RGB[0].i[i], &gp2x_video_RGB[0].p[i], 0x26000);

              gp2x_video_RGB[0].p[3]=gp2x_video_RGB[0].p[0]; gp2x_video_RGB[0].i[3]=gp2x_video_RGB[0].i[0];
              gp2x_video_RGB[0].p[4]=gp2x_video_RGB[0].p[1]; gp2x_video_RGB[0].i[4]=gp2x_video_RGB[0].i[1];
              gp2x_video_RGB[0].p[5]=gp2x_video_RGB[0].p[2]; gp2x_video_RGB[0].i[5]=gp2x_video_RGB[0].i[2];

              for(i=0;i<4;i++)
              {
               gp2x_mem_allocupper(&gp2x_video_YUV[i].i[0],&gp2x_video_YUV[i].p[0],0x4B000);
               gp2x_mem_allocupper(&gp2x_video_YUV[i].i[1],&gp2x_video_YUV[i].p[1],0x4B000);
              }

              gp2x_save_registers(0);
              gp2x_940t_reset(1);
              gp2x_940t_pause(1);
              gp2x_sound_volume(100,100);

               //enable stereo in older firmwares than v1.4.0
              gp2x_sound_stereo(1);
        }

  //set YUV pixel width for A/B top/bottom regions 
  gp2x_memregs[0x2892>>1] = gp2x_memregs[0x2894>>1] = gp2x_memregs[0x28BC>>1] = gp2x_memregs[0x28BE>>1] = 320*4; /*YUV*/

  //set YUV input memory for regions/skip alpha blending/priority to region A
  gp2x_memregs[0x2884>>1]=0;     
  //set RGB pixel width (8,15/16)
  gp2x_memregs[0x290C>>1]=320*((bpp+1)/8);
  //set RGB bpp (8,15/16) ; enable RGB window 1
  gp2x_memregs[0x28DA>>1]=(((bpp+1)/8)<<9)|0x0AB;                                                                /*RGB: 8/15/16/24bpp and 5 layers active*/

  //clear video 
  memset(gp2x_video_cursor_virt, 0,  0x4000);
  memset(gp2x_video_RGB[0].p[0], 0, 0x26000);
  memset(gp2x_video_RGB[0].p[1], 0, 0x26000);
  memset(gp2x_video_RGB[0].p[2], 0, 0x26000);
  memset(gp2x_video_YUV[0].p[0], 0, 0x4B000);
  memset(gp2x_video_YUV[0].p[1], 0, 0x4B000);
  memset(gp2x_video_YUV[1].p[0], 0, 0x4B000);
  memset(gp2x_video_YUV[1].p[1], 0, 0x4B000);
  memset(gp2x_video_YUV[2].p[0], 0, 0x4B000);
  memset(gp2x_video_YUV[2].p[1], 0, 0x4B000);
  memset(gp2x_video_YUV[3].p[0], 0, 0x4B000);
  memset(gp2x_video_YUV[3].p[1], 0, 0x4B000);

  //clear icounters+pcounters at gp2x_video_layer structs
  memset(&gp2x_video_RGB[0].i[6], 0, sizeof(unsigned long)*2);
  memset(&gp2x_video_YUV[0].i[6], 0, sizeof(unsigned long)*2);
  memset(&gp2x_video_YUV[1].i[6], 0, sizeof(unsigned long)*2);
  memset(&gp2x_video_YUV[2].i[6], 0, sizeof(unsigned long)*2);
  memset(&gp2x_video_YUV[3].i[6], 0, sizeof(unsigned long)*2);

  //set some video settings
  if(first) {   gp2x_video_setluminance(0,0);
                gp2x_video_setdithering(0); 
                gp2x_video_setgammacorrection(1.0);                                                            
                gp2x_video_RGB_setcolorkey(0,0,0);
               
                #if 1
                gp2x_video_RGB_setwindows(0x11,-1,-1,-1,319,239);              
                gp2x_video_RGB_setscaling(320,240); 
                gp2x_video_YUV_setparts(-1,-1,-1,-1,319,239);
                gp2x_video_YUV_setscaling(0,320,240);
                gp2x_video_YUV_setscaling(1,320,240);
                gp2x_video_YUV_setscaling(2,320,240);
                gp2x_video_YUV_setscaling(3,320,240);
                #endif
                                                     
               //gp2x_tv_setmode(3);  

                gp2x_video_cursor_setup((unsigned char *)gp2x_video_cursor_default, 16, 2,255,255,255,0xF, 1,0,0,0,0xF);
                gp2x_video_cursor_show(0); }

  //set default 8bpp settings: palette + sextuple buffering
  if(bpp==8) {  gp2x_video_RGB_color8(0x00,0x00,0x00,0);
                gp2x_video_RGB_color8(0x00,0x00,0x00,254);
                gp2x_video_RGB_color8(0xFF,0xFF,0xFF,255);
                gp2x_video_RGB_setpalette(); 
	            gp2x_video_RGB[0].i[3]+=320*240, gp2x_video_RGB[0].p[3]=(void*)(((short*)gp2x_video_RGB[0].p[3])+320*240/2),
                gp2x_video_RGB[0].i[4]+=320*240, gp2x_video_RGB[0].p[4]=(void*)(((short*)gp2x_video_RGB[0].p[4])+320*240/2);
                gp2x_video_RGB[0].i[5]+=320*240, gp2x_video_RGB[0].p[5]=(void*)(((short*)gp2x_video_RGB[0].p[5])+320*240/2);
		gp2x_printfchar=gp2x_printfchar8;
      	} else  gp2x_printfchar=gp2x_printfchar15;
		
  //update screen pointers
  gp2x_video_RGB_flip(0);
  gp2x_video_YUV_flip(0); gp2x_video_YUV_flip(1); gp2x_video_YUV_flip(2); gp2x_video_YUV_flip(3);

  //set sound
  ioctl(gp2x_dev[3], SNDCTL_DSP_SPEED,  &rate);
  ioctl(gp2x_dev[3], SNDCTL_DSP_SETFMT, &bits);
  ioctl(gp2x_dev[3], SNDCTL_DSP_STEREO, &stereo);

  //set some default options, launches sound engine, and ready to rock... :-)
  if(first) {   audio_buf_info abi;
                int frag = 0x80000|7;
                
                ioctl(gp2x_dev[3], SNDCTL_DSP_SETFRAGMENT, &frag);
                ioctl(gp2x_dev[3], SNDCTL_DSP_GETOSPACE, &abi);

                gp2x_sound_buffer=(unsigned short *)malloc(16*(4+(abi.fragsize<<(stereo+(bits==16)))));
                gp2x_sound_buffer[1]=(gp2x_sound_buffer[0]=(abi.fragsize)) << (stereo + (bits==16));
                gp2x_sound_buffer[2]=(1000000000/Hz)&0xFFFF;
                gp2x_sound_buffer[3]=(1000000000/Hz)>>16;

                gp2x_dualcore_clock(200);

                gp2x_frames = 1;
                memset(gp2x_timer, 1, 16*sizeof(unsigned long));

                // turn on LCD if not using tv out. turn off LCD if using tv out
                gp2x_misc_lcd( 1^((gp2x_memregs[0x2800>>1]>>9)&1) );

                gp2x_misc_led(gp2x_misc_battery() < 15 );

                pthread_create( &gp2x_sound_thread, NULL, gp2x_sound_play, NULL);
                atexit(gp2x_deinit);  
                first=0;

                gp2x_profiles_calibration = gp2x_memregl[0x0A00>>2];

                printf(MINILIB_VERSION "\n"); }
}

unsigned char gp2x_keyboard_read(void)
{
    unsigned char key;

    if (gp2x_dev[8] == -1) {
        return 0;
    }
    read(gp2x_dev[8], &key, 1);

    if ((key & 0x80)) {
        key = 0;
    } else {
        key = key & 0x7F;
    }
    return key;
}

unsigned int gp2x_keyboard_readext(void)
{
    if (gp2x_dev[8] == -1) {
        return 0;
    }
    return read(gp2x_dev[8], keybuffer, 64);
}

unsigned int gp2x_keyboard_ledget(void)
{
    unsigned long int leds;

    ioctl(gp2x_dev[8], KDGETLED, &leds);
    return leds;
}

void gp2x_keyboard_ledset(unsigned int led)
{
    ioctl(gp2x_dev[8], KDSETLED, led);
}

/* Private: gp2x_deinit_
   Function which deinits library.

   Credits:
   rlyeh */

static void gp2x_deinit_(void)
{
  gp2x_init(1000, 16, 44100,16,1,60, 1);

  ioctl(gp2x_dev[6], MINIMAL_MMU_UNINSTALLHACK, NULL);

  while((gp2x_exit++)<1000000);

  gp2x_sound_volume(70, 70); 

  gp2x_save_registers(2); 

  gp2x_940t_reset(1);
  gp2x_940t_pause(1);

  gp2x_video_setdithering(2); 
  gp2x_video_setluminance(0,0);
  gp2x_video_setgammacorrection(1.0);
  gp2x_video_cursor_show(0);

  gp2x_tv_setmode(LCD);

  #if 1
  gp2x_video_RGB_setscaling(320,240);
  gp2x_video_RGB_setcolorkey(0,0,0);
  gp2x_video_RGB_setwindows(0x11,-1,-1,-1,319,239);
  gp2x_video_YUV_setparts(-1,-1,-1,-1,319,239);
  #endif

  munmap((void *)gp2x_blitter,      0x100);
  munmap((void *)gp2x_dualcore_ram, 0x2000000);
  munmap((void *)gp2x_memregl,      0x10000);

  gp2x_joystick_deinit();
  
  { int i; for(i=0;i<8;i++) if(gp2x_dev[i]) close(gp2x_dev[i]); }

  fcloseall();

  gp2x_module_deinit();

  free((void *)gp2x_sound_buffer);

  free((void *)gp2x_profiles_area);
  
  sync();
}


/* Function: gp2x_deinit
   This function unsets the whole library and returns to GP2X menu.

   Note:    
   - You can exit() your program, or call this function directly.
   - If you plan to exit your app by rebooting, check <gp2x_reboot> instead.

   See also:
   <gp2x_init>, <gp2x_reboot>

   Credits:
   rlyeh */

void gp2x_deinit(void)
{
 gp2x_deinit_();

 chdir("/usr/gp2x");
 execl("gp2xmenu","gp2xmenu",NULL);
}


/* Function: gp2x_reboot
   This function unsets the whole library and reboots the GP2X.

   See also:
   <gp2x_deinit>

   Credits:
   rlyeh */

void gp2x_reboot(void)
{
 gp2x_dualcore_launch_HH(NULL);
}
                      



/*
   ,--------------------.
   |                    |X
   |   GP2X - MISC      |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/

/* Function: gp2x_misc_led
   This function enables or disables the red battery led (GPIO OUT H4).

   Parameters:
   on (0..1) - led off (0) or on (1)  

   Credits:
   K-teto */

void gp2x_misc_led(int on)
{
 if(!gp2x_f200)
 { 
    if(!on) gp2x_memregs[0x106E>>1]|=0x10; else gp2x_memregs[0x106E>>1]&=~0x10;
 }
}


/* Function: gp2x_misc_memorycard
   This function if any memory card is inserted in.
   
   Returns:
   This function returns 0 if no memory card is found, 1 if unlocked memory card is found, and 2 if locked memory card is found.

   Credits:
   rlyeh */

int gp2x_misc_memorycard(void)
{
 int sdin     = ((1<<14) ^ (gp2x_memregs[0x1190>>1] & (1<<14))) >> 14,
     sdlocked = (gp2x_memregs[0x1186>>1] & (1<<2) ) >> 1;

 return sdin | sdlocked;
}


/* Function: gp2x_misc_headphones
   This function detects if headphones are plugged in.
   
   Returns:
   If headphones are plugged this function returns 1, else returns 0.

   Credits:
   rlyeh */

int gp2x_misc_headphones(void)
{
 return ((1<<11) ^ (gp2x_memregs[0x1196>>1] & (1<<11))) > 0;
}


/* Function: gp2x_misc_lcd
   This function enables or disables the LCD backlight.

   Parameters:
   on (0..1) - turns LCD backlight off (0) or on (1)  

   Credits:
   RobBrown, Coder_TimT */
 
void gp2x_misc_lcd(int on)
{
 if(gp2x_f200)
 {
  if(on) gp2x_memregs[0x1076 >> 1] |= 0x0800; else gp2x_memregs[0x1076 >> 1] &= ~0x0800;
 }
 else
 {
  if(on) gp2x_memregs[0x106E>>1] |= 4; else gp2x_memregs[0x106E>>1] &= ~4;
 }
}

/* lcd_poweroff

Coder_TimT

if(gp2x_f200)
{
 if(on) gp2x_memregs[0x106E >> 1] |= 0x0002; else gp2x_memregs[0x106E >> 1] &= ~0x0002;
}

*/

/* Function: gp2x_misc_battery
   This function returns a percentage (0-100) corresponding to the battery levels.

   Credits:
   Squidge (original battery code) 
   rlyeh */
 
int gp2x_misc_battery(void)
{ 
 int value = 0;
 
 if(gp2x_f200)
    return 50;
 
 read (gp2x_dev[5], &value, 2);
 return ( value-678 < 0 ? 0 : ((value-678)*100 > 33800 ? 100 : ((value-678)*100) / 338 ) );
}


/* Function: gp2x_misc_flushcache
   This function flush cache through SWI 0x9f0002

   Credits:
   rlyeh */

#if 1
void gp2x_misc_flushcache(void)
{
 asm volatile (".word 0xef9f0002"); // swi #0x9f0002
}
#else
void gp2x_misc_flushcache(void) __attribute__((naked));
{
 asm volatile (".word 0xef9f0002"); // swi #0x9f0002
 asm volatile (".word 0xe1a0f00e"); // mov pc, lr
}
#endif


#if 0
/* Function: gp2x_misc_eeprom
   This function returns data from GP2X EEPROM (I2C bus at address 0x0050).

   Notes at given string:
   - length is 87 characters wide (including trailing '\0').
   - at 0x0000, there is a substring of 32 hexadecimal ASCII characters (checksum?).
   - at 0x0020, there is a substring of 24 ASCII characters (string on the 'Info' screen at gp2xmenu).
   - at 0x0048, there are 14 bytes containing the LCD registers settings.

   Credits:
   theoddbot */

const char *gp2x_misc_eeprom(void)
{
  struct i2c_msg { unsigned short addr, flags, len; unsigned char *buf; } msg;
  struct i2c_rdwr_ioctl_data { struct i2c_msg *msgs; int nmsgs; } data;

  static char buffer[86+1];
  int i, fd;

  memset(buffer,0,86+1);

  if( (fd = open("/dev/i2c-0", O_RDWR)) ) {
  data.msgs  = &msg;
  data.nmsgs = 1;

  msg.addr  = 0x0050;       //0x0050 = EEPROM ADDRESS
  msg.flags = 0;            //0 = write
  msg.len   = 2;
  msg.buf   = (unsigned char *)"\0\0";
  ioctl(fd, 0x0707, &data); //0x0707 = I2C_RDWR

  msg.flags = 1;            //1 = read
  msg.len   = 1;
  for (i = 0; i < 86; i++) {
  msg.buf = (unsigned char *)buffer + i;
  ioctl(fd, 0x0707, &data); } //I2C_RDWR

  close(fd);
  }

  return buffer;
}
#else
char *gp2x_misc_eeprom(void)
{
  static char buffer[86+1];
  int i;

  memset(buffer,0,86+1);

  for (i = 0; i < 86; i++) buffer[i] = gp2x_i2c_read('P', i);

  return buffer;
}
#endif




  











/*
   ,--------------------.
   |                    |X
   |   GP2X - IMAGE     |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/


/* Function: gp2x_image_tga
   This function decodes a 24bpp TGA file into a YUV or RGB16 <gp2x_rect>.
   It allocates needed memory as well.
   
   Parameters:
   fp - open handle to read TGA from.
   r  - <gp2x_rect> pointer to write decoded pixels.
 
   Note:
   - You must set gp2x_rect.type (RECT_RGB16, RECT_YUV) and gp2x_rect.solid (0, 1) before calling this function.
   - You must call <gp2x_image_free> after using the image in order to deallocating used memory.
   
   Return:
   0 - error
   1 - ok
   
   See also:
   <gp2x_image_bmp>, <gp2x_image_free>

   Credits:
   rlyeh */

int gp2x_image_tga(FILE *fp, gp2x_rect *r)
{
 struct TGA_HEADER {         unsigned char  IDsize, palette, type;        // size of string, palettized, image type (0=none,1=indexed,2=rgb,3=grey,+8=rle packed)
                             short          palettestart,palettelength;   // first color map entry in palette, number of colors
                             unsigned char  palettebits;                  // number of bits per palette entry 15,16,24,32
                             short          x,y,w,h;                      // x,y offsets // WxH size
                             unsigned char  bpp, descriptor;              // 8,16,24,32  // descriptor bits (00vhaaaa vflipbit hflipbit alphabits)
                    }        __attribute__ ((__packed__)) tga ;

 void *a=NULL; int R, G, B, ix, ixl, iy, i=0; unsigned int *v32; unsigned short *v16;

 fread(&tga,sizeof(tga),1,fp);

 if(tga.palette==0 && tga.type==2 && tga.bpp==24 && (r->type * tga.w * tga.h) > 0 && r->type > 1 ) 
 if((a = malloc( r->type * tga.w * tga.h )) != NULL)
 {
  fseek(fp, tga.IDsize, SEEK_CUR);

  r->x=r->y=0, r->w=tga.w, r->h=tga.h, r->data=a, v16=(unsigned short *)a, v32=(unsigned int *)a;
   
  if( (tga.descriptor & 0x10)) i+=(tga.w-1), ixl=tga.w, ix=-1;   else ixl=-tga.w, ix=1;
  if(!(tga.descriptor & 0x20)) i+=(tga.w*(tga.h-1)), iy=-tga.w;  else iy = tga.w;

  if(r->type==RECT_YUV)   while(tga.h--) { while(tga.w--) { B=fgetc(fp);G=fgetc(fp);R=fgetc(fp); v32[i]=gp2x_video_YUV_color(R,G,B);   i+=ix; }; i+=(ixl+iy); tga.w=r->w; }
  if(r->type==RECT_RGB16) while(tga.h--) { while(tga.w--) { B=fgetc(fp);G=fgetc(fp);R=fgetc(fp); v16[i]=gp2x_video_RGB_color16(R,G,B); i+=ix; }; i+=(ixl+iy); tga.w=r->w; }
 }

 return (a!=NULL);
}


/* Function: gp2x_image_bmp
   This function decodes a 8/24bpp BMP file into a YUV, RGB8 or RGB16 <gp2x_rect>.
   It allocates needed memory as well.
   
   Parameters:
   fp - open handle to read BMP from.
   r  - <gp2x_rect> pointer to write decoded pixels.
   paletteIndex - first GP2X palette index to write our decoded palette to. Usually this is 0.
 
   Note:
   - You must set gp2x_rect.type (RECT_RGB16, RECT_YUV) and gp2x_rect.solid (0, 1) before calling this function.
   - You must call <gp2x_image_free> after using the image in order to deallocating used memory.
   - When using 8bpp BMPs, you must call gp2x_video_setpalette() later to update palette changes.
   
   Return:
   0 - error
   1 - ok
   
   See also:
   <gp2x_image_bmp>, <gp2x_image_free>
   
   Credits:
   rlyeh */

int gp2x_image_bmp(FILE *fp, gp2x_rect *r, unsigned char paletteIndex)
{
 struct BMP_HEADER 
 {
  unsigned short filetype;                           // 'BM'
  unsigned short pad1[8];                            // unused stuff (filesize, offset, etc)
  unsigned short w[2],h[2];                          // w x h
  unsigned short colorPlanes,bpp;                    // 1 // bits per pixel
  unsigned short cm[2];                              // compression type (0!)
  unsigned short iS[2];                              // size of image in bytes (some bmps got this to 0)
  unsigned short pad2[4];                            // unused stuff (metric)
  unsigned short cU[2];                              // number of colors used by the bitmap
  unsigned short pad3[2];                            // unused stuff (important colors)
 } __attribute__ ((__packed__)) bmp;

 unsigned char  *pal = NULL, *img = NULL; 
 unsigned long i,j,c,d=0,size;

 fread(&bmp,sizeof(bmp),1,fp);

 if(bmp.filetype == 0x4D42 && bmp.colorPlanes==1 && (bmp.cm[0]+(bmp.cm[1]<<16) == 0) )
 {
  if(bmp.bpp==8)
  {
   pal  = (unsigned char  *)malloc(4*(bmp.cU[0]+(bmp.cU[1]<<16)));
   fread(pal,4*(bmp.cU[0]+(bmp.cU[1]<<16)),1,fp);
   for(c=0;c<(unsigned)(bmp.cU[0]+(bmp.cU[1]<<16));c++) gp2x_video_RGB_color8(pal[c*4+2], pal[c*4+1], pal[c*4+0], c+paletteIndex);
  }

  fseek(fp,0L,SEEK_END);
  if((size=ftell(fp)-54+4*(bmp.cU[0]+(bmp.cU[1]<<16))) > 0) img = (unsigned char *)malloc(size);
  fseek(fp,54+4*(bmp.cU[0]+(bmp.cU[1]<<16)),SEEK_SET);

  if(img!=NULL)
  {
  fread(img,size,1,fp);

  r->x=r->y=0, r->w=bmp.w[0]+(bmp.w[1]<<16), r->h=bmp.h[0]+(bmp.h[1]<<16), r->data = (void *)malloc(r->type*(size/(bmp.bpp/8)));

  for(c=r->w*(r->h-1)-1,d=0,j=r->h;j;j--, d+=(4-(r->w%4))%4, c-=r->w*2)
   for(                     i=r->w;i;i--)
   {
    if(bmp.bpp==8  && r->type==RECT_RGB8)  { ((unsigned char *)(r->data))[c++] = img[d++]+paletteIndex; }
    if(bmp.bpp==8  && r->type==RECT_RGB16) { unsigned char *p = &pal[img[d++]*4]; ((unsigned short *)(r->data))[c++] = gp2x_video_RGB_color16(p[2], p[1], p[0]); }
    if(bmp.bpp==8  && r->type==RECT_YUV)   { unsigned char *p = &pal[img[d++]*4]; ((unsigned long  *)(r->data))[c++] = gp2x_video_YUV_color(p[2], p[1], p[0]); }
    if(bmp.bpp==24 && r->type==RECT_RGB16) { ((unsigned short *)(r->data))[c++] = gp2x_video_RGB_color16(img[d+2], img[d+1], img[d+0]); d+=3; }
    if(bmp.bpp==24 && r->type==RECT_YUV)   { ((unsigned long  *)(r->data))[c++] = gp2x_video_YUV_color(img[d+2], img[d+1], img[d+0]); d+=3; }
   }
  }

  if(img) free(img);
  if(pal) free(pal);
 }

 return d!=0;
}


/* Function: gp2x_image_free
   This function deallocates any memory taken by <gp2x_image_tga> or <gp2x_image_bmp>.
   
   Parameters:
   r  - <gp2x_rect> pointer to deallocate memory from.
 
   See also:
   <gp2x_image_bmp>, <gp2x_image_tga>
   
   Credits:
   rlyeh */

void gp2x_image_free(gp2x_rect *r)
{
 free(r->data);
 r->type=0;
}


#if 0

// Pasi Ojala gave me permission to embed his gunzip library into Minimal Library.
// This will lead to handle internally zip, gz and png file formats.

// http://www.cs.tut.fi/~albert/Dev/gunzip/gunzip.c

/*
   ,--------------------.
   |                    |X
   |   GP2X - GUNZIP    |X
   |                    |X
   `--------------------'X
    XXXXXXXXXXXXXXXXXXXXXX 
*/

void gp2x_file_gunzip(FILE *in, FILE *out)
{
  int GZRead(FILE*, FILE*, int);
  (void)GZRead(in, out, 0);
}

void gp2x_file_unzip(FILE *in, FILE *out)
{
  char header[64]; unsigned long csize, usize, crc32, skip;

  fread(header,64,1,in);

  skip=30;
  skip+=(header[27]<<8)|header[26]; //filename length
  skip+=(header[29]<<8)|header[28]; //extra fields length

  crc32=(header[17]<<24)|(header[16]<<16)|(header[15]<<8)|(header[14]);
  csize=(header[21]<<24)|(header[20]<<16)|(header[19]<<8)|(header[18]);
  usize=(header[25]<<24)|(header[24]<<16)|(header[23]<<8)|(header[22]);

  //loadpacked ( skip, size, crc );
  fseek(in, skip, SEEK_CUR);

  while(csize--) fputc(fgetc(in), out);
}


#endif

#if 0

MORE TO DOs:

# open2x - minimal.o bug
if minimal.o in current dir
    use it
    
    else
    
        modprobe /lib/modules/minimal.o
        use it
        
        else 

                if(/etc/open2x)
                    use internal open2x minimal.o
                else
                    use internal gph minimal.o
        
done
           
paeryn, Propeller, joanvr.

# f100 - f200     
change START/HOME button in API

# TO SUPPORT (from gp2xpectrum):
PM sleep 
sound restored at exit

# TO DO; LAB TEST : 120 HZ mode + simulated blur with coarse scaler

int x, nx, adjx;
int y, ny, adjy;

adjx = ( int( float( nx / x ) ) * x != nx);
adjy = ( int( float( ny / y ) ) * y != ny);

if(adjx || adjy)
{
 installtimer();

 in each loop do through
     scale( x , y, nx-1,  ny-1);
     scale( x , y, nx+0, ny-1);
     scale( x , y, nx+1, ny-1);

     scale( x , y, nx-1,  ny+0);
     scale( x , y, nx+0, ny+0);
     scale( x , y, nx+1, ny+0);

     scale( x , y, nx-1,  ny+1);
     scale( x , y, nx+0, ny+1);
     scale( x , y, nx+1, ny+1);
}


#endif
