#include "Header.h"
#include "Dp.h"

/* This file has code for the basic Davis-Putnam propositional decision
 * procedure.
 */

/*
Todo:
    2. proper exit for max_mem limit.
    3. proper exit codes for unsat, all_models.
*/

/* #define TRACE */

/****** Macros ******/

#define SUBSUMED(c) (Subsume ? c->subsumer : subsumed(c))

/****** External Variables and Routines ******/

extern int Save_interval;
extern FILE *Save_file;
extern FILE *Restore_file;

extern int First_order;            /* it's really a first-order problem */
extern int print_model(FILE *fp);  /* to print first-order models */

/****** Global Static Variables ******/

static int Subsume;            /* whether or not to mark subsumed clauses */
static int Check_time;         /* whether or not there is a time limit */

static int Num_clauses;
static Clause_ptr Clauses;
static int Greatest_atom;
static Atom_ptr Atoms;         /* malloced when we know how many there are */
static int Num_selectable_clauses;
static Clause_ptr *Selectable_clauses;  /* for select_atom() */
 
static int *Unit_queue;          /* queue for unit propogation */
static int Unit_queue_size;      /* Actually, one less than this will fit. */
static int Unit_queue_first, Unit_queue_last;

static Gen_ptr_ptr gen_ptr_avail;      /* list of available gen_ptr nodes */

static int Next_save;
static int Num_to_restore, Restore_count;
static int Num_assignments_for_save;
static int *Assignments_for_save;

/*************
 *
 *    get_gen_ptr() -- allocate a gen_ptr node.
 *
 *************/

static Gen_ptr_ptr get_gen_ptr(void)
{
    Gen_ptr_ptr p;
    
    if (!gen_ptr_avail)
        p = tp_alloc(sizeof(struct gen_ptr));
    else {
        p = gen_ptr_avail;
        gen_ptr_avail = gen_ptr_avail->next;
        }
    p->next = NULL;
    return(p);
}  /* get_gen_ptr */

/*************
 *
 *    free_gen_ptr() -- free a gen_ptr node.
 *
 *************/

static void free_gen_ptr(Gen_ptr_ptr p)
{
    p->next = gen_ptr_avail;
    gen_ptr_avail = p;
}  /* free_gen_ptr */

/*************
 *
 *   init_dp()
 *
 *************/

void init_dp()
{
    init_options();
    init_stats();
    init_clocks();
}  /* init_dp */

/*************
 *
 *   unsatisfiable_exit()
 *
 *************/

void unsatisfiable_exit()
{
    exit(UNSATISFIABLE_EXIT);
}  /* unsatisfiable_exit */

/*************
 *
 *   exit_if_over_time_limit()
 *
 *************/

void exit_if_over_time_limit()
{
    int sec = run_time() / 1000;
    if (sec > Parms[MAX_TP_SECONDS].val) {
	output_stats(stdout);
	printf("\nExit due to max_seconds parameter.\n");
	exit(MAX_SECONDS_EXIT);
	}
}  /* exit_if_over_time_limit */

/*************
 *
 *   subsumed()
 *
 *   Check if any of the literals is assigned TRUE.
 *
 *************/

int subsumed(Clause_ptr c)
{
    int i;
    for (i = 0; i < c->num_pos; i++)
	if (Atoms[c->pos[i]].value == ATOM_TRUE)
	    return(1);
    for (i = 0; i < c->num_neg; i++)
	if (Atoms[c->neg[i]].value == ATOM_FALSE)
	    return(1);
    return(0);
}  /* subsumed */

/*************
 *
 *   init_unit_queue()
 *
 *************/

static void init_unit_queue()
{
    Unit_queue_first = 0;
    Unit_queue_last = -1;
}  /* init_unit_queue */

/*************
 *
 *   unit_enqueue(c)
 *
 *   It is assumed that there is exactly 1 active literal.  The clause
 *   may be subsumed.  The queue is not allowed to have duplicate or
 *   complementary literals.
 *
 *************/

static int unit_enqueue(Clause_ptr c)
{
    ATOM_INT *ip;
    int lit, ev, rc, i;

    /* First find the unit (*ip gets atom, lit gets literal).
     * If subsumption is disabled, then a clause may have "true"
     * literals that are included in the "active" count; in this
     * case, c's "active" literal is already true, so it is not
     * enqueued.
     */

    if (c->active_pos == 1) {
	for (ip=c->pos, i = 0;
	     i < c->num_pos && Atoms[*ip].value != ATOM_NOT_ASSIGNED;
	     ip++, i++);
	if (i == c->num_pos) {
	    /* Subsumption must be off, and c is really subsumed. */
#ifdef TRACE
	    printf("		ENQUEUE: subsumed: "); p_clause(c);
#endif
	    return(NOT_DETERMINED);
	    }
	else
	    lit = *ip;
	}
    else {
	for (ip=c->neg, i = 0;
	     i < c->num_neg && Atoms[*ip].value != ATOM_NOT_ASSIGNED;
	     ip++, i++);
	if (i == c->num_neg) {
	    /* Subsumption must be off, and c is really subsumed. */
#ifdef TRACE
	    printf("		ENQUEUE: subsumed: "); p_clause(c);
#endif
	    return(NOT_DETERMINED);
	    }
	else
	    lit = -(*ip);
	}

    ev = Atoms[*ip].enqueued_value;

    if (ev == NOT_DETERMINED) {
	Atoms[*ip].enqueued_value = lit > 0 ? ATOM_TRUE : ATOM_FALSE;
	Unit_queue_last++;
	if (Unit_queue_last == Unit_queue_size)
	    abend("unit_enqueue: queue too small");
	Unit_queue[Unit_queue_last] = lit;
#ifdef TRACE
	printf("		ENQUEUE: %d ok.\n", lit);
#endif
	return(NOT_DETERMINED);
	}
    else if ((ev == ATOM_TRUE  && lit < 0) ||
	     (ev == ATOM_FALSE && lit > 0)) {
#ifdef TRACE
	printf("		ENQUEUE: %d unsat.\n", lit);
#endif
	return(UNSATISFIABLE);
	}
    else {
#ifdef TRACE
	printf("		ENQUEUE: %d already in queue.\n", lit);
#endif
	return(NOT_DETERMINED);
	}
}  /* unit_enqueue */

/*************
 *
 *   unit_dequeue()
 *
 *   Also reset the enqueued_value field of the atom.
 *
 *************/

static int unit_dequeue()
{
    int lit;

    if (Unit_queue_last < Unit_queue_first)
      return(0);
    else {
      lit = Unit_queue[Unit_queue_first];
      Unit_queue_first++;
      Atoms[abs(lit)].enqueued_value = NOT_DETERMINED;
#ifdef TRACE
      printf("		dequeue: %d.\n", lit);
#endif
      return(lit);
    }
}  /* unit_dequeue */

/*************
 *
 *   p_clause()
 *
 *************/

void p_clause(Clause_ptr c)
{
    int j;
    
    printf("( ");
    for (j = 0; j < c->num_neg; j++)
	printf("-%d ", c->neg[j]);
    for (j = 0; j < c->num_pos; j++)
	printf("%d ", c->pos[j]);
    printf(") active_neg=%d, active_pos=%d, subsumer=%d\n", c->active_neg,
	    c->active_pos, c->subsumer);
}  /* p_clause */

/*************
 *
 *   p_atom()
 *
 *************/

void p_atom(int i)
{
    int j;
    Atom_ptr a = Atoms + i;

    printf("Atom=%d, value=%d, pos_occ=", i, a->value);
    for (j = 0; j < a->num_pos_occ; j++)
	printf("%x,", (unsigned) a->pos_occ[j]);
    printf(" neg_occ=");
    for (j = 0; j < a->num_neg_occ; j++)
	printf("%x,", (unsigned) a->neg_occ[j]);
    printf(".\n");
}  /* p_atom */

/*************
 *
 *   insert_dp_clause()
 *
 *   Take a propositional clause (array of integers) and insert it into
 *   the set of DP clauses.
 *
 *************/

void insert_dp_clause(int c[], int n)
{
    static Clause_ptr Prev = NULL;
    Clause_ptr d;
    int i, np, nn;

    /* Find out how many pos and neg literals, so that arrays can be allocated.
     */

    np = nn = 0;
    for (i = 0; i < n; i++) {
	if (c[i] > 0)
	    np++;
        else
	    nn++;
	if (abs(c[i]) > Greatest_atom) {
	    if (abs(c[i]) >= ATOM_INT_MAX)
		abend("insert_dp_clause, atom_too_big.");
	    Greatest_atom = abs(c[i]);
	    }
	}

    /* Allocate the clause and the literal arrays. */
	
    d = tp_alloc(sizeof(struct clause));
    d->next = NULL; d->subsumer = 0;
    d->num_pos = np; d->active_pos = np;
    d->num_neg = nn; d->active_neg = nn;
    if (Prev)
	Prev->next = d;
    else
	Clauses = d;
    Prev = d;

    d->pos = tp_alloc(np * sizeof(ATOM_INT));  /* ok if np==0 */
    d->neg = tp_alloc(nn * sizeof(ATOM_INT));  /* ok if nn==0 */

    /* Fill in the literal arrays (neg literals are stored positively). */

    np = nn = 0;
    for (i = 0; i < n; i++) {
	if (c[i] > 0)
	    d->pos[np++] = c[i];
	else
	    d->neg[nn++] = -c[i];
	}
    Num_clauses++;
    Stats[INPUT_CLAUSES] = Num_clauses;
    Stats[GREATEST_ATOM] = Greatest_atom;
#if 0
    p_clause(d);
#endif
}  /* insert_dp_clause */

/*************
 *
 *   read_one_clause(fp)
 *
 *   Read a propositional clause (sequence of nonzero integers, 
 *   terminated with 0) from a file, and insert it into the DP clause set.
 *   (This is not used for first-order problems.)
 *
 *   Return  EOF at EOF, 0 if error, 1 if clase read successfully.
 *
 *************/

static int read_one_clause(FILE *fp)
{
    int c[LIT_INT_MAX], num_lits, lit, rc;

    num_lits = 0;
    do {
	rc = fscanf(fp, "%d", &lit);
	if (rc == EOF)
	    return(EOF);
	else if (rc == 0)
	    return(0);
	else if (lit != 0)
	    c[num_lits++] = lit;
	} while (lit != 0);

    insert_dp_clause(c, num_lits);

    return(1);
}  /* read_one_clause */

/*************
 *
 *   read_all_clauses(fp)
 *
 *   (This is not used for first-order problems.)
 *   Abend if an error is found.
 *
 *************/

int read_all_clauses(FILE *fp)
{
    int rc = 1;

    while (rc == 1)
	rc = read_one_clause(fp);

    if (rc == 0)
	abend("Propositional input must be all integers; if you have\nfirst-order input, the domain size must be given (-n).");

    return(Num_clauses);
}  /* read_all_clauses */

/*************
 *
 *   more_setup()
 *
 *   Assume all clauses have been inserted into the DP set.  This routine
 *   sets up the unit propogation queue, and the atom array (which is
 *   used for accessing clauses that contain a given atom).
 *
 *************/

int more_setup()
{
    int i, j;
    Clause_ptr c;
    Atom_ptr a;

#if 0
    for (c = Clauses; c; c = c->next)
	p_clause(c);
#endif
    /* Allocate queue for unit propogation (Unit_queue is a global variable). */

    Unit_queue_size = Greatest_atom + 1;
    i = Unit_queue_size * sizeof(Clause_ptr *);
    Unit_queue = malloc(i);
    Stats[MEM_MALLOCED] += i;
    init_unit_queue();

    /* Now set up the atom array (indexed with 1--Greatest_atom).
     * Each atom struct has lists of positive occurrences and
     * negative occurrences.  To save space, the lists are stored
     * as arrays; but we don't know how big the arrays should be
     * until we process the clauses.  So we make two passes through
     * the clauses.
     */

    i = (Greatest_atom+1) * sizeof(struct atom);
    Atoms = malloc(i);
    Stats[MEM_MALLOCED] += i;

    for (i = 0, a = Atoms; i <= Greatest_atom; i++, a++) {
	a->value = ATOM_NOT_ASSIGNED;
	a->enqueued_value = ATOM_NOT_ASSIGNED;
	a->num_pos_occ = 0;
	a->pos_occ = NULL;
	a->num_neg_occ = 0;
	a->neg_occ = NULL;
	}

    /* For each literal of each clause, increment occurrence count
     * in atom structure.
     */

    for (c = Clauses; c; c = c->next) {
	for (j = 0; j < c->num_pos; j++) {
	    a = Atoms + c->pos[j];
	    a->num_pos_occ++;
	    }
	for (j = 0; j < c->num_neg; j++) {
	    a = Atoms + c->neg[j];
	    a->num_neg_occ++;
	    }
	}

    /* For each atom, allocate occurrence arrays, and reset occurrence
     * counts to 0.
     */

    for (i = 0, a = Atoms; i <= Greatest_atom; i++, a++) {
	Stats[LIT_OCC_INPUT] += a->num_pos_occ;
	a->pos_occ = tp_alloc(a->num_pos_occ * sizeof(Clause_ptr));
	a->num_pos_occ = 0;

	Stats[LIT_OCC_INPUT] += a->num_neg_occ;
	a->neg_occ = tp_alloc(a->num_neg_occ * sizeof(Clause_ptr));
	a->num_neg_occ = 0;
	}

    /* For each literal of each clause, add clause to occurrence
     * array of atom.
     */

    for (c = Clauses; c; c = c->next) {
	for (j = 0; j < c->num_pos; j++) {
	    a = Atoms + c->pos[j];
	    a->pos_occ[a->num_pos_occ++] = c;
	    }
	for (j = 0; j < c->num_neg; j++) {
	    a = Atoms + c->neg[j];
	    a->neg_occ[a->num_neg_occ++] = c;
	    }
	}

    return(Num_clauses);
}  /* more_setup */

/*************
 *
 *   select_init()
 *
 *   Initialize the mechanism for selecting atoms for splitting;
 *
 *************/

static void select_init()
{
    Clause_ptr c;
    int j;
    
    /* We will select literals from (smallest) positive clauses for splitting.
     * To avoid scanning all clauses, we build an array of pointers
     * to all of the nonsubsumed clauses with 2 or more positive literals.
     * (Horn clauses will be handled by unit propogation.)
     */

    for (c = Clauses, Num_selectable_clauses = 0; c; c = c->next) {
	/* Recall that even if subsumption is disabled, it is done
	 * during preprocessing, which has just occurred.
	 */
	if (!c->subsumer && c->num_pos >= 2)
	    Num_selectable_clauses++;
	}

    Stats[SELECTABLE_CLAUSES] = Num_selectable_clauses;

    /* Set up array of selectable clauses for select_atom(). */

    j = Num_selectable_clauses * sizeof(Clause_ptr *);
    Selectable_clauses = malloc(j);
    Stats[MEM_MALLOCED] += j;

    for (c = Clauses, j = 0; c; c = c->next)
	if (!c->subsumer && c->num_pos >= 2)
	    Selectable_clauses[j++] = c;
}  /* select_init */

/*************
 *
 *   select_atom()
 *
 *   Go through the array of selectable clauses, and select the first active
 *   literal in the first shortest nonsubsumed clause.
 *
 *************/

static int select_atom()
{
    ATOM_INT *ip;
    int min, atom, i;
    Clause_ptr c;

    min = LIT_INT_MAX;
    atom = 0;
#if 1
    for (i = 0; i < Num_selectable_clauses; i++)
#else
    for (i = Num_selectable_clauses-1; i >= 0; i--)
#endif
	{
	c = Selectable_clauses[i];
	if (c->active_neg == 0 && c->active_pos < min && !SUBSUMED(c)) {
	    min = c->active_pos;
	    for (ip = c->pos; Atoms[*ip].value != ATOM_NOT_ASSIGNED; ip++);
	    atom = *ip;
	    if (atom == 0)
		abend("HERE it is!!");
	    }
	}
    return(atom);
}  /* select_atom */

/*************
 *
 *   atom_value()
 *
 *   Return the value of an atom.  This can be called from outside code,
 *   (i.e., first-order model printer) so the defined symbols are not used.
 *
 *************/

int atom_value(int atom)
{
    int rc = -1;
    if (atom < 1 || atom > Greatest_atom)
	abend("atom_value, atom out of range");
    switch(Atoms[atom].value) {
      case ATOM_FALSE: rc = 0; break;
      case ATOM_TRUE:  rc = 1; break;
      case ATOM_NOT_ASSIGNED: rc = 2; break;
      default: abend("atom_value, bad value");
	}
    return(rc);
}  /* atom_value */

/*************
 *
 *   model()
 *
 *   this routine is called when a model has been found.
 *
 *************/

static void model()
{
    int i;
    static int next_message = 1;

    Stats[MODELS]++;
    if (Flags[PRINT_MODELS].val) {
	if (!First_order) {
	    printf("\nModel #%d:", Stats[MODELS]);
	    for (i = 1; i <= Greatest_atom; i++)
		if (Atoms[i].value == ATOM_TRUE)
		    printf(" %d", i);
	    printf("\n");
	    printf("end_of_model\n");
	    }
	else
	    print_model(stdout);  /* print first_order model */
	}
    
    if (Flags[PRINT_MODELS_PORTABLE].val) {
	if (!First_order) {
	    printf("\nModel #%d:", Stats[MODELS]);
	    for (i = 1; i <= Greatest_atom; i++)
		if (Atoms[i].value == ATOM_TRUE)
		    printf(" %d", i);
	    printf("\n");
	    printf("end_of_model\n");
	    }
	else
	    print_model_portable(stdout);  /* print first_order model */
	}
    
    if (Flags[PRINT_MODELS_IVY].val) {
	if (!First_order) {
	    printf("\nModel #%d:", Stats[MODELS]);
	    for (i = 1; i <= Greatest_atom; i++)
		if (Atoms[i].value == ATOM_TRUE)
		    printf(" %d", i);
	    printf("\n");
	    printf("end_of_model\n");
	    }
	else
	    print_model_ivy(stdout);  /* print first_order model */
	}
    
    if (!Flags[PRINT_MODELS_PORTABLE].val && !Flags[PRINT_MODELS].val) {
	if (Stats[MODELS] == next_message) {
	    printf("\nThe %d%s model has been found.\n", Stats[MODELS],
		   next_message == 1 ? "st" : "th");
	    next_message *= 10;
	    }
	}

    fflush(stdout);

    if (Stats[MODELS] >= Parms[MAX_MODELS].val) {
	output_stats(stdout);
	printf("\nExit due to max_models parameter.\n");
	exit(MAX_MODELS_EXIT);
	}
}  /* model */

/*************
 *
 *   assign(lit) - make a literal true
 *
 *   1. Make the assignment in the atom array.
 *   2. (optional) Mark nonsubsumed clauses that contain it as subsumed by it.
 *   3. Unit resolution on clauses that are not marked as subsumed.
 *      If a unit is found, enqueue it for unit propogation.
 *   4. If empty clause is found, return UNSATISFIABLE else NOT_DETERMINED.
 *
 *************/

static int assign(int lit)
{
    Atom_ptr a;
    Clause_ptr c;
    int atom, value, i, rc, n, np, nn;
    
    if (lit < 0)
	{ atom = -lit; value = ATOM_FALSE; }
    else
	{ atom = lit; value = ATOM_TRUE; }

    rc = NOT_DETERMINED;
    a = Atoms + atom;

    np = a->num_pos_occ;
    nn = a->num_neg_occ;

    if (a->value != ATOM_NOT_ASSIGNED)
	abend("assign: atom already assigned");
    else
	a->value = value;

    if (value == ATOM_TRUE) {  /* Assign true to atom */
	if (Subsume) 
	    for (i = 0; i < np; i++) {
		c = a->pos_occ[i];
		if (!c->subsumer)
		    c->subsumer = atom;
		}
	for (i = 0; i < nn; i++) {
	    c = a->neg_occ[i];
	    if (!c->subsumer) {
		c->active_neg--;
		n = c->active_pos + c->active_neg;
		if (n == 0)
		    rc = UNSATISFIABLE;
		else if (n == 1 && unit_enqueue(c) == UNSATISFIABLE)
		    rc = UNSATISFIABLE;
		}
	    }
	}

    else {  /* Assign false to atom */
	if (Subsume)
	    for (i = 0; i < nn; i++) {
		c = a->neg_occ[i];
		if (!c->subsumer)
		    c->subsumer = atom;
		}
	for (i = 0; i < np; i++) {
	    c = a->pos_occ[i];
	    if (!c->subsumer) {
		c->active_pos--;
		n = c->active_neg + c->active_pos;
		if (n == 0)
		    rc = UNSATISFIABLE;
		else if (n == 1 && unit_enqueue(c) == UNSATISFIABLE)
		    rc = UNSATISFIABLE;
		}
	    }
	}
    if (rc == UNSATISFIABLE)
      Stats[FAILED_PATHS]++;
    return(rc);
}  /* assign */

/*************
 *
 *   unassign(lit) -- undo an assignment
 *
 *************/

static void unassign(int lit)
{
    Atom_ptr a;
    Clause_ptr c;
    int i, atom, np, nn;
    
    atom = abs(lit);
    a = Atoms + atom;

    if (a->value == ATOM_NOT_ASSIGNED)
	abend("unassign: atom not assigned");

    np = a->num_pos_occ;
    nn = a->num_neg_occ;
    
    if (a->value == ATOM_TRUE) {  /* Unassign true. */
	if (Subsume)
	    for (i = 0; i < np; i++) {
		c = a->pos_occ[i];
		if (c->subsumer == atom)
		    c->subsumer = 0;
		}
	for (i = 0; i < nn; i++) {
	    c = a->neg_occ[i];
	    if (!c->subsumer)
		c->active_neg++;
	    }
	}
    else {  /* Unassign false. */
	if (Subsume)
	    for (i = 0; i < nn; i++) {
		c = a->neg_occ[i];
		if (c->subsumer == atom)
		    c->subsumer = 0;
		}
	for (i = 0; i < np; i++) {
	    c = a->pos_occ[i];
	    if (!c->subsumer)
		c->active_pos++;
	    }
	}
    a->value = ATOM_NOT_ASSIGNED;
}  /* unassign */

/*************
 *
 *   save_if_appropriate()
 *
 *   Format of save entry:
 *
 *   seconds  models number-of-assignments  <assignments>
 *
 *************/

void save_if_appropriate()
{
    int cpu_seconds, i;

    if ((Restore_count == Num_to_restore) && Stats[SPLITS] % 1 == 0) {
	cpu_seconds = run_time() / 1000;
	if (cpu_seconds >= Next_save) {
	    Next_save += Save_interval;
	    fprintf(Save_file, "\n%10d %10d %10d   ", cpu_seconds,
		   Stats[MODELS], Num_assignments_for_save);
	    for (i = 0; i < Num_assignments_for_save; i++)
		fprintf(Save_file, " %d", Assignments_for_save[i]);
	    fprintf(Save_file, "\n");
	    fflush(Save_file);
	    }
	}
}  /* save_if_appropriate */

/*************
 *
 *   restore_assignments()
 *
 *************/

void restore_assignments()
{
    int sec, n, models, i;
    if (Assignments_for_save == NULL)
	Assignments_for_save = malloc(Greatest_atom * sizeof(ATOM_INT));
	
    fscanf(Restore_file, "%d %d %d", &sec, &models, &n);
    printf("\nRestoring %d assignments from %d-second save (previously found %d models).\n", n, sec, models);
    fflush(stdout);
    for (i = 0; i < n; i++)
	fscanf(Restore_file, "%d", Assignments_for_save + i);
    Num_to_restore = n;
    Restore_count = 0;

    for (i = 0; i < n; i++)
	printf(" %d", Assignments_for_save[i]);
    printf("\n"); fflush(stdout);
}  /* restore_assignments */

/*************
 *
 *   dp(depth) -- the kernel of the Davis-Putnam procedure
 *
 *   depth: current recursion level---for debugging only.
 *
 *************/

static void dp(int depth)
{
    int lit, unit, rc, j;
    Gen_ptr_ptr unit_assignments, p1;

    if (Check_time)
	exit_if_over_time_limit();

    if (Save_interval > 0)
	save_if_appropriate();

    if (Restore_count < Num_to_restore) {
	lit = Assignments_for_save[Restore_count++];
	j = (lit > 0 ? 0 : 1);
	}
    else {
	lit = select_atom();
	j = 0;
	}

    if (lit == 0) {
	model();
	return;
	}

    Stats[SPLITS]++;

    for (; j < 2; j++) {
	init_unit_queue();
#ifdef TRACE
	{ 
	int i;
	for (i = 0; i < depth; i++) printf("   ");
	printf(" %3d\n", lit);
	}
#endif
	rc = assign(lit);
	if (Save_interval > 0)  /* no-op if currently restoring */
	    Assignments_for_save[Num_assignments_for_save++] = lit;

	/* Unit propogation */

	unit_assignments = NULL;
	while (rc == NOT_DETERMINED && (unit = unit_dequeue()) != 0) {
	    Stats[UNIT_ASSIGNS]++;
#ifdef TRACE
	    { 
	    int i;
	    for (i = 0; i < depth; i++) printf("   ");
	    printf(" %3d*\n", unit);
	    }
#endif
	    rc = assign(unit);
	    /* Save the assignment so that it can be undone. */
	    p1 = get_gen_ptr();
	    p1->u.i = unit;
	    p1->next = unit_assignments;
	    unit_assignments = p1;
	    }

	if (rc == NOT_DETERMINED)
	    dp(depth+1);
	else  /* rc == UNSATISFIABLE, so empty queue to reset any enqueued_values */
	    while ((unit = unit_dequeue()) != 0);

	/* Undo unit propogation assignments */

	while (unit_assignments) {
#ifdef TRACE
	    { 
	    int i;
	    for (i = 0; i < depth; i++) printf("   ");
	    printf("[%3d]*\n", unit_assignments->u.i);
	    }
#endif
	    unassign(unit_assignments->u.i);
	    p1 = unit_assignments;
	    unit_assignments = unit_assignments->next;
	    free_gen_ptr(p1);
	    }

#ifdef TRACE
	{ 
	int i;
	for (i = 0; i < depth; i++) printf("   ");
	printf("[%3d]\n", lit);
	}
#endif
	unassign(lit);

	if (Save_interval > 0)
	    Num_assignments_for_save--;
	if (j == 0)
	    lit = -lit;
	}
}  /* dp */

/*************
 *
 *   delete_pointers_to_subsumed_clauses()
 *
 *   This routine is called after the initial unit propogation (before
 *   any splitting).  It goes through the occurrence lists in the
 *   atom array and deletes references to subsumed clauses.  The reason
 *   for this is simply to speed traversals during assign() and unassign().
 *
 *************/

static void delete_pointers_to_subsumed_clauses()
{
    int i, j, k;
    Atom_ptr a;

    /* The set of occurrences, say positive, of an atom, is kept as
     * an array of pointers to clauses.  Pointers to subsumed clauses
     * are removed, and the the others are shifted left the apropriate
     * amount.  Same for negative occurrences.
     */

    for (i = 1; i <= Greatest_atom; i++) {
	a = Atoms + i;
	for (j = 0, k = 0; j < a->num_pos_occ; j++)
	    if (a->pos_occ[j]->subsumer == 0)
		a->pos_occ[k++] = a->pos_occ[j];
	a->num_pos_occ = k;
	Stats[LIT_OCC_AFTER_SUB] += k;

	for (j = 0, k = 0; j < a->num_neg_occ; j++)
	    if (a->neg_occ[j]->subsumer == 0)
		a->neg_occ[k++] = a->neg_occ[j];
	a->num_neg_occ = k;
	Stats[LIT_OCC_AFTER_SUB] += k;
	}
}  /* delete_pointers_to_subsumed_clauses */

/*************
 *
 *   dp_prover()
 *
 *   This is the top (nonrecursive) routine of the Davis-Putnam procedure.
 *
 *************/

int dp_prover()
{
    int rc, unit, n, j;
    Clause_ptr c;

    Subsume = Flags[SUBSUME].val;
    Check_time = (Parms[MAX_TP_SECONDS].val < INT_MAX);

    if (Save_interval > 0) {
	Next_save = Save_interval;
	Assignments_for_save = malloc(Greatest_atom * sizeof(ATOM_INT));
	}

    if (Restore_file != NULL)
	restore_assignments();

    /* Initial unit propogation and subsumption. */

    Subsume = 1;  /* Always mark subsumed clauses while preprocessing. */

    CLOCK_START(PREPROCESS_TIME)
    for (c=Clauses, rc=NOT_DETERMINED; c && rc == NOT_DETERMINED; c = c->next)
	if (c->num_pos + c->num_neg == 1)
	    rc = unit_enqueue(c);

    while (rc == NOT_DETERMINED && (unit = unit_dequeue()) != 0) {
	Stats[PREPROCESS_UNIT_ASSIGNS]++;
	rc = assign(unit);
	}

    if (Stats[PREPROCESS_UNIT_ASSIGNS] > 0)
	delete_pointers_to_subsumed_clauses();
    else
	Stats[LIT_OCC_AFTER_SUB] = Stats[LIT_OCC_INPUT];

    for (c = Clauses, n=0; c; c = c->next)
	if (c->subsumer)
	    n++;

    Stats[CLAUSES_AFTER_SUB] = Stats[INPUT_CLAUSES] - n;

    Subsume = Flags[SUBSUME].val;

    /* Done with initial unit propogation and subsumption. */

    select_init();  /* Initialize the selection mechanism for splitting. */

    printf("\nAfter all unit preprocessing, %d atoms are still unassigned;\n"
           "%d clauses remain; %d of those are selectable; %d K allocated;\n"
	   "the current user CPU time is %.2f seconds.\n",
	   Greatest_atom - Stats[PREPROCESS_UNIT_ASSIGNS],
	   Stats[CLAUSES_AFTER_SUB], Stats[SELECTABLE_CLAUSES],
	   Stats[MEM_MALLOCED]/1024 + total_mem(),
	   run_time() / 1000.);
    fflush(stdout);
 
    CLOCK_STOP(PREPROCESS_TIME)

    if (rc == NOT_DETERMINED)	
	dp(0);  /* recursive Davis-Putnam routine */

    if (Stats[MODELS] == 0)
	return(UNSATISFIABLE_EXIT);
    else
	return(ALL_MODELS_EXIT);
}  /* dp_prover */
