/*
 * Copyright (c) 2021, 2023-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"
#include "../../bfloat.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void a64_hybrid_fp32bf16fp32_mmla_6x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<float> A_arg,
    size_t M, size_t N, const bfloat16 *B_ptr, IndirectOutputArg<float> output_arg,
    const float *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const bfloat16 *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const float *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 176f\n"
      "cmp %x[M], #0x4\n"
      "bgt 141f\n"
      "beq 106f\n"
      "cmp %x[M], #0x2\n"
      "bgt 71f\n"
      "beq 36f\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "cbz x12, 3f\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "b 15f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 14f\n"
      "cmp x11, #0x10\n"
      "bge 12f\n"
      "tbz x11, #3, 7f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "tbz x11, #2, 5f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "tbz x11, #1, 4f\n"
      "ldr d16, [x9], #0x8\n"
      "mov x20, #0x38\n"
      "tbz x11, #0, 11f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "b 11f\n"
      "4:"  // Height 1: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x11, #0, 11f\n"
      "ldr s16, [x9, #0x0]\n"
      "b 11f\n"
      "5:"  // Height 1: Partial accumulate: partial_2_8
      "tbz x11, #1, 6f\n"
      "ldr d11, [x9], #0x8\n"
      "mov x20, #0x28\n"
      "tbz x11, #0, 11f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "b 11f\n"
      "6:"  // Height 1: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x11, #0, 11f\n"
      "ldr s11, [x9, #0x0]\n"
      "b 11f\n"
      "7:"  // Height 1: Partial accumulate: partial_4_0
      "tbz x11, #2, 9f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "tbz x11, #1, 8f\n"
      "ldr d10, [x9], #0x8\n"
      "mov x20, #0x18\n"
      "tbz x11, #0, 11f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "b 11f\n"
      "8:"  // Height 1: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x11, #0, 11f\n"
      "ldr s10, [x9, #0x0]\n"
      "b 11f\n"
      "9:"  // Height 1: Partial accumulate: partial_2_0
      "tbz x11, #1, 10f\n"
      "ldr d9, [x9], #0x8\n"
      "mov x20, #0x8\n"
      "tbz x11, #0, 11f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "b 11f\n"
      "10:"  // Height 1: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "mov x20, #0x0\n"
      "11:"  // Height 1: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 13f\n"
      "12:"  // Height 1: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "13:"  // Height 1: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "b 15f\n"
      "14:"  // Height 1: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "15:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "16:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 17f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 18f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "b 18f\n"
      "17:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "18:"  // Height 1: input setup done
      "cmp x27, #0x4\n"
      "blt 21f\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      "ldr q6, [x10, #0x0]\n"
      "cmp x27, #0x8\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 20f\n"
      "19:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "cmp x27, #0x8\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x6e52ec09  // bfmmla v9.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x6e51ec0d  // bfmmla v13.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x6e52ec0a  // bfmmla v10.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x6e51ec0e  // bfmmla v14.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e52ec0b  // bfmmla v11.4s, v0.8h, v18.8h\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x6e51ec0f  // bfmmla v15.4s, v0.8h, v17.8h\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 19b\n"
      "20:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x6e52ec09  // bfmmla v9.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x6e51ec0d  // bfmmla v13.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x6e52ec0a  // bfmmla v10.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x6e51ec0e  // bfmmla v14.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e52ec0b  // bfmmla v11.4s, v0.8h, v18.8h\n"
      ".inst 0x6e51ec0f  // bfmmla v15.4s, v0.8h, v17.8h\n"
      "21:"  // Height 1: Multiply loop: Main loop skip
      "cbz x27, 24f\n"
      "cbz x27, 24f\n"
      "tbz x27, #1, 22f\n"
      "ldr d0, [x26], #0x8\n"
      "tbz x27, #0, 23f\n"
      "ld1 { v0.s }[2], [x26]\n"
      "b 23f\n"
      "22:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr s0, [x26, #0x0]\n"
      "23:"  // Height 1: Multiply loop: Ragged operand read: Done
      "ldr q18, [x10, #0x0]\n"
      "ldr q17, [x10, #0x10]\n"
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x6e52ec08  // bfmmla v8.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x6e51ec0c  // bfmmla v12.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x6e52ec09  // bfmmla v9.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x6e51ec0d  // bfmmla v13.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x6e52ec0a  // bfmmla v10.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x6e51ec0e  // bfmmla v14.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e52ec0b  // bfmmla v11.4s, v0.8h, v18.8h\n"
      ".inst 0x6e51ec0f  // bfmmla v15.4s, v0.8h, v17.8h\n"
      "24:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 16b\n"
      "uzp1 v8.2d, v8.2d, v12.2d\n"
      "uzp1 v9.2d, v9.2d, v13.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "uzp1 v10.2d, v10.2d, v14.2d\n"
      "uzp1 v11.2d, v11.2d, v15.2d\n"
      "tbz %x[flags], #1, 25f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v18.4s }, [x21]\n"
      "ld1r { v17.4s }, [x20]\n"
      "fmin v8.4s, v8.4s, v18.4s\n"
      "fmin v9.4s, v9.4s, v18.4s\n"
      "fmin v10.4s, v10.4s, v18.4s\n"
      "fmin v11.4s, v11.4s, v18.4s\n"
      "fmax v8.4s, v8.4s, v17.4s\n"
      "fmax v9.4s, v9.4s, v17.4s\n"
      "fmax v10.4s, v10.4s, v17.4s\n"
      "fmax v11.4s, v11.4s, v17.4s\n"
      "25:"  // Height 1: No activation
      "cmp x11, #0x10\n"
      "bge 34f\n"
      "tbz x11, #3, 29f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "st1 { v9.4s }, [x9], #0x10\n"
      "tbz x11, #2, 27f\n"
      "st1 { v10.4s }, [x9], #0x10\n"
      "tbz x11, #1, 26f\n"
      "str d11, [x9], #0x8\n"
      "tbz x11, #0, 33f\n"
      "st1 { v11.s }[2], [x9]\n"
      "b 33f\n"
      "26:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x11, #0, 33f\n"
      "str s11, [x9, #0x0]\n"
      "b 33f\n"
      "27:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x11, #1, 28f\n"
      "str d10, [x9], #0x8\n"
      "tbz x11, #0, 33f\n"
      "st1 { v10.s }[2], [x9]\n"
      "b 33f\n"
      "28:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x11, #0, 33f\n"
      "str s10, [x9, #0x0]\n"
      "b 33f\n"
      "29:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x11, #2, 31f\n"
      "st1 { v8.4s }, [x9], #0x10\n"
      "tbz x11, #1, 30f\n"
      "str d9, [x9], #0x8\n"
      "tbz x11, #0, 33f\n"
      "st1 { v9.s }[2], [x9]\n"
      "b 33f\n"
      "30:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x11, #0, 33f\n"
      "str s9, [x9, #0x0]\n"
      "b 33f\n"
      "31:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x11, #1, 32f\n"
      "str d8, [x9], #0x8\n"
      "tbz x11, #0, 33f\n"
      "st1 { v8.s }[2], [x9]\n"
      "b 33f\n"
      "32:"  // Height 1: Partial direct writeback: partial_1_0
      "str s8, [x9, #0x0]\n"
      "33:"  // Height 1: Partial direct writeback: Done
      "b 35f\n"
      "34:"  // Height 1: Full writeback
      "str q8, [x9, #0x0]\n"
      "str q9, [x9, #0x10]\n"
      "str q10, [x9, #0x20]\n"
      "str q11, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "35:"  // Height 1: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 2b\n"
      "b 212f\n"
      "36:"  // Height 2
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "37:"  // Height 2: Column loop
      "cbz x12, 38f\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "b 50f\n"
      "38:"  // Height 2: no bias
      "tbz %x[flags], #0, 49f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x26, x9, x20, LSL #2\n"
      "bge 47f\n"
      "tbz x11, #3, 42f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x26], #0x10\n"
      "tbz x11, #2, 40f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x26], #0x10\n"
      "tbz x11, #1, 39f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "mov x20, #0x38\n"
      "tbz x11, #0, 46f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x26]\n"
      "b 46f\n"
      "39:"  // Height 2: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x11, #0, 46f\n"
      "ldr s16, [x9, #0x0]\n"
      "ldr s15, [x26, #0x0]\n"
      "b 46f\n"
      "40:"  // Height 2: Partial accumulate: partial_2_8
      "tbz x11, #1, 41f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "mov x20, #0x28\n"
      "tbz x11, #0, 46f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x26]\n"
      "b 46f\n"
      "41:"  // Height 2: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x11, #0, 46f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s14, [x26, #0x0]\n"
      "b 46f\n"
      "42:"  // Height 2: Partial accumulate: partial_4_0
      "tbz x11, #2, 44f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "tbz x11, #1, 43f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "mov x20, #0x18\n"
      "tbz x11, #0, 46f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x26]\n"
      "b 46f\n"
      "43:"  // Height 2: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x11, #0, 46f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s13, [x26, #0x0]\n"
      "b 46f\n"
      "44:"  // Height 2: Partial accumulate: partial_2_0
      "tbz x11, #1, 45f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "mov x20, #0x8\n"
      "tbz x11, #0, 46f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x26]\n"
      "b 46f\n"
      "45:"  // Height 2: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "ldr s12, [x26, #0x0]\n"
      "mov x20, #0x0\n"
      "46:"  // Height 2: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 48f\n"
      "47:"  // Height 2: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x26, #0x0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "48:"  // Height 2: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "b 50f\n"
      "49:"  // Height 2: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "50:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "51:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 52f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 53f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "b 53f\n"
      "52:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #2\n"
      "53:"  // Height 2: input setup done
      "cmp x27, #0x4\n"
      "blt 56f\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      "ld1 { v1.4s }, [x25], #0x10\n"
      "cmp x27, #0x8\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 55f\n"
      "54:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "cmp x27, #0x8\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      "ld1 { v1.4s }, [x25], #0x10\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x6e52ec09  // bfmmla v9.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x6e51ec0d  // bfmmla v13.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x6e52ec0a  // bfmmla v10.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x6e51ec0e  // bfmmla v14.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e52ec0b  // bfmmla v11.4s, v0.8h, v18.8h\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x6e51ec0f  // bfmmla v15.4s, v0.8h, v17.8h\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 54b\n"
      "55:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x6e52ec09  // bfmmla v9.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x6e51ec0d  // bfmmla v13.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x6e52ec0a  // bfmmla v10.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x6e51ec0e  // bfmmla v14.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e52ec0b  // bfmmla v11.4s, v0.8h, v18.8h\n"
      ".inst 0x6e51ec0f  // bfmmla v15.4s, v0.8h, v17.8h\n"
      "56:"  // Height 2: Multiply loop: Main loop skip
      "cbz x27, 59f\n"
      "cbz x27, 59f\n"
      "tbz x27, #1, 57f\n"
      "ldr d0, [x26], #0x8\n"
      "ldr d1, [x25], #0x8\n"
      "tbz x27, #0, 58f\n"
      "ld1 { v0.s }[2], [x26]\n"
      "ld1 { v1.s }[2], [x25]\n"
      "b 58f\n"
      "57:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr s0, [x26, #0x0]\n"
      "ldr s1, [x25, #0x0]\n"
      "58:"  // Height 2: Multiply loop: Ragged operand read: Done
      "ldr q18, [x10, #0x0]\n"
      "ldr q17, [x10, #0x10]\n"
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x6e52ec08  // bfmmla v8.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x20]\n"
      ".inst 0x6e51ec0c  // bfmmla v12.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x30]\n"
      ".inst 0x6e52ec09  // bfmmla v9.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x40]\n"
      ".inst 0x6e51ec0d  // bfmmla v13.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x50]\n"
      ".inst 0x6e52ec0a  // bfmmla v10.4s, v0.8h, v18.8h\n"
      "ldr q18, [x10, #0x60]\n"
      ".inst 0x6e51ec0e  // bfmmla v14.4s, v0.8h, v17.8h\n"
      "ldr q17, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e52ec0b  // bfmmla v11.4s, v0.8h, v18.8h\n"
      ".inst 0x6e51ec0f  // bfmmla v15.4s, v0.8h, v17.8h\n"
      "59:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 51b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v6.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x26, x9, x20, LSL #2\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "tbz %x[flags], #1, 60f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v18.4s }, [x21]\n"
      "ld1r { v17.4s }, [x20]\n"
      "fmin v6.4s, v6.4s, v18.4s\n"
      "fmin v12.4s, v12.4s, v18.4s\n"
      "fmin v13.4s, v13.4s, v18.4s\n"
      "fmin v14.4s, v14.4s, v18.4s\n"
      "fmin v8.4s, v8.4s, v18.4s\n"
      "fmin v9.4s, v9.4s, v18.4s\n"
      "fmin v10.4s, v10.4s, v18.4s\n"
      "fmin v11.4s, v11.4s, v18.4s\n"
      "fmax v6.4s, v6.4s, v17.4s\n"
      "fmax v12.4s, v12.4s, v17.4s\n"
      "fmax v13.4s, v13.4s, v17.4s\n"
      "fmax v14.4s, v14.4s, v17.4s\n"
      "fmax v8.4s, v8.4s, v17.4s\n"
      "fmax v9.4s, v9.4s, v17.4s\n"
      "fmax v10.4s, v10.4s, v17.4s\n"
      "fmax v11.4s, v11.4s, v17.4s\n"
      "60:"  // Height 2: No activation
      "cmp x11, #0x10\n"
      "bge 69f\n"
      "tbz x11, #3, 64f\n"
      "st1 { v6.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x26], #0x10\n"
      "st1 { v9.4s }, [x26], #0x10\n"
      "tbz x11, #2, 62f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x26], #0x10\n"
      "tbz x11, #1, 61f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x26], #0x8\n"
      "tbz x11, #0, 68f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x26]\n"
      "b 68f\n"
      "61:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x11, #0, 68f\n"
      "str s14, [x9, #0x0]\n"
      "str s11, [x26, #0x0]\n"
      "b 68f\n"
      "62:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x11, #1, 63f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x26], #0x8\n"
      "tbz x11, #0, 68f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x26]\n"
      "b 68f\n"
      "63:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x11, #0, 68f\n"
      "str s13, [x9, #0x0]\n"
      "str s10, [x26, #0x0]\n"
      "b 68f\n"
      "64:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x11, #2, 66f\n"
      "st1 { v6.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x26], #0x10\n"
      "tbz x11, #1, 65f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x26], #0x8\n"
      "tbz x11, #0, 68f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x26]\n"
      "b 68f\n"
      "65:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x11, #0, 68f\n"
      "str s12, [x9, #0x0]\n"
      "str s9, [x26, #0x0]\n"
      "b 68f\n"
      "66:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x11, #1, 67f\n"
      "str d6, [x9], #0x8\n"
      "str d8, [x26], #0x8\n"
      "tbz x11, #0, 68f\n"
      "st1 { v6.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x26]\n"
      "b 68f\n"
      "67:"  // Height 2: Partial direct writeback: partial_1_0
      "str s6, [x9, #0x0]\n"
      "str s8, [x26, #0x0]\n"
      "68:"  // Height 2: Partial direct writeback: Done
      "b 70f\n"
      "69:"  // Height 2: Full writeback
      "str q6, [x9, #0x0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x26, #0x0]\n"
      "str q9, [x26, #0x10]\n"
      "str q10, [x26, #0x20]\n"
      "str q11, [x26, #0x30]\n"
      "70:"  // Height 2: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 37b\n"
      "b 212f\n"
      "71:"  // Height 3
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "72:"  // Height 3: Column loop
      "cbz x12, 73f\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v12.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v21.16b, v13.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v22.16b, v14.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v23.16b, v15.16b\n"
      "b 85f\n"
      "73:"  // Height 3: no bias
      "tbz %x[flags], #0, 84f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "bge 82f\n"
      "tbz x11, #3, 77f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x26], #0x10\n"
      "ld1 { v18.4s }, [x25], #0x10\n"
      "tbz x11, #2, 75f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x26], #0x10\n"
      "ld1 { v19.4s }, [x25], #0x10\n"
      "tbz x11, #1, 74f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d24, [x25], #0x8\n"
      "tbz x11, #0, 81f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x26]\n"
      "ld1 { v24.s }[2], [x25]\n"
      "b 81f\n"
      "74:"  // Height 3: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x11, #0, 81f\n"
      "ldr s16, [x9, #0x0]\n"
      "ldr s15, [x26, #0x0]\n"
      "ldr s24, [x25, #0x0]\n"
      "b 81f\n"
      "75:"  // Height 3: Partial accumulate: partial_2_8
      "tbz x11, #1, 76f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d19, [x25], #0x8\n"
      "tbz x11, #0, 81f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x26]\n"
      "ld1 { v19.s }[2], [x25]\n"
      "b 81f\n"
      "76:"  // Height 3: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x11, #0, 81f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s14, [x26, #0x0]\n"
      "ldr s19, [x25, #0x0]\n"
      "b 81f\n"
      "77:"  // Height 3: Partial accumulate: partial_4_0
      "tbz x11, #2, 79f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "tbz x11, #1, 78f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d18, [x25], #0x8\n"
      "tbz x11, #0, 81f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x26]\n"
      "ld1 { v18.s }[2], [x25]\n"
      "b 81f\n"
      "78:"  // Height 3: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x11, #0, 81f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s13, [x26, #0x0]\n"
      "ldr s18, [x25, #0x0]\n"
      "b 81f\n"
      "79:"  // Height 3: Partial accumulate: partial_2_0
      "tbz x11, #1, 80f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "tbz x11, #0, 81f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x26]\n"
      "ld1 { v17.s }[2], [x25]\n"
      "b 81f\n"
      "80:"  // Height 3: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "ldr s12, [x26, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s17, [x25, #0x0]\n"
      "81:"  // Height 3: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 83f\n"
      "82:"  // Height 3: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x26, #0x0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q17, [x25, #0x0]\n"
      "ldr q18, [x25, #0x10]\n"
      "ldr q19, [x25, #0x20]\n"
      "ldr q24, [x25, #0x30]\n"
      "83:"  // Height 3: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "b 85f\n"
      "84:"  // Height 3: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "85:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "86:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 87f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 88f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "b 88f\n"
      "87:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "88:"  // Height 3: input setup done
      "cmp x27, #0x4\n"
      "blt 91f\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      "ld1 { v1.4s }, [x25], #0x10\n"
      "cmp x27, #0x8\n"
      "ld1 { v2.4s }, [x24], #0x10\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 90f\n"
      "89:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "cmp x27, #0x8\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      "ld1 { v1.4s }, [x25], #0x10\n"
      ".inst 0x6e46ec50  // bfmmla v16.4s, v2.8h, v6.8h\n"
      ".inst 0x6e47ec54  // bfmmla v20.4s, v2.8h, v7.8h\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x6e5aec09  // bfmmla v9.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec51  // bfmmla v17.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x6e59ec0d  // bfmmla v13.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec55  // bfmmla v21.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x6e5aec0a  // bfmmla v10.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec52  // bfmmla v18.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x6e59ec0e  // bfmmla v14.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec56  // bfmmla v22.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e5aec0b  // bfmmla v11.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec53  // bfmmla v19.4s, v2.8h, v26.8h\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x6e59ec0f  // bfmmla v15.4s, v0.8h, v25.8h\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      ".inst 0x6e59ec57  // bfmmla v23.4s, v2.8h, v25.8h\n"
      "ld1 { v2.4s }, [x24], #0x10\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 89b\n"
      "90:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x6e46ec50  // bfmmla v16.4s, v2.8h, v6.8h\n"
      ".inst 0x6e47ec54  // bfmmla v20.4s, v2.8h, v7.8h\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x6e5aec09  // bfmmla v9.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec51  // bfmmla v17.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x6e59ec0d  // bfmmla v13.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec55  // bfmmla v21.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x6e5aec0a  // bfmmla v10.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec52  // bfmmla v18.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x6e59ec0e  // bfmmla v14.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec56  // bfmmla v22.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e5aec0b  // bfmmla v11.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec53  // bfmmla v19.4s, v2.8h, v26.8h\n"
      ".inst 0x6e59ec0f  // bfmmla v15.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec57  // bfmmla v23.4s, v2.8h, v25.8h\n"
      "91:"  // Height 3: Multiply loop: Main loop skip
      "cbz x27, 94f\n"
      "cbz x27, 94f\n"
      "tbz x27, #1, 92f\n"
      "ldr d0, [x26], #0x8\n"
      "ldr d1, [x25], #0x8\n"
      "ldr d2, [x24], #0x8\n"
      "tbz x27, #0, 93f\n"
      "ld1 { v0.s }[2], [x26]\n"
      "ld1 { v1.s }[2], [x25]\n"
      "ld1 { v2.s }[2], [x24]\n"
      "b 93f\n"
      "92:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr s0, [x26, #0x0]\n"
      "ldr s1, [x25, #0x0]\n"
      "ldr s2, [x24, #0x0]\n"
      "93:"  // Height 3: Multiply loop: Ragged operand read: Done
      "ldr q26, [x10, #0x0]\n"
      "ldr q25, [x10, #0x10]\n"
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x6e5aec50  // bfmmla v16.4s, v2.8h, v26.8h\n"
      ".inst 0x6e59ec54  // bfmmla v20.4s, v2.8h, v25.8h\n"
      ".inst 0x6e5aec08  // bfmmla v8.4s, v0.8h, v26.8h\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x6e59ec0c  // bfmmla v12.4s, v0.8h, v25.8h\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x6e5aec09  // bfmmla v9.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec51  // bfmmla v17.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x6e59ec0d  // bfmmla v13.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec55  // bfmmla v21.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x6e5aec0a  // bfmmla v10.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec52  // bfmmla v18.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x6e59ec0e  // bfmmla v14.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec56  // bfmmla v22.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e5aec0b  // bfmmla v11.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec53  // bfmmla v19.4s, v2.8h, v26.8h\n"
      ".inst 0x6e59ec0f  // bfmmla v15.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec57  // bfmmla v23.4s, v2.8h, v25.8h\n"
      "94:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 86b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v6.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "uzp1 v16.2d, v16.2d, v20.2d\n"
      "uzp1 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v19.2d, v19.2d, v23.2d\n"
      "tbz %x[flags], #1, 95f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v26.4s }, [x21]\n"
      "ld1r { v25.4s }, [x20]\n"
      "fmin v6.4s, v6.4s, v26.4s\n"
      "fmin v12.4s, v12.4s, v26.4s\n"
      "fmin v13.4s, v13.4s, v26.4s\n"
      "fmin v14.4s, v14.4s, v26.4s\n"
      "fmin v8.4s, v8.4s, v26.4s\n"
      "fmin v9.4s, v9.4s, v26.4s\n"
      "fmin v10.4s, v10.4s, v26.4s\n"
      "fmin v11.4s, v11.4s, v26.4s\n"
      "fmin v16.4s, v16.4s, v26.4s\n"
      "fmin v17.4s, v17.4s, v26.4s\n"
      "fmin v18.4s, v18.4s, v26.4s\n"
      "fmin v19.4s, v19.4s, v26.4s\n"
      "fmax v6.4s, v6.4s, v25.4s\n"
      "fmax v12.4s, v12.4s, v25.4s\n"
      "fmax v13.4s, v13.4s, v25.4s\n"
      "fmax v14.4s, v14.4s, v25.4s\n"
      "fmax v8.4s, v8.4s, v25.4s\n"
      "fmax v9.4s, v9.4s, v25.4s\n"
      "fmax v10.4s, v10.4s, v25.4s\n"
      "fmax v11.4s, v11.4s, v25.4s\n"
      "fmax v16.4s, v16.4s, v25.4s\n"
      "fmax v17.4s, v17.4s, v25.4s\n"
      "fmax v18.4s, v18.4s, v25.4s\n"
      "fmax v19.4s, v19.4s, v25.4s\n"
      "95:"  // Height 3: No activation
      "cmp x11, #0x10\n"
      "bge 104f\n"
      "tbz x11, #3, 99f\n"
      "st1 { v6.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x26], #0x10\n"
      "st1 { v9.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "st1 { v17.4s }, [x25], #0x10\n"
      "tbz x11, #2, 97f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x26], #0x10\n"
      "st1 { v18.4s }, [x25], #0x10\n"
      "tbz x11, #1, 96f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x26], #0x8\n"
      "str d19, [x25], #0x8\n"
      "tbz x11, #0, 103f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x26]\n"
      "st1 { v19.s }[2], [x25]\n"
      "b 103f\n"
      "96:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x11, #0, 103f\n"
      "str s14, [x9, #0x0]\n"
      "str s11, [x26, #0x0]\n"
      "str s19, [x25, #0x0]\n"
      "b 103f\n"
      "97:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x11, #1, 98f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x26], #0x8\n"
      "str d18, [x25], #0x8\n"
      "tbz x11, #0, 103f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x26]\n"
      "st1 { v18.s }[2], [x25]\n"
      "b 103f\n"
      "98:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x11, #0, 103f\n"
      "str s13, [x9, #0x0]\n"
      "str s10, [x26, #0x0]\n"
      "str s18, [x25, #0x0]\n"
      "b 103f\n"
      "99:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x11, #2, 101f\n"
      "st1 { v6.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x26], #0x10\n"
      "st1 { v16.4s }, [x25], #0x10\n"
      "tbz x11, #1, 100f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x26], #0x8\n"
      "str d17, [x25], #0x8\n"
      "tbz x11, #0, 103f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x26]\n"
      "st1 { v17.s }[2], [x25]\n"
      "b 103f\n"
      "100:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x11, #0, 103f\n"
      "str s12, [x9, #0x0]\n"
      "str s9, [x26, #0x0]\n"
      "str s17, [x25, #0x0]\n"
      "b 103f\n"
      "101:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x11, #1, 102f\n"
      "str d6, [x9], #0x8\n"
      "str d8, [x26], #0x8\n"
      "str d16, [x25], #0x8\n"
      "tbz x11, #0, 103f\n"
      "st1 { v6.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x26]\n"
      "st1 { v16.s }[2], [x25]\n"
      "b 103f\n"
      "102:"  // Height 3: Partial direct writeback: partial_1_0
      "str s6, [x9, #0x0]\n"
      "str s8, [x26, #0x0]\n"
      "str s16, [x25, #0x0]\n"
      "103:"  // Height 3: Partial direct writeback: Done
      "b 105f\n"
      "104:"  // Height 3: Full writeback
      "str q6, [x9, #0x0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x26, #0x0]\n"
      "str q9, [x26, #0x10]\n"
      "str q10, [x26, #0x20]\n"
      "str q11, [x26, #0x30]\n"
      "str q16, [x25, #0x0]\n"
      "str q17, [x25, #0x10]\n"
      "str q18, [x25, #0x20]\n"
      "str q19, [x25, #0x30]\n"
      "105:"  // Height 3: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 72b\n"
      "b 212f\n"
      "106:"  // Height 4
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "107:"  // Height 4: Column loop
      "cbz x12, 108f\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v12.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v21.16b, v13.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v22.16b, v14.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v23.16b, v15.16b\n"
      "b 120f\n"
      "108:"  // Height 4: no bias
      "tbz %x[flags], #0, 119f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "bge 117f\n"
      "tbz x11, #3, 112f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x26], #0x10\n"
      "ld1 { v18.4s }, [x25], #0x10\n"
      "ld1 { v21.4s }, [x24], #0x10\n"
      "tbz x11, #2, 110f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x26], #0x10\n"
      "ld1 { v19.4s }, [x25], #0x10\n"
      "ld1 { v22.4s }, [x24], #0x10\n"
      "tbz x11, #1, 109f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d24, [x25], #0x8\n"
      "ldr d23, [x24], #0x8\n"
      "tbz x11, #0, 116f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x26]\n"
      "ld1 { v24.s }[2], [x25]\n"
      "ld1 { v23.s }[2], [x24]\n"
      "b 116f\n"
      "109:"  // Height 4: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x11, #0, 116f\n"
      "ldr s16, [x9, #0x0]\n"
      "ldr s15, [x26, #0x0]\n"
      "ldr s24, [x25, #0x0]\n"
      "ldr s23, [x24, #0x0]\n"
      "b 116f\n"
      "110:"  // Height 4: Partial accumulate: partial_2_8
      "tbz x11, #1, 111f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d19, [x25], #0x8\n"
      "ldr d22, [x24], #0x8\n"
      "tbz x11, #0, 116f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x26]\n"
      "ld1 { v19.s }[2], [x25]\n"
      "ld1 { v22.s }[2], [x24]\n"
      "b 116f\n"
      "111:"  // Height 4: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x11, #0, 116f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s14, [x26, #0x0]\n"
      "ldr s19, [x25, #0x0]\n"
      "ldr s22, [x24, #0x0]\n"
      "b 116f\n"
      "112:"  // Height 4: Partial accumulate: partial_4_0
      "tbz x11, #2, 114f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "tbz x11, #1, 113f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d18, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "tbz x11, #0, 116f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x26]\n"
      "ld1 { v18.s }[2], [x25]\n"
      "ld1 { v21.s }[2], [x24]\n"
      "b 116f\n"
      "113:"  // Height 4: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x11, #0, 116f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s13, [x26, #0x0]\n"
      "ldr s18, [x25, #0x0]\n"
      "ldr s21, [x24, #0x0]\n"
      "b 116f\n"
      "114:"  // Height 4: Partial accumulate: partial_2_0
      "tbz x11, #1, 115f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "tbz x11, #0, 116f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x26]\n"
      "ld1 { v17.s }[2], [x25]\n"
      "ld1 { v20.s }[2], [x24]\n"
      "b 116f\n"
      "115:"  // Height 4: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "ldr s12, [x26, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s17, [x25, #0x0]\n"
      "ldr s20, [x24, #0x0]\n"
      "116:"  // Height 4: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 118f\n"
      "117:"  // Height 4: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x26, #0x0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q17, [x25, #0x0]\n"
      "ldr q18, [x25, #0x10]\n"
      "ldr q19, [x25, #0x20]\n"
      "ldr q24, [x25, #0x30]\n"
      "ldr q20, [x24, #0x0]\n"
      "ldr q21, [x24, #0x10]\n"
      "ldr q22, [x24, #0x20]\n"
      "ldr q23, [x24, #0x30]\n"
      "118:"  // Height 4: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "b 120f\n"
      "119:"  // Height 4: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "120:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "121:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 122f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 123f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "b 123f\n"
      "122:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "123:"  // Height 4: input setup done
      "cmp x27, #0x4\n"
      "blt 126f\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      "ld1 { v2.4s }, [x24], #0x10\n"
      "cmp x27, #0x8\n"
      "ld1 { v1.4s }, [x25], #0x10\n"
      "ld1 { v3.4s }, [x23], #0x10\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 125f\n"
      "124:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "cmp x27, #0x8\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      "ld1 { v1.4s }, [x25], #0x10\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      "ld1 { v3.4s }, [x23], #0x10\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      ".inst 0x6e46ec50  // bfmmla v16.4s, v2.8h, v6.8h\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x6e47ec54  // bfmmla v20.4s, v2.8h, v7.8h\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x6e5aec09  // bfmmla v9.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec51  // bfmmla v17.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x6e59ec0d  // bfmmla v13.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec55  // bfmmla v21.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x6e5aec0a  // bfmmla v10.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec52  // bfmmla v18.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x6e59ec0e  // bfmmla v14.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec56  // bfmmla v22.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e5aec0b  // bfmmla v11.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec53  // bfmmla v19.4s, v2.8h, v26.8h\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x6e59ec0f  // bfmmla v15.4s, v0.8h, v25.8h\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      ".inst 0x6e59ec57  // bfmmla v23.4s, v2.8h, v25.8h\n"
      "ld1 { v2.4s }, [x24], #0x10\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 124b\n"
      "125:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      ".inst 0x6e46ec50  // bfmmla v16.4s, v2.8h, v6.8h\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x6e47ec54  // bfmmla v20.4s, v2.8h, v7.8h\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x6e5aec09  // bfmmla v9.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec51  // bfmmla v17.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x6e59ec0d  // bfmmla v13.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec55  // bfmmla v21.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x6e5aec0a  // bfmmla v10.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec52  // bfmmla v18.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x6e59ec0e  // bfmmla v14.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec56  // bfmmla v22.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e5aec0b  // bfmmla v11.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec53  // bfmmla v19.4s, v2.8h, v26.8h\n"
      ".inst 0x6e59ec0f  // bfmmla v15.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec57  // bfmmla v23.4s, v2.8h, v25.8h\n"
      "126:"  // Height 4: Multiply loop: Main loop skip
      "cbz x27, 129f\n"
      "cbz x27, 129f\n"
      "tbz x27, #1, 127f\n"
      "ldr d0, [x26], #0x8\n"
      "ldr d1, [x25], #0x8\n"
      "ldr d2, [x24], #0x8\n"
      "ldr d3, [x23], #0x8\n"
      "tbz x27, #0, 128f\n"
      "ld1 { v0.s }[2], [x26]\n"
      "ld1 { v1.s }[2], [x25]\n"
      "ld1 { v2.s }[2], [x24]\n"
      "ld1 { v3.s }[2], [x23]\n"
      "b 128f\n"
      "127:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr s0, [x26, #0x0]\n"
      "ldr s1, [x25, #0x0]\n"
      "ldr s2, [x24, #0x0]\n"
      "ldr s3, [x23, #0x0]\n"
      "128:"  // Height 4: Multiply loop: Ragged operand read: Done
      "ldr q26, [x10, #0x0]\n"
      "ldr q25, [x10, #0x10]\n"
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      ".inst 0x6e5aec08  // bfmmla v8.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec50  // bfmmla v16.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x20]\n"
      ".inst 0x6e59ec0c  // bfmmla v12.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec54  // bfmmla v20.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x30]\n"
      ".inst 0x6e5aec09  // bfmmla v9.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec51  // bfmmla v17.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x40]\n"
      ".inst 0x6e59ec0d  // bfmmla v13.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec55  // bfmmla v21.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x50]\n"
      ".inst 0x6e5aec0a  // bfmmla v10.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec52  // bfmmla v18.4s, v2.8h, v26.8h\n"
      "ldr q26, [x10, #0x60]\n"
      ".inst 0x6e59ec0e  // bfmmla v14.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec56  // bfmmla v22.4s, v2.8h, v25.8h\n"
      "ldr q25, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e5aec0b  // bfmmla v11.4s, v0.8h, v26.8h\n"
      ".inst 0x6e5aec53  // bfmmla v19.4s, v2.8h, v26.8h\n"
      ".inst 0x6e59ec0f  // bfmmla v15.4s, v0.8h, v25.8h\n"
      ".inst 0x6e59ec57  // bfmmla v23.4s, v2.8h, v25.8h\n"
      "129:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 121b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v6.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "tbz %x[flags], #1, 130f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v26.4s }, [x21]\n"
      "ld1r { v25.4s }, [x20]\n"
      "fmin v6.4s, v6.4s, v26.4s\n"
      "fmin v12.4s, v12.4s, v26.4s\n"
      "fmin v13.4s, v13.4s, v26.4s\n"
      "fmin v14.4s, v14.4s, v26.4s\n"
      "fmin v8.4s, v8.4s, v26.4s\n"
      "fmin v9.4s, v9.4s, v26.4s\n"
      "fmin v10.4s, v10.4s, v26.4s\n"
      "fmin v11.4s, v11.4s, v26.4s\n"
      "fmin v15.4s, v15.4s, v26.4s\n"
      "fmin v20.4s, v20.4s, v26.4s\n"
      "fmin v21.4s, v21.4s, v26.4s\n"
      "fmin v22.4s, v22.4s, v26.4s\n"
      "fmin v16.4s, v16.4s, v26.4s\n"
      "fmin v17.4s, v17.4s, v26.4s\n"
      "fmin v18.4s, v18.4s, v26.4s\n"
      "fmin v19.4s, v19.4s, v26.4s\n"
      "fmax v6.4s, v6.4s, v25.4s\n"
      "fmax v12.4s, v12.4s, v25.4s\n"
      "fmax v13.4s, v13.4s, v25.4s\n"
      "fmax v14.4s, v14.4s, v25.4s\n"
      "fmax v8.4s, v8.4s, v25.4s\n"
      "fmax v9.4s, v9.4s, v25.4s\n"
      "fmax v10.4s, v10.4s, v25.4s\n"
      "fmax v11.4s, v11.4s, v25.4s\n"
      "fmax v15.4s, v15.4s, v25.4s\n"
      "fmax v20.4s, v20.4s, v25.4s\n"
      "fmax v21.4s, v21.4s, v25.4s\n"
      "fmax v22.4s, v22.4s, v25.4s\n"
      "fmax v16.4s, v16.4s, v25.4s\n"
      "fmax v17.4s, v17.4s, v25.4s\n"
      "fmax v18.4s, v18.4s, v25.4s\n"
      "fmax v19.4s, v19.4s, v25.4s\n"
      "130:"  // Height 4: No activation
      "cmp x11, #0x10\n"
      "bge 139f\n"
      "tbz x11, #3, 134f\n"
      "st1 { v6.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x26], #0x10\n"
      "st1 { v9.4s }, [x26], #0x10\n"
      "st1 { v15.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x25], #0x10\n"
      "st1 { v16.4s }, [x24], #0x10\n"
      "st1 { v17.4s }, [x24], #0x10\n"
      "tbz x11, #2, 132f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x26], #0x10\n"
      "st1 { v21.4s }, [x25], #0x10\n"
      "st1 { v18.4s }, [x24], #0x10\n"
      "tbz x11, #1, 131f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x26], #0x8\n"
      "str d22, [x25], #0x8\n"
      "str d19, [x24], #0x8\n"
      "tbz x11, #0, 138f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x26]\n"
      "st1 { v22.s }[2], [x25]\n"
      "st1 { v19.s }[2], [x24]\n"
      "b 138f\n"
      "131:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x11, #0, 138f\n"
      "str s14, [x9, #0x0]\n"
      "str s11, [x26, #0x0]\n"
      "str s22, [x25, #0x0]\n"
      "str s19, [x24, #0x0]\n"
      "b 138f\n"
      "132:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x11, #1, 133f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x26], #0x8\n"
      "str d21, [x25], #0x8\n"
      "str d18, [x24], #0x8\n"
      "tbz x11, #0, 138f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x26]\n"
      "st1 { v21.s }[2], [x25]\n"
      "st1 { v18.s }[2], [x24]\n"
      "b 138f\n"
      "133:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x11, #0, 138f\n"
      "str s13, [x9, #0x0]\n"
      "str s10, [x26, #0x0]\n"
      "str s21, [x25, #0x0]\n"
      "str s18, [x24, #0x0]\n"
      "b 138f\n"
      "134:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x11, #2, 136f\n"
      "st1 { v6.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x26], #0x10\n"
      "st1 { v15.4s }, [x25], #0x10\n"
      "st1 { v16.4s }, [x24], #0x10\n"
      "tbz x11, #1, 135f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x26], #0x8\n"
      "str d20, [x25], #0x8\n"
      "str d17, [x24], #0x8\n"
      "tbz x11, #0, 138f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x26]\n"
      "st1 { v20.s }[2], [x25]\n"
      "st1 { v17.s }[2], [x24]\n"
      "b 138f\n"
      "135:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x11, #0, 138f\n"
      "str s12, [x9, #0x0]\n"
      "str s9, [x26, #0x0]\n"
      "str s20, [x25, #0x0]\n"
      "str s17, [x24, #0x0]\n"
      "b 138f\n"
      "136:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x11, #1, 137f\n"
      "str d6, [x9], #0x8\n"
      "str d8, [x26], #0x8\n"
      "str d15, [x25], #0x8\n"
      "str d16, [x24], #0x8\n"
      "tbz x11, #0, 138f\n"
      "st1 { v6.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x26]\n"
      "st1 { v15.s }[2], [x25]\n"
      "st1 { v16.s }[2], [x24]\n"
      "b 138f\n"
      "137:"  // Height 4: Partial direct writeback: partial_1_0
      "str s6, [x9, #0x0]\n"
      "str s8, [x26, #0x0]\n"
      "str s15, [x25, #0x0]\n"
      "str s16, [x24, #0x0]\n"
      "138:"  // Height 4: Partial direct writeback: Done
      "b 140f\n"
      "139:"  // Height 4: Full writeback
      "str q6, [x9, #0x0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x26, #0x0]\n"
      "str q9, [x26, #0x10]\n"
      "str q10, [x26, #0x20]\n"
      "str q11, [x26, #0x30]\n"
      "str q15, [x25, #0x0]\n"
      "str q20, [x25, #0x10]\n"
      "str q21, [x25, #0x20]\n"
      "str q22, [x25, #0x30]\n"
      "str q16, [x24, #0x0]\n"
      "str q17, [x24, #0x10]\n"
      "str q18, [x24, #0x20]\n"
      "str q19, [x24, #0x30]\n"
      "140:"  // Height 4: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 107b\n"
      "b 212f\n"
      "141:"  // Height 5
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "142:"  // Height 5: Column loop
      "cbz x12, 143f\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v12.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v21.16b, v13.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v22.16b, v14.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v23.16b, v15.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v28.16b, v12.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v29.16b, v13.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v30.16b, v14.16b\n"
      "mov v27.16b, v11.16b\n"
      "mov v31.16b, v15.16b\n"
      "b 155f\n"
      "143:"  // Height 5: no bias
      "tbz %x[flags], #0, 154f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "bge 152f\n"
      "tbz x11, #3, 147f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v25.4s }, [x23], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x26], #0x10\n"
      "ld1 { v18.4s }, [x25], #0x10\n"
      "ld1 { v21.4s }, [x24], #0x10\n"
      "ld1 { v26.4s }, [x23], #0x10\n"
      "tbz x11, #2, 145f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x26], #0x10\n"
      "ld1 { v19.4s }, [x25], #0x10\n"
      "ld1 { v22.4s }, [x24], #0x10\n"
      "ld1 { v27.4s }, [x23], #0x10\n"
      "tbz x11, #1, 144f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d24, [x25], #0x8\n"
      "ldr d23, [x24], #0x8\n"
      "ldr d6, [x23], #0x8\n"
      "tbz x11, #0, 151f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x26]\n"
      "ld1 { v24.s }[2], [x25]\n"
      "ld1 { v23.s }[2], [x24]\n"
      "ld1 { v6.s }[2], [x23]\n"
      "b 151f\n"
      "144:"  // Height 5: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x11, #0, 151f\n"
      "ldr s16, [x9, #0x0]\n"
      "ldr s15, [x26, #0x0]\n"
      "ldr s24, [x25, #0x0]\n"
      "ldr s23, [x24, #0x0]\n"
      "ldr s6, [x23, #0x0]\n"
      "b 151f\n"
      "145:"  // Height 5: Partial accumulate: partial_2_8
      "tbz x11, #1, 146f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d19, [x25], #0x8\n"
      "ldr d22, [x24], #0x8\n"
      "ldr d27, [x23], #0x8\n"
      "tbz x11, #0, 151f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x26]\n"
      "ld1 { v19.s }[2], [x25]\n"
      "ld1 { v22.s }[2], [x24]\n"
      "ld1 { v27.s }[2], [x23]\n"
      "b 151f\n"
      "146:"  // Height 5: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x11, #0, 151f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s14, [x26, #0x0]\n"
      "ldr s19, [x25, #0x0]\n"
      "ldr s22, [x24, #0x0]\n"
      "ldr s27, [x23, #0x0]\n"
      "b 151f\n"
      "147:"  // Height 5: Partial accumulate: partial_4_0
      "tbz x11, #2, 149f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v25.4s }, [x23], #0x10\n"
      "tbz x11, #1, 148f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d18, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "ldr d26, [x23], #0x8\n"
      "tbz x11, #0, 151f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x26]\n"
      "ld1 { v18.s }[2], [x25]\n"
      "ld1 { v21.s }[2], [x24]\n"
      "ld1 { v26.s }[2], [x23]\n"
      "b 151f\n"
      "148:"  // Height 5: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x11, #0, 151f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s13, [x26, #0x0]\n"
      "ldr s18, [x25, #0x0]\n"
      "ldr s21, [x24, #0x0]\n"
      "ldr s26, [x23, #0x0]\n"
      "b 151f\n"
      "149:"  // Height 5: Partial accumulate: partial_2_0
      "tbz x11, #1, 150f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "ldr d25, [x23], #0x8\n"
      "tbz x11, #0, 151f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x26]\n"
      "ld1 { v17.s }[2], [x25]\n"
      "ld1 { v20.s }[2], [x24]\n"
      "ld1 { v25.s }[2], [x23]\n"
      "b 151f\n"
      "150:"  // Height 5: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "ldr s12, [x26, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s17, [x25, #0x0]\n"
      "ldr s20, [x24, #0x0]\n"
      "ldr s25, [x23, #0x0]\n"
      "151:"  // Height 5: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 153f\n"
      "152:"  // Height 5: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x26, #0x0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q17, [x25, #0x0]\n"
      "ldr q18, [x25, #0x10]\n"
      "ldr q19, [x25, #0x20]\n"
      "ldr q24, [x25, #0x30]\n"
      "ldr q20, [x24, #0x0]\n"
      "ldr q21, [x24, #0x10]\n"
      "ldr q22, [x24, #0x20]\n"
      "ldr q23, [x24, #0x30]\n"
      "ldr q25, [x23, #0x0]\n"
      "ldr q26, [x23, #0x10]\n"
      "ldr q27, [x23, #0x20]\n"
      "ldr q6, [x23, #0x30]\n"
      "153:"  // Height 5: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "zip1 v24.2d, v25.2d, v28.2d\n"
      "zip2 v28.2d, v25.2d, v28.2d\n"
      "zip1 v25.2d, v26.2d, v29.2d\n"
      "zip2 v29.2d, v26.2d, v29.2d\n"
      "zip1 v26.2d, v27.2d, v30.2d\n"
      "zip2 v30.2d, v27.2d, v30.2d\n"
      "zip1 v27.2d, v6.2d, v31.2d\n"
      "zip2 v31.2d, v6.2d, v31.2d\n"
      "b 155f\n"
      "154:"  // Height 5: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "movi v28.16b, #0x0\n"
      "movi v29.16b, #0x0\n"
      "movi v30.16b, #0x0\n"
      "movi v31.16b, #0x0\n"
      "155:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "156:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 157f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 158f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "b 158f\n"
      "157:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "158:"  // Height 5: input setup done
      "cmp x27, #0x4\n"
      "blt 161f\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      "ld1 { v2.4s }, [x24], #0x10\n"
      "cmp x27, #0x8\n"
      "ld1 { v1.4s }, [x25], #0x10\n"
      "ld1 { v3.4s }, [x23], #0x10\n"
      "ld1 { v4.4s }, [x22], #0x10\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 160f\n"
      "159:"  // Height 5: Multiply loop: Main loop head
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x0ea16884  // bfcvtn v4.4h, v4.4s\n"
      "cmp x27, #0x8\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      "ld1 { v1.4s }, [x25], #0x10\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      "ld1 { v3.4s }, [x23], #0x10\n"
      ".inst 0x6e46ec98  // bfmmla v24.4s, v4.8h, v6.8h\n"
      ".inst 0x6e47ec9c  // bfmmla v28.4s, v4.8h, v7.8h\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      ".inst 0x6e46ec50  // bfmmla v16.4s, v2.8h, v6.8h\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x6e47ec54  // bfmmla v20.4s, v2.8h, v7.8h\n"
      "ldr q5, [x10, #0x30]\n"
      ".inst 0x6e46ec09  // bfmmla v9.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec51  // bfmmla v17.4s, v2.8h, v6.8h\n"
      ".inst 0x6e46ec99  // bfmmla v25.4s, v4.8h, v6.8h\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x6e45ec0d  // bfmmla v13.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec55  // bfmmla v21.4s, v2.8h, v5.8h\n"
      ".inst 0x6e45ec9d  // bfmmla v29.4s, v4.8h, v5.8h\n"
      "ldr q5, [x10, #0x50]\n"
      ".inst 0x6e46ec0a  // bfmmla v10.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec52  // bfmmla v18.4s, v2.8h, v6.8h\n"
      ".inst 0x6e46ec9a  // bfmmla v26.4s, v4.8h, v6.8h\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x6e45ec0e  // bfmmla v14.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec56  // bfmmla v22.4s, v2.8h, v5.8h\n"
      ".inst 0x6e45ec9e  // bfmmla v30.4s, v4.8h, v5.8h\n"
      "ldr q5, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e46ec0b  // bfmmla v11.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec53  // bfmmla v19.4s, v2.8h, v6.8h\n"
      ".inst 0x6e46ec9b  // bfmmla v27.4s, v4.8h, v6.8h\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x6e45ec0f  // bfmmla v15.4s, v0.8h, v5.8h\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      ".inst 0x6e45ec57  // bfmmla v23.4s, v2.8h, v5.8h\n"
      "ld1 { v2.4s }, [x24], #0x10\n"
      ".inst 0x6e45ec9f  // bfmmla v31.4s, v4.8h, v5.8h\n"
      "ld1 { v4.4s }, [x22], #0x10\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 159b\n"
      "160:"  // Height 5: Multiply loop: Single iteration only
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x0ea16884  // bfcvtn v4.4h, v4.4s\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      ".inst 0x6e46ec98  // bfmmla v24.4s, v4.8h, v6.8h\n"
      ".inst 0x6e47ec9c  // bfmmla v28.4s, v4.8h, v7.8h\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      ".inst 0x6e46ec50  // bfmmla v16.4s, v2.8h, v6.8h\n"
      "ldr q3, [x10, #0x20]\n"
      ".inst 0x6e47ec54  // bfmmla v20.4s, v2.8h, v7.8h\n"
      "ldr q1, [x10, #0x30]\n"
      ".inst 0x6e43ec09  // bfmmla v9.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec51  // bfmmla v17.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec99  // bfmmla v25.4s, v4.8h, v3.8h\n"
      "ldr q3, [x10, #0x40]\n"
      ".inst 0x6e41ec0d  // bfmmla v13.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec55  // bfmmla v21.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9d  // bfmmla v29.4s, v4.8h, v1.8h\n"
      "ldr q1, [x10, #0x50]\n"
      ".inst 0x6e43ec0a  // bfmmla v10.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec52  // bfmmla v18.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec9a  // bfmmla v26.4s, v4.8h, v3.8h\n"
      "ldr q3, [x10, #0x60]\n"
      ".inst 0x6e41ec0e  // bfmmla v14.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec56  // bfmmla v22.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9e  // bfmmla v30.4s, v4.8h, v1.8h\n"
      "ldr q1, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e43ec0b  // bfmmla v11.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec53  // bfmmla v19.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec9b  // bfmmla v27.4s, v4.8h, v3.8h\n"
      ".inst 0x6e41ec0f  // bfmmla v15.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec57  // bfmmla v23.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9f  // bfmmla v31.4s, v4.8h, v1.8h\n"
      "161:"  // Height 5: Multiply loop: Main loop skip
      "cbz x27, 164f\n"
      "cbz x27, 164f\n"
      "tbz x27, #1, 162f\n"
      "ldr d0, [x26], #0x8\n"
      "ldr d1, [x25], #0x8\n"
      "ldr d2, [x24], #0x8\n"
      "ldr d3, [x23], #0x8\n"
      "ldr d4, [x22], #0x8\n"
      "tbz x27, #0, 163f\n"
      "ld1 { v0.s }[2], [x26]\n"
      "ld1 { v1.s }[2], [x25]\n"
      "ld1 { v2.s }[2], [x24]\n"
      "ld1 { v3.s }[2], [x23]\n"
      "ld1 { v4.s }[2], [x22]\n"
      "b 163f\n"
      "162:"  // Height 5: Multiply loop: Ragged operand read: partial_1_0
      "ldr s0, [x26, #0x0]\n"
      "ldr s1, [x25, #0x0]\n"
      "ldr s2, [x24, #0x0]\n"
      "ldr s3, [x23, #0x0]\n"
      "ldr s4, [x22, #0x0]\n"
      "163:"  // Height 5: Multiply loop: Ragged operand read: Done
      "ldr q6, [x10, #0x0]\n"
      "ldr q5, [x10, #0x10]\n"
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      ".inst 0x0ea16884  // bfcvtn v4.4h, v4.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec98  // bfmmla v24.4s, v4.8h, v6.8h\n"
      ".inst 0x6e45ec0c  // bfmmla v12.4s, v0.8h, v5.8h\n"
      ".inst 0x6e45ec9c  // bfmmla v28.4s, v4.8h, v5.8h\n"
      ".inst 0x6e46ec50  // bfmmla v16.4s, v2.8h, v6.8h\n"
      "ldr q3, [x10, #0x20]\n"
      ".inst 0x6e45ec54  // bfmmla v20.4s, v2.8h, v5.8h\n"
      "ldr q1, [x10, #0x30]\n"
      ".inst 0x6e43ec09  // bfmmla v9.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec51  // bfmmla v17.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec99  // bfmmla v25.4s, v4.8h, v3.8h\n"
      "ldr q3, [x10, #0x40]\n"
      ".inst 0x6e41ec0d  // bfmmla v13.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec55  // bfmmla v21.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9d  // bfmmla v29.4s, v4.8h, v1.8h\n"
      "ldr q1, [x10, #0x50]\n"
      ".inst 0x6e43ec0a  // bfmmla v10.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec52  // bfmmla v18.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec9a  // bfmmla v26.4s, v4.8h, v3.8h\n"
      "ldr q3, [x10, #0x60]\n"
      ".inst 0x6e41ec0e  // bfmmla v14.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec56  // bfmmla v22.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9e  // bfmmla v30.4s, v4.8h, v1.8h\n"
      "ldr q1, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e43ec0b  // bfmmla v11.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec53  // bfmmla v19.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec9b  // bfmmla v27.4s, v4.8h, v3.8h\n"
      ".inst 0x6e41ec0f  // bfmmla v15.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec57  // bfmmla v23.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9f  // bfmmla v31.4s, v4.8h, v1.8h\n"
      "164:"  // Height 5: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 156b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v6.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "uzp1 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v27.2d, v27.2d, v31.2d\n"
      "tbz %x[flags], #1, 165f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v1.4s }, [x21]\n"
      "ld1r { v0.4s }, [x20]\n"
      "fmin v6.4s, v6.4s, v1.4s\n"
      "fmin v12.4s, v12.4s, v1.4s\n"
      "fmin v13.4s, v13.4s, v1.4s\n"
      "fmin v14.4s, v14.4s, v1.4s\n"
      "fmin v8.4s, v8.4s, v1.4s\n"
      "fmin v9.4s, v9.4s, v1.4s\n"
      "fmin v10.4s, v10.4s, v1.4s\n"
      "fmin v11.4s, v11.4s, v1.4s\n"
      "fmin v15.4s, v15.4s, v1.4s\n"
      "fmin v20.4s, v20.4s, v1.4s\n"
      "fmin v21.4s, v21.4s, v1.4s\n"
      "fmin v22.4s, v22.4s, v1.4s\n"
      "fmin v16.4s, v16.4s, v1.4s\n"
      "fmin v17.4s, v17.4s, v1.4s\n"
      "fmin v18.4s, v18.4s, v1.4s\n"
      "fmin v19.4s, v19.4s, v1.4s\n"
      "fmin v24.4s, v24.4s, v1.4s\n"
      "fmin v25.4s, v25.4s, v1.4s\n"
      "fmin v26.4s, v26.4s, v1.4s\n"
      "fmin v27.4s, v27.4s, v1.4s\n"
      "fmax v6.4s, v6.4s, v0.4s\n"
      "fmax v12.4s, v12.4s, v0.4s\n"
      "fmax v13.4s, v13.4s, v0.4s\n"
      "fmax v14.4s, v14.4s, v0.4s\n"
      "fmax v8.4s, v8.4s, v0.4s\n"
      "fmax v9.4s, v9.4s, v0.4s\n"
      "fmax v10.4s, v10.4s, v0.4s\n"
      "fmax v11.4s, v11.4s, v0.4s\n"
      "fmax v15.4s, v15.4s, v0.4s\n"
      "fmax v20.4s, v20.4s, v0.4s\n"
      "fmax v21.4s, v21.4s, v0.4s\n"
      "fmax v22.4s, v22.4s, v0.4s\n"
      "fmax v16.4s, v16.4s, v0.4s\n"
      "fmax v17.4s, v17.4s, v0.4s\n"
      "fmax v18.4s, v18.4s, v0.4s\n"
      "fmax v19.4s, v19.4s, v0.4s\n"
      "fmax v24.4s, v24.4s, v0.4s\n"
      "fmax v25.4s, v25.4s, v0.4s\n"
      "fmax v26.4s, v26.4s, v0.4s\n"
      "fmax v27.4s, v27.4s, v0.4s\n"
      "165:"  // Height 5: No activation
      "cmp x11, #0x10\n"
      "bge 174f\n"
      "tbz x11, #3, 169f\n"
      "st1 { v6.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x26], #0x10\n"
      "st1 { v9.4s }, [x26], #0x10\n"
      "st1 { v15.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x25], #0x10\n"
      "st1 { v16.4s }, [x24], #0x10\n"
      "st1 { v17.4s }, [x24], #0x10\n"
      "st1 { v24.4s }, [x23], #0x10\n"
      "st1 { v25.4s }, [x23], #0x10\n"
      "tbz x11, #2, 167f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x26], #0x10\n"
      "st1 { v21.4s }, [x25], #0x10\n"
      "st1 { v18.4s }, [x24], #0x10\n"
      "st1 { v26.4s }, [x23], #0x10\n"
      "tbz x11, #1, 166f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x26], #0x8\n"
      "str d22, [x25], #0x8\n"
      "str d19, [x24], #0x8\n"
      "str d27, [x23], #0x8\n"
      "tbz x11, #0, 173f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x26]\n"
      "st1 { v22.s }[2], [x25]\n"
      "st1 { v19.s }[2], [x24]\n"
      "st1 { v27.s }[2], [x23]\n"
      "b 173f\n"
      "166:"  // Height 5: Partial direct writeback: partial_1_12
      "tbz x11, #0, 173f\n"
      "str s14, [x9, #0x0]\n"
      "str s11, [x26, #0x0]\n"
      "str s22, [x25, #0x0]\n"
      "str s19, [x24, #0x0]\n"
      "str s27, [x23, #0x0]\n"
      "b 173f\n"
      "167:"  // Height 5: Partial direct writeback: partial_2_8
      "tbz x11, #1, 168f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x26], #0x8\n"
      "str d21, [x25], #0x8\n"
      "str d18, [x24], #0x8\n"
      "str d26, [x23], #0x8\n"
      "tbz x11, #0, 173f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x26]\n"
      "st1 { v21.s }[2], [x25]\n"
      "st1 { v18.s }[2], [x24]\n"
      "st1 { v26.s }[2], [x23]\n"
      "b 173f\n"
      "168:"  // Height 5: Partial direct writeback: partial_1_8
      "tbz x11, #0, 173f\n"
      "str s13, [x9, #0x0]\n"
      "str s10, [x26, #0x0]\n"
      "str s21, [x25, #0x0]\n"
      "str s18, [x24, #0x0]\n"
      "str s26, [x23, #0x0]\n"
      "b 173f\n"
      "169:"  // Height 5: Partial direct writeback: partial_4_0
      "tbz x11, #2, 171f\n"
      "st1 { v6.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x26], #0x10\n"
      "st1 { v15.4s }, [x25], #0x10\n"
      "st1 { v16.4s }, [x24], #0x10\n"
      "st1 { v24.4s }, [x23], #0x10\n"
      "tbz x11, #1, 170f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x26], #0x8\n"
      "str d20, [x25], #0x8\n"
      "str d17, [x24], #0x8\n"
      "str d25, [x23], #0x8\n"
      "tbz x11, #0, 173f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x26]\n"
      "st1 { v20.s }[2], [x25]\n"
      "st1 { v17.s }[2], [x24]\n"
      "st1 { v25.s }[2], [x23]\n"
      "b 173f\n"
      "170:"  // Height 5: Partial direct writeback: partial_1_4
      "tbz x11, #0, 173f\n"
      "str s12, [x9, #0x0]\n"
      "str s9, [x26, #0x0]\n"
      "str s20, [x25, #0x0]\n"
      "str s17, [x24, #0x0]\n"
      "str s25, [x23, #0x0]\n"
      "b 173f\n"
      "171:"  // Height 5: Partial direct writeback: partial_2_0
      "tbz x11, #1, 172f\n"
      "str d6, [x9], #0x8\n"
      "str d8, [x26], #0x8\n"
      "str d15, [x25], #0x8\n"
      "str d16, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "tbz x11, #0, 173f\n"
      "st1 { v6.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x26]\n"
      "st1 { v15.s }[2], [x25]\n"
      "st1 { v16.s }[2], [x24]\n"
      "st1 { v24.s }[2], [x23]\n"
      "b 173f\n"
      "172:"  // Height 5: Partial direct writeback: partial_1_0
      "str s6, [x9, #0x0]\n"
      "str s8, [x26, #0x0]\n"
      "str s15, [x25, #0x0]\n"
      "str s16, [x24, #0x0]\n"
      "str s24, [x23, #0x0]\n"
      "173:"  // Height 5: Partial direct writeback: Done
      "b 175f\n"
      "174:"  // Height 5: Full writeback
      "str q6, [x9, #0x0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x26, #0x0]\n"
      "str q9, [x26, #0x10]\n"
      "str q10, [x26, #0x20]\n"
      "str q11, [x26, #0x30]\n"
      "str q15, [x25, #0x0]\n"
      "str q20, [x25, #0x10]\n"
      "str q21, [x25, #0x20]\n"
      "str q22, [x25, #0x30]\n"
      "str q16, [x24, #0x0]\n"
      "str q17, [x24, #0x10]\n"
      "str q18, [x24, #0x20]\n"
      "str q19, [x24, #0x30]\n"
      "str q24, [x23, #0x0]\n"
      "str q25, [x23, #0x10]\n"
      "str q26, [x23, #0x20]\n"
      "str q27, [x23, #0x30]\n"
      "175:"  // Height 5: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 142b\n"
      "b 212f\n"
      "176:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x18\n"
      "ldr x12, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd x20, x21, x20, x9\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "177:"  // Height 6: Column loop
      "cbz x12, 178f\n"
      "ldr q8, [x12, #0x0]\n"
      "ldr q9, [x12, #0x10]\n"
      "ldr q10, [x12, #0x20]\n"
      "ldr q11, [x12, #0x30]\n"
      "add x12, x12, #0x40\n"
      "zip2 v12.2d, v8.2d, v8.2d\n"
      "zip1 v8.2d, v8.2d, v8.2d\n"
      "zip2 v13.2d, v9.2d, v9.2d\n"
      "zip1 v9.2d, v9.2d, v9.2d\n"
      "zip2 v14.2d, v10.2d, v10.2d\n"
      "zip1 v10.2d, v10.2d, v10.2d\n"
      "zip2 v15.2d, v11.2d, v11.2d\n"
      "zip1 v11.2d, v11.2d, v11.2d\n"
      "mov v16.16b, v8.16b\n"
      "mov v20.16b, v12.16b\n"
      "mov v17.16b, v9.16b\n"
      "mov v21.16b, v13.16b\n"
      "mov v18.16b, v10.16b\n"
      "mov v22.16b, v14.16b\n"
      "mov v19.16b, v11.16b\n"
      "mov v23.16b, v15.16b\n"
      "mov v24.16b, v8.16b\n"
      "mov v28.16b, v12.16b\n"
      "mov v25.16b, v9.16b\n"
      "mov v29.16b, v13.16b\n"
      "mov v26.16b, v10.16b\n"
      "mov v30.16b, v14.16b\n"
      "mov v27.16b, v11.16b\n"
      "mov v31.16b, v15.16b\n"
      "b 190f\n"
      "178:"  // Height 6: no bias
      "tbz %x[flags], #0, 189f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "cmp x11, #0x10\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "bge 187f\n"
      "tbz x11, #3, 182f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v25.4s }, [x23], #0x10\n"
      "ld1 { v28.4s }, [x22], #0x10\n"
      "ld1 { v10.4s }, [x9], #0x10\n"
      "ld1 { v13.4s }, [x26], #0x10\n"
      "ld1 { v18.4s }, [x25], #0x10\n"
      "ld1 { v21.4s }, [x24], #0x10\n"
      "ld1 { v26.4s }, [x23], #0x10\n"
      "ld1 { v29.4s }, [x22], #0x10\n"
      "tbz x11, #2, 180f\n"
      "ld1 { v11.4s }, [x9], #0x10\n"
      "ld1 { v14.4s }, [x26], #0x10\n"
      "ld1 { v19.4s }, [x25], #0x10\n"
      "ld1 { v22.4s }, [x24], #0x10\n"
      "ld1 { v27.4s }, [x23], #0x10\n"
      "ld1 { v30.4s }, [x22], #0x10\n"
      "tbz x11, #1, 179f\n"
      "ldr d16, [x9], #0x8\n"
      "ldr d15, [x26], #0x8\n"
      "mov x20, #0x38\n"
      "ldr d24, [x25], #0x8\n"
      "ldr d23, [x24], #0x8\n"
      "ldr d6, [x23], #0x8\n"
      "ldr d31, [x22], #0x8\n"
      "tbz x11, #0, 186f\n"
      "ld1 { v16.s }[2], [x9]\n"
      "ld1 { v15.s }[2], [x26]\n"
      "ld1 { v24.s }[2], [x25]\n"
      "ld1 { v23.s }[2], [x24]\n"
      "ld1 { v6.s }[2], [x23]\n"
      "ld1 { v31.s }[2], [x22]\n"
      "b 186f\n"
      "179:"  // Height 6: Partial accumulate: partial_1_12
      "mov x20, #0x30\n"
      "tbz x11, #0, 186f\n"
      "ldr s16, [x9, #0x0]\n"
      "ldr s15, [x26, #0x0]\n"
      "ldr s24, [x25, #0x0]\n"
      "ldr s23, [x24, #0x0]\n"
      "ldr s6, [x23, #0x0]\n"
      "ldr s31, [x22, #0x0]\n"
      "b 186f\n"
      "180:"  // Height 6: Partial accumulate: partial_2_8
      "tbz x11, #1, 181f\n"
      "ldr d11, [x9], #0x8\n"
      "ldr d14, [x26], #0x8\n"
      "mov x20, #0x28\n"
      "ldr d19, [x25], #0x8\n"
      "ldr d22, [x24], #0x8\n"
      "ldr d27, [x23], #0x8\n"
      "ldr d30, [x22], #0x8\n"
      "tbz x11, #0, 186f\n"
      "ld1 { v11.s }[2], [x9]\n"
      "ld1 { v14.s }[2], [x26]\n"
      "ld1 { v19.s }[2], [x25]\n"
      "ld1 { v22.s }[2], [x24]\n"
      "ld1 { v27.s }[2], [x23]\n"
      "ld1 { v30.s }[2], [x22]\n"
      "b 186f\n"
      "181:"  // Height 6: Partial accumulate: partial_1_8
      "mov x20, #0x20\n"
      "tbz x11, #0, 186f\n"
      "ldr s11, [x9, #0x0]\n"
      "ldr s14, [x26, #0x0]\n"
      "ldr s19, [x25, #0x0]\n"
      "ldr s22, [x24, #0x0]\n"
      "ldr s27, [x23, #0x0]\n"
      "ldr s30, [x22, #0x0]\n"
      "b 186f\n"
      "182:"  // Height 6: Partial accumulate: partial_4_0
      "tbz x11, #2, 184f\n"
      "ld1 { v9.4s }, [x9], #0x10\n"
      "ld1 { v12.4s }, [x26], #0x10\n"
      "ld1 { v17.4s }, [x25], #0x10\n"
      "ld1 { v20.4s }, [x24], #0x10\n"
      "ld1 { v25.4s }, [x23], #0x10\n"
      "ld1 { v28.4s }, [x22], #0x10\n"
      "tbz x11, #1, 183f\n"
      "ldr d10, [x9], #0x8\n"
      "ldr d13, [x26], #0x8\n"
      "mov x20, #0x18\n"
      "ldr d18, [x25], #0x8\n"
      "ldr d21, [x24], #0x8\n"
      "ldr d26, [x23], #0x8\n"
      "ldr d29, [x22], #0x8\n"
      "tbz x11, #0, 186f\n"
      "ld1 { v10.s }[2], [x9]\n"
      "ld1 { v13.s }[2], [x26]\n"
      "ld1 { v18.s }[2], [x25]\n"
      "ld1 { v21.s }[2], [x24]\n"
      "ld1 { v26.s }[2], [x23]\n"
      "ld1 { v29.s }[2], [x22]\n"
      "b 186f\n"
      "183:"  // Height 6: Partial accumulate: partial_1_4
      "mov x20, #0x10\n"
      "tbz x11, #0, 186f\n"
      "ldr s10, [x9, #0x0]\n"
      "ldr s13, [x26, #0x0]\n"
      "ldr s18, [x25, #0x0]\n"
      "ldr s21, [x24, #0x0]\n"
      "ldr s26, [x23, #0x0]\n"
      "ldr s29, [x22, #0x0]\n"
      "b 186f\n"
      "184:"  // Height 6: Partial accumulate: partial_2_0
      "tbz x11, #1, 185f\n"
      "ldr d9, [x9], #0x8\n"
      "ldr d12, [x26], #0x8\n"
      "mov x20, #0x8\n"
      "ldr d17, [x25], #0x8\n"
      "ldr d20, [x24], #0x8\n"
      "ldr d25, [x23], #0x8\n"
      "ldr d28, [x22], #0x8\n"
      "tbz x11, #0, 186f\n"
      "ld1 { v9.s }[2], [x9]\n"
      "ld1 { v12.s }[2], [x26]\n"
      "ld1 { v17.s }[2], [x25]\n"
      "ld1 { v20.s }[2], [x24]\n"
      "ld1 { v25.s }[2], [x23]\n"
      "ld1 { v28.s }[2], [x22]\n"
      "b 186f\n"
      "185:"  // Height 6: Partial accumulate: partial_1_0
      "ldr s9, [x9, #0x0]\n"
      "ldr s12, [x26, #0x0]\n"
      "mov x20, #0x0\n"
      "ldr s17, [x25, #0x0]\n"
      "ldr s20, [x24, #0x0]\n"
      "ldr s25, [x23, #0x0]\n"
      "ldr s28, [x22, #0x0]\n"
      "186:"  // Height 6: Partial accumulate: Done
      "sub x9, x9, x20\n"
      "b 188f\n"
      "187:"  // Height 6: full accumulate
      "ldr q9, [x9, #0x0]\n"
      "ldr q10, [x9, #0x10]\n"
      "ldr q11, [x9, #0x20]\n"
      "ldr q16, [x9, #0x30]\n"
      "ldr q12, [x26, #0x0]\n"
      "ldr q13, [x26, #0x10]\n"
      "ldr q14, [x26, #0x20]\n"
      "ldr q15, [x26, #0x30]\n"
      "ldr q17, [x25, #0x0]\n"
      "ldr q18, [x25, #0x10]\n"
      "ldr q19, [x25, #0x20]\n"
      "ldr q24, [x25, #0x30]\n"
      "ldr q20, [x24, #0x0]\n"
      "ldr q21, [x24, #0x10]\n"
      "ldr q22, [x24, #0x20]\n"
      "ldr q23, [x24, #0x30]\n"
      "ldr q25, [x23, #0x0]\n"
      "ldr q26, [x23, #0x10]\n"
      "ldr q27, [x23, #0x20]\n"
      "ldr q6, [x23, #0x30]\n"
      "ldr q28, [x22, #0x0]\n"
      "ldr q29, [x22, #0x10]\n"
      "ldr q30, [x22, #0x20]\n"
      "ldr q31, [x22, #0x30]\n"
      "188:"  // Height 6: MMLA fixup
      "zip1 v8.2d, v9.2d, v12.2d\n"
      "zip2 v12.2d, v9.2d, v12.2d\n"
      "zip1 v9.2d, v10.2d, v13.2d\n"
      "zip2 v13.2d, v10.2d, v13.2d\n"
      "zip1 v10.2d, v11.2d, v14.2d\n"
      "zip2 v14.2d, v11.2d, v14.2d\n"
      "zip1 v11.2d, v16.2d, v15.2d\n"
      "zip2 v15.2d, v16.2d, v15.2d\n"
      "zip1 v16.2d, v17.2d, v20.2d\n"
      "zip2 v20.2d, v17.2d, v20.2d\n"
      "zip1 v17.2d, v18.2d, v21.2d\n"
      "zip2 v21.2d, v18.2d, v21.2d\n"
      "zip1 v18.2d, v19.2d, v22.2d\n"
      "zip2 v22.2d, v19.2d, v22.2d\n"
      "zip1 v19.2d, v24.2d, v23.2d\n"
      "zip2 v23.2d, v24.2d, v23.2d\n"
      "zip1 v24.2d, v25.2d, v28.2d\n"
      "zip2 v28.2d, v25.2d, v28.2d\n"
      "zip1 v25.2d, v26.2d, v29.2d\n"
      "zip2 v29.2d, v26.2d, v29.2d\n"
      "zip1 v26.2d, v27.2d, v30.2d\n"
      "zip2 v30.2d, v27.2d, v30.2d\n"
      "zip1 v27.2d, v6.2d, v31.2d\n"
      "zip2 v31.2d, v6.2d, v31.2d\n"
      "b 190f\n"
      "189:"  // Height 6: no accumulate
      "movi v8.16b, #0x0\n"
      "movi v9.16b, #0x0\n"
      "movi v10.16b, #0x0\n"
      "movi v11.16b, #0x0\n"
      "movi v12.16b, #0x0\n"
      "movi v13.16b, #0x0\n"
      "movi v14.16b, #0x0\n"
      "movi v15.16b, #0x0\n"
      "movi v16.16b, #0x0\n"
      "movi v17.16b, #0x0\n"
      "movi v18.16b, #0x0\n"
      "movi v19.16b, #0x0\n"
      "movi v20.16b, #0x0\n"
      "movi v21.16b, #0x0\n"
      "movi v22.16b, #0x0\n"
      "movi v23.16b, #0x0\n"
      "movi v24.16b, #0x0\n"
      "movi v25.16b, #0x0\n"
      "movi v26.16b, #0x0\n"
      "movi v27.16b, #0x0\n"
      "movi v28.16b, #0x0\n"
      "movi v29.16b, #0x0\n"
      "movi v30.16b, #0x0\n"
      "movi v31.16b, #0x0\n"
      "190:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "191:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 192f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 193f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "add x21, x21, x20, LSL #2\n"
      "b 193f\n"
      "192:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "add x21, x22, x21, LSL #2\n"
      "193:"  // Height 6: input setup done
      "cmp x27, #0x4\n"
      "blt 196f\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      "ld1 { v2.4s }, [x24], #0x10\n"
      "cmp x27, #0x8\n"
      "ld1 { v4.4s }, [x22], #0x10\n"
      "ld1 { v1.4s }, [x25], #0x10\n"
      "ld1 { v3.4s }, [x23], #0x10\n"
      "ld1 { v5.4s }, [x21], #0x10\n"
      "ldr q6, [x10, #0x0]\n"
      "ldr q7, [x10, #0x10]\n"
      "blt 195f\n"
      "194:"  // Height 6: Multiply loop: Main loop head
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x0ea16884  // bfcvtn v4.4h, v4.4s\n"
      "cmp x27, #0x8\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      "ld1 { v1.4s }, [x25], #0x10\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      "ld1 { v3.4s }, [x23], #0x10\n"
      ".inst 0x4ea168a4  // bfcvtn2 v4.8h, v5.4s\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      "ld1 { v5.4s }, [x21], #0x10\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      ".inst 0x6e46ec50  // bfmmla v16.4s, v2.8h, v6.8h\n"
      ".inst 0x6e47ec54  // bfmmla v20.4s, v2.8h, v7.8h\n"
      ".inst 0x6e46ec98  // bfmmla v24.4s, v4.8h, v6.8h\n"
      "ldr q6, [x10, #0x20]\n"
      ".inst 0x6e47ec9c  // bfmmla v28.4s, v4.8h, v7.8h\n"
      "ldr q7, [x10, #0x30]\n"
      ".inst 0x6e46ec09  // bfmmla v9.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec51  // bfmmla v17.4s, v2.8h, v6.8h\n"
      ".inst 0x6e46ec99  // bfmmla v25.4s, v4.8h, v6.8h\n"
      "ldr q6, [x10, #0x40]\n"
      ".inst 0x6e47ec0d  // bfmmla v13.4s, v0.8h, v7.8h\n"
      ".inst 0x6e47ec55  // bfmmla v21.4s, v2.8h, v7.8h\n"
      ".inst 0x6e47ec9d  // bfmmla v29.4s, v4.8h, v7.8h\n"
      "ldr q7, [x10, #0x50]\n"
      ".inst 0x6e46ec0a  // bfmmla v10.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec52  // bfmmla v18.4s, v2.8h, v6.8h\n"
      ".inst 0x6e46ec9a  // bfmmla v26.4s, v4.8h, v6.8h\n"
      "ldr q6, [x10, #0x60]\n"
      ".inst 0x6e47ec0e  // bfmmla v14.4s, v0.8h, v7.8h\n"
      ".inst 0x6e47ec56  // bfmmla v22.4s, v2.8h, v7.8h\n"
      ".inst 0x6e47ec9e  // bfmmla v30.4s, v4.8h, v7.8h\n"
      "ldr q7, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e46ec0b  // bfmmla v11.4s, v0.8h, v6.8h\n"
      ".inst 0x6e46ec53  // bfmmla v19.4s, v2.8h, v6.8h\n"
      ".inst 0x6e46ec9b  // bfmmla v27.4s, v4.8h, v6.8h\n"
      "ldr q6, [x10, #0x0]\n"
      ".inst 0x6e47ec0f  // bfmmla v15.4s, v0.8h, v7.8h\n"
      "ld1 { v0.4s }, [x26], #0x10\n"
      ".inst 0x6e47ec57  // bfmmla v23.4s, v2.8h, v7.8h\n"
      "ld1 { v2.4s }, [x24], #0x10\n"
      ".inst 0x6e47ec9f  // bfmmla v31.4s, v4.8h, v7.8h\n"
      "ld1 { v4.4s }, [x22], #0x10\n"
      "ldr q7, [x10, #0x10]\n"
      "bge 194b\n"
      "195:"  // Height 6: Multiply loop: Single iteration only
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      "sub x27, x27, #0x4\n"
      "prfm pldl1keep, [x26, #0x80]\n"
      ".inst 0x0ea16884  // bfcvtn v4.4h, v4.4s\n"
      "prfm pldl1keep, [x25, #0x80]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      ".inst 0x4ea168a4  // bfcvtn2 v4.8h, v5.4s\n"
      ".inst 0x6e46ec08  // bfmmla v8.4s, v0.8h, v6.8h\n"
      ".inst 0x6e47ec0c  // bfmmla v12.4s, v0.8h, v7.8h\n"
      ".inst 0x6e46ec50  // bfmmla v16.4s, v2.8h, v6.8h\n"
      ".inst 0x6e47ec54  // bfmmla v20.4s, v2.8h, v7.8h\n"
      ".inst 0x6e46ec98  // bfmmla v24.4s, v4.8h, v6.8h\n"
      "ldr q3, [x10, #0x20]\n"
      ".inst 0x6e47ec9c  // bfmmla v28.4s, v4.8h, v7.8h\n"
      "ldr q1, [x10, #0x30]\n"
      ".inst 0x6e43ec09  // bfmmla v9.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec51  // bfmmla v17.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec99  // bfmmla v25.4s, v4.8h, v3.8h\n"
      "ldr q3, [x10, #0x40]\n"
      ".inst 0x6e41ec0d  // bfmmla v13.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec55  // bfmmla v21.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9d  // bfmmla v29.4s, v4.8h, v1.8h\n"
      "ldr q1, [x10, #0x50]\n"
      ".inst 0x6e43ec0a  // bfmmla v10.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec52  // bfmmla v18.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec9a  // bfmmla v26.4s, v4.8h, v3.8h\n"
      "ldr q3, [x10, #0x60]\n"
      ".inst 0x6e41ec0e  // bfmmla v14.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec56  // bfmmla v22.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9e  // bfmmla v30.4s, v4.8h, v1.8h\n"
      "ldr q1, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e43ec0b  // bfmmla v11.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec53  // bfmmla v19.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec9b  // bfmmla v27.4s, v4.8h, v3.8h\n"
      ".inst 0x6e41ec0f  // bfmmla v15.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec57  // bfmmla v23.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9f  // bfmmla v31.4s, v4.8h, v1.8h\n"
      "196:"  // Height 6: Multiply loop: Main loop skip
      "cbz x27, 199f\n"
      "cbz x27, 199f\n"
      "tbz x27, #1, 197f\n"
      "ldr d0, [x26], #0x8\n"
      "ldr d1, [x25], #0x8\n"
      "ldr d2, [x24], #0x8\n"
      "ldr d3, [x23], #0x8\n"
      "ldr d4, [x22], #0x8\n"
      "ldr d5, [x21], #0x8\n"
      "tbz x27, #0, 198f\n"
      "ld1 { v0.s }[2], [x26]\n"
      "ld1 { v1.s }[2], [x25]\n"
      "ld1 { v2.s }[2], [x24]\n"
      "ld1 { v3.s }[2], [x23]\n"
      "ld1 { v4.s }[2], [x22]\n"
      "ld1 { v5.s }[2], [x21]\n"
      "b 198f\n"
      "197:"  // Height 6: Multiply loop: Ragged operand read: partial_1_0
      "ldr s0, [x26, #0x0]\n"
      "ldr s1, [x25, #0x0]\n"
      "ldr s2, [x24, #0x0]\n"
      "ldr s3, [x23, #0x0]\n"
      "ldr s4, [x22, #0x0]\n"
      "ldr s5, [x21, #0x0]\n"
      "198:"  // Height 6: Multiply loop: Ragged operand read: Done
      "ldr q7, [x10, #0x0]\n"
      "ldr q6, [x10, #0x10]\n"
      ".inst 0x0ea16800  // bfcvtn v0.4h, v0.4s\n"
      ".inst 0x0ea16842  // bfcvtn v2.4h, v2.4s\n"
      ".inst 0x0ea16884  // bfcvtn v4.4h, v4.4s\n"
      ".inst 0x4ea16820  // bfcvtn2 v0.8h, v1.4s\n"
      ".inst 0x4ea16862  // bfcvtn2 v2.8h, v3.4s\n"
      ".inst 0x4ea168a4  // bfcvtn2 v4.8h, v5.4s\n"
      ".inst 0x6e47ec08  // bfmmla v8.4s, v0.8h, v7.8h\n"
      ".inst 0x6e46ec0c  // bfmmla v12.4s, v0.8h, v6.8h\n"
      ".inst 0x6e47ec50  // bfmmla v16.4s, v2.8h, v7.8h\n"
      ".inst 0x6e47ec98  // bfmmla v24.4s, v4.8h, v7.8h\n"
      "ldr q3, [x10, #0x20]\n"
      ".inst 0x6e46ec54  // bfmmla v20.4s, v2.8h, v6.8h\n"
      ".inst 0x6e46ec9c  // bfmmla v28.4s, v4.8h, v6.8h\n"
      "ldr q1, [x10, #0x30]\n"
      ".inst 0x6e43ec09  // bfmmla v9.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec51  // bfmmla v17.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec99  // bfmmla v25.4s, v4.8h, v3.8h\n"
      "ldr q3, [x10, #0x40]\n"
      ".inst 0x6e41ec0d  // bfmmla v13.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec55  // bfmmla v21.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9d  // bfmmla v29.4s, v4.8h, v1.8h\n"
      "ldr q1, [x10, #0x50]\n"
      ".inst 0x6e43ec0a  // bfmmla v10.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec52  // bfmmla v18.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec9a  // bfmmla v26.4s, v4.8h, v3.8h\n"
      "ldr q3, [x10, #0x60]\n"
      ".inst 0x6e41ec0e  // bfmmla v14.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec56  // bfmmla v22.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9e  // bfmmla v30.4s, v4.8h, v1.8h\n"
      "ldr q1, [x10, #0x70]\n"
      "add x10, x10, #0x80\n"
      ".inst 0x6e43ec0b  // bfmmla v11.4s, v0.8h, v3.8h\n"
      ".inst 0x6e43ec53  // bfmmla v19.4s, v2.8h, v3.8h\n"
      ".inst 0x6e43ec9b  // bfmmla v27.4s, v4.8h, v3.8h\n"
      ".inst 0x6e41ec0f  // bfmmla v15.4s, v0.8h, v1.8h\n"
      ".inst 0x6e41ec57  // bfmmla v23.4s, v2.8h, v1.8h\n"
      ".inst 0x6e41ec9f  // bfmmla v31.4s, v4.8h, v1.8h\n"
      "199:"  // Height 6: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 191b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "uzp1 v6.2d, v8.2d, v12.2d\n"
      "uzp2 v8.2d, v8.2d, v12.2d\n"
      "prfm pstl1keep, [x9, #0x0]\n"
      "uzp1 v12.2d, v9.2d, v13.2d\n"
      "uzp2 v9.2d, v9.2d, v13.2d\n"
      "uzp1 v13.2d, v10.2d, v14.2d\n"
      "uzp2 v10.2d, v10.2d, v14.2d\n"
      "add x26, x9, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "uzp1 v14.2d, v11.2d, v15.2d\n"
      "uzp2 v11.2d, v11.2d, v15.2d\n"
      "prfm pstl1keep, [x26, #0x0]\n"
      "add x23, x24, x20, LSL #2\n"
      "uzp1 v15.2d, v16.2d, v20.2d\n"
      "uzp2 v16.2d, v16.2d, v20.2d\n"
      "prfm pstl1keep, [x25, #0x0]\n"
      "add x22, x23, x20, LSL #2\n"
      "uzp1 v20.2d, v17.2d, v21.2d\n"
      "uzp2 v17.2d, v17.2d, v21.2d\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "uzp1 v21.2d, v18.2d, v22.2d\n"
      "uzp2 v18.2d, v18.2d, v22.2d\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "uzp1 v22.2d, v19.2d, v23.2d\n"
      "uzp2 v19.2d, v19.2d, v23.2d\n"
      "uzp1 v23.2d, v24.2d, v28.2d\n"
      "uzp2 v24.2d, v24.2d, v28.2d\n"
      "uzp1 v28.2d, v25.2d, v29.2d\n"
      "uzp2 v25.2d, v25.2d, v29.2d\n"
      "uzp1 v29.2d, v26.2d, v30.2d\n"
      "uzp2 v26.2d, v26.2d, v30.2d\n"
      "uzp1 v30.2d, v27.2d, v31.2d\n"
      "uzp2 v27.2d, v27.2d, v31.2d\n"
      "tbz %x[flags], #1, 200f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1r { v1.4s }, [x21]\n"
      "ld1r { v0.4s }, [x20]\n"
      "fmin v6.4s, v6.4s, v1.4s\n"
      "fmin v12.4s, v12.4s, v1.4s\n"
      "fmin v13.4s, v13.4s, v1.4s\n"
      "fmin v14.4s, v14.4s, v1.4s\n"
      "fmin v8.4s, v8.4s, v1.4s\n"
      "fmin v9.4s, v9.4s, v1.4s\n"
      "fmin v10.4s, v10.4s, v1.4s\n"
      "fmin v11.4s, v11.4s, v1.4s\n"
      "fmin v15.4s, v15.4s, v1.4s\n"
      "fmin v20.4s, v20.4s, v1.4s\n"
      "fmin v21.4s, v21.4s, v1.4s\n"
      "fmin v22.4s, v22.4s, v1.4s\n"
      "fmin v16.4s, v16.4s, v1.4s\n"
      "fmin v17.4s, v17.4s, v1.4s\n"
      "fmin v18.4s, v18.4s, v1.4s\n"
      "fmin v19.4s, v19.4s, v1.4s\n"
      "fmin v23.4s, v23.4s, v1.4s\n"
      "fmin v28.4s, v28.4s, v1.4s\n"
      "fmin v29.4s, v29.4s, v1.4s\n"
      "fmin v30.4s, v30.4s, v1.4s\n"
      "fmin v24.4s, v24.4s, v1.4s\n"
      "fmin v25.4s, v25.4s, v1.4s\n"
      "fmin v26.4s, v26.4s, v1.4s\n"
      "fmin v27.4s, v27.4s, v1.4s\n"
      "fmax v6.4s, v6.4s, v0.4s\n"
      "fmax v12.4s, v12.4s, v0.4s\n"
      "fmax v13.4s, v13.4s, v0.4s\n"
      "fmax v14.4s, v14.4s, v0.4s\n"
      "fmax v8.4s, v8.4s, v0.4s\n"
      "fmax v9.4s, v9.4s, v0.4s\n"
      "fmax v10.4s, v10.4s, v0.4s\n"
      "fmax v11.4s, v11.4s, v0.4s\n"
      "fmax v15.4s, v15.4s, v0.4s\n"
      "fmax v20.4s, v20.4s, v0.4s\n"
      "fmax v21.4s, v21.4s, v0.4s\n"
      "fmax v22.4s, v22.4s, v0.4s\n"
      "fmax v16.4s, v16.4s, v0.4s\n"
      "fmax v17.4s, v17.4s, v0.4s\n"
      "fmax v18.4s, v18.4s, v0.4s\n"
      "fmax v19.4s, v19.4s, v0.4s\n"
      "fmax v23.4s, v23.4s, v0.4s\n"
      "fmax v28.4s, v28.4s, v0.4s\n"
      "fmax v29.4s, v29.4s, v0.4s\n"
      "fmax v30.4s, v30.4s, v0.4s\n"
      "fmax v24.4s, v24.4s, v0.4s\n"
      "fmax v25.4s, v25.4s, v0.4s\n"
      "fmax v26.4s, v26.4s, v0.4s\n"
      "fmax v27.4s, v27.4s, v0.4s\n"
      "200:"  // Height 6: No activation
      "cmp x11, #0x10\n"
      "bge 209f\n"
      "tbz x11, #3, 204f\n"
      "st1 { v6.4s }, [x9], #0x10\n"
      "st1 { v12.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x26], #0x10\n"
      "st1 { v9.4s }, [x26], #0x10\n"
      "st1 { v15.4s }, [x25], #0x10\n"
      "st1 { v20.4s }, [x25], #0x10\n"
      "st1 { v16.4s }, [x24], #0x10\n"
      "st1 { v17.4s }, [x24], #0x10\n"
      "st1 { v23.4s }, [x23], #0x10\n"
      "st1 { v28.4s }, [x23], #0x10\n"
      "st1 { v24.4s }, [x22], #0x10\n"
      "st1 { v25.4s }, [x22], #0x10\n"
      "tbz x11, #2, 202f\n"
      "st1 { v13.4s }, [x9], #0x10\n"
      "st1 { v10.4s }, [x26], #0x10\n"
      "st1 { v21.4s }, [x25], #0x10\n"
      "st1 { v18.4s }, [x24], #0x10\n"
      "st1 { v29.4s }, [x23], #0x10\n"
      "st1 { v26.4s }, [x22], #0x10\n"
      "tbz x11, #1, 201f\n"
      "str d14, [x9], #0x8\n"
      "str d11, [x26], #0x8\n"
      "str d22, [x25], #0x8\n"
      "str d19, [x24], #0x8\n"
      "str d30, [x23], #0x8\n"
      "str d27, [x22], #0x8\n"
      "tbz x11, #0, 208f\n"
      "st1 { v14.s }[2], [x9]\n"
      "st1 { v11.s }[2], [x26]\n"
      "st1 { v22.s }[2], [x25]\n"
      "st1 { v19.s }[2], [x24]\n"
      "st1 { v30.s }[2], [x23]\n"
      "st1 { v27.s }[2], [x22]\n"
      "b 208f\n"
      "201:"  // Height 6: Partial direct writeback: partial_1_12
      "tbz x11, #0, 208f\n"
      "str s14, [x9, #0x0]\n"
      "str s11, [x26, #0x0]\n"
      "str s22, [x25, #0x0]\n"
      "str s19, [x24, #0x0]\n"
      "str s30, [x23, #0x0]\n"
      "str s27, [x22, #0x0]\n"
      "b 208f\n"
      "202:"  // Height 6: Partial direct writeback: partial_2_8
      "tbz x11, #1, 203f\n"
      "str d13, [x9], #0x8\n"
      "str d10, [x26], #0x8\n"
      "str d21, [x25], #0x8\n"
      "str d18, [x24], #0x8\n"
      "str d29, [x23], #0x8\n"
      "str d26, [x22], #0x8\n"
      "tbz x11, #0, 208f\n"
      "st1 { v13.s }[2], [x9]\n"
      "st1 { v10.s }[2], [x26]\n"
      "st1 { v21.s }[2], [x25]\n"
      "st1 { v18.s }[2], [x24]\n"
      "st1 { v29.s }[2], [x23]\n"
      "st1 { v26.s }[2], [x22]\n"
      "b 208f\n"
      "203:"  // Height 6: Partial direct writeback: partial_1_8
      "tbz x11, #0, 208f\n"
      "str s13, [x9, #0x0]\n"
      "str s10, [x26, #0x0]\n"
      "str s21, [x25, #0x0]\n"
      "str s18, [x24, #0x0]\n"
      "str s29, [x23, #0x0]\n"
      "str s26, [x22, #0x0]\n"
      "b 208f\n"
      "204:"  // Height 6: Partial direct writeback: partial_4_0
      "tbz x11, #2, 206f\n"
      "st1 { v6.4s }, [x9], #0x10\n"
      "st1 { v8.4s }, [x26], #0x10\n"
      "st1 { v15.4s }, [x25], #0x10\n"
      "st1 { v16.4s }, [x24], #0x10\n"
      "st1 { v23.4s }, [x23], #0x10\n"
      "st1 { v24.4s }, [x22], #0x10\n"
      "tbz x11, #1, 205f\n"
      "str d12, [x9], #0x8\n"
      "str d9, [x26], #0x8\n"
      "str d20, [x25], #0x8\n"
      "str d17, [x24], #0x8\n"
      "str d28, [x23], #0x8\n"
      "str d25, [x22], #0x8\n"
      "tbz x11, #0, 208f\n"
      "st1 { v12.s }[2], [x9]\n"
      "st1 { v9.s }[2], [x26]\n"
      "st1 { v20.s }[2], [x25]\n"
      "st1 { v17.s }[2], [x24]\n"
      "st1 { v28.s }[2], [x23]\n"
      "st1 { v25.s }[2], [x22]\n"
      "b 208f\n"
      "205:"  // Height 6: Partial direct writeback: partial_1_4
      "tbz x11, #0, 208f\n"
      "str s12, [x9, #0x0]\n"
      "str s9, [x26, #0x0]\n"
      "str s20, [x25, #0x0]\n"
      "str s17, [x24, #0x0]\n"
      "str s28, [x23, #0x0]\n"
      "str s25, [x22, #0x0]\n"
      "b 208f\n"
      "206:"  // Height 6: Partial direct writeback: partial_2_0
      "tbz x11, #1, 207f\n"
      "str d6, [x9], #0x8\n"
      "str d8, [x26], #0x8\n"
      "str d15, [x25], #0x8\n"
      "str d16, [x24], #0x8\n"
      "str d23, [x23], #0x8\n"
      "str d24, [x22], #0x8\n"
      "tbz x11, #0, 208f\n"
      "st1 { v6.s }[2], [x9]\n"
      "st1 { v8.s }[2], [x26]\n"
      "st1 { v15.s }[2], [x25]\n"
      "st1 { v16.s }[2], [x24]\n"
      "st1 { v23.s }[2], [x23]\n"
      "st1 { v24.s }[2], [x22]\n"
      "b 208f\n"
      "207:"  // Height 6: Partial direct writeback: partial_1_0
      "str s6, [x9, #0x0]\n"
      "str s8, [x26, #0x0]\n"
      "str s15, [x25, #0x0]\n"
      "str s16, [x24, #0x0]\n"
      "str s23, [x23, #0x0]\n"
      "str s24, [x22, #0x0]\n"
      "208:"  // Height 6: Partial direct writeback: Done
      "b 210f\n"
      "209:"  // Height 6: Full writeback
      "str q6, [x9, #0x0]\n"
      "str q12, [x9, #0x10]\n"
      "str q13, [x9, #0x20]\n"
      "str q14, [x9, #0x30]\n"
      "add x9, x9, #0x40\n"
      "str q8, [x26, #0x0]\n"
      "str q9, [x26, #0x10]\n"
      "str q10, [x26, #0x20]\n"
      "str q11, [x26, #0x30]\n"
      "str q15, [x25, #0x0]\n"
      "str q20, [x25, #0x10]\n"
      "str q21, [x25, #0x20]\n"
      "str q22, [x25, #0x30]\n"
      "str q16, [x24, #0x0]\n"
      "str q17, [x24, #0x10]\n"
      "str q18, [x24, #0x20]\n"
      "str q19, [x24, #0x30]\n"
      "str q23, [x23, #0x0]\n"
      "str q28, [x23, #0x10]\n"
      "str q29, [x23, #0x20]\n"
      "str q30, [x23, #0x30]\n"
      "str q24, [x22, #0x0]\n"
      "str q25, [x22, #0x10]\n"
      "str q26, [x22, #0x20]\n"
      "str q27, [x22, #0x30]\n"
      "210:"  // Height 6: Writeback done
      "subs x11, x11, #0x10\n"
      "bgt 177b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 212f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 211f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "211:"  // Update direct input
      "mov x20, #0x18\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "212:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x11", "x12", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
