
/*
 Copyright 2013, Kovid Goyal <kovid at kovidgoyal.net>
 Released under the GPLv3 License
 */

(function() {
  var FontStats, fl_pat, font_dict, font_usage, log, process_font_face_rule, process_sheet, process_sheet_for_pseudo, process_style_rule;

  if (typeof window !== "undefined" && window !== null ? window.calibre_utils : void 0) {
    log = window.calibre_utils.log;
  }

  font_dict = function(style, computed) {
    var f, family, fams, i, len;
    if (computed == null) {
      computed = false;
    }
    if (computed) {
      fams = [];
      family = style.getPropertyCSSValue('font-family');
      if (family.cssValueType === CSSValue.CSS_PRIMITIVE_VALUE) {
        fams.push(family.getStringValue());
      } else {
        for (i = 0, len = family.length; i < len; i++) {
          f = family[i];
          fams.push(f.getStringValue());
        }
      }
    } else {
      fams = style.getPropertyValue('font-family');
    }
    return {
      'font-family': fams,
      'font-weight': style.getPropertyValue('font-weight'),
      'font-style': style.getPropertyValue('font-style'),
      'font-stretch': style.getPropertyValue('font-stretch'),
      'text-transform': style.getPropertyValue('text-transform'),
      'font-variant': style.getPropertyValue('font-variant')
    };
  };

  font_usage = function(node) {
    var ans, child, i, len, ref, style, text;
    style = window.getComputedStyle(node, null);
    ans = font_dict(style, true);
    text = [];
    ref = node.childNodes;
    for (i = 0, len = ref.length; i < len; i++) {
      child = ref[i];
      if (child.nodeType === Node.TEXT_NODE) {
        text.push(child.nodeValue);
      }
    }
    ans['text'] = text;
    return ans;
  };

  process_sheet = function(sheet, font_faces) {
    var i, len, ref, results, rule;
    ref = sheet.cssRules;
    results = [];
    for (i = 0, len = ref.length; i < len; i++) {
      rule = ref[i];
      if (rule.type === rule.FONT_FACE_RULE) {
        results.push(process_font_face_rule(rule, font_faces));
      } else if (rule.type === rule.IMPORT_RULE && rule.styleSheet) {
        results.push(process_sheet(rule.styleSheet, font_faces));
      } else {
        results.push(void 0);
      }
    }
    return results;
  };

  process_font_face_rule = function(rule, font_faces) {
    var fd;
    fd = font_dict(rule.style);
    fd['src'] = rule.style.getPropertyValue('src');
    return font_faces.push(fd);
  };

  fl_pat = /:{1,2}(first-letter|first-line)/i;

  process_sheet_for_pseudo = function(sheet, rules) {
    var ff, i, len, m, pseudo, ref, results, rule, st;
    ref = sheet.cssRules;
    results = [];
    for (i = 0, len = ref.length; i < len; i++) {
      rule = ref[i];
      if (rule.type === rule.STYLE_RULE) {
        st = rule.selectorText;
        m = fl_pat.exec(st);
        if (m) {
          pseudo = m[1].toLowerCase();
          ff = rule.style.getPropertyValue('font-family');
          if (ff) {
            results.push(process_style_rule(st, rule.style, rules, pseudo));
          } else {
            results.push(void 0);
          }
        } else {
          results.push(void 0);
        }
      } else if (rule.type === rule.IMPORT_RULE && rule.styleSheet) {
        results.push(process_sheet_for_pseudo(rule.styleSheet, rules));
      } else {
        results.push(void 0);
      }
    }
    return results;
  };

  process_style_rule = function(selector_text, style, rules, pseudo) {
    var element, fd, i, len, ref, results, text;
    selector_text = selector_text.replace(fl_pat, '');
    fd = font_dict(style);
    ref = document.querySelectorAll(selector_text);
    results = [];
    for (i = 0, len = ref.length; i < len; i++) {
      element = ref[i];
      text = element.innerText;
      if (text) {
        results.push(rules.push([fd, text, pseudo]));
      } else {
        results.push(void 0);
      }
    }
    return results;
  };

  FontStats = (function() {
    function FontStats() {
      if (!this instanceof arguments.callee) {
        throw new Error('FontStats constructor called as function');
      }
    }

    FontStats.prototype.get_font_face_rules = function() {
      var font_faces, i, len, ref, sheet;
      font_faces = [];
      ref = document.styleSheets;
      for (i = 0, len = ref.length; i < len; i++) {
        sheet = ref[i];
        process_sheet(sheet, font_faces);
      }
      return py_bridge.value = font_faces;
    };

    FontStats.prototype.get_font_usage = function() {
      var ans, busage, i, len, node, ref, usage;
      ans = [];
      busage = font_usage(document.body);
      if (busage !== null) {
        ans.push(busage);
      }
      ref = document.body.getElementsByTagName('*');
      for (i = 0, len = ref.length; i < len; i++) {
        node = ref[i];
        usage = font_usage(node);
        if (usage !== null) {
          ans.push(usage);
        }
      }
      return py_bridge.value = ans;
    };

    FontStats.prototype.get_pseudo_element_font_usage = function() {
      var ans, i, len, ref, sheet;
      ans = [];
      ref = document.styleSheets;
      for (i = 0, len = ref.length; i < len; i++) {
        sheet = ref[i];
        process_sheet_for_pseudo(sheet, ans);
      }
      return py_bridge.value = ans;
    };

    FontStats.prototype.get_font_families = function() {
      var ans, family, i, j, len, len1, node, ref, rule, rules, style;
      ans = {};
      ref = document.getElementsByTagName('*');
      for (i = 0, len = ref.length; i < len; i++) {
        node = ref[i];
        rules = document.defaultView.getMatchedCSSRules(node, '');
        if (rules) {
          for (j = 0, len1 = rules.length; j < len1; j++) {
            rule = rules[j];
            style = rule.style;
            family = style.getPropertyValue('font-family');
            if (family) {
              ans[family] = true;
            }
          }
        }
        if (node.getAttribute('style')) {
          family = node.style.getPropertyValue('font-family');
          if (family) {
            ans[family] = true;
          }
        }
      }
      return py_bridge.value = ans;
    };

    return FontStats;

  })();

  if (typeof window !== "undefined" && window !== null) {
    window.font_stats = new FontStats();
  }

}).call(this);
