package org.openscience.cdk.forcefield.mmff;

import org.junit.Test;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.silent.AtomContainer;

import static org.openscience.cdk.interfaces.IBond.Order.DOUBLE;
import static org.openscience.cdk.interfaces.IBond.Order.SINGLE;
import static org.openscience.cdk.interfaces.IBond.Order.TRIPLE;

/**
 * This test class is generated from MMFF94hypervalent.mol2 and  MMFF94opti.log available at
 * http://server.ccl.net/cca/data/MMFF94/.
 */
public abstract class AbstractMmffAtomTypeValidationSuiteTest {

    /**
     * Generated test for AGLYSL01
     *
     * @cdk.inchi InChI=1S/C2H5NO2/c3-1-2(4)5/h1,3H2,(H,4,5)
     */
    @Test
    public void testAGLYSL01() {
        IAtomContainer container = new AtomContainer(10, 9, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 9, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(6, 8, SINGLE);

        String[] expected = {"CR", "COO", "HNR", "HNR", "HC", "HC", "HOCO", "NR", "OC=O", "O=CO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for AMHTAR01
     *
     * @cdk.inchi InChI=1S/C4H6O6/c5-1(3(7)8)2(6)4(9)10/h1-2,5-6H,(H,7,8)(H,9,10)/p-1/t1-,2-/m1/s1
     */
    @Test
    public void testAMHTAR01() {
        IAtomContainer container = new AtomContainer(15, 14, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 9, DOUBLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 11, SINGLE);

        String[] expected = {"OC=O", "O=CO", "OR", "OR", "O2CM", "O2CM", "COO", "CR", "CR", "CO2M", "HC", "HC", "HOR",
                "HOR", "HOCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for AMPTRB10
     *
     * @cdk.inchi InChI=1S/C8H10N6/c1-3-4(2)12-7-5(11-3)6(9)13-8(10)14-7/h1-2H3,(H4,9,10,12,13,14)/p+1
     */
    @Test
    public void testAMPTRB10() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 14, SINGLE);
        container.addBond(0, 13, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 6, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 12, DOUBLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, SINGLE);

        String[] expected = {"NPD+", "CB", "NC=N", "NPYD", "CB", "NC=C", "CB", "NPYD", "CB", "CR", "CB", "CR", "NPYD",
                "CB", "HPD+", "HNCN", "HNCN", "HNCC", "HNCC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for ARGIND11
     *
     * @cdk.inchi InChI=1S/C6H14N4O2/c7-4(5(11)12)2-1-3-10-6(8)9/h4H,1-3,7H2,(H,11,12)(H4,8,9,10)/t4-/m0/s1
     */
    @Test
    public void testARGIND11() {
        IAtomContainer container = new AtomContainer(26, 25, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addBond(0, 25, SINGLE);
        container.addBond(0, 24, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 23, SINGLE);
        container.addBond(5, 22, DOUBLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(15, 22, SINGLE);
        container.addBond(16, 21, SINGLE);
        container.addBond(17, 21, SINGLE);
        container.addBond(18, 23, SINGLE);
        container.addBond(19, 23, SINGLE);

        String[] expected = {"CO2M", "CR", "CR", "CR", "CR", "CGD+", "HNR", "HC", "HNR", "HC", "HC", "HC", "HC", "HC",
                "HC", "HGD+", "HGD+", "HGD+", "HGD+", "HGD+", "NR", "NGD+", "NGD+", "NGD+", "O2CM", "O2CM"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BAOXLM01
     *
     * @cdk.inchi InChI=1S/C2H2O4/c3-1(4)2(5)6/h(H,3,4)(H,5,6)/p-2
     */
    @Test
    public void testBAOXLM01() {
        IAtomContainer container = new AtomContainer(6, 5, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addBond(0, 2, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 5, SINGLE);

        String[] expected = {"O2CM", "O2CM", "CO2M", "CO2M", "O2CM", "O2CM"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BBSPRT10
     *
     * @cdk.inchi InChI=1S/C9H8BrNO3S2/c10-6-1-3-7(4-2-6)16(13,14)11-8-5-15-9(8)12/h1-4,8,11H,5H2/t8-/m0/s1
     */
    @Test
    public void testBBSPRT10() {
        IAtomContainer container = new AtomContainer(24, 25, 0, 0);
        container.addAtom(newAtm("Br", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 13, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(2, 4, DOUBLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 22, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(15, 23, SINGLE);

        String[] expected = {"BR", "S", "SO2N", "O=C", "O2S", "O2S", "NSO2", "C=OS", "CR4R", "CR4R", "CB", "CB", "CB",
                "CB", "CB", "CB", "HNSO", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BEVJER10
     *
     * @cdk.inchi InChI=1S/C16N8/c17-1-9(2-18)13-14(10(3-19)4-20)16(12(7-23)8-24)15(13)11(5-21)6-22
     */
    @Test
    public void testBEVJER10() {
        IAtomContainer container = new AtomContainer(24, 24, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(1, 7, DOUBLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 4, TRIPLE);
        container.addBond(5, 6, TRIPLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 9, TRIPLE);
        container.addBond(10, 11, TRIPLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 14, DOUBLE);
        container.addBond(13, 15, DOUBLE);
        container.addBond(14, 16, SINGLE);
        container.addBond(14, 17, SINGLE);
        container.addBond(15, 18, SINGLE);
        container.addBond(15, 19, SINGLE);
        container.addBond(16, 20, TRIPLE);
        container.addBond(17, 21, TRIPLE);
        container.addBond(18, 22, TRIPLE);
        container.addBond(19, 23, TRIPLE);

        String[] expected = {"CE4R", "CE4R", "C=C", "CSP", "NSP", "CSP", "NSP", "C=C", "CSP", "NSP", "CSP", "NSP",
                "CE4R", "CE4R", "C=C", "C=C", "CSP", "CSP", "CSP", "CSP", "NSP", "NSP", "NSP", "NSP"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BEWCUB
     *
     * @cdk.inchi InChI=1S/C23H23N3O9S/c1-13(11-34-15(3)28)20-18-10-19(36(33)9-8-24-14(2)27)21(25(18)22(20)29)23(30)35-12-16-4-6-17(7-5-16)26(31)32/h4-9,18H,10-12H2,1-3H3,(H,24,27)/b9-8+,20-13+/t18-,36-/m1/s1
     */
    @Test
    public void testBEWCUB() {
        IAtomContainer container = new AtomContainer(59, 61, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 23, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 36, SINGLE);
        container.addBond(3, 37, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 38, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, DOUBLE);
        container.addBond(6, 22, DOUBLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 15, DOUBLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 39, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 40, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 41, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(14, 42, SINGLE);
        container.addBond(14, 43, SINGLE);
        container.addBond(14, 44, SINGLE);
        container.addBond(16, 45, SINGLE);
        container.addBond(16, 46, SINGLE);
        container.addBond(16, 47, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(17, 48, SINGLE);
        container.addBond(17, 49, SINGLE);
        container.addBond(18, 19, SINGLE);
        container.addBond(19, 20, DOUBLE);
        container.addBond(19, 21, SINGLE);
        container.addBond(21, 50, SINGLE);
        container.addBond(21, 51, SINGLE);
        container.addBond(21, 52, SINGLE);
        container.addBond(23, 24, DOUBLE);
        container.addBond(23, 25, SINGLE);
        container.addBond(25, 26, SINGLE);
        container.addBond(26, 27, SINGLE);
        container.addBond(26, 53, SINGLE);
        container.addBond(26, 54, SINGLE);
        container.addBond(27, 28, DOUBLE);
        container.addBond(27, 32, SINGLE);
        container.addBond(28, 29, SINGLE);
        container.addBond(28, 57, SINGLE);
        container.addBond(29, 30, DOUBLE);
        container.addBond(29, 58, SINGLE);
        container.addBond(30, 31, SINGLE);
        container.addBond(30, 33, SINGLE);
        container.addBond(31, 32, DOUBLE);
        container.addBond(31, 55, SINGLE);
        container.addBond(32, 56, SINGLE);
        container.addBond(33, 34, SINGLE);
        container.addBond(33, 35, DOUBLE);

        String[] expected = {"NC=O", "C=C", "C=C", "CR", "CR4R", "CE4R", "C=ON", "C=C", "S=O", "C=C", "C=C", "NC=O",
                "C=ON", "O=CN", "CR", "O=S", "CR", "CR", "OC=O", "COO", "O=CO", "CR", "O=CN", "COO", "O=CO", "OC=O",
                "CR", "CB", "CB", "CB", "CB", "CB", "CB", "NO2", "O2N", "O2N", "HC", "HC", "HC", "HC", "HC", "HNCO",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BEWKUJ04
     *
     * @cdk.inchi InChI=1S/C11H10N3O2S/c12-9-4-6-10(7-5-9)17(15,16)14-11-3-1-2-8-13-11/h1-8H,12H2/q-1/p+1
     */
    @Test
    public void testBEWKUJ04() {
        IAtomContainer container = new AtomContainer(28, 29, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", -1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(4, 21, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 27, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 12, DOUBLE);
        container.addBond(6, 11, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 25, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(16, 24, SINGLE);

        String[] expected = {"SO2N", "O2S", "O2S", "NM", "NC=C", "NPD+", "CB", "CB", "CB", "CB", "CB", "CB", "CB",
                "CB", "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HNCC", "HNCC", "HC", "HC", "HC", "HC", "HPD+"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BIHKEI01
     *
     * @cdk.inchi InChI=1S/C8H9NO3/c1-4-7(10)6-5(2-9-4)3-12-8(6)11/h2,8,10-11H,3H2,1H3/t8-/m1/s1
     */
    @Test
    public void testBIHKEI01() {
        IAtomContainer container = new AtomContainer(21, 22, 0, 0);
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 5, DOUBLE);
        container.addBond(4, 11, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 11, SINGLE);

        String[] expected = {"OM2", "OR", "OR", "NPD+", "CB", "CB", "CR", "CB", "CB", "CR", "CR", "CB", "HOR", "HPD+",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BIPDEJ02
     *
     * @cdk.inchi InChI=1S/C8H9FN2O3/c9-5-4-11(6-2-1-3-14-6)8(13)10-7(5)12/h4,6H,1-3H2,(H,10,12,13)/t6-/m0/s1
     */
    @Test
    public void testBIPDEJ02() {
        IAtomContainer container = new AtomContainer(23, 24, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);

        String[] expected = {"NC=O", "CONN", "O=CN", "NC=O", "C=ON", "O=CN", "C=C", "F", "C=C", "CR", "CR", "CR", "CR",
                "OR", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BIPJUF10
     *
     * @cdk.inchi InChI=1S/C7H4N2O6/c10-7(11)5-2-1-4(8(12)13)3-6(5)9(14)15/h1-3H,(H,10,11)
     */
    @Test
    public void testBIPJUF10() {
        IAtomContainer container = new AtomContainer(19, 19, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 11, DOUBLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 14, DOUBLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "COO", "OC=O", "O=CO", "NO2", "O2N", "O2N", "NO2",
                "O2N", "O2N", "HOCO", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BIPYCL01
     *
     * @cdk.inchi InChI=1S/C10H8N2/c1-3-7-11-9(5-1)10-6-2-4-8-12-10/h1-8H/p+2
     */
    @Test
    public void testBIPYCL01() {
        IAtomContainer container = new AtomContainer(22, 23, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 12, SINGLE);
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 11, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 21, SINGLE);

        String[] expected = {"NPD+", "CB", "CB", "CB", "CB", "CB", "NPD+", "CB", "CB", "CB", "CB", "CB", "HPD+", "HC",
                "HC", "HC", "HC", "HPD+", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BITNAT10
     *
     * @cdk.inchi InChI=1S/C12H11N3S/c1-8-7-9(2)15(14-8)12-13-10-5-3-4-6-11(10)16-12/h3-7H,1-2H3
     */
    @Test
    public void testBITNAT10() {
        IAtomContainer container = new AtomContainer(27, 29, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 17, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(14, 22, SINGLE);
        container.addBond(14, 21, SINGLE);
        container.addBond(15, 26, SINGLE);
        container.addBond(15, 25, SINGLE);
        container.addBond(15, 24, SINGLE);

        String[] expected = {"C5A", "CB", "CB", "CB", "CB", "C5B", "N5B", "C5A", "STHI", "NPYL", "C5A", "C5B", "C5B",
                "N5A", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BIYBIU10
     *
     * @cdk.inchi InChI=1S/C12H9NS/c1-3-7-11-9(5-1)13-10-6-2-4-8-12(10)14-11/h1-8,13H
     */
    @Test
    public void testBIYBIU10() {
        IAtomContainer container = new AtomContainer(23, 25, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(12, 11, SINGLE);
        container.addBond(12, 9, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 10, SINGLE);
        container.addBond(13, 6, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(10, 20, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "NC=C", "S", "CB", "CB", "HC",
                "HC", "HC", "HC", "HC", "HC", "HNCC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BODKOU
     *
     * @cdk.inchi InChI=1S/C6H8N4O2S.C4H8N2S2/c1-12-10-4(5(7)11)3-2-13-6(8)9-3;1-3-5-6-4(7-2)8-3/h2H,1H3,(H2,7,11)(H2,8,9);3,5H,1-2H3/b10-4-;/t;3-/m.1/s1
     */
    @Test
    public void testBODKOU() {
        IAtomContainer container = new AtomContainer(37, 37, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(21, 28, SINGLE);
        container.addBond(21, 27, SINGLE);
        container.addBond(21, 22, SINGLE);
        container.addBond(22, 24, SINGLE);
        container.addBond(22, 23, DOUBLE);
        container.addBond(24, 35, SINGLE);
        container.addBond(24, 25, DOUBLE);
        container.addBond(25, 26, SINGLE);
        container.addBond(26, 34, SINGLE);
        container.addBond(34, 31, SINGLE);
        container.addBond(34, 30, SINGLE);
        container.addBond(34, 29, SINGLE);
        container.addBond(35, 19, DOUBLE);
        container.addBond(35, 16, SINGLE);
        container.addBond(16, 17, DOUBLE);
        container.addBond(17, 20, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(18, 19, SINGLE);
        container.addBond(19, 36, SINGLE);
        container.addBond(20, 33, SINGLE);
        container.addBond(20, 32, SINGLE);

        String[] expected = {"CR", "S", "C=N", "S", "CR", "NR", "N=C", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HNR", "N5B", "C5A", "STHI", "C5A", "NC=N", "NC=O", "C=ON", "O=CN", "C=N", "N=C", "OR", "HNCO", "HNCO",
                "HC", "HC", "HC", "HNCN", "HNCN", "CR", "C5B", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BSALAP01
     *
     * @cdk.inchi InChI=1S/C12H9BrN2O/c13-10-4-5-11(16)9(7-10)8-15-12-3-1-2-6-14-12/h1-8,16H/b15-8+
     */
    @Test
    public void testBSALAP01() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("Br", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(2, 10, DOUBLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 11, DOUBLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(15, 24, SINGLE);

        String[] expected = {"BR", "OC=C", "N=C", "NPYD", "CB", "CB", "CB", "CB", "CB", "CB", "C=N", "CB", "CB", "CB",
                "CB", "CB", "HOCC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BUPSLB10
     *
     * @cdk.inchi InChI=1S/C10H24P2S2/c1-9(2,3)11(7,13)12(8,14)10(4,5)6/h1-8H3/t11-,12+
     */
    @Test
    public void testBUPSLB10() {
        IAtomContainer container = new AtomContainer(38, 37, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 19, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(19, 20, DOUBLE);
        container.addBond(19, 21, SINGLE);
        container.addBond(19, 22, SINGLE);
        container.addBond(21, 23, SINGLE);
        container.addBond(21, 24, SINGLE);
        container.addBond(21, 25, SINGLE);
        container.addBond(22, 26, SINGLE);
        container.addBond(22, 27, SINGLE);
        container.addBond(22, 28, SINGLE);
        container.addBond(23, 29, SINGLE);
        container.addBond(23, 30, SINGLE);
        container.addBond(23, 31, SINGLE);
        container.addBond(24, 32, SINGLE);
        container.addBond(24, 33, SINGLE);
        container.addBond(24, 34, SINGLE);
        container.addBond(25, 35, SINGLE);
        container.addBond(25, 36, SINGLE);
        container.addBond(25, 37, SINGLE);

        String[] expected = {"PTET", "S-P", "CR", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "PTET", "S-P", "CR", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BUPSLD10
     *
     * @cdk.inchi InChI=1S/C10H24P2S3/c1-9(2,3)11(7,13)15-12(8,14)10(4,5)6/h1-8H3/t11-,12-/m0/s1
     */
    @Test
    public void testBUPSLD10() {
        IAtomContainer container = new AtomContainer(39, 38, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(11, 27, SINGLE);
        container.addBond(11, 28, SINGLE);
        container.addBond(11, 29, SINGLE);
        container.addBond(12, 30, SINGLE);
        container.addBond(12, 31, SINGLE);
        container.addBond(12, 32, SINGLE);
        container.addBond(13, 33, SINGLE);
        container.addBond(13, 34, SINGLE);
        container.addBond(13, 35, SINGLE);
        container.addBond(14, 36, SINGLE);
        container.addBond(14, 37, SINGLE);
        container.addBond(14, 38, SINGLE);

        String[] expected = {"PTET", "PTET", "S-P", "S-P", "S", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR",
                "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BUYTIY10
     *
     * @cdk.inchi InChI=1S/C11H12N4O4/c12-3-6-4-15(14-2-1-13-10(6)14)11-9(18)8(17)7(5-16)19-11/h1-2,4,7-9,11,16-18H,5H2/t7-,8-,9-,11-/m1/s1
     */
    @Test
    public void testBUYTIY10() {
        IAtomContainer container = new AtomContainer(31, 33, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 7, DOUBLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(8, 9, TRIPLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 18, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(16, 28, SINGLE);
        container.addBond(16, 29, SINGLE);
        container.addBond(17, 30, SINGLE);

        String[] expected = {"N5B", "C5B", "C5A", "NPYL", "NC=C", "C=C", "C=C", "C5A", "CSP", "NSP", "CR", "CR", "OR",
                "CR", "OR", "CR", "CR", "OR", "OR", "HC", "HC", "HC", "HC", "HC", "HOR", "HC", "HOR", "HC", "HC", "HC",
                "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BUYTOE10
     *
     * @cdk.inchi InChI=1S/C11H12N4O4/c12-3-6-4-13-15-2-1-14(10(6)15)11-9(18)8(17)7(5-16)19-11/h1-2,4,7-9,11,16-18H,5H2/t7-,8-,9-,11-/m1/s1
     */
    @Test
    public void testBUYTOE10() {
        IAtomContainer container = new AtomContainer(31, 33, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(8, 9, TRIPLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(15, 18, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 29, SINGLE);
        container.addBond(16, 28, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(17, 30, SINGLE);

        String[] expected = {"NC=C", "C=C", "C=C", "NPYL", "N5A", "C5B", "C5B", "C5A", "CSP", "NSP", "CR", "CR", "OR",
                "CR", "OR", "CR", "CR", "OR", "OR", "HC", "HC", "HC", "HC", "HC", "HOR", "HC", "HOR", "HC", "HC", "HC",
                "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BUYXEY10
     *
     * @cdk.inchi InChI=1S/C11H20N2O4S/c1-7(2)9(11(16)17)13-10(15)8(12-6-14)4-5-18-3/h6-9H,4-5H2,1-3H3,(H,12,14)(H,13,15)(H,16,17)/t8-,9-/m0/s1
     */
    @Test
    public void testBUYXEY10() {
        IAtomContainer container = new AtomContainer(38, 37, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 24, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 25, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 28, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 29, SINGLE);
        container.addBond(7, 32, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 30, SINGLE);
        container.addBond(9, 34, SINGLE);
        container.addBond(9, 35, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 27, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 26, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 33, SINGLE);
        container.addBond(14, 31, SINGLE);
        container.addBond(14, 36, SINGLE);
        container.addBond(14, 37, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 17, DOUBLE);
        container.addBond(16, 21, SINGLE);

        String[] expected = {"O=CN", "C=ON", "NC=O", "CR", "C=ON", "O=CN", "CR", "CR", "S", "CR", "NC=O", "CR", "CR",
                "CR", "CR", "COO", "OC=O", "O=CO", "HC", "HC", "HC", "HOCO", "HC", "HC", "HNCO", "HC", "HC", "HNCO",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for BYITOT02
     *
     * @cdk.inchi InChI=1S/C11H12N2O2S/c1-13-7-8-15-16-11(13)12-10(14)9-5-3-2-4-6-9/h2-6H,7-8H2,1H3/b12-11-
     */
    @Test
    public void testBYITOT02() {
        IAtomContainer container = new AtomContainer(28, 29, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(15, 25, SINGLE);
        container.addBond(15, 26, SINGLE);
        container.addBond(15, 27, SINGLE);

        String[] expected = {"-OS", "S", "C=N", "NC=N", "CR", "CR", "N=C", "C=ON", "O=CN", "CB", "CB", "CB", "CB",
                "CB", "CB", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CABWEH10
     *
     * @cdk.inchi InChI=1S/C10H10O2S/c11-13(12)7-6-10(8-13)9-4-2-1-3-5-9/h1-6H,7-8H2
     */
    @Test
    public void testCABWEH10() {
        IAtomContainer container = new AtomContainer(23, 24, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(1, 22, SINGLE);
        container.addBond(1, 12, DOUBLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 21, SINGLE);

        String[] expected = {"SO2", "CB", "O2S", "O2S", "CR", "C=C", "C=C", "CR", "CB", "CB", "CB", "CB", "CB", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CAFORM07
     *
     * @cdk.inchi InChI=1S/CH2O2/c2-1-3/h1H,(H,2,3)/p-1
     */
    @Test
    public void testCAFORM07() {
        IAtomContainer container = new AtomContainer(4, 3, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);

        String[] expected = {"CO2M", "O2CM", "O2CM", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CAGREH10
     *
     * @cdk.inchi InChI=1S/C7H5NO6S2/c9-15(10)8-7(13-16(11,12)14-15)6-4-2-1-3-5-6/h1-5H
     */
    @Test
    public void testCAGREH10() {
        IAtomContainer container = new AtomContainer(21, 22, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, DOUBLE);
        container.addBond(0, 5, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(1, 7, DOUBLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(15, 20, SINGLE);

        String[] expected = {"SO4", "SO3", "OSO3", "OSO3", "O2S", "O2S", "O2S", "O2S", "N=C", "C=N", "CB", "CB", "CB",
                "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CALXES20
     *
     * @cdk.inchi InChI=1S/C7H13N3O4/c1-4(8)7(14)10-2-5(11)9-3-6(12)13/h4H,2-3,8H2,1H3,(H,9,11)(H,10,14)(H,12,13)/t4-/m0/s1
     */
    @Test
    public void testCALXES20() {
        IAtomContainer container = new AtomContainer(27, 26, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(0, 16, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(2, 17, SINGLE);
        container.addBond(2, 18, SINGLE);
        container.addBond(2, 19, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 13, SINGLE);

        String[] expected = {"NR+", "CR", "CR", "C=ON", "O=CN", "NC=O", "CR", "C=ON", "O=CN", "NC=O", "CR", "CO2M",
                "O2CM", "O2CM", "HNR+", "HNR+", "HNR+", "HC", "HC", "HC", "HC", "HNCO", "HC", "HC", "HNCO", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CAMALD03
     *
     * @cdk.inchi InChI=1S/C3H4O4/c4-2(5)1-3(6)7/h1H2,(H,4,5)(H,6,7)/p-2
     */
    @Test
    public void testCAMALD03() {
        IAtomContainer container = new AtomContainer(9, 8, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(2, 5, SINGLE);

        String[] expected = {"CO2M", "CR", "CO2M", "O2CM", "O2CM", "O2CM", "O2CM", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CEFMEN
     *
     * @cdk.inchi InChI=1S/C16H17N9O5S3/c1-24-16(20-22-23-24)33-4-6-3-31-13-9(12(27)25(13)10(6)14(28)29)19-11(26)8(21-30-2)7-5-32-15(17)18-7/h5,9,13H,3-4H2,1-2H3,(H2,17,18)(H,19,26)(H,28,29)/b21-8-/t9-,13-/m1/s1
     */
    @Test
    public void testCEFMEN() {
        IAtomContainer container = new AtomContainer(50, 53, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 17, SINGLE);
        container.addBond(0, 20, SINGLE);
        container.addBond(1, 28, SINGLE);
        container.addBond(1, 29, SINGLE);
        container.addBond(2, 30, SINGLE);
        container.addBond(2, 31, SINGLE);
        container.addBond(3, 22, DOUBLE);
        container.addBond(4, 23, DOUBLE);
        container.addBond(5, 23, SINGLE);
        container.addBond(5, 35, SINGLE);
        container.addBond(6, 24, DOUBLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 26, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(9, 38, SINGLE);
        container.addBond(10, 25, DOUBLE);
        container.addBond(11, 27, SINGLE);
        container.addBond(11, 29, DOUBLE);
        container.addBond(12, 29, SINGLE);
        container.addBond(12, 41, SINGLE);
        container.addBond(12, 42, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 31, SINGLE);
        container.addBond(13, 32, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 31, DOUBLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(17, 33, SINGLE);
        container.addBond(17, 34, SINGLE);
        container.addBond(18, 19, DOUBLE);
        container.addBond(18, 30, SINGLE);
        container.addBond(19, 23, SINGLE);
        container.addBond(20, 21, SINGLE);
        container.addBond(20, 36, SINGLE);
        container.addBond(21, 22, SINGLE);
        container.addBond(21, 37, SINGLE);
        container.addBond(24, 25, SINGLE);
        container.addBond(25, 27, SINGLE);
        container.addBond(26, 39, SINGLE);
        container.addBond(26, 45, SINGLE);
        container.addBond(26, 46, SINGLE);
        container.addBond(27, 28, DOUBLE);
        container.addBond(28, 40, SINGLE);
        container.addBond(30, 43, SINGLE);
        container.addBond(30, 44, SINGLE);
        container.addBond(32, 47, SINGLE);
        container.addBond(32, 48, SINGLE);
        container.addBond(32, 49, SINGLE);

        String[] expected = {"S", "STHI", "S", "O=CN", "O=CO", "OC=O", "O=CN", "OR", "NC=O", "NC=O", "N=C", "N5B",
                "NC=N", "NPYL", "N5A", "N5B", "N5B", "CR", "C=C", "C=C", "CR4R", "CR4R", "C=ON", "COO", "C=ON", "C=N",
                "CR", "C5B", "C5A", "C5A", "CR", "C5A", "CR", "HC", "HC", "HOCO", "HC", "HC", "HNCO", "HC", "HC",
                "HNCN", "HNCN", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CETROI01
     *
     * @cdk.inchi InChI=1S/C10H15NO5S/c1-4-6(2-11)17-10-9(15-4)8(14)7(13)5(3-12)16-10/h4-10,12-14H,3H2,1H3/t4-,5+,6+,7+,8-,9+,10-/m0/s1
     */
    @Test
    public void testCETROI01() {
        IAtomContainer container = new AtomContainer(32, 33, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(2, 18, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(6, 21, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(10, 11, TRIPLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(13, 29, SINGLE);
        container.addBond(14, 30, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(16, 31, SINGLE);

        String[] expected = {"S", "CR", "CR", "OR", "CR", "CR", "CR", "CR", "OR", "CR", "CSP", "NSP", "CR", "OR", "OR",
                "CR", "OR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HOR", "HOR", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CEWCUC10
     *
     * @cdk.inchi InChI=1S/C18H15OP/c19-20(16-10-4-1-5-11-16,17-12-6-2-7-13-17)18-14-8-3-9-15-18/h1-15H
     */
    @Test
    public void testCEWCUC10() {
        IAtomContainer container = new AtomContainer(35, 37, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 20, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 21, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 22, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 27, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(13, 29, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 30, SINGLE);
        container.addBond(16, 17, DOUBLE);
        container.addBond(16, 31, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(17, 32, SINGLE);
        container.addBond(18, 19, DOUBLE);
        container.addBond(18, 33, SINGLE);
        container.addBond(19, 34, SINGLE);

        String[] expected = {"PO", "OP", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB",
                "CB", "CB", "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CEWVIJ10
     *
     * @cdk.inchi InChI=1S/C10H14N4O2/c1-6(2)4-14-8-7(11-5-12-8)9(15)13(3)10(14)16/h5-6H,4H2,1-3H3,(H,11,12)
     */
    @Test
    public void testCEWVIJ10() {
        IAtomContainer container = new AtomContainer(30, 31, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, DOUBLE);
        container.addBond(1, 9, DOUBLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 10, DOUBLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(15, 29, SINGLE);

        String[] expected = {"O=CN", "O=CN", "NC=O", "NC=O", "NPYL", "N5B", "CONN", "C5B", "C5A", "C=ON", "C5A", "CR",
                "CR", "CR", "CR", "CR", "HPYL", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CEWYIM30
     *
     * @cdk.inchi InChI=1S/C12H8O2/c13-9-5-1-3-7-8-4-2-6-10(14)12(8)11(7)9/h1-6,13-14H
     */
    @Test
    public void testCEWYIM30() {
        IAtomContainer container = new AtomContainer(22, 24, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 14, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 13, DOUBLE);

        String[] expected = {"OC=C", "OC=C", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB",
                "HOCC", "HOCC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIHWUL10
     *
     * @cdk.inchi InChI=1S/C6H8O3/c1-3(7-1)5-6(9-5)4-2-8-4/h3-6H,1-2H2/t3-,4-,5+,6+/m1/s1
     */
    @Test
    public void testCIHWUL10() {
        IAtomContainer container = new AtomContainer(17, 19, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(8, 16, SINGLE);

        String[] expected = {"OR", "OR", "OR", "CR3R", "CR3R", "CR3R", "CR3R", "CR3R", "CR3R", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIJXOI10
     *
     * @cdk.inchi InChI=1S/C9H13NO/c1-5-3-7-4-8(11)9(5)6(2)10-7/h5,7,9H,3-4H2,1-2H3/p+1/t5-,7-,9-/m0/s1
     */
    @Test
    public void testCIJXOI10() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 12, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 9, DOUBLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 10, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(9, 11, SINGLE);

        String[] expected = {"CR", "CR", "CR", "CR", "C=N", "C=OR", "CR", "CR", "CR", "N+=C", "O=CR", "HNC+", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIKSEU10
     *
     * @cdk.inchi InChI=1S/C10H5N5/c11-4-7-5-12-8-2-1-3-9-13-6-14-10(7)15(8)9/h1-3,5-6H
     */
    @Test
    public void testCIKSEU10() {
        IAtomContainer container = new AtomContainer(20, 22, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 11, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(13, 14, TRIPLE);

        String[] expected = {"N=C", "C=N", "N=C", "C=N", "C=C", "C=C", "N=C", "C=N", "C=C", "C=C", "C=C", "C=C",
                "NC=N", "CSP", "NSP", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CILBII
     *
     * @cdk.inchi InChI=1S/C7H10N4O14/c1-20-7-6(25-11(18)19)5(24-10(16)17)4(23-9(14)15)3(22-7)2-21-8(12)13/h3-7H,2H2,1H3/t3-,4-,5+,6-,7-/m1/s1
     */
    @Test
    public void testCILBII() {
        IAtomContainer container = new AtomContainer(35, 35, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 17, SINGLE);
        container.addBond(0, 25, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 26, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 27, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 28, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(4, 29, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 30, SINGLE);
        container.addBond(5, 31, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 32, SINGLE);
        container.addBond(6, 33, SINGLE);
        container.addBond(6, 34, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 21, DOUBLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(13, 22, DOUBLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(15, 23, SINGLE);
        container.addBond(16, 23, DOUBLE);
        container.addBond(18, 24, SINGLE);
        container.addBond(19, 24, SINGLE);
        container.addBond(20, 24, DOUBLE);

        String[] expected = {"CR", "CR", "CR", "CR", "CR", "CR", "CR", "OR", "ONO2", "O2NO", "O2NO", "ONO2", "O2NO",
                "O2NO", "ONO2", "O2NO", "O2NO", "OR", "ONO2", "O2NO", "O2NO", "NO3", "NO3", "NO3", "NO3", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CILDOQ
     *
     * @cdk.inchi InChI=1S/C6H9N3O4S/c7-6(14)9-8-3(5(12)13)1-2-4(10)11/h1-2H2,(H,10,11)(H,12,13)(H3,7,9,14)/b8-3+
     */
    @Test
    public void testCILDOQ() {
        IAtomContainer container = new AtomContainer(23, 22, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, DOUBLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(2, 10, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 22, SINGLE);
        container.addBond(4, 13, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 9, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(12, 21, SINGLE);

        String[] expected = {"S=C", "OC=O", "O=CO", "OC=O", "O=CO", "NC=S", "N=C", "NC=S", "C=SN", "C=N", "COO", "CR",
                "CR", "COO", "HNCS", "HNCS", "HNCS", "HOCO", "HC", "HC", "HC", "HC", "HOCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CILWUP11
     *
     * @cdk.inchi InChI=1S/C3H3N3/c1-2-4-6-5-3-1/h1-3H
     */
    @Test
    public void testCILWUP11() {
        IAtomContainer container = new AtomContainer(9, 9, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, SINGLE);

        String[] expected = {"NPYD", "NPYD", "NPYD", "CB", "CB", "CB", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIMRUL10
     *
     * @cdk.inchi InChI=1S/C12H12ClNO3/c1-17-11(16)12(7-9(12)13)14-10(15)8-5-3-2-4-6-8/h2-6,9H,7H2,1H3,(H,14,15)/t9-,12-/m0/s1
     */
    @Test
    public void testCIMRUL10() {
        IAtomContainer container = new AtomContainer(29, 30, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 8, DOUBLE);
        container.addBond(2, 15, DOUBLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 20, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(16, 26, SINGLE);
        container.addBond(16, 27, SINGLE);
        container.addBond(16, 28, SINGLE);

        String[] expected = {"CL", "O=CN", "O=CO", "OC=O", "NC=O", "CR3R", "CR3R", "CR3R", "C=ON", "CB", "CB", "CB",
                "CB", "CB", "CB", "COO", "CR", "HC", "HC", "HC", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CINVIE
     *
     * @cdk.inchi InChI=1S/C10H11N3O8/c1-9(15)7(14)6(11(16)17)4-10(13(20)21,5-2-3-5)8(9)12(18)19/h4-5,8,15H,2-3H2,1H3/t8-,9-,10-/m1/s1
     */
    @Test
    public void testCINVIE() {
        IAtomContainer container = new AtomContainer(32, 33, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 11, DOUBLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 21, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 8, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 9, DOUBLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 10, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 17, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(15, 23, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(17, 19, SINGLE);
        container.addBond(17, 24, SINGLE);
        container.addBond(18, 19, SINGLE);
        container.addBond(18, 25, SINGLE);
        container.addBond(18, 26, SINGLE);
        container.addBond(19, 27, SINGLE);
        container.addBond(19, 28, SINGLE);
        container.addBond(20, 29, SINGLE);
        container.addBond(20, 30, SINGLE);
        container.addBond(20, 31, SINGLE);

        String[] expected = {"O=CR", "OR", "O2N", "O2N", "O2N", "O2N", "O2N", "O2N", "NO2", "NO2", "NO2", "C=OR", "CR",
                "CR", "CR", "C=C", "C=C", "CR3R", "CR3R", "CR3R", "CR", "HOR", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIPVOM
     *
     * @cdk.inchi InChI=1S/C8H11N5O2/c1-4-15-8(14)6-5(2)10-13-7(6)9-11-12(13)3/h4H2,1-3H3
     */
    @Test
    public void testCIPVOM() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(14, 23, SINGLE);

        String[] expected = {"N=N", "N=N", "NN=N", "NPYL", "N5A", "C5B", "C5B", "C5A", "CR", "CR", "O=CO", "COO",
                "OC=O", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIPYAB10
     *
     * @cdk.inchi InChI=1S/C12N6/c13-1-7(2-14)10-11(8(3-15)4-16)12(10)9(5-17)6-18
     */
    @Test
    public void testCIPYAB10() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 4, DOUBLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 12, TRIPLE);
        container.addBond(7, 13, TRIPLE);
        container.addBond(8, 14, TRIPLE);
        container.addBond(9, 15, TRIPLE);
        container.addBond(10, 16, TRIPLE);
        container.addBond(11, 17, TRIPLE);

        String[] expected = {"C=C", "C=C", "C=C", "C=C", "C=C", "C=C", "CSP", "CSP", "CSP", "CSP", "CSP", "CSP", "NSP",
                "NSP", "NSP", "NSP", "NSP", "NSP"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CISMOG
     *
     * @cdk.inchi InChI=1S/C6H2N4S2/c1-2-4-6(10-12-8-4)5-3(1)7-11-9-5/h1-2H
     */
    @Test
    public void testCISMOG() {
        IAtomContainer container = new AtomContainer(14, 16, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(2, 4, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 8, DOUBLE);
        container.addBond(7, 9, DOUBLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(10, 13, DOUBLE);
        container.addBond(12, 13, SINGLE);

        String[] expected = {"STHI", "N5A", "N5A", "C5B", "C5B", "CB", "HC", "C5B", "CB", "N5A", "C5B", "HC", "STHI",
                "N5A"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CISPOJ
     *
     * @cdk.inchi InChI=1S/C8H4N8O4/c17-15(18)4-1-6-5(7(2-4)16(19)20)3-9-14(6)8-10-12-13-11-8/h1-3H,(H,10,11,12,13)
     */
    @Test
    public void testCISPOJ() {
        IAtomContainer container = new AtomContainer(24, 26, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 20, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 21, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 13, DOUBLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(14, 16, SINGLE);
        container.addBond(14, 17, DOUBLE);
        container.addBond(15, 18, SINGLE);
        container.addBond(15, 19, DOUBLE);

        String[] expected = {"CB", "C5A", "C5B", "CB", "CB", "CB", "C5B", "C5A", "N5B", "N5B", "N5A", "NPYL", "NPYL",
                "N5A", "NO2", "NO2", "O2N", "O2N", "O2N", "O2N", "HC", "HC", "HC", "HPYL"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CITDIS
     *
     * @cdk.inchi InChI=1S/C8H5Cl3O2/c1-4(12)13-8-6(10)3-2-5(9)7(8)11/h2-3H,1H3
     */
    @Test
    public void testCITDIS() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 5, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 9, DOUBLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 16, SINGLE);
        container.addBond(14, 17, SINGLE);

        String[] expected = {"CB", "CB", "CL", "CB", "CL", "CB", "HC", "CB", "HC", "CB", "CL", "OC=O", "COO", "O=CO",
                "CR", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CITNOI10
     *
     * @cdk.inchi InChI=1S/C11H13NO2S/c1-2-15-10(13)8-12-11(14)9-6-4-3-5-7-9/h3-7H,2,8H2,1H3,(H,12,14)
     */
    @Test
    public void testCITNOI10() {
        IAtomContainer container = new AtomContainer(28, 28, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(2, 8, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 26, SINGLE);
        container.addBond(14, 27, SINGLE);

        String[] expected = {"S", "O=C", "O=CN", "NC=O", "CR", "CR", "C=OS", "CR", "C=ON", "CB", "CB", "CB", "CB",
                "CB", "CB", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CITPEA10
     *
     * @cdk.inchi InChI=1S/C12H15NOS2/c1-3-16-12(15)9(2)13-11(14)10-7-5-4-6-8-10/h4-9H,3H2,1-2H3,(H,13,14)/t9-/m0/s1
     */
    @Test
    public void testCITPEA10() {
        IAtomContainer container = new AtomContainer(31, 31, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(2, 9, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 26, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 29, SINGLE);
        container.addBond(15, 30, SINGLE);

        String[] expected = {"S", "S=C", "O=CN", "NC=O", "CR", "CR", "CSS", "CR", "CR", "C=ON", "CB", "CB", "CB", "CB",
                "CB", "CB", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CITSED10
     *
     * @cdk.inchi InChI=1S/CH4N2O/c2-1-3-4/h1,4H,(H2,2,3)/p+1
     */
    @Test
    public void testCITSED10() {
        IAtomContainer container = new AtomContainer(9, 8, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, SINGLE);

        String[] expected = {"NCN+", "CNN+", "NCN+", "-O-", "HO", "HNN+", "HC", "HNN+", "HNN+"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIVCEP02
     *
     * @cdk.inchi InChI=1S/C12H15N3O6/c16-10-13(1-7-4-19-7)11(17)15(3-9-6-21-9)12(18)14(10)2-8-5-20-8/h7-9H,1-6H2/t7-,8-,9-/m0/s1
     */
    @Test
    public void testCIVCEP02() {
        IAtomContainer container = new AtomContainer(36, 39, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, DOUBLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(13, 16, DOUBLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(14, 17, SINGLE);
        container.addBond(14, 18, DOUBLE);
        container.addBond(15, 19, SINGLE);
        container.addBond(15, 20, SINGLE);
        container.addBond(15, 21, SINGLE);
        container.addBond(17, 22, SINGLE);
        container.addBond(19, 23, SINGLE);
        container.addBond(19, 24, SINGLE);
        container.addBond(19, 25, SINGLE);
        container.addBond(22, 26, SINGLE);
        container.addBond(22, 27, SINGLE);
        container.addBond(22, 28, SINGLE);
        container.addBond(23, 24, SINGLE);
        container.addBond(24, 29, SINGLE);
        container.addBond(24, 30, SINGLE);
        container.addBond(26, 31, SINGLE);
        container.addBond(26, 32, SINGLE);
        container.addBond(26, 33, SINGLE);
        container.addBond(31, 32, SINGLE);
        container.addBond(32, 34, SINGLE);
        container.addBond(32, 35, SINGLE);

        String[] expected = {"O=CN", "OR", "NC=O", "CONN", "CR", "CR3R", "CR3R", "HC", "HC", "HC", "HC", "HC", "NC=O",
                "CONN", "CONN", "CR", "O=CN", "NC=O", "O=CN", "CR3R", "HC", "HC", "CR", "OR", "CR3R", "HC", "CR3R",
                "HC", "HC", "HC", "HC", "OR", "CR3R", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIVLAU02
     *
     * @cdk.inchi InChI=1S/C11H14O6/c1-14-9(12)5-7-6(10(13)15-2)8(5)11(7)16-3-4-17-11/h5-8H,3-4H2,1-2H3/t5-,6+,7+,8-
     */
    @Test
    public void testCIVLAU02() {
        IAtomContainer container = new AtomContainer(31, 33, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 17, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 19, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 20, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 21, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 28, SINGLE);
        container.addBond(16, 29, SINGLE);
        container.addBond(16, 30, SINGLE);

        String[] expected = {"CR4R", "CR4R", "CR4R", "CR4R", "CR4R", "OR", "CR", "CR", "OR", "COO", "O=CO", "OC=O",
                "CR", "COO", "O=CO", "OC=O", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIXWAH
     *
     * @cdk.inchi InChI=1S/C7H10O4/c8-4-1-5-3-6(2-4)11-7(9)10-5/h4-6,8H,1-3H2/t4-,5+,6-
     */
    @Test
    public void testCIXWAH() {
        IAtomContainer container = new AtomContainer(21, 22, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 10, DOUBLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(9, 20, SINGLE);

        String[] expected = {"OR", "OC=O", "O=CO", "OC=O", "CR", "CR", "CR", "CR", "CR", "CR", "COOO", "HOR", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIYNUT
     *
     * @cdk.inchi InChI=1S/C6N4O2S2/c11-5-1-2(8-13-7-1)6(12)4-3(5)9-14-10-4
     */
    @Test
    public void testCIYNUT() {
        IAtomContainer container = new AtomContainer(14, 16, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 9, DOUBLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 11, DOUBLE);
        container.addBond(10, 12, DOUBLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(12, 13, SINGLE);

        String[] expected = {"STHI", "O=CR", "N5A", "N5A", "C5B", "C5B", "C=OR", "C=OR", "C5B", "O=CR", "C5B", "N5A",
                "N5A", "STHI"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIZFIA
     *
     * @cdk.inchi InChI=1S/C9H13NO3S/c1-6-5-8-4-3-7(2)14(12)10(8)9(11)13-6/h3-4,6-8H,5H2,1-2H3/t6-,7+,8+,14-/m0/s1
     */
    @Test
    public void testCIZFIA() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(3, 9, DOUBLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(13, 26, SINGLE);

        String[] expected = {"S=O", "OC=O", "O=S", "O=CN", "NC=O", "CR", "C=C", "C=C", "CR", "C=ON", "CR", "CR", "CR",
                "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIZJAW
     *
     * @cdk.inchi InChI=1S/C12H30Cl2Si4/c1-15(2,3)11(13)17(7,8)12(14,16(4,5)6)18(11,9)10/h1-10H3/t11-,12+
     */
    @Test
    public void testCIZJAW() {
        IAtomContainer container = new AtomContainer(48, 48, 0, 0);
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(10, 24, SINGLE);
        container.addBond(11, 29, SINGLE);
        container.addBond(11, 28, SINGLE);
        container.addBond(11, 27, SINGLE);
        container.addBond(12, 32, SINGLE);
        container.addBond(12, 31, SINGLE);
        container.addBond(12, 30, SINGLE);
        container.addBond(13, 35, SINGLE);
        container.addBond(13, 34, SINGLE);
        container.addBond(13, 33, SINGLE);
        container.addBond(14, 38, SINGLE);
        container.addBond(14, 37, SINGLE);
        container.addBond(14, 36, SINGLE);
        container.addBond(15, 41, SINGLE);
        container.addBond(15, 40, SINGLE);
        container.addBond(15, 39, SINGLE);
        container.addBond(16, 44, SINGLE);
        container.addBond(16, 43, SINGLE);
        container.addBond(16, 42, SINGLE);
        container.addBond(17, 47, SINGLE);
        container.addBond(17, 46, SINGLE);
        container.addBond(17, 45, SINGLE);

        String[] expected = {"SI", "SI", "SI", "SI", "CL", "CL", "CR4R", "CR4R", "CR", "CR", "CR", "CR", "CR", "CR",
                "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIZWUD
     *
     * @cdk.inchi InChI=1S/C6H4N4O/c7-5-9-10(11)6-3-1-2-4-8-6/h1-4H/b10-9-
     */
    @Test
    public void testCIZWUD() {
        IAtomContainer container = new AtomContainer(15, 15, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 6, DOUBLE);
        container.addBond(5, 9, TRIPLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 8, DOUBLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CSP", "NPYD", "N2OX", "N=N", "NSP", "OXN", "HC", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIZYEP
     *
     * @cdk.inchi InChI=1S/C12H14N2S/c1-14(2)12(15-3)11(9-13)10-7-5-4-6-8-10/h4-8H,1-3H3/b12-11+
     */
    @Test
    public void testCIZYEP() {
        IAtomContainer container = new AtomContainer(29, 29, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 14, TRIPLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(11, 25, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(12, 28, SINGLE);

        String[] expected = {"S", "CSP", "C=C", "C=C", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "CR", "CR", "NC=C",
                "NSP", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CIZZUG
     *
     * @cdk.inchi InChI=1S/C6H6N4O3S/c11-5-7-1-2-9(5)6-8-3-4(14-6)10(12)13/h3H,1-2H2,(H,7,11)
     */
    @Test
    public void testCIZZUG() {
        IAtomContainer container = new AtomContainer(20, 21, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, DOUBLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 13, DOUBLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 19, SINGLE);

        String[] expected = {"STHI", "C5A", "N5B", "C5B", "C5A", "NO2", "O2N", "O2N", "NC=O", "CONN", "NC=O", "CR",
                "CR", "O=CN", "HC", "HNCO", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COBKIN01
     *
     * @cdk.inchi InChI=1S/C7H12N2O4/c1-3-13-6(11)7(5(8)10)4-9(7)12-2/h3-4H2,1-2H3,(H2,8,10)/t7-,9+/m0/s1
     */
    @Test
    public void testCOBKIN01() {
        IAtomContainer container = new AtomContainer(25, 25, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(1, 8, DOUBLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 9, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 19, SINGLE);

        String[] expected = {"OR", "O=CN", "OC=O", "O=CO", "NR", "NC=O", "CR3R", "CR3R", "C=ON", "COO", "CR", "CR",
                "CR", "HNCO", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COCXUN
     *
     * @cdk.inchi InChI=1S/C12H6N4/c13-7-15-11-5-6-12(16-8-14)10-4-2-1-3-9(10)11/h1-6H/b15-11+,16-12+
     */
    @Test
    public void testCOCXUN() {
        IAtomContainer container = new AtomContainer(22, 23, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 13, DOUBLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 17, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 15, DOUBLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 12, TRIPLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 14, TRIPLE);

        String[] expected = {"C=N", "C=C", "C=C", "C=N", "CB", "CB", "CB", "CB", "CB", "CB", "CSP", "CSP", "NSP",
                "N=C", "NSP", "N=C", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COGDEH
     *
     * @cdk.inchi InChI=1S/C11H6N6/c1-2-4-8-7(3-1)5-6-9-10(8)12-11-13-15-16-17(11)14-9/h1-6H
     */
    @Test
    public void testCOGDEH() {
        IAtomContainer container = new AtomContainer(23, 26, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 7, DOUBLE);
        container.addBond(5, 8, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 20, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 21, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(16, 22, SINGLE);

        String[] expected = {"N5B", "N5B", "N5A", "NPYL", "N=C", "N=C", "C5A", "C=N", "C=N", "C=C", "C=C", "CB", "CB",
                "CB", "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COGYAY
     *
     * @cdk.inchi InChI=1S/C8H6ClN3/c9-8-6-10-12(11-8)7-4-2-1-3-5-7/h1-6H
     */
    @Test
    public void testCOGYAY() {
        IAtomContainer container = new AtomContainer(18, 19, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 17, SINGLE);

        String[] expected = {"CL", "N5A", "NPYL", "N5A", "C5B", "C5B", "CB", "CB", "CB", "CB", "CB", "CB", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COHKOZ
     *
     * @cdk.inchi InChI=1S/C5H4N4OS/c1-10-4(7)5-8-3(2-6)9-11-5/h7H,1H3/b7-4+
     */
    @Test
    public void testCOHKOZ() {
        IAtomContainer container = new AtomContainer(15, 15, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(4, 8, TRIPLE);
        container.addBond(5, 9, DOUBLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 14, SINGLE);

        String[] expected = {"STHI", "OC=N", "N5A", "N5B", "NSP", "N=C", "C5B", "C5A", "CSP", "C=N", "CR", "HN=C",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COJFIQ
     *
     * @cdk.inchi InChI=1S/C7H9N5/c1-12(2)7-5-6(9-3-8-5)10-4-11-7/h3-4H,1-2H3,(H,8,9,10,11)/p+2
     */
    @Test
    public void testCOJFIQ() {
        IAtomContainer container = new AtomContainer(23, 24, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 12, SINGLE);
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 22, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 19, SINGLE);

        String[] expected = {"NPD+", "CB", "NPYD", "C5", "C5", "CB", "NC=N", "NIM+", "CIM+", "NIM+", "CR", "CR",
                "HPD+", "HIM+", "HC", "HIM+", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COKDEL
     *
     * @cdk.inchi InChI=1S/C8H14O2S/c1-6-7(8(2,3)4)10-11(5)9/h1,7H,2-5H3/t7-,11+/m0/s1
     */
    @Test
    public void testCOKDEL() {
        IAtomContainer container = new AtomContainer(25, 24, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(5, 6, TRIPLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(10, 24, SINGLE);

        String[] expected = {"S=O", "O=S", "OS=O", "CR", "CR", "CSP", "CSP", "CR", "CR", "CR", "CR", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COKROJ
     *
     * @cdk.inchi InChI=1S/C10H10N3O3S/c1-7-6-10(12-16-7)13-17(14,15)9-4-2-8(11)3-5-9/h2-6H,11H2,1H3/q-1
     */
    @Test
    public void testCOKROJ() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addBond(0, 13, SINGLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(6, 21, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(16, 26, SINGLE);
        container.addBond(17, 26, SINGLE);
        container.addBond(17, 22, SINGLE);
        container.addBond(17, 18, DOUBLE);
        container.addBond(18, 19, SINGLE);
        container.addBond(19, 20, DOUBLE);
        container.addBond(20, 23, SINGLE);
        container.addBond(20, 21, SINGLE);
        container.addBond(21, 22, DOUBLE);
        container.addBond(24, 26, DOUBLE);
        container.addBond(25, 26, DOUBLE);

        String[] expected = {"HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCC", "HNCC", "OFUR", "N5A", "C5B",
                "C5B", "C5A", "CR", "NM", "CB", "CB", "CB", "CB", "CB", "CB", "NC=C", "O2S", "O2S", "SO2N"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COLZUY
     *
     * @cdk.inchi InChI=1S/C12H26N4P2/c1-17-13-5-3-6-14(17)10-12-16-8-4-7-15(11-9-13)18(16)2/h3-12H2,1-2H3/t17-,18-
     */
    @Test
    public void testCOLZUY() {
        IAtomContainer container = new AtomContainer(44, 46, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 21, SINGLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 23, SINGLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 25, SINGLE);
        container.addBond(5, 24, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 27, SINGLE);
        container.addBond(7, 26, SINGLE);
        container.addBond(8, 30, SINGLE);
        container.addBond(8, 29, SINGLE);
        container.addBond(8, 28, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(10, 32, SINGLE);
        container.addBond(10, 31, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(12, 34, SINGLE);
        container.addBond(12, 33, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(13, 36, SINGLE);
        container.addBond(13, 35, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 17, SINGLE);
        container.addBond(15, 39, SINGLE);
        container.addBond(15, 38, SINGLE);
        container.addBond(15, 37, SINGLE);
        container.addBond(16, 41, SINGLE);
        container.addBond(16, 40, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(17, 43, SINGLE);
        container.addBond(17, 42, SINGLE);

        String[] expected = {"P", "CR", "CR", "NR", "CR", "CR", "NR", "CR", "CR", "NR", "CR", "P", "CR", "CR", "NR",
                "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COMDIR
     *
     * @cdk.inchi InChI=1S/C11H8Br2O2/c12-7-4-8(14)11(13)6-2-1-5(3-6)9(11)10(7)15/h1-2,4-6,9H,3H2/t5-,6+,9+,11+/m1/s1
     */
    @Test
    public void testCOMDIR() {
        IAtomContainer container = new AtomContainer(23, 25, 0, 0);
        container.addAtom(newAtm("Br", 0));
        container.addAtom(newAtm("Br", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 4, DOUBLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(14, 21, SINGLE);
        container.addBond(14, 22, SINGLE);

        String[] expected = {"BR", "BR", "O=CR", "O=CR", "C=OR", "C=C", "C=C", "C=OR", "CR", "CR", "CR", "C=C", "C=C",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COMKAQ
     *
     * @cdk.inchi InChI=1S/C7H7NO2/c8-2-4-1-5-3-10-7(9)6(4)5/h4-6H,1,3H2/t4-,5+,6+/m0/s1
     */
    @Test
    public void testCOMKAQ() {
        IAtomContainer container = new AtomContainer(17, 18, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(8, 9, TRIPLE);

        String[] expected = {"COO", "O=CO", "OC=O", "CR", "CR4R", "CR4R", "CR4R", "CR4R", "CSP", "NSP", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COMWOQ
     *
     * @cdk.inchi InChI=1S/C9H7NO/c11-6-7-5-10-9-4-2-1-3-8(7)9/h1-6,10H
     */
    @Test
    public void testCOMWOQ() {
        IAtomContainer container = new AtomContainer(18, 19, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 11, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 10, DOUBLE);

        String[] expected = {"NPYL", "C5A", "C5B", "CB", "CB", "CB", "CB", "C5A", "C5B", "C=OR", "O=CR", "HPYL", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COMWUW
     *
     * @cdk.inchi InChI=1S/C11H14N2/c1-13(2)8-9-7-12-11-6-4-3-5-10(9)11/h3-7,12H,8H2,1-2H3
     */
    @Test
    public void testCOMWUW() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 13, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(11, 26, SINGLE);
        container.addBond(11, 25, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(12, 21, SINGLE);

        String[] expected = {"NPYL", "NR", "C5A", "C5B", "CB", "CB", "CB", "CB", "C5A", "C5B", "CR", "CR", "CR",
                "HPYL", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CONBAI
     *
     * @cdk.inchi InChI=1S/C12H14F2N4O4/c1-15-5-6-12(14,8(20)18(4)10(22)16(6)2)11(5,13)7(19)17(3)9(15)21/h5-6H,1-4H3/t5-,6+,11-,12+
     */
    @Test
    public void testCONBAI() {
        IAtomContainer container = new AtomContainer(36, 38, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(1, 22, SINGLE);
        container.addBond(1, 23, SINGLE);
        container.addBond(1, 24, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 25, SINGLE);
        container.addBond(5, 26, SINGLE);
        container.addBond(5, 27, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(10, 28, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(12, 29, SINGLE);
        container.addBond(12, 30, SINGLE);
        container.addBond(12, 31, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 17, SINGLE);
        container.addBond(16, 32, SINGLE);
        container.addBond(16, 33, SINGLE);
        container.addBond(16, 34, SINGLE);
        container.addBond(17, 18, DOUBLE);
        container.addBond(17, 19, SINGLE);
        container.addBond(19, 20, SINGLE);
        container.addBond(19, 21, SINGLE);
        container.addBond(21, 35, SINGLE);

        String[] expected = {"NC=O", "CR", "CONN", "O=CN", "NC=O", "CR", "C=ON", "O=CN", "CR4R", "F", "CR4R", "NC=O",
                "CR", "CONN", "O=CN", "NC=O", "CR", "C=ON", "O=CN", "CR4R", "F", "CR4R", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CONFAM
     *
     * @cdk.inchi InChI=1S/C2H5NOS/c1-4-2(3)5/h1H3,(H2,3,5)
     */
    @Test
    public void testCONFAM() {
        IAtomContainer container = new AtomContainer(10, 9, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 8, SINGLE);

        String[] expected = {"S=C", "C=SN", "CR", "OC=S", "NC=S", "HC", "HC", "HC", "HNCS", "HNCS"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CONLIA
     *
     * @cdk.inchi InChI=1S/C7H18N2/c1-8(2)5-6-9(3,4)7-8/h5-7H2,1-4H3/q+2
     */
    @Test
    public void testCONLIA() {
        IAtomContainer container = new AtomContainer(27, 27, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(8, 26, SINGLE);

        String[] expected = {"NR+", "NR+", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CORDOC
     *
     * @cdk.inchi InChI=1S/C9H9ClO3S/c1-14(12,13)9(10)8(11)7-5-3-2-4-6-7/h2-6,9H,1H3/t9-/m1/s1
     */
    @Test
    public void testCORDOC() {
        IAtomContainer container = new AtomContainer(23, 23, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 11, DOUBLE);
        container.addBond(0, 12, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 13, DOUBLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 22, SINGLE);

        String[] expected = {"SO2", "CL", "CR", "CR", "C=OR", "CB", "CB", "CB", "CB", "CB", "CB", "O2S", "O2S", "O=CR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CORWUB10
     *
     * @cdk.inchi InChI=1S/C2H3NOS2/c4-1-3-2(5)6/h1H,(H2,3,4,5,6)/p-1
     */
    @Test
    public void testCORWUB10() {
        IAtomContainer container = new AtomContainer(8, 7, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 5, DOUBLE);

        String[] expected = {"S2CM", "S2CM", "CS2M", "NC=O", "C=ON", "O=CN", "HNCO", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COSFAR
     *
     * @cdk.inchi InChI=1S/C8H7NOS/c1-9-6-4-2-3-5-7(6)11-8(9)10/h2-5H,1H3
     */
    @Test
    public void testCOSFAR() {
        IAtomContainer container = new AtomContainer(18, 19, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 9, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(10, 17, SINGLE);

        String[] expected = {"S", "O=CN", "NC=O", "CB", "CB", "CB", "CB", "CB", "CB", "C=ON", "CR", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COSSEI
     *
     * @cdk.inchi InChI=1S/C4H8N2S4/c7-3(8)5-1-2-6-4(9)10/h1-2H2,(H2,5,7,8)(H2,6,9,10)/p-2
     */
    @Test
    public void testCOSSEI() {
        IAtomContainer container = new AtomContainer(16, 15, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", -1));
        container.addBond(0, 2, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(12, 14, DOUBLE);

        String[] expected = {"S2CM", "S2CM", "CS2M", "NC=S", "CR", "HNCS", "HC", "HC", "CR", "NC=S", "HC", "HC",
                "CS2M", "HNCS", "S2CM", "S2CM"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COSWIQ
     *
     * @cdk.inchi InChI=1S/C11H10N2O/c1-3-10-6-4-9(8-12)5-7-11(13-10)14-2/h3-7H,1H2,2H3/b6-4-,7-5-,9-4+,9-5+,10-6-,11-7+,13-10?,13-11+
     */
    @Test
    public void testCOSWIQ() {
        IAtomContainer container = new AtomContainer(24, 24, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 11, TRIPLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 22, SINGLE);

        String[] expected = {"N=C", "C=N", "C=C", "C=C", "C=C", "C=C", "C=C", "C=C", "OC=N", "CR", "CSP", "NSP", "C=C",
                "C=C", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COTMON
     *
     * @cdk.inchi InChI=1S/C7H8ClN3S/c8-5-1-3-6(4-2-5)10-7(12)11-9/h1-4H,9H2,(H2,10,11,12)
     */
    @Test
    public void testCOTMON() {
        IAtomContainer container = new AtomContainer(20, 20, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(0, 13, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 11, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 19, SINGLE);

        String[] expected = {"NR", "NC=S", "C=SN", "NC=S", "CB", "CB", "CB", "CB", "CB", "CB", "CL", "S=C", "HNR",
                "HNR", "HNCS", "HNCS", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COTPEG
     *
     * @cdk.inchi InChI=1S/C5H6O5/c6-3(5(9)10)1-2-4(7)8/h1-2H2,(H,7,8)(H,9,10)/p-1
     */
    @Test
    public void testCOTPEG() {
        IAtomContainer container = new AtomContainer(15, 14, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(3, 9, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 14, SINGLE);

        String[] expected = {"O2CM", "O2CM", "O=CR", "O=CO", "OC=O", "CO2M", "C=OR", "CR", "CR", "COO", "HOCO", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COTRIM
     *
     * @cdk.inchi InChI=1S/C11H9N3O2/c15-13-8-6-11(7-9-13)14(16)12-10-4-2-1-3-5-10/h1-9H/b14-12-
     */
    @Test
    public void testCOTRIM() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 9, DOUBLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(15, 24, SINGLE);

        String[] expected = {"OXN", "OXN", "NPOX", "N2OX", "N=N", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB",
                "CB", "CB", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COVHUQ
     *
     * @cdk.inchi InChI=1S/C4H6O2S5/c1-7-3(5)9-11-10-4(6)8-2/h1-2H3
     */
    @Test
    public void testCOVHUQ() {
        IAtomContainer container = new AtomContainer(17, 16, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(13, 16, SINGLE);

        String[] expected = {"CR", "C=OS", "O=C", "S", "S", "S", "HC", "HC", "HC", "S", "C=OS", "O=C", "S", "CR", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COVMAB
     *
     * @cdk.inchi InChI=1S/C10H20N2/c1-9(2,3)11-7-8-12-10(4,5)6/h7-8H,1-6H3/b11-7+,12-8+
     */
    @Test
    public void testCOVMAB() {
        IAtomContainer container = new AtomContainer(32, 31, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(16, 17, DOUBLE);
        container.addBond(16, 18, SINGLE);
        container.addBond(17, 19, SINGLE);
        container.addBond(19, 20, SINGLE);
        container.addBond(19, 21, SINGLE);
        container.addBond(19, 22, SINGLE);
        container.addBond(20, 23, SINGLE);
        container.addBond(20, 24, SINGLE);
        container.addBond(20, 25, SINGLE);
        container.addBond(21, 26, SINGLE);
        container.addBond(21, 27, SINGLE);
        container.addBond(21, 28, SINGLE);
        container.addBond(22, 29, SINGLE);
        container.addBond(22, 30, SINGLE);
        container.addBond(22, 31, SINGLE);

        String[] expected = {"N=C", "C=N", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "C=N", "N=C", "HC", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COVXIU
     *
     * @cdk.inchi InChI=1S/C12H14N4O2S/c1-8-7-9(2)15-12(14-8)16-19(17,18)11-5-3-10(13)4-6-11/h3-7H,13H2,1-2H3,(H,14,15,16)
     */
    @Test
    public void testCOVXIU() {
        IAtomContainer container = new AtomContainer(33, 34, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 21, SINGLE);
        container.addBond(5, 22, SINGLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(6, 24, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 9, DOUBLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(15, 18, SINGLE);
        container.addBond(17, 27, SINGLE);
        container.addBond(17, 28, SINGLE);
        container.addBond(17, 29, SINGLE);
        container.addBond(18, 30, SINGLE);
        container.addBond(18, 31, SINGLE);
        container.addBond(18, 32, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "NC=C", "SO2N", "O2S", "O2S", "NSO2", "CB", "NPYD",
                "CB", "CB", "CB", "NPYD", "CR", "CR", "HC", "HC", "HC", "HC", "HNCC", "HNCC", "HNSO", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COWTIR
     *
     * @cdk.inchi InChI=1S/C10H16NO4P/c1-7(2)11-10(16(13,14)15)8-5-3-4-6-9(8)12/h3-7,10-12H,1-2H3,(H2,13,14,15)/t10-/m1/s1
     */
    @Test
    public void testCOWTIR() {
        IAtomContainer container = new AtomContainer(32, 32, 0, 0);
        container.addAtom(newAtm("P", 1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 27, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 29, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 30, SINGLE);
        container.addBond(15, 31, SINGLE);

        String[] expected = {"PO3", "OPO2", "O2P", "O2P", "OC=C", "NR+", "CR", "CR", "CR", "CR", "CB", "CB", "CB",
                "CB", "CB", "CB", "HOP", "HC", "HNR+", "HNR+", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HOCC", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COXBAS
     *
     * @cdk.inchi InChI=1S/C5H11O3PS3/c1-7-9(6,8-2)5-11-3-10-4-12-5/h5H,3-4H2,1-2H3
     */
    @Test
    public void testCOXBAS() {
        IAtomContainer container = new AtomContainer(23, 23, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 4, DOUBLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 19, SINGLE);

        String[] expected = {"PO3", "S", "S", "S", "OP", "OPO2", "OPO2", "CR", "CR", "CR", "CR", "CR", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COXZEU
     *
     * @cdk.inchi InChI=1S/C3H7N3S2/c1-8-3(7)6-2(4)5/h1H3,(H4,4,5,6,7)
     */
    @Test
    public void testCOXZEU() {
        IAtomContainer container = new AtomContainer(15, 14, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 8, SINGLE);

        String[] expected = {"S", "S=C", "N=C", "NC=N", "NC=N", "CGD", "CSS", "CR", "HC", "HC", "HC", "HNCN", "HNCN",
                "HNCN", "HNCN"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COYMOS
     *
     * @cdk.inchi InChI=1S/C10H11N3O2/c11-8(14)10(9(12)15)6-13(10)7-4-2-1-3-5-7/h1-5H,6H2,(H2,11,14)(H2,12,15)/t13-/m0/s1
     */
    @Test
    public void testCOYMOS() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(14, 25, SINGLE);

        String[] expected = {"O=CN", "O=CN", "NC=O", "NC=O", "NC=C", "C=ON", "C=ON", "CR3R", "CR3R", "CB", "CB", "CB",
                "CB", "CB", "CB", "HNCO", "HNCO", "HNCO", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COYNAF
     *
     * @cdk.inchi InChI=1S/C8H6N2O2/c9-10-7(11)5-3-1-2-4-6(5)8(10)12/h1-4H,9H2
     */
    @Test
    public void testCOYNAF() {
        IAtomContainer container = new AtomContainer(18, 19, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(0, 13, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 4, DOUBLE);
        container.addBond(3, 11, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(10, 11, SINGLE);

        String[] expected = {"NR", "NC=O", "O=CN", "O=CN", "C=ON", "CB", "CB", "CB", "CB", "CB", "CB", "C=ON", "HNR",
                "HNR", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for COYVIV
     *
     * @cdk.inchi InChI=1S/C9H10N4O2S2/c1-11-17(14,15)7-5-3-2-4-6(7)8-12-13-9(10)16-8/h2-5,11H,1H3,(H2,10,13)
     */
    @Test
    public void testCOYVIV() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 14, DOUBLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 23, SINGLE);
        container.addBond(16, 24, SINGLE);
        container.addBond(16, 25, SINGLE);
        container.addBond(16, 26, SINGLE);

        String[] expected = {"STHI", "C5A", "N5B", "N5B", "C5A", "CB", "CB", "CB", "CB", "CB", "CB", "NC=N", "SO2N",
                "O2S", "O2S", "NSO2", "CR", "HNCN", "HNCN", "HC", "HC", "HC", "HC", "HNSO", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUBTUO
     *
     * @cdk.inchi InChI=1S/C12H14N2/c1-9-4-3-5-11(10(9)2)6-12-7-13-8-14-12/h3-5,7-8H,6H2,1-2H3,(H,13,14)/p+1
     */
    @Test
    public void testCUBTUO() {
        IAtomContainer container = new AtomContainer(29, 30, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 10, DOUBLE);
        container.addBond(0, 16, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(7, 28, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(9, 11, DOUBLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(13, 26, SINGLE);
        container.addBond(13, 27, SINGLE);

        String[] expected = {"NIM+", "NIM+", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "C5", "CIM+", "C5", "CR", "CR",
                "HC", "HC", "HIM+", "HC", "HIM+", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUCDAF
     *
     * @cdk.inchi InChI=1S/C12H12O3/c1-6-4-3-5-7-8(6)12(2,14)11-10(15-11)9(7)13/h3-5,10-11,14H,1-2H3/t10-,11-,12+/m0/s1
     */
    @Test
    public void testCUCDAF() {
        IAtomContainer container = new AtomContainer(27, 29, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 20, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 24, SINGLE);

        String[] expected = {"OR", "OR", "O=CR", "CR", "CR3R", "CR3R", "C=OR", "CB", "CB", "CB", "CB", "CB", "CB",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HOR", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUCHOX
     *
     * @cdk.inchi InChI=1S/C11H14O4/c1-6(12)10-5-11(10,7(2)13)9(3,15)4-8(10)14/h15H,4-5H2,1-3H3/t9-,10-,11+/m0/s1
     */
    @Test
    public void testCUCHOX() {
        IAtomContainer container = new AtomContainer(29, 30, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 23, SINGLE);
        container.addBond(1, 24, SINGLE);
        container.addBond(1, 25, SINGLE);
        container.addBond(2, 7, DOUBLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 13, DOUBLE);
        container.addBond(4, 10, DOUBLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(14, 27, SINGLE);
        container.addBond(14, 28, SINGLE);

        String[] expected = {"OR", "CR", "O=CR", "C=OR", "O=CR", "CR3R", "CR", "C=OR", "CR", "CR3R", "C=OR", "CR3R",
                "CR", "O=CR", "CR", "HOR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUCHUD
     *
     * @cdk.inchi InChI=1S/C9H8N4O3S/c1-12-5-11-7(13(15)16)9(12)17-8-6(14)3-2-4-10-8/h2-5,14H,1H3
     */
    @Test
    public void testCUCHUD() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 14, DOUBLE);
        container.addBond(15, 21, SINGLE);
        container.addBond(15, 22, SINGLE);
        container.addBond(15, 23, SINGLE);
        container.addBond(16, 24, SINGLE);

        String[] expected = {"S", "C5A", "C5B", "N5B", "C5A", "NPYL", "CB", "NPYD", "CB", "CB", "CB", "CB", "NO2",
                "O2N", "O2N", "CR", "OC=C", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HOCC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUDJAM
     *
     * @cdk.inchi InChI=1S/C6H10N4O3/c1-3(8-12)6(9-13)10-2-4(10)5(7)11/h4,12-13H,2H2,1H3,(H2,7,11)/b8-3+,9-6-/t4-,10+/m1/s1
     */
    @Test
    public void testCUDJAM() {
        IAtomContainer container = new AtomContainer(23, 23, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 12, DOUBLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(4, 7, DOUBLE);
        container.addBond(5, 8, DOUBLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(11, 22, SINGLE);

        String[] expected = {"O=CN", "-O-", "-O-", "NC=N", "N=C", "N=C", "NC=O", "C=N", "C=N", "CR3R", "CR3R", "CR",
                "C=ON", "HO", "HO", "HNCO", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUDNEU
     *
     * @cdk.inchi InChI=1S/C10H22O2Si/c1-12-10(13(2,3)4)8-6-5-7-9(10)11/h9,11H,5-8H2,1-4H3/t9-,10+/m1/s1
     */
    @Test
    public void testCUDNEU() {
        IAtomContainer container = new AtomContainer(35, 35, 0, 0);
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(5, 22, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(8, 26, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 27, SINGLE);
        container.addBond(9, 28, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 29, SINGLE);
        container.addBond(10, 30, SINGLE);
        container.addBond(11, 31, SINGLE);
        container.addBond(12, 32, SINGLE);
        container.addBond(12, 33, SINGLE);
        container.addBond(12, 34, SINGLE);

        String[] expected = {"SI", "OR", "OR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "HOR", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUDPAS
     *
     * @cdk.inchi InChI=1S/C6H5N3O4/c10-5-3-7-2-1-4(5)6(8-11)9(12)13/h1-3,10-11H/p-1/b8-6+
     */
    @Test
    public void testCUDPAS() {
        IAtomContainer container = new AtomContainer(17, 17, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 12, DOUBLE);

        String[] expected = {"NPYD", "CB", "CB", "CB", "CB", "CB", "OC=C", "C=N", "N=C", "OM2", "NO2", "O2N", "O2N",
                "HC", "HC", "HC", "HOCC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUDPOG
     *
     * @cdk.inchi InChI=1S/C10H11N3S/c1-10(2)12-11-9(14)13(10)8-6-4-3-5-7-8/h3-7H,1-2H3
     */
    @Test
    public void testCUDPOG() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, DOUBLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(6, 24, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(13, 18, SINGLE);

        String[] expected = {"S=C", "NC=S", "N=N", "N=N", "C=SN", "CR", "CR", "CR", "CB", "CB", "CB", "CB", "CB", "CB",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUDREY
     *
     * @cdk.inchi InChI=1S/C3H4N2S/c4-3-5-1-2-6-3/h1-2H,(H2,4,5)/p+1
     */
    @Test
    public void testCUDREY() {
        IAtomContainer container = new AtomContainer(11, 11, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 8, SINGLE);

        String[] expected = {"STHI", "NIM+", "NCN+", "CIM+", "C5B", "C5A", "HIM+", "HC", "HC", "HNN+", "HNN+"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUFFAK
     *
     * @cdk.inchi InChI=1S/C20H12O5/c21-11-5-7-15-17(9-11)24-18-10-12(22)6-8-16(18)20(15)14-4-2-1-3-13(14)19(23)25-20/h1-10,21-22H
     */
    @Test
    public void testCUFFAK() {
        IAtomContainer container = new AtomContainer(37, 41, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(1, 27, SINGLE);
        container.addBond(1, 36, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 30, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 29, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 33, SINGLE);
        container.addBond(5, 31, SINGLE);
        container.addBond(5, 34, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 35, SINGLE);
        container.addBond(10, 12, DOUBLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(14, 16, DOUBLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 18, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(18, 20, DOUBLE);
        container.addBond(19, 26, SINGLE);
        container.addBond(20, 22, SINGLE);
        container.addBond(21, 28, SINGLE);
        container.addBond(22, 24, SINGLE);
        container.addBond(22, 34, SINGLE);
        container.addBond(23, 32, SINGLE);
        container.addBond(24, 26, DOUBLE);
        container.addBond(24, 33, SINGLE);
        container.addBond(25, 36, SINGLE);
        container.addBond(26, 28, SINGLE);
        container.addBond(28, 30, DOUBLE);
        container.addBond(30, 32, SINGLE);
        container.addBond(32, 33, DOUBLE);
        container.addBond(34, 35, SINGLE);
        container.addBond(35, 36, DOUBLE);

        String[] expected = {"OC=C", "CB", "OC=C", "CB", "OC=C", "CB", "OC=O", "COO", "O=CO", "HOCC", "CB", "HOCC",
                "CB", "HC", "CB", "HC", "CB", "HC", "CB", "HC", "CB", "HC", "CR", "HC", "CB", "HC", "CB", "HC", "CB",
                "HC", "CB", "HC", "CB", "CB", "CB", "CB", "CB"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUGBEL
     *
     * @cdk.inchi InChI=1S/C12H14N2O2/c1-2-16-12-10(7-13)8-5-3-4-6-9(8)11(15)14-12/h2-6H2,1H3,(H,14,15)
     */
    @Test
    public void testCUGBEL() {
        IAtomContainer container = new AtomContainer(30, 31, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 7, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 15, DOUBLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 25, SINGLE);
        container.addBond(11, 26, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(12, 29, SINGLE);
        container.addBond(13, 14, TRIPLE);

        String[] expected = {"C=C", "C=C", "C=C", "CR", "CR", "CR", "CR", "C=C", "C=ON", "NC=O", "OC=C", "CR", "CR",
                "CSP", "NSP", "O=CN", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCO", "HC", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUGGOA
     *
     * @cdk.inchi InChI=1S/C7H6Cl5NO4/c1-6(10,5(14)15)2-3(8)4(9)7(11,12)13(16)17/h2,4H,1H3,(H,14,15)/b3-2+/t4-,6-/m1/s1
     */
    @Test
    public void testCUGGOA() {
        IAtomContainer container = new AtomContainer(23, 22, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 10, DOUBLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(16, 22, SINGLE);
        container.addBond(16, 21, SINGLE);
        container.addBond(16, 20, SINGLE);

        String[] expected = {"CL", "CL", "CL", "CL", "CL", "O=CO", "OC=O", "O2N", "O2N", "NO2", "COO", "CR", "C=C",
                "C=C", "CR", "CR", "CR", "HOCO", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUGLOF
     *
     * @cdk.inchi InChI=1S/C10H9N3O6/c14-11(15)9-3-1-2-4-5(3)6(9)7(4)10(8(2)9,12(16)17)13(18)19/h2-8H,1H2/t2-,3+,4-,5+,6+,7-,8+,9-/m1/s1
     */
    @Test
    public void testCUGLOF() {
        IAtomContainer container = new AtomContainer(28, 32, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 19, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 21, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 22, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 23, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 24, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 25, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(9, 27, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 14, DOUBLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 16, DOUBLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 18, DOUBLE);

        String[] expected = {"CR4R", "CR", "CR", "CR4R", "CR4R", "CR4R", "CR4R", "CR4R", "CR", "CR", "NO2", "NO2",
                "NO2", "O2N", "O2N", "O2N", "O2N", "O2N", "O2N", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUJYUB10
     *
     * @cdk.inchi InChI=1S/C8H10N2OS/c1-10-6-4-3-5-7(10)8(9-11)12-2/h3-6H,1-2H3/p+1
     */
    @Test
    public void testCUJYUB10() {
        IAtomContainer container = new AtomContainer(23, 23, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 10, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 11, DOUBLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 20, SINGLE);

        String[] expected = {"S", "-O-", "CB", "CB", "CB", "CB", "CR", "CB", "C=N", "CR", "NPD+", "N=C", "HO", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CULGEV10
     *
     * @cdk.inchi InChI=1S/C8H9NO3S/c1-13(10,11)9-8(12-9)7-5-3-2-4-6-7/h2-6,8H,1H3/t8-,9-/m0/s1
     */
    @Test
    public void testCULGEV10() {
        IAtomContainer container = new AtomContainer(22, 23, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(12, 21, SINGLE);

        String[] expected = {"SO2N", "NSO2", "O2S", "O2S", "OR", "CR3R", "CB", "CB", "CB", "CB", "CB", "CB", "CR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CULHIA10
     *
     * @cdk.inchi InChI=1S/C12H24N2O/c15-14(10-5-2-6-11-14)12-9-13-7-3-1-4-8-13/h1-12H2
     */
    @Test
    public void testCULHIA10() {
        IAtomContainer container = new AtomContainer(39, 40, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(0, 16, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 19, SINGLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 21, SINGLE);
        container.addBond(3, 22, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 23, SINGLE);
        container.addBond(4, 24, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 25, SINGLE);
        container.addBond(6, 26, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 27, SINGLE);
        container.addBond(7, 28, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 29, SINGLE);
        container.addBond(9, 30, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 31, SINGLE);
        container.addBond(10, 32, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 33, SINGLE);
        container.addBond(11, 34, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 35, SINGLE);
        container.addBond(12, 36, SINGLE);
        container.addBond(13, 37, SINGLE);
        container.addBond(13, 38, SINGLE);

        String[] expected = {"CR", "CR", "CR", "CR", "CR", "NR", "CR", "CR", "N3OX", "CR", "CR", "CR", "CR", "CR",
                "OXN", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CULVEK
     *
     * @cdk.inchi InChI=1S/C8H16ClO3P/c1-6(2)7-8(3,4)5-11-13(9,10)12-7/h6-7H,5H2,1-4H3/t7-,13+/m0/s1
     */
    @Test
    public void testCULVEK() {
        IAtomContainer container = new AtomContainer(29, 29, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(11, 25, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(12, 28, SINGLE);

        String[] expected = {"PO3", "CL", "OPO2", "OP", "OPO2", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUNVAI
     *
     * @cdk.inchi InChI=1S/C10H10N4O5/c1-6(7(2)15)11-12-9-4-3-8(13(16)17)5-10(9)14(18)19/h3-5,12H,1-2H3/b11-6+
     */
    @Test
    public void testCUNVAI() {
        IAtomContainer container = new AtomContainer(29, 29, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 20, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 12, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(9, 14, DOUBLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(10, 16, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(17, 25, SINGLE);
        container.addBond(17, 24, SINGLE);
        container.addBond(17, 23, SINGLE);
        container.addBond(18, 28, SINGLE);
        container.addBond(18, 27, SINGLE);
        container.addBond(18, 26, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "C=N", "C=OR", "O=CR", "NO2", "NO2", "NC=C", "N=C",
                "O2N", "O2N", "O2N", "O2N", "CR", "CR", "HC", "HC", "HC", "HNCC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUNVEM
     *
     * @cdk.inchi InChI=1S/C10H14ClNO2S/c1-8(2,3)15-9(6-12)5-10(9,11)7(13)14-4/h5H2,1-4H3/t9-,10+/m1/s1
     */
    @Test
    public void testCUNVEM() {
        IAtomContainer container = new AtomContainer(29, 29, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 9, DOUBLE);
        container.addBond(4, 8, TRIPLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 27, SINGLE);
        container.addBond(7, 28, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 26, SINGLE);

        String[] expected = {"CL", "S", "OC=O", "O=CO", "NSP", "CR3R", "CR3R", "CR3R", "CSP", "COO", "CR", "CR", "CR",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CURZIY
     *
     * @cdk.inchi InChI=1S/C8H12O3S/c1-2-5-4-3-6(12(5,9)10)8-7(4)11-8/h4-8H,2-3H2,1H3/t4-,5-,6-,7-,8+/m1/s1
     */
    @Test
    public void testCURZIY() {
        IAtomContainer container = new AtomContainer(24, 26, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(11, 23, SINGLE);

        String[] expected = {"SO2", "OR", "O2S", "O2S", "CR", "CR", "CR", "CR3R", "CR3R", "CR", "CR", "CR", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUVFOO
     *
     * @cdk.inchi InChI=1S/C6H11NO4S2/c7-4(6(10)11)3-13-12-2-1-5(8)9/h4H,1-3,7H2,(H,8,9)(H,10,11)/t4-/m0/s1
     */
    @Test
    public void testCUVFOO() {
        IAtomContainer container = new AtomContainer(24, 23, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(12, 23, SINGLE);

        String[] expected = {"O2CM", "O2CM", "CO2M", "CR", "NR+", "CR", "S", "S", "CR", "CR", "COO", "O=CO", "OC=O",
                "HC", "HNR+", "HNR+", "HNR+", "HC", "HC", "HC", "HC", "HC", "HC", "HOCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUVGAB
     *
     * @cdk.inchi InChI=1S/C6H13O4P/c1-5(7)3-4-6(2,8)11(5,9)10/h7-8H,3-4H2,1-2H3,(H,9,10)/t5-,6-/m0/s1
     */
    @Test
    public void testCUVGAB() {
        IAtomContainer container = new AtomContainer(24, 24, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(10, 21, SINGLE);

        String[] expected = {"PO2", "OPO", "OP", "OR", "OR", "CR", "CR", "CR", "CR", "CR", "CR", "HOP", "HOR", "HOR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUVJOS
     *
     * @cdk.inchi InChI=1S/CH6N2/c1-3-2/h3H,2H2,1H3/p+1
     */
    @Test
    public void testCUVJOS() {
        IAtomContainer container = new AtomContainer(10, 9, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 9, SINGLE);

        String[] expected = {"NR", "NR+", "CR", "HNR", "HNR", "HNR+", "HNR+", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CUYRAP
     *
     * @cdk.inchi InChI=1S/C12H12O4/c1-7(13)10-6-9-11(15-3)4-8(14-2)5-12(9)16-10/h4-6H,1-3H3
     */
    @Test
    public void testCUYRAP() {
        IAtomContainer container = new AtomContainer(28, 29, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(1, 12, DOUBLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(13, 20, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(14, 21, SINGLE);
        container.addBond(14, 22, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(15, 23, SINGLE);
        container.addBond(15, 24, SINGLE);
        container.addBond(15, 27, SINGLE);

        String[] expected = {"OFUR", "O=CR", "OC=C", "OC=C", "C5A", "C5B", "CB", "CB", "CB", "CB", "C5A", "C5B",
                "C=OR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CYANAM01
     *
     * @cdk.inchi InChI=1S/CH2N2/c2-1-3/h2H2
     */
    @Test
    public void testCYANAM01() {
        IAtomContainer container = new AtomContainer(5, 4, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, TRIPLE);

        String[] expected = {"NC%N", "NSP", "CSP", "HNC%", "HNC%"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for CYGUAN01
     *
     * @cdk.inchi InChI=1S/C11H14ClN5/c1-11(2)16-9(13)15-10(14)17(11)8-5-3-7(12)4-6-8/h3-6H,1-2H3,(H4,13,14,15,16)/p+1
     */
    @Test
    public void testCYGUAN01() {
        IAtomContainer container = new AtomContainer(32, 33, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 14, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 23, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(10, 27, SINGLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(11, 16, DOUBLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 29, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(15, 30, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 31, SINGLE);

        String[] expected = {"CL", "NCN+", "CR", "NC=N", "CGD", "N=C", "CNN+", "CR", "CR", "NC=N", "NCN+", "CB", "CB",
                "CB", "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC", "HC", "HNCN", "HNCN", "HNCN", "HNN+", "HNN+",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DABHAP
     *
     * @cdk.inchi InChI=1S/C12H14N4O3S/c1-8-14-11(7-12(15-8)19-2)16-20(17,18)10-5-3-9(13)4-6-10/h3-7H,13H2,1-2H3,(H,14,15,16)
     */
    @Test
    public void testDABHAP() {
        IAtomContainer container = new AtomContainer(34, 35, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 21, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(5, 23, SINGLE);
        container.addBond(6, 24, SINGLE);
        container.addBond(6, 25, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 9, DOUBLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 19, SINGLE);
        container.addBond(16, 27, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(18, 28, SINGLE);
        container.addBond(18, 29, SINGLE);
        container.addBond(18, 30, SINGLE);
        container.addBond(19, 31, SINGLE);
        container.addBond(19, 32, SINGLE);
        container.addBond(19, 33, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "NC=C", "SO2N", "O2S", "O2S", "N=C", "C=N", "C=C",
                "C=C", "N=C", "C=N", "NC=N", "OC=C", "CR", "CR", "HC", "HC", "HC", "HC", "HNCC", "HNCC", "HC", "HNCN",
                "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DABLIB
     *
     * @cdk.inchi InChI=1S/C8H8F4/c9-7(10)3-1-4-6(2-5(3)7)8(4,11)12/h3-6H,1-2H2/t3-,4-,5+,6+
     */
    @Test
    public void testDABLIB() {
        IAtomContainer container = new AtomContainer(20, 22, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 19, SINGLE);

        String[] expected = {"CR3R", "CR", "CR3R", "F", "F", "HC", "HC", "CR3R", "CR3R", "HC", "CR", "HC", "CR3R",
                "CR3R", "HC", "HC", "HC", "HC", "F", "F"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DACSAB
     *
     * @cdk.inchi InChI=1S/C11H15FNO2P/c1-10(11-6-3-2-4-7-11)13-8-5-9-15-16(13,12)14/h2-4,6-7,10H,5,8-9H2,1H3/t10-,16+/m0/s1
     */
    @Test
    public void testDACSAB() {
        IAtomContainer container = new AtomContainer(31, 32, 0, 0);
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(6, 21, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 26, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 29, SINGLE);
        container.addBond(15, 30, SINGLE);

        String[] expected = {"F", "OP", "OPO", "PO2", "NR", "CR", "CR", "CR", "CR", "CR", "CB", "CB", "CB", "CB", "CB",
                "CB", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DACYIP
     *
     * @cdk.inchi InChI=1S/C10H22NPS/c1-8-9(2,3)12(13,11(6)7)10(8,4)5/h8H,1-7H3/t8-,12-
     */
    @Test
    public void testDACYIP() {
        IAtomContainer container = new AtomContainer(35, 35, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 21, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 22, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 21, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(21, 24, SINGLE);
        container.addBond(21, 25, SINGLE);
        container.addBond(22, 26, SINGLE);
        container.addBond(22, 27, SINGLE);
        container.addBond(22, 28, SINGLE);
        container.addBond(24, 29, SINGLE);
        container.addBond(24, 30, SINGLE);
        container.addBond(24, 31, SINGLE);
        container.addBond(25, 32, SINGLE);
        container.addBond(25, 33, SINGLE);
        container.addBond(25, 34, SINGLE);

        String[] expected = {"S-P", "PTET", "NR", "CR4R", "CR4R", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "CR4R", "CR", "HC", "CR", "CR", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DADDAN
     *
     * @cdk.inchi InChI=1S/C12H15NO4/c14-10-1-2-11(15)17-9-4-6-13-5-3-8(7-16-10)12(9)13/h3,9,12H,1-2,4-7H2/t9-,12-/m1/s1
     */
    @Test
    public void testDADDAN() {
        IAtomContainer container = new AtomContainer(32, 34, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 12, SINGLE);
        container.addBond(0, 13, SINGLE);
        container.addBond(1, 13, DOUBLE);
        container.addBond(2, 16, DOUBLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 28, SINGLE);
        container.addBond(14, 29, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 30, SINGLE);
        container.addBond(15, 31, SINGLE);

        String[] expected = {"OC=O", "O=CO", "O=CO", "OC=O", "NR", "C=C", "C=C", "CR", "CR", "CR", "CR", "CR", "CR",
                "COO", "CR", "CR", "COO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DADLAV
     *
     * @cdk.inchi InChI=1S/C7H8N2O2/c1-3-5(4(2)10)6(8)7(11)9-3/h1,8H2,2H3,(H,9,11)
     */
    @Test
    public void testDADLAV() {
        IAtomContainer container = new AtomContainer(19, 19, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, DOUBLE);
        container.addBond(1, 8, DOUBLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 10, DOUBLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 15, SINGLE);

        String[] expected = {"O=CN", "O=CR", "NC=O", "NC=C", "C=ON", "C=C", "C=C", "C=C", "C=OR", "CR", "C=C", "HNCO",
                "HNCC", "HNCC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DADLEZ
     *
     * @cdk.inchi InChI=1S/C7H8N2O2/c1-3(10)5-4(2)11-7(9)6(5)8/h9H,2,8H2,1H3/b9-7-
     */
    @Test
    public void testDADLEZ() {
        IAtomContainer container = new AtomContainer(19, 19, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 9, DOUBLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 11, SINGLE);

        String[] expected = {"OC=C", "C=N", "C=C", "C=C", "C=C", "N=C", "NC=C", "C=OR", "O=CR", "C=C", "CR", "HC",
                "HC", "HC", "HNCC", "HNCC", "HC", "HC", "HN=C"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAFKIE
     *
     * @cdk.inchi InChI=1S/C8H11N3/c1-7-3-5-8(6-4-7)10-11-9-2/h3-6H,1-2H3,(H,9,10)
     */
    @Test
    public void testDAFKIE() {
        IAtomContainer container = new AtomContainer(22, 22, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 21, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "CR", "CR", "N=N", "N=N", "NN=N", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNNN"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAFPUV
     *
     * @cdk.inchi InChI=1S/C2H2N4O3/c3-1-2(6(7)8)5-9-4-1/h(H2,3,4)
     */
    @Test
    public void testDAFPUV() {
        IAtomContainer container = new AtomContainer(11, 11, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(4, 8, DOUBLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 8, SINGLE);

        String[] expected = {"OFUR", "O2N", "O2N", "N5A", "N5A", "NO2", "NC=N", "C5B", "C5B", "HNCN", "HNCN"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAGTUA
     *
     * @cdk.inchi InChI=1S/C4H12N2O6S2/c1-11-5(13(3,7)8)6(12-2)14(4,9)10/h1-4H3
     */
    @Test
    public void testDAGTUA() {
        IAtomContainer container = new AtomContainer(26, 25, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 6, DOUBLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(1, 7, DOUBLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(13, 25, SINGLE);

        String[] expected = {"SO2N", "SO2N", "OR", "O2S", "OR", "O2S", "O2S", "O2S", "NSO2", "NSO2", "CR", "CR", "CR",
                "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAHBAP
     *
     * @cdk.inchi InChI=1S/C8H10N2O3/c1-9(2)8(12)6-4-3-5-7(11)10(6)13/h3-5,13H,1-2H3
     */
    @Test
    public void testDAHBAP() {
        IAtomContainer container = new AtomContainer(23, 23, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 16, SINGLE);
        container.addBond(1, 9, DOUBLE);
        container.addBond(2, 10, DOUBLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);

        String[] expected = {"-O-", "O=CN", "O=CN", "NC=O", "NC=O", "C=C", "C=C", "C=C", "C=C", "C=ON", "C=ON", "CR",
                "CR", "HC", "HC", "HC", "HO", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAHNAB
     *
     * @cdk.inchi InChI=1S/C6H16Cl2Si2/c1-9(2,7)5-6-10(3,4)8/h5-6H2,1-4H3
     */
    @Test
    public void testDAHNAB() {
        IAtomContainer container = new AtomContainer(26, 25, 0, 0);
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(13, 16, SINGLE);
        container.addBond(14, 17, SINGLE);
        container.addBond(14, 18, SINGLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(18, 20, SINGLE);
        container.addBond(18, 21, SINGLE);
        container.addBond(18, 22, SINGLE);
        container.addBond(19, 23, SINGLE);
        container.addBond(19, 24, SINGLE);
        container.addBond(19, 25, SINGLE);

        String[] expected = {"SI", "CL", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "CR", "SI",
                "HC", "HC", "CL", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAJXER
     *
     * @cdk.inchi InChI=1S/C11H14N4O2/c1-11-7-6-8(11)15(10(17)13-9(11)16)4-2-3-14(6)5-12-7/h5-8H,2-4H2,1H3,(H,13,16,17)/t6-,7+,8-,11+/m0/s1
     */
    @Test
    public void testDAJXER() {
        IAtomContainer container = new AtomContainer(31, 34, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(16, 29, SINGLE);
        container.addBond(16, 30, SINGLE);

        String[] expected = {"NC=O", "CONN", "O=CN", "NC=O", "C=ON", "O=CN", "CR4R", "CR4R", "CR", "NC=N", "C=N",
                "N=C", "CR4R", "CR4R", "CR", "CR", "CR", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAKBAS
     *
     * @cdk.inchi InChI=1S/C15H13N3O3S/c1-8-13(9(2)19)14(11(7-16)15(22)17-8)10-5-3-4-6-12(10)18(20)21/h3-6,14,17,22H,1-2H3/p-1/t14-/m1/s1
     */
    @Test
    public void testDAKBAS() {
        IAtomContainer container = new AtomContainer(34, 35, 0, 0);
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 20, DOUBLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 6, DOUBLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 12, TRIPLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(7, 11, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(13, 26, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 28, SINGLE);
        container.addBond(16, 17, DOUBLE);
        container.addBond(17, 29, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(18, 30, SINGLE);
        container.addBond(18, 19, DOUBLE);
        container.addBond(20, 21, SINGLE);
        container.addBond(21, 33, SINGLE);
        container.addBond(21, 32, SINGLE);
        container.addBond(21, 31, SINGLE);

        String[] expected = {"SM", "O=CR", "O2N", "O2N", "NC=C", "NSP", "NO2", "C=C", "C=C", "C=C", "CR", "C=C", "CSP",
                "CR", "CB", "CB", "CB", "CB", "CB", "CB", "C=OR", "CR", "HNCC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAKCEX
     *
     * @cdk.inchi InChI=1S/C8H4N8/c9-13-10-7-5-3-1-2-4-6(5)8-11-14-15-16(8)12-7/h1-4H
     */
    @Test
    public void testDAKCEX() {
        IAtomContainer container = new AtomContainer(20, 22, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", -1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 9, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 12, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 15, DOUBLE);

        String[] expected = {"N5B", "N5B", "N5A", "NPYL", "N=C", "C=N", "CB", "CB", "C5A", "CB", "CB", "CB", "CB",
                "N=N", "=N=", "NAZT", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAKDOI
     *
     * @cdk.inchi InChI=1S/C16H11ClN4O3S/c17-8-1-3-10-11(5-8)13(7-15(22)23)21(19-16(10)24)9-2-4-14-12(6-9)18-20-25-14/h1-6,13H,7H2,(H,19,24)(H,22,23)/t13-/m0/s1
     */
    @Test
    public void testDAKDOI() {
        IAtomContainer container = new AtomContainer(36, 39, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 17, SINGLE);
        container.addBond(0, 21, DOUBLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 25, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 26, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 27, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 28, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 29, SINGLE);
        container.addBond(8, 30, SINGLE);
        container.addBond(9, 22, DOUBLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 15, DOUBLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 31, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 32, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(15, 33, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(17, 34, SINGLE);
        container.addBond(19, 20, DOUBLE);
        container.addBond(20, 24, SINGLE);
        container.addBond(23, 35, SINGLE);

        String[] expected = {"C=ON", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "CR", "COO", "CB", "CB", "CB", "C5A",
                "C5B", "CB", "CL", "NC=O", "NC=C", "N5B", "N5A", "O=CN", "O=CO", "OC=O", "STHI", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HNCO", "HOCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DANCUQ
     *
     * @cdk.inchi InChI=1S/C4H8S4Si/c1-2-6-9(5-1)7-3-4-8-9/h1-4H2
     */
    @Test
    public void testDANCUQ() {
        IAtomContainer container = new AtomContainer(17, 18, 0, 0);
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(12, 16, SINGLE);

        String[] expected = {"SI", "S", "S", "CR", "CR", "HC", "HC", "HC", "HC", "S", "S", "CR", "CR", "HC", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAPSUO03
     *
     * @cdk.inchi InChI=1S/C12H12N2O2S/c13-9-1-5-11(6-2-9)17(15,16)12-7-3-10(14)4-8-12/h1-8H,13-14H2
     */
    @Test
    public void testDAPSUO03() {
        IAtomContainer container = new AtomContainer(29, 30, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 11, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(3, 26, SINGLE);
        container.addBond(3, 25, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 28, SINGLE);
        container.addBond(4, 27, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 10, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 23, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(16, 24, SINGLE);

        String[] expected = {"SO2", "O2S", "O2S", "NC=C", "NC=C", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB",
                "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCC", "HNCC", "HNCC", "HNCC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DARDEF
     *
     * @cdk.inchi InChI=1S/C10H14N4S2/c1-6-8(13-5-12-6)9-14-7(3-15-9)4-16-10(14)11-2/h5,7,9H,3-4H2,1-2H3,(H,12,13)/b11-10-/t7-,9-/m0/s1
     */
    @Test
    public void testDARDEF() {
        IAtomContainer container = new AtomContainer(30, 32, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 16, SINGLE);
        container.addBond(0, 17, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 8, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 26, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(15, 29, SINGLE);

        String[] expected = {"CR", "S", "C=N", "NC=N", "CR", "S", "CR", "CR", "N=C", "CR", "C5B", "C5A", "NPYL", "C5A",
                "N5B", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HPYL", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DARPOB10
     *
     * @cdk.inchi InChI=1S/C10H25N2PSi/c1-8-12(9-2)10(11(3)4)13-14(5,6)7/h8-9H2,1-7H3
     */
    @Test
    public void testDARPOB10() {
        IAtomContainer container = new AtomContainer(39, 38, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(6, 21, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(9, 38, SINGLE);
        container.addBond(9, 27, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(10, 29, SINGLE);
        container.addBond(10, 28, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 32, SINGLE);
        container.addBond(11, 31, SINGLE);
        container.addBond(11, 30, SINGLE);
        container.addBond(12, 34, SINGLE);
        container.addBond(12, 33, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 37, SINGLE);
        container.addBond(13, 36, SINGLE);
        container.addBond(13, 35, SINGLE);

        String[] expected = {"-P=C", "SI", "NC=P", "NC=P", "CR", "CR", "CR", "C=P", "CR", "CR", "CR", "CR", "CR", "CR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DARXID
     *
     * @cdk.inchi InChI=1S/C11H12N2S/c1-9-8-11(14)13(12(9)2)10-6-4-3-5-7-10/h3-8H,1-2H3
     */
    @Test
    public void testDARXID() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(13, 25, SINGLE);

        String[] expected = {"S=C", "NC=S", "NC=C", "C=SN", "C=C", "C=C", "CR", "CR", "CB", "CB", "CB", "CB", "CB",
                "CB", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DARZEB
     *
     * @cdk.inchi InChI=1S/C9H11N3O/c1-2-12-9(13)7-5-3-4-6-8(7)10-11-12/h3-6,9,13H,2H2,1H3/t9-/m0/s1
     */
    @Test
    public void testDARZEB() {
        IAtomContainer container = new AtomContainer(24, 25, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 21, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(12, 19, SINGLE);

        String[] expected = {"N=N", "N=N", "NN=N", "CR", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "CR", "OR", "HC",
                "HC", "HC", "HC", "HC", "HC", "HOR", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAVWEC
     *
     * @cdk.inchi InChI=1S/C9H9Br2N3O7/c1-7(12(16)17)4(10)5(11)8(2,13(18)19)9(3,6(7)15)14(20)21/h1-3H3/t7-,8+,9-/m1/s1
     */
    @Test
    public void testDAVWEC() {
        IAtomContainer container = new AtomContainer(30, 30, 0, 0);
        container.addAtom(newAtm("Br", 0));
        container.addAtom(newAtm("Br", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 14, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(10, 27, SINGLE);
        container.addBond(10, 28, SINGLE);
        container.addBond(10, 29, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 16, DOUBLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 18, DOUBLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(13, 20, DOUBLE);

        String[] expected = {"BR", "BR", "C=OR", "CR", "C=C", "C=C", "CR", "CR", "CR", "CR", "CR", "NO2", "NO2", "NO2",
                "O=CR", "O2N", "O2N", "O2N", "O2N", "O2N", "O2N", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAVXED
     *
     * @cdk.inchi InChI=1S/C10H12N4O9/c1-5-6(11(16)17)9(3,13(20)21)10(4,14(22)23)7(15)8(5,2)12(18)19/h1-4H3/t8-,9+,10+/m0/s1
     */
    @Test
    public void testDAVXED() {
        IAtomContainer container = new AtomContainer(35, 35, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 14, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 32, SINGLE);
        container.addBond(6, 33, SINGLE);
        container.addBond(6, 34, SINGLE);
        container.addBond(7, 29, SINGLE);
        container.addBond(7, 30, SINGLE);
        container.addBond(7, 31, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(9, 27, SINGLE);
        container.addBond(9, 28, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 16, DOUBLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 18, DOUBLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(12, 20, DOUBLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(13, 22, DOUBLE);

        String[] expected = {"C=OR", "CR", "C=C", "C=C", "CR", "CR", "CR", "CR", "CR", "CR", "NO2", "NO2", "NO2",
                "NO2", "O=CR", "O2N", "O2N", "O2N", "O2N", "O2N", "O2N", "O2N", "O2N", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAWXII
     *
     * @cdk.inchi InChI=1S/C6H7F4N3O/c1-11-3-4(13(14)12-2)6(9,10)5(3,7)8/h11H,1-2H3/b13-12-
     */
    @Test
    public void testDAWXII() {
        IAtomContainer container = new AtomContainer(21, 21, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 18, SINGLE);

        String[] expected = {"CE4R", "CE4R", "CR4R", "CR4R", "NC=C", "CR", "N2OX", "N=N", "CR", "OXN", "F", "F", "F",
                "F", "HNCC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAWYUV
     *
     * @cdk.inchi InChI=1S/C9H5ClO2/c10-7-5-9(11)12-8-4-2-1-3-6(7)8/h1-5H
     */
    @Test
    public void testDAWYUV() {
        IAtomContainer container = new AtomContainer(17, 18, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 11, DOUBLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 10, DOUBLE);

        String[] expected = {"CL", "OC=O", "COO", "C=C", "C=C", "CB", "CB", "CB", "CB", "CB", "CB", "O=CO", "HC", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAYWEF
     *
     * @cdk.inchi InChI=1S/C6H12O7P2/c7-14(9-3-1-4-10-14)13-15(8)11-5-2-6-12-15/h1-6H2
     */
    @Test
    public void testDAYWEF() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 11, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 26, SINGLE);

        String[] expected = {"PO4", "PO4", "OPO3", "OPO3", "OPO3", "OP", "CR", "CR", "CR", "OPO3", "OPO3", "OP", "CR",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DAZVEF
     *
     * @cdk.inchi InChI=1S/C12H12O3/c13-7-10(1-2-10)8(14)12(5-6-12)9(15)11(7)3-4-11/h1-6H2
     */
    @Test
    public void testDAZVEF() {
        IAtomContainer container = new AtomContainer(27, 30, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 26, SINGLE);

        String[] expected = {"C=OR", "O=CR", "CR3R", "CR3R", "CR3R", "C=OR", "O=CR", "CR3R", "CR3R", "CR3R", "C=OR",
                "O=CR", "CR3R", "CR3R", "CR3R", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEBMOM01
     *
     * @cdk.inchi InChI=1S/C4H4O4/c5-3(6)1-2-4(7)8/h1-2H,(H,5,6)(H,7,8)/p-1/b2-1-
     */
    @Test
    public void testDEBMOM01() {
        IAtomContainer container = new AtomContainer(11, 10, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, DOUBLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 10, SINGLE);

        String[] expected = {"O=CO", "OC=O", "O2CM", "O2CM", "COO", "C=C", "C=C", "CO2M", "HOCO", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DECJAW
     *
     * @cdk.inchi InChI=1S/C7H8N2O4/c10-6-3-1-2-8-4(7(11)12)5(3)13-9-6/h4,8H,1-2H2,(H,9,10)(H,11,12)/t4-/m0/s1
     */
    @Test
    public void testDECJAW() {
        IAtomContainer container = new AtomContainer(21, 22, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 8, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(12, 20, SINGLE);

        String[] expected = {"OFUR", "N5A", "C5B", "C5B", "CR", "CR", "NR+", "CR", "C5A", "CO2M", "O2CM", "O2CM",
                "OC=N", "HC", "HC", "HC", "HC", "HNR+", "HNR+", "HC", "HOCN"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DECKUR
     *
     * @cdk.inchi InChI=1S/C10H15N3O2/c1-12-6-7(9(11)14)10(15)13-5-3-2-4-8(12)13/h6,8H,2-5H2,1H3,(H2,11,14)/t8-/m1/s1
     */
    @Test
    public void testDECKUR() {
        IAtomContainer container = new AtomContainer(30, 31, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 13, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(14, 27, SINGLE);
        container.addBond(14, 28, SINGLE);
        container.addBond(14, 29, SINGLE);

        String[] expected = {"NC=C", "C=C", "C=C", "C=ON", "NC=O", "CR", "CR", "CR", "CR", "CR", "C=ON", "O=CN",
                "NC=O", "O=CN", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCO", "HNCO", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DECRIM
     *
     * @cdk.inchi InChI=1S/C10H13N2O/c1-11(2)10-12(3)8-6-4-5-7-9(8)13-10/h4-7H,1-3H3/q+1
     */
    @Test
    public void testDECRIM() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(10, 24, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);

        String[] expected = {"OFUR", "NIM+", "NCN+", "CIM+", "C5A", "CB", "CB", "CB", "CB", "C5B", "CR", "CR", "CR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEDCIY
     *
     * @cdk.inchi InChI=1S/C9H6Cl3NO/c1-9(12)5-2-4(10)3-6(11)7(5)13-8(9)14/h2-3H,1H3,(H,13,14)/t9-/m0/s1
     */
    @Test
    public void testDEDCIY() {
        IAtomContainer container = new AtomContainer(20, 21, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(3, 11, DOUBLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 19, SINGLE);

        String[] expected = {"CL", "CL", "CL", "O=CN", "NC=O", "CB", "CB", "CB", "CB", "CB", "CB", "C=ON", "CR", "CR",
                "HC", "HC", "HNCO", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEDSIO
     *
     * @cdk.inchi InChI=1S/C11H14INO3/c1-11(2,3)16-10(15)4-5-13-8(7-12)6-9(13)14/h8H,6-7H2,1-3H3/t8-/m1/s1
     */
    @Test
    public void testDEDSIO() {
        IAtomContainer container = new AtomContainer(30, 30, 0, 0);
        container.addAtom(newAtm("I", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(7, 8, TRIPLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(15, 29, SINGLE);

        String[] expected = {"I", "NC=O", "O=CN", "C=ON", "CR4R", "CR4R", "CR", "CSP", "CSP", "COO", "O=CO", "OC=O",
                "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEFGIE
     *
     * @cdk.inchi InChI=1S/C8H10N4O2/c1-12(14)11-10-7-4-2-6(3-5-7)8(9)13/h2-5,10H,1H3,(H2,9,13)/b12-11-
     */
    @Test
    public void testDEFGIE() {
        IAtomContainer container = new AtomContainer(24, 24, 0, 0);
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 13, DOUBLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(12, 21, SINGLE);

        String[] expected = {"OXN", "O=CN", "NC=C", "N=N", "N2OX", "NC=O", "CB", "CB", "CB", "CB", "CB", "CB", "CR",
                "C=ON", "HNCC", "HC", "HC", "HNCO", "HNCO", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEFLEF
     *
     * @cdk.inchi InChI=1S/C8H8N6O8/c1-3(15)9-5-6(12(7(9)17)14(21)22)10(4(2)16)8(18)11(5)13(19)20/h5-6H,1-2H3/t5-,6-/m1/s1
     */
    @Test
    public void testDEFLEF() {
        IAtomContainer container = new AtomContainer(30, 31, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 23, SINGLE);
        container.addBond(0, 18, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 4, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 10, DOUBLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 29, SINGLE);
        container.addBond(9, 28, SINGLE);
        container.addBond(9, 27, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 15, DOUBLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(16, 18, SINGLE);
        container.addBond(16, 17, DOUBLE);
        container.addBond(18, 19, SINGLE);
        container.addBond(19, 21, DOUBLE);
        container.addBond(19, 20, SINGLE);
        container.addBond(20, 26, SINGLE);
        container.addBond(20, 25, SINGLE);
        container.addBond(20, 24, SINGLE);

        String[] expected = {"CR", "NC=O", "NO2", "O2N", "O2N", "CONN", "O=CN", "NC=O", "C=ON", "CR", "O=CN", "CR",
                "NC=O", "NO2", "O2N", "O2N", "CONN", "O=CN", "NC=O", "C=ON", "CR", "O=CN", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEFPUZ
     *
     * @cdk.inchi InChI=1S/C4H8N4S/c1-6-4-7-9-3(5)8(4)2/h5H,1-2H3,(H,6,7)/p+1
     */
    @Test
    public void testDEFPUZ() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 15, SINGLE);

        String[] expected = {"STHI", "N5A", "NIM+", "NC=N", "NCN+", "C5B", "CIM+", "CR", "CR", "HNCN", "HNN+", "HNN+",
                "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEFTUD
     *
     * @cdk.inchi InChI=1S/C10H11N5O/c1-14(2)9-11-10(16)15(13-12-9)8-6-4-3-5-7-8/h3-7H,1-2H3
     */
    @Test
    public void testDEFTUD() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 23, SINGLE);
        container.addBond(5, 22, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 26, SINGLE);
        container.addBond(6, 25, SINGLE);
        container.addBond(6, 24, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(15, 20, SINGLE);

        String[] expected = {"NC=O", "N=N", "N=N", "C=N", "NC=N", "CR", "CR", "N=C", "CONN", "O=CN", "CB", "CB", "CB",
                "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEFVAL
     *
     * @cdk.inchi InChI=1S/C11H13N5O/c1-8-4-6-9(7-5-8)13-16-11(17)12-10(14-16)15(2)3/h4-7H,1-3H3/p+1
     */
    @Test
    public void testDEFVAL() {
        IAtomContainer container = new AtomContainer(31, 32, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 26, SINGLE);
        container.addBond(7, 25, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(8, 29, SINGLE);
        container.addBond(8, 28, SINGLE);
        container.addBond(8, 27, SINGLE);
        container.addBond(9, 30, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 15, DOUBLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 16, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 20, SINGLE);
        container.addBond(16, 23, SINGLE);
        container.addBond(16, 22, SINGLE);
        container.addBond(16, 21, SINGLE);

        String[] expected = {"N=N", "N+=N", "CONN", "O=CN", "N=C", "C=N", "NC=N", "CR", "CR", "NC=C", "CB", "CB", "CB",
                "CB", "CB", "CB", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HNCC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEFYUI
     *
     * @cdk.inchi InChI=1S/C6H9NS2/c7-5-3-1-2-4(5)6(8)9/h1-3,7H2,(H,8,9)
     */
    @Test
    public void testDEFYUI() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 17, SINGLE);

        String[] expected = {"S=C", "S", "NC=C", "CSS", "C=C", "C=C", "CR", "CR", "CR", "HS", "HNCC", "HNCC", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEGLUW
     *
     * @cdk.inchi InChI=1S/C3H7ClN2/c1-6(2)3(4)5/h5H,1-2H3/p+1
     */
    @Test
    public void testDEGLUW() {
        IAtomContainer container = new AtomContainer(14, 13, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(5, 11, SINGLE);

        String[] expected = {"CL", "NCN+", "NCN+", "CNN+", "CR", "CR", "HNN+", "HNN+", "HC", "HC", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEGRIQ
     *
     * @cdk.inchi InChI=1S/C17H18N2/c1-12-3-5-16-14(7-12)9-18-11-19(16)10-15-8-13(2)4-6-17(15)18/h3-8H,9-11H2,1-2H3
     */
    @Test
    public void testDEGRIQ() {
        IAtomContainer container = new AtomContainer(37, 40, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 13, SINGLE);
        container.addBond(0, 19, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 21, SINGLE);
        container.addBond(4, 13, DOUBLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 22, SINGLE);
        container.addBond(5, 23, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(14, 17, SINGLE);
        container.addBond(14, 18, SINGLE);
        container.addBond(14, 29, SINGLE);
        container.addBond(14, 30, SINGLE);
        container.addBond(15, 31, SINGLE);
        container.addBond(15, 32, SINGLE);
        container.addBond(15, 33, SINGLE);
        container.addBond(16, 34, SINGLE);
        container.addBond(16, 35, SINGLE);
        container.addBond(16, 36, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CR", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "CB", "CR",
                "CR", "CR", "NC=C", "NC=C", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEKRUG
     *
     * @cdk.inchi InChI=1S/C6H5N3O5/c1-9-4(12)7-2(10)6(9)3(11)8-5(13)14-6/h1H3,(H2,7,8,10,11,12,13)/p-1/t6-/m1/s1
     */
    @Test
    public void testDEKRUG() {
        IAtomContainer container = new AtomContainer(18, 19, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 12, DOUBLE);

        String[] expected = {"NC=O", "CR", "CONN", "O=CN", "NC=O", "C=ON", "O=CN", "CR", "C=ON", "O=CN", "NM", "C=ON",
                "O=CN", "OC=O", "HC", "HC", "HC", "HNCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEMBIG
     *
     * @cdk.inchi InChI=1S/C7H7ClN6O2/c8-1-2-14-7(16)13-3-10-4(5(9)15)6(13)11-12-14/h3H,1-2H2,(H2,9,15)
     */
    @Test
    public void testDEMBIG() {
        IAtomContainer container = new AtomContainer(23, 24, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 12, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 15, DOUBLE);
        container.addBond(14, 21, SINGLE);
        container.addBond(14, 22, SINGLE);

        String[] expected = {"N=N", "N=N", "NC=O", "CONN", "NPYL", "C5A", "N5B", "C5B", "C5A", "CR", "CR", "CL",
                "O=CN", "C=ON", "NC=O", "O=CN", "HC", "HC", "HC", "HC", "HC", "HNCO", "HNCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEPKEO
     *
     * @cdk.inchi InChI=1S/C9H5N3OS/c1-2-6-8(11-3-1)14-9-7(13-6)10-4-5-12-9/h1-5H
     */
    @Test
    public void testDEPKEO() {
        IAtomContainer container = new AtomContainer(19, 21, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 13, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(12, 13, DOUBLE);

        String[] expected = {"S", "CB", "NPYD", "CB", "CB", "NPYD", "CB", "OC=C", "CB", "CB", "CB", "CB", "NPYD", "CB",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DERZUV
     *
     * @cdk.inchi InChI=1S/C9H13F3N4O2/c1-8(2)6(15(3)4)16(7(18)13-8)14-5(17)9(10,11)12/h1-4H3,(H-,13,14,17,18)
     */
    @Test
    public void testDERZUV() {
        IAtomContainer container = new AtomContainer(31, 31, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(13, 20, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(14, 22, SINGLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 17, SINGLE);
        container.addBond(16, 25, SINGLE);
        container.addBond(16, 26, SINGLE);
        container.addBond(16, 27, SINGLE);
        container.addBond(17, 28, SINGLE);
        container.addBond(17, 29, SINGLE);
        container.addBond(17, 30, SINGLE);

        String[] expected = {"NCN+", "CONN", "O=CN", "NC=O", "CR", "CNN+", "N=C", "C=N", "OM2", "CR", "F", "F", "F",
                "CR", "CR", "NCN+", "CR", "CR", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DESWUT
     *
     * @cdk.inchi InChI=1S/C6H8N2O2S/c1-2-10-6(9)4-5(7)11-3-8-4/h3H,2,7H2,1H3
     */
    @Test
    public void testDESWUT() {
        IAtomContainer container = new AtomContainer(19, 19, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(10, 17, SINGLE);

        String[] expected = {"STHI", "C5A", "N5B", "C5B", "C5A", "NC=C", "COO", "O=CO", "OC=O", "CR", "CR", "HNCC",
                "HNCC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DESYOP
     *
     * @cdk.inchi InChI=1S/C3H4N2/c1-3(5)2-4/h1,5H2
     */
    @Test
    public void testDESYOP() {
        IAtomContainer container = new AtomContainer(9, 8, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, TRIPLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 8, SINGLE);

        String[] expected = {"NSP", "NC=C", "CSP", "C=C", "C=C", "HNCC", "HNCC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEWHOC
     *
     * @cdk.inchi InChI=1S/C12H10N2O5/c1-2-14-7-4-9-8(18-5-19-9)3-6(7)11(15)10(13-14)12(16)17/h3-4H,2,5H2,1H3,(H,16,17)
     */
    @Test
    public void testDEWHOC() {
        IAtomContainer container = new AtomContainer(29, 31, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 17, SINGLE);
        container.addBond(0, 16, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 8, DOUBLE);
        container.addBond(2, 19, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(4, 9, DOUBLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 16, DOUBLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(15, 23, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(17, 25, SINGLE);
        container.addBond(17, 24, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(18, 28, SINGLE);
        container.addBond(18, 27, SINGLE);
        container.addBond(18, 26, SINGLE);

        String[] expected = {"NC=C", "N=C", "OC=O", "O=CO", "O=CR", "OC=C", "OC=C", "COO", "C=N", "C=OR", "CB", "CB",
                "CB", "CR", "CB", "CB", "CB", "CR", "CR", "HOCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEWJEU
     *
     * @cdk.inchi InChI=1S/C10H22N2Si2/c1-13(2,3)11-7-9-12(10-8-11)14(4,5)6/h7-10H,1-6H3
     */
    @Test
    public void testDEWJEU() {
        IAtomContainer container = new AtomContainer(36, 36, 0, 0);
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 18, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 19, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(18, 20, SINGLE);
        container.addBond(18, 21, SINGLE);
        container.addBond(19, 20, SINGLE);
        container.addBond(19, 22, SINGLE);
        container.addBond(20, 23, SINGLE);
        container.addBond(23, 24, SINGLE);
        container.addBond(23, 25, SINGLE);
        container.addBond(23, 26, SINGLE);
        container.addBond(24, 27, SINGLE);
        container.addBond(24, 28, SINGLE);
        container.addBond(24, 29, SINGLE);
        container.addBond(25, 30, SINGLE);
        container.addBond(25, 31, SINGLE);
        container.addBond(25, 32, SINGLE);
        container.addBond(26, 33, SINGLE);
        container.addBond(26, 34, SINGLE);
        container.addBond(26, 35, SINGLE);

        String[] expected = {"SI", "NC=C", "C=C", "C=C", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "C=C", "C=C", "NC=C", "HC", "HC", "SI", "CR", "CR", "CR", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEXCIS
     *
     * @cdk.inchi InChI=1S/C11H13NO2/c1-7(13)9-10(12-11(9)14)8-5-3-2-4-6-8/h2-7,9-10,13H,1H3,(H,12,14)/t7-,9+,10-/m1/s1
     */
    @Test
    public void testDEXCIS() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 26, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 25, SINGLE);

        String[] expected = {"O=CN", "C=ON", "CR4R", "CR4R", "NC=O", "CR", "CR", "OR", "CB", "CB", "CB", "CB", "CB",
                "CB", "HC", "HC", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEXGIW
     *
     * @cdk.inchi InChI=1S/C9H9N3O3/c13-5-10-9(15)6-1-3-12-4-2-7(14)11-8(6)12/h1,3,5H,2,4H2,(H,11,14)(H,10,13,15)
     */
    @Test
    public void testDEXGIW() {
        IAtomContainer container = new AtomContainer(24, 25, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 23, SINGLE);

        String[] expected = {"NC=O", "C=ON", "O=CN", "CR", "CR", "NPYL", "C5A", "C5B", "C5B", "C5A", "C=ON", "O=CN",
                "NC=O", "C=ON", "O=CN", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HNCO", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEZDUH
     *
     * @cdk.inchi InChI=1S/C8H6F6/c9-5(10)1-3-4-2-6(11,12)7(3,13)8(4,5)14/h3-4H,1-2H2/t3-,4-,7+,8+/m1/s1
     */
    @Test
    public void testDEZDUH() {
        IAtomContainer container = new AtomContainer(20, 22, 0, 0);
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 19, SINGLE);

        String[] expected = {"F", "F", "F", "F", "F", "F", "CR4R", "CR4R", "CR", "CR", "CR4R", "CR4R", "CR", "CR",
                "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEZNIF
     *
     * @cdk.inchi InChI=1S/C11H13NO2S/c1-15(13,14)11-10-5-3-2-4-9(8-10)6-7-12-11/h2-7,9-10H,8H2,1H3/t9-,10-/m0/s1
     */
    @Test
    public void testDEZNIF() {
        IAtomContainer container = new AtomContainer(28, 29, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 14, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 26, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 27, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(14, 23, SINGLE);

        String[] expected = {"SO2", "O2S", "O2S", "N=C", "C=N", "CR", "C=C", "C=C", "C=C", "C=C", "CR", "C=C", "C=C",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DEZXEL
     *
     * @cdk.inchi InChI=1S/C8H5Cl2NO2/c9-5-2-1-3-7(10)11-6(5)4-13-8(11)12/h1-3H,4H2
     */
    @Test
    public void testDEZXEL() {
        IAtomContainer container = new AtomContainer(18, 19, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, DOUBLE);

        String[] expected = {"NC=O", "C=C", "C=C", "C=C", "C=C", "C=C", "C=C", "CR", "OC=O", "C=ON", "O=CN", "CL",
                "CL", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DHOADS01
     *
     * @cdk.inchi InChI=1S/C10H11N5O2/c11-9-8-10(13-4-12-9)15(5-14-8)7-2-1-6(3-16)17-7/h1-2,4-7,16H,3H2,(H2,11,12,13)/t6-,7+/m0/s1
     */
    @Test
    public void testDHOADS01() {
        IAtomContainer container = new AtomContainer(28, 30, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 11, DOUBLE);
        container.addBond(0, 17, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(3, 10, DOUBLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 13, DOUBLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 20, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(16, 27, SINGLE);

        String[] expected = {"CB", "C5A", "C5B", "CB", "C5A", "CR", "C=C", "C=C", "CR", "CR", "NPYD", "NPYD", "NC=N",
                "N5B", "NPYL", "OR", "OR", "HC", "HNCN", "HNCN", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DICKIJ
     *
     * @cdk.inchi InChI=1S/C16H12N2S/c1-2-4-14-12(3-1)9-13(10-18-7-6-17-11-18)15-5-8-19-16(14)15/h1-9,11H,10H2/p+1
     */
    @Test
    public void testDICKIJ() {
        IAtomContainer container = new AtomContainer(32, 35, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 12, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 21, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(5, 22, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(13, 27, SINGLE);
        container.addBond(13, 26, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 18, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(16, 29, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(17, 30, SINGLE);
        container.addBond(17, 18, DOUBLE);
        container.addBond(18, 31, SINGLE);

        String[] expected = {"STHI", "C5A", "C5B", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "C5B", "C5A", "CR",
                "NIM+", "CIM+", "NIM+", "C5", "C5", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HIM+",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DICPUA
     *
     * @cdk.inchi InChI=1S/C8H6N2O2/c11-10-8(6-9-12-10)7-4-2-1-3-5-7/h1-6H
     */
    @Test
    public void testDICPUA() {
        IAtomContainer container = new AtomContainer(18, 19, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, DOUBLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 9, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(9, 10, SINGLE);

        String[] expected = {"C5B", "C5B", "CB", "CB", "CB", "CB", "CB", "CB", "N5AX", "N5A", "OFUR", "OXN", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DICRAI
     *
     * @cdk.inchi InChI=1S/C8H6N2O2/c11-10-6-8(9-12-10)7-4-2-1-3-5-7/h1-6H
     */
    @Test
    public void testDICRAI() {
        IAtomContainer container = new AtomContainer(18, 19, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, DOUBLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 9, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 10, SINGLE);

        String[] expected = {"C5B", "C5B", "CB", "CB", "CB", "CB", "CB", "CB", "N5A", "N5AX", "OFUR", "OXN", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DICYIX
     *
     * @cdk.inchi InChI=1S/C5H15NSi/c1-6(2)7(3,4)5/h1-5H3
     */
    @Test
    public void testDICYIX() {
        IAtomContainer container = new AtomContainer(22, 21, 0, 0);
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(6, 21, SINGLE);

        String[] expected = {"SI", "NR", "CR", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DICYOD
     *
     * @cdk.inchi InChI=1S/C8H10Cl4N2O2/c1-3-14(4-2)7(10)13-6(16)8(11,12)5(9)15/h3-4H2,1-2H3/b13-7+
     */
    @Test
    public void testDICYOD() {
        IAtomContainer container = new AtomContainer(26, 25, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 8, DOUBLE);
        container.addBond(5, 10, DOUBLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 11, DOUBLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(13, 20, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 21, SINGLE);
        container.addBond(14, 22, SINGLE);
        container.addBond(15, 23, SINGLE);
        container.addBond(15, 24, SINGLE);
        container.addBond(15, 25, SINGLE);

        String[] expected = {"CL", "CL", "CL", "CL", "O=C", "O=CN", "N=C", "NC=N", "C=O", "CR", "C=ON", "C=N", "CR",
                "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIDYOE
     *
     * @cdk.inchi InChI=1S/C7H11NO5/c1-12-7(11)6(10)5(9)4-2-3-8-13-4/h3-6,9-10H,2H2,1H3/t4-,5-,6+/m1/s1
     */
    @Test
    public void testDIDYOE() {
        IAtomContainer container = new AtomContainer(24, 24, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(3, 11, DOUBLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(12, 23, SINGLE);

        String[] expected = {"OR", "OR", "OR", "O=CO", "OC=O", "N=C", "C=N", "CR", "CR", "CR", "CR", "COO", "CR",
                "HOR", "HOR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIFSIU
     *
     * @cdk.inchi InChI=1S/C12H8N2O5S/c15-13(16)9-5-1-3-7-11(9)20(19)12-8-4-2-6-10(12)14(17)18/h1-8H
     */
    @Test
    public void testDIFSIU() {
        IAtomContainer container = new AtomContainer(28, 29, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 6, DOUBLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 7, DOUBLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 17, DOUBLE);
        container.addBond(16, 24, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(17, 25, SINGLE);
        container.addBond(18, 19, DOUBLE);
        container.addBond(18, 26, SINGLE);
        container.addBond(19, 27, SINGLE);

        String[] expected = {"S=O", "O=S", "O2N", "O2N", "O2N", "O2N", "NO2", "NO2", "CB", "CB", "CB", "CB", "CB",
                "CB", "CB", "CB", "CB", "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIGCOL
     *
     * @cdk.inchi InChI=1S/C2H4N2S4/c5-1(6)3-4-2(7)8/h(H2,3,5,6)(H2,4,7,8)/p-2
     */
    @Test
    public void testDIGCOL() {
        IAtomContainer container = new AtomContainer(10, 9, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", -1));
        container.addBond(0, 3, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(6, 9, SINGLE);

        String[] expected = {"S2CM", "S2CM", "NC=S", "CS2M", "HNCS", "NC=S", "CS2M", "HNCS", "S2CM", "S2CM"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIGCUR
     *
     * @cdk.inchi InChI=1S/C6H10S6/c1-9-5(7)11-3-4-12-6(8)10-2/h3-4H2,1-2H3
     */
    @Test
    public void testDIGCUR() {
        IAtomContainer container = new AtomContainer(22, 21, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(15, 17, SINGLE);
        container.addBond(17, 18, SINGLE);
        container.addBond(18, 19, SINGLE);
        container.addBond(18, 20, SINGLE);
        container.addBond(18, 21, SINGLE);

        String[] expected = {"S=C", "S", "S", "CR", "CSS", "CR", "HC", "HC", "HC", "HC", "HC", "CR", "S", "HC", "HC",
                "CSS", "S=C", "S", "CR", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIGLEK
     *
     * @cdk.inchi InChI=1S/C17H14N2O7/c20-12-3-1-2-11(15(12)21)16(22)18-14-13(26-17(14)23)8-9-4-6-10(7-5-9)19(24)25/h1-7,13-14,20-21H,8H2,(H,18,22)/t13-,14+/m1/s1
     */
    @Test
    public void testDIGLEK() {
        IAtomContainer container = new AtomContainer(40, 42, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, DOUBLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 19, DOUBLE);
        container.addBond(3, 21, SINGLE);
        container.addBond(3, 39, SINGLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(4, 38, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 37, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 27, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(12, 29, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 30, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(15, 31, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(17, 18, DOUBLE);
        container.addBond(17, 32, SINGLE);
        container.addBond(18, 33, SINGLE);
        container.addBond(19, 20, SINGLE);
        container.addBond(20, 21, DOUBLE);
        container.addBond(20, 25, SINGLE);
        container.addBond(21, 22, SINGLE);
        container.addBond(22, 23, DOUBLE);
        container.addBond(23, 24, SINGLE);
        container.addBond(23, 34, SINGLE);
        container.addBond(24, 25, DOUBLE);
        container.addBond(24, 35, SINGLE);
        container.addBond(25, 36, SINGLE);

        String[] expected = {"O=CO", "OC=O", "O=CN", "OC=C", "OC=C", "O2N", "O2N", "NC=O", "NO2", "COO", "CR4R",
                "CR4R", "CR", "CB", "CB", "CB", "CB", "CB", "CB", "C=ON", "CB", "CB", "CB", "CB", "CB", "CB", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCO", "HOCC", "HOCC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIHTET
     *
     * @cdk.inchi InChI=1S/C9H8N2O2/c1-5-3-7(12)6-4-10-11(2)8(6)9(5)13/h3-4H,1-2H3
     */
    @Test
    public void testDIHTET() {
        IAtomContainer container = new AtomContainer(21, 22, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(1, 8, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(12, 17, SINGLE);

        String[] expected = {"O=CR", "O=CR", "NPYL", "N5A", "C5B", "C=OR", "C=C", "C=C", "C=OR", "C5A", "C5B", "CR",
                "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIKGAF
     *
     * @cdk.inchi InChI=1S/C7H10Cl3NO3S/c1-4-6(2,3)11(15(4,13)14)5(12)7(8,9)10/h4H,1-3H3/t4-/m1/s1
     */
    @Test
    public void testDIKGAF() {
        IAtomContainer container = new AtomContainer(25, 25, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 14, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(3, 5, DOUBLE);
        container.addBond(3, 6, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 13, DOUBLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(10, 24, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(13, 14, SINGLE);

        String[] expected = {"CL", "CL", "CL", "SO2N", "O=CN", "O2S", "O2S", "NSO2", "CR4R", "CR4R", "CR", "CR", "CR",
                "C=ON", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIKGEJ
     *
     * @cdk.inchi InChI=1S/C7H10Cl3NO3S/c1-4-6(2,3)14-5(7(8,9)10)11-15(4,12)13/h4H,1-3H3/t4-/m1/s1
     */
    @Test
    public void testDIKGEJ() {
        IAtomContainer container = new AtomContainer(25, 25, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 6, DOUBLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(7, 9, DOUBLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(13, 20, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 21, SINGLE);
        container.addBond(14, 23, SINGLE);

        String[] expected = {"SO2N", "CL", "CL", "CL", "OC=N", "O2S", "O2S", "N=C", "CR", "C=N", "CR", "CR", "CR",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIKWID
     *
     * @cdk.inchi InChI=1S/C12H21N3O2/c16-15-11-7-3-1-5-9(11)13-10-6-2-4-8-12(10,15)17-14-11/h9-10,13-14,16H,1-8H2/t9-,10+,11-,12-/m0/s1
     */
    @Test
    public void testDIKWID() {
        IAtomContainer container = new AtomContainer(38, 41, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 13, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 19, SINGLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 21, SINGLE);
        container.addBond(3, 22, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 23, SINGLE);
        container.addBond(4, 24, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(5, 25, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 26, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 27, SINGLE);
        container.addBond(7, 28, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 29, SINGLE);
        container.addBond(8, 30, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 31, SINGLE);
        container.addBond(9, 32, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 33, SINGLE);
        container.addBond(10, 34, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(12, 35, SINGLE);
        container.addBond(13, 16, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 36, SINGLE);
        container.addBond(16, 37, SINGLE);

        String[] expected = {"CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "NR", "NR", "NR",
                "OR", "-O-", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HNR", "HNR", "HO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIKYUR
     *
     * @cdk.inchi InChI=1S/C8H6Cl2O2/c9-8(10)5-2-1-4(3-11)6(5)7(8)12/h1,3,5-6H,2H2/t5-,6+/m1/s1
     */
    @Test
    public void testDIKYUR() {
        IAtomContainer container = new AtomContainer(18, 19, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 11, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 17, SINGLE);

        String[] expected = {"CL", "CL", "O=CR", "O=CR", "CR4R", "C=OR", "CR4R", "C=C", "C=C", "CR", "CR4R", "C=OR",
                "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DILCOQ
     *
     * @cdk.inchi InChI=1S/C11H17NO3/c1-6(2)7-8(13)12-9(7)15-10(14)11(3,4)5/h9H,1-5H3,(H,12,13)/t9-/m0/s1
     */
    @Test
    public void testDILCOQ() {
        IAtomContainer container = new AtomContainer(32, 32, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 12, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(11, 25, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(13, 26, SINGLE);
        container.addBond(13, 27, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(14, 29, SINGLE);
        container.addBond(14, 30, SINGLE);
        container.addBond(14, 31, SINGLE);

        String[] expected = {"NC=O", "C=ON", "O=CN", "CE4R", "CR4R", "OC=O", "COO", "O=CO", "CR", "CR", "CR", "CR",
                "C=C", "CR", "CR", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIMYIH10
     *
     * @cdk.inchi InChI=1S/C10H6N2O6S2/c13-11(14)8-6-9(12(15)16)19-10(8)20(17,18)7-4-2-1-3-5-7/h1-6H
     */
    @Test
    public void testDIMYIH10() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 18, SINGLE);
        container.addBond(0, 19, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 13, DOUBLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(11, 15, DOUBLE);
        container.addBond(16, 19, DOUBLE);
        container.addBond(17, 19, DOUBLE);

        String[] expected = {"C5A", "C5B", "C5B", "C5A", "CB", "CB", "CB", "CB", "CB", "CB", "NO2", "NO2", "O2N",
                "O2N", "O2N", "O2N", "O2S", "O2S", "STHI", "SO2", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIPDAH10
     *
     * @cdk.inchi InChI=1S/C10H12N2O5/c1-4-2-6(14)11-10-12(4)9-8(17-10)7(15)5(3-13)16-9/h2,5,7-9,13,15H,3H2,1H3/t5-,7-,8+,9-/m1/s1
     */
    @Test
    public void testDIPDAH10() {
        IAtomContainer container = new AtomContainer(29, 31, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, DOUBLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 27, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 28, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 24, SINGLE);
        container.addBond(16, 25, SINGLE);
        container.addBond(16, 26, SINGLE);

        String[] expected = {"O=CN", "OC=N", "OR", "OR", "OR", "NC=N", "N=C", "CR", "C=N", "C=ON", "C=C", "C=C", "CR",
                "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HOR", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIPDIP10
     *
     * @cdk.inchi InChI=1S/C5H10N4/c6-2-1-4-3-8-5(7)9-4/h3H,1-2,6H2,(H3,7,8,9)/p+2
     */
    @Test
    public void testDIPDIP10() {
        IAtomContainer container = new AtomContainer(21, 21, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 4, DOUBLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 19, SINGLE);

        String[] expected = {"NIM+", "NIM+", "NGD+", "NR+", "CIM+", "C5", "C5", "CR", "CR", "HIM+", "HIM+", "HGD+",
                "HGD+", "HC", "HC", "HC", "HNR+", "HNR+", "HNR+", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIRMIA
     *
     * @cdk.inchi InChI=1S/CH5NO3S/c1-6(4,5)2-3/h2-3H,1H3
     */
    @Test
    public void testDIRMIA() {
        IAtomContainer container = new AtomContainer(11, 10, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(5, 9, SINGLE);

        String[] expected = {"SO2N", "O2S", "O2S", "-O-", "NSO2", "CR", "HNSO", "HC", "HC", "HC", "HO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DISHES
     *
     * @cdk.inchi InChI=1S/C12H13NO/c1-8-7-12(2,3)13-11-5-4-9(14)6-10(8)11/h4-7H,1-3H3
     */
    @Test
    public void testDISHES() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(13, 26, SINGLE);

        String[] expected = {"O=CR", "N=C", "C=N", "C=C", "C=C", "C=OR", "C=C", "C=C", "C=C", "C=C", "CR", "CR", "CR",
                "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DISJOE
     *
     * @cdk.inchi InChI=1S/C4H6ClNO4/c1-2-10-4(7)3(5)6(8)9/h7H,2H2,1H3/p-1/b4-3+
     */
    @Test
    public void testDISJOE() {
        IAtomContainer container = new AtomContainer(15, 14, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 14, SINGLE);

        String[] expected = {"CL", "O2N", "O2N", "OM2", "OC=C", "NO2", "C=C", "C=C", "CR", "CR", "HC", "HC", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DITRAZ
     *
     * @cdk.inchi InChI=1S/C2H7N3S/c1-6-2(3)5-4/h4H2,1H3,(H2,3,5)/p+1
     */
    @Test
    public void testDITRAZ() {
        IAtomContainer container = new AtomContainer(14, 13, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(5, 11, SINGLE);

        String[] expected = {"S", "CNN+", "NCN+", "NCN+", "NR", "CR", "HNN+", "HNN+", "HNN+", "HNR", "HNR", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DITYAG10
     *
     * @cdk.inchi InChI=1S/C12H13ClO/c13-11-6-4-10(5-7-11)12(14)8-9-2-1-3-9/h4-7,9H,1-3,8H2
     */
    @Test
    public void testDITYAG10() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 8, DOUBLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(13, 26, SINGLE);

        String[] expected = {"CL", "O=CR", "CB", "CB", "CB", "CB", "CB", "CB", "C=OR", "CR", "CR4R", "CR4R", "CR4R",
                "CR4R", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIVJUN
     *
     * @cdk.inchi InChI=1S/C5H9N3OS/c1-6-4-5(8-9)10-3-2-7-4/h9H,2-3H2,1H3,(H,6,7)/p+1/b8-5-
     */
    @Test
    public void testDIVJUN() {
        IAtomContainer container = new AtomContainer(20, 20, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 6, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 11, SINGLE);

        String[] expected = {"C=N", "CNN+", "CR", "CR", "CR", "NCN+", "N=C", "NCN+", "-O-", "S", "HNN+", "HO", "HNN+",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIVTUX
     *
     * @cdk.inchi InChI=1S/C11H12O2/c12-11-7-3-1-2-4-9(7)13-10-6-5-8(10)11/h1-4,8,10-12H,5-6H2/t8-,10+,11+/m1/s1
     */
    @Test
    public void testDIVTUX() {
        IAtomContainer container = new AtomContainer(25, 27, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 24, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 23, SINGLE);

        String[] expected = {"OR", "OC=C", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "CR4R", "CR4R", "CR4R", "CR4R",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIVVEJ
     *
     * @cdk.inchi InChI=1S/C2H6N4O/c3-1(4)6-2(5)7/h(H6,3,4,5,6,7)/p+1
     */
    @Test
    public void testDIVVEJ() {
        IAtomContainer container = new AtomContainer(14, 13, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 13, SINGLE);

        String[] expected = {"O=CN", "CONN", "CGD+", "NC=O", "NGD+", "NGD+", "NGD+", "HNCO", "HNCO", "HGD+", "HGD+",
                "HGD+", "HGD+", "HGD+"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIVWEK
     *
     * @cdk.inchi InChI=1S/C18H12O4/c19-16-14(11-12-7-3-1-4-8-12)22-17(18(20)21)15(16)13-9-5-2-6-10-13/h1-11H,(H,20,21)/b14-11-
     */
    @Test
    public void testDIVWEK() {
        IAtomContainer container = new AtomContainer(34, 36, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 12, DOUBLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 27, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 11, DOUBLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 28, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 31, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 24, SINGLE);
        container.addBond(11, 26, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(15, 33, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(16, 30, SINGLE);
        container.addBond(17, 18, DOUBLE);
        container.addBond(17, 32, SINGLE);
        container.addBond(18, 29, SINGLE);
        container.addBond(19, 20, DOUBLE);
        container.addBond(19, 21, SINGLE);
        container.addBond(21, 22, SINGLE);

        String[] expected = {"OC=C", "C=C", "C=OR", "C=C", "C=C", "C=C", "CB", "CB", "CB", "CB", "CB", "CB", "O=CR",
                "CB", "CB", "CB", "CB", "CB", "CB", "COO", "O=CO", "OC=O", "HOCO", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIWCOB
     *
     * @cdk.inchi InChI=1S/CH8N6/c2-5-1(6-3)7-4/h2-4H2,(H2,5,6,7)/p+1
     */
    @Test
    public void testDIWCOB() {
        IAtomContainer container = new AtomContainer(16, 15, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 14, SINGLE);

        String[] expected = {"CGD+", "NGD+", "NR", "NGD+", "NR", "NGD+", "NR", "HGD+", "HNR", "HNR", "HGD+", "HNR",
                "HNR", "HGD+", "HNR", "HNR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIXJEZ
     *
     * @cdk.inchi InChI=1S/C7H9N5/c1-12(2)7-5-6(9-3-8-5)10-4-11-7/h3-4H,1-2H3,(H,8,9,10,11)/p+1
     */
    @Test
    public void testDIXJEZ() {
        IAtomContainer container = new AtomContainer(22, 23, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 19, SINGLE);

        String[] expected = {"NPYD", "CB", "NPD+", "C5A", "C5B", "CB", "NC=N", "N5B", "C5A", "NPYL", "CR", "CR", "HC",
                "HPD+", "HC", "HPYL", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIYDIY
     *
     * @cdk.inchi InChI=1S/C9H8O4/c10-5-7-3-1-2-4-8(7)13-6-9(11)12/h1-5H,6H2,(H,11,12)
     */
    @Test
    public void testDIYDIY() {
        IAtomContainer container = new AtomContainer(21, 21, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 19, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "OC=C", "CR", "COO", "O=CO", "OC=O", "C=OR", "O=CR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HOCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIYPOQ
     *
     * @cdk.inchi InChI=1S/C9H15N5O3/c1-3(15)6(16)4-2-11-7-5(12-4)8(17)14-9(10)13-7/h3-4,6,12,15-16H,2H2,1H3,(H4,10,11,13,14,17)/p+2/t3-,4+,6-/m0/s1
     */
    @Test
    public void testDIYPOQ() {
        IAtomContainer container = new AtomContainer(34, 35, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 17, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 18, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 11, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 20, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(14, 29, SINGLE);
        container.addBond(14, 16, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 30, SINGLE);
        container.addBond(16, 33, SINGLE);
        container.addBond(16, 32, SINGLE);
        container.addBond(16, 31, SINGLE);

        String[] expected = {"NGD+", "CGD+", "NGD+", "C=ON", "NR+", "CR", "CR", "NC=C", "C=C", "C=C", "NGD+", "O=CN",
                "CR", "OR", "CR", "OR", "CR", "HGD+", "HGD+", "HNR+", "HNR+", "HC", "HC", "HC", "HNCC", "HGD+", "HGD+",
                "HC", "HOR", "HC", "HOR", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIYPUW
     *
     * @cdk.inchi InChI=1S/C5H8N2O/c8-5(6-1-2-6)7-3-4-7/h1-4H2
     */
    @Test
    public void testDIYPUW() {
        IAtomContainer container = new AtomContainer(16, 17, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(11, 15, SINGLE);

        String[] expected = {"O=CN", "CONN", "NC=O", "CR3R", "CR3R", "HC", "HC", "HC", "HC", "NC=O", "CR3R", "CR3R",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DIZPUX
     *
     * @cdk.inchi InChI=1S/C10H15NO/c1-7(12)9-4-2-3-8-5-6-10(9)11-8/h4,8,10-11H,2-3,5-6H2,1H3/p+1/t8-,10-/m1/s1
     */
    @Test
    public void testDIZPUX() {
        IAtomContainer container = new AtomContainer(28, 29, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(6, 21, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 25, SINGLE);
        container.addBond(11, 26, SINGLE);
        container.addBond(11, 27, SINGLE);

        String[] expected = {"O=CR", "NR+", "CR", "CR", "CR", "CR", "CR", "CR", "C=C", "C=C", "C=OR", "CR", "HNR+",
                "HNR+", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DMEOXA01
     *
     * @cdk.inchi InChI=1S/C4H6O4/c1-7-3(5)4(6)8-2/h1-2H3
     */
    @Test
    public void testDMEOXA01() {
        IAtomContainer container = new AtomContainer(14, 13, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 13, SINGLE);

        String[] expected = {"COO", "CR", "O=CO", "OC=O", "HC", "HC", "HC", "COO", "O=CO", "OC=O", "CR", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOCCIH
     *
     * @cdk.inchi InChI=1S/C8H6S4/c9-7(10)5-1-2-6(4-3-5)8(11)12/h1-4H,(H,9,10)(H,11,12)/p-2
     */
    @Test
    public void testDOCCIH() {
        IAtomContainer container = new AtomContainer(16, 16, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", -1));
        container.addBond(0, 2, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 9, DOUBLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(8, 10, DOUBLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(13, 14, DOUBLE);

        String[] expected = {"S2CM", "S2CM", "CS2M", "CB", "CB", "CB", "HC", "HC", "CB", "CB", "CB", "HC", "HC",
                "CS2M", "S2CM", "S2CM"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOCFIK
     *
     * @cdk.inchi InChI=1S/C8Cl2N8S3/c9-5(7-13-3(1-11)15-19-7)17-21-18-6(10)8-14-4(2-12)16-20-8/b17-5-,18-6-
     */
    @Test
    public void testDOCFIK() {
        IAtomContainer container = new AtomContainer(21, 22, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(9, 10, TRIPLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 18, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(16, 19, SINGLE);
        container.addBond(17, 18, DOUBLE);
        container.addBond(19, 20, TRIPLE);

        String[] expected = {"S", "N=C", "C=N", "CL", "STHI", "N5A", "C5B", "N5B", "C5A", "CSP", "NSP", "N=C", "C=N",
                "CL", "STHI", "N5A", "C5B", "N5B", "C5A", "CSP", "NSP"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOCWUN
     *
     * @cdk.inchi InChI=1S/C9H11N3O3/c1-6-4-3-5-12(9(6)13)8(11-15)7(2)10-14/h3-5,14-15H,1-2H3/b10-7+,11-8-
     */
    @Test
    public void testDOCWUN() {
        IAtomContainer container = new AtomContainer(26, 26, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, DOUBLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 13, DOUBLE);
        container.addBond(5, 12, DOUBLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(14, 23, SINGLE);

        String[] expected = {"O=CN", "-O-", "-O-", "NC=O", "N=C", "N=C", "CR", "C=ON", "C=C", "C=C", "C=C", "C=C",
                "C=N", "C=N", "CR", "HO", "HO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DODNOZ
     *
     * @cdk.inchi InChI=1S/C6H15NO6S3/c1-4-14(8,9)7(15(10,11)5-2)16(12,13)6-3/h4-6H2,1-3H3
     */
    @Test
    public void testDODNOZ() {
        IAtomContainer container = new AtomContainer(31, 30, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(10, 24, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 13, DOUBLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(14, 27, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(15, 29, SINGLE);
        container.addBond(15, 30, SINGLE);

        String[] expected = {"NSO2", "SO2N", "O2S", "O2S", "CR", "CR", "SO2N", "O2S", "O2S", "CR", "CR", "SO2N", "O2S",
                "O2S", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DODNUF
     *
     * @cdk.inchi InChI=1S/C4H11NO4S2/c1-3-10(6,7)5-11(8,9)4-2/h5H,3-4H2,1-2H3
     */
    @Test
    public void testDODNUF() {
        IAtomContainer container = new AtomContainer(22, 21, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 21, SINGLE);

        String[] expected = {"NSO2", "SO2N", "O2S", "O2S", "CR", "CR", "SO2N", "O2S", "O2S", "CR", "CR", "HNSO", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOJPAT
     *
     * @cdk.inchi InChI=1S/C8H10O2S/c9-11(10)3-7-5-1-2-6(5)8(7)4-11/h1-2,5-8H,3-4H2/t5-,6+,7+,8-
     */
    @Test
    public void testDOJPAT() {
        IAtomContainer container = new AtomContainer(21, 23, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 20, SINGLE);

        String[] expected = {"SO2", "O2S", "O2S", "CR", "CR4R", "CR4R", "CE4R", "CE4R", "CR4R", "CR4R", "CR", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DONFOB
     *
     * @cdk.inchi InChI=1S/C5H13NO6S4/c1-6(2)5(3-13-15(7,8)9)4-14-16(10,11)12/h5H,3-4H2,1-2H3,(H,7,8,9)(H,10,11,12)/p-1
     */
    @Test
    public void testDONFOB() {
        IAtomContainer container = new AtomContainer(28, 27, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 6, DOUBLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 8, DOUBLE);
        container.addBond(1, 9, DOUBLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 27, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(15, 25, SINGLE);
        container.addBond(15, 26, SINGLE);

        String[] expected = {"SO3", "SO3", "S", "S", "NR+", "O3S", "O3S", "O3S", "O3S", "O3S", "O3S", "CR", "CR", "CR",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNR+"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOSNOO
     *
     * @cdk.inchi InChI=1S/C3H6N2O3S/c4-3(6)5-1-2-9(5,7)8/h1-2H2,(H2,4,6)
     */
    @Test
    public void testDOSNOO() {
        IAtomContainer container = new AtomContainer(15, 15, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 13, DOUBLE);
        container.addBond(1, 14, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 12, SINGLE);

        String[] expected = {"NSO2", "SO2N", "CR4R", "HC", "HC", "CR4R", "HC", "HC", "CONN", "O=CN", "NC=O", "HNCO",
                "HNCO", "O2S", "O2S"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOTNIJ
     *
     * @cdk.inchi InChI=1S/C8H7NS2/c10-6-9(7-11)8-4-2-1-3-5-8/h1-7H
     */
    @Test
    public void testDOTNIJ() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(11, 14, DOUBLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(13, 17, SINGLE);

        String[] expected = {"S=C", "NC=S", "C=SN", "CB", "CB", "CB", "CB", "HC", "HC", "HC", "HC", "C=SN", "CB", "CB",
                "S=C", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOTVEN
     *
     * @cdk.inchi InChI=1S/C3H5N3O3S/c4-2-1-3(7)6-10(8,9)5-2/h1H2,(H2,4,5)(H,6,7)
     */
    @Test
    public void testDOTVEN() {
        IAtomContainer container = new AtomContainer(15, 15, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, DOUBLE);
        container.addBond(0, 6, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 8, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(9, 10, SINGLE);

        String[] expected = {"SO2N", "NSO2", "C=ON", "CR", "C=N", "N=C", "O2S", "O2S", "O=CN", "NC=N", "HNCN", "HC",
                "HNSO", "HNCN", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOTWOY
     *
     * @cdk.inchi InChI=1S/C12H12Cl2O3/c1-2-17-9-5-3-8(4-6-9)11(10(15)16)7-12(11,13)14/h3-6H,2,7H2,1H3,(H,15,16)/t11-/m0/s1
     */
    @Test
    public void testDOTWOY() {
        IAtomContainer container = new AtomContainer(29, 30, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 14, DOUBLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(13, 26, SINGLE);
        container.addBond(13, 27, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(15, 17, SINGLE);

        String[] expected = {"CL", "CL", "CR3R", "CR3R", "CR3R", "COO", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "CR",
                "O=CO", "OC=O", "OC=C", "HOCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOWDEY
     *
     * @cdk.inchi InChI=1S/C6H5N3O2S/c1-12-4-3(2-7)5(10)11-6(8)9-4/h1H3,(H2,8,9)
     */
    @Test
    public void testDOWDEY() {
        IAtomContainer container = new AtomContainer(17, 17, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 11, DOUBLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 10, TRIPLE);

        String[] expected = {"OC=O", "C=N", "N=C", "C=C", "C=C", "COO", "NC=N", "S", "CR", "CSP", "NSP", "O=CO",
                "HNCN", "HNCN", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOXXAP
     *
     * @cdk.inchi InChI=1S/C3H7N3O4/c4-2(3(7)8)1-6(10)5-9/h2,9H,1,4H2,(H,7,8)/b6-5-/t2-/m0/s1
     */
    @Test
    public void testDOXXAP() {
        IAtomContainer container = new AtomContainer(17, 16, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 6, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(9, 16, SINGLE);

        String[] expected = {"CO2M", "CR", "CR", "NR+", "N2OX", "N=N", "O2CM", "O2CM", "OXN", "-O-", "HC", "HC", "HC",
                "HNR+", "HNR+", "HNR+", "HO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOXZOF
     *
     * @cdk.inchi InChI=1S/C8H7NO/c10-9-8-5-6-3-1-2-4-7(6)8/h1-4,10H,5H2/b9-8-
     */
    @Test
    public void testDOXZOF() {
        IAtomContainer container = new AtomContainer(17, 18, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 9, DOUBLE);

        String[] expected = {"-O-", "N=C", "C=N", "CR4R", "CB", "CB", "CB", "CB", "CB", "CB", "HO", "HC", "HC", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOZFON
     *
     * @cdk.inchi InChI=1S/C4H7F2N3O3/c1-8(2)7-3(10)4(5,6)9(11)12/h1-2H3,(H,7,10)/p+1
     */
    @Test
    public void testDOZFON() {
        IAtomContainer container = new AtomContainer(20, 19, 0, 0);
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 8, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 7, DOUBLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 16, SINGLE);

        String[] expected = {"F", "F", "O=CN", "O2N", "O2N", "NC=O", "NR+", "NO2", "C=ON", "CR", "CR", "CR", "HNR+",
                "HC", "HC", "HC", "HC", "HC", "HC", "HNCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DOZNIP
     *
     * @cdk.inchi InChI=1S/C2H2N7O4/c10-8(11)6-1-3-2(5-4-1)7-9(12)13/h(H2-,3,4,5,6,7)/q-1
     */
    @Test
    public void testDOZNIP() {
        IAtomContainer container = new AtomContainer(15, 15, 0, 0);
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", -1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 4, DOUBLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 10, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 11, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 12, DOUBLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 10, SINGLE);

        String[] expected = {"O2N", "O2N", "O2N", "O2N", "NO2", "NC=N", "N5M", "N5M", "N5M", "NC=N", "NO2", "C5", "C5",
                "HNCN", "HNCN"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUBNET
     *
     * @cdk.inchi InChI=1S/C8H9N3O2/c1-9-7(12)10-5-2-3-6(4-5)11(10)8(9)13/h2-3,5-6H,4H2,1H3/t5-,6+
     */
    @Test
    public void testDUBNET() {
        IAtomContainer container = new AtomContainer(22, 24, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, DOUBLE);
        container.addBond(1, 11, DOUBLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(12, 21, SINGLE);

        String[] expected = {"O=CN", "O=CN", "NC=O", "NC=O", "NC=O", "CR", "C=C", "C=C", "CR", "CR", "CONN", "CONN",
                "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUDMUK
     *
     * @cdk.inchi InChI=1S/C5H10N4O/c1-4-6-7-5(2)9(3-10)8-4/h5,10H,3H2,1-2H3/t5-/m1/s1
     */
    @Test
    public void testDUDMUK() {
        IAtomContainer container = new AtomContainer(20, 20, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 18, SINGLE);

        String[] expected = {"NR", "N=C", "C=N", "N=N", "N=N", "CR", "CR", "CR", "CR", "OR", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HOR", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUGMUN
     *
     * @cdk.inchi InChI=1S/C8H6N2O/c9-6-8(10-11)7-4-2-1-3-5-7/h1-5,11H/b10-8+
     */
    @Test
    public void testDUGMUN() {
        IAtomContainer container = new AtomContainer(17, 17, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(2, 4, TRIPLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(10, 16, SINGLE);

        String[] expected = {"-O-", "N=C", "NSP", "C=N", "CSP", "CB", "CB", "CB", "CB", "CB", "CB", "HO", "HC", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUGWIL01
     *
     * @cdk.inchi InChI=1S/C3H9NO4S2/c1-4(9(2,5)6)10(3,7)8/h1-3H3
     */
    @Test
    public void testDUGWIL01() {
        IAtomContainer container = new AtomContainer(19, 18, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 4, DOUBLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 18, SINGLE);

        String[] expected = {"SO2N", "SO2N", "O2S", "O2S", "O2S", "O2S", "NSO2", "CR", "CR", "CR", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUJHEV
     *
     * @cdk.inchi InChI=1S/C6H7NO3S/c1-6(8)7-2-4-11(9,10)5-3-7/h2-5H,1H3
     */
    @Test
    public void testDUJHEV() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, DOUBLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(9, 17, SINGLE);

        String[] expected = {"SO2", "O2S", "O2S", "C=C", "C=C", "C=C", "C=C", "NC=O", "C=ON", "CR", "O=CN", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUJMEA
     *
     * @cdk.inchi InChI=1S/C12H18N2O3/c1-13-5-9-17-12(10-4-3-7-15-10)11(13)16-8-6-14(12)2/h3-4,7,11H,5-6,8-9H2,1-2H3/t11-,12-/m0/s1
     */
    @Test
    public void testDUJMEA() {
        IAtomContainer container = new AtomContainer(35, 37, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 21, SINGLE);
        container.addBond(2, 22, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 23, SINGLE);
        container.addBond(3, 24, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 25, SINGLE);
        container.addBond(5, 26, SINGLE);
        container.addBond(5, 27, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 28, SINGLE);
        container.addBond(8, 29, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 30, SINGLE);
        container.addBond(9, 31, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 32, SINGLE);
        container.addBond(11, 33, SINGLE);
        container.addBond(11, 34, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(13, 16, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(15, 19, SINGLE);
        container.addBond(16, 20, SINGLE);

        String[] expected = {"CR", "OR", "CR", "CR", "NR", "CR", "CR", "OR", "CR", "CR", "NR", "CR", "C5A", "C5B",
                "OFUR", "C5A", "C5B", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUKVAG
     *
     * @cdk.inchi InChI=1S/C2H4N4S/c3-1-5-6-2(4)7-1/h(H2,3,5)(H2,4,6)/p+1
     */
    @Test
    public void testDUKVAG() {
        IAtomContainer container = new AtomContainer(12, 12, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 6, SINGLE);

        String[] expected = {"STHI", "NIM+", "N5B", "NCN+", "NC=N", "CIM+", "C5A", "HIM+", "HNN+", "HNN+", "HNCN",
                "HNCN"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUKWUB
     *
     * @cdk.inchi InChI=1S/C6H13NO7S/c7-3-5(9)4(8)2(14-6(3)10)1-15(11,12)13/h2-6,8-10H,1,7H2,(H,11,12,13)/t2-,3-,4-,5-,6+/m1/s1
     */
    @Test
    public void testDUKWUB() {
        IAtomContainer container = new AtomContainer(28, 28, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(1, 26, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(4, 27, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 24, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 25, SINGLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(14, 17, SINGLE);
        container.addBond(14, 18, SINGLE);

        String[] expected = {"SO3", "OR", "OR", "O2S", "OSO2", "O2S", "OR", "OR", "NR", "CR", "CR", "CR", "CR", "CR",
                "CR", "HNR", "HNR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HOR", "HOR", "HOR", "HOS"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DULTIN
     *
     * @cdk.inchi InChI=1S/C8H12BrNO2/c9-6-3-10-2-1-5-7(10)8(6,4-11)12-5/h5-7,11H,1-4H2/t5-,6-,7+,8+/m1/s1
     */
    @Test
    public void testDULTIN() {
        IAtomContainer container = new AtomContainer(24, 26, 0, 0);
        container.addAtom(newAtm("Br", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(11, 23, SINGLE);

        String[] expected = {"BR", "CR4R", "CR", "CR", "NR", "CR", "CR", "CR4R", "CR4R", "CR", "OR", "OR", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUMHIC
     *
     * @cdk.inchi InChI=1S/C8H12O5S/c1-9-6-5-4(12-8(14)13-5)3-11-7(6)10-2/h4-7H,3H2,1-2H3/t4-,5-,6+,7-/m0/s1
     */
    @Test
    public void testDUMHIC() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(13, 25, SINGLE);

        String[] expected = {"OR", "CR", "CR", "OC=S", "C=S", "S=C", "CR", "OC=S", "CR", "OR", "CR", "CR", "OR", "CR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUMPAC
     *
     * @cdk.inchi InChI=1S/C7H14N4O5/c1-8(10(12)13)3-7(5-16-6-7)4-9(2)11(14)15/h3-6H2,1-2H3
     */
    @Test
    public void testDUMPAC() {
        IAtomContainer container = new AtomContainer(30, 30, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(3, 19, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 20, SINGLE);
        container.addBond(4, 21, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 22, SINGLE);
        container.addBond(5, 23, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(7, 25, SINGLE);
        container.addBond(7, 26, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, DOUBLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(12, 29, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 15, DOUBLE);

        String[] expected = {"OR", "CR4R", "CR4R", "CR4R", "CR", "CR", "NR", "CR", "NO2", "O2N", "O2N", "NR", "CR",
                "NO2", "O2N", "O2N", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUPHEB
     *
     * @cdk.inchi InChI=1S/C9H9ClN2O2S/c1-15-9(6-12(13)14)11-8-5-3-2-4-7(8)10/h2-6,11H,1H3/b9-6+
     */
    @Test
    public void testDUPHEB() {
        IAtomContainer container = new AtomContainer(24, 24, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 14, DOUBLE);
        container.addBond(12, 23, SINGLE);

        String[] expected = {"S", "CL", "C=C", "C=C", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "NO2", "NC=C", "O2N",
                "O2N", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUPTAJ
     *
     * @cdk.inchi InChI=1S/C10H9NO2S/c12-14(13,11-8-4-5-9-11)10-6-2-1-3-7-10/h1-9H
     */
    @Test
    public void testDUPTAJ() {
        IAtomContainer container = new AtomContainer(23, 24, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(5, 7, DOUBLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 22, SINGLE);

        String[] expected = {"NPYL", "C5A", "C5B", "C5B", "C5A", "SO2N", "O2S", "O2S", "CB", "CB", "CB", "CB", "CB",
                "CB", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DURDID
     *
     * @cdk.inchi InChI=1S/C6H10N2O2/c1-5-4-6(2,3)8(10)7(5)9/h4H,1-3H3
     */
    @Test
    public void testDURDID() {
        IAtomContainer container = new AtomContainer(20, 20, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 9, SINGLE);

        String[] expected = {"C=C", "C=C", "CR", "CR", "CR", "CR", "N2OX", "N2OX", "OXN", "OXN", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUTHIJ
     *
     * @cdk.inchi InChI=1S/C12H14N2O3/c1-15-12(11-13-6-7-14-11)8-16-9-4-2-3-5-10(9)17-12/h2-5H,6-8H2,1H3,(H,13,14)/p+1/t12-/m1/s1
     */
    @Test
    public void testDUTHIJ() {
        IAtomContainer container = new AtomContainer(32, 34, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 12, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 18, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 19, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 20, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(6, 21, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 16, DOUBLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 27, SINGLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 31, SINGLE);
        container.addBond(11, 30, SINGLE);
        container.addBond(11, 29, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(15, 23, SINGLE);
        container.addBond(16, 28, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "CR", "CR", "CNN+", "CR", "CR", "CR", "OC=C", "OC=C",
                "OR", "NCN+", "NCN+", "HC", "HC", "HC", "HC", "HC", "HC", "HNN+", "HC", "HC", "HC", "HC", "HNN+", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUVHUX10
     *
     * @cdk.inchi InChI=1S/C10H8N2S3/c1-7-9(14-15-10(7)13)3-2-8-6-11-4-5-12-8/h2-6H,1H3/b3-2+
     */
    @Test
    public void testDUVHUX10() {
        IAtomContainer container = new AtomContainer(23, 24, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 13, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 11, DOUBLE);
        container.addBond(3, 5, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 6, DOUBLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(14, 20, SINGLE);
        container.addBond(14, 21, SINGLE);
        container.addBond(14, 22, SINGLE);

        String[] expected = {"S", "S", "S=C", "NPYD", "NPYD", "CB", "CB", "CB", "CB", "C=C", "C=C", "CSS", "C=C",
                "C=C", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUVXIB
     *
     * @cdk.inchi InChI=1S/C5H5N5O2/c1-10-3-2(7-9-8-3)4(11)6-5(10)12/h1H3,(H,6,11,12)(H,7,8,9)
     */
    @Test
    public void testDUVXIB() {
        IAtomContainer container = new AtomContainer(17, 18, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 12, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 9, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 8, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 6, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 11, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 13, SINGLE);

        String[] expected = {"NC=O", "CONN", "NC=O", "C5B", "C5B", "C=ON", "N5A", "NPYL", "N5A", "O=CN", "CR", "O=CN",
                "HNCO", "HC", "HC", "HC", "HPYL"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUWGAD
     *
     * @cdk.inchi InChI=1S/C5H13ClN2P/c1-7-4-5-8(2,3)9(7)6/h4-5H2,1-3H3/q+1/t9-/m1/s1
     */
    @Test
    public void testDUWGAD() {
        IAtomContainer container = new AtomContainer(22, 22, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 19, SINGLE);

        String[] expected = {"P", "CL", "NR", "CR", "CR", "CR", "NR+", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUWKUB
     *
     * @cdk.inchi InChI=1S/C5H6Cl3N5O2/c1-14-13-4(5(6,7)8)10-3-2(9)11-15-12-3/h1H3,(H2,9,11)(H,10,12,13)
     */
    @Test
    public void testDUWKUB() {
        IAtomContainer container = new AtomContainer(21, 21, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 12, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 7, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 16, SINGLE);
        container.addBond(14, 17, SINGLE);

        String[] expected = {"CL", "CL", "CL", "OFUR", "C5B", "OR", "NC=N", "N5A", "N5A", "C5B", "C=N", "N=C", "CR",
                "NC=N", "CR", "HC", "HC", "HC", "HNCN", "HNCN", "HNCN"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUWRIW
     *
     * @cdk.inchi InChI=1S/C6H9NO6/c8-2-4-6(10)3(9)1-5(13-4)7(11)12/h1,3-4,6,8-10H,2H2/t3-,4-,6+/m1/s1
     */
    @Test
    public void testDUWRIW() {
        IAtomContainer container = new AtomContainer(22, 22, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, DOUBLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 21, SINGLE);

        String[] expected = {"C=C", "NO2", "O2N", "O2N", "C=C", "CR", "OR", "CR", "OR", "CR", "OC=C", "CR", "OR", "HC",
                "HC", "HOR", "HC", "HOR", "HC", "HC", "HC", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUXTIZ
     *
     * @cdk.inchi InChI=1S/C2H4N2O4/c5-1(3-7)2(6)4-8/h7-8H,(H,3,5)(H,4,6)
     */
    @Test
    public void testDUXTIZ() {
        IAtomContainer container = new AtomContainer(12, 11, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(9, 11, SINGLE);

        String[] expected = {"C=ON", "NC=O", "-O-", "O=CN", "HNCO", "HO", "C=ON", "NC=O", "O=CN", "-O-", "HNCO", "HO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUXWUO
     *
     * @cdk.inchi InChI=1S/C11H12N2O2S2/c1-8(14)13-10(9-6-4-3-5-7-9)16-11(12-13)17(2)15/h3-7,10H,1-2H3/t10-,17+/m0/s1
     */
    @Test
    public void testDUXWUO() {
        IAtomContainer container = new AtomContainer(29, 30, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 13, DOUBLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 16, SINGLE);
        container.addBond(16, 26, SINGLE);
        container.addBond(16, 27, SINGLE);
        container.addBond(16, 28, SINGLE);

        String[] expected = {"S", "CR", "NC=O", "N=C", "C=N", "CB", "CB", "CB", "CB", "CB", "CB", "C=ON", "CR", "O=CN",
                "S=O", "O=S", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUXXAV
     *
     * @cdk.inchi InChI=1S/C11H12N2O3S2/c1-8(14)13-10(9-6-4-3-5-7-9)18(16)11(12-13)17(2)15/h3-7,10H,1-2H3/t10-,17-,18+/m0/s1
     */
    @Test
    public void testDUXXAV() {
        IAtomContainer container = new AtomContainer(30, 31, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 17, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 13, DOUBLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 16, SINGLE);
        container.addBond(16, 27, SINGLE);
        container.addBond(16, 28, SINGLE);
        container.addBond(16, 29, SINGLE);

        String[] expected = {"S=O", "CR", "NC=O", "N=C", "C=N", "CB", "CB", "CB", "CB", "CB", "CB", "C=ON", "CR",
                "O=CN", "S=O", "O=S", "CR", "O=S", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUYNOA
     *
     * @cdk.inchi InChI=1S/C11H22N4O/c1-8-5-9(2)14(12)10(3,6-8)16-11(4,7-8)15(9)13/h5-7,12-13H2,1-4H3/t8-,9+,10-,11+
     */
    @Test
    public void testDUYNOA() {
        IAtomContainer container = new AtomContainer(38, 40, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(10, 27, SINGLE);
        container.addBond(10, 28, SINGLE);
        container.addBond(11, 29, SINGLE);
        container.addBond(11, 30, SINGLE);
        container.addBond(11, 31, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(13, 35, SINGLE);
        container.addBond(13, 36, SINGLE);
        container.addBond(13, 37, SINGLE);
        container.addBond(14, 32, SINGLE);
        container.addBond(14, 33, SINGLE);
        container.addBond(14, 34, SINGLE);
        container.addBond(15, 18, SINGLE);
        container.addBond(15, 19, SINGLE);

        String[] expected = {"CR", "OR", "CR", "NR", "CR", "CR", "CR", "CR", "NR", "CR", "CR", "CR", "NR", "CR", "CR",
                "NR", "HNR", "HNR", "HNR", "HNR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUYPES
     *
     * @cdk.inchi InChI=1S/C20H23N2O2/c1-21(2)20(22(3)4)24-19(17-13-9-6-10-14-17)15-18(23)16-11-7-5-8-12-16/h5-15H,1-4H3/q+1/b19-15+
     */
    @Test
    public void testDUYPES() {
        IAtomContainer container = new AtomContainer(47, 48, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 11, DOUBLE);
        container.addBond(2, 8, DOUBLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(4, 26, SINGLE);
        container.addBond(4, 25, SINGLE);
        container.addBond(4, 24, SINGLE);
        container.addBond(5, 29, SINGLE);
        container.addBond(5, 28, SINGLE);
        container.addBond(5, 27, SINGLE);
        container.addBond(6, 32, SINGLE);
        container.addBond(6, 31, SINGLE);
        container.addBond(6, 30, SINGLE);
        container.addBond(7, 35, SINGLE);
        container.addBond(7, 34, SINGLE);
        container.addBond(7, 33, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(10, 36, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 37, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 38, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(15, 39, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 40, SINGLE);
        container.addBond(16, 17, DOUBLE);
        container.addBond(17, 41, SINGLE);
        container.addBond(18, 23, SINGLE);
        container.addBond(18, 19, DOUBLE);
        container.addBond(19, 42, SINGLE);
        container.addBond(19, 20, SINGLE);
        container.addBond(20, 43, SINGLE);
        container.addBond(20, 21, DOUBLE);
        container.addBond(21, 44, SINGLE);
        container.addBond(21, 22, SINGLE);
        container.addBond(22, 45, SINGLE);
        container.addBond(22, 23, DOUBLE);
        container.addBond(23, 46, SINGLE);

        String[] expected = {"OC=C", "O=CR", "NCN+", "NCN+", "CR", "CR", "CR", "CR", "CNN+", "C=C", "C=C", "C=OR",
                "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for DUYRAQ
     *
     * @cdk.inchi InChI=1S/C9H22N2P2/c1-8(2,3)12-7-13(11-10-12)9(4,5)6/h10-11H,7H2,1-6H3/t12-,13-/m1/s1
     */
    @Test
    public void testDUYRAQ() {
        IAtomContainer container = new AtomContainer(35, 35, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(10, 24, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(13, 29, SINGLE);
        container.addBond(13, 30, SINGLE);
        container.addBond(13, 31, SINGLE);
        container.addBond(14, 32, SINGLE);
        container.addBond(14, 33, SINGLE);
        container.addBond(14, 34, SINGLE);

        String[] expected = {"P", "P", "NR", "HNR", "NR", "HNR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FACMIF
     *
     * @cdk.inchi InChI=1S/C10H13N2OP/c1-6-10-8-3-4-9(5-8)14(10)12(11-6)7(2)13/h3-4,8-10H,5H2,1-2H3/t8-,9+,10+,14-/m0/s1
     */
    @Test
    public void testFACMIF() {
        IAtomContainer container = new AtomContainer(27, 29, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 4, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 6, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(13, 26, SINGLE);

        String[] expected = {"P", "O=CN", "NC=O", "N=C", "C=ON", "CR", "C=N", "CR", "CR", "CR", "CR", "CR", "C=C",
                "C=C", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FACREG
     *
     * @cdk.inchi InChI=1S/C5H9NO6/c7-1-2-3(8)4(9)5(12-2)6(10)11/h2-5,7-9H,1H2/t2-,3-,4-,5-/m1/s1
     */
    @Test
    public void testFACREG() {
        IAtomContainer container = new AtomContainer(21, 21, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("O", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(19, 20, SINGLE);

        String[] expected = {"CR", "NO2", "O2N", "O2N", "CR", "OR", "CR", "OR", "CR", "OR", "CR", "HC", "HC", "HOR",
                "HC", "HOR", "HC", "HC", "HC", "HOR", "OR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FACYAJ
     *
     * @cdk.inchi InChI=1S/C10H5NO3/c11-3-10-7-2-9-8(13-5-14-9)1-6(7)4-12-10/h1-2,4H,5H2
     */
    @Test
    public void testFACYAJ() {
        IAtomContainer container = new AtomContainer(19, 21, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 8, DOUBLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(9, 10, TRIPLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(13, 18, SINGLE);

        String[] expected = {"C5A", "OFUR", "C5A", "C5B", "CB", "CB", "CB", "CB", "C5B", "CSP", "NSP", "OC=C", "OC=C",
                "CR", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FADMIG
     *
     * @cdk.inchi InChI=1S/C7H6N2O/c10-7-5-3-1-2-4-6(5)8-9-7/h1-4H,(H2,8,9,10)
     */
    @Test
    public void testFADMIG() {
        IAtomContainer container = new AtomContainer(16, 17, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 9, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 8, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 8, SINGLE);

        String[] expected = {"NC=C", "NC=O", "C=ON", "CB", "CB", "CB", "CB", "CB", "CB", "O=CN", "HNCC", "HNCO", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FADVEL
     *
     * @cdk.inchi InChI=1S/C3H9NO7S3/c1-12(5,6)4(13(2,7)8)11-14(3,9)10/h1-3H3
     */
    @Test
    public void testFADVEL() {
        IAtomContainer container = new AtomContainer(23, 22, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 4, DOUBLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(1, 7, DOUBLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 8, DOUBLE);
        container.addBond(2, 9, DOUBLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(13, 20, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(13, 22, SINGLE);

        String[] expected = {"SO2N", "SO3", "SO2N", "O2S", "O2S", "OSO2", "O2S", "O2S", "O2S", "O2S", "NSO2", "CR",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FADVUB
     *
     * @cdk.inchi InChI=1S/C5H12N2O3S/c1-11(7,10)3-2-4(6)5(8)9/h4,7H,2-3,6H2,1H3,(H,8,9)/t4-,11-/m0/s1
     */
    @Test
    public void testFADVUB() {
        IAtomContainer container = new AtomContainer(23, 22, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 9, DOUBLE);

        String[] expected = {"SNO", "O-S", "NSO", "CR", "CR", "CR", "CR", "NR+", "CO2M", "O2CM", "O2CM", "HNSO", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNR+", "HNR+", "HNR+"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAGBUK
     *
     * @cdk.inchi InChI=1S/C10H24N4O2P/c1-11(2)9(12(3)4)17(15,16)10(13(5)6)14(7)8/h1-8H3/q+1
     */
    @Test
    public void testFAGBUK() {
        IAtomContainer container = new AtomContainer(41, 40, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 22, SINGLE);
        container.addBond(0, 21, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 4, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(22, 24, SINGLE);
        container.addBond(22, 23, DOUBLE);
        container.addBond(23, 26, SINGLE);
        container.addBond(23, 25, SINGLE);
        container.addBond(24, 28, SINGLE);
        container.addBond(24, 27, SINGLE);
        container.addBond(25, 31, SINGLE);
        container.addBond(25, 30, SINGLE);
        container.addBond(25, 29, SINGLE);
        container.addBond(26, 34, SINGLE);
        container.addBond(26, 33, SINGLE);
        container.addBond(26, 32, SINGLE);
        container.addBond(27, 37, SINGLE);
        container.addBond(27, 36, SINGLE);
        container.addBond(27, 35, SINGLE);
        container.addBond(28, 40, SINGLE);
        container.addBond(28, 39, SINGLE);
        container.addBond(28, 38, SINGLE);

        String[] expected = {"PO2", "O2P", "NCN+", "NCN+", "CNN+", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "O2P", "CNN+", "NCN+", "NCN+", "CR", "CR", "CR", "CR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAGCOF
     *
     * @cdk.inchi InChI=1S/C10H11NO3S/c1-15(13,14)11-9(7-10(11)12)8-5-3-2-4-6-8/h2-6,9H,7H2,1H3/t9-/m0/s1
     */
    @Test
    public void testFAGCOF() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(14, 25, SINGLE);

        String[] expected = {"SO2N", "O=CN", "O2S", "O2S", "NSO2", "C=ON", "CR4R", "CR4R", "CB", "CB", "CB", "CB",
                "CB", "CB", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAGLII
     *
     * @cdk.inchi InChI=1S/C6H4ClO2P/c7-10-8-5-3-1-2-4-6(5)9-10/h1-4H
     */
    @Test
    public void testFAGLII() {
        IAtomContainer container = new AtomContainer(14, 15, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);

        String[] expected = {"CL", "P", "OPO", "OPO", "CB", "CB", "CB", "CB", "CB", "CB", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAGVEO
     *
     * @cdk.inchi InChI=1S/C4O4/c5-1-2(6)4(8)3(1)7
     */
    @Test
    public void testFAGVEO() {
        IAtomContainer container = new AtomContainer(8, 8, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addBond(0, 4, DOUBLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);

        String[] expected = {"O=CR", "O=CR", "O=CR", "O=CR", "C=OR", "C=OR", "C=OR", "C=OR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAGZOC
     *
     * @cdk.inchi InChI=1S/C6H2Br2O4/c7-1-3(9)5(11)2(8)6(12)4(1)10/h9,12H/p-2
     */
    @Test
    public void testFAGZOC() {
        IAtomContainer container = new AtomContainer(12, 12, 0, 0);
        container.addAtom(newAtm("Br", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("Br", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 4, DOUBLE);
        container.addBond(3, 5, DOUBLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 9, DOUBLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(9, 11, SINGLE);

        String[] expected = {"BR", "OM2", "O=CR", "C=C", "C=OR", "C=C", "C=OR", "C=C", "O=CR", "C=C", "OM2", "BR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAHPUZ
     *
     * @cdk.inchi InChI=1S/C12H12N2O2S/c15-17(16)7-8-5-6-10-9-3-1-2-4-11(9)14(17)12(10)13-8/h1-6,8,10,12-13H,7H2/t8-,10+,12-/m1/s1
     */
    @Test
    public void testFAHPUZ() {
        IAtomContainer container = new AtomContainer(29, 32, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 16, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 16, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(16, 26, SINGLE);
        container.addBond(16, 27, SINGLE);

        String[] expected = {"SO2N", "O2S", "O2S", "NSO2", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "CR", "C=C",
                "C=C", "CR", "NR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAHSUC
     *
     * @cdk.inchi InChI=1S/C8H11N5O/c1-9-6-5-7(12(2)4-10-6)13(3)8(14)11-5/h4H,1-3H3,(H,11,14)/p+1
     */
    @Test
    public void testFAHSUC() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 9, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(13, 15, SINGLE);

        String[] expected = {"NPYD", "CB", "NPD+", "CB", "CB", "CB", "NC=O", "CONN", "NC=O", "O=CN", "CR", "CR",
                "NC=N", "CR", "HC", "HC", "HC", "HC", "HC", "HNCO", "HC", "HC", "HNCN", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAHYUI
     *
     * @cdk.inchi InChI=1S/C9H11NOS/c1-10(11)9(12-2)8-6-4-3-5-7-8/h3-7H,1-2H3/b10-9-
     */
    @Test
    public void testFAHYUI() {
        IAtomContainer container = new AtomContainer(23, 23, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 9, DOUBLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(11, 20, SINGLE);

        String[] expected = {"S", "OXN", "N2OX", "CB", "CB", "CB", "CB", "CB", "CB", "C=N", "CR", "CR", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAHZET
     *
     * @cdk.inchi InChI=1S/C10H9NO4S/c1-7-9(8-5-3-2-4-6-8)15-16(13,14)11-10(7)12/h2-6H,1H3,(H,11,12)
     */
    @Test
    public void testFAHZET() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 4, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 8, DOUBLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(15, 20, SINGLE);

        String[] expected = {"SO3", "OSO2", "O=CN", "O2S", "O2S", "NSO3", "C=C", "C=C", "C=ON", "CR", "CB", "CB", "CB",
                "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC", "HNSO", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAJWIW
     *
     * @cdk.inchi InChI=1S/C5H7N3O3/c1-10-4-6-3(9)7-5(8-4)11-2/h1-2H3,(H,6,7,8,9)
     */
    @Test
    public void testFAJWIW() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 7, DOUBLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 9, DOUBLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 10, DOUBLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 17, SINGLE);

        String[] expected = {"OC=N", "O=CN", "OC=N", "NC=O", "N=C", "N=C", "CR", "CONN", "CR", "C=N", "C=N", "HNCO",
                "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAMHAC
     *
     * @cdk.inchi InChI=1S/C10H14NOPS/c1-8-9-6-4-5-7-10(9)13(14,12-8)11(2)3/h4-8H,1-3H3/t8-,13-/m1/s1
     */
    @Test
    public void testFAMHAC() {
        IAtomContainer container = new AtomContainer(28, 29, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(13, 26, SINGLE);
        container.addBond(13, 27, SINGLE);

        String[] expected = {"S-P", "PO", "-OP", "NR", "CR", "CR", "CR", "CB", "CB", "CB", "CB", "CB", "CB", "CR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAMYUN
     *
     * @cdk.inchi InChI=1S/C4H4BrNO2/c5-6-3(7)1-2-4(6)8/h1-2H2
     */
    @Test
    public void testFAMYUN() {
        IAtomContainer container = new AtomContainer(12, 12, 0, 0);
        container.addAtom(newAtm("Br", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 7, DOUBLE);

        String[] expected = {"BR", "NC=O", "C=ON", "O=CN", "CR", "CR", "C=ON", "O=CN", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAPLUD
     *
     * @cdk.inchi InChI=1S/C2H4N3O3PS/c6-1-3-2(7)5-9(8,10)4-1/h(H4,3,4,5,6,7,8,10)/p-1
     */
    @Test
    public void testFAPLUD() {
        IAtomContainer container = new AtomContainer(13, 13, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, DOUBLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 8, DOUBLE);
        container.addBond(5, 9, DOUBLE);

        String[] expected = {"PO2", "NC=O", "NC=O", "NC=O", "CONN", "CONN", "SM", "O2P", "O=CN", "O=CN", "HNCO",
                "HNCO", "HNCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FARMAM
     *
     * @cdk.inchi InChI=1S/C12H8N4/c1-3-7-15-9(5-1)13-11-12(15)16-8-4-2-6-10(16)14-11/h1-8H
     */
    @Test
    public void testFARMAM() {
        IAtomContainer container = new AtomContainer(24, 27, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(0, 16, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 18, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 19, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 14, DOUBLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 15, SINGLE);

        String[] expected = {"C=C", "C=C", "C=C", "C=C", "C5A", "N5B", "C5B", "N5B", "C5A", "C=C", "C=C", "C=C", "C=C",
                "NPYL", "C5A", "NPYL", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FARSOG
     *
     * @cdk.inchi InChI=1S/C12N8S2/c13-1-5(2-14)7-9-11(19-21-17-9)8(6(3-15)4-16)12-10(7)18-22-20-12
     */
    @Test
    public void testFARSOG() {
        IAtomContainer container = new AtomContainer(22, 24, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(5, 7, DOUBLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 11, TRIPLE);
        container.addBond(10, 12, TRIPLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 17, DOUBLE);
        container.addBond(14, 18, DOUBLE);
        container.addBond(15, 19, TRIPLE);
        container.addBond(16, 20, TRIPLE);
        container.addBond(17, 21, SINGLE);
        container.addBond(18, 21, SINGLE);

        String[] expected = {"STHI", "N5A", "C5B", "C5B", "N5A", "C=C", "C=C", "C=C", "C=C", "CSP", "CSP", "NSP",
                "NSP", "C5B", "C5B", "CSP", "CSP", "N5A", "N5A", "NSP", "NSP", "STHI"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FARWEA
     *
     * @cdk.inchi InChI=1S/C3H6N2S2/c1-2(4)5-3(6)7/h1H3,(H3,4,5,6,7)/p-1
     */
    @Test
    public void testFARWEA() {
        IAtomContainer container = new AtomContainer(12, 11, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, DOUBLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 11, SINGLE);

        String[] expected = {"S2CM", "S2CM", "N=C", "NC=N", "CS2M", "C=N", "CR", "HNCN", "HNCN", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FASGUB
     *
     * @cdk.inchi InChI=1S/CH8N6/c2-5-1(6-3)7-4/h2-4H2,(H2,5,6,7)
     */
    @Test
    public void testFASGUB() {
        IAtomContainer container = new AtomContainer(15, 14, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(6, 14, SINGLE);

        String[] expected = {"CGD", "N=C", "NR", "NC=N", "NR", "NC=N", "NR", "HNR", "HNR", "HNCN", "HNR", "HNR",
                "HNCN", "HNR", "HNR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FASJIS
     *
     * @cdk.inchi InChI=1S/C10H19N5O2P2S/c1-13(2)18(16)11-14(3)19(20,15(4)12-18)17-10-8-6-5-7-9-10/h5-9H,1-4H3,(H2,11,12,16)/t18-,19+
     */
    @Test
    public void testFASJIS() {
        IAtomContainer container = new AtomContainer(39, 40, 0, 0);
        container.addAtom(newAtm("P", 1));
        container.addAtom(newAtm("P", 1));
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 26, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(15, 25, SINGLE);
        container.addBond(16, 27, SINGLE);
        container.addBond(16, 28, SINGLE);
        container.addBond(16, 29, SINGLE);
        container.addBond(17, 30, SINGLE);
        container.addBond(17, 31, SINGLE);
        container.addBond(17, 32, SINGLE);
        container.addBond(18, 33, SINGLE);
        container.addBond(18, 34, SINGLE);
        container.addBond(18, 35, SINGLE);
        container.addBond(19, 36, SINGLE);
        container.addBond(19, 37, SINGLE);
        container.addBond(19, 38, SINGLE);

        String[] expected = {"PO", "PO", "S-P", "-OP", "OP", "NR", "NR", "NR", "NR", "NR", "CB", "CB", "CB", "CB",
                "CB", "CB", "CR", "CR", "CR", "CR", "HNR", "HC", "HC", "HC", "HC", "HC", "HNR", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FATLIV
     *
     * @cdk.inchi InChI=1S/C12H16N3O3P/c1-18-11-4-2-3-10(9-11)12(16)13-19(17,14-5-6-14)15-7-8-15/h2-4,9H,5-8H2,1H3,(H,13,16,17)
     */
    @Test
    public void testFATLIV() {
        IAtomContainer container = new AtomContainer(35, 37, 0, 0);
        container.addAtom(newAtm("P", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(2, 7, DOUBLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(15, 26, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(16, 28, SINGLE);
        container.addBond(16, 29, SINGLE);
        container.addBond(17, 30, SINGLE);
        container.addBond(17, 31, SINGLE);
        container.addBond(18, 32, SINGLE);
        container.addBond(18, 33, SINGLE);
        container.addBond(18, 34, SINGLE);

        String[] expected = {"PO", "OP", "O=CN", "OC=C", "NC=O", "NR", "NR", "C=ON", "CB", "CB", "CB", "CB", "CB",
                "CB", "CR3R", "CR3R", "CR3R", "CR3R", "CR", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAXFUF10
     *
     * @cdk.inchi InChI=1S/C5H4N4O/c10-5-3-1-8-9-4(3)6-2-7-5/h1-2H,(H2,6,7,8,9,10)/p+1
     */
    @Test
    public void testFAXFUF10() {
        IAtomContainer container = new AtomContainer(15, 16, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, DOUBLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 6, DOUBLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 9, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(9, 13, SINGLE);

        String[] expected = {"O=CN", "NCN+", "NCN+", "NPYL", "N5A", "CNN+", "C5B", "C5B", "C=ON", "C5A", "HNN+", "HC",
                "HNN+", "HC", "HPYL"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAXVAB
     *
     * @cdk.inchi InChI=1S/C8H8N2O3S/c1-10-8(11)6-4-2-3-5-7(6)9-14(10,12)13/h2-5,9H,1H3
     */
    @Test
    public void testFAXVAB() {
        IAtomContainer container = new AtomContainer(22, 23, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(3, 12, DOUBLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(13, 20, SINGLE);
        container.addBond(13, 19, SINGLE);

        String[] expected = {"SO2N", "O2S", "O2S", "O=CN", "NSO2", "NSO2", "CB", "CB", "CB", "CB", "CB", "CB", "C=ON",
                "CR", "HNSO", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAXVEF
     *
     * @cdk.inchi InChI=1S/C9H14N/c1-2-3-7-10-8-5-4-6-9-10/h4-6,8-9H,2-3,7H2,1H3/q+1
     */
    @Test
    public void testFAXVEF() {
        IAtomContainer container = new AtomContainer(24, 24, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(9, 21, SINGLE);

        String[] expected = {"NPD+", "CB", "CB", "CB", "CB", "CB", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAXVIJ
     *
     * @cdk.inchi InChI=1S/C12H6F3N3O2/c13-12(14,15)7-5-8-11(9(6-7)18(19)20)17-4-2-1-3-10(17)16-8/h1-6H
     */
    @Test
    public void testFAXVIJ() {
        IAtomContainer container = new AtomContainer(26, 28, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 20, SINGLE);
        container.addBond(0, 19, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 21, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 22, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 23, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 18, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 18, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 17, DOUBLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(18, 19, SINGLE);

        String[] expected = {"C=C", "C=C", "C=C", "C=C", "C5A", "N5B", "C5B", "CB", "CB", "CR", "F", "F", "F", "CB",
                "CB", "NO2", "O2N", "O2N", "C5A", "NPYL", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAZBAJ
     *
     * @cdk.inchi InChI=1S/C5H4N2OS/c1-9-5(8)4(2-6)3-7/h8H,1H3/p-1
     */
    @Test
    public void testFAZBAJ() {
        IAtomContainer container = new AtomContainer(12, 11, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(4, 6, TRIPLE);
        container.addBond(5, 7, TRIPLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 11, SINGLE);

        String[] expected = {"S", "OM2", "C=C", "C=C", "CSP", "CSP", "NSP", "NSP", "CR", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FAZKUM
     *
     * @cdk.inchi InChI=1S/C2Cl8Si2/c3-1(11(5,6)7)2(4)12(8,9)10/b2-1+
     */
    @Test
    public void testFAZKUM() {
        IAtomContainer container = new AtomContainer(12, 11, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 11, SINGLE);

        String[] expected = {"CL", "CL", "CL", "CL", "SI", "C=C", "C=C", "CL", "SI", "CL", "CL", "CL"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FBATNB
     *
     * @cdk.inchi InChI=1S/C10H11FN4O6/c1-10(2,3)12(11)9-7(14(18)19)4-6(13(16)17)5-8(9)15(20)21/h4-5H,1-3H3
     */
    @Test
    public void testFBATNB() {
        IAtomContainer container = new AtomContainer(32, 32, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addBond(0, 22, SINGLE);
        container.addBond(0, 6, DOUBLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 21, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(2, 23, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 24, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 25, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(23, 27, DOUBLE);
        container.addBond(23, 26, SINGLE);
        container.addBond(24, 29, DOUBLE);
        container.addBond(24, 28, SINGLE);
        container.addBond(25, 31, DOUBLE);
        container.addBond(25, 30, SINGLE);

        String[] expected = {"CB", "CR", "CB", "CB", "CB", "CB", "CB", "CR", "CR", "CR", "F", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "NC=C", "NO2", "NO2", "NO2", "O2N", "O2N", "O2N", "O2N",
                "O2N", "O2N"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FECXEQ
     *
     * @cdk.inchi InChI=1S/C9H12N2O/c1-7(2)8-4-5-11(3)10-6-9(8)12/h4-6H,1-3H3
     */
    @Test
    public void testFECXEQ() {
        IAtomContainer container = new AtomContainer(24, 24, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 10, DOUBLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 11, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(9, 10, SINGLE);

        String[] expected = {"C=N", "C=OR", "C=C", "C=C", "C=C", "C=C", "CR", "CR", "CR", "NC=C", "N=C", "O=CR", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEGSEP
     *
     * @cdk.inchi InChI=1S/C4H8NO4P/c6-3-1-2-4(5-3)10(7,8)9/h4H,1-2H2,(H,5,6)(H2,7,8,9)/t4-/m0/s1
     */
    @Test
    public void testFEGSEP() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("P", 1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(4, 9, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 17, SINGLE);

        String[] expected = {"PO3", "OPO2", "OPO2", "OP", "O=CN", "NC=O", "CR", "CR", "CR", "C=ON", "HOP", "HOP", "HC",
                "HNCO", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEHDAX
     *
     * @cdk.inchi InChI=1S/C5H7NO/c1-3-2-4(3)5(6)7/h4H,1-2H2,(H2,6,7)/t4-/m0/s1
     */
    @Test
    public void testFEHDAX() {
        IAtomContainer container = new AtomContainer(14, 14, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 13, SINGLE);

        String[] expected = {"C=C", "CR3R", "CR3R", "C=C", "C=ON", "O=CN", "NC=O", "HC", "HC", "HC", "HC", "HC",
                "HNCO", "HNCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEJJEJ
     *
     * @cdk.inchi InChI=1S/C12H11NO/c1-2-10-11(13-12(10)14)8-9-6-4-3-5-7-9/h3-7,11H,1,8H2,(H,13,14)/t11-/m1/s1
     */
    @Test
    public void testFEJJEJ() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(13, 24, SINGLE);

        String[] expected = {"O=CN", "C=ON", "CE4R", "CR4R", "NC=O", "=C=", "C=C", "CR", "CB", "CB", "CB", "CB", "CB",
                "CB", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEJKIO
     *
     * @cdk.inchi InChI=1S/C12H22N2/c1-2-11(1)9-13-5-7-14(8-6-13)10-12-3-4-12/h11-12H,1-10H2/p+2
     */
    @Test
    public void testFEJKIO() {
        IAtomContainer container = new AtomContainer(38, 40, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(19, 21, SINGLE);
        container.addBond(19, 22, SINGLE);
        container.addBond(19, 23, SINGLE);
        container.addBond(20, 21, SINGLE);
        container.addBond(20, 24, SINGLE);
        container.addBond(20, 25, SINGLE);
        container.addBond(21, 26, SINGLE);
        container.addBond(21, 27, SINGLE);
        container.addBond(26, 28, SINGLE);
        container.addBond(26, 29, SINGLE);
        container.addBond(26, 30, SINGLE);
        container.addBond(28, 31, SINGLE);
        container.addBond(28, 32, SINGLE);
        container.addBond(28, 33, SINGLE);
        container.addBond(31, 32, SINGLE);
        container.addBond(31, 34, SINGLE);
        container.addBond(31, 35, SINGLE);
        container.addBond(32, 36, SINGLE);
        container.addBond(32, 37, SINGLE);

        String[] expected = {"NR+", "CR", "CR", "CR", "CR3R", "CR3R", "CR3R", "HNR+", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "CR", "CR", "NR+", "HC", "HC", "HC", "HC", "CR", "HNR+", "CR3R",
                "HC", "HC", "CR3R", "CR3R", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FELYIE
     *
     * @cdk.inchi InChI=1S/C7H9NO2S/c9-4-3-8-7(11)6-2-1-5-10-6/h1-2,5,9H,3-4H2,(H,8,11)
     */
    @Test
    public void testFELYIE() {
        IAtomContainer container = new AtomContainer(20, 20, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, DOUBLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 19, SINGLE);

        String[] expected = {"S=C", "OFUR", "OR", "NC=S", "C=SN", "C5A", "C5B", "C5B", "C5A", "CR", "CR", "HC", "HC",
                "HC", "HNCS", "HOR", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FELYUQ
     *
     * @cdk.inchi InChI=1S/C10H11NO3/c1-7(12)9-3-5-10(6-4-9)11(14)8(2)13/h3-6,14H,1-2H3
     */
    @Test
    public void testFELYUQ() {
        IAtomContainer container = new AtomContainer(25, 25, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 22, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "C=OR", "O=CR", "CR", "NC=O", "-O-", "C=ON", "O=CN",
                "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HO", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEMGEJ
     *
     * @cdk.inchi InChI=1S/C10H14O2/c11-10(12)9-7-5-3-1-2-4-6-8(7)9/h3,5,7-9H,1-2,4,6H2,(H,11,12)/p-1/b5-3-/t7-,8+,9+/m1/s1
     */
    @Test
    public void testFEMGEJ() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 11, DOUBLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 24, SINGLE);

        String[] expected = {"O2CM", "O2CM", "CR3R", "C=C", "C=C", "CR", "CR", "CR", "CR", "CR3R", "CR3R", "CO2M",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FENCOQ
     *
     * @cdk.inchi InChI=1S/C5H7N3O3/c1-3(6-9)5-4(2)8(10)11-7-5/h9H,1-2H3/b6-3+
     */
    @Test
    public void testFENCOQ() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 6, DOUBLE);
        container.addBond(4, 7, DOUBLE);
        container.addBond(5, 9, DOUBLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 14, SINGLE);

        String[] expected = {"OFUR", "OXN", "-O-", "N5AX", "N5A", "N=C", "C5B", "C5B", "CR", "C=N", "CR", "HO", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FENHAH
     *
     * @cdk.inchi InChI=1S/C5H7N3O2/c1-3-7-4(8-10-3)5(9)6-2/h1-2H3,(H,6,9)
     */
    @Test
    public void testFENHAH() {
        IAtomContainer container = new AtomContainer(17, 17, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(9, 16, SINGLE);

        String[] expected = {"C5B", "N5B", "C5A", "OFUR", "N5A", "C=ON", "O=CN", "NC=O", "CR", "CR", "HNCO", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FENJIR
     *
     * @cdk.inchi InChI=1S/C9H17N3S2/c1-8(2)9(11(3)4)12(5-6-14-9)7(13)10-8/h5-6H2,1-4H3,(H,10,13)/t9-/m1/s1
     */
    @Test
    public void testFENJIR() {
        IAtomContainer container = new AtomContainer(31, 32, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(2, 17, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(13, 29, SINGLE);
        container.addBond(13, 30, SINGLE);

        String[] expected = {"S", "CR", "CR", "NC=S", "C=SN", "S=C", "NC=S", "CR", "CR", "NR", "CR", "CR", "CR", "CR",
                "HC", "HC", "HC", "HC", "HNCS", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FENJOX
     *
     * @cdk.inchi InChI=1S/C10H19N3S2/c1-5-10(2,8-11-6-7-15-8)12-9(14)13(3)4/h5-7H2,1-4H3,(H,12,14)/t10-/m0/s1
     */
    @Test
    public void testFENJOX() {
        IAtomContainer container = new AtomContainer(34, 34, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 33, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 29, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(13, 27, SINGLE);
        container.addBond(14, 32, SINGLE);
        container.addBond(14, 31, SINGLE);
        container.addBond(14, 30, SINGLE);

        String[] expected = {"S", "C=N", "N=C", "CR", "CR", "CR", "CR", "CR", "CR", "NC=S", "C=SN", "S=C", "NC=S",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCS", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FENJUD
     *
     * @cdk.inchi InChI=1S/C9H14N2OS2/c12-7-9(3-1-2-4-9)10-8(14)11(7)5-6-13/h13H,1-6H2,(H,10,14)
     */
    @Test
    public void testFENJUD() {
        IAtomContainer container = new AtomContainer(28, 29, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(13, 27, SINGLE);

        String[] expected = {"NC=S", "C=SN", "S=C", "NC=O", "C=ON", "O=CN", "CR", "CR", "CR", "CR", "CR", "CR", "CR",
                "S", "HNCS", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HS"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FENNUH
     *
     * @cdk.inchi InChI=1S/C5H13N2P/c1-6(2)5(8)7(3)4/h8H,1-4H3
     */
    @Test
    public void testFENNUH() {
        IAtomContainer container = new AtomContainer(21, 20, 0, 0);
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 18, SINGLE);

        String[] expected = {"-P=C", "NC=P", "NC=P", "C=P", "CR", "CR", "CR", "CR", "HP", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FENYIG
     *
     * @cdk.inchi InChI=1S/C12H16N4O4/c1-12(2,3)20-11(19)7-5-4-6-15-9(17)8(14-13)10(18)16(7)15/h7H,4-6H2,1-3H3/t7-/m1/s1
     */
    @Test
    public void testFENYIG() {
        IAtomContainer container = new AtomContainer(36, 37, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(2, 11, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 19, SINGLE);
        container.addBond(16, 18, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(17, 29, SINGLE);
        container.addBond(17, 28, SINGLE);
        container.addBond(17, 27, SINGLE);
        container.addBond(18, 32, SINGLE);
        container.addBond(18, 31, SINGLE);
        container.addBond(18, 30, SINGLE);
        container.addBond(19, 35, SINGLE);
        container.addBond(19, 34, SINGLE);
        container.addBond(19, 33, SINGLE);

        String[] expected = {"C=ON", "O=CN", "C=N", "C=ON", "O=CN", "NC=O", "CR", "CR", "CR", "CR", "NC=O", "=N=",
                "NAZT", "COO", "O=CO", "OC=O", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEPWAY
     *
     * @cdk.inchi InChI=1S/C3H6N2S2/c1-2(4)5-3(6)7/h1H3,(H3,4,5,6,7)
     */
    @Test
    public void testFEPWAY() {
        IAtomContainer container = new AtomContainer(13, 12, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, DOUBLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 5, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 12, SINGLE);

        String[] expected = {"S2CM", "S2CM", "NCN+", "NCN+", "CS2M", "CNN+", "CR", "HNN+", "HNN+", "HNN+", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEPWOM
     *
     * @cdk.inchi InChI=1S/C4F4N2S2/c5-3(6,11-1-9)4(7,8)12-2-10
     */
    @Test
    public void testFEPWOM() {
        IAtomContainer container = new AtomContainer(12, 11, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("F", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 5, TRIPLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(10, 11, TRIPLE);

        String[] expected = {"S", "F", "F", "NSP", "CR", "CSP", "CR", "S", "F", "F", "CSP", "NSP"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FESCAH
     *
     * @cdk.inchi InChI=1S/C3H3NO3/c5-1-4(2-6)3-7/h1-3H
     */
    @Test
    public void testFESCAH() {
        IAtomContainer container = new AtomContainer(10, 9, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(4, 6, DOUBLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 8, DOUBLE);
        container.addBond(5, 9, SINGLE);

        String[] expected = {"NC=O", "C=ON", "O=CN", "HC", "C=ON", "C=ON", "O=CN", "HC", "O=CN", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FESMIZ
     *
     * @cdk.inchi InChI=1S/C9H9Cl3N2O2/c10-9(11,12)8(15)16-14-7(13)6-4-2-1-3-5-6/h1-5,8,15H,(H2,13,14)/t8-/m1/s1
     */
    @Test
    public void testFESMIZ() {
        IAtomContainer container = new AtomContainer(25, 25, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 12, DOUBLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(14, 24, SINGLE);

        String[] expected = {"CL", "CL", "CL", "CR", "CR", "C=N", "CB", "CB", "CB", "CB", "CB", "CB", "N=C", "NC=N",
                "OR", "OR", "HC", "HC", "HC", "HC", "HC", "HC", "HNCN", "HNCN", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FETRUR
     *
     * @cdk.inchi InChI=1S/C6H12N4/c1-7-2-9-4-8(1)5-10(3-7)6-9/h1-6H2/p+1
     */
    @Test
    public void testFETRUR() {
        IAtomContainer container = new AtomContainer(23, 25, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(13, 20, SINGLE);
        container.addBond(20, 21, SINGLE);
        container.addBond(20, 22, SINGLE);

        String[] expected = {"NR+", "CR", "NR", "CR", "HNR+", "HC", "HC", "HC", "CR", "CR", "HC", "CR", "NR", "NR",
                "HC", "HC", "HC", "HC", "HC", "HC", "CR", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FETWOQ
     *
     * @cdk.inchi InChI=1S/C11H13NO5/c13-5-3-4-8(11(5)17)9(10(4)16)12-6(14)1-2-7(12)15/h4-5,8-9,11,13,17H,1-3H2/t4-,5-,8+,9+,11-/m1/s1
     */
    @Test
    public void testFETWOQ() {
        IAtomContainer container = new AtomContainer(30, 32, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 17, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 21, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 23, SINGLE);
        container.addBond(6, 14, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 15, DOUBLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(9, 27, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 16, DOUBLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(13, 29, SINGLE);

        String[] expected = {"CR4R", "CR", "CR", "CR", "CR4R", "CR4R", "C=OR", "C=ON", "CR", "CR", "C=ON", "NC=O",
                "OR", "OR", "O=CR", "O=CN", "O=CN", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HOR", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEVNUP
     *
     * @cdk.inchi InChI=1S/C5H6Cl3N3O2/c1-11-3(12)5(6,7)2(10-8)9-4(11)13/h3,12H,1H3,(H,9,10,13)/t3-/m0/s1
     */
    @Test
    public void testFEVNUP() {
        IAtomContainer container = new AtomContainer(19, 19, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(12, 18, SINGLE);

        String[] expected = {"CL", "CL", "CL", "NC=O", "CR", "CONN", "O=CN", "NC=O", "C=N", "N=C", "CR", "CR", "OR",
                "HC", "HC", "HC", "HNCO", "HC", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEYLUQ
     *
     * @cdk.inchi InChI=1S/C4H4N6O5/c11-9(12)7-1-2-8(10(13)14)4-3(7)5-15-6-4/h1-2H2
     */
    @Test
    public void testFEYLUQ() {
        IAtomContainer container = new AtomContainer(19, 20, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 9, DOUBLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(4, 10, DOUBLE);
        container.addBond(5, 13, DOUBLE);
        container.addBond(6, 14, DOUBLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(13, 14, SINGLE);

        String[] expected = {"OFUR", "O2N", "O2N", "O2N", "O2N", "N5A", "N5A", "NC=N", "NC=N", "NO2", "NO2", "CR",
                "CR", "C5B", "C5B", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEZPOP
     *
     * @cdk.inchi InChI=1S/C2H4N2O2S2/c5-1(7)3-4-2(6)8/h(H2,3,5,7)(H2,4,6,8)/p-2
     */
    @Test
    public void testFEZPOP() {
        IAtomContainer container = new AtomContainer(10, 9, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", -1));
        container.addBond(0, 3, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(6, 9, SINGLE);

        String[] expected = {"S=C", "OM2", "NC=S", "C=SN", "HNCS", "NC=S", "C=SN", "HNCS", "S=C", "OM2"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FEZRUX
     *
     * @cdk.inchi InChI=1S/C12H11N5O/c1-8-14-15-12-13-10(7-11(18)17(12)16-8)9-5-3-2-4-6-9/h2-7H,1H3,(H,13,15)(H,14,16)
     */
    @Test
    public void testFEZRUX() {
        IAtomContainer container = new AtomContainer(29, 31, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 18, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 8, DOUBLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(6, 21, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 26, SINGLE);
        container.addBond(16, 17, DOUBLE);
        container.addBond(16, 27, SINGLE);
        container.addBond(17, 28, SINGLE);

        String[] expected = {"O=CN", "NC=O", "NC=N", "N=C", "NC=N", "N=C", "CR", "C=N", "CGD", "C=ON", "C=C", "C=C",
                "CB", "CB", "CB", "CB", "CB", "CB", "HNCN", "HNCN", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIBLIL
     *
     * @cdk.inchi InChI=1S/C11H12N2O4/c14-11(12-7-1-2-8-17-12)9-3-5-10(6-4-9)13(15)16/h3-6H,1-2,7-8H2
     */
    @Test
    public void testFIBLIL() {
        IAtomContainer container = new AtomContainer(29, 30, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 21, SINGLE);
        container.addBond(2, 22, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 23, SINGLE);
        container.addBond(3, 24, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 25, SINGLE);
        container.addBond(4, 26, SINGLE);
        container.addBond(5, 27, SINGLE);
        container.addBond(5, 28, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(13, 20, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 16, DOUBLE);

        String[] expected = {"NC=O", "OR", "CR", "CR", "CR", "CR", "C=ON", "O=CN", "CB", "CB", "CB", "CB", "CB", "CB",
                "NO2", "O2N", "O2N", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FICDOK
     *
     * @cdk.inchi InChI=1S/C5H7N3O3/c1-7-3-6-4(8(9)10)5(7)11-2/h3H,1-2H3
     */
    @Test
    public void testFICDOK() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, DOUBLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(10, 17, SINGLE);

        String[] expected = {"NPYL", "CR", "C5A", "N5B", "C5B", "NO2", "O2N", "O2N", "C5A", "OC=C", "CR", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIFGUW
     *
     * @cdk.inchi InChI=1S/C11H20O2S/c1-9(2)6-10(3,4)8-11(7-9,13-8)14(5)12/h8H,6-7H2,1-5H3/t8-,11-,14-/m1/s1
     */
    @Test
    public void testFIFGUW() {
        IAtomContainer container = new AtomContainer(34, 35, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(10, 24, SINGLE);
        container.addBond(11, 25, SINGLE);
        container.addBond(11, 26, SINGLE);
        container.addBond(11, 27, SINGLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(12, 29, SINGLE);
        container.addBond(12, 30, SINGLE);
        container.addBond(13, 31, SINGLE);
        container.addBond(13, 32, SINGLE);
        container.addBond(13, 33, SINGLE);

        String[] expected = {"S=O", "O=S", "OR", "CR3R", "CR3R", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIGYID
     *
     * @cdk.inchi InChI=1S/C3H7N/c4-3-1-2-3/h3H,1-2,4H2
     */
    @Test
    public void testFIGYID() {
        IAtomContainer container = new AtomContainer(11, 11, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 10, SINGLE);

        String[] expected = {"CR3R", "CR3R", "CR3R", "NR", "HC", "HC", "HC", "HC", "HC", "HNR", "HNR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIHXID
     *
     * @cdk.inchi InChI=1S/C11H13BrN2O5/c12-2-1-6-4-14(11(18)13-10(6)17)9-3-7(16)8(5-15)19-9/h1-2,4,7-9,15-16H,3,5H2,(H,13,17,18)/b2-1-/t7-,8+,9+/m0/s1
     */
    @Test
    public void testFIHXID() {
        IAtomContainer container = new AtomContainer(32, 33, 0, 0);
        container.addAtom(newAtm("Br", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 18, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 9, DOUBLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 30, SINGLE);
        container.addBond(4, 12, DOUBLE);
        container.addBond(5, 15, SINGLE);
        container.addBond(5, 31, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(6, 16, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(7, 25, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(10, 24, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 26, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(13, 27, SINGLE);
        container.addBond(14, 16, DOUBLE);
        container.addBond(14, 17, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(15, 29, SINGLE);
        container.addBond(16, 19, SINGLE);
        container.addBond(17, 18, DOUBLE);
        container.addBond(17, 20, SINGLE);
        container.addBond(18, 21, SINGLE);

        String[] expected = {"BR", "OR", "O=CN", "OR", "O=CN", "OR", "NC=O", "NC=O", "CR", "CONN", "CR", "CR", "C=ON",
                "CR", "C=C", "CR", "C=C", "C=C", "C=C", "HC", "HC", "HC", "HC", "HC", "HC", "HNCO", "HC", "HC", "HC",
                "HC", "HOR", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIKJAK
     *
     * @cdk.inchi InChI=1S/C11H6N6/c12-2-8-1-10(4-13,5-14)3-11(6-15,7-16)9(8)17/h1,3,17H2
     */
    @Test
    public void testFIKJAK() {
        IAtomContainer container = new AtomContainer(23, 23, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 19, SINGLE);
        container.addBond(2, 20, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 21, SINGLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 12, TRIPLE);
        container.addBond(7, 13, TRIPLE);
        container.addBond(8, 14, TRIPLE);
        container.addBond(9, 15, TRIPLE);
        container.addBond(10, 16, TRIPLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 18, SINGLE);

        String[] expected = {"C=C", "C=C", "CR", "CR", "CR", "CR", "CSP", "CSP", "CSP", "CSP", "CSP", "NC=C", "NSP",
                "NSP", "NSP", "NSP", "NSP", "HNCC", "HNCC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIKZOO10
     *
     * @cdk.inchi InChI=1S/C9H7NS2/c11-9-10-8(6-12-9)7-4-2-1-3-5-7/h1-6H,(H,10,11)
     */
    @Test
    public void testFIKZOO10() {
        IAtomContainer container = new AtomContainer(19, 20, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 11, DOUBLE);
        container.addBond(9, 18, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "C=C", "C=C", "C=SN", "NC=S", "S", "S=C", "HC", "HC",
                "HC", "HC", "HC", "HC", "HNCS"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FILGEM
     *
     * @cdk.inchi InChI=1S/C12H9N3O/c1-8-11(9-2-4-14-5-3-9)6-10(7-13)12(16)15-8/h2-6H,1H3,(H,15,16)
     */
    @Test
    public void testFILGEM() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 12, SINGLE);
        container.addBond(2, 15, DOUBLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(6, 13, TRIPLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(9, 14, DOUBLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(12, 16, SINGLE);

        String[] expected = {"C=C", "C=C", "C=ON", "C=C", "C=C", "CR", "CSP", "CB", "CB", "CB", "CB", "CB", "NC=O",
                "NSP", "NPYD", "O=CN", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FILNOD
     *
     * @cdk.inchi InChI=1S/C9H11NO2S/c1-9(2)7-5-3-4-6-8(7)13(12)10(9)11/h3-6,11H,1-2H3/t13-/m1/s1
     */
    @Test
    public void testFILNOD() {
        IAtomContainer container = new AtomContainer(24, 25, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(12, 23, SINGLE);

        String[] expected = {"S=O", "O=S", "-O-", "NR", "CR", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "CR", "HO",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FINBIN
     *
     * @cdk.inchi InChI=1S/C6H3Cl9O3P2/c7-4(8,9)1-16-20-17-2(5(10,11)12)19(1)3(18-20)6(13,14)15/h1-3H/t1-,2-,3-,19-,20+/m0/s1
     */
    @Test
    public void testFINBIN() {
        IAtomContainer container = new AtomContainer(23, 24, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 17, SINGLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(2, 17, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(13, 16, SINGLE);
        container.addBond(14, 17, SINGLE);
        container.addBond(14, 20, SINGLE);
        container.addBond(15, 18, SINGLE);
        container.addBond(15, 21, SINGLE);
        container.addBond(16, 19, SINGLE);
        container.addBond(16, 22, SINGLE);

        String[] expected = {"CL", "CL", "CL", "CL", "CL", "CL", "CL", "CL", "CL", "P", "P", "OPO2", "OPO2", "OPO2",
                "CR", "CR", "CR", "CR", "CR", "CR", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FINPEX
     *
     * @cdk.inchi InChI=1S/C6H11NO3/c7-6(5(9)10)1-4(2-6)3-8/h4,8H,1-3,7H2,(H,9,10)/p+1/t4-,6+
     */
    @Test
    public void testFINPEX() {
        IAtomContainer container = new AtomContainer(22, 22, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 8, SINGLE);
        container.addBond(0, 21, SINGLE);
        container.addBond(1, 8, DOUBLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 17, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(3, 19, SINGLE);
        container.addBond(3, 20, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(9, 16, SINGLE);

        String[] expected = {"OC=O", "O=CO", "OR", "NR+", "CR4R", "CR4R", "CR4R", "CR4R", "COO", "CR", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HOR", "HNR+", "HNR+", "HNR+", "HOCO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FITGIY
     *
     * @cdk.inchi InChI=1S/C9H14N2O3/c1-4-14-8(12)6-5(2)11-9(3,13)7(6)10/h13H,4,10H2,1-3H3/t9-/m0/s1
     */
    @Test
    public void testFITGIY() {
        IAtomContainer container = new AtomContainer(28, 28, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 9, DOUBLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(13, 27, SINGLE);

        String[] expected = {"N=C", "C=N", "CR", "C=C", "COO", "O=CO", "OC=O", "CR", "CR", "C=C", "NC=C", "CR", "CR",
                "OR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCC", "HNCC", "HC", "HC", "HC", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FITSEG
     *
     * @cdk.inchi InChI=1S/C6H3N3/c7-1-4-5(2-8)6(4)3-9/h4-6H/t4-,5-,6-
     */
    @Test
    public void testFITSEG() {
        IAtomContainer container = new AtomContainer(12, 12, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 8, SINGLE);
        container.addBond(1, 2, TRIPLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 5, TRIPLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(9, 11, TRIPLE);

        String[] expected = {"CR3R", "CSP", "NSP", "CR3R", "CSP", "NSP", "HC", "HC", "CR3R", "CSP", "HC", "NSP"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FITTIL
     *
     * @cdk.inchi InChI=1S/C6H7N5/c1-11-3-10-5(7)4-6(11)9-2-8-4/h2-3H,1H3,(H2,7,8,9)/p+1
     */
    @Test
    public void testFITTIL() {
        IAtomContainer container = new AtomContainer(19, 20, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(10, 16, SINGLE);

        String[] expected = {"NPYD", "CB", "NPD+", "C5B", "C5A", "CB", "NPYL", "C5A", "N5B", "NC=N", "CR", "HC",
                "HPYL", "HNCN", "HNCN", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIVNUT
     *
     * @cdk.inchi InChI=1S/C12H15N3OS/c1-7-5-15(3)6-10(11(7)16)8(2)9(4-13)12(14)17/h5-6,8-9H,1-3H3,(H2,14,17)/t8-,9-/m1/s1
     */
    @Test
    public void testFIVNUT() {
        IAtomContainer container = new AtomContainer(32, 32, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 14, DOUBLE);
        container.addBond(1, 7, DOUBLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(4, 15, TRIPLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 24, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(13, 26, SINGLE);
        container.addBond(13, 27, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(16, 29, SINGLE);
        container.addBond(16, 30, SINGLE);
        container.addBond(16, 31, SINGLE);

        String[] expected = {"S=C", "O=CR", "NC=C", "NC=S", "NSP", "C=C", "C=C", "C=OR", "C=C", "C=C", "CR", "CR",
                "CR", "CR", "C=SN", "CSP", "CR", "HNCS", "HNCS", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIVRAD
     *
     * @cdk.inchi InChI=1S/C12H15N3OS/c1-2-3-8-13-12(16)14-11-9-6-4-5-7-10(9)17-15-11/h4-7H,2-3,8H2,1H3,(H2,13,14,15,16)
     */
    @Test
    public void testFIVRAD() {
        IAtomContainer container = new AtomContainer(32, 33, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 12, DOUBLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 31, SINGLE);
        container.addBond(16, 30, SINGLE);
        container.addBond(16, 29, SINGLE);

        String[] expected = {"STHI", "O=CN", "N5A", "NC=O", "NC=O", "C5B", "C5B", "C5A", "CB", "CB", "CB", "CB",
                "CONN", "CR", "CR", "CR", "CR", "HNCO", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIXPIL
     *
     * @cdk.inchi InChI=1S/C11H10N2O3/c1-7-3-5-9(6-4-7)13-10(8(2)14)11(15)16-12-13/h3-6H,1-2H3
     */
    @Test
    public void testFIXPIL() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 17, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 15, DOUBLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(10, 24, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(12, 13, SINGLE);

        String[] expected = {"CB", "CB", "CB", "CB", "CB", "CB", "C5B", "C5A", "C=OR", "CR", "CR", "N5B+", "N5A",
                "OFUR", "OM2", "O=CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIYBIY
     *
     * @cdk.inchi InChI=1S/C10H11NO3S/c1-8-7-15(13,11-10(12)14-8)9-5-3-2-4-6-9/h2-6,8H,7H2,1H3/t8-,15-/m1/s1
     */
    @Test
    public void testFIYBIY() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 13, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 4, DOUBLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(2, 11, DOUBLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(14, 23, SINGLE);

        String[] expected = {"SNO", "O-S", "O=CN", "OC=O", "NSO", "CB", "CB", "CB", "CB", "CB", "CB", "C=ON", "CR",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIZGEA
     *
     * @cdk.inchi InChI=1S/C11H7Cl4NO2S2/c1-6-2-4-7(5-3-6)20(17,18)16-19-10(14)8(12)9(13)11(19)15/h2-5H,1H3
     */
    @Test
    public void testFIZGEA() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 12, DOUBLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 20, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 21, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(16, 19, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(17, 22, SINGLE);
        container.addBond(17, 18, DOUBLE);
        container.addBond(18, 23, SINGLE);
        container.addBond(19, 26, SINGLE);
        container.addBond(19, 25, SINGLE);
        container.addBond(19, 24, SINGLE);

        String[] expected = {">S=N", "C=C", "C=C", "C=C", "C=C", "CL", "CL", "CL", "CL", "NSO2", "SO2N", "O2S", "O2S",
                "CB", "CB", "CB", "CB", "CB", "CB", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIZGOK
     *
     * @cdk.inchi InChI=1S/C7H5Cl4NO3S/c1-2-15-7(13)12-16(14)5(10)3(8)4(9)6(16)11/h2H2,1H3
     */
    @Test
    public void testFIZGOK() {
        IAtomContainer container = new AtomContainer(21, 21, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 9, DOUBLE);
        container.addBond(0, 15, DOUBLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 16, SINGLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(14, 18, SINGLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(14, 20, SINGLE);

        String[] expected = {"SNO", "C=C", "C=C", "C=C", "C=C", "CL", "CL", "CL", "CL", "NSO", "C=ON", "O=CN", "OC=O",
                "CR", "CR", "O-S", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FIZJED
     *
     * @cdk.inchi InChI=1S/C4H6N4O2/c1-9-3-5-7-4(10-2)8-6-3/h1-2H3
     */
    @Test
    public void testFIZJED() {
        IAtomContainer container = new AtomContainer(16, 16, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 8, DOUBLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 13, SINGLE);

        String[] expected = {"NPYD", "NPYD", "CB", "OC=N", "CR", "HC", "HC", "HC", "CB", "NPYD", "NPYD", "OC=N", "CR",
                "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOBJUB01
     *
     * @cdk.inchi InChI=1S/C4H7NO2/c5-4(1-2-4)3(6)7/h1-2,5H2,(H,6,7)
     */
    @Test
    public void testFOBJUB01() {
        IAtomContainer container = new AtomContainer(14, 14, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 13, SINGLE);

        String[] expected = {"O2CM", "O2CM", "NR+", "CO2M", "CR3R", "CR3R", "CR3R", "HNR+", "HNR+", "HNR+", "HC", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FODTUN
     *
     * @cdk.inchi InChI=1S/C9H14N2O2/c1-6(12)11-5-7-3-9(11,4-7)8(13)10-2/h7H,3-5H2,1-2H3,(H,10,13)/t7-,9+
     */
    @Test
    public void testFODTUN() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 13, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 26, SINGLE);

        String[] expected = {"CR", "C=ON", "O=CN", "NC=O", "CR4R", "CR4R", "CR4R", "CR4R", "CR", "C=ON", "O=CN",
                "NC=O", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCO", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOGBIM
     *
     * @cdk.inchi InChI=1S/C8H16NOSi/c1-11(2)7-9-6-4-3-5-8(9)10-11/h3-7H2,1-2H3/q+1
     */
    @Test
    public void testFOGBIM() {
        IAtomContainer container = new AtomContainer(27, 28, 0, 0);
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 10, DOUBLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 22, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(9, 10, SINGLE);

        String[] expected = {"SI", "OC=N", "N+=C", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "C=N", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOGVIG01
     *
     * @cdk.inchi InChI=1S/C8H15N7O2S3/c9-6(15-20(12,16)17)1-2-18-3-5-4-19-8(13-5)14-7(10)11/h4H,1-3H2,(H2,9,15)(H2,12,16,17)(H4,10,11,13,14)
     */
    @Test
    public void testFOGVIG01() {
        IAtomContainer container = new AtomContainer(35, 35, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 20, SINGLE);
        container.addBond(0, 21, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 22, SINGLE);
        container.addBond(1, 23, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 25, SINGLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 27, SINGLE);
        container.addBond(11, 28, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 29, SINGLE);
        container.addBond(12, 30, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(15, 17, DOUBLE);
        container.addBond(15, 18, SINGLE);
        container.addBond(18, 31, SINGLE);
        container.addBond(18, 32, SINGLE);
        container.addBond(19, 33, SINGLE);
        container.addBond(19, 34, SINGLE);

        String[] expected = {"NC=N", "NC=N", "CGD", "N=C", "C5A", "N5B", "C5B", "C5A", "STHI", "CR", "S", "CR", "CR",
                "C=N", "N=C", "SO2N", "O2S", "O2S", "NSO2", "NC=N", "HNCN", "HNCN", "HNCN", "HNCN", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HNSO", "HNSO", "HNCN", "HNCN"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOHXEF
     *
     * @cdk.inchi InChI=1S/C8H12N4S2/c1-2-5-14-7-10-6-9-3-4-12(6)8(13)11-7/h2-5H2,1H3,(H,9,10,11,13)
     */
    @Test
    public void testFOHXEF() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 7, DOUBLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 14, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(9, 16, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(13, 23, SINGLE);
        container.addBond(13, 24, SINGLE);
        container.addBond(13, 25, SINGLE);

        String[] expected = {"S=C", "S", "NC=N", "CGD", "N=C", "C=N", "N=C", "C=SN", "NC=S", "CR", "CR", "CR", "CR",
                "CR", "HNCN", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOHYAC
     *
     * @cdk.inchi InChI=1S/C7H7ClO2S/c1-6-2-4-7(5-3-6)11(8,9)10/h2-5H,1H3
     */
    @Test
    public void testFOHYAC() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 3, DOUBLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(10, 17, SINGLE);

        String[] expected = {"SO2", "CL", "O2S", "O2S", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOJBEL
     *
     * @cdk.inchi InChI=1S/C8H16O2S/c1-3-4-8-5-6-10-7(2)11(8)9/h7-8H,3-6H2,1-2H3/t7-,8+,11-/m0/s1
     */
    @Test
    public void testFOJBEL() {
        IAtomContainer container = new AtomContainer(27, 27, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 19, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(10, 24, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(10, 26, SINGLE);

        String[] expected = {"S=O", "OR", "O=S", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOJPAV
     *
     * @cdk.inchi InChI=1S/C12H9NO2S2/c1-7-2-4-8(5-3-7)10-11-13(12(15)17-10)9(14)6-16-11/h2-5H,6H2,1H3
     */
    @Test
    public void testFOJPAV() {
        IAtomContainer container = new AtomContainer(26, 28, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 19, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 20, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 16, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 21, SINGLE);
        container.addBond(15, 22, SINGLE);
        container.addBond(16, 23, SINGLE);
        container.addBond(16, 24, SINGLE);
        container.addBond(16, 25, SINGLE);

        String[] expected = {"S", "CR", "C=ON", "O=CN", "NC=O", "C=ON", "O=CN", "S", "C=C", "C=C", "CB", "CB", "CB",
                "CB", "CB", "CB", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FONCOA
     *
     * @cdk.inchi InChI=1S/C10H16N2O2/c1-6(2)5-7-10(14)12-4-3-8(12)9(13)11-7/h6-8H,3-5H2,1-2H3,(H,11,13)/t7-,8-/m1/s1
     */
    @Test
    public void testFONCOA() {
        IAtomContainer container = new AtomContainer(30, 31, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 16, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 9, DOUBLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 29, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(13, 27, SINGLE);
        container.addBond(13, 26, SINGLE);

        String[] expected = {"CR", "CR", "C=ON", "CR", "O=CN", "C=ON", "NC=O", "CR", "CR4R", "O=CN", "NC=O", "CR4R",
                "CR4R", "CR", "HC", "HC", "HC", "HC", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FORGOI
     *
     * @cdk.inchi InChI=1S/C8H7IO2/c1-11-8(10)6-2-4-7(9)5-3-6/h2-5H,1H3
     */
    @Test
    public void testFORGOI() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("I", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 9, DOUBLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 11, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(10, 17, SINGLE);

        String[] expected = {"I", "O=CO", "OC=O", "CB", "CB", "CB", "CB", "CB", "CB", "COO", "CR", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FORHEZ
     *
     * @cdk.inchi InChI=1S/C5H6N2O2S/c1-4(7(8)9)5(2)10-3-6/h1-2H3/b5-4-
     */
    @Test
    public void testFORHEZ() {
        IAtomContainer container = new AtomContainer(16, 15, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(3, 15, SINGLE);
        container.addBond(4, 5, TRIPLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, DOUBLE);

        String[] expected = {"CR", "C=C", "C=C", "CR", "CSP", "NSP", "NO2", "O2N", "O2N", "S", "HC", "HC", "HC", "HC",
                "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FORJIF
     *
     * @cdk.inchi InChI=1S/C10H2Cl10/c11-3-1-5(13)4(12)2(7(3,15)9(5,17)18)8(3,16)10(19,20)6(1,4)14/h1-2H/t1-,2-,3-,4-,5-,6+,7-,8+
     */
    @Test
    public void testFORJIF() {
        IAtomContainer container = new AtomContainer(22, 26, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(13, 16, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(14, 20, SINGLE);
        container.addBond(16, 21, SINGLE);

        String[] expected = {"CL", "CL", "CL", "CL", "CL", "CR4R", "CR4R", "CR4R", "CR4R", "CR", "HC", "CR4R", "CR4R",
                "CR", "CR4R", "CL", "CR4R", "HC", "CL", "CL", "CL", "CL"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FORJUR
     *
     * @cdk.inchi InChI=1S/C11H14N4S/c1-7-5-4-6-8(2)9(7)15-10(12)13-11(14-15)16-3/h4-6H,1-3H3,(H2,12,13,14)
     */
    @Test
    public void testFORJUR() {
        IAtomContainer container = new AtomContainer(30, 31, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 7, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 28, SINGLE);
        container.addBond(6, 29, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(13, 20, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(14, 22, SINGLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(15, 25, SINGLE);
        container.addBond(15, 26, SINGLE);
        container.addBond(15, 27, SINGLE);

        String[] expected = {"S", "NPYL", "N5A", "C5B", "N5B", "C5A", "NC=N", "CB", "CB", "CB", "CB", "CB", "CB", "CR",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCN", "HNCN"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FORTAH
     *
     * @cdk.inchi InChI=1S/C6H14N2S2/c1-7(2)5(6(9)10)8(3)4/h5H,1-4H3,(H,9,10)/p-1
     */
    @Test
    public void testFORTAH() {
        IAtomContainer container = new AtomContainer(23, 22, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 22, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 12, SINGLE);
        container.addBond(5, 11, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 19, SINGLE);

        String[] expected = {"S2CM", "S2CM", "CS2M", "CR", "NR", "CR", "CR", "NR", "CR", "CR", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOSDIA
     *
     * @cdk.inchi InChI=1S/C8H12Cl3N4PS/c9-8(10,11)7-12-1-2-15(7)16(17,13-3-4-13)14-5-6-14/h1-6H2
     */
    @Test
    public void testFOSDIA() {
        IAtomContainer container = new AtomContainer(29, 31, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("P", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 16, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 13, DOUBLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(13, 16, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 25, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(15, 28, SINGLE);

        String[] expected = {"CL", "CL", "CL", "S-P", "PTET", "NR", "NR", "NC=N", "N=C", "CR3R", "CR3R", "CR3R",
                "CR3R", "C=N", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOVHUT
     *
     * @cdk.inchi InChI=1S/C4H9N5/c1-2-7-3(5)9-4(6)8-2/h3H,5H2,1H3,(H3,6,7,8,9)/t3-/m1/s1
     */
    @Test
    public void testFOVHUT() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 11, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 5, DOUBLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(12, 15, SINGLE);

        String[] expected = {"N=C", "C=N", "NC=N", "HNCN", "HNCN", "N=C", "CR", "NR", "HNR", "HNR", "NC=N", "C=N",
                "CR", "HC", "HNCN", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOVJIJ
     *
     * @cdk.inchi InChI=1S/C10H12O3/c1-5-4-7-8(9(5)12)10(7,3)13-6(2)11/h4,7-8H,1-3H3/t7-,8-,10+/m1/s1
     */
    @Test
    public void testFOVJIJ() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, DOUBLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(2, 11, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 14, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 17, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 19, SINGLE);
        container.addBond(10, 20, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 22, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(12, 24, SINGLE);

        String[] expected = {"O=CR", "OC=O", "O=CO", "CR3R", "C=OR", "C=C", "C=C", "CR3R", "CR", "CR3R", "CR", "COO",
                "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOVRAJ
     *
     * @cdk.inchi InChI=1S/C14H14N4O/c1-10-12(15)18-8-7-16-14(13(18)17-10)19-9-11-5-3-2-4-6-11/h2-8H,9,15H2,1H3/p+1
     */
    @Test
    public void testFOVRAJ() {
        IAtomContainer container = new AtomContainer(34, 36, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 19, SINGLE);
        container.addBond(0, 8, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 10, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 20, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 29, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 30, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(16, 31, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(17, 32, SINGLE);
        container.addBond(17, 18, DOUBLE);
        container.addBond(18, 33, SINGLE);

        String[] expected = {"NIM+", "C5", "C5", "NIM+", "C=C", "C=C", "N=C", "C=N", "CIM+", "CR", "NC=C", "OC=N",
                "CR", "CB", "CB", "CB", "CB", "CB", "CB", "HIM+", "HC", "HC", "HC", "HC", "HC", "HNCC", "HNCC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOVRUD
     *
     * @cdk.inchi InChI=1S/C14H14N4O/c1-10-8-13(19)16-17-14(10)11-2-4-12(5-3-11)18-7-6-15-9-18/h2-7,9-10H,8H2,1H3,(H,16,19)/p+1/t10-/m0/s1
     */
    @Test
    public void testFOVRUD() {
        IAtomContainer container = new AtomContainer(34, 36, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 21, SINGLE);
        container.addBond(3, 20, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(7, 25, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 26, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 27, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 28, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 29, SINGLE);
        container.addBond(14, 18, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 30, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(16, 31, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(17, 32, SINGLE);
        container.addBond(17, 18, DOUBLE);
        container.addBond(18, 33, SINGLE);

        String[] expected = {"N=C", "NC=O", "C=ON", "CR", "CR", "C=N", "O=CN", "CR", "CB", "CB", "CB", "CB", "CB",
                "CB", "NIM+", "CIM+", "NIM+", "C5", "C5", "HNCO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HIM+", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOWBEY
     *
     * @cdk.inchi InChI=1S/C9H8ClN5/c1-9(2)5-6(12-14-9)7(10)13-15-4-3-11-8(5)15/h3-4H,1-2H3
     */
    @Test
    public void testFOWBEY() {
        IAtomContainer container = new AtomContainer(23, 25, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 12, DOUBLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 11, DOUBLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(14, 20, SINGLE);
        container.addBond(14, 21, SINGLE);
        container.addBond(14, 22, SINGLE);

        String[] expected = {"CL", "N5B", "C5B", "C5A", "NPYL", "N=C", "C=N", "C=C", "N=N", "N=N", "CR", "C=C", "C5A",
                "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOWPOW
     *
     * @cdk.inchi InChI=1S/C10H9N7O2/c1-19-7-4-2-6(3-5-7)11-8-9(18)12-10-13-15-16-17(10)14-8/h2-5H,1H3,(H,11,14)(H,12,13,16,18)
     */
    @Test
    public void testFOWPOW() {
        IAtomContainer container = new AtomContainer(28, 30, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 11, DOUBLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(2, 9, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 10, DOUBLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(12, 17, DOUBLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 21, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 22, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 16, DOUBLE);
        container.addBond(16, 23, SINGLE);
        container.addBond(16, 17, SINGLE);
        container.addBond(17, 24, SINGLE);
        container.addBond(18, 27, SINGLE);
        container.addBond(18, 26, SINGLE);
        container.addBond(18, 25, SINGLE);

        String[] expected = {"O=CN", "OC=C", "N5B", "N5B", "N5A", "NPYL", "N=C", "NC=O", "NC=N", "C5A", "C=N", "C=ON",
                "CB", "CB", "CB", "CB", "CB", "CB", "CR", "HNCO", "HNCN", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOWVES
     *
     * @cdk.inchi InChI=1S/C8O4S4/c9-5-1-2(6(10)15-5)14-4-3(13-1)7(11)16-8(4)12
     */
    @Test
    public void testFOWVES() {
        IAtomContainer container = new AtomContainer(16, 18, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 6, DOUBLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(4, 8, DOUBLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 9, DOUBLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 12, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(10, 14, DOUBLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(13, 15, DOUBLE);

        String[] expected = {"S", "S", "O=C", "O=C", "C=C", "C=C", "C=OS", "C=OS", "C=C", "C=C", "C=OS", "S", "S",
                "C=OS", "O=C", "O=C"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOWZAS
     *
     * @cdk.inchi InChI=1S/C12H14O3/c13-4-15-12-5-1-6-7-3-9(14)10(6)8(2-5)11(7)12/h4-8,10-12H,1-3H2/t5-,6-,7-,8-,10+,11+,12+/m1/s1
     */
    @Test
    public void testFOWZAS() {
        IAtomContainer container = new AtomContainer(29, 32, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 17, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 18, SINGLE);
        container.addBond(3, 19, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(4, 20, SINGLE);
        container.addBond(5, 13, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(8, 24, SINGLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 12, DOUBLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(10, 27, SINGLE);
        container.addBond(11, 13, SINGLE);
        container.addBond(11, 14, DOUBLE);
        container.addBond(11, 28, SINGLE);

        String[] expected = {"CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "CR", "C=OR", "CR", "COO", "O=CR", "OC=O",
                "O=CO", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOYMAH
     *
     * @cdk.inchi InChI=1S/C7H8N2/c8-7(9)6-4-2-1-3-5-6/h1-5H,(H3,8,9)/p+1
     */
    @Test
    public void testFOYMAH() {
        IAtomContainer container = new AtomContainer(18, 18, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 15, SINGLE);
        container.addBond(0, 14, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 17, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 13, SINGLE);

        String[] expected = {"NCN+", "NCN+", "CNN+", "CB", "CB", "CB", "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC",
                "HNN+", "HNN+", "HNN+", "HNN+"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FOYNUC
     *
     * @cdk.inchi InChI=1S/C12H11NO3/c14-13(15)11-7-9-6-5-8-3-1-2-4-10(8)12(9)16-11/h5-7H,1-4H2
     */
    @Test
    public void testFOYNUC() {
        IAtomContainer container = new AtomContainer(27, 29, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(5, 20, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 21, SINGLE);
        container.addBond(6, 22, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 23, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(8, 26, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 15, DOUBLE);

        String[] expected = {"OFUR", "C5A", "C5B", "CB", "CB", "CR", "CR", "CR", "CR", "CB", "CB", "C5B", "C5A", "NO2",
                "O2N", "O2N", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUCMIZ
     *
     * @cdk.inchi InChI=1S/C2H7Cl2NSi/c1-5(2)6(3)4/h6H,1-2H3
     */
    @Test
    public void testFUCMIZ() {
        IAtomContainer container = new AtomContainer(13, 12, 0, 0);
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 9, SINGLE);
        container.addBond(5, 8, SINGLE);

        String[] expected = {"CL", "CL", "SI", "NR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HSI", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUCMUL
     *
     * @cdk.inchi InChI=1S/C4H2N2S2/c1-5-3-4(7-1)6-2-8-3/h1-2H
     */
    @Test
    public void testFUCMUL() {
        IAtomContainer container = new AtomContainer(10, 11, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(2, 5, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, DOUBLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 9, SINGLE);

        String[] expected = {"STHI", "N5B", "C5", "C5A", "HC", "C5", "C5A", "N5B", "STHI", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUCTIG01
     *
     * @cdk.inchi InChI=1S/NO3/c2-1(3)4/q-1
     */
    @Test
    public void testFUCTIG01() {
        IAtomContainer container = new AtomContainer(4, 3, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("N", 1));
        container.addBond(0, 3, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 3, SINGLE);

        String[] expected = {"O3N", "O3N", "O3N", "NO3"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUCWIJ
     *
     * @cdk.inchi InChI=1S/C11H15N5O4/c1-15-3-14-10-6(9(15)12)13-4-16(10)11-8(19)7(18)5(2-17)20-11/h3-5,7-8,11-12,17-19H,2H2,1H3/b12-9+/t5-,7-,8-,11-/m1/s1
     */
    @Test
    public void testFUCWIJ() {
        IAtomContainer container = new AtomContainer(35, 37, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(5, 10, DOUBLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(9, 29, SINGLE);
        container.addBond(9, 30, SINGLE);
        container.addBond(9, 31, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(14, 16, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(15, 19, SINGLE);
        container.addBond(15, 27, SINGLE);
        container.addBond(15, 28, SINGLE);
        container.addBond(17, 32, SINGLE);
        container.addBond(18, 33, SINGLE);
        container.addBond(19, 34, SINGLE);

        String[] expected = {"NC=N", "C=N", "N=C", "C5A", "C5B", "C=N", "N5B", "C5A", "NPYL", "CR", "N=C", "CR", "CR",
                "CR", "CR", "CR", "OR", "OR", "OR", "OR", "HC", "HC", "HN=C", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HOR", "HOR", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUCWOP
     *
     * @cdk.inchi InChI=1S/C10H9N6/c11-10-14-8-7(12-6-13-8)9(15-10)16-4-2-1-3-5-16/h1-6H,(H3,11,12,13,14,15)/q+1
     */
    @Test
    public void testFUCWOP() {
        IAtomContainer container = new AtomContainer(25, 27, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 17, SINGLE);
        container.addBond(2, 16, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 15, DOUBLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 22, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 24, SINGLE);

        String[] expected = {"NPYD", "CB", "NC=N", "NPYD", "C5A", "C5B", "CB", "N5B", "C5A", "NPYL", "NPD+", "CB",
                "CB", "CB", "CB", "CB", "HNCN", "HNCN", "HC", "HPYL", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUDPOJ
     *
     * @cdk.inchi InChI=1S/C3H2O3/c4-1-2(5)3(1)6/h4-5H
     */
    @Test
    public void testFUDPOJ() {
        IAtomContainer container = new AtomContainer(8, 8, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 5, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 7, SINGLE);

        String[] expected = {"O=CR", "OC=C", "C=OR", "C=C", "HOCC", "C=C", "OC=C", "HOCC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUDXUX
     *
     * @cdk.inchi InChI=1S/C3H6N2/c1-5-3-2-4/h5H,3H2,1H3/p+1
     */
    @Test
    public void testFUDXUX() {
        IAtomContainer container = new AtomContainer(12, 11, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 2, TRIPLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 6, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 11, SINGLE);

        String[] expected = {"NSP", "NR+", "CSP", "CR", "CR", "HNR+", "HNR+", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUFDIT
     *
     * @cdk.inchi InChI=1S/C5H8N2O5/c1-5(10)2(8)6-4(9)7-3(5)12-11/h3,10-11H,1H3,(H2,6,7,8,9)/t3-,5-/m0/s1
     */
    @Test
    public void testFUFDIT() {
        IAtomContainer container = new AtomContainer(20, 20, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 6, DOUBLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 8, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(5, 10, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 16, SINGLE);
        container.addBond(7, 17, SINGLE);
        container.addBond(9, 18, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 19, SINGLE);

        String[] expected = {"NC=O", "CONN", "NC=O", "C=ON", "CR", "CR", "O=CN", "CR", "O=CN", "OR", "OR", "-O-",
                "HNCO", "HNCO", "HC", "HC", "HC", "HC", "HOR", "HO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUGWIN
     *
     * @cdk.inchi InChI=1S/C10H13N5O3/c11-10-8-6(12-3-13-10)7(14-15-8)9-5(17)1-4(2-16)18-9/h3-5,9,16-17H,1-2H2,(H,14,15)(H2,11,12,13)/p+1/t4-,5+,9+/m0/s1
     */
    @Test
    public void testFUGWIN() {
        IAtomContainer container = new AtomContainer(32, 34, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 15, SINGLE);
        container.addBond(0, 11, SINGLE);
        container.addBond(1, 31, SINGLE);
        container.addBond(1, 12, SINGLE);
        container.addBond(2, 14, DOUBLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(3, 30, SINGLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(4, 8, DOUBLE);
        container.addBond(5, 28, SINGLE);
        container.addBond(5, 10, DOUBLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(6, 29, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(7, 27, SINGLE);
        container.addBond(7, 26, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(8, 20, SINGLE);
        container.addBond(9, 13, DOUBLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 14, SINGLE);
        container.addBond(11, 18, SINGLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 15, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 23, SINGLE);
        container.addBond(16, 25, SINGLE);
        container.addBond(16, 22, SINGLE);
        container.addBond(17, 24, SINGLE);
        container.addBond(17, 19, SINGLE);

        String[] expected = {"OR", "OR", "N5A", "OR", "NPYD", "NPD+", "NPYL", "NC=C", "CB", "C5A", "C5B", "CR", "CR",
                "CB", "C5B", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HNCC", "HNCC", "HPD+",
                "HPYL", "HOR", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUHFAP
     *
     * @cdk.inchi InChI=1S/C2H7NO2/c1-4-3-5-2/h3H,1-2H3
     */
    @Test
    public void testFUHFAP() {
        IAtomContainer container = new AtomContainer(12, 11, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 11, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 8, SINGLE);

        String[] expected = {"NR", "OR", "CR", "HC", "HC", "HC", "OR", "CR", "HC", "HC", "HC", "HNR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUHSEG
     *
     * @cdk.inchi InChI=1S/C9H6N2O2/c10-7-9(11(12)13)6-8-4-2-1-3-5-8/h1-6H/b9-6+
     */
    @Test
    public void testFUHSEG() {
        IAtomContainer container = new AtomContainer(19, 19, 0, 0);
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 3, DOUBLE);
        container.addBond(2, 4, TRIPLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 13, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 15, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 17, SINGLE);
        container.addBond(12, 18, SINGLE);

        String[] expected = {"O2N", "O2N", "NSP", "NO2", "CSP", "C=C", "C=C", "CB", "CB", "CB", "CB", "CB", "CB", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FULRAF
     *
     * @cdk.inchi InChI=1S/C8H14N6O4/c9-5(10)6-12-8(11)14(13-6)7-4(17)3(16)2(1-15)18-7/h2-4,7,15-17H,1H2,(H3,9,10)(H2,11,12,13)/p+1/t2-,3-,4-,7-/m1/s1
     */
    @Test
    public void testFULRAF() {
        IAtomContainer container = new AtomContainer(33, 34, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 8, SINGLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 21, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 23, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 26, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 27, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 29, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(15, 30, SINGLE);
        container.addBond(16, 31, SINGLE);
        container.addBond(17, 32, SINGLE);

        String[] expected = {"NPYL", "N5A", "C5B", "N5B", "C5A", "CNN+", "NCN+", "NCN+", "NC=N", "CR", "CR", "CR",
                "CR", "CR", "OR", "OR", "OR", "OR", "HNN+", "HNN+", "HNN+", "HNN+", "HNCN", "HNCN", "HC", "HC", "HC",
                "HC", "HC", "HC", "HOR", "HOR", "HOR"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUNSIQ
     *
     * @cdk.inchi InChI=1S/C4H7NO7S/c1-2-12-4(6)3(5(7)8)13(9,10)11/h3H,2H2,1H3,(H,9,10,11)/p-1/t3-/m0/s1
     */
    @Test
    public void testFUNSIQ() {
        IAtomContainer container = new AtomContainer(19, 18, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 5, DOUBLE);
        container.addBond(0, 6, DOUBLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(1, 9, DOUBLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 11, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(4, 8, DOUBLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(12, 16, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 18, SINGLE);

        String[] expected = {"SO3", "O=CO", "OC=O", "O2N", "O2N", "O3S", "O3S", "O3S", "NO2", "COO", "CR", "CR", "CR",
                "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUNXOB
     *
     * @cdk.inchi InChI=1S/C12H10O3/c1-14-11-4-2-9(3-5-11)12(13)10-6-7-15-8-10/h2-8H,1H3
     */
    @Test
    public void testFUNXOB() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 16, SINGLE);
        container.addBond(4, 17, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 7, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 18, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 19, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(11, 12, DOUBLE);
        container.addBond(11, 20, SINGLE);
        container.addBond(12, 21, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 22, SINGLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(14, 24, SINGLE);

        String[] expected = {"OFUR", "C5A", "C5B", "C5B", "C5A", "C=OR", "O=CR", "CB", "CB", "CB", "CB", "CB", "CB",
                "OC=C", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUPJUV
     *
     * @cdk.inchi InChI=1S/C11H13N3O3/c1-11(2,3)17-10(15)13-8-6-4-5-7-9(8)14(16)12-13/h4-7H,1-3H3
     */
    @Test
    public void testFUPJUV() {
        IAtomContainer container = new AtomContainer(30, 31, 0, 0);
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 19, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(14, 24, SINGLE);
        container.addBond(14, 23, SINGLE);
        container.addBond(15, 22, SINGLE);
        container.addBond(15, 21, SINGLE);
        container.addBond(15, 20, SINGLE);
        container.addBond(25, 29, SINGLE);
        container.addBond(25, 28, SINGLE);
        container.addBond(25, 27, SINGLE);

        String[] expected = {"N5BX", "OXN", "N5A", "NPYL", "C5A", "CB", "CB", "CB", "CB", "C5B", "C=ON", "O=CN",
                "OC=O", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "CR", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUPKIK
     *
     * @cdk.inchi InChI=1S/C7H7N3O3S/c1-14(11,12)13-10-7-5-3-2-4-6(7)8-9-10/h2-5H,1H3
     */
    @Test
    public void testFUPKIK() {
        IAtomContainer container = new AtomContainer(21, 22, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 8, SINGLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 14, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 12, DOUBLE);
        container.addBond(10, 13, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(13, 19, SINGLE);
        container.addBond(13, 20, SINGLE);

        String[] expected = {"N5B", "N5A", "NPYL", "CB", "CB", "CB", "CB", "C5B", "C5A", "OSO2", "SO3", "O2S", "O2S",
                "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUPKOQ
     *
     * @cdk.inchi InChI=1S/C12H10N6O3S/c1-21-15-10(7-6-22-12(13)14-7)11(19)17-8-4-2-3-5-9(8)18(20)16-17/h2-6H,1H3,(H2,13,14)/b15-10-
     */
    @Test
    public void testFUPKOQ() {
        IAtomContainer container = new AtomContainer(32, 34, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(2, 9, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 22, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 24, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(12, 19, DOUBLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(13, 17, SINGLE);
        container.addBond(13, 14, DOUBLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(14, 15, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 18, SINGLE);
        container.addBond(16, 17, DOUBLE);
        container.addBond(18, 28, SINGLE);
        container.addBond(18, 27, SINGLE);
        container.addBond(19, 20, SINGLE);
        container.addBond(20, 21, SINGLE);
        container.addBond(21, 31, SINGLE);
        container.addBond(21, 30, SINGLE);
        container.addBond(21, 29, SINGLE);

        String[] expected = {"NPYL", "N5A", "N5BX", "C5B", "C5A", "CB", "CB", "CB", "CB", "OXN", "C=ON", "O=CN", "C=N",
                "C5B", "C5A", "STHI", "C5A", "N5B", "NC=N", "N=C", "OR", "CR", "HC", "HC", "HC", "HC", "HC", "HNCN",
                "HNCN", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUPTOZ
     *
     * @cdk.inchi InChI=1S/C6H2Cl8OS2/c7-1(8)2-16-3(5(9,10)11)15-4(17-2)6(12,13)14/h3-4H/t3-,4-/m1/s1
     */
    @Test
    public void testFUPTOZ() {
        IAtomContainer container = new AtomContainer(19, 19, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addAtom(newAtm("Cl", 0));
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 13, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 6, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 11, SINGLE);
        container.addBond(9, 12, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 15, SINGLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(15, 17, SINGLE);
        container.addBond(15, 18, SINGLE);

        String[] expected = {"OR", "S", "S", "C=C", "C=C", "CL", "CL", "CR", "HC", "CR", "CL", "CL", "CL", "CR", "HC",
                "CR", "CL", "CL", "CL"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUPZEV
     *
     * @cdk.inchi InChI=1S/C12H18O3/c1-7-5-6-9-11(3,4)10(12(7,9)14)15-8(2)13/h5,9-10,14H,6H2,1-4H3/t9-,10+,12+/m0/s1
     */
    @Test
    public void testFUPZEV() {
        IAtomContainer container = new AtomContainer(33, 34, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(1, 9, SINGLE);
        container.addBond(1, 10, SINGLE);
        container.addBond(2, 10, DOUBLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 9, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(4, 12, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 16, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 17, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 19, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 13, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 21, SINGLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 26, SINGLE);
        container.addBond(13, 27, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(13, 29, SINGLE);
        container.addBond(14, 30, SINGLE);
        container.addBond(14, 31, SINGLE);
        container.addBond(14, 32, SINGLE);

        String[] expected = {"OR", "OC=O", "O=CO", "CR4R", "C=C", "C=C", "CR", "CR4R", "CR4R", "CR4R", "COO", "CR",
                "CR", "CR", "CR", "HOR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUSPEO
     *
     * @cdk.inchi InChI=1S/C3H4O2/c4-2-1-3-5/h1-4H/p-1/b2-1+
     */
    @Test
    public void testFUSPEO() {
        IAtomContainer container = new AtomContainer(8, 7, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 7, SINGLE);
        container.addBond(3, 4, SINGLE);

        String[] expected = {"O=CR", "C=OR", "C=C", "C=C", "OM2", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUTCEC
     *
     * @cdk.inchi InChI=1S/C10H12N2O/c1-10(2)9(13)12(11-10)8-6-4-3-5-7-8/h3-7,11H,1-2H3
     */
    @Test
    public void testFUTCEC() {
        IAtomContainer container = new AtomContainer(25, 26, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 6, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(1, 13, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 4, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 12, DOUBLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(4, 15, SINGLE);
        container.addBond(4, 16, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 20, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 22, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 23, SINGLE);
        container.addBond(11, 24, SINGLE);

        String[] expected = {"NC=O", "NR", "CR4R", "C=ON", "CR", "CR", "CB", "CB", "CB", "CB", "CB", "CB", "O=CN",
                "HNR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUTZEZ
     *
     * @cdk.inchi InChI=1S/CH6OSi/c1-2-3/h1,3H3
     */
    @Test
    public void testFUTZEZ() {
        IAtomContainer container = new AtomContainer(9, 8, 0, 0);
        container.addAtom(newAtm("Si", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(2, 6, SINGLE);
        container.addBond(2, 8, SINGLE);

        String[] expected = {"SI", "OR", "CR", "HSI", "HSI", "HC", "HC", "HSI", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUVDOP
     *
     * @cdk.inchi InChI=1S/C12H15N3O6/c1-4(16)19-10-7-11(20-5(2)17)9-12(21-6(3)18)8(10)14-13(7)15(9)14/h7-12H,1-3H3/t7-,8+,9-,10-,11+,12-,13+,14-,15+
     */
    @Test
    public void testFUVDOP() {
        IAtomContainer container = new AtomContainer(36, 39, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(2, 13, SINGLE);
        container.addBond(2, 17, SINGLE);
        container.addBond(3, 15, DOUBLE);
        container.addBond(4, 16, DOUBLE);
        container.addBond(5, 17, DOUBLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 8, SINGLE);
        container.addBond(6, 10, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(8, 14, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 22, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 23, SINGLE);
        container.addBond(12, 13, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 25, SINGLE);
        container.addBond(14, 26, SINGLE);
        container.addBond(15, 18, SINGLE);
        container.addBond(16, 19, SINGLE);
        container.addBond(17, 20, SINGLE);
        container.addBond(18, 27, SINGLE);
        container.addBond(18, 28, SINGLE);
        container.addBond(18, 29, SINGLE);
        container.addBond(19, 30, SINGLE);
        container.addBond(19, 31, SINGLE);
        container.addBond(19, 32, SINGLE);
        container.addBond(20, 33, SINGLE);
        container.addBond(20, 34, SINGLE);
        container.addBond(20, 35, SINGLE);

        String[] expected = {"OC=O", "OC=O", "OC=O", "O=CO", "O=CO", "O=CO", "NR", "NR", "NR", "CR", "CR", "CR", "CR",
                "CR", "CR", "COO", "COO", "COO", "CR", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUVMUE
     *
     * @cdk.inchi InChI=1S/C9H18N2O2/c1-10(2)8(12)6-5-7-9(13)11(3)4/h5-7H2,1-4H3/p+2
     */
    @Test
    public void testFUVMUE() {
        IAtomContainer container = new AtomContainer(33, 32, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 1));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 17, SINGLE);
        container.addBond(0, 16, SINGLE);
        container.addBond(0, 15, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(1, 19, SINGLE);
        container.addBond(1, 18, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 5, SINGLE);
        container.addBond(3, 4, DOUBLE);
        container.addBond(4, 13, SINGLE);
        container.addBond(5, 22, SINGLE);
        container.addBond(5, 21, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 24, SINGLE);
        container.addBond(6, 23, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 26, SINGLE);
        container.addBond(7, 25, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 10, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 14, SINGLE);
        container.addBond(10, 12, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(11, 29, SINGLE);
        container.addBond(11, 28, SINGLE);
        container.addBond(11, 27, SINGLE);
        container.addBond(12, 32, SINGLE);
        container.addBond(12, 31, SINGLE);
        container.addBond(12, 30, SINGLE);

        String[] expected = {"CR", "CR", "NC=O", "C=ON", "O=+", "CR", "CR", "CR", "C=ON", "O=+", "NC=O", "CR", "CR",
                "HO=+", "HO=+", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUVNEP
     *
     * @cdk.inchi InChI=1S/C17H20N2O/c1-18(2)15-9-5-13(6-10-15)17(20)14-7-11-16(12-8-14)19(3)4/h5-12H,1-4H3
     */
    @Test
    public void testFUVNEP() {
        IAtomContainer container = new AtomContainer(40, 41, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 11, DOUBLE);
        container.addBond(1, 5, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(2, 19, SINGLE);
        container.addBond(2, 18, SINGLE);
        container.addBond(2, 15, SINGLE);
        container.addBond(3, 25, SINGLE);
        container.addBond(3, 24, SINGLE);
        container.addBond(3, 23, SINGLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(4, 21, SINGLE);
        container.addBond(4, 20, SINGLE);
        container.addBond(5, 10, DOUBLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 32, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 33, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 11, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 34, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 35, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(13, 39, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(14, 38, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(15, 16, SINGLE);
        container.addBond(16, 37, SINGLE);
        container.addBond(16, 17, DOUBLE);
        container.addBond(17, 36, SINGLE);
        container.addBond(18, 28, SINGLE);
        container.addBond(18, 27, SINGLE);
        container.addBond(18, 26, SINGLE);
        container.addBond(19, 31, SINGLE);
        container.addBond(19, 30, SINGLE);
        container.addBond(19, 29, SINGLE);

        String[] expected = {"O=CR", "NC=C", "NC=C", "CR", "CR", "CB", "CB", "CB", "CB", "CB", "CB", "C=OR", "CB",
                "CB", "CB", "CB", "CB", "CB", "CR", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUVXOJ
     *
     * @cdk.inchi InChI=1S/C8H9NO2/c1-11-8-4-2-7(3-5-8)6-9-10/h2-6,10H,1H3/p+1/b9-6-
     */
    @Test
    public void testFUVXOJ() {
        IAtomContainer container = new AtomContainer(21, 21, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 4, SINGLE);
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 19, SINGLE);
        container.addBond(2, 10, DOUBLE);
        container.addBond(3, 13, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(3, 11, SINGLE);
        container.addBond(4, 9, DOUBLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(5, 14, SINGLE);
        container.addBond(5, 6, DOUBLE);
        container.addBond(6, 15, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(7, 10, SINGLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(9, 17, SINGLE);
        container.addBond(10, 18, SINGLE);

        String[] expected = {"OC=C", "-O-", "N+=C", "CR", "CB", "CB", "CB", "CB", "CB", "CB", "C=N", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC", "HNC+", "HO"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUWMOZ
     *
     * @cdk.inchi InChI=1S/C9H10N2O4S/c12-9-11(6-7-15-9)16(13,14)10-8-4-2-1-3-5-8/h1-5,10H,6-7H2
     */
    @Test
    public void testFUWMOZ() {
        IAtomContainer container = new AtomContainer(26, 27, 0, 0);
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 22, SINGLE);
        container.addBond(4, 23, SINGLE);
        container.addBond(5, 24, SINGLE);
        container.addBond(5, 25, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(6, 8, DOUBLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 21, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(10, 15, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(11, 16, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(13, 14, SINGLE);
        container.addBond(13, 18, SINGLE);
        container.addBond(14, 15, DOUBLE);
        container.addBond(14, 19, SINGLE);
        container.addBond(15, 20, SINGLE);

        String[] expected = {"OC=O", "C=ON", "O=CN", "NSO2", "CR", "CR", "SO2N", "O2S", "O2S", "NSO2", "CB", "CB",
                "CB", "CB", "CB", "CB", "HC", "HC", "HC", "HC", "HC", "HNSO", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUWTUM
     *
     * @cdk.inchi InChI=1S/C5H9P3S3/c1-5-2-6-7(3-5)11-8(9,4-5)10-6/h2-4H2,1H3/t5-,6+,7-,8-
     */
    @Test
    public void testFUWTUM() {
        IAtomContainer container = new AtomContainer(20, 22, 0, 0);
        container.addAtom(newAtm("S", -1));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("P", 1));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 3, SINGLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(2, 5, SINGLE);
        container.addBond(3, 6, SINGLE);
        container.addBond(4, 5, SINGLE);
        container.addBond(4, 7, SINGLE);
        container.addBond(5, 8, SINGLE);
        container.addBond(6, 9, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 12, SINGLE);
        container.addBond(7, 9, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(7, 14, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(8, 16, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 17, SINGLE);
        container.addBond(10, 18, SINGLE);
        container.addBond(10, 19, SINGLE);

        String[] expected = {"S-P", "S", "S", "PTET", "P", "P", "CR", "CR", "CR", "CR", "CR", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUXXAX
     *
     * @cdk.inchi InChI=1S/C5H15N2O3PS/c6-2-1-3-7-4-5-12-11(8,9)10/h7H,1-6H2,(H2,8,9,10)
     */
    @Test
    public void testFUXXAX() {
        IAtomContainer container = new AtomContainer(27, 26, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("P", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, SINGLE);
        container.addBond(0, 5, SINGLE);
        container.addBond(1, 2, DOUBLE);
        container.addBond(1, 3, SINGLE);
        container.addBond(1, 4, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(5, 17, SINGLE);
        container.addBond(5, 18, SINGLE);
        container.addBond(6, 7, SINGLE);
        container.addBond(6, 19, SINGLE);
        container.addBond(6, 20, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(7, 12, SINGLE);
        container.addBond(7, 13, SINGLE);
        container.addBond(8, 9, SINGLE);
        container.addBond(8, 21, SINGLE);
        container.addBond(8, 22, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(9, 23, SINGLE);
        container.addBond(9, 24, SINGLE);
        container.addBond(10, 11, SINGLE);
        container.addBond(10, 25, SINGLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(11, 14, SINGLE);
        container.addBond(11, 15, SINGLE);
        container.addBond(11, 16, SINGLE);

        String[] expected = {"S", "PO3", "O3P", "O3P", "O3P", "CR", "CR", "NR+", "CR", "CR", "CR", "NR+", "HNR+",
                "HNR+", "HNR+", "HNR+", "HNR+", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for FUXZED
     *
     * @cdk.inchi InChI=1S/C8H8N2O4S/c1-13-5-7(3-9)15(11,12)8(4-10)6-14-2/h5-6H,1-2H3/b7-5+,8-6+
     */
    @Test
    public void testFUXZED() {
        IAtomContainer container = new AtomContainer(23, 22, 0, 0);
        container.addAtom(newAtm("S", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 1, DOUBLE);
        container.addBond(0, 2, DOUBLE);
        container.addBond(0, 7, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(3, 8, SINGLE);
        container.addBond(3, 12, SINGLE);
        container.addBond(4, 10, SINGLE);
        container.addBond(4, 14, SINGLE);
        container.addBond(5, 11, TRIPLE);
        container.addBond(6, 13, TRIPLE);
        container.addBond(7, 8, DOUBLE);
        container.addBond(7, 11, SINGLE);
        container.addBond(8, 15, SINGLE);
        container.addBond(9, 10, DOUBLE);
        container.addBond(9, 13, SINGLE);
        container.addBond(10, 16, SINGLE);
        container.addBond(12, 17, SINGLE);
        container.addBond(12, 18, SINGLE);
        container.addBond(12, 19, SINGLE);
        container.addBond(14, 20, SINGLE);
        container.addBond(14, 21, SINGLE);
        container.addBond(14, 22, SINGLE);

        String[] expected = {"SO2", "O2S", "O2S", "OC=C", "OC=C", "NSP", "NSP", "C=C", "C=C", "C=C", "C=C", "CSP",
                "CR", "CSP", "CR", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for GADHEY
     *
     * @cdk.inchi InChI=1S/C14H18N2/c1-15(2)12-9-5-7-11-8-6-10-13(14(11)12)16(3)4/h5-10H,1-4H3/p+1
     */
    @Test
    public void testGADHEY() {
        IAtomContainer container = new AtomContainer(35, 36, 0, 0);
        container.addAtom(newAtm("N", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 13, SINGLE);
        container.addBond(0, 12, SINGLE);
        container.addBond(0, 2, SINGLE);
        container.addBond(1, 16, SINGLE);
        container.addBond(1, 15, SINGLE);
        container.addBond(1, 14, SINGLE);
        container.addBond(1, 8, SINGLE);
        container.addBond(2, 7, SINGLE);
        container.addBond(2, 3, DOUBLE);
        container.addBond(3, 17, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 18, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 19, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 11, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(9, 20, SINGLE);
        container.addBond(9, 10, SINGLE);
        container.addBond(10, 21, SINGLE);
        container.addBond(10, 11, DOUBLE);
        container.addBond(11, 22, SINGLE);
        container.addBond(12, 25, SINGLE);
        container.addBond(12, 24, SINGLE);
        container.addBond(12, 23, SINGLE);
        container.addBond(13, 28, SINGLE);
        container.addBond(13, 27, SINGLE);
        container.addBond(13, 26, SINGLE);
        container.addBond(14, 31, SINGLE);
        container.addBond(14, 30, SINGLE);
        container.addBond(14, 29, SINGLE);
        container.addBond(15, 34, SINGLE);
        container.addBond(15, 33, SINGLE);
        container.addBond(15, 32, SINGLE);

        String[] expected = {"NC=C", "NR+", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CB", "CR", "CR",
                "CR", "CR", "HNR+", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC", "HC", "HC", "HC"};
        assertMatchingTypes(container, assign(container), expected);
    }

    /**
     * Generated test for GAFNUW
     *
     * @cdk.inchi InChI=1S/C12H12N2O5/c1-6(15)19-11-3-7-2-9(11)8-4-12(14(17)18)13(16)5-10(7)8/h4-5,7,9,11H,2-3H2,1H3/t7-,9-,11+/m1/s1
     */
    @Test
    public void testGAFNUW() {
        IAtomContainer container = new AtomContainer(31, 33, 0, 0);
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("C", 0));
        container.addAtom(newAtm("N", 1));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("O", 0));
        container.addAtom(newAtm("O", -1));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addAtom(newAtm("H", 0));
        container.addBond(0, 19, SINGLE);
        container.addBond(0, 10, SINGLE);
        container.addBond(0, 9, SINGLE);
        container.addBond(0, 1, SINGLE);
        container.addBond(1, 20, SINGLE);
        container.addBond(1, 11, SINGLE);
        container.addBond(1, 2, SINGLE);
        container.addBond(2, 22, SINGLE);
        container.addBond(2, 21, SINGLE);
        container.addBond(2, 3, SINGLE);
        container.addBond(3, 23, SINGLE);
        container.addBond(3, 10, SINGLE);
        container.addBond(3, 4, SINGLE);
        container.addBond(4, 9, SINGLE);
        container.addBond(4, 5, DOUBLE);
        container.addBond(5, 24, SINGLE);
        container.addBond(5, 6, SINGLE);
        container.addBond(6, 18, SINGLE);
        container.addBond(6, 7, DOUBLE);
        container.addBond(7, 15, SINGLE);
        container.addBond(7, 8, SINGLE);
        container.addBond(8, 25, SINGLE);
        container.addBond(8, 9, DOUBLE);
        container.addBond(10, 27, SINGLE);
        container.addBond(10, 26, SINGLE);
        container.addBond(11, 12, SINGLE);
        container.addBond(12, 14, SINGLE);
        container.addBond(12, 13, DOUBLE);
        container.addBond(14, 30, SINGLE);
        container.addBond(14, 29, SINGLE);
        container.addBond(14, 28, SINGLE);
        container.addBond(15, 17, DOUBLE);
        container.addBond(15, 16, SINGLE);

        String[] expected = {"CR", "CR", "CR", "CR", "CB", "CB", "NPOX", "CB", "CB", "CB", "CR", "OC=O", "COO", "O=CO",
                "CR", "NO2", "O2N", "O2N", "OXN", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC", "HC",
                "HC"};
        assertMatchingTypes(container, assign(