# Django REST framework 2.2

The 2.2 release represents an important point for REST framework, with the addition of Python 3 support, and the introduction of an official deprecation policy.

## Python 3 support

Thanks to some fantastic work from [Xavier Ordoquy][xordoquy], Django REST framework 2.2 now supports Python 3.  You'll need to be running Django 1.5, and it's worth keeping in mind that Django's Python 3 support is currently [considered experimental][django-python-3].

Django 1.6's Python 3 support is expected to be officially labeled as 'production-ready'.

If you want to start ensuring that your own projects are Python 3 ready, we can highly recommend Django's [Porting to Python 3][porting-python-3] documentation.

Django REST framework's Python 2.6 support now requires 2.6.5 or above, in line with [Django 1.5's Python compatibility][python-compat].

## Deprecation policy

We've now introduced an official deprecation policy, which is in line with [Django's deprecation policy][django-deprecation-policy].  This policy will make it easy for you to continue to track the latest, greatest version of REST framework.

The timeline for deprecation works as follows:

* Version 2.2 introduces some API changes as detailed in the release notes.  It remains fully backwards compatible with 2.1, but will raise `PendingDeprecationWarning` warnings if you use bits of API that are due to be deprecated.  These warnings are silent by default, but can be explicitly enabled when you're ready to start migrating any required changes.  For example if you start running your tests using `python -Wd manage.py test`, you'll be warned of any API changes you need to make.

* Version 2.3 will escalate these warnings to `DeprecationWarning`, which is loud by default.

* Version 2.4 will remove the deprecated bits of API entirely.

Note that in line with Django's policy, any parts of the framework not mentioned in the documentation should generally be considered private API, and may be subject to change.

## Community

As of the 2.2 merge, we've also hit an impressive milestone.  The number of committers listed in [the credits][credits], is now at over **one hundred individuals**.  Each name on that list represents at least one merged pull request, however large or small.

Our [mailing list][mailing-list] and #restframework IRC channel are also very active, and we've got a really impressive rate of development both on REST framework itself, and on third party packages such as the great [django-rest-framework-docs][django-rest-framework-docs] package from [Marc Gibbons][marcgibbons].

---

## API changes

The 2.2 release makes a few changes to the API, in order to make it more consistent, simple, and easier to use.

### Cleaner to-many related fields

The `ManyRelatedField()` style is being deprecated in favor of a new `RelatedField(many=True)` syntax.

For example, if a user is associated with multiple questions, which we want to represent using a primary key relationship, we might use something like the following:

    class UserSerializer(serializers.HyperlinkedModelSerializer):
        questions = serializers.PrimaryKeyRelatedField(many=True)

        class Meta:
            fields = ('username', 'questions')

The new syntax is cleaner and more obvious, and the change will also make the documentation cleaner, simplify the internal API, and make writing custom relational fields easier.

The change also applies to serializers.  If you have a nested serializer, you should start using `many=True` for to-many relationships.  For example, a serializer representation of an Album that can contain many Tracks might look something like this:

    class TrackSerializer(serializer.ModelSerializer):
        class Meta:
            model = Track
            fields = ('name', 'duration')

    class AlbumSerializer(serializer.ModelSerializer):
        tracks = TrackSerializer(many=True)

        class Meta:
            model = Album
            fields = ('album_name', 'artist', 'tracks')

Additionally, the change also applies when serializing or deserializing data.  For example to serialize a queryset of models you should now use the `many=True` flag.

    serializer = SnippetSerializer(Snippet.objects.all(), many=True)
    serializer.data

This more explicit behavior on serializing and deserializing data [makes integration with non-ORM backends such as MongoDB easier][564], as instances to be serialized can include the `__iter__` method, without incorrectly triggering list-based serialization, or requiring workarounds.

The implicit to-many behavior on serializers, and the `ManyRelatedField` style classes will continue to function, but will raise a `PendingDeprecationWarning`, which can be made visible using the `-Wd` flag.

**Note**: If you need to forcibly turn off the implicit "`many=True` for `__iter__` objects" behavior, you can now do so by specifying `many=False`.  This will become the default (instead of the current default of `None`) once the deprecation of the implicit behavior is finalised in version 2.4.

### Cleaner optional relationships

Serializer relationships for nullable Foreign Keys will change from using the current `null=True` flag, to instead using `required=False`.

For example, is a user account has an optional foreign key to a company, that you want to express using a hyperlink, you might use the following field in a `Serializer` class:

    current_company = serializers.HyperlinkedRelatedField(required=False)

This is in line both with the rest of the serializer fields API, and with Django's `Form` and `ModelForm` API.

Using `required` throughout the serializers API means you won't need to consider if a particular field should take `blank` or `null` arguments instead of `required`, and also means there will be more consistent behavior for how fields are treated when they are not present in the incoming data.

The `null=True` argument will continue to function, and will imply `required=False`, but will raise a `PendingDeprecationWarning`.

### Cleaner CharField syntax

The `CharField` API previously took an optional `blank=True` argument, which was intended to differentiate between null CharField input, and blank CharField input.

In keeping with Django's CharField API, REST framework's `CharField` will only ever return the empty string, for missing or `None` inputs.  The `blank` flag will no longer be in use, and you should instead just use the `required=<bool>` flag.  For example:

    extra_details = CharField(required=False)

The `blank` keyword argument will continue to function, but will raise a `PendingDeprecationWarning`.

### Simpler object-level permissions

Custom permissions classes previously used the signature `.has_permission(self, request, view, obj=None)`.  This method would be called twice, firstly for the global permissions check, with the `obj` parameter set to `None`, and again for the object-level permissions check when appropriate, with the `obj` parameter set to the relevant model instance.

The global permissions check and object-level permissions check are now separated into two separate methods, which gives a cleaner, more obvious API.

* Global permission checks now use the `.has_permission(self, request, view)` signature.
* Object-level permission checks use a new method `.has_object_permission(self, request, view, obj)`.

For example, the following custom permission class:

    class IsOwner(permissions.BasePermission):
        """
        Custom permission to only allow owners of an object to view or edit it.
        Model instances are expected to include an `owner` attribute.
        """

        def has_permission(self, request, view, obj=None):
            if obj is None:
                # Ignore global permissions check
                return True

            return obj.owner == request.user

Now becomes:

    class IsOwner(permissions.BasePermission):
        """
        Custom permission to only allow owners of an object to view or edit it.
        Model instances are expected to include an `owner` attribute.
        """

        def has_object_permission(self, request, view, obj):
            return obj.owner == request.user

If you're overriding the `BasePermission` class, the old-style signature will continue to function, and will correctly handle both global and object-level permissions checks, but its use will raise a `PendingDeprecationWarning`.

Note also that the usage of the internal APIs for permission checking on the `View` class has been cleaned up slightly, and is now documented and subject to the deprecation policy in all future versions.

### More explicit hyperlink relations behavior

When using a serializer with a `HyperlinkedRelatedField` or `HyperlinkedIdentityField`, the hyperlinks would previously use absolute URLs if the serializer context included a `'request'` key, and fall back to using relative URLs otherwise.  This could lead to non-obvious behavior, as it might not be clear why some serializers generated absolute URLs, and others do not.

From version 2.2 onwards, serializers with hyperlinked relationships *always* require a `'request'` key to be supplied in the context dictionary.  The implicit behavior will continue to function, but its use will raise a `PendingDeprecationWarning`.

[xordoquy]: https://github.com/xordoquy
[django-python-3]: https://docs.djangoproject.com/en/stable/faq/install/#can-i-use-django-with-python-3
[porting-python-3]: https://docs.djangoproject.com/en/stable/topics/python3/
[python-compat]: https://docs.djangoproject.com/en/stable/releases/1.5/#python-compatibility
[django-deprecation-policy]: https://docs.djangoproject.com/en/stable/internals/release-process/#internal-release-deprecation-policy
[credits]: http://www.django-rest-framework.org/topics/credits
[mailing-list]: https://groups.google.com/forum/?fromgroups#!forum/django-rest-framework
[django-rest-framework-docs]: https://github.com/marcgibbons/django-rest-framework-docs
[marcgibbons]: https://github.com/marcgibbons/
[issues]: https://github.com/encode/django-rest-framework/issues
[564]: https://github.com/encode/django-rest-framework/issues/564
