#include <QPainter>
#include <QCursor>
#include <QMessageBox>

#include "qtdpuser.h"
#include "fitting.h"
#include "QFits2dFit.h"
#include "QFitsSingleBuffer.h"
#include "QFitsMainWindow.h"

QFits2dFit::QFits2dFit(QFitsMainWindow *parent) :
                                       QDialog(dynamic_cast<QWidget*>(parent)) {
    myParent = parent;
    int i, maxheight, maxwidth;
    zoom = zoomIndex+1;
    cenx = ceny = 0;
    QLabel *widthlabel = new QLabel("Fit window half width:", this);
    widthlabel->adjustSize();
    maxheight = widthlabel->height();
    fitwindow = new QSpinBox(this);
    fitwindow->setMinimum(3);
    fitwindow->setMaximum(999);
    fitwindow->setSingleStep(1);
    fitwindow->setValue(20);
    savefitwidth = 20;
    fitwindow->adjustSize();
    if (fitwindow->height() > maxheight) {
        maxheight = fitwindow->height();
    }
    refitButton = new QPushButton("Fit again", this);
    refitButton->adjustSize();
    if (refitButton->height() > maxheight) {
        maxheight = refitButton->height();
    }
    widthlabel->setGeometry(10, 155, widthlabel->width(), maxheight);
    fitwindow->setGeometry(widthlabel->x() + widthlabel->width() + 5,
                           widthlabel->y(),
                           fitwindow->width(),
                           maxheight);
    refitButton->setGeometry(fitwindow->x() + fitwindow->width() + 5,
                             widthlabel->y(),
                             refitButton->width(),
                             maxheight);
    fitfunction = new QComboBox(this);
    fitfunction->setGeometry(10, refitButton->y() + refitButton->height() + 5,
                             280, 25);
    fitfunction->addItem("Gauss");
//    fitfunction->addItem("Moffat");
    fitfunction->addItem("Sersic");

    fitslm = new QCheckBox("fix sersic index at:", this);
    fitslm->adjustSize();
    maxheight = fitslm->height();
    slm = new QDoubleSpinBox(this);
    slm->setRange(0.1, 5.0);
    slm->setValue(1.0);
    slm->setSingleStep(0.1);
    slm->adjustSize();
    if (slm->height() > maxheight) {
        maxheight = slm->height();
    }
    fitslm->setGeometry(10, fitfunction->y() + fitfunction->height() + 5,
                        fitslm->width(), maxheight);
    fitslm->hide();
    slm->setGeometry(fitslm->x() + fitslm->width() + 5, fitslm->y(),
                     slm->width(),maxheight);
    slm->hide();

    resultLabel1 = new QLabel(" \n \n \n \n \n \n \n \n \n ", this);
    resultLabel1->adjustSize();
    resultLabel1->move(10, fitslm->y() + fitslm->height() + 5);
    resultLabel2 = new QLabel(" \n \n \n \n \n \n \n ", this);
    resultLabel2->move(resultLabel1->x() + 10, resultLabel1->y());
    resultLabel2->setAlignment(Qt::AlignRight);
    resultLabel3 = new QLabel(" \n \n \n \n \n \n \n ", this);
    resultLabel3->move(resultLabel2->x() + 10, resultLabel1->y());

    maxwidth = refitButton->x() + refitButton->width() + 10;
    if (maxwidth < 300) {
        maxwidth = 300;
    }

    createButton = new QPushButton("Create new buffer from fit", this);
    createButton->setGeometry(10, resultLabel1->y() + resultLabel1->height() + 10,
                              maxwidth - 20, 30);
    closeButton = new QPushButton("Close", this);
    closeButton->setGeometry(maxwidth / 2 - 50,
                             createButton->y() + createButton->height() + 10,
                             100, 30);

    connect(fitwindow, SIGNAL(valueChanged(int)),
            this, SLOT(zoomChanged(int)));
    connect(fitfunction, SIGNAL(activated(const QString &)),
            this, SLOT(fitfunctionChanged(const QString &)));
    connect(closeButton, SIGNAL(clicked()),
            this, SLOT(close()));
    connect(refitButton, SIGNAL(clicked()),
            this, SLOT(refit()));
    connect(createButton, SIGNAL(clicked()),
            this, SLOT(createFitFunction()));

    resize(maxwidth, closeButton->y() + closeButton->height() + 20);
    setFixedSize(size());

    Qt::WindowFlags flags = windowFlags();
    flags |= Qt::WindowStaysOnTopHint;
    setWindowFlags(flags);
}

void QFits2dFit::paintEvent(QPaintEvent *) {
    QFitsSingleBuffer *sb = fitsMainWindow->getActualSB();
    if (sb != NULL) {
        double  fitx        = 0.,
                fity        = 0.,
                fitw1       = 0.,
                fitw2       = 0.,
                fitangle    = 0.;
        int     x           = savex - 1,
                y           = sb->getImage()->height() - savey,
                imagesize   = 0,
                imageoffset = 0,
                xoff        = 0,
                yoff        = 0;

        if (fitfunction->currentText() == "Gauss") {
            fitx = fitresult[2];
            fity = fitresult[3];
            fitw1 = fitresult[4];
            fitw2 = fitresult[5];
            fitangle = fitresult[6];
        } else if (fitfunction->currentText() == "Sersic") {
            fitx = fitresult[3];
            fity = fitresult[4];
            fitw1 = fitresult[2];
            fitw2 = fitresult[2] * fitresult[6];
            fitangle = fitresult[5];
        }

        if (zoom > zoomIndex) {
            imagesize = 150 / (zoom - (zoomIndex-1));
        } else if (zoom < zoomIndex) {
            imagesize = 150 * (zoomIndex+1 - zoom);
        } else {
            imagesize = 150;
        }
        if (imagesize % 2) {
            imagesize++;
        }

        QImage subimage = sb->getImage()->copy(x - imagesize / 2, y - imagesize / 2,
                                               imagesize + 1, imagesize + 1);
        if (zoom > zoomIndex) {
            subimage = subimage.scaled(imagesize * (zoom - (zoomIndex-1)),
                                       imagesize*(zoom - (zoomIndex-1)));
        } else if (zoom < zoomIndex) {
            subimage = subimage.scaled(imagesize / (zoomIndex+1-zoom),
                                       imagesize/(zoomIndex+1-zoom));
        }

        imageoffset = (150 - subimage.width());

        QPainter p;
        p.begin(this);
        p.drawImage((width() - 150 + imageoffset) / 2, imageoffset / 2, subimage);
        p.setPen(QColor(0, 200, 0));
        p.translate(width() / 2, 150 / 2);
        if (zoom >= zoomIndex) {
            p.drawRect(-savefitwidth*(zoom - (zoomIndex-1)),
                       -savefitwidth*(zoom - (zoomIndex-1)),
                       (savefitwidth*2+1)*(zoom - (zoomIndex-1)),
                       (savefitwidth*2+1)*(zoom - (zoomIndex-1)));
            p.translate((fitx - savex) * (double)(zoom - (zoomIndex-1)) - 1,
                        -(fity - savey) * (double)(zoom - (zoomIndex-1)));
            p.rotate(-fitangle);
            p.rotate(-sb->getRotation());
            if ((sb->getRotation() == 90) || (sb->getRotation() == 180))
            {
                p.translate(0, -(zoom - (zoomIndex-1))/2);
            }
            if ((sb->getRotation() == 180) || (sb->getRotation() == 270))
            {
                p.translate((zoom - (zoomIndex-1))/2, 0);
            }
            xoff = fitw1 * (zoom - (zoomIndex-1));
            yoff = fitw2 * (zoom - (zoomIndex-1));
        } else if (zoom < zoomIndex) {
            p.drawRect(-savefitwidth/(zoomIndex+1 - zoom),
                       -savefitwidth/(zoomIndex+1 - zoom),
                       (savefitwidth*2+1)/(zoomIndex+1 - zoom),
                       (savefitwidth*2+1)/(zoomIndex+1 - zoom));
            p.translate((fitx - savex) / (double)(zoomIndex+1 - zoom) - 1,
                        -(fity - savey) / (double)(zoomIndex+1 - zoom));
            p.rotate(-fitangle);
            p.rotate(-sb->getRotation());
            xoff = fitw1 / (zoomIndex+1 - zoom);
            yoff = fitw2 / (zoomIndex+1 - zoom);
            if ((sb->getRotation() == 90) || (sb->getRotation() == 180))
            {
                p.translate(0, -(zoomIndex+1 - zoom)/2);
            }
            if ((sb->getRotation() == 180) || (sb->getRotation() == 270)) {
                p.translate((zoomIndex+1 - zoom)/2, 0);
            }
        }
        p.drawEllipse(-(int)(xoff/2.-.5), -(int)(yoff/2.-.5),
                      (int)(xoff+.5), (int)(yoff+.5));
        p.end();
    }
}

void QFits2dFit::resizeEvent(QResizeEvent *e) {
    update();
}

void QFits2dFit::zoomChanged(int value) {
    zoom = 150 / (value * 2) + 8;
    update();
}

void QFits2dFit::fitfunctionChanged(const QString &newfunction) {
    if (newfunction == "Sersic") {
        slm->show();
        fitslm->show();
    } else {
        slm->hide();
        fitslm->hide();
    }
    refit();
}

void QFits2dFit::centre(int cx, int cy, int rx, int ry) {
    cenx = rx;
    ceny = ry;
    cencx = cx;
    cency = cy;
}

void QFits2dFit::fitGauss() {
    QFitsSingleBuffer *sb = fitsMainWindow->getActualSB();
    if (sb != NULL) {
        if ((sb->Naxis(1) < 4) || (sb->Naxis(2) < 4)) {
            QMessageBox::information(this, "QFitsView", "Gaussfit needs at least a "
                                           "4 pixel wide image");
            return;
        }
        savex = cencx;
        savey = cency;
        fitx = cenx;
        fity = ceny;
        int maxwidth = sb->Naxis(1);
        if (sb->Naxis(2) < sb->Naxis(1)) {
            maxwidth = sb->Naxis(2);
        }
        fitwindow->setMaximum(maxwidth / 2 - 1);

        refit();
    }
}

void QFits2dFit::refit() {
    QFitsSingleBuffer *sb = fitsMainWindow->getActualSB();
    if ((sb != NULL) && (sb->getDpData()->type == typeFits)) {
        Fits    result;
        QString text1, text2, text3;
        double  slmvalue = -1.0,
                chisq = 0.;
        char    **lab = NULL,
                *labgauss[]  = {"Offset:", "Height:", "Image X:", "Image Y:",
                                "FWHM1:", "FWHM2:", "Angle:" },
                *labmoffat[] = {"Offset:", "Height:", "Power:", "Image X:", "Image Y:",
                                "FWHM1:", "FWHM2:", "Angle:" },
                *labsersic[] = {"Offset:", "Height:", "Re:", "Image X:", "Image Y:",
                                "Angle:", "Diskiness:", "Sersic index:" };

        if (fitslm->isChecked()) {
            slmvalue = slm->value();
        }

        savefitwidth = fitwindow->value();
        if (sb->getDpData()->fvalue->Naxis(3) > 1) {
            if (fitfunction->currentText() == "Gauss") {
                chisq = gauss2dsimplefit(result, *(sb->getCubeDisplay2D()),
                                         fitx, fity, fitwindow->value());
            } else if (fitfunction->currentText() == "Moffat") {
                chisq = moffat2dsimplefit(result, *(sb->getCubeDisplay2D()),
                                          fitx, fity, fitwindow->value());
            } else if (fitfunction->currentText() == "Sersic") {
                chisq = sersic2dsimplefit(result, *(sb->getCubeDisplay2D()),
                                          fitx, fity, fitwindow->value(), slmvalue);
            }
        } else {
            QReadLocker locker(&buffersLock);
            if (fitfunction->currentText() == "Gauss") {
                chisq = gauss2dsimplefit(result, *(sb->getDpData()->fvalue),
                                         fitx, fity, fitwindow->value());
    //        } else if (fitfunction->currentText() == "Moffat") {
    //            chisq = moffat2dsimplefit(result, *(sb->getDpData()->fvalue),
    //                                      fitx, fity, fitwindow->value());
            } else if (fitfunction->currentText() == "Sersic") {
                chisq = sersic2dsimplefit(result, *(sb->getDpData()->fvalue),
                                          fitx, fity, fitwindow->value(), slmvalue);
            }
        }

        for (int i = 0; i < result.Nelements(); i++) {
            fitresult[i] = result.r8data[i];
        }

        if (fitfunction->currentText() == "Gauss") {
            lab = labgauss;
        } else if (fitfunction->currentText() == "Moffat") {
            lab = labmoffat;
        } else if (fitfunction->currentText() == "Sersic") {
            lab = labsersic;
        }

        for (int i = 0; i < result.Nelements() / 2; i++) {
            text1 += QString(lab[i]) + "\n";
            text2 += QString::number(result.r8data[i], 'g', 5) + " +- \n";
            text3 += QString::number(result.r8data[i+result.Nelements() / 2], 'g', 3) + "\n";
        }
        if (fitfunction->currentText() == "Gauss") {
            text1 += "flux:\n";
            text2 += QString::number(M_PI * result.r8data[1] *
                                     fabs(result.r8data[4] * result.r8data[5]) /
                                                     (4.0*log(2.0)), 'g', 7) + "\n";
        }

        text1 += "chisq:";
        text2 += QString::number(chisq, 'g', 3);

        resultLabel1->setText(text1);
        resultLabel1->adjustSize();
        resultLabel2->setText(text2);
        resultLabel2->adjustSize();
        resultLabel2->move(resultLabel1->x() + resultLabel1->width() + 5,
                           resultLabel1->y());
        resultLabel3->setText(text3);
        resultLabel3->adjustSize();
        resultLabel3->move(resultLabel2->x() + resultLabel2->width() + 5,
                           resultLabel2->y());

        show();
        raise();
        update();
    }
}

void QFits2dFit::createFitFunction() {
    QFitsSingleBuffer *sb = fitsMainWindow->getActualSB();
    if (sb != NULL) {
        QString cmd = freeBufferName().c_str();
        cmd += " = ";
        cmd += QString::number(fitresult[0]) + " + ";
        cmd += QString::number(fitresult[1]) + " * ";

        if (fitfunction->currentText() == "Gauss") {
            cmd += "gauss(";
            cmd += QString::number(fitresult[2]) + ", ";
            cmd += QString::number(fitresult[3]) + ", ";
            cmd += QString::number(fitresult[4]) + ", ";
            cmd += QString::number(fitresult[5]) + ", ";
            cmd += QString::number(fitresult[6]) + ", ";
            cmd += "naxis1 = " + QString::number(sb->Naxis(1));
            cmd += ", naxis2 = " + QString::number(sb->Naxis(2));
            cmd += ")";
        } else if (fitfunction->currentText() == "Sersic") {
            cmd += "sersic2d(";
            cmd += QString::number(fitresult[2]) + ", ";
            cmd += QString::number(fitresult[3]) + ", ";
            cmd += QString::number(fitresult[4]) + ", ";
            cmd += QString::number(fitresult[5]) + ", ";
            cmd += QString::number(fitresult[6]) + ", ";
            if (fitslm->isChecked()) {
                cmd += QString::number(slm->value()) + ", ";
            } else {
                cmd += QString::number(fitresult[7]) + ", ";
            }
            cmd += "naxis1 = " + QString::number(sb->Naxis(1));
            cmd += ", naxis2 = " + QString::number(sb->Naxis(2));
            cmd += ")";
        }

        dpuser_widget->executeCommand(cmd);
    }
}
