/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file preferences.c
 * \brief Preferences dialog
 */

#include <ffgtk.h>
#include <preferences.h>

#define NAME_COUNT  28

void bluetoothSetList( GtkWidget *psWidget );
int bluetoothSetHeadset( const gchar *pnName );
GList *getActions( void );
struct sAction *findAction( const gchar *pnName );
struct sAction *createAction( const gchar *pnName );
void removeAction( const gchar *pnName );

GList *getDtmfActions( void );
struct sDtmf *findDtmfAction( const gchar *pnName );
struct sDtmf *createDtmfAction( const gchar *pnName );
void removeDtmfAction( const gchar *pnName );

void transferPassword( struct sProfile *psProfile, gchar *pnPlugin );

int assistant( void );
static GtkWidget *psPreferencesDialog = NULL;
static GtkTreeModel *psListStore = NULL;
static GtkWidget *psNotebook = NULL;
static GtkWidget *psRouterHostEntry = NULL;
static GtkWidget *psRouterPasswordEntry = NULL;
static GtkWidget *psRouterLineAccessCodeEntry = NULL;
static GtkWidget *psRouterInternationalPrefixEntry = NULL;
static GtkWidget *psRouterCountryCodeEntry = NULL;
static GtkWidget *psRouterNationalPrefixEntry = NULL;
static GtkWidget *psRouterAreaCodeEntry = NULL;
static GtkWidget *psRouterPortComboBox = NULL;
static GtkWidget *psRouterVerifyButton = NULL;
static GtkWidget *psRouterDetectButton = NULL;
static GtkWidget *psRouterBoxLabel = NULL;

static GtkWidget *psCallMonitorBox = NULL;
static GtkWidget *psCallMonitorTypeButton = NULL;
static GtkWidget *psCallMonitorDateTimeButton = NULL;
static GtkWidget *psCallMonitorNameButton = NULL;
static GtkWidget *psCallMonitorCompanyButton = NULL;
static GtkWidget *psCallMonitorCityButton = NULL;
static GtkWidget *psCallMonitorNumberButton = NULL;
static GtkWidget *psCallMonitorLocalNameButton = NULL;
static GtkWidget *psCallMonitorLocalNumberButton = NULL;
static GtkWidget *psCallMonitorDurationButton = NULL;

static GtkWidget *psCallMonitorSystemRadio = NULL;
static GtkWidget *psCallMonitorFfgtkRadio = NULL;
static GtkWidget *psCallMonitorPopupPositionComboBox = NULL;
static GtkWidget *psCallMonitorTimeSpinButton = NULL;
static GtkWidget *psCallMonitorReverseInCheckButton = NULL;
static GtkWidget *psCallMonitorReverseOutCheckButton = NULL;
static GtkWidget *psCallMonitorMsnTreeView = NULL;
static GtkWidget *psCallMonitorPlaySoundsCheckButton = NULL;
static GtkWidget *psCallMonitorIncomingSoundChooser = NULL;
static GtkWidget *psCallMonitorOutgoingSoundChooser = NULL;
static GtkWidget *psCallMonitorLoadListOnStartupCheckButton = NULL;
static GtkWidget *psCallMonitorReplaceOnLoad = NULL;

static GtkWidget *psVoiceBoxUserEntry = NULL;
static GtkWidget *psVoiceBoxPasswordEntry = NULL;

static GtkWidget *psFaxSenderEntry = NULL;
static GtkWidget *psFaxIdEntry = NULL;
static GtkWidget *psFaxMsnEntry = NULL;
static GtkWidget *psFaxBitrateComboBox = NULL;
static GtkWidget *psFaxControllerComboBox = NULL;
static GtkWidget *psFaxAcceptIncomingCheckButton = NULL;
static GtkWidget *psFaxEcmToggle = NULL;
static GtkWidget *psFaxReportToggle = NULL;
static GtkWidget *psFaxReportDir = NULL;

static GtkWidget *psActionsTreeView = NULL;
static GtkListStore *psActionsStore = NULL;
static GtkWidget *psActionsAddButton = NULL;
static GtkWidget *psActionsDelButton = NULL;
static GtkWidget *psActionsEditButton = NULL;
static GtkWidget *psActionsDescriptionLabel = NULL;
static GtkWidget *psActionsInStartToggle = NULL;
static GtkWidget *psActionsInStopToggle = NULL;
static GtkWidget *psActionsOutStartToggle = NULL;
static GtkWidget *psActionsOutStopToggle = NULL;
static GtkWidget *psActionsInRingToggle = NULL;
static GtkWidget *psActionsInMissedToggle = NULL;
static GtkWidget *psActionsOutDialToggle = NULL;

static GtkWidget *psPluginsReverseComboBox = NULL;
static GtkWidget *psPluginsAddressComboBox = NULL;
static GtkWidget *psPluginsAddressBookEdit = NULL;
static GtkWidget *psPluginsPasswordComboBox = NULL;
static GtkWidget *psPluginsAudioComboBox = NULL;
static GtkWidget *psPluginsAudioEdit = NULL;

static GtkWidget *psMiscToolbarStyleComboBox = NULL;
static GtkWidget *psMiscNumberFormatComboBox = NULL;

static GtkWidget *psDebugFfgtkComboBox = NULL;
static GtkWidget *psDebugFaxComboBox = NULL;

static struct sAction *psSelectedAction = NULL;

static GtkWidget *psSoftphoneMsnEntry = NULL;
static GtkWidget *psSoftphoneControllerComboBox = NULL;
static GtkWidget *psBluetoothComboBox = NULL;
static GtkWidget *psDtmfActionsTreeView = NULL;
static GtkListStore *psDtmfActionsStore = NULL;
static GtkWidget *psDtmfActionAddButton = NULL;
static GtkWidget *psDtmfActionDelButton = NULL;
static GtkWidget *psDtmfActionEditButton = NULL;

struct sPorts {
	char *pnName;
	int nType;
};

static struct sPorts asPortNames[ NAME_COUNT ] = {
	/* Analog */
	{ "Analog1", PORT_ANALOG1 },
	{ "Analog2", PORT_ANALOG2 },
	{ "Analog3", PORT_ANALOG3 },
	/* ISDN */
	{ "IsdnAll", PORT_ISDNALL },
	{ "Isdn1", PORT_ISDN1 },
	{ "Isdn2", PORT_ISDN2 },
	{ "Isdn3", PORT_ISDN3 },
	{ "Isdn4", PORT_ISDN4 },
	{ "Isdn5", PORT_ISDN5 },
	{ "Isdn6", PORT_ISDN6 },
	{ "Isdn7", PORT_ISDN7 },
	{ "Isdn8", PORT_ISDN8 },
	/* DECT */
	{ "Dect1", PORT_DECT1 },
	{ "Dect2", PORT_DECT2 },
	{ "Dect3", PORT_DECT3 },
	{ "Dect4", PORT_DECT4 },
	{ "Dect5", PORT_DECT5 },
	{ "Dect6", PORT_DECT6 },
	/* IP-Phone */
	{ "IPPhone1", PORT_IP1 },
	{ "IPPhone2", PORT_IP2 },
	{ "IPPhone3", PORT_IP3 },
	{ "IPPhone4", PORT_IP4 },
	{ "IPPhone5", PORT_IP5 },
	{ "IPPhone6", PORT_IP6 },
	{ "IPPhone7", PORT_IP7 },
	{ "IPPhone8", PORT_IP8 },
	{ "IPPhone9", PORT_IP9 },
	{ "IPPhone10", PORT_IP10 }
};

/**
 * \brief Set standard port names
 * \param psProfile profile containing the router information
 */
void routerSetStandardPorts( struct sProfile *psProfile ) {
	int nIndex;

	for ( nIndex = 0; nIndex < NAME_COUNT; nIndex++ ) {
		gchar *pnTmp = g_strdup_printf( "/ffgtk/router/%s", asPortNames[ nIndex ].pnName );
		prefsSetString( psProfile, pnTmp, "" );
		g_free( pnTmp );
	}
}

/**
 * \brief Add exisiting router ports to combobox
 * \param psProfile profile containing the router information
 * \param psComboBox combo box where we add the information
 * \return error code (0=success, else error)
 */
void routerAddComboBoxPorts( struct sProfile *psProfile, GtkWidget *psComboBox ) {
	const char *pnName = NULL;
	const gchar *pnPortName = NULL;
	GtkListStore *psStore;
	int nPorts;

	if ( psProfile == NULL ) {
		return;
	}

	nPorts = routerGetPorts( psProfile );

	psStore = GTK_LIST_STORE( gtk_combo_box_get_model( GTK_COMBO_BOX( psComboBox ) ) );
	gtk_list_store_clear( psStore );

	pnName = routerGetDefaultFon( psProfile );

	if ( nPorts & PORT_ANALOG1 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 0 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ANALOG2 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 1 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ANALOG3 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 2 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ISDNALL ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 3 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ISDN1 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 4 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ISDN2 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 5 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ISDN3 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 6 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ISDN4 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 7 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ISDN5 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 8 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ISDN6 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 9 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ISDN7 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 10 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_ISDN8 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 11 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_DECT1 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 12 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_DECT2 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 13 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_DECT3 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 14 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_DECT4 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 15 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_DECT5 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 16 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_DECT6 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 17 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_IP1 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 18 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_IP2 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 19 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_IP3 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 20 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_IP4 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 21 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_IP5 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 22 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_IP6 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 23 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_IP7 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 24 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_IP8 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 25 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_IP9 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 26 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
	if ( nPorts & PORT_IP10 ) {
		pnPortName = routerGetPortName( psProfile, asPortNames[ 27 ].pnName );
		if ( pnPortName != NULL && strlen( pnPortName ) > 0 ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), pnPortName );
		}
	}
#ifdef HAVE_FAXOPHONE
	gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBox ), _( "Softphone" ) );
#endif

	if ( pnName != NULL && strlen( pnName ) > 0 ) {
		GtkTreeModel *psModel = gtk_combo_box_get_model( GTK_COMBO_BOX( psComboBox ) );
		gboolean bValid;
		GtkTreeIter sIter;
		int nIndex = 0;
		bool bSet = FALSE;

		bValid = gtk_tree_model_get_iter_first( psModel, &sIter);
		while ( bValid ) {
			gchar *pnText;
			gtk_tree_model_get( psModel, &sIter, 0, &pnText, -1 );
			bValid = gtk_tree_model_iter_next( psModel, &sIter );
			if ( strcmp( pnName, pnText ) == 0 ) {
				gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBox ), nIndex );
				bSet = TRUE;
			}
			g_free( pnText );
			nIndex++;
		}
		if ( bSet == FALSE ) {
			gtk_combo_box_set_active ( GTK_COMBO_BOX( psComboBox ), 0 );
		}
	} else {
		gtk_combo_box_set_active ( GTK_COMBO_BOX( psComboBox ), 0 );
	}
}

/**
 * \brief Get router port type
 * \param psProfile profile structure
 * \param pnName router port name
 * \return router port type
 */
int routerGetPortType( struct sProfile *psProfile, const char *pnName ) {
	const char *pnPortName;
	int nIndex;

	if ( pnName != NULL ) {
		for ( nIndex = 0; nIndex < NAME_COUNT; nIndex++ ) {
			pnPortName = routerGetPortName( psProfile, asPortNames[ nIndex ].pnName );
			if ( strcmp( pnName, pnPortName ) == 0 ) {
				return asPortNames[ nIndex ].nType;
			}
		}
	} else {
		return asPortNames[ 0 ].nType;
	}

	return 0;
}

/** Get and set softphone msn */
CREATE_STRING_PREFS( softphone, Msn, "/ffgtk/softphone/msn" );
/** Get and set softphone controller */
CREATE_INT_PREFS( softphone, Controller, "/ffgtk/softphone/controller" );
/** Get and set softphone headset */
CREATE_STRING_PREFS( softphone, Headset, "/ffgtk/softphone/headset" );

/** Get and set msn */
CREATE_STRING_PREFS( fax, SenderMsn, "/ffgtk/fax/msn" );
/** Get and set sender name */
CREATE_STRING_PREFS( fax, SenderName, "/ffgtk/fax/sender" );
/** Get and set sender number */
CREATE_STRING_PREFS( fax, SenderNumber, "/ffgtk/fax/faxid" );
/** Get and set fax controller */
CREATE_INT_PREFS( fax, Controller, "/ffgtk/fax/controller" );
/** Get and set bitrate index */
CREATE_INT_PREFS( fax, BitrateIndex, "/ffgtk/fax/bitrate" );
/** Get and set report dir */
CREATE_PATH_PREFS( fax, ReportDir, "/ffgtk/fax/reportdir" );
/** Get and set report dir state */
CREATE_BOOL_PREFS( fax, ReportToggle, "/ffgtk/fax/report" );
/** Get and set error correction mode state */
CREATE_BOOL_PREFS( fax, Ecm, "/ffgtk/fax/ecm" );
/** Get and set name visible flag */
CREATE_BOOL_PREFS( callMonitor, Name, "/ffgtk/callmonitor/name_visible" );
/** Get and set current password type (0 = local, 1 = gnome keyring) */
CREATE_INT_PREFS( password, Type, "/ffgtk/misc/password_type" );
/** Get and set popup position */
CREATE_INT_PREFS( callMonitor, PopupPosition, "/ffgtk/callmonitor/popupposition" );
/** Get and set play sounds state */
CREATE_BOOL_PREFS( callMonitor, PlaySounds, "/ffgtk/callmonitor/playsound" );
/** Get and set call monitor paned position */
CREATE_INT_PREFS( callMonitor, PanedPosition, "/ffgtk/callmonitor/paned_position" );
/** Get and set accept fax state */
CREATE_BOOL_PREFS( fax, Accept, "/ffgtk/fax/accept" );
/** Get and set default router fon */
CREATE_STRING_PREFS( router, DefaultFon, "/ffgtk/router/fon" );
/** Get and set type visible flag */
CREATE_BOOL_PREFS( callMonitor, Type, "/ffgtk/callmonitor/type_visible" );
/** Get and set date/time visible flag */
CREATE_BOOL_PREFS( callMonitor, DateTime, "/ffgtk/callmonitor/datetime_visible" );
/** Get and set company visible flag */
CREATE_BOOL_PREFS( callMonitor, Company, "/ffgtk/callmonitor/company_visible" );
/** Get and set city visible flag */
CREATE_BOOL_PREFS( callMonitor, City, "/ffgtk/callmonitor/city_visible" );
/** Get and set number visible flag */
CREATE_BOOL_PREFS( callMonitor, Number, "/ffgtk/callmonitor/number_visible" );
/** Get and set local name visible flag */
CREATE_BOOL_PREFS( callMonitor, LocalName, "/ffgtk/callmonitor/localname_visible" );
/** Get and set local number visible flag */
CREATE_BOOL_PREFS( callMonitor, LocalNumber, "/ffgtk/callmonitor/localnumber_visible" );
/** Get and set duration visible flag */
CREATE_BOOL_PREFS( callMonitor, Duration, "/ffgtk/callmonitor/duration_visible" );
/** Get and set incoming reverse lookup state */
CREATE_BOOL_PREFS( callMonitor, ReverseLookupIn, "/ffgtk/callmonitor/in" );
/** Get and set outgoing reverse lookup state */
CREATE_BOOL_PREFS( callMonitor, ReverseLookupOut, "/ffgtk/callmonitor/out" );
/** Get and set load on startup state */
CREATE_BOOL_PREFS( callMonitor, LoadOnStartup, "/ffgtk/callmonitor/loadonstartup" );
/** Get and set replace on load */
CREATE_BOOL_PREFS( callMonitor, ReplaceOnLoad, "/ffgtk/callmonitor/replaceload" );
/** Get and set voice box user name */
CREATE_STRING_PREFS( voiceBox, User, "/ffgtk/voicebox/user" );
/** Get and set voice box password */
CREATE_STRING_PREFS( voiceBox, Password, "/ffgtk/voicebox/password" );
/** Get and set address book number format */
CREATE_INT_PREFS( addressBook, NumberFormat, "/ffgtk/addressbook/numberformat" );
/** Get and set incoming sound */
CREATE_PATH_PREFS( callMonitor, IncomingSound, "/ffgtk/callmonitor/incoming_sound" );
/** Get and set outgoing sound */
CREATE_PATH_PREFS( callMonitor, OutgoingSound, "/ffgtk/callmonitor/outgoing_sound" );
/** Get and set router host */
CREATE_STRING_PREFS( router, Host, "/ffgtk/router/host" );
/** Get and set firmware version */
CREATE_STRING_PREFS( router, Firmware, "/ffgtk/router/firmware" );
/** Get and set popup display time */
CREATE_INT_PREFS( callMonitor, DisplayTime, "/ffgtk/callmonitor/displaytime" );
/** Get and set country code */
CREATE_STRING_PREFS( router, CountryCode, "/ffgtk/router/countrycode" );
/** Get and set line access code */
CREATE_STRING_PREFS( router, LineAccessCode, "/ffgtk/router/lineaccesscode" );
/** Get and set area code */
CREATE_STRING_PREFS( router, AreaCode, "/ffgtk/router/areacode" );
/** Get and set national prefix */
CREATE_STRING_PREFS( router, NationalPrefix, "/ffgtk/router/nationalprefix" );
/** Get and set international prefix */
CREATE_STRING_PREFS( router, InternationalPrefix, "/ffgtk/router/internationalprefix" );
/** Get and set toolbar style */
CREATE_INT_PREFS( misc, ToolbarStyle, "/ffgtk/misc/toolbarstyle" );
/** Get and set lookup plugin name */
CREATE_STRING_PREFS( plugins, Lookup, "/ffgtk/plugins/lookup" );
/** Get and set password plugin */
CREATE_STRING_PREFS( plugins, Password, "/ffgtk/plugins/password" );
/** Get and set address book plugin */
CREATE_STRING_PREFS( plugins, Address, "/ffgtk/plugins/book" );
/** Get and set fax plugin */
CREATE_STRING_PREFS( plugins, Fax, "/ffgtk/plugins/fax" );
/** Get and set fax debug level */
CREATE_INT_PREFS( debug, FaxLevel, "/ffgtk/debug/fax" );
/** Get and set fax id */
CREATE_STRING_PREFS( fax, Id, "/ffgtk/fax/faxid" );
/** Get and set fax msn */
CREATE_STRING_PREFS( fax, Msn, "/ffgtk/fax/msn" );
/** Get and set audio plugin */
CREATE_STRING_PREFS( plugins, Audio, "/ffgtk/plugins/audio" );
/** Get and set system notification */
CREATE_BOOL_PREFS( callMonitor, SystemNotification, "/ffgtk/callmonitor/systemnotification" );
/** Get and set ffgtk notification */
CREATE_BOOL_PREFS( callMonitor, FfgtkNotification, "/ffgtk/callmonitor/ffgtknotification" );

/** 
 * \brief Set fax bitrate
 * \param psProfile profile structure
 * \param nBitrate bitrate
 */
void faxSetBitrate( struct sProfile *psProfile, int nBitrate ) {
	prefsSetInt( psProfile, "/ffgtk/fax/bitrate", nBitrate );
}

/**
 * \brief Get bitrate
 * \param psProfile profile structure
 * \return bitrate for faxing
 */
int faxGetBitrate( struct sProfile *psProfile ) {
	int anRates[ 4 ] = { 2400, 4800, 9600, 14400 };
	int nBitrate = faxGetBitrateIndex( psProfile );

	if ( nBitrate < 0 || nBitrate > 3 ) {
		return 14400;
	}

	return anRates[ nBitrate ];
}

/**
 * \brief Get fritzfax debug level
 * \param psProfile profile structure
 * \return fritzfax debug level
 */
int debugGetLevel( struct sProfile *psProfile ) {
	if ( psProfile == NULL ) {
		return 3;
	}

	return prefsGetInt( psProfile, "/ffgtk/debug/ffgtk" );
}

/**
 * \brief Get incoming call state
 * \param psProfile profile structure
 * \param pnNumber check state for this number
 * \return incoming call state
 */
gboolean callMonitorGetIncomingState( struct sProfile *psProfile, const gchar *pnNumber ) {
	int nType = routerGetFonType( psProfile, pnNumber );

	if ( nType == -1 ) {
		return FALSE;
	}

	return routerGetFonFlags( psProfile, nType ) & 2;
}

/**
 * \brief Get outgoing call state
 * \param psProfile profile structure
 * \param pnNumber check state for this number
 * \return outgoing call state
 */
gboolean callMonitorGetOutgoingState( struct sProfile *psProfile, const gchar *pnNumber ) {
	int nType = routerGetFonType( psProfile, pnNumber );

	if ( nType == -1 ) {
		return FALSE;
	}

	return routerGetFonFlags( psProfile, nType ) & 4;
}

/**
 * \brief Set call monitor size
 * \param psProfile profile structure
 * \param nWidth width
 * \param nHeight height
 */
void callMonitorSetSize( struct sProfile *psProfile, gint nWidth, gint nHeight ) {
	prefsSetInt( psProfile, "/ffgtk/callmonitor/width", nWidth );
	prefsSetInt( psProfile, "/ffgtk/callmonitor/height", nHeight );
}

/**
 * \brief Get call monitor size
 * \param psProfile profile structure
 * \param pnWidth width
 * \param pnHeight height
 */
void callMonitorGetSize( struct sProfile *psProfile, gint *pnWidth, gint *pnHeight ) {
	if ( psProfile != NULL ) {
		*pnWidth = prefsGetInt( psProfile, "/ffgtk/callmonitor/width" );
		*pnHeight = prefsGetInt( psProfile, "/ffgtk/callmonitor/height" );
	}
}

/**
 * \brief Set mute flag
 * \param psProfile profile structure
 * \param bSet Mute flag set
 */
void callMonitorSetMute( struct sProfile *psProfile, bool bSet ) {
	if ( psProfile == NULL ) {
		return;
	}

	prefsSetBool( psProfile, "/ffgtk/callmonitor/mute", bSet );
}

/**
 * \brief Get mute flag
 * \param psProfile profile structure
 * \return mute flag
 */
gboolean callMonitorGetMute( struct sProfile *psProfile ) {
	if ( psProfile == NULL ) {
		return FALSE;
	}

	return prefsGetBool( psProfile, "/ffgtk/callmonitor/mute" );
}


/**
 * \brief Get router password
 * \param psProfile profile structure
 * \return router password
 */
const gchar *routerGetPassword( struct sProfile *psProfile ) {
	struct sPassword *psPassword = NULL;

	psPassword = getDefaultPasswordPlugin( psProfile );
	if ( psPassword != NULL ) {
		return psPassword -> GetPassword( psProfile );
	}

	return prefsGetString( psProfile, "/ffgtk/router/password" );
}

/**
 * \brief Set router password
 * \param psProfile profile structure
 * \param pnPassword password
 */
void routerSetPassword( struct sProfile *psProfile, const gchar *pnPassword ) {
	struct sPassword *psPassword = NULL;

	psPassword = getDefaultPasswordPlugin( psProfile );
	if ( psPassword != NULL ) {
		psPassword -> SetPassword( psProfile, pnPassword );
		return;
	}

	prefsSetString( psProfile, "/ffgtk/router/password", pnPassword );
}

/**
 * \brief Remove router password
 * \param psProfile profile structure
 */
void routerRemovePassword( struct sProfile *psProfile ) {
	struct sPassword *psPassword = NULL;

	psPassword = getDefaultPasswordPlugin( psProfile );
	if ( psPassword != NULL ) {
		psPassword -> RemovePassword( psProfile );
		return;
	}

	prefsSetString( psProfile, "/ffgtk/router/password", "" );
}

/**
 * \brief Set router ports
 * \param psProfile profile structure
 * \param nPorts ports present on router
 */
void routerSetPorts( struct sProfile *psProfile, int nPorts ) {
	prefsSetInt( psProfile, "/ffgtk/router/ports", nPorts );
}

/**
 * \brief Get port name
 * \param psProfile profile structure
 * \param pnPort port id
 * \return port name
 */
const char *routerGetPortName( struct sProfile *psProfile, char *pnPort ) {
	char *pnVar = g_strdup_printf( "/ffgtk/router/%s", pnPort );
	const char *pnName;

	pnName = prefsGetString( psProfile, pnVar );
	g_free( pnVar );

	return pnName;
}

/**
 * \brief Set port name
 * \param psProfile profile structure
 * \param nType port type
 * \param pnName port name
 */
void routerSetPortName( struct sProfile *psProfile, int nType, char *pnName ) {
	int nIndex = 0;

	while ( nType >> 1 ) {
		nIndex++;
		nType >>= 1;
	}

	char *pnVar = g_strdup_printf( "/ffgtk/router/%s", asPortNames[ nIndex ].pnName );

	prefsSetString( psProfile, pnVar, pnName );
	g_free( pnVar );
}

/**
 * \brief Get router ports
 * \param psProfile profile structure
 * \return ports present on router
 */
int routerGetPorts( struct sProfile *psProfile ) {
	return prefsGetInt( psProfile, "/ffgtk/router/ports" );
}

/**
 * \brief Set router fon number
 * \param psProfile profile structure
 * \param nType which fon
 * \param pnMsn number
 */
void routerSetFonNumber( struct sProfile *psProfile, int nType, const gchar *pnMsn ) {
	gchar *pnTmp;

	switch ( nType ) {
		case 0 ... 9:
			pnTmp = g_strdup_printf( "/ffgtk/router/msn%d/number", nType );
			break;
		case 10:
			pnTmp = g_strdup_printf( "/ffgtk/router/pots/number" );
			break;
		case 11 ... 29:
			pnTmp = g_strdup_printf( "/ffgtk/router/sip%d/number", nType - 11 );
			break;
		case 30 ... 39:
			pnTmp = g_strdup_printf( "/ffgtk/router/ip%d/number", nType - 30 );
			break;
		default:
			Debug( KERN_WARNING, "Wrong type (%d), abort!\n", nType );
			return;
	}

	prefsSetString( psProfile, pnTmp, pnMsn );
	g_free( pnTmp );
}

/**
 * \brief Clear fon numbers
 * \param psProfile profile structure
 */
void routerClearNumbers( struct sProfile *psProfile ) {
	gint nIndex;

	for ( nIndex = 0; nIndex < 39; nIndex++ ) {
		routerSetFonNumber( psProfile, nIndex, "" );
	}
}

/**
 * \brief Get router fon number
 * \param psProfile profile structure
 * \param nType which fon
 * \return fon number or NULL
 */
const gchar *routerGetFonNumber( struct sProfile *psProfile, int nType ) {
	gchar *pnTmp = NULL;
	const gchar *pnMsn = NULL;

	switch ( nType ) {
		case 0 ... 9:
			pnTmp = g_strdup_printf( "/ffgtk/router/msn%d/number", nType );
			break;
		case 10:
			pnTmp = g_strdup_printf( "/ffgtk/router/pots/number" );
			break;
		case 11 ... 29:
			pnTmp = g_strdup_printf( "/ffgtk/router/sip%d/number", nType - 11 );
			break;
		case 30 ... 39:
			pnTmp = g_strdup_printf( "/ffgtk/router/ip%d/number", nType - 30 );
			break;
		default:
			Debug( KERN_WARNING, "Wrong type (%d), about\n", nType );
			return NULL;
	}
	pnMsn = prefsGetString( psProfile, pnTmp );
	g_free( pnTmp );

	return pnMsn;
}

/**
 * \brief Set router fon flags (DIAL/IN/OUT)
 * \param psProfile profile structure
 * \param nType which fon
 * \param nFlags fon flags
 */
void routerSetFonFlags( struct sProfile *psProfile, int nType, int nFlags ) {
	gchar *pnTmp;

	switch ( nType ) {
		case 0 ... 9:
			pnTmp = g_strdup_printf( "/ffgtk/router/msn%d/flags", nType );
			break;
		case 10:
			pnTmp = g_strdup_printf( "/ffgtk/router/pots/flags" );
			break;
		case 11 ... 29:
			pnTmp = g_strdup_printf( "/ffgtk/router/sip%d/flags", nType - 11 );
			break;
		case 30 ... 39:
			pnTmp = g_strdup_printf( "/ffgtk/router/ip%d/flags", nType - 30 );
			break;
		default:
			Debug( KERN_WARNING, "Wrong type (%d), abort!\n", nType );
			return;
	}
	prefsSetInt( psProfile, pnTmp, nFlags );
	g_free( pnTmp );
}

/**
 * \brief Get router fon flags (DIAL/IN/OUT)
 * \param psProfile profile structure
 * \param nType which fon
 * \return fon flag
 */
int routerGetFonFlags( struct sProfile *psProfile, int nType ) {
	gchar *pnTmp = NULL;
	int nFlags = 0;

	switch ( nType ) {
		case 0 ... 9:
			pnTmp = g_strdup_printf( "/ffgtk/router/msn%d/flags", nType );
			break;
		case 10:
			pnTmp = g_strdup_printf( "/ffgtk/router/pots/flags" );
			break;
		case 11 ... 29:
			pnTmp = g_strdup_printf( "/ffgtk/router/sip%d/flags", nType - 11 );
			break;
		case 30 ... 39:
			pnTmp = g_strdup_printf( "/ffgtk/router/ip%d/flags", nType - 30 );
			break;
	}
	nFlags = prefsGetInt( psProfile, pnTmp );
	g_free( pnTmp );

	return nFlags;
}

/**
 * \brief Get router fon type
 * \param psProfile profile structure
 * \param pnNumber number
 * \return fon type
 */
int routerGetFonType( struct sProfile *psProfile, const gchar *pnNumber ) {
	const gchar *pnFon = NULL;
	int nIndex;
	gchar *pnFullCheck = NULL;
	gchar *pnFullNumber = NULL;

	if ( pnNumber == NULL ) {
		return -1;
	}

	pnFullNumber = fullNumber( pnNumber, TRUE );
	if ( pnFullNumber == NULL ) {
		return -1;
	}

	for ( nIndex = 0; nIndex < 39; nIndex++ ) {
		pnFon = routerGetFonNumber( psProfile, nIndex );
		if ( pnFon != NULL && strlen( pnFon ) > 0 ) {
			pnFullCheck = fullNumber( pnFon, TRUE );

			if ( !strcmp( pnFullCheck, pnFullNumber ) ) {
				g_free( pnFullCheck );
				g_free( pnFullNumber );
				return nIndex;
			}
			g_free( pnFullCheck );
		}
	}

	g_free( pnFullNumber );

	return -1;
}

/**
 * \brief Set fax sender
 * \param psProfile profile structure
 * \param pnSender fax sender name
 */
void faxSetSender( struct sProfile *psProfile, const gchar *pnSender ) {
	prefsSetString( psProfile, "/ffgtk/fax/sender", pnSender );
}

/**
 * \brief Set debug level
 * \param psProfile profile structure
 * \param nLevel debug level
 */
void debugSetLevel( struct sProfile *psProfile, int nLevel ) {
	prefsSetInt( psProfile, "/ffgtk/debug/ffgtk", nLevel );
}

/**
 * \brief Get action flags
 * \param psProfile profile structure
 * \param pnName action name
 * \return action flags
 */
int getActionFlags( struct sProfile *psProfile, const gchar *pnName ) {
	gchar *pnTmp = g_strdup_printf( "/ffgtk/actions/%s", pnName );
	int nFlags = 0;

	nFlags = prefsGetInt( psProfile, pnTmp );
	g_free( pnTmp );

	return nFlags;
}

/**
 * \brief Set action flags
 * \param psProfile profile structure
 * \param pnName action name
 * \param nFlags action flags
 */
void setActionFlags( struct sProfile *psProfile, const gchar *pnName, int nFlags ) {
	gchar *pnTmp = g_strdup_printf( "/ffgtk/actions/%s", pnName );

	prefsSetInt( psProfile, pnTmp, nFlags );
	g_free( pnTmp );
}

/**
 * \brief Load action information
 * \param psAction action structure
 */
static void loadAction( struct sAction *psAction ) {
	if ( psAction != NULL ) {
		gtk_label_set_text( GTK_LABEL( psActionsDescriptionLabel ), psAction -> pnDescription );
		int nFlags = getActionFlags( getActiveProfile(), psAction -> pnName );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psActionsInStartToggle ), nFlags & 0x01 );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psActionsInStopToggle ), nFlags & 0x02 );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psActionsOutStartToggle ), nFlags & 0x04 );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psActionsOutStopToggle ), nFlags & 0x08 );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psActionsInRingToggle ), nFlags & 0x10 );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psActionsInMissedToggle ), nFlags & 0x20 );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psActionsOutDialToggle ), nFlags & 0x40 );
	}
}

/**
 * \brief Save action information
 * \param psAction action structure
 */
static void saveAction( struct sAction *psAction ) {
	if ( psAction != NULL ) {
		int nFlags = getActionFlags( getActiveProfile(), psAction -> pnName );

		if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psActionsInStartToggle ) ) == TRUE ) {
			nFlags |= 0x01;
		} else {
			nFlags &= ~0x01;
		}
		if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psActionsInStopToggle ) ) == TRUE ) {
			nFlags |= 0x02;
		} else {
			nFlags &= ~0x02;
		}
		if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psActionsOutStartToggle ) ) == TRUE ) {
			nFlags |= 0x04;
		} else {
			nFlags &= ~0x04;
		}
		if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psActionsOutStopToggle ) ) == TRUE ) {
			nFlags |= 0x08;
		} else {
			nFlags &= ~0x08;
		}
		if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psActionsInRingToggle ) ) == TRUE ) {
			nFlags |= 0x10;
		} else {
			nFlags &= ~0x10;
		}
		if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psActionsInMissedToggle ) ) == TRUE ) {
			nFlags |= 0x20;
		} else {
			nFlags &= ~0x20;
		}
		if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psActionsOutDialToggle ) ) == TRUE ) {
			nFlags |= 0x40;
		} else {
			nFlags &= ~0x40;
		}

		setActionFlags( getActiveProfile(), psAction -> pnName, nFlags );
	}
}

/**
 * \brief Save profile information
 * \param psProfile profile structure
 */
void saveProfile( struct sProfile *psProfile ) {
	struct sFax *psFax;
	int nOldController;
	int nOldDebug;
	gchar *pnTmp;

	if ( psProfile == NULL ) {
		return;
	}

	nOldController = faxGetController( psProfile );
	nOldDebug = debugGetFaxLevel( psProfile );

	/* Router - Tab */
	routerSetHost( psProfile, gtk_entry_get_text( GTK_ENTRY( psRouterHostEntry ) ) );
	pnTmp = g_strdup( gtk_entry_get_text( GTK_ENTRY( psRouterPasswordEntry ) ) );
	routerSetPassword( psProfile, pnTmp );
	g_free( pnTmp );
	routerSetDefaultFon( psProfile, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psRouterPortComboBox ) ) );
	routerSetLineAccessCode( psProfile, gtk_entry_get_text( GTK_ENTRY( psRouterLineAccessCodeEntry ) ) );
	routerSetInternationalPrefix( psProfile, gtk_entry_get_text( GTK_ENTRY( psRouterInternationalPrefixEntry ) ) );
	routerSetCountryCode( psProfile, gtk_entry_get_text( GTK_ENTRY( psRouterCountryCodeEntry ) ) );
	routerSetNationalPrefix( psProfile, gtk_entry_get_text( GTK_ENTRY( psRouterNationalPrefixEntry ) ) );
	routerSetAreaCode( psProfile, gtk_entry_get_text( GTK_ENTRY( psRouterAreaCodeEntry ) ) );

	/* Call monitor - Tab */
	callMonitorSetType( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorTypeButton ) ) );
	callMonitorSetDateTime( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorDateTimeButton ) ) );
	callMonitorSetName( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorNameButton ) ) );
	callMonitorSetCompany( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorCompanyButton ) ) );
	callMonitorSetCity( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorCityButton ) ) );
	callMonitorSetNumber( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorNumberButton ) ) );
	callMonitorSetLocalName( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorLocalNameButton ) ) );
	callMonitorSetLocalNumber( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorLocalNumberButton ) ) );
	callMonitorSetDuration( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorDurationButton ) ) );

	/* Voice Box - Tab */
	voiceBoxSetUser( psProfile, gtk_entry_get_text( GTK_ENTRY( psVoiceBoxUserEntry ) ) );
	voiceBoxSetPassword( psProfile, gtk_entry_get_text( GTK_ENTRY( psVoiceBoxPasswordEntry ) ) );

	/* Softphone - Tab */
	softphoneSetMsn( psProfile, gtk_entry_get_text( GTK_ENTRY( psSoftphoneMsnEntry ) ) );
	softphoneSetController( psProfile, gtk_combo_box_get_active( GTK_COMBO_BOX( psSoftphoneControllerComboBox ) ) );
	softphoneSetHeadset( psProfile, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psBluetoothComboBox ) ) );
#if HAVE_FAXOPHONE &&  HAVE_DBUS
	bluetoothSetHeadset( softphoneGetHeadset( psProfile ) );
#endif

	/* Notification - Tab */
	callMonitorSetSystemNotification( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorSystemRadio ) ) );
	callMonitorSetFfgtkNotification( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorFfgtkRadio ) ) );
	callMonitorSetPopupPosition( psProfile, gtk_combo_box_get_active( GTK_COMBO_BOX( psCallMonitorPopupPositionComboBox ) ) );
	callMonitorSetDisplayTime( psProfile, gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( psCallMonitorTimeSpinButton ) ) );
	callMonitorSetReverseLookupIn( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorReverseInCheckButton ) ) );
	callMonitorSetReverseLookupOut( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorReverseOutCheckButton ) ) );
	callMonitorSetPlaySounds( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorPlaySoundsCheckButton ) ) );
	callMonitorSetLoadOnStartup( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorLoadListOnStartupCheckButton ) ) );
	callMonitorSetIncomingSound( psProfile, gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( psCallMonitorIncomingSoundChooser ) ) );
	callMonitorSetOutgoingSound( psProfile, gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( psCallMonitorOutgoingSoundChooser ) ) );

	/* Fax - Tab */
	faxSetSender( psProfile, gtk_entry_get_text( GTK_ENTRY( psFaxSenderEntry ) ) );
	faxSetId( psProfile, gtk_entry_get_text( GTK_ENTRY( psFaxIdEntry ) ) );
	faxSetMsn( psProfile, gtk_entry_get_text( GTK_ENTRY( psFaxMsnEntry ) ) );
	faxSetBitrate( psProfile, gtk_combo_box_get_active( GTK_COMBO_BOX( psFaxBitrateComboBox ) ) );
	faxSetController( psProfile, gtk_combo_box_get_active( GTK_COMBO_BOX( psFaxControllerComboBox ) ) );
	faxSetAccept( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psFaxAcceptIncomingCheckButton ) ) );
	faxSetEcm( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psFaxEcmToggle ) ) );
	faxSetReportToggle( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psFaxReportToggle ) ) );
	pnTmp = g_strdup_printf( "%s/", gtk_file_chooser_get_current_folder( GTK_FILE_CHOOSER( psFaxReportDir ) ) );
	Debug( KERN_DEBUG, "Want to set report dir to: '%s'\n", pnTmp );
	faxSetReportDir( psProfile, pnTmp );
	g_free( pnTmp );
	Debug( KERN_DEBUG, "fax report dir set to: '%s'\n", faxGetReportDir( psProfile ) );

	/* Actions - Tab */

	/* Plugins - Tab */
	pluginsSetLookup( psProfile, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsReverseComboBox ) ) );
	setDefaultPlugin( psProfile, PLUGIN_TYPE_LOOKUP, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsReverseComboBox ) ) );
	pluginsSetAddress( psProfile, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsAddressComboBox ) ) );
	setDefaultPlugin( psProfile, PLUGIN_TYPE_BOOK, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsAddressComboBox ) ) );
	transferPassword( psProfile, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsPasswordComboBox ) ) );
	pluginsSetPassword( psProfile, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsPasswordComboBox ) ) );
	setDefaultPlugin( psProfile, PLUGIN_TYPE_PASSWORD, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsPasswordComboBox ) ) );
	pluginsSetAudio( psProfile, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsAudioComboBox ) ) );
	setDefaultPlugin( psProfile, PLUGIN_TYPE_AUDIO, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsAudioComboBox ) ) );

	/* Misc - Tab */
	miscSetToolbarStyle( psProfile, gtk_combo_box_get_active( GTK_COMBO_BOX( psMiscToolbarStyleComboBox ) ) );
	addressBookSetNumberFormat( psProfile, gtk_combo_box_get_active( GTK_COMBO_BOX( psMiscNumberFormatComboBox ) ) );
	callMonitorSetReplaceOnLoad( psProfile, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCallMonitorReplaceOnLoad ) ) );

	/* Debug - Tab */
	debugSetLevel( psProfile, gtk_combo_box_get_active( GTK_COMBO_BOX( psDebugFfgtkComboBox ) ) );
	//debugSetFaxLevel( psProfile, gtk_combo_box_get_active( GTK_COMBO_BOX( psDebugCapiFaxComboBox ) ) );

	if ( ( faxGetController( psProfile ) != nOldController ) || ( debugGetFaxLevel( psProfile ) != nOldDebug ) ) {
		psFax = getDefaultFaxPlugin( psProfile );
		if ( psFax != NULL ) {
			psFax -> Close();
			if ( psFax -> Init( faxGetController( psProfile ), debugGetFaxLevel( psProfile ) ) ) {
				setDefaultPlugin( psProfile, PLUGIN_TYPE_FAX, NULL );
			}
		}
	}
}

/**
 * \brief Address book entry changed - callback
 * \param psWidget button widget
 * \param pUserData Book preferences button
 */
static void addressBookChanged( GtkWidget *psWidget, gpointer pUserData ) {
	struct sPlugin *psPlugin = NULL;
	struct sAddressBook *psBook = NULL;

	psPlugin = findPlugin( PLUGIN_TYPE_BOOK, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsAddressComboBox ) ) );
	if ( psPlugin != NULL ) {
		psBook = psPlugin -> pData;

		gtk_widget_set_sensitive( pUserData, psBook -> Preferences != NULL );
	} else {
		gtk_widget_set_sensitive( pUserData, FALSE );
	}
}

/**
 * \brief Call address book preferences
 * \param psWidget button widget
 * \param pUserData UNUSED
 */
static void addressBookPreferences( GtkWidget *psWidget, gpointer pUserData ) {
	struct sPlugin *psPlugin = NULL;
	struct sAddressBook *psBook = NULL;

	psPlugin = findPlugin( PLUGIN_TYPE_BOOK, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsAddressComboBox ) ) );
	if ( psPlugin != NULL ) {
		psBook = psPlugin -> pData;

		psBook -> Preferences();
	}
}

/**
 * \brief Audio entry changed - callback
 * \param psWidget button widget
 * \param pUserData Book preferences button
 */
static void audioChanged( GtkWidget *psWidget, gpointer pUserData ) {
	struct sPlugin *psPlugin = NULL;
	struct sAudio *psAudio = NULL;

	psPlugin = findPlugin( PLUGIN_TYPE_AUDIO, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsAudioComboBox ) ) );
	if ( psPlugin != NULL ) {
		psAudio = psPlugin -> pData;

		gtk_widget_set_sensitive( pUserData, psAudio -> Preferences != NULL );
	} else {
		gtk_widget_set_sensitive( pUserData, FALSE );
	}
}

/**
 * \brief Call audio preferences
 * \param psWidget button widget
 * \param pUserData UNUSED
 */
static void audioPreferences( GtkWidget *psWidget, gpointer pUserData ) {
	struct sPlugin *psPlugin = NULL;
	struct sAudio *psAudio = NULL;

	psPlugin = findPlugin( PLUGIN_TYPE_AUDIO, gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPluginsAudioComboBox ) ) );
	if ( psPlugin != NULL ) {
		psAudio = psPlugin -> pData;

		psAudio -> Preferences();
	}
}

/**
 * \brief Load profile information
 * \param psProfile profile structure
 */
void loadProfile( struct sProfile *psProfile ) {
	GList *psList = NULL;
	GtkTreeModel *psModel = NULL;
	GtkListStore *psListStore = NULL;
	GtkTreeIter sIter;
	struct sPlugin *psPlugin = NULL;
	struct sLookup *psLookup = NULL;
	int nCount = 0;
	gchar *pnTmp;
	int nIndex;
	const gchar *pnNumber;
	gchar *pnType = NULL;

	/* Router - Tab */
	gtk_entry_set_text( GTK_ENTRY( psRouterHostEntry ), routerGetHost( psProfile ) );
	pnTmp = ( char * ) routerGetPassword( psProfile );
	if ( pnTmp != NULL ) {
		gtk_entry_set_text( GTK_ENTRY( psRouterPasswordEntry ), pnTmp );
	}
	gtk_entry_set_text( GTK_ENTRY( psRouterLineAccessCodeEntry ), routerGetLineAccessCode( psProfile ) );
	gtk_entry_set_text( GTK_ENTRY( psRouterInternationalPrefixEntry ), routerGetInternationalPrefix( psProfile ) );
	gtk_entry_set_text( GTK_ENTRY( psRouterCountryCodeEntry ), routerGetCountryCode( psProfile ) );
	gtk_entry_set_text( GTK_ENTRY( psRouterNationalPrefixEntry ), routerGetNationalPrefix( psProfile ) );
	gtk_entry_set_text( GTK_ENTRY( psRouterAreaCodeEntry ), routerGetAreaCode( psProfile ) );
	routerAddComboBoxPorts( psProfile, psRouterPortComboBox );
	pnTmp = g_strdup_printf( "%s (%s)", routerGetName( psProfile ), routerGetVersion( psProfile ) );
	gtk_label_set_text( GTK_LABEL( psRouterBoxLabel ), pnTmp );
	g_free( pnTmp );

	/* Call monitor - Tab */
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorTypeButton ), callMonitorGetType( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorDateTimeButton ), callMonitorGetDateTime( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorNameButton ), callMonitorGetName( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorCompanyButton ), callMonitorGetCompany( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorCityButton ), callMonitorGetCity( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorNumberButton ), callMonitorGetNumber( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorLocalNameButton ), callMonitorGetLocalName( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorLocalNumberButton ), callMonitorGetLocalNumber( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorDurationButton ), callMonitorGetDuration( psProfile ) );

	/* Voice Box - Tab */
	gtk_entry_set_text( GTK_ENTRY( psVoiceBoxUserEntry ), voiceBoxGetUser( psProfile ) );
	gtk_entry_set_text( GTK_ENTRY( psVoiceBoxPasswordEntry ), voiceBoxGetPassword( psProfile ) );

	/* Softphone - Tab */
	gtk_entry_set_text( GTK_ENTRY( psSoftphoneMsnEntry ), softphoneGetMsn( psProfile ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psSoftphoneControllerComboBox ), softphoneGetController( psProfile ) );
#if HAVE_FAXOPHONE && HAVE_DBUS
	bluetoothSetList( psBluetoothComboBox );
#else
	gtk_widget_set_sensitive( GTK_WIDGET( psBluetoothComboBox ), FALSE );
#endif


	/* Notification - Tab */
	gtk_combo_box_set_active( GTK_COMBO_BOX( psCallMonitorPopupPositionComboBox ), callMonitorGetPopupPosition( psProfile ) );
	//gtk_spin_button_set_value( GTK_SPIN_BUTTON( psCallMonitorTimeSpinButton ), callMonitorGetDisplayTime( psProfile ) );
	
	gtk_spin_button_set_value( GTK_SPIN_BUTTON( psCallMonitorTimeSpinButton ), 5 );
	gtk_widget_set_sensitive( GTK_WIDGET( psCallMonitorTimeSpinButton ), FALSE );

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorReverseInCheckButton ), callMonitorGetReverseLookupIn( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorReverseOutCheckButton ), callMonitorGetReverseLookupOut( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorPlaySoundsCheckButton ), callMonitorGetPlaySounds( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorLoadListOnStartupCheckButton ), callMonitorGetLoadOnStartup( psProfile ) );
/*#ifdef HAVE_LIBNOTIFY
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorFfgtkRadio ), callMonitorGetFfgtkNotification( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorSystemRadio ), callMonitorGetSystemNotification( psProfile ) );
#else*/
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorFfgtkRadio ), TRUE );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorSystemRadio ), FALSE );
	gtk_widget_set_sensitive( GTK_WIDGET( psCallMonitorFfgtkRadio ), FALSE );
	gtk_widget_set_sensitive( GTK_WIDGET( psCallMonitorSystemRadio ), FALSE );
//#endif
	g_signal_emit_by_name( G_OBJECT( psCallMonitorSystemRadio ), "toggled", G_TYPE_NONE );

	psModel = gtk_tree_view_get_model( GTK_TREE_VIEW( psCallMonitorMsnTreeView ) );
	psListStore = GTK_LIST_STORE( psModel );
	gtk_list_store_clear( GTK_LIST_STORE( psListStore ) );
	for ( nIndex = 0; nIndex < 29; nIndex++ ) {
		pnNumber = routerGetFonNumber( psProfile, nIndex );
		if ( pnNumber != NULL && strlen( pnNumber ) > 0 ) {
			if ( nIndex >= 0 && nIndex < 10 ) {
				pnType = g_strdup_printf( "MSN%d", nIndex );
			} else if ( nIndex == 10 ) {
				pnType = g_strdup( "POTS" );
			} else {
				pnType = g_strdup_printf( "SIP%d", nIndex - 11 );
			}
			gtk_list_store_append( psListStore, &sIter );
			gtk_list_store_set( psListStore, &sIter,
			                   0, pnType,
							   1, pnNumber,
							   2, ( routerGetFonFlags( psProfile, nIndex ) & 1 ),
							   3, ( routerGetFonFlags( psProfile, nIndex ) & 2 ),
							   4, ( routerGetFonFlags( psProfile, nIndex ) & 4 ), -1 );
			if ( pnType != NULL ) {
				g_free( pnType );
				pnType = NULL;
			}
		}
	}

	gtk_file_chooser_set_filename( GTK_FILE_CHOOSER( psCallMonitorIncomingSoundChooser ), callMonitorGetIncomingSound( psProfile ) );
	gtk_file_chooser_set_filename( GTK_FILE_CHOOSER( psCallMonitorOutgoingSoundChooser ), callMonitorGetOutgoingSound( psProfile ) );

	/* Fax - Tab */
	gtk_entry_set_text( GTK_ENTRY( psFaxSenderEntry ), faxGetSenderName( psProfile ) );
	gtk_entry_set_text( GTK_ENTRY( psFaxIdEntry ), faxGetSenderNumber( psProfile ) );
	gtk_entry_set_text( GTK_ENTRY( psFaxMsnEntry ), faxGetSenderMsn( psProfile ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psFaxBitrateComboBox ), faxGetBitrateIndex( psProfile ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psFaxControllerComboBox ), faxGetController( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psFaxAcceptIncomingCheckButton ), faxGetAccept( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psFaxEcmToggle ), faxGetEcm( psProfile ) );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psFaxReportToggle ), faxGetReportToggle( psProfile ) );
	Debug( KERN_DEBUG, "Set fax report folder to: '%s'\n", faxGetReportDir( psProfile ) );
	gtk_file_chooser_set_current_folder( GTK_FILE_CHOOSER( psFaxReportDir ), faxGetReportDir( psProfile ) );
	pnTmp = gtk_file_chooser_get_current_folder( GTK_FILE_CHOOSER( psFaxReportDir ) );
	Debug( KERN_DEBUG, "File-chooser got: '%s'\n", pnTmp );

	/* Actions - Tab */
	loadAction( psSelectedAction );

	/* Plugins - Tab */
	psModel = gtk_combo_box_get_model( GTK_COMBO_BOX( psPluginsReverseComboBox ) );
	psListStore = GTK_LIST_STORE( psModel );
	gtk_list_store_clear( GTK_LIST_STORE( psListStore ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psPluginsReverseComboBox ), _( "None" ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psPluginsReverseComboBox ), nCount );

	for ( psList = getLookupList( routerGetCountryCode( psProfile ) ); psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psLookup = psList -> data;
		gtk_combo_box_append_text( GTK_COMBO_BOX( psPluginsReverseComboBox ), psLookup -> pnService );
		nCount++;
		if ( strcmp( psLookup -> pnService, pluginsGetLookup( psProfile ) ) == 0 ) {
			gtk_combo_box_set_active( GTK_COMBO_BOX( psPluginsReverseComboBox ), nCount );
		}
	}

	nCount = 0;
	psModel = gtk_combo_box_get_model( GTK_COMBO_BOX( psPluginsAddressComboBox ) );
	psListStore = GTK_LIST_STORE( psModel );
	gtk_list_store_clear( GTK_LIST_STORE( psListStore ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psPluginsAddressComboBox ), _( "None" ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psPluginsAddressComboBox ), 0 );
	for ( psList = psPluginList; psList != NULL; psList = psList -> next ) {
		psPlugin = psList -> data;
		if ( psPlugin && psPlugin -> nType == PLUGIN_TYPE_BOOK ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psPluginsAddressComboBox ), psPlugin -> pnName );
			nCount++;
			if ( psPlugin -> pData == getDefaultBookPlugin( psProfile ) ) {
				gtk_combo_box_set_active( GTK_COMBO_BOX( psPluginsAddressComboBox ), nCount );
			}
		}
	}

	nCount = 0;
	psModel = gtk_combo_box_get_model( GTK_COMBO_BOX( psPluginsPasswordComboBox ) );
	psListStore = GTK_LIST_STORE( psModel );
	gtk_list_store_clear( GTK_LIST_STORE( psListStore ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psPluginsPasswordComboBox ), _( "None" ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psPluginsPasswordComboBox ), 0 );
	for ( psList = psPluginList; psList != NULL; psList = psList -> next ) {
		psPlugin = psList -> data;
		if ( psPlugin && psPlugin -> nType == PLUGIN_TYPE_PASSWORD ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psPluginsPasswordComboBox ), psPlugin -> pnName );
			nCount++;
			if ( psPlugin -> pData == getDefaultPasswordPlugin( psProfile ) ) {
				gtk_combo_box_set_active( GTK_COMBO_BOX( psPluginsPasswordComboBox ), nCount );
			}
		}
	}

	nCount = 0;
	psModel = gtk_combo_box_get_model( GTK_COMBO_BOX( psPluginsAudioComboBox ) );
	psListStore = GTK_LIST_STORE( psModel );
	gtk_list_store_clear( GTK_LIST_STORE( psListStore ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psPluginsAudioComboBox ), _( "None" ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psPluginsAudioComboBox ), 0 );
	for ( psList = psPluginList; psList != NULL; psList = psList -> next ) {
		psPlugin = psList -> data;
		if ( psPlugin && psPlugin -> nType == PLUGIN_TYPE_AUDIO ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( psPluginsAudioComboBox ), psPlugin -> pnName );
			nCount++;
			if ( psPlugin -> pData == getDefaultAudioPlugin( psProfile ) ) {
				gtk_combo_box_set_active( GTK_COMBO_BOX( psPluginsAudioComboBox ), nCount );
			}
		}
	}

	/* Misc */
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCallMonitorReplaceOnLoad ), callMonitorGetReplaceOnLoad( psProfile ) );

	/* Debug - Tab */
	gtk_combo_box_set_active( GTK_COMBO_BOX( psDebugFfgtkComboBox ), debugGetLevel( psProfile ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psDebugFaxComboBox ), debugGetFaxLevel( psProfile ) );
}

/**
 * \brief Update action treeview and fields
 */
void updateActions( void ) {
	GList *psList = getActions();
	struct sAction *psAction;
	GtkTreeIter sIter;
	gboolean bSet = FALSE;

	if ( psActionsStore == NULL ) {
		return;
	}

	gtk_list_store_clear( psActionsStore );

	while ( psList != NULL ) {
		psAction = psList -> data;
		if ( psAction != NULL ) {
			gtk_list_store_append( psActionsStore, &sIter );
			gtk_list_store_set( psActionsStore, &sIter, 0, g_locale_to_utf8( psAction -> pnName, -1, 0, 0, 0 ), -1 );
		}
		if ( bSet != TRUE ) {
			gtk_tree_selection_select_iter( gtk_tree_view_get_selection( GTK_TREE_VIEW( psActionsTreeView ) ), &sIter );
			loadAction( psAction );
			psSelectedAction = psAction;
			bSet = TRUE;
		}

		psList = psList -> next;
	}
}

/**
 * \brief Update dtmf treeview and fields
 */
void updateDtmfActions( void ) {
	GList *psList = getDtmfActions();
	struct sDtmf *psDtmf;
	GtkTreeIter sIter;

	if ( psDtmfActionsStore == NULL ) {
		return;
	}

	gtk_list_store_clear( psDtmfActionsStore );

	while ( psList != NULL ) {
		psDtmf = psList -> data;
		if ( psDtmf != NULL ) {
			gtk_list_store_append( psDtmfActionsStore, &sIter );
			gtk_list_store_set( psDtmfActionsStore, &sIter, 0, g_locale_to_utf8( psDtmf -> pnName, -1, 0, 0, 0 ), -1 );
		}

		psList = psList -> next;
	}
}

/**
 * \brief Save current action and set new selected action
 * \param psAction new action structure
 */
void setAction( struct sAction *psAction ) {
	if ( psSelectedAction != NULL ) {
		saveAction( psSelectedAction );
	}

	if ( psAction != NULL ) {
		loadAction( psAction );
		psSelectedAction = psAction;
	}
}

/**
 * \brief Handle preferences response
 * \param psWidget widget
 * \param nResponse response id
 * \param pData window pointer
 */
static void preferencesResponse( GtkWidget *psWidget, gint nResponse, gpointer pData ) {
	GList *psList = getProfiles();
	struct sProfile *psProfile;

	if ( nResponse == 1 ) {
		saveAction( psSelectedAction );
		saveProfile( getActiveProfile() );

		while ( psList != NULL ) {
			psProfile = psList -> data;
			if ( psProfile != NULL ) {
				SavePreferences( psProfile );
			}
			psList = psList -> next;
		}
		saveProfiles();
	}

	gtk_widget_destroy( GTK_WIDGET( pData ) );
	psPreferencesDialog = NULL;
	routerLock( getActiveProfile(), FALSE );
}

/**
 * \brief Action selected callback
 * \param psTreeView action treeview
 * \param psEvent button event structure
 * \param pUserData UNUSED
 * \return handled flag
 */
gboolean actionSelected( GtkTreeView *psTreeView, GdkEventButton *psEvent, gpointer pUserData ) {
	if ( ( psEvent -> type == GDK_BUTTON_PRESS ) && ( psEvent -> button == 1 ) ) {
		GtkTreeSelection *psSelection = NULL;
		GtkTreeModel *psModel = NULL;
		GtkTreeIter sIter;
		gchar *pnName = NULL;
		struct sAction *psAction = NULL;

		psSelection = gtk_tree_view_get_selection( GTK_TREE_VIEW( psTreeView ) );

		if ( gtk_tree_selection_count_selected_rows( psSelection ) <= 1 ) {
			GtkTreePath *psPath;

			/* Get tree path for row that was clicked */
			if ( gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW( psTreeView ), (gint) psEvent -> x, (gint) psEvent -> y, &psPath, NULL, NULL, NULL ) ) {
				gtk_tree_selection_unselect_all( psSelection );
				gtk_tree_selection_select_path( psSelection, psPath );

				psModel = gtk_tree_view_get_model( psTreeView );
				if ( gtk_tree_model_get_iter( psModel, &sIter, psPath ) ) {
					gtk_tree_model_get( psModel, &sIter, 0, &pnName, -1 );
					if ( pnName == NULL ) {
						gtk_tree_path_free( psPath );
						return FALSE;
					}
					psAction = findAction( pnName );
					setAction( psAction );
				}
				gtk_tree_path_free( psPath );
			} else {
				return FALSE;
			}
		} else {
			return FALSE;
		}
	}

	return FALSE;
}

/**
 * \brief Delete action
 * \param psButton Delete button widget
 * \param pUserData tree view widget
 */
void delAction( GtkButton *psButton, gpointer pUserData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( pUserData );

	if ( gtk_tree_selection_get_selected( psSelection, &psModel, &sSelectedIter ) ) {
		GValue sName = { 0 };

		gtk_tree_model_get_value( psModel, &sSelectedIter, 0, &sName );

		GtkWidget *psDialog = gtk_message_dialog_new( NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION,
			GTK_BUTTONS_OK_CANCEL, _( "Do you want to delete the action '%s'?" ), g_value_get_string( &sName ) );
		gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Delete action" ) );
		gint nResult = gtk_dialog_run( GTK_DIALOG( psDialog ) );
		gtk_widget_destroy( psDialog );

		if ( nResult != GTK_RESPONSE_OK ) {
			return;
		}

		removeAction( g_value_get_string( &sName ) );

		g_value_unset( &sName );

		saveActions();
		updateActions();
	}
}

/**
 * \brief Delete dtmf
 * \param psButton Delete button widget
 * \param pUserData tree view widget
 */
void dtmfActionsDel( GtkButton *psButton, gpointer pUserData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( pUserData );

	if ( gtk_tree_selection_get_selected( psSelection, &psModel, &sSelectedIter ) ) {
		GValue sName = { 0 };

		gtk_tree_model_get_value( psModel, &sSelectedIter, 0, &sName );

		GtkWidget *psDialog = gtk_message_dialog_new( NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION,
			GTK_BUTTONS_OK_CANCEL, _( "Do you want to delete the dtmf action '%s'?" ), g_value_get_string( &sName ) );
		gtk_window_set_title( GTK_WINDOW( psDialog ), _( "Delete dtmf action" ) );
		gint nResult = gtk_dialog_run( GTK_DIALOG( psDialog ) );
		gtk_widget_destroy( psDialog );

		if ( nResult != GTK_RESPONSE_OK ) {
			return;
		}

		removeDtmfAction( g_value_get_string( &sName ) );

		g_value_unset( &sName );

		saveDtmfActions();
		updateDtmfActions();
	}
}

/**
 * \brief Verify button callback
 * \param psButton verify button callback
 * \param pUserData UNUSED
 */
static void verifyCallback( GtkButton *psButton, gpointer pUserData ) {
	GtkWidget *psDialog;
	gint nError = -1;

	saveProfile( getActiveProfile() );

	routerLock( getActiveProfile(), FALSE );
	nError = routerLogin( getActiveProfile() );
	if ( nError != 0 ) {
		if ( nError == 33485 ) {
			ShowError( FALSE, _( "Login blocked" ), _( "ffgtk has blocked login until valid password is given" ) );
		} else if ( nError < 0 ) {
			RouterLoginError( FALSE, -nError );
		} else {
			ShowError( FALSE, _( "Network error" ), curl_easy_strerror( nError ) );
		}
		routerLock( getActiveProfile(), TRUE );
	} else {
		routerLogout( getActiveProfile() );
		psDialog = gtk_message_dialog_new( pUserData, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_INFO, GTK_BUTTONS_CLOSE, _( "Password for router is valid" ) );
		gtk_dialog_run( GTK_DIALOG( psDialog ) );
		gtk_widget_destroy( GTK_WIDGET( psDialog ) );
	}
}

static gboolean bFinishedPref = FALSE;

/**
 * \brief Thread for detecting router informations
 * \param pData UNUSED
 * \return NULL
 */
gpointer detectThread( gpointer pData ) {
	gint nError = -1;

	routerSetStandardPorts( getActiveProfile() );

	nError = routerGetPhoneSettings( getActiveProfile() );
	if ( nError != 0 ) {
		if ( nError == 33485 ) {
			ShowError( TRUE, _( "Login blocked" ), _( "ffgtk has blocked login until valid password is given" ) );
		} else if ( nError < 0 ) {
			RouterLoginError( TRUE, -nError );
		} else {
			ShowError( TRUE, _( "Network error" ), curl_easy_strerror( nError ) );
		}
		goto end;
	}

	//#warning LINE ACCESS?
	if ( getActiveProfile() -> sRouterInfo.pnInternationalPrefix != NULL ) {
		routerSetInternationalPrefix( getActiveProfile(), getActiveProfile() -> sRouterInfo.pnInternationalPrefix );
	}
	if ( getActiveProfile() -> sRouterInfo.pnCountryCode != NULL ) {
		routerSetCountryCode( getActiveProfile(), getActiveProfile() -> sRouterInfo.pnCountryCode );
	}
	if ( getActiveProfile() -> sRouterInfo.pnNationalPrefix != NULL ) {
		routerSetNationalPrefix( getActiveProfile(), getActiveProfile() -> sRouterInfo.pnNationalPrefix );
	}
	if ( getActiveProfile() -> sRouterInfo.pnAreaCode != NULL ) {
		routerSetAreaCode( getActiveProfile(), getActiveProfile() -> sRouterInfo.pnAreaCode );
	}

	nError = routerFirmwareDetect( getActiveProfile() );
	if ( nError != 0 ) {
		if ( nError == 33485 ) {
			ShowError( TRUE, _( "Login blocked" ), _( "ffgtk has blocked login until valid password is given" ) );
		} else if ( nError < 0 ) {
			RouterLoginError( TRUE, -nError );
		} else {
			ShowError( TRUE, _( "Network error" ), curl_easy_strerror( nError ) );
		}
		goto end;
	}

	nError = routerGetActivePorts( getActiveProfile() );
	if ( nError != 0 ) {
		if ( nError == 33485 ) {
			ShowError( TRUE, _( "Login blocked" ), _( "ffgtk has blocked login until valid password is given" ) );
		} else if ( nError < 0 ) {
			RouterLoginError( TRUE, -nError );
		} else {
			ShowError( TRUE, _( "Network error" ), curl_easy_strerror( nError ) );
		}
		goto end;
	}

end:
	bFinishedPref = TRUE;

	return NULL;
}

/**
 * \brief Detect firmware callback
 * \param psButton button widget
 * \param pUserData UNUSED
 */
static void detectCallback( GtkButton *psButton, gpointer pUserData ) {
	GdkCursor *psCursor = NULL;

	bFinishedPref = FALSE;

	saveProfile( getActiveProfile() );

	psCursor = gdk_cursor_new_for_display( gtk_widget_get_display( psPreferencesDialog ), GDK_WATCH );
	gdk_window_set_cursor( gtk_widget_get_window( psPreferencesDialog ), psCursor );
	if ( psCursor != NULL ) {
#if GTK_MAJOR_VERSION < 3
		gdk_cursor_unref( psCursor );
#else
		g_object_unref( psCursor );
#endif
	}

	GTK_FLUSH;

	CREATE_THREAD( "detect", detectThread, NULL );

	while ( bFinishedPref == FALSE ) GTK_FLUSH;

	loadProfile( getActiveProfile() );

	gdk_window_set_cursor( gtk_widget_get_window( GTK_WIDGET( psPreferencesDialog ) ), NULL );
}

/**
 * \brief Action dial flags toggled
 * \param psToggle cell renderer toggle
 * \param pnPathStr path string
 * \param puserData number of toggle
 */
static void dialToggled( GtkCellRendererToggle *psToggle, gchar *pnPathStr, gpointer pUserData ) {
	GtkTreeModel *psModel = gtk_tree_view_get_model( GTK_TREE_VIEW( psCallMonitorMsnTreeView ) );
	GtkTreeIter sIter;
	GtkTreePath *psPath = gtk_tree_path_new_from_string( pnPathStr );
	gchar *pnNumber;
	gboolean bDial;
	int nNum = GPOINTER_TO_INT( pUserData );
	int nFlags;
	int nType = 10;

	gtk_tree_model_get_iter( psModel, &sIter, psPath );
	gtk_tree_model_get( psModel, &sIter, 1, &pnNumber, nNum, &bDial, -1 );

	nType = routerGetFonType( getActiveProfile(), pnNumber );
	if ( nType != -1 ) {
		nFlags = routerGetFonFlags( getActiveProfile(), nType );
		if ( bDial == TRUE ) {
			nFlags &= ~( 1 << ( nNum - 2 ) );
		} else {
			nFlags |= ( 1 << ( nNum - 2 ) );
		}
		routerSetFonFlags( getActiveProfile(), nType, nFlags ); 

		bDial ^= 1;
		gtk_list_store_set( GTK_LIST_STORE( psModel ), &sIter, nNum, bDial, -1 );
	}

	gtk_tree_path_free( psPath );
}

static gint nActionMsnFlags = 0;

/**
 * \brief Action selected toggld
 * \param psToggle cell renderer toggle
 * \param pnPathStr path string
 * \param pUserData number of toggle
 */
static void selectedToggled( GtkCellRendererToggle *psToggle, gchar *pnPathStr, gpointer pUserData ) {
	GtkTreeModel *psModel = gtk_tree_view_get_model( GTK_TREE_VIEW( pUserData ) );
	GtkTreeIter sIter;
	GtkTreePath *psPath = gtk_tree_path_new_from_string( pnPathStr );
	gchar *pnNumber;
	gboolean bDial;
	gint nType = 0;

	gtk_tree_model_get_iter( psModel, &sIter, psPath );
	gtk_tree_model_get( psModel, &sIter, 0, &bDial, 2, &pnNumber, -1 );

	nType = routerGetFonType( getActiveProfile(), pnNumber );
	if ( nType != -1 ) {
		nType = ( 1 << ( nType + ACTION_TYPE_MAX ) );
		if ( bDial == TRUE ) {
			nActionMsnFlags &= ~nType;
		} else {
			nActionMsnFlags |= nType;
		}
		Debug( KERN_DEBUG, "%d %x\n", nType, nActionMsnFlags );

		bDial ^= 1;

		gtk_list_store_set( GTK_LIST_STORE( psModel ), &sIter, 0, bDial, -1 );
	}

	gtk_tree_path_free( psPath );
}

/**
 * \brief Action response callback
 * \param psWidget action widget
 * \param nResult response id result
 * \param puserData UNUSED
 */
static void actionsResponse( GtkWidget *psWidget, gint nResult, gpointer pUserData ) {
	GtkWidget *psNameEntry = g_object_get_data( G_OBJECT( psWidget ), "name_entry" );
	GtkWidget *psDescriptionEntry = g_object_get_data( G_OBJECT( psWidget ), "description_entry" );
	GtkWidget *psExecEntry = g_object_get_data( G_OBJECT( psWidget ), "exec_entry" );
	struct sAction *psAction = NULL;
	const gchar *pnActionName = gtk_entry_get_text( GTK_ENTRY( psNameEntry ) );
	const gchar *pnActionDescription = gtk_entry_get_text( GTK_ENTRY( psDescriptionEntry ) );
	const gchar *pnActionExec = gtk_entry_get_text( GTK_ENTRY( psExecEntry ) );

	if ( nResult == 1 && pnActionName != NULL && strlen( pnActionName ) > 0 ) {
		if ( pUserData != NULL ) {
			psAction = pUserData;

			g_free( psAction -> pnName );
			psAction -> pnName = g_strdup( pnActionName );
			g_free( psAction -> pnDescription );
			psAction -> pnDescription = NULL;
			g_free( psAction -> pnExec );
			psAction -> pnExec = NULL;
		} else {
			psAction = createAction( pnActionName );
		}

		if ( psAction != NULL ) {
			if ( pnActionDescription != NULL ) {
				psAction -> pnDescription = g_strdup( pnActionDescription );
			}
			if ( pnActionExec != NULL ) {
				psAction -> pnExec = g_strdup( pnActionExec );
			}
			setActionFlags( getActiveProfile(), psAction -> pnName, nActionMsnFlags );

			saveActions();
			updateActions();
		}
	}

	gtk_widget_destroy( GTK_WIDGET( psWidget ) );
}

/**
 * \brief Dtmf action response callback
 * \param psWidget action widget
 * \param nResult response id result
 * \param puserData UNUSED
 */
static void dtmfActionsResponse( GtkWidget *psWidget, gint nResult, gpointer pUserData ) {
	GtkWidget *psNameEntry = g_object_get_data( G_OBJECT( psWidget ), "name_entry" );
	GtkWidget *psCallerEntry = g_object_get_data( G_OBJECT( psWidget ), "caller_entry" );
	GtkWidget *psCalledEntry = g_object_get_data( G_OBJECT( psWidget ), "called_entry" );
	GtkWidget *psCodeEntry = g_object_get_data( G_OBJECT( psWidget ), "code_entry" );
	GtkWidget *psExecEntry = g_object_get_data( G_OBJECT( psWidget ), "exec_entry" );
	struct sDtmf *psDtmf = NULL;
	const gchar *pnDtmfActionName = gtk_entry_get_text( GTK_ENTRY( psNameEntry ) );
	const gchar *pnDtmfActionCaller = gtk_entry_get_text( GTK_ENTRY( psCallerEntry ) );
	const gchar *pnDtmfActionCalled = gtk_entry_get_text( GTK_ENTRY( psCalledEntry ) );
	const gchar *pnDtmfActionCode = gtk_entry_get_text( GTK_ENTRY( psCodeEntry ) );
	const gchar *pnDtmfActionExec = gtk_entry_get_text( GTK_ENTRY( psExecEntry ) );

	if ( nResult == 1 && pnDtmfActionName != NULL && strlen( pnDtmfActionName ) > 0 ) {
		if ( pUserData != NULL ) {
			psDtmf = pUserData;

			g_free( psDtmf -> pnName );
			psDtmf -> pnName = g_strdup( pnDtmfActionName );
			g_free( psDtmf -> pnCaller );
			psDtmf -> pnCaller = g_strdup( pnDtmfActionCaller );
			g_free( psDtmf -> pnCalled );
			psDtmf -> pnCalled = g_strdup( pnDtmfActionCalled );
			g_free( psDtmf -> pnCode );
			psDtmf -> pnCode = g_strdup( pnDtmfActionCode );
			g_free( psDtmf -> pnExecute );
			psDtmf -> pnExecute = NULL;
		} else {
			psDtmf = createDtmfAction( pnDtmfActionName );
		}

		if ( psDtmf != NULL ) {
			if ( pnDtmfActionCaller != NULL ) {
				psDtmf -> pnCaller = g_strdup( pnDtmfActionCaller );
			}
			if ( pnDtmfActionCalled != NULL ) {
				psDtmf -> pnCalled = g_strdup( pnDtmfActionCalled );
			}
			if ( pnDtmfActionCode != NULL ) {
				psDtmf -> pnCode = g_strdup( pnDtmfActionCode );
			}
			if ( pnDtmfActionExec != NULL ) {
				psDtmf -> pnExecute = g_strdup( pnDtmfActionExec );
			}

			saveDtmfActions();
			updateDtmfActions();
		}
	}

	gtk_widget_destroy( GTK_WIDGET( psWidget ) );
}

/**
 * \brief Search for programm callback
 * \param psButton button widget
 * \param pUserData UNUSED
 */
static void searchWindow( GtkWidget *psButton, gpointer pUserData ) {
	GtkWidget *psDialog = pUserData;
	GtkWidget *psExecEntry = g_object_get_data( G_OBJECT( psDialog ), "exec_entry" );
	GtkWidget *psChooser = gtk_file_chooser_dialog_new( _( "Select programm" ), GTK_WINDOW( psDialog ), GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL );
	gint nResult = 0;

	nResult = gtk_dialog_run( GTK_DIALOG( psChooser ) );
	if ( nResult == GTK_RESPONSE_ACCEPT ) {
		gchar *pnFileName;

		pnFileName = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER( psChooser ) );
		gtk_entry_set_text( GTK_ENTRY( psExecEntry ), pnFileName );

		g_free( pnFileName );
	}

	gtk_widget_destroy( GTK_WIDGET( psChooser ) );
}

/**
 * \brief Add actions callback
 * \param Button widget
 * \param pUserData UNUSED
 */
static void actionsAdd( GtkWidget *psButton, gpointer pUserData ) {
	GtkTreeModel *psModel = NULL;
	GtkListStore *psStore = NULL;
	GtkBuilder *psBuilder = NULL;
	GtkWidget *psDialog = NULL;
	GtkWidget *psNameEntry = NULL;
	GtkWidget *psDescriptionEntry = NULL;
	GtkWidget *psExecEntry = NULL;
	GtkWidget *psSearchButton = NULL;
	struct sAction *psAction = NULL;
	GError *psError = NULL;
	gchar *pnUiFile = NULL;
	gint nIndex = 0;
	const gchar *pnNumber = NULL;
	gchar *pnType = NULL;
	GtkTreeIter sIter;
	GtkWidget *psActionsMsnTreeView = NULL;
	GtkCellRenderer *psRenderer = NULL;
	GtkTreeViewColumn *psCol = NULL;

	nActionMsnFlags = 0;

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "actions.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
		Debug( KERN_WARNING, "Error: %s\n", psError -> message );
		g_error_free( psError );
		g_free( pnUiFile );
		return;
	}
	g_free( pnUiFile );

	gtk_builder_connect_signals( psBuilder, NULL );

	psDialog = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsDialog" ) );
	if ( psDialog == NULL ) {
		return;
	}

	psNameEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNameEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "name_entry", psNameEntry );
	psDescriptionEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDescriptionEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "description_entry", psDescriptionEntry );
	psExecEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psExecEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "exec_entry", psExecEntry );
	psSearchButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psSearchButton" ) );
	g_signal_connect( G_OBJECT( psSearchButton ), "clicked", G_CALLBACK( searchWindow ), psDialog );

	psActionsMsnTreeView = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psMsnTreeView" ) );
	psStore = gtk_list_store_new( 3, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING );
	psModel = GTK_TREE_MODEL( psStore );
	gtk_tree_view_set_model( GTK_TREE_VIEW( psActionsMsnTreeView ), psModel );

	psRenderer = gtk_cell_renderer_toggle_new();
	psCol = gtk_tree_view_column_new_with_attributes( _( "Selected" ), psRenderer, "active", 0, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psActionsMsnTreeView ), psCol );
	g_signal_connect( G_OBJECT( psRenderer ), "toggled", G_CALLBACK( selectedToggled ), psActionsMsnTreeView );

	psRenderer = gtk_cell_renderer_text_new();
	psCol = gtk_tree_view_column_new_with_attributes( _( "Type" ), psRenderer, "text", 1, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psActionsMsnTreeView ), psCol );

	psRenderer = gtk_cell_renderer_text_new();
	psCol = gtk_tree_view_column_new_with_attributes( _( "Number" ), psRenderer, "text", 2, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psActionsMsnTreeView ), psCol );

	if ( pUserData != NULL ) {
		psAction = pUserData;

		if ( psAction -> pnName != NULL ) {
			gtk_entry_set_text( GTK_ENTRY( psNameEntry ), psAction -> pnName );
		}
		if ( psAction -> pnDescription != NULL ) {
			gtk_entry_set_text( GTK_ENTRY( psDescriptionEntry ), psAction -> pnDescription );
		}
		if ( psAction -> pnExec != NULL ) {
			gtk_entry_set_text( GTK_ENTRY( psExecEntry ), psAction -> pnExec );
		}
		nActionMsnFlags = getActionFlags( getActiveProfile(), psAction -> pnName );
	}

	gtk_list_store_clear( GTK_LIST_STORE( psStore ) );
	for ( nIndex = 0; nIndex < 29; nIndex++ ) {
		pnNumber = routerGetFonNumber( getActiveProfile(), nIndex );
		if ( pnNumber != NULL && strlen( pnNumber ) > 0 ) {
			if ( nIndex >= 0 && nIndex < 10 ) {
				pnType = g_strdup_printf( "MSN%d", nIndex );
			} else if ( nIndex == 10 ) {
				pnType = g_strdup( "POTS" );
			} else {
				pnType = g_strdup_printf( "SIP%d", nIndex - 11 );
			}
			gtk_list_store_append( psStore, &sIter );
			gtk_list_store_set( psStore, &sIter, 0, nActionMsnFlags & ( 1 << ( nIndex + ACTION_TYPE_MAX ) ), 1, pnType, 2, pnNumber, -1 );
			if ( pnType != NULL ) {
				g_free( pnType );
				pnType = NULL;
			}
		}
	}

	g_signal_connect( G_OBJECT( psDialog ), "response", G_CALLBACK( actionsResponse ), pUserData );

	gtk_widget_show_all( psDialog );
}

/**
 * \brief Actions edit callback
 * \param psButton button pressed
 * \param pUserData action treeview
 */
static void actionsEdit( GtkWidget *psButton, gpointer pUserData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( pUserData );
	struct sAction *psAction = NULL;

	if ( gtk_tree_selection_get_selected( psSelection, &psModel, &sSelectedIter ) ) {
		GValue sName = { 0 };

		gtk_tree_model_get_value( psModel, &sSelectedIter, 0, &sName );

		psAction = findAction( g_value_get_string( &sName ) );

		g_value_unset( &sName );

		if ( psAction != NULL ) {
			actionsAdd( NULL, psAction );
		}
	}
}

/**
 * \brief Add dtmf callback
 * \param Button widget
 * \param pUserData UNUSED
 */
static void dtmfActionsAdd( GtkWidget *psButton, gpointer pUserData ) {
	GtkBuilder *psBuilder = NULL;
	GtkWidget *psDialog = NULL;
	GtkWidget *psNameEntry = NULL;
	GtkWidget *psCallerEntry = NULL;
	GtkWidget *psCalledEntry = NULL;
	GtkWidget *psCodeEntry = NULL;
	GtkWidget *psExecEntry = NULL;
	struct sDtmf *psDtmf = NULL;
	GError *psError = NULL;
	gchar *pnUiFile = NULL;

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "dtmf.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
		Debug( KERN_WARNING, "Error: %s\n", psError -> message );
		g_error_free( psError );
		g_free( pnUiFile );
		return;
	}
	g_free( pnUiFile );

	gtk_builder_connect_signals( psBuilder, NULL );

	psDialog = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDtmfDialog" ) );
	if ( psDialog == NULL ) {
		return;
	}

	psNameEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNameEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "name_entry", psNameEntry );
	psCallerEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallerEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "caller_entry", psCallerEntry );
	psCalledEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCalledEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "called_entry", psCalledEntry );
	psCodeEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCodeEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "code_entry", psCodeEntry );
	psExecEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psExecEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "exec_entry", psExecEntry );

	if ( pUserData != NULL ) {
		psDtmf = pUserData;

		if ( psDtmf -> pnName != NULL ) {
			gtk_entry_set_text( GTK_ENTRY( psNameEntry ), psDtmf -> pnName );
		}
		if ( psDtmf -> pnCaller != NULL ) {
			gtk_entry_set_text( GTK_ENTRY( psCallerEntry ), psDtmf -> pnCaller );
		}
		if ( psDtmf -> pnCalled != NULL ) {
			gtk_entry_set_text( GTK_ENTRY( psCalledEntry ), psDtmf -> pnCalled );
		}
		if ( psDtmf -> pnCode != NULL ) {
			gtk_entry_set_text( GTK_ENTRY( psCodeEntry ), psDtmf -> pnCode );
		}
		if ( psDtmf -> pnExecute != NULL ) {
			gtk_entry_set_text( GTK_ENTRY( psExecEntry ), psDtmf -> pnExecute );
		}
	}

	g_signal_connect( G_OBJECT( psDialog ), "response", G_CALLBACK( dtmfActionsResponse ), pUserData );

	gtk_widget_show_all( psDialog );
}
/**
 * \brief Dtmf edit callback
 * \param psButton button pressed
 * \param pUserData action treeview
 */
static void dtmfActionsEdit( GtkWidget *psButton, gpointer pUserData ) {
	GtkTreeIter sSelectedIter;
	GtkTreeModel *psModel;
	GtkTreeSelection *psSelection = gtk_tree_view_get_selection( pUserData );
	struct sDtmf *psDtmf = NULL;

	if ( gtk_tree_selection_get_selected( psSelection, &psModel, &sSelectedIter ) ) {
		GValue sName = { 0 };

		gtk_tree_model_get_value( psModel, &sSelectedIter, 0, &sName );

		psDtmf = findDtmfAction( g_value_get_string( &sName ) );

		g_value_unset( &sName );

		if ( psDtmf != NULL ) {
			dtmfActionsAdd( NULL, psDtmf );
		}
	}
}

 /**
 * \brief Open log file
 * \param psWidget file chooser
 * \param pUserData toggle button
 */
void openLogCallback( GtkWidget *psWidget, gpointer pUserData ) {
	gchar *pnExec = NULL;

	/* create execution command line  */
	pnExec = g_strdup_printf( "%s/ffgtk.log", getUserDir() );

	/* execute the standard ui-independent way in opening the default browser */
	osExecute( pnExec );

	/* free command line */
	g_free( pnExec );
}

/**
 * \brief Fax sender entry insert text - callback
 * We only want normal ascii chars for sender name
 * \param psEdit editable entry widget
 * \param pnNewText new text to be inserted
 * \param nNewText length of pnNewText
 * \param pnPosition current position pointer
 * \param pUserData user data UNUSED
 */
static void faxSenderEntryInsertText( GtkEditable *psEdit, gchar *pnNewText, gint nNewTextLength, gint *pnPosition, gpointer pUserData ) {
	gint nIndex = 0;

	if ( pnNewText == NULL ) {
		return;
	}

	for ( nIndex = 0; nIndex < nNewTextLength; nIndex++ ) {
		if ( !isascii( pnNewText[ nIndex ] ) ) {
			g_signal_stop_emission_by_name( GTK_WIDGET( psEdit ), "insert-text" );
		}
	}
}

void debugFfgtkComboBoxChanged( GtkWidget *psBox, gpointer pUserData ) {
	Debug( KERN_DEBUG, "Called and changed to %d\n", gtk_combo_box_get_active( GTK_COMBO_BOX( psBox ) ) );
	debugSetLevel( getActiveProfile(), gtk_combo_box_get_active( GTK_COMBO_BOX( psBox ) ) );
}

void debugCapiFaxComboBoxChanged( GtkWidget *psBox, gpointer pUserData ) {
	Debug( KERN_DEBUG, "Called and changed to %d\n", gtk_combo_box_get_active( GTK_COMBO_BOX( psBox ) ) );
	debugSetFaxLevel( getActiveProfile(), gtk_combo_box_get_active( GTK_COMBO_BOX( psBox ) ) );
}

/**
 * \brief Open preferences dialog
 * \return error code
 */
int preferences( void ) {
	GtkBuilder *psBuilder;
	GtkWidget *psSoftphoneTable = NULL;
	GtkWidget *psFaxTable = NULL;
	GtkListStore *psStore;
	GError *psError = NULL;
	gchar *pnUiFile;
	GtkFileFilter *psFilter;
	GtkCellRenderer *psRenderer = NULL;
	GtkTreeViewColumn *psCol = NULL;

	if ( psPreferencesDialog != NULL ) {
		gtk_window_present( GTK_WINDOW( psPreferencesDialog ) );
		return 0;
	}

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "preferences.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
		Debug( KERN_WARNING, "Error: %s\n", psError -> message );
		g_error_free( psError );
		g_free( pnUiFile );
		return 1;
	}
	g_free( pnUiFile );

	psPreferencesDialog = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPreferencesDialog" ) );
	if ( psPreferencesDialog == NULL ) {
		return 2;
	}
	g_signal_connect( G_OBJECT( psPreferencesDialog ), "response", G_CALLBACK( preferencesResponse ), psPreferencesDialog );

	psNotebook = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNotebook" ) );

	/* Router - Tab */
	psRouterHostEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterHostEntry" ) );
	psRouterPasswordEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterPasswordEntry" ) );
	psRouterLineAccessCodeEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterLineAccessCodeEntry" ) );
	psRouterInternationalPrefixEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterInternationalPrefixEntry" ) );
	psRouterCountryCodeEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterCountryCodeEntry" ) );
	psRouterNationalPrefixEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterNationalPrefixEntry" ) );
	psRouterAreaCodeEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterAreaCodeEntry" ) );
	psRouterPortComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterPortComboBox" ) );
	psRouterBoxLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterBoxLabel" ) );
	psRouterVerifyButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterVerifyButton" ) );
	g_signal_connect( G_OBJECT( psRouterVerifyButton ), "clicked", G_CALLBACK( verifyCallback ), psPreferencesDialog );
	psRouterDetectButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRouterDetectButton" ) );
	g_signal_connect( G_OBJECT( psRouterDetectButton ), "clicked", G_CALLBACK( detectCallback ), NULL );

	/* Call-Monitor - Tab */
	psCallMonitorBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorBox" ) );
	psCallMonitorTypeButton = gtk_check_button_new_with_label( _( "Type" ) );
	gtk_box_pack_start(GTK_BOX( psCallMonitorBox ), psCallMonitorTypeButton, FALSE, TRUE, 2 );
	psCallMonitorDateTimeButton = gtk_check_button_new_with_label( _( "Date/Time" ) );
	gtk_box_pack_start( GTK_BOX( psCallMonitorBox ), psCallMonitorDateTimeButton, FALSE, TRUE, 2 );
	psCallMonitorNameButton = gtk_check_button_new_with_label( _( "Name" ) );
	gtk_box_pack_start( GTK_BOX( psCallMonitorBox ), psCallMonitorNameButton, FALSE, TRUE, 2 );
	psCallMonitorCompanyButton = gtk_check_button_new_with_label( _( "Company" ) );
	gtk_box_pack_start( GTK_BOX( psCallMonitorBox ), psCallMonitorCompanyButton, FALSE, TRUE, 2 );
	psCallMonitorCityButton = gtk_check_button_new_with_label( _( "City" ) );
	gtk_box_pack_start( GTK_BOX( psCallMonitorBox ), psCallMonitorCityButton, FALSE, TRUE, 2 );
	psCallMonitorNumberButton = gtk_check_button_new_with_label( _( "Number" ) );
	gtk_box_pack_start( GTK_BOX( psCallMonitorBox ), psCallMonitorNumberButton, FALSE, TRUE, 2 );
	psCallMonitorLocalNameButton = gtk_check_button_new_with_label( _( "Local Name" ) );
	gtk_box_pack_start( GTK_BOX( psCallMonitorBox ), psCallMonitorLocalNameButton, FALSE, TRUE, 2 );
	psCallMonitorLocalNumberButton = gtk_check_button_new_with_label( _( "Local Number" ) );
	gtk_box_pack_start( GTK_BOX( psCallMonitorBox ), psCallMonitorLocalNumberButton, FALSE, TRUE, 2 );
	psCallMonitorDurationButton = gtk_check_button_new_with_label( _( "Duration" ) );
	gtk_box_pack_start( GTK_BOX( psCallMonitorBox ), psCallMonitorDurationButton, FALSE, TRUE, 2 );

	/* Voice Box - Tab */
	psVoiceBoxUserEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psVoiceBoxUserEntry" ) );
	psVoiceBoxPasswordEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psVoiceBoxPasswordEntry" ) );

	/* Notification - Tab */
	psCallMonitorPopupPositionComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorPopupPositionComboBox" ) );
	psStore = GTK_LIST_STORE( gtk_combo_box_get_model( GTK_COMBO_BOX( psCallMonitorPopupPositionComboBox ) ) );
	gtk_list_store_clear( psStore );

	psCallMonitorSystemRadio = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psSystemRadio" ) );
	psCallMonitorFfgtkRadio = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFfgtkRadio" ) );

#ifndef HAVE_LIBNOTIFY
	gtk_widget_set_sensitive( GTK_WIDGET( psCallMonitorSystemRadio ), FALSE );
#endif

	gtk_combo_box_append_text( GTK_COMBO_BOX( psCallMonitorPopupPositionComboBox ), _( "Top left" ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psCallMonitorPopupPositionComboBox ), _( "Top right" ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psCallMonitorPopupPositionComboBox ), _( "Bottom right" ) );
	gtk_combo_box_append_text( GTK_COMBO_BOX( psCallMonitorPopupPositionComboBox ), _( "Bottom left" ) );

	psCallMonitorTimeSpinButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorTimeSpinButton" ) );
	psCallMonitorReverseInCheckButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorReverseInCheckButton" ) );
	psCallMonitorReverseOutCheckButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorReverseOutCheckButton" ) );
	psCallMonitorPlaySoundsCheckButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorPlaySoundsCheckButton" ) );
	psCallMonitorLoadListOnStartupCheckButton =	GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorLoadListOnStartupCheckButton" ) );

	psCallMonitorMsnTreeView = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorMsnTreeView" ) );
	psStore = gtk_list_store_new( 5, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN );
	psListStore = GTK_TREE_MODEL( psStore );
	gtk_tree_view_set_model( GTK_TREE_VIEW( psCallMonitorMsnTreeView ), psListStore );

	psRenderer = gtk_cell_renderer_text_new();
	psCol = gtk_tree_view_column_new_with_attributes( _( "Type" ), psRenderer, "text", 0, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorMsnTreeView ), psCol );

	psRenderer = gtk_cell_renderer_text_new();
	psCol = gtk_tree_view_column_new_with_attributes( _( "Number" ), psRenderer, "text", 1, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorMsnTreeView ), psCol );

	psRenderer = gtk_cell_renderer_toggle_new();
	psCol = gtk_tree_view_column_new_with_attributes( _( "Dial" ), psRenderer, "active", 2, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorMsnTreeView ), psCol );
	g_signal_connect( G_OBJECT( psRenderer ), "toggled", G_CALLBACK( dialToggled ), GINT_TO_POINTER( 2 ) );

	psRenderer = gtk_cell_renderer_toggle_new();
	psCol = gtk_tree_view_column_new_with_attributes( _( "Incoming" ), psRenderer, "active", 3, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorMsnTreeView ), psCol );
	g_signal_connect( G_OBJECT( psRenderer ), "toggled", G_CALLBACK( dialToggled ), GINT_TO_POINTER( 3 ) );

	psRenderer = gtk_cell_renderer_toggle_new();
	psCol = gtk_tree_view_column_new_with_attributes( _( "Outgoing" ), psRenderer, "active", 4, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psCallMonitorMsnTreeView ), psCol );
	g_signal_connect( G_OBJECT( psRenderer ), "toggled", G_CALLBACK( dialToggled ), GINT_TO_POINTER( 4 ) );

	psCallMonitorIncomingSoundChooser = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorIncomingSoundChooser" ) );
	psFilter = gtk_file_filter_new();
	gtk_file_filter_add_mime_type( psFilter, "audio/wav" );
	gtk_file_filter_add_mime_type( psFilter, "audio/mp3" );
	gtk_file_filter_add_mime_type( psFilter, "audio/ogg" );

	gtk_file_chooser_set_filter( GTK_FILE_CHOOSER( psCallMonitorIncomingSoundChooser ), psFilter );
	psCallMonitorOutgoingSoundChooser = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorOutgoingSoundChooser" ) );
	gtk_file_chooser_set_filter( GTK_FILE_CHOOSER( psCallMonitorOutgoingSoundChooser ), psFilter );

	/* Fax - Tab */
	psFaxTable = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxTable" ) );
	psFaxSenderEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxSenderEntry" ) );
	g_signal_connect( G_OBJECT( psFaxSenderEntry ), "insert-text", G_CALLBACK( faxSenderEntryInsertText ), NULL );
	psFaxIdEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxIdEntry" ) );
	psFaxMsnEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxMsnEntry" ) );
	psFaxBitrateComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxBitrateComboBox" ) );
	psFaxControllerComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxControllerComboBox" ) );
	psStore = GTK_LIST_STORE( gtk_combo_box_get_model( GTK_COMBO_BOX( psFaxControllerComboBox ) ) );
	gtk_list_store_clear( psStore );

	if ( routerGetPorts( getActiveProfile() ) & PORT_ISDNALL ) {
		gtk_combo_box_append_text( GTK_COMBO_BOX( psFaxControllerComboBox ), _( "ISDN-Controller 1" ) );
		gtk_combo_box_append_text( GTK_COMBO_BOX( psFaxControllerComboBox ), _( "ISDN-Controller 2" ) );
		gtk_combo_box_append_text( GTK_COMBO_BOX( psFaxControllerComboBox ), _( "Outside line ISDN" ) );
		gtk_combo_box_append_text( GTK_COMBO_BOX( psFaxControllerComboBox ), _( "Outside line Analog" ) );
		gtk_combo_box_append_text( GTK_COMBO_BOX( psFaxControllerComboBox ), _( "Internet" ) );
	} else {
		gtk_combo_box_append_text( GTK_COMBO_BOX( psFaxControllerComboBox ), _( "ISDN-Controller 1" ) );
		gtk_combo_box_append_text( GTK_COMBO_BOX( psFaxControllerComboBox ), _( "ISDN-Controller 2" ) );
		gtk_combo_box_append_text( GTK_COMBO_BOX( psFaxControllerComboBox ), _( "Outside line Analog" ) );
		gtk_combo_box_append_text( GTK_COMBO_BOX( psFaxControllerComboBox ), _( "Outside line ISDN" ) );
		gtk_combo_box_append_text( GTK_COMBO_BOX( psFaxControllerComboBox ), _( "Internet" ) );
	}

	psBluetoothComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psBluetoothComboBox" ) );

	psStore = GTK_LIST_STORE( gtk_combo_box_get_model( GTK_COMBO_BOX( psBluetoothComboBox ) ) );
	gtk_list_store_clear( psStore );

	psFaxAcceptIncomingCheckButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxAcceptIncomingCheckButton" ) );

	psFaxEcmToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxEcmToggle" ) );
	psFaxReportToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxReportToggle" ) );
	psFaxReportDir = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psFaxReportDir" ) );

	/* Softphone - Tab */
	psSoftphoneTable = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psSoftphoneTable" ) );
	psSoftphoneMsnEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psSoftphoneMsnEntry" ) );
	psSoftphoneControllerComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psSoftphoneControllerComboBox" ) );
	psDtmfActionsTreeView = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDtmfTreeView" ) );
	psDtmfActionsStore = gtk_list_store_new( 1, G_TYPE_STRING );
	psListStore = GTK_TREE_MODEL( psDtmfActionsStore );
	gtk_tree_view_set_model( GTK_TREE_VIEW( psDtmfActionsTreeView ), psListStore );

	psRenderer = gtk_cell_renderer_text_new();
	psCol = gtk_tree_view_column_new_with_attributes( _( "Name" ), psRenderer, "text", 0, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psDtmfActionsTreeView ), psCol );

	psDtmfActionAddButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDtmfAddButton" ) );
	g_signal_connect( G_OBJECT( psDtmfActionAddButton ), "clicked", G_CALLBACK( dtmfActionsAdd ), NULL );
	psDtmfActionDelButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDtmfDelButton" ) );
	g_signal_connect( G_OBJECT( psDtmfActionDelButton ), "clicked", G_CALLBACK( dtmfActionsDel ), psDtmfActionsTreeView );
	psDtmfActionEditButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDtmfEditButton" ) );
	g_signal_connect( G_OBJECT( psDtmfActionEditButton ), "clicked", G_CALLBACK( dtmfActionsEdit ), psDtmfActionsTreeView );

	/* Actions - Tab */
	psActionsTreeView = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsTreeView" ) );
	psActionsStore = gtk_list_store_new( 1, G_TYPE_STRING );
	psListStore = GTK_TREE_MODEL( psActionsStore );
	gtk_tree_view_set_model( GTK_TREE_VIEW( psActionsTreeView ), psListStore );
	g_signal_connect( psActionsTreeView, "button-press-event", (GCallback) actionSelected, NULL );

	psRenderer = gtk_cell_renderer_text_new();
	psCol = gtk_tree_view_column_new_with_attributes( _( "Name" ), psRenderer, "text", 0, NULL );
	gtk_tree_view_append_column( GTK_TREE_VIEW( psActionsTreeView ), psCol );

	psActionsAddButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsAddButton" ) );
	g_signal_connect( G_OBJECT( psActionsAddButton ), "clicked", G_CALLBACK( actionsAdd ), NULL );
	psActionsDelButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsDelButton" ) );
	g_signal_connect( G_OBJECT( psActionsDelButton ), "clicked", G_CALLBACK( delAction ), psActionsTreeView );
	psActionsEditButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsEditButton" ) );
	g_signal_connect( G_OBJECT( psActionsEditButton ), "clicked", G_CALLBACK( actionsEdit ), psActionsTreeView );

	psActionsInStartToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsInStartToggle" ) );
	psActionsInStopToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsInStopToggle" ) );
	psActionsOutStartToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsOutStartToggle" ) );
	psActionsOutStopToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsOutStopToggle" ) );
	psActionsInRingToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsInRingToggle" ) );
	psActionsInMissedToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsInMissedToggle" ) );
	psActionsOutDialToggle = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsOutDialToggle" ) );

	psActionsDescriptionLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psActionsDescriptionLabel" ) );

	/* Plugins - Tab */
	psPluginsReverseComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPluginsReverseComboBox" ) );
	psPluginsAddressComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPluginsAddressComboBox" ) );
	psPluginsAddressBookEdit = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPluginsAddressEdit" ) );
	g_signal_connect( G_OBJECT( psPluginsAddressBookEdit ), "clicked", G_CALLBACK( addressBookPreferences ), NULL );
	g_signal_connect( G_OBJECT( psPluginsAddressComboBox ), "changed", G_CALLBACK( addressBookChanged ), psPluginsAddressBookEdit );
	gtk_widget_set_sensitive( GTK_WIDGET( psPluginsAddressBookEdit ), TRUE );
	psPluginsPasswordComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPluginsPasswordComboBox" ) );
	psPluginsAudioComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPluginsAudioComboBox" ) );
	psPluginsAudioEdit = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPluginsAudioEdit" ) );
	g_signal_connect( G_OBJECT( psPluginsAudioEdit ), "clicked", G_CALLBACK( audioPreferences ), NULL );
	g_signal_connect( G_OBJECT( psPluginsAudioComboBox ), "changed", G_CALLBACK( audioChanged ), psPluginsAudioEdit );

	/* Misc - Tab */
	psMiscToolbarStyleComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psMiscToolbarStyleComboBox" ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psMiscToolbarStyleComboBox ), miscGetToolbarStyle( getActiveProfile() ) );
	psCallMonitorReplaceOnLoad = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psCallMonitorReplaceOnLoad" ) );
	psMiscNumberFormatComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psMiscNumberFormatComboBox" ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( psMiscNumberFormatComboBox ), addressBookGetNumberFormat( getActiveProfile() ) );

	/* Debug - Tab */
	psDebugFfgtkComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDebugFfgtkComboBox" ) );
	psDebugFaxComboBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDebugCapiFaxComboBox" ) );
	//psDebugOpenLogButton =  GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDebugOpenLogButton" ) );

	loadProfile( getActiveProfile() );
	updateActions();

	updateDtmfActions();

	gtk_builder_connect_signals( psBuilder, NULL );

#ifndef HAVE_FAXOPHONE
	gtk_widget_set_sensitive( GTK_WIDGET( psSoftphoneTable ), FALSE );
	gtk_widget_set_sensitive( GTK_WIDGET( psFaxTable ), FALSE );
#else
	gtk_widget_set_sensitive( GTK_WIDGET( psSoftphoneTable ), TRUE );
	gtk_widget_set_sensitive( GTK_WIDGET( psFaxTable ), TRUE );
#endif

	gtk_widget_show_all( psPreferencesDialog );

	return 0;
}
