/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file ssdp.c
 * \brief Simple Service Discovery Protocol (SSDP) functions
 * based upon the work of
 * - Nicolas Justin+Gregrory Kokanosky
 * - Frank Zschockelt
 */

#include <ffgtk.h>
#ifndef G_OS_WIN32
#include <netdb.h>
#include <sys/socket.h>
#else
#include <winsock2.h>
#endif

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>

#include "ssdp.h"

/**
 * \brief Parse the URL found at the "LOCATION:" entry of
 * the SSDP reply in 'buffer'. 'hostname' and 'port'
 * allocated, the caller is responsible to free them.
 * 'hostname' is guaranteed to be null terminated.
 * \param pnBuffer buffer pointer
 * \param ppnHostName where to store hostname to
 * \param pnPort where to store port number to
 * \return 0 on success, -1 otherwise.
 */
static int getHostFromSsdp( char *pnBuffer, char **ppnHostName, unsigned short *pnPort ) {
	char *pnPos1, *pnPos2;
	char pnTmp;

	if ( ( pnPos1 = strstr( pnBuffer, "LOCATION:" ) ) == NULL ) {
		return -1;
	}

	/* find first / */
	if ( ( pnPos1 = strchr( pnPos1, '/' ) ) == NULL ) {
		return -1;
	}

	pnPos1 += 2;

	/* search the ':' */
	if ( ( pnPos2 = strchr( pnPos1, ':' ) ) == NULL ) {
		return -1;
	}

	pnTmp = *pnPos2;
	*pnPos2 = '\0';

	/* copy the hostname */
	if ( ( *ppnHostName = strdup( pnPos1 ) ) == NULL ) {
		return -1;
	}
	*pnPos2 = pnTmp;

	/* copy the port number */
	pnPos1 = pnPos2 + 1;

	if ( ( pnPos2 = strchr( pnPos1, '\r' ) ) == NULL ) {
		return -1;
	}

	pnTmp = *pnPos2;
	*pnPos2 = '\0';
	*pnPort = atoi( pnPos1 );
	*pnPos2 = pnTmp;

	return 0;
}


/**
 * \brief Call this function if you want automatic discovery.
 * A SSDP:discover is sent to the multicast address 239.255.255.250:1900, if
 * a response is received, hostname and port are set.
 * \param ppnHost host name to store to
 * \param pnPort port name to store to
 * \return 0 if succeed, or -1 on error.
 */
int ssdpDiscover( char **ppnHost, unsigned short *pnPort ) {
	struct sockaddr_in sSockName;
	struct hostent *psHostName;
	int nSock;
	size_t nRet;
	char anData[] = 
		"M-SEARCH * HTTP/1.1\r\n"
		"Host: 239.255.255.250:1900\r\n"
		"Man: \"ssdp:discover\"\r\n"
		"ST: urn:rootdevice\r\n"
		"MX: 3\r\n"
		"\r\n";
	char anBuffer[ RESPONSE_BUFFER_LEN ];
	ssize_t nLen = RESPONSE_BUFFER_LEN;
	fd_set sFds;
	struct timeval sTimeOut;

	psHostName = gethostbyname( SSDP_MULTICAST );
	psHostName -> h_addrtype = AF_INET;

	if ( ( nSock = socket( PF_INET, SOCK_DGRAM, 0 ) ) == -1 ) {
		return -1;
	}

	memset( ( char * ) &sSockName, 0, sizeof( struct sockaddr_in ) );
	sSockName.sin_family = AF_INET;
	sSockName.sin_port = htons( SSDP_PORT );
	sSockName.sin_addr.s_addr = *( ( unsigned long * )( psHostName -> h_addr_list[ 0 ] ) );

	nRet = sendto( nSock, anData, strlen( anData ), 0, ( struct sockaddr * ) &sSockName, sizeof( struct sockaddr_in ) );
	if ( nRet != strlen( anData ) ) {
		return -1;
	}

	/* Get response */
	FD_ZERO( &sFds );
	FD_SET( nSock, &sFds );
	sTimeOut.tv_sec = 3;
	sTimeOut.tv_usec = 0;
	if ( select( nSock + 1, &sFds, NULL, NULL, &sTimeOut ) < 0 ) {
		return -1;
	}

	if ( FD_ISSET( nSock, &sFds ) ) {
		if ( ( nLen = recv( nSock, anBuffer, nLen, MSG_PEEK ) ) == -1 ) {
			return -1;
		}
		anBuffer[ nLen ] = '\0';
		close( nSock );
    
		/* Check the HTTP response code */
		if ( strncmp( anBuffer, "HTTP/1.1 200 OK", 12 ) != 0 ) {
			return -1;
		}
    
		if ( getHostFromSsdp( anBuffer, ppnHost, pnPort ) == -1 ) {
			return -1;
		}

		return 0;
	} else {
		return -1;
	}
}

