/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION: gnome-scan-plugin
 * @short_description: Scan operation plugin
 * @include: gnome-scan.h
 *
 * A #GnomeScanPlugin extends the processing pipeline. Only processing
 * plugin derive directly #GnomeScanPlugin. Source and Sink must
 * derive from #GnomeScanner and #GnomeScanSink. Basically, a plugin
 * handle common properties such as @name, @blurb and @params. Each
 * plugins as a per instance list of params. Those params are exposed
 * to the UI and save in #GnomeScanSettings.
 **/

#include "gnome-scan-plugin.h"
#include "gnome-scan-param-specs.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_TYPE_SCAN_PLUGIN, GnomeScanPluginPrivate))

typedef struct _GnomeScanPluginPrivate GnomeScanPluginPrivate;

struct _GnomeScanPluginPrivate
{
	gchar*			name;
	gchar*			blurb;
};

enum
{
	PROP_0,
	PROP_NAME,
	PROP_BLURB,
	PROP_PARAMS
};

enum
{
	PARAMS_CHANGED,
	LAST_SIGNAL
};

static GObjectClass* parent_class = NULL;
static guint signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE (GnomeScanPlugin, gnome_scan_plugin, G_TYPE_OBJECT);

static void
gnome_scan_plugin_init (GnomeScanPlugin *object)
{
	object->params = g_param_spec_pool_new (FALSE);
}

static void
gnome_scan_plugin_finalize (GObject *object)
{
	/* TODO: Add deinitalization code here */

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_scan_plugin_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_PLUGIN (object));

	switch (prop_id)
		{
		case PROP_NAME:
			GET_PRIVATE (object)->name = g_value_dup_string (value);
			break;
		case PROP_BLURB:
			GET_PRIVATE (object)->blurb = g_value_dup_string (value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
		}
}

static void
gnome_scan_plugin_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_PLUGIN (object));

	switch (prop_id)
		{
		case PROP_NAME:
			g_value_set_string (value, gnome_scan_plugin_get_name (GNOME_SCAN_PLUGIN (object)));
			break;
		case PROP_BLURB:
			g_value_set_string (value, GET_PRIVATE (object)->blurb);
			break;
		case PROP_PARAMS:
			g_value_set_pointer (value, GNOME_SCAN_PLUGIN (object)->params);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
		}
}

static void
gnome_scan_plugin_class_init (GnomeScanPluginClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));

	g_type_class_add_private (klass, sizeof (GnomeScanPluginPrivate));
	object_class->finalize = gnome_scan_plugin_finalize;
	object_class->set_property = gnome_scan_plugin_set_property;
	object_class->get_property = gnome_scan_plugin_get_property;

	/**
	 * GnomeScanPlugin:name:
	 *
	 * The plugin public name.
	 **/
	g_object_class_install_property (object_class,
					 PROP_NAME,
					 g_param_spec_string ("name",
							      "Name",
							      "Plugin public name",
							      NULL,
							      G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

	/**
	 * GnomeScanPlugin:blurb:
	 *
	 * The plugin public blurb.
	 **/
	g_object_class_install_property (object_class,
					 PROP_BLURB,
					 g_param_spec_string ("blurb",
							      "Blurb",
							      "Plugin public description.",
							      NULL,
							      G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

	/**
	 * GnomeScanPlugin:params:
	 *
	 * The list of per instance #GSParamSpec.
	 **/
	g_object_class_install_property (object_class,
					 PROP_PARAMS,
					 g_param_spec_pointer ("params",
							       "Parameters",
							       "Pool of plugin parameters specification.",
							       G_PARAM_READABLE));
	
	/**
	 * GnomeScanPlugin::params-changed:
	 * @plugin:	the emitting #GnomeScanPlugin
	 * @spec:	the spec of the changed parameter
	 *
	 * Triggered when one param of plugin has been added, changed,
	 * removed. Needs to be renamed to param-changed.
	 **/
	signals[PARAMS_CHANGED] =
		g_signal_new ("params-changed",
			      G_OBJECT_CLASS_TYPE (klass),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeScanPluginClass, changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__POINTER,
			      G_TYPE_NONE, 1, G_TYPE_POINTER);
}


/**
 * gnome_scan_plugin_get_name:
 * @plugin: a #GnomeScanPlugin
 * 
 * Returns: the @plugin name
 **/
gchar*
gnome_scan_plugin_get_name (GnomeScanPlugin *plugin)
{
	g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), NULL);
	return g_strdup (GET_PRIVATE (plugin)->name);
}

/**
 * gnome_scan_plugin_params_add:
 * @plugin: a #GnomeScanPlugin
 * @param: a #GSParamSpec
 * 
 * See: %gnome-scan-param-specs
 **/
void
gnome_scan_plugin_params_add (GnomeScanPlugin *plugin, GParamSpec *param)
{
	g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
	g_param_spec_pool_insert (plugin->params, param, G_OBJECT_TYPE (plugin));
	gnome_scan_plugin_params_changed (plugin, param);
}

/**
 * gnome_scan_plugin_params_remove:
 * @plugin: a #GnomeScanPlugin
 * @param: a #GSParamSpec
 * 
 * Remove a param spec from the param list.
 **/
void
gnome_scan_plugin_params_remove (GnomeScanPlugin *plugin, GParamSpec *param)
{
	g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
	g_param_spec_pool_remove (plugin->params, param);
	gnome_scan_plugin_params_changed (plugin, NULL);
}

/**
 * gnome_scan_plugin_params_clear:
 * @plugin: a #GnomeScanPlugin
 * 
 * Remove all instance parameters from a plugin.
 **/
void
gnome_scan_plugin_params_clear (GnomeScanPlugin *plugin)
{
	g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
	GList *node;
	
	node = g_param_spec_pool_list_owned (plugin->params, G_OBJECT_TYPE (plugin));

	for (; node; node = node->next)
		g_param_spec_pool_remove (plugin->params, node->data);
	
	gnome_scan_plugin_params_changed (plugin, NULL);
}

/**
 * gnome_scan_plugin_get_params:
 * @plugin: a #GnomeScanPlugin
 * 
 * Retrieve all instance paramters from @plugin.
 * 
 * Returns: a GList of #GSParamSpec.
 **/
GList*
gnome_scan_plugin_get_params (GnomeScanPlugin *plugin)
{
	g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), NULL);
	GList *params = NULL;
	
	params = g_param_spec_pool_list_owned (plugin->params,
					       G_OBJECT_TYPE (plugin));
	params = g_list_sort (params, (GCompareFunc) gs_param_spec_cmp_index);
	return params;
}

/**
 * gnome_scan_plugin_params_get_groups:
 * @plugin: a #GnomeScanPlugin
 *
 * The options must be ordered by groups in order this function not to
 * duplicate groups. When writing a #GnomeScanPlugin, ensure that
 * sorting option by index keep groups's options sibling.
 *
 * Returns: a #GSList of groups #GQuark.
 **/
GSList*
gnome_scan_plugin_params_get_groups (GnomeScanPlugin *plugin)
{
	GSList* groups = NULL;
	GList* node = gnome_scan_plugin_get_params(plugin);
	GQuark cur_group = 0, last_group = 0;
	
	for (; node ; node = node->next) {
		cur_group = gs_param_spec_get_group (node->data);
		if (!last_group || last_group != cur_group) {
			groups = g_slist_append (groups, (gpointer) cur_group);
			last_group = cur_group;
		}
	}
	
	return groups;
}

/**
 * gnome_scan_plugin_params_get_other_groups:
 * @plugin: a #GnomeScanPlugin
 * @known_group0: the first known group #GQuark
 * @...: A 0 terminated list of group #GQuark
 *
 * Return all group #GQuark except the one listed as parameter. This
 * is useful if you want e.g. all param group except preview params
 * group.
 *
 * Returns: a #GSList of #GQuark.
 **/
GSList*
gnome_scan_plugin_params_get_other_groups (GnomeScanPlugin *plugin,
					   GQuark known_group0, ...)
{
	va_list args;
	va_start (args, known_group0);
	GSList *groups = gnome_scan_plugin_params_get_groups (plugin);
	GQuark known_group;
	
	/* list known group not to list */
	known_group = known_group0;
	while (known_group) {
		groups = g_slist_remove_all (groups, (gpointer) known_group);
		known_group = va_arg (args, GQuark);
	}
	
	return groups;
}

/**
 * gnome_scan_plugin_get_param_group:
 * @plugin: a #GnomeScanPlugin
 * @group: a group #GQuark
 * 
 * Return all instance parameters which have @group as group.
 * 
 * Returns: a GList of #GSParamSpec.
 **/
GSList*
gnome_scan_plugin_get_param_group (GnomeScanPlugin *plugin, GQuark group)
{
	g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), NULL);
	GSList *list = NULL;
	gint i;
	guint n;
	GQuark pgroup;
	GParamSpec **specs =
		g_param_spec_pool_list (plugin->params,
					G_OBJECT_TYPE (plugin), &n);
	
	for (i = 0; i < n; i++) {
		pgroup = gs_param_spec_get_group (specs[i]);
		if (pgroup == group)
			list = g_slist_append (list, specs[i]);
	}

	list = g_slist_sort (list, (GCompareFunc) gs_param_spec_cmp_index);
	
	return list;
}

/**
 * gnome_scan_plugin_get_param_groups:
 * @plugin:	a #GnomeScanPlugin
 * @group0:	The first group #GQuark
 * @...:	A 0 terminated list of group @GQuark
 *
 * List all params which belong to groups.
 *
 * Returns: a #GSList of #GParamSpec
 **/
GSList*
gnome_scan_plugin_get_param_groups (GnomeScanPlugin *plugin,
				    GQuark group0, ...)
{
	GSList *groups = NULL;
	GQuark group;
	va_list args;
	va_start(args, group0);

	group = group0;
	while (group) {
		groups = g_slist_concat (groups,
					 gnome_scan_plugin_get_param_group (plugin, group));
		group = va_arg(args, GQuark);
	}
	
	return groups;
}

/**
 * gnome_scan_plugin_params_lookup:
 * @plugin:	a #GnomeScanPlugin
 * @name:	The name of the searched parameter
 *
 * Search the param spec pool for parameter @name.
 *
 * Returns:	a #GParamSpec or NULL.
 **/
GParamSpec*
gnome_scan_plugin_params_lookup	(GnomeScanPlugin *plugin,
				 const gchar *name)
{
	return g_param_spec_pool_lookup (plugin->params,
					 name,
					 G_OBJECT_TYPE (plugin),
					 FALSE);
}

/**
 * gnome_scan_plugin_params_foreach:
 * @plugin:	#GnomeScanPlugin
 * @func:	user callback
 * @user_data:	pointer to data passed as argument to @func.
 *
 * Execute @func for each @plugin's params.
 **/
void
gnome_scan_plugin_params_foreach (GnomeScanPlugin *plugin,
				  GFunc func,
				  gpointer user_data)
{
	GList *list = g_param_spec_pool_list_owned (plugin->params, G_OBJECT_TYPE (plugin));
	g_list_foreach (list, func, user_data);
}

#define	gs_exec_class_func_ret(k,f,d,a...)	if(k->f){return klass->f(a);}else{return d;}
#define	gs_exec_class_func(k,f,a...)		if(k->f){klass->f(a);}

/**
 * gnome_scan_plugin_configure:
 * @plugin: a #GnomeScanPlugin
 * @settings: a #GnomeScanPlugin
 *
 * Note:	Implemented by derived class, called by #GnomeScanJob.
 * 
 * Search in @settings value for its instance parameters.
 **/
void
gnome_scan_plugin_configure (GnomeScanPlugin *plugin, GnomeScanSettings *settings)
{
	g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin) && GNOME_IS_SCAN_SETTINGS (settings));
	GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
	gs_exec_class_func(klass, configure, plugin, settings);
}

/**
 * gnome_scan_plugin_get_child_nodes:
 * @plugin:	a #GnomeScanPlugin:
 * @root:	a root #GeglNode
 *
 * Note:	Implemented by derived class, called by #GnomeScanJob.
 *
 * Return the list of #GeglNode to append to the
 * pipeline. #GnomeScanJob link "input" and "output" pads of sibling
 * nodes, not auxiliary pad.
 *
 * Returns:	a #GList of #GeglNode
 **/
GList*
gnome_scan_plugin_get_child_nodes (GnomeScanPlugin *plugin,
				   GeglNode *root)
{
	g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), NULL);
	GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
	gs_exec_class_func_ret(klass, get_child_nodes, NULL, plugin, root);
}

/**
 * gnome_scan_plugin_configure_frame:
 * @plugin:	a #GnomeScanPlugin 
 *
 * Note:	Implemented by derived class, called by #GnomeScanJob.
 *
 * This is used especially for per frame GEGL pipeline configuration.
 **/
void
gnome_scan_plugin_configure_frame (GnomeScanPlugin *plugin)
{
	g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
	GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
	gs_exec_class_func(klass, configure_frame, plugin);
}

/**
 * gnome_scan_plugin_start_frame:
 * @plugin:	a #GnomeScanPlugin
 *
 * Note:	Implemented by derived class, called by #GnomeScanJob.
 *
 * Returns:	wether the frame is to be processed or aborted.
 **/
gboolean
gnome_scan_plugin_start_frame	(GnomeScanPlugin *plugin)
{
	g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), FALSE);
	GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
	gs_exec_class_func_ret(klass, start_frame, TRUE, plugin);
}

/**
 * gnome_scan_plugin_work:
 * @plugin:	a #GnomeScanPlugin
 * @progress:	a pointer to a place where to store progress
 *
 * Note:	Implemented by derived class, called by #GnomeScanJob.
 *
 * Do an iteration of plugin work. Store the amount of work done in
 * progress as a fraction, from 0 to 1. progress set to 1 means work
 * completed.
 *
 * Returns:	Wether more work iteration needs to be done.
 **/
gboolean
gnome_scan_plugin_work (GnomeScanPlugin *plugin,
			gdouble *progress)
{
	g_return_val_if_fail (GNOME_IS_SCAN_PLUGIN (plugin), FALSE);
	GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
	*progress = 1.;
	gs_exec_class_func_ret(klass, work, FALSE, plugin, progress);
}


/**
 * gnome_scan_plugin_end_frame:
 * @plugin:	a #GnomeScanPlugin
 *
 * Note:	Implemented by derived class, called by #GnomeScanJob.
 **/
void
gnome_scan_plugin_end_frame (GnomeScanPlugin *plugin)
{
	g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
	GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
	gs_exec_class_func(klass, end_frame, plugin);
}


/**
 * gnome_scan_plugin_end_scan:
 * @plugin:	a #GnomeScanPlugin
 *
 * Note:	Implemented by derived class, called by #GnomeScanJob.
 **/
void
gnome_scan_plugin_end_scan (GnomeScanPlugin *plugin)
{
	g_return_if_fail (GNOME_IS_SCAN_PLUGIN (plugin));
	GnomeScanPluginClass *klass = GNOME_SCAN_PLUGIN_GET_CLASS (plugin);
	gs_exec_class_func(klass, end_scan, plugin);
}

/**
 * gnome_scan_plugin_params_changed:
 * @plugin:	a #GnomeScanPlugin
 * @spec:	a #GParamSpec from @plugin
 *
 * Emit #GnomeScanPlugin::changed for @spec param. For use in derived
 * class.
 */
void
gnome_scan_plugin_params_changed	(GnomeScanPlugin *plugin,
					 GParamSpec *spec)
{
	g_signal_emit (plugin, signals[PARAMS_CHANGED], 0, spec);
}

