/* Contributed by Michael Werle <gnustep@michaelwerle.com>
 */
#import <Foundation/Foundation.h>
#import "Testing.h"

#define kTestCaseCount		(6)

typedef struct
{
  const char	*plist;
  size_t	plistLength;
  const char	*text;
  size_t	textLength;
} TestCase;


static const TestCase sTestCases[kTestCaseCount];

static BOOL RunTestCase(const TestCase *testCase, unsigned idx);


int main (int argc, const char * argv[])
{
  NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
  unsigned i;

  for (i = 0; i < kTestCaseCount; i++)
    {
      PASS((RunTestCase(&sTestCases[i], i + 1)),
	"xml propertylist parse test %d", i)
    }
    
  [pool release];
  return 0;
}


static BOOL
RunTestCase(const TestCase *testCase, unsigned idx)
{
  NSData	*data = nil;
  NSString	*errorDesc = nil;
  id		plist = nil;
  NSString	*plistValue = nil;
  NSString	*expectedValue = nil;
  
  NS_DURING
    {
      data = [NSData dataWithBytes: testCase->plist
			    length: testCase->plistLength];
      if (data == nil)
	{
	  NSLog(@"Test case %u FAILED: could not construct plist data.", idx);
	  return NO;
	}
      
      plist = [NSPropertyListSerialization propertyListFromData: data
		mutabilityOption: NSPropertyListImmutable
		format: NULL
		errorDescription: &errorDesc];
      
      if (plist == nil)
	{
	  NSLog(@"Test case %u FAILED: could not parse property list."
	    @" Error string: %@", idx, errorDesc);
	  return NO;
	}
      
      if (![plist isKindOfClass: [NSArray class]] || [plist count] != 1)
	{
	  NSLog(@"Test case %u FAILED: plist content is not array with "
	    @"one member.", idx);
	  return NO;
	}
      
      plistValue = [plist objectAtIndex: 0];
      if (![plistValue isKindOfClass: [NSString class]])
	{
	  NSLog(@"Test case %u FAILED: plist content is not an "
	    @"array with one string.", idx);
	  return NO;
	}
      
      data = [NSData dataWithBytes: testCase->text
			    length: testCase->textLength];
      if (data == nil)
	{
	  NSLog(@"Test case %u FAILED: could not construct string data.", idx);
	  return NO;
	}
      
      expectedValue = [[NSString alloc] initWithData: data
					    encoding: NSUTF8StringEncoding];
      [expectedValue autorelease];
      if (expectedValue == nil)
	{
	  NSLog(@"Test case %u FAILED: could not construct string from data "
	    @"(must be UTF-8).", idx);
	  return NO;
	}
      
      if ([plistValue compare: expectedValue options: 0] != NSOrderedSame)
	{
	  NSLog(@"Test case %u FAILED: read string is wrong (expected \"%@\", "
	    @"got \"%@\").", idx, expectedValue, plistValue);
	  return NO;
	}
    }  
  NS_HANDLER
    {
      NSLog(@"Test case %u FAILED: exception thrown - %@ : %@", idx,
	[localException name], [localException reason]);
      return NO;
    }
  NS_ENDHANDLER
  return YES;
}


/*	Test data follows. Data entries are UTF-8 strings, expressed as byte
	arrays to minimize risk of encoding problems.
*/

// Cyrillic string.
static const char sTestPlist1[] =
{
	0x3C, 0x3F, 0x78, 0x6D, 0x6C, 0x20, 0x76, 0x65, 0x72, 0x73,
	0x69, 0x6F, 0x6E, 0x3D, 0x22, 0x31, 0x2E, 0x30, 0x22, 0x20,
	0x65, 0x6E, 0x63, 0x6F, 0x64, 0x69, 0x6E, 0x67, 0x3D, 0x22,
	0x55, 0x54, 0x46, 0x2D, 0x38, 0x22, 0x3F, 0x3E, 0x0A, 0x3C,
	0x21, 0x44, 0x4F, 0x43, 0x54, 0x59, 0x50, 0x45, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x50, 0x55, 0x42, 0x4C, 0x49,
	0x43, 0x20, 0x22, 0x2D, 0x2F, 0x2F, 0x47, 0x4E, 0x55, 0x73,
	0x74, 0x65, 0x70, 0x2F, 0x2F, 0x44, 0x54, 0x44, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x30, 0x2E, 0x39, 0x2F, 0x2F,
	0x45, 0x4E, 0x22, 0x20, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3A,
	0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x67, 0x6E, 0x75, 0x73,
	0x74, 0x65, 0x70, 0x2E, 0x6F, 0x72, 0x67, 0x2F, 0x70, 0x6C,
	0x69, 0x73, 0x74, 0x2D, 0x30, 0x5F, 0x39, 0x2E, 0x78, 0x6D,
	0x6C, 0x22, 0x3E, 0x0A, 0x3C, 0x70, 0x6C, 0x69, 0x73, 0x74,
	0x20, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x22,
	0x30, 0x2E, 0x39, 0x22, 0x3E, 0x0A, 0x3C, 0x61, 0x72, 0x72,
	0x61, 0x79, 0x3E, 0x0A, 0x09, 0x3C, 0x73, 0x74, 0x72, 0x69,
	0x6E, 0x67, 0x3E, 0xD0, 0x93, 0xD0, 0xBE, 0xD0, 0xB2, 0xD0,
	0xBE, 0xD1, 0x80, 0xD0, 0xB8, 0xD1, 0x82, 0x20, 0xD1, 0x81,
	0xD1, 0x82, 0xD0, 0xB0, 0xD0, 0xBD, 0xD1, 0x86, 0xD0, 0xB8,
	0xD1, 0x8F, 0x20, 0x4C, 0x61, 0x65, 0x6E, 0x69, 0x6E, 0x2E,
	0x20, 0xD0, 0x9C, 0xD1, 0x8B, 0x20, 0xD0, 0xB7, 0xD0, 0xB0,
	0xD0, 0xB3, 0xD1, 0x80, 0xD1, 0x83, 0xD0, 0xB7, 0xD0, 0xB8,
	0xD0, 0xBB, 0xD0, 0xB8, 0x20, 0xD1, 0x81, 0xD1, 0x82, 0xD1,
	0x8B, 0xD0, 0xBA, 0xD0, 0xBE, 0xD0, 0xB2, 0xD0, 0xBE, 0xD1,
	0x87, 0xD0, 0xBD, 0xD1, 0x8B, 0xD0, 0xB5, 0x20, 0xD0, 0xB8,
	0xD0, 0xBD, 0xD1, 0x81, 0xD1, 0x82, 0xD1, 0x80, 0xD1, 0x83,
	0xD0, 0xBA, 0xD1, 0x86, 0xD0, 0xB8, 0xD0, 0xB8, 0x20, 0xD0,
	0xBD, 0xD0, 0xB0, 0x20, 0x3C, 0x2F, 0x73, 0x74, 0x72, 0x69,
	0x6E, 0x67, 0x3E, 0x0A, 0x3C, 0x2F, 0x61, 0x72, 0x72, 0x61,
	0x79, 0x3E, 0x0A, 0x3C, 0x2F, 0x70, 0x6C, 0x69, 0x73, 0x74,
	0x3E, 0x0A
};
static const char sTestText1[] =
{
	0xD0, 0x93, 0xD0, 0xBE, 0xD0, 0xB2, 0xD0, 0xBE, 0xD1, 0x80,
	0xD0, 0xB8, 0xD1, 0x82, 0x20, 0xD1, 0x81, 0xD1, 0x82, 0xD0,
	0xB0, 0xD0, 0xBD, 0xD1, 0x86, 0xD0, 0xB8, 0xD1, 0x8F, 0x20,
	0x4C, 0x61, 0x65, 0x6E, 0x69, 0x6E, 0x2E, 0x20, 0xD0, 0x9C,
	0xD1, 0x8B, 0x20, 0xD0, 0xB7, 0xD0, 0xB0, 0xD0, 0xB3, 0xD1,
	0x80, 0xD1, 0x83, 0xD0, 0xB7, 0xD0, 0xB8, 0xD0, 0xBB, 0xD0,
	0xB8, 0x20, 0xD1, 0x81, 0xD1, 0x82, 0xD1, 0x8B, 0xD0, 0xBA,
	0xD0, 0xBE, 0xD0, 0xB2, 0xD0, 0xBE, 0xD1, 0x87, 0xD0, 0xBD,
	0xD1, 0x8B, 0xD0, 0xB5, 0x20, 0xD0, 0xB8, 0xD0, 0xBD, 0xD1,
	0x81, 0xD1, 0x82, 0xD1, 0x80, 0xD1, 0x83, 0xD0, 0xBA, 0xD1,
	0x86, 0xD0, 0xB8, 0xD0, 0xB8, 0x20, 0xD0, 0xBD, 0xD0, 0xB0,
	0x20
};


// "A & B", using &amp;
static const char sTestPlist2[] =
{
	0x3C, 0x3F, 0x78, 0x6D, 0x6C, 0x20, 0x76, 0x65, 0x72, 0x73,
	0x69, 0x6F, 0x6E, 0x3D, 0x22, 0x31, 0x2E, 0x30, 0x22, 0x20,
	0x65, 0x6E, 0x63, 0x6F, 0x64, 0x69, 0x6E, 0x67, 0x3D, 0x22,
	0x55, 0x54, 0x46, 0x2D, 0x38, 0x22, 0x3F, 0x3E, 0x0A, 0x3C,
	0x21, 0x44, 0x4F, 0x43, 0x54, 0x59, 0x50, 0x45, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x50, 0x55, 0x42, 0x4C, 0x49,
	0x43, 0x20, 0x22, 0x2D, 0x2F, 0x2F, 0x47, 0x4E, 0x55, 0x73,
	0x74, 0x65, 0x70, 0x2F, 0x2F, 0x44, 0x54, 0x44, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x30, 0x2E, 0x39, 0x2F, 0x2F,
	0x45, 0x4E, 0x22, 0x20, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3A,
	0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x67, 0x6E, 0x75, 0x73,
	0x74, 0x65, 0x70, 0x2E, 0x6F, 0x72, 0x67, 0x2F, 0x70, 0x6C,
	0x69, 0x73, 0x74, 0x2D, 0x30, 0x5F, 0x39, 0x2E, 0x78, 0x6D,
	0x6C, 0x22, 0x3E, 0x0A, 0x3C, 0x70, 0x6C, 0x69, 0x73, 0x74,
	0x20, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x22,
	0x30, 0x2E, 0x39, 0x22, 0x3E, 0x0A, 0x3C, 0x61, 0x72, 0x72,
	0x61, 0x79, 0x3E, 0x0A, 0x09, 0x3C, 0x73, 0x74, 0x72, 0x69,
	0x6E, 0x67, 0x3E, 0x41, 0x20, 0x26, 0x61, 0x6D, 0x70, 0x3B,
	0x20, 0x42, 0x3C, 0x2F, 0x73, 0x74, 0x72, 0x69, 0x6E, 0x67,
	0x3E, 0x0A, 0x3C, 0x2F, 0x61, 0x72, 0x72, 0x61, 0x79, 0x3E,
	0x0A, 0x3C, 0x2F, 0x70, 0x6C, 0x69, 0x73, 0x74, 0x3E, 0x0A
};
static const char sTestText2[] =
{
	0x41, 0x20, 0x26, 0x20, 0x42
};


// Euro symbol (U+20AC), literal
static const char sTestPlist3[] =
{
	0x3C, 0x3F, 0x78, 0x6D, 0x6C, 0x20, 0x76, 0x65, 0x72, 0x73,
	0x69, 0x6F, 0x6E, 0x3D, 0x22, 0x31, 0x2E, 0x30, 0x22, 0x20,
	0x65, 0x6E, 0x63, 0x6F, 0x64, 0x69, 0x6E, 0x67, 0x3D, 0x22,
	0x55, 0x54, 0x46, 0x2D, 0x38, 0x22, 0x3F, 0x3E, 0x0A, 0x3C,
	0x21, 0x44, 0x4F, 0x43, 0x54, 0x59, 0x50, 0x45, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x50, 0x55, 0x42, 0x4C, 0x49,
	0x43, 0x20, 0x22, 0x2D, 0x2F, 0x2F, 0x47, 0x4E, 0x55, 0x73,
	0x74, 0x65, 0x70, 0x2F, 0x2F, 0x44, 0x54, 0x44, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x30, 0x2E, 0x39, 0x2F, 0x2F,
	0x45, 0x4E, 0x22, 0x20, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3A,
	0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x67, 0x6E, 0x75, 0x73,
	0x74, 0x65, 0x70, 0x2E, 0x6F, 0x72, 0x67, 0x2F, 0x70, 0x6C,
	0x69, 0x73, 0x74, 0x2D, 0x30, 0x5F, 0x39, 0x2E, 0x78, 0x6D,
	0x6C, 0x22, 0x3E, 0x0A, 0x3C, 0x70, 0x6C, 0x69, 0x73, 0x74,
	0x20, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x22,
	0x30, 0x2E, 0x39, 0x22, 0x3E, 0x0A, 0x3C, 0x61, 0x72, 0x72,
	0x61, 0x79, 0x3E, 0x0A, 0x09, 0x3C, 0x73, 0x74, 0x72, 0x69,
	0x6E, 0x67, 0x3E, 0xE2, 0x82, 0xAC, 0x3C, 0x2F, 0x73, 0x74,
	0x72, 0x69, 0x6E, 0x67, 0x3E, 0x0A, 0x3C, 0x2F, 0x61, 0x72,
	0x72, 0x61, 0x79, 0x3E, 0x0A, 0x3C, 0x2F, 0x70, 0x6C, 0x69,
	0x73, 0x74, 0x3E, 0x0A
};
static const char sTestText3[] =
{
	0xE2, 0x82, 0xAC
};


// Euro symbol (U+20AC), escaped as &#8364;
static const char sTestPlist4[] =
{
	0x3C, 0x3F, 0x78, 0x6D, 0x6C, 0x20, 0x76, 0x65, 0x72, 0x73,
	0x69, 0x6F, 0x6E, 0x3D, 0x22, 0x31, 0x2E, 0x30, 0x22, 0x20,
	0x65, 0x6E, 0x63, 0x6F, 0x64, 0x69, 0x6E, 0x67, 0x3D, 0x22,
	0x55, 0x54, 0x46, 0x2D, 0x38, 0x22, 0x3F, 0x3E, 0x0A, 0x3C,
	0x21, 0x44, 0x4F, 0x43, 0x54, 0x59, 0x50, 0x45, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x50, 0x55, 0x42, 0x4C, 0x49,
	0x43, 0x20, 0x22, 0x2D, 0x2F, 0x2F, 0x47, 0x4E, 0x55, 0x73,
	0x74, 0x65, 0x70, 0x2F, 0x2F, 0x44, 0x54, 0x44, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x30, 0x2E, 0x39, 0x2F, 0x2F,
	0x45, 0x4E, 0x22, 0x20, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3A,
	0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x67, 0x6E, 0x75, 0x73,
	0x74, 0x65, 0x70, 0x2E, 0x6F, 0x72, 0x67, 0x2F, 0x70, 0x6C,
	0x69, 0x73, 0x74, 0x2D, 0x30, 0x5F, 0x39, 0x2E, 0x78, 0x6D,
	0x6C, 0x22, 0x3E, 0x0A, 0x3C, 0x70, 0x6C, 0x69, 0x73, 0x74,
	0x20, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x22,
	0x30, 0x2E, 0x39, 0x22, 0x3E, 0x0A, 0x3C, 0x61, 0x72, 0x72,
	0x61, 0x79, 0x3E, 0x0A, 0x09, 0x3C, 0x73, 0x74, 0x72, 0x69,
	0x6E, 0x67, 0x3E, 0xE2, 0x82, 0xAC, 0x3C, 0x2F, 0x73, 0x74,
	0x72, 0x69, 0x6E, 0x67, 0x3E, 0x0A, 0x3C, 0x2F, 0x61, 0x72,
	0x72, 0x61, 0x79, 0x3E, 0x0A, 0x3C, 0x2F, 0x70, 0x6C, 0x69,
	0x73, 0x74, 0x3E, 0x0A
};
static const char sTestText4[] =
{
	0xE2, 0x82, 0xAC
};


// Cruzeiro symbol (U+20A2), literal
static const char sTestPlist5[] =
{
	0x3C, 0x3F, 0x78, 0x6D, 0x6C, 0x20, 0x76, 0x65, 0x72, 0x73,
	0x69, 0x6F, 0x6E, 0x3D, 0x22, 0x31, 0x2E, 0x30, 0x22, 0x20,
	0x65, 0x6E, 0x63, 0x6F, 0x64, 0x69, 0x6E, 0x67, 0x3D, 0x22,
	0x55, 0x54, 0x46, 0x2D, 0x38, 0x22, 0x3F, 0x3E, 0x0A, 0x3C,
	0x21, 0x44, 0x4F, 0x43, 0x54, 0x59, 0x50, 0x45, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x50, 0x55, 0x42, 0x4C, 0x49,
	0x43, 0x20, 0x22, 0x2D, 0x2F, 0x2F, 0x47, 0x4E, 0x55, 0x73,
	0x74, 0x65, 0x70, 0x2F, 0x2F, 0x44, 0x54, 0x44, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x30, 0x2E, 0x39, 0x2F, 0x2F,
	0x45, 0x4E, 0x22, 0x20, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3A,
	0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x67, 0x6E, 0x75, 0x73,
	0x74, 0x65, 0x70, 0x2E, 0x6F, 0x72, 0x67, 0x2F, 0x70, 0x6C,
	0x69, 0x73, 0x74, 0x2D, 0x30, 0x5F, 0x39, 0x2E, 0x78, 0x6D,
	0x6C, 0x22, 0x3E, 0x0A, 0x3C, 0x70, 0x6C, 0x69, 0x73, 0x74,
	0x20, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x22,
	0x30, 0x2E, 0x39, 0x22, 0x3E, 0x0A, 0x3C, 0x61, 0x72, 0x72,
	0x61, 0x79, 0x3E, 0x0A, 0x09, 0x3C, 0x73, 0x74, 0x72, 0x69,
	0x6E, 0x67, 0x3E, 0xE2, 0x82, 0xA2, 0x3C, 0x2F, 0x73, 0x74,
	0x72, 0x69, 0x6E, 0x67, 0x3E, 0x0A, 0x3C, 0x2F, 0x61, 0x72,
	0x72, 0x61, 0x79, 0x3E, 0x0A, 0x3C, 0x2F, 0x70, 0x6C, 0x69,
	0x73, 0x74, 0x3E, 0x0A
};
static const char sTestText5[] =
{
	0xE2, 0x82, 0xA2
};


// "test " with trailing space
static const char sTestPlist6[] =
{
	0x3C, 0x3F, 0x78, 0x6D, 0x6C, 0x20, 0x76, 0x65, 0x72, 0x73,
	0x69, 0x6F, 0x6E, 0x3D, 0x22, 0x31, 0x2E, 0x30, 0x22, 0x20,
	0x65, 0x6E, 0x63, 0x6F, 0x64, 0x69, 0x6E, 0x67, 0x3D, 0x22,
	0x55, 0x54, 0x46, 0x2D, 0x38, 0x22, 0x3F, 0x3E, 0x0A, 0x3C,
	0x21, 0x44, 0x4F, 0x43, 0x54, 0x59, 0x50, 0x45, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x50, 0x55, 0x42, 0x4C, 0x49,
	0x43, 0x20, 0x22, 0x2D, 0x2F, 0x2F, 0x47, 0x4E, 0x55, 0x73,
	0x74, 0x65, 0x70, 0x2F, 0x2F, 0x44, 0x54, 0x44, 0x20, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x20, 0x30, 0x2E, 0x39, 0x2F, 0x2F,
	0x45, 0x4E, 0x22, 0x20, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3A,
	0x2F, 0x2F, 0x77, 0x77, 0x77, 0x2E, 0x67, 0x6E, 0x75, 0x73,
	0x74, 0x65, 0x70, 0x2E, 0x6F, 0x72, 0x67, 0x2F, 0x70, 0x6C,
	0x69, 0x73, 0x74, 0x2D, 0x30, 0x5F, 0x39, 0x2E, 0x78, 0x6D,
	0x6C, 0x22, 0x3E, 0x0A, 0x3C, 0x70, 0x6C, 0x69, 0x73, 0x74,
	0x20, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x22,
	0x30, 0x2E, 0x39, 0x22, 0x3E, 0x0A, 0x3C, 0x61, 0x72, 0x72,
	0x61, 0x79, 0x3E, 0x0A, 0x09, 0x3C, 0x73, 0x74, 0x72, 0x69,
	0x6E, 0x67, 0x3E, 0x74, 0x65, 0x73, 0x74, 0x20, 0x3C, 0x2F,
	0x73, 0x74, 0x72, 0x69, 0x6E, 0x67, 0x3E, 0x0A, 0x3C, 0x2F,
	0x61, 0x72, 0x72, 0x61, 0x79, 0x3E, 0x0A, 0x3C, 0x2F, 0x70,
	0x6C, 0x69, 0x73, 0x74, 0x3E, 0x0A
};
static const char sTestText6[] =
{
	0x74, 0x65, 0x73, 0x74, 0x20
};


static const TestCase sTestCases[kTestCaseCount] =
{
	{
		sTestPlist1, sizeof sTestPlist1,
		sTestText1, sizeof sTestText1
	},
	{
		sTestPlist2, sizeof sTestPlist2,
		sTestText2, sizeof sTestText2
	},
	{
		sTestPlist3, sizeof sTestPlist3,
		sTestText3, sizeof sTestText3
	},
	{
		sTestPlist4, sizeof sTestPlist4,
		sTestText4, sizeof sTestText4
	},
	{
		sTestPlist5, sizeof sTestPlist5,
		sTestText5, sizeof sTestText5
	},
	{
		sTestPlist6, sizeof sTestPlist6,
		sTestText6, sizeof sTestText6
	}
};
