;;; GNU Guix --- Functional package management for GNU
;;; Copyright © 2015, 2017, 2022 Ricardo Wurmus <rekado@elephly.net>
;;; Copyright © 2016 Ben Woodcroft <donttrustben@gmail.com>
;;; Copyright © 2020 Martin Becze <mjbecze@riseup.net>
;;; Copyright © 2021 Sarah Morgensen <iskarian@mgsn.dev>
;;;
;;; This file is part of GNU Guix.
;;;
;;; GNU Guix is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; GNU Guix is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Guix.  If not, see <http://www.gnu.org/licenses/>.

(define-module (test-import-utils)
  #:use-module (guix tests)
  #:use-module (guix import utils)
  #:use-module ((guix licenses) #:prefix license:)
  #:use-module (guix packages)
  #:use-module (guix build-system)
  #:use-module (gnu packages)
  #:use-module (srfi srfi-64)
  #:use-module (ice-9 match))

(test-begin "import-utils")

(test-equal "beautify-description: use double spacing"
  "\
Trust me Mr. Hendrix, M. Night Shyamalan et al.  \
Differences are hard to spot,
e.g. in CLOS vs. GOOPS."
  (beautify-description
   "
Trust me Mr. Hendrix, M. Night Shyamalan et al. \
Differences are hard to spot, e.g. in CLOS vs. GOOPS."))

(test-equal "beautify-description: transform fragment into sentence"
  "This package provides a function to establish world peace"
  (beautify-description "A function to establish world peace"))

(test-equal "beautify-description: remove single quotes"
  "CRAN likes to quote acronyms and function names."
  (beautify-description "CRAN likes to 'quote' acronyms and 'function' names."))

(test-equal "beautify-description: escape @"
  "This @@ is not Texinfo syntax.  Neither is this %@@>%."
  (beautify-description "This @ is not Texinfo syntax.  Neither is this %@>%."))

(test-equal "license->symbol"
  'license:lgpl2.0
  (license->symbol license:lgpl2.0))

(test-equal "recursive-import"
  '((package                         ;package expressions in topological order
      (name "bar"))
    (package
      (name "foo")
      (inputs `(("bar" ,bar)))))
  (recursive-import "foo"
                    #:repo 'repo
                    #:repo->guix-package
                    (match-lambda*
                      (("foo" #:version #f #:repo 'repo)
                       (values '(package
                                  (name "foo")
                                  (inputs `(("bar" ,bar))))
                               '("bar")))
                      (("bar" #:version #f #:repo 'repo)
                       (values '(package
                                  (name "bar"))
                               '())))
                    #:guix-name identity))

(test-equal "recursive-import: skip false packages (toplevel)"
  '()
  (recursive-import "foo"
                    #:repo 'repo
                    #:repo->guix-package
                    (match-lambda*
                      (("foo" #:version #f #:repo 'repo)
                       (values #f '())))
                    #:guix-name identity))

(test-equal "recursive-import: skip false packages (dependency)"
  '((package
      (name "foo")
      (inputs `(("bar" ,bar)))))
  (recursive-import "foo"
                    #:repo 'repo
                    #:repo->guix-package
                    (match-lambda*
                      (("foo" #:version #f #:repo 'repo)
                       (values '(package
                                  (name "foo")
                                  (inputs `(("bar" ,bar))))
                               '("bar")))
                      (("bar" #:version #f #:repo 'repo)
                       (values #f '())))
                    #:guix-name identity))

(test-assert "alist->package with simple source"
  (let* ((meta '(("name" . "hello")
                 ("version" . "2.10")
                 ("source" .
                  ;; Use a 'file://' URI so that we don't cause a download.
                  ,(string-append "file://"
                                  (search-path %load-path "guix.scm")))
                 ("build-system" . "gnu")
                 ("home-page" . "https://gnu.org")
                 ("synopsis" . "Say hi")
                 ("description" . "This package says hi.")
                 ("license" . "GPL-3.0+")))
         (pkg (alist->package meta)))
    (and (package? pkg)
         (license:license? (package-license pkg))
         (build-system? (package-build-system pkg))
         (origin? (package-source pkg)))))

(test-assert "alist->package with explicit source"
  (let* ((meta '(("name" . "hello")
                 ("version" . "2.10")
                 ("source" . (("method" . "url-fetch")
                              ("uri"    . "mirror://gnu/hello/hello-2.10.tar.gz")
                              ("sha256" .
                               (("base32" .
                                 "0ssi1wpaf7plaswqqjwigppsg5fyh99vdlb9kzl7c9lng89ndq1i")))))
                 ("build-system" . "gnu")
                 ("home-page" . "https://gnu.org")
                 ("synopsis" . "Say hi")
                 ("description" . "This package says hi.")
                 ("license" . "GPL-3.0+")))
         (pkg (alist->package meta)))
    (and (package? pkg)
         (license:license? (package-license pkg))
         (build-system? (package-build-system pkg))
         (origin? (package-source pkg))
         (equal? (origin-sha256 (package-source pkg))
                 (base32 "0ssi1wpaf7plaswqqjwigppsg5fyh99vdlb9kzl7c9lng89ndq1i")))))

(test-equal "alist->package with false license"  ;<https://bugs.gnu.org/30470>
  'license-is-false
  (let* ((meta '(("name" . "hello")
                 ("version" . "2.10")
                 ("source" . (("method" . "url-fetch")
                              ("uri"    . "mirror://gnu/hello/hello-2.10.tar.gz")
                              ("sha256" .
                               (("base32" .
                                 "0ssi1wpaf7plaswqqjwigppsg5fyh99vdlb9kzl7c9lng89ndq1i")))))
                 ("build-system" . "gnu")
                 ("home-page" . "https://gnu.org")
                 ("synopsis" . "Say hi")
                 ("description" . "This package says hi.")
                 ("license" . #f))))
    ;; Note: Use 'or' because comparing with #f otherwise succeeds when
    ;; there's an exception instead of an actual #f.
    (or (package-license (alist->package meta))
        'license-is-false)))

(test-equal "alist->package with SPDX license name 1/2"  ;<https://bugs.gnu.org/45453>
  license:expat
  (let* ((meta '(("name" . "hello")
                 ("version" . "2.10")
                 ("source" . (("method" . "url-fetch")
                              ("uri"    . "mirror://gnu/hello/hello-2.10.tar.gz")
                              ("sha256" .
                               (("base32" .
                                 "0ssi1wpaf7plaswqqjwigppsg5fyh99vdlb9kzl7c9lng89ndq1i")))))
                 ("build-system" . "gnu")
                 ("home-page" . "https://gnu.org")
                 ("synopsis" . "Say hi")
                 ("description" . "This package says hi.")
                 ("license" . "expat"))))
    (package-license (alist->package meta))))

(test-equal "alist->package with SPDX license name 2/2"  ;<https://bugs.gnu.org/45453>
  license:expat
  (let* ((meta '(("name" . "hello")
                 ("version" . "2.10")
                 ("source" . (("method" . "url-fetch")
                              ("uri"    . "mirror://gnu/hello/hello-2.10.tar.gz")
                              ("sha256" .
                               (("base32" .
                                 "0ssi1wpaf7plaswqqjwigppsg5fyh99vdlb9kzl7c9lng89ndq1i")))))
                 ("build-system" . "gnu")
                 ("home-page" . "https://gnu.org")
                 ("synopsis" . "Say hi")
                 ("description" . "This package says hi.")
                 ("license" . "MIT"))))
    (package-license (alist->package meta))))

(test-equal "alist->package with dependencies"
  `(("gettext" ,(specification->package "gettext")))
  (let* ((meta '(("name" . "hello")
                 ("version" . "2.10")
                 ("source" . (("method" . "url-fetch")
                              ("uri"    . "mirror://gnu/hello/hello-2.10.tar.gz")
                              ("sha256" .
                               (("base32" .
                                 "0ssi1wpaf7plaswqqjwigppsg5fyh99vdlb9kzl7c9lng89ndq1i")))))
                 ("build-system" . "gnu")
                 ("home-page" . "https://gnu.org")
                 ("synopsis" . "Say hi")
                 ("description" . "This package says hi.")
                                                  ;
                 ;; Note: As with Guile-JSON 3.x, JSON arrays are represented
                 ;; by vectors.
                 ("native-inputs" . #("gettext"))

                 ("license" . #f))))
    (package-native-inputs (alist->package meta))))

(test-assert "alist->package with properties"
  (let* ((meta '(("name" . "hello")
                 ("version" . "2.10")
                 ("source" .
                  ;; Use a 'file://' URI so that we don't cause a download.
                  ,(string-append "file://"
                                  (search-path %load-path "guix.scm")))
                 ("build-system" . "gnu")
                 ("properties" . (("hidden?" . #t)
                                  ("upstream-name" . "hello-upstream")))
                 ("home-page" . "https://gnu.org")
                 ("synopsis" . "Say hi")
                 ("description" . "This package says hi.")
                 ("license" . "GPL-3.0+")))
         (pkg (alist->package meta)))
    (and (package? pkg)
         (equal? (package-upstream-name pkg) "hello-upstream")
         (hidden-package? pkg))))

(test-equal "spdx-string->license"
  '(license:gpl3+ license:agpl3 license:gpl2+)
  (map spdx-string->license
       '("GPL-3.0-oR-LaTeR" "AGPL-3.0" "GPL-2.0+")))

(test-end "import-utils")
