/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

#include "Array.h"
#include <gc/gc_allocator.h>

#include <iostream>
#include "Heap.h"
#include "ValueFuns.h"
#include "stdfuns.h"
#include <cstring>

#define ALLOC_SHIFT 10
#define ALLOC (1u << m_alloc)
#define ALLOC_MASK ((1u << m_alloc)-1)

#define ALLOC_CHUNK (Value**)GC_MALLOC_UNCOLLECTABLE(sizeof(Value*)*(1 << m_alloc))

Array::Array(ukint size):m_offset(0),m_size(0)
{
    // Make an initial block
    if (size != 0) {
	// allocate next bigger power of 2. Use a couple of shortcuts for
	// really small arrays so we don't waste too much time.
	if (size<8) { m_alloc=3; } else
	if (size<16) { m_alloc=4; } else
	if (size<32) { m_alloc=5; } else
	{ 
	    m_alloc = ukint(log2((double)size))+1;
	}
    } else {
	m_alloc = ALLOC_SHIFT;
    }
    m_blocks = (Value***)GC_MALLOC_UNCOLLECTABLE(sizeof(Value**));
    m_blocks[0] = ALLOC_CHUNK;
    m_numBlocks = 1;
}

Array::~Array() {
    for(ukint i=0;i<m_numBlocks;++i) {
	GC_FREE(m_blocks[i]);
    }
    GC_FREE(m_blocks);
}

void Array::expectedSize(ukint size)
{
    // No-op now
}

void Array::resize(ukint i)
{
    if (i<(m_size-m_offset)) {
	m_size = i+m_offset;
    } else {
        // This is slow. Better to just set the size, and remember how
        // much is allocated. But this'll do for now.
        while((m_size-m_offset)<i) {
            static Value* empty = new Value(NULL,KVT_NULL);
            push_back(empty);
        }
    }
}

void Array::unshift(Value* v) {
  if (m_offset == 0) {
    // just need to add a new block onto the left of the blocks,
    // and update the offset
    ukint chunk = m_size >> m_alloc;
    //    cerr<<m_offset<<" "<<m_size<<" "<<ALLOC<<" "<<m_numBlocks<<" "<<chunk<<endl;
    if (m_numBlocks <= chunk+1) { 
      m_numBlocks = chunk+2;
      m_blocks = (Value***)GC_REALLOC(m_blocks, m_numBlocks*sizeof(Value**));
      memmove(m_blocks+1,m_blocks,(chunk+1)*sizeof(Value**));
    } else {
      // in this case, we have at least one spare chunk on the right
      // may as well use it
      GC_FREE(m_blocks[chunk+1]);
      memmove(m_blocks+1,m_blocks,(chunk+1)*sizeof(Value**));
      //      m_size -= ALLOC;
    }
    m_blocks[0] = ALLOC_CHUNK;
    m_offset = ALLOC-1;
    m_blocks[0][m_offset] = v;
    m_size += ALLOC;
  } else {
    m_offset--;
    ukint chunk = m_offset >> m_alloc;
    ukint chunkpos = m_offset & ALLOC_MASK;
    m_blocks[chunk][chunkpos] = v;
  }
  //  

}

Value* Array::shift()
{
    Value* head = lookup(0);
    m_offset++;
    //    m_size--;
    return head;
}

ukint Array::reserved_size() {
    return m_numBlocks*ALLOC;
}

void Array::push_back(Value *v)
{

    ukint chunk = (m_size) >> m_alloc;
    ukint chunkpos = (m_size) & ALLOC_MASK;

    if ((chunk+1)>m_numBlocks) {
      // maybe there's a spare block on the left we can use

      if (m_offset > ALLOC) {
	//        cerr<<"A:"<<chunk<<" "<<m_numBlocks<<" "<<m_offset<<" "<<ALLOC<<" "<<endl;
	m_offset -= ALLOC;
	m_size -= ALLOC;
	GC_FREE(m_blocks[0]);
	chunk--;
	memmove(m_blocks,m_blocks+1,chunk*sizeof(Value**));
	//        cerr<<"B:"<<chunk<<" "<<m_numBlocks<<" "<<m_offset<<" "<<ALLOC<<" "<<endl;
      } else {
	m_numBlocks = chunk+1;
	m_blocks = (Value***)GC_REALLOC(m_blocks, m_numBlocks*sizeof(Value**));
      }
      m_blocks[chunk] = ALLOC_CHUNK;
    }

    m_blocks[chunk][chunkpos] = v;
    ++m_size;
}

Value* Array::lookup(ukint i)
{
//    if (i<ALLOC) return m_blocks[0][i];
    i += m_offset;
    if ((i+1)>m_size) {
      //      cerr<<i<<m_offset<<m_size<<ALLOC<<endl;
      return NULL;
    }
    ukint chunk = i >> m_alloc;
    ukint chunkpos = i & ALLOC_MASK;
    //    cerr<<i<<" "<<m_offset<<" "<<m_size<<" "<<ALLOC<<" "<<chunk<<" "<<chunkpos<<" "<<m_numBlocks<<endl;

    return m_blocks[chunk][chunkpos];
}

// TODO: These are too slow, since we can peek directly rather than lookup.
// CIM: On the other hand, there's something to be said for using lookup
bool Array::eq(Array* x, map<Value*, Value*>& done)
{
    if (x->size()!=size()) return false;
    for(ukint i=0;i<(m_size-m_offset);++i) {
	Value* idx = lookup(i);
	Value* xidx = x->lookup(i);
	if (!funtable_eq_aux(idx,xidx,done)) return false;
    }
    return true;
}

kint Array::cmp(Array* x)
{
    if (x->size()<size()) return -1;
    if (x->size()>size()) return 1;
    for(ukint i=0;i<(m_size-m_offset);++i) {
	Value* idx = lookup(i);
	Value* xidx = x->lookup(i);
	kint cmp = funtable_compare(NULL,idx,xidx);
	if (cmp!=0) return cmp;
    }
    return 0;
}

Array* Array::copy(VMState* vm, map<Value*,Value*>& done)
{
    Array* a = new(vm->getAPool()) Array(m_size-m_offset);
    for(ukint i=0;i<(m_size-m_offset);++i) {
	Value* copied;
	Value* x = lookup(i);
	// if x is already copied, return the copied value
	if (done.find(x)!=done.end()) {
	    copied = done[x];
	}
	else {
	    copied = funtable_copy_aux(vm, x, done);
	    done[x] = copied;
	}
	a->push_back(copied);
    }
    return a;
}

ukint Array::memusage()
{
    ukint size = 0;
    for(ukint i=0;i<m_numBlocks;++i) {
	size+=sizeof(Value**);
	size+=sizeof(Value*)*ALLOC;
	for(ukint j=0;j<ALLOC;++j) {
	    if (m_blocks[i][j]!=NULL)
		size+=funtable_memusage(m_blocks[i][j]);
	}
    }
    return size;
}
