/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcas_db_read.c
    \brief  the LCAS database reader
    \author Martijn Steenbakkers for the EU DataGrid.
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "lcas_log.h"
#include "_lcas_db_read.h"

/******************************************************************************
                                Definitions
******************************************************************************/
#define MAXDBENTRIES 250 /*!< maximum number of LCAS database entries \internal */
#define MAXPAIRS     2   /*!< maximum number of variable-value pairs that will be parsed per line
                            \internal */

#define WHITESPACE_CHARS " \t\n" /*!< \internal */
#define QUOTING_CHARS    "\"" /*!< \internal */
#define ESCAPING_CHARS   "\\" /*!< \internal */
#define COMMENT_CHARS    "#" /*!< \internal */

/*! Characters separating variable-value pairs in the lcas database file \internal */
#define PAIR_SEP_CHARS ","
/*! Characters separating variables from values \internal */
#define VARVAL_SEP_CHARS "="

/*
 * Characters that terminate pairs, values and variables in the lcas database file. This
 * is a combination of whitespace and separators.
 */
/*!
    Characters that terminate pairs in the lcas database file. This
    is a combination of whitespace and separators. \internal
 */
#define PAIR_TERMINATOR_CHARS PAIR_SEP_CHARS WHITESPACE_CHARS
/*!
    Characters that terminate variables and values in the lcas database file. This
    is a combination of whitespace and separators. \internal
 */
#define VARVAL_TERMINATOR_CHARS VARVAL_SEP_CHARS WHITESPACE_CHARS

#ifndef NUL
#define NUL '\0' /*!< \internal */
#endif

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int lcas_db_read_entries(FILE *);
static int lcas_db_parse_line(char *, lcas_db_entry_t **);
static int lcas_db_parse_pair(char *, char **, char **);
static int lcas_db_parse_string(char **);

/******************************************************************************
                       Define module specific variables
******************************************************************************/
static lcas_db_entry_t *  lcas_db_list=NULL; /*!< list of database entries */

/******************************************************************************
    Function: lcas_db_read
    documentation in _lcas_db_read.h
******************************************************************************/
lcas_db_entry_t ** lcas_db_read(
        char * lcas_db_fname
)
{
    FILE * lcas_db_fhandle;
    int    no_entries;

    lcas_db_fhandle = fopen(lcas_db_fname, "r");
    if (lcas_db_fhandle == NULL)
    {
        /* Cannot open file */
        return NULL;
    }

    no_entries=lcas_db_read_entries(lcas_db_fhandle);
    if (no_entries < 0)
    {
        lcas_log(0,"lcas.mod-lcas_db_read(): Parse error in line %d of %s\n",
            -(no_entries), lcas_db_fname);
        fclose(lcas_db_fhandle);
        return NULL;
    }
    else if (no_entries > MAXDBENTRIES)
    {
        lcas_log(0,"lcas.mod-lcas_db_read(): Too many entries found in %s\n",
            lcas_db_fname);
        lcas_log(0,"lcas.mod-lcas_db_read(): Only the first %d entries are used\n",
            MAXDBENTRIES);
    }
    fclose(lcas_db_fhandle);

    return &lcas_db_list;
}

/*!
    \fn lcas_db_read_entries(
        FILE * dbstream
        )
    \brief Read db entries from stream and fill a lsit of db entries
    \param dbstream database stream
    \return the number of entries found (failure --> negative number)
    \internal
*/
static int lcas_db_read_entries(
        FILE * dbstream
)
{
    char               line[1024];
    int                nlines=0;
    int                no_entries=0;
    lcas_db_entry_t *  entry=NULL;

    nlines=0;
    no_entries=0;
/*    lcas_db_fill_entry(no_entries,NULL); */
    while (fgets(line, sizeof(line), dbstream))
    {
        ++nlines;

        if (! lcas_db_parse_line(line, &entry))
        {
            /* parse error, return line number */
            if (entry != NULL) free(entry);
            return -(nlines);
        }
        if (entry != NULL)
        {
            lcas_log_debug(3,"line %d: %s, %s\n",nlines,entry->pluginname,entry->pluginargs);
            /* entry found */
            ++no_entries;
            if (no_entries > MAXDBENTRIES)
            {
                if (entry != NULL) free(entry);
                return no_entries;
            }
            if ( lcas_db_fill_entry(&lcas_db_list, entry)==NULL )
            {
                /* error filling lcas_db */
                if (entry != NULL) free(entry);
                return -(nlines);
            }
            if (entry != NULL) free(entry);
            entry=NULL;
        }
        else
        {
            /* no entry found, but no error */
            continue;
        }
    }
    if (entry != NULL) free(entry);
    return no_entries;
}

/******************************************************************************
    Function: lcas_db_fill_entry
    documentation in _lcas_db_read.h
******************************************************************************/
lcas_db_entry_t * lcas_db_fill_entry(
        lcas_db_entry_t ** list,
        lcas_db_entry_t * entry
)
{
    lcas_db_entry_t * plist;

    if (entry == NULL)
    {
        lcas_log(0,"lcas.mod-lcas_db_fill_entry(): error null entry\n");
        return NULL;
    }

    if (!(*list))
    {
        lcas_log_debug(2,"lcas.mod-lcas_db_fill_entry(): creating first list entry\n");
        *list=plist=(lcas_db_entry_t *)malloc(sizeof(lcas_db_entry_t));
    }
    else
    {
        lcas_log_debug(2,"lcas.mod-lcas_db_fill_entry(): creating new list entry\n");
        plist=*list;
        while (plist->next) plist=plist->next;
        plist=plist->next=(lcas_db_entry_t *)malloc(sizeof(lcas_db_entry_t));
    }
    if (!plist)
    {
        lcas_log(0,"lcas.mod-lcas_db_fill_entry(): error creating new list entry\n");
        return NULL;
    }
    plist->next=NULL;

    strncpy(plist->pluginname,entry->pluginname,LCAS_MAXPATHLEN);
    (plist->pluginname)[LCAS_MAXPATHLEN]=NUL;

    strncpy(plist->pluginargs,entry->pluginargs,LCAS_MAXARGSTRING);
    (plist->pluginargs)[LCAS_MAXARGSTRING]=NUL;

    return plist;
}

/*!
    \fn lcas_db_parse_line(
        char * line,
        lcas_db_entry_t ** entry
        )
    \brief Parses database line and fills database structure
    \param line database line
    \param entry pointer to a pointer to a database structure (can/should be freed afterwards)
    \retval 1 succes.
    \retval 0 failure.
    \internal
*/
static int lcas_db_parse_line(
        char * line,
        lcas_db_entry_t ** entry
)
{
    char *            var_val_pairs[MAXPAIRS];
    char *            var;
    char *            val;
    int               ipair;
    int               no_pairs;
    size_t            len;
    lcas_db_entry_t * tmp_entry=NULL;

    /* Check arguments */
    if ((line == NULL) || (*entry != NULL))
    {
        lcas_log(0,"lcas.mod-lcas_db_parse_line(): something wrong with arguments\n");
        goto error;
    }

    /* Skip over leading whitespace */
    line += strspn(line, WHITESPACE_CHARS);

    /* Check for comment at start of line and ignore line if present */
    if (strchr(COMMENT_CHARS, *line) != NULL) 
    {
        /* Ignore line, return NULL entry. */
        *entry=NULL;
        return 1;
    }

    /* Check for empty line */
    if (*line == NUL)
    {
        /* Empty line, return NULL entry. */
        *entry=NULL;
        return 1;
    }

    /* Look for variable-value pairs by checking the PAIR_SEP_CHARS */
    ipair=0;
    len=0;
    while (*line != NUL)
    {
        len=strcspn(line, PAIR_SEP_CHARS);
        if (len > 0)
        {
            var_val_pairs[ipair] = line;
            ipair++;
            line+=len;
            if (*line == NUL)
            {
                /* probably end of line */
                continue;
            }
            if (strchr(PAIR_SEP_CHARS, *line) != NULL)
            {
                *line=NUL;
                line += 1;
            }
        }
        else
        {
            /* len = 0, so *line=PAIR_SEP_CHARS */
            line += 1;
        }
        line += strspn(line, WHITESPACE_CHARS);
    }
    no_pairs=ipair;
    if (no_pairs)
    {
        tmp_entry=malloc(sizeof(*tmp_entry));
        if (tmp_entry == NULL)
        {
            lcas_log(0,"lcas.mod-lcas_db_parse_line(): error allocating memory\n");
            goto error;
        }

        for (ipair=0; ipair < no_pairs; ipair++)
        {
            int rc=0;
            lcas_log_debug(3,"pair %d:%s-endpair\n",ipair, var_val_pairs[ipair]);
            rc=lcas_db_parse_pair(var_val_pairs[ipair], &var, &val);
            if (! rc)
            {
                /* error parsing variable-value pair */
                lcas_log(0,"lcas.mod-lcas_db_parse_line(): error parsing variable-value pair\n");
                goto error;
            }
            lcas_log_debug(3,"var: %s, value: %s\n",var,val);

            if (strncmp(var,"pluginname",strlen("pluginname")) == 0)
            {
                /* found plugin name */
                strncpy(tmp_entry->pluginname,val,LCAS_MAXPATHLEN);
                (tmp_entry->pluginname)[LCAS_MAXPATHLEN]=NUL;
            }
            else if (strncmp(var,"pluginargs",strlen("pluginargs")) == 0)
            {
                /* found plugin database */
                strncpy(tmp_entry->pluginargs,val,LCAS_MAXARGSTRING);
                (tmp_entry->pluginargs)[LCAS_MAXARGSTRING]=NUL;
            }
            else
            {
                /* unknown option: do nothing */
            }
        }
    }

    /* succes */
    *entry=tmp_entry;
    return 1;

  error:
    if (tmp_entry != NULL) free(tmp_entry);
    return 0;
}

/*!
    \fn lcas_db_parse_pair(
        char * pair,
        char ** pvar,
        char ** pval
        )
    \brief Parses a database variable-value pair and returns the variable name and its value
    \param pair string containing the pair
    \param pvar pointer to the variable string
    \param pval pointer to the value string
    \retval 1 succes.
    \retval 0 failure.
    \internal
*/
static int lcas_db_parse_pair(
        char * pair,
        char ** pvar,
        char ** pval
)
{
    int    len;
    char * var;
    char * val;

    if (pair == NULL)
    {
        lcas_log(0,"lcas.mod-lcas_db_parse_pair(): cannot parse empty pair\n");
        return 0;
    }

    /* Skip over leading whitespace */
    pair += strspn(pair, WHITESPACE_CHARS);
    if (*pair == NUL)
    {
        lcas_log(0,"lcas.mod-lcas_db_parse_pair(): cannot parse empty pair\n");
        return 0;
    }

    var=pair;
    len=strcspn(pair, VARVAL_SEP_CHARS);
    pair+=len;
    if (*pair == NUL)
    {
        /* Cannot find '=' */
        lcas_log(0,"lcas.mod-lcas_db_parse_pair(): cannot find =\n");
        return 0;
    }

    if (strchr(VARVAL_SEP_CHARS, *pair) != NULL)
    {
        /* Found ' var =' and replace '=' with NUL*/
        *pair=NUL;
        if (! lcas_db_parse_string(&var))
        {
            /* error parsing variable name */
            return 0;
        }
        pair+=1;
        if (*pair==NUL)
        {
            /* No value found */
/*            val=NULL; */
            val=pair;
            *pvar=var;
            *pval=val;
            return 1;
        }
        else
        {
            /* Skip over leading whitespace */
            pair += strspn(pair, WHITESPACE_CHARS);
            if (*pair == NUL)
            {
                /* No value found */
/*                val=NULL; */
                val=pair;
                *pvar=var;
                *pval=val;
                return 1;
            }
            val=pair;
            if (! lcas_db_parse_string(&val))
            {
                /* error parsing value */
                return 0;
            }
        }
    }
    else
    {
        /* Cannot find '=' */
        lcas_log(0,"lcas.mod-lcas_db_parse_pair(): cannot find =\n");
        return 0;
    }

    /* success */
    *pvar=var;
    *pval=val;
    return 1;
}

/*!
    \fn lcas_db_parse_string(
        char ** pstring
        )
    \brief Takes a string and removes prepending and trailing spaces and quotes (unless escaped)
    \param pstring pointer to a pointer to a char
    \retval 1 succes.
    \retval 0 failure.
    \internal
*/
static int lcas_db_parse_string(
        char ** pstring
)
{
    char * end;
    char * string;

    string=*pstring;

    if (*string==NUL)
    {
        lcas_log(0,"lcas.mod-lcas_db_parse_string(): error parsing null string\n");
        return 0;
    }

    /* Is string quoted ? */
    if (strchr(QUOTING_CHARS, *string) != NULL)
    {
        /*
         * Yes, skip over opening quote and look for unescaped
         * closing double quote
         */
        string++;
        end=string;
        do
        {
            end += strcspn(end, QUOTING_CHARS);
            if (*end == NUL)
            {
                lcas_log(0,"lcas.mod-lcas_db_parse_string(): missing closing quote\n");
                return 0; /* Missing closing quote */
            }

            /* Make sure it's not escaped */
        }
        while (strchr(ESCAPING_CHARS, *(end - 1)) != NULL);
    }
    else
    {
        end = string + strcspn(string, WHITESPACE_CHARS);
    }
    *end=NUL;
    *pstring=string;

    return 1;
}

/******************************************************************************
    Function: lcas_db_clean_list
    documentation in _lcas_db_read.h
******************************************************************************/
int lcas_db_clean_list(
        lcas_db_entry_t ** list
)
{
    lcas_db_entry_t * entry;

    entry=*list;
    while (entry)
    {
        lcas_db_entry_t * next_entry;
        lcas_log_debug(2,"cleaning db entry for module %s\n",entry->pluginname);
        next_entry=entry->next;
        free(entry);
        entry=next_entry;
    }
    *list=entry=NULL;
    return 0;
}

/******************************************************************************
    Function: lcas_db_clean
    documentation in _lcas_db_read.h
******************************************************************************/
int lcas_db_clean()
{
    if(lcas_db_clean_list(&lcas_db_list) != 0)
    {
        lcas_log(0,"lcas.mod-lcas_db_clean() error: could not clean list\n");
        return 1;
    }
    return 0;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/src/lcas_db_read.c,v $
    $Date: 2010-06-15 15:32:10 $
    $Revision: 2.11 $
    $Author: okoeroo $
******************************************************************************/
