/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


/***************************************************************************
   grid-proxy-verify.c

 Sample C program that verifies a Globus GSI proxy.
 The following checks are performed:
 - certificate chain is verified , including proxy certs
 - proxy itself is verified (SUBJ/CN=proxy vs ISSUER etc)
 - proxy private key is matched against proxy public key
 - file permissions of proxy file are checked

 Build instructions:
    gcc -o grid-proxy-verify grid-proxy-verify.c \
        -I<OPENSSL-INCLUDE> -L<OPENSSL-LIB> -lssl -lcrypto
 (code is CFLAGS="-Wall -g -Wuninitialized -O" clean)

 ***************************************************************************/

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

#include <openssl/err.h>
#include <openssl/x509v3.h>

#include "_verify_log.h"
#include "verify_x509.h"

#define PROGNAME    "verify-proxy-tool"


static char  *fileName  = NULL;

static void print_usage( void );


static void print_usage( void )
{
    printf( "%s\n", PROGNAME);
    printf( "Usage:\n");
    printf( "  %s [-h|--help] [-d|--debug] [-q|--quiet] [-t|--atnotbefore] [proxy]\n\n", PROGNAME);
    printf( "Repeat -d/--debug multiple times to get more debugging output.\n");
    printf( "If no proxy is specified then %s is used.\n", fileName);

    exit(0);
}


int main( int argc, char **argv )
{
    const char       *CA_dir = NULL;

    int               i = 0;
    int		      result = 0;
    unsigned long     errcode=0,err = 0;
    char             *long_opt;
    struct stat       my_stat;
    int               log_lvl = 0;
    const char       *reason=NULL;
    char              data[255];
    int               rc=0;
    unsigned int      verify_at_notbefore=0;

    internal_verify_x509_data_t * verify_data = NULL;


    fileName = getenv( "X509_USER_PROXY" );
    if ( fileName == NULL )
    {
        snprintf( data, 255, "/tmp/x509up_u%u", getuid() );
	fileName = data;
    }

    for (i = 1; i < argc; i++)
    {
        if ( (strlen(argv[i]) >= 2 ) && ( argv[i][0] == '-' ) )
        {
            switch (argv[i][1])
            {
                case '-': long_opt = argv[i]+2;
                          if ( strcmp( long_opt, "help" ) == 0 )
                              print_usage();
                          else if ( strcmp( long_opt, "debug") == 0 )
                          {
                              log_lvl = verify_get_log_level();
                              log_lvl++;
                              verify_set_log_level (log_lvl);
                          }
                          else if ( strcmp( long_opt, "quiet") == 0 )
                              verify_set_log_level (0);
			  else if ( strcmp( long_opt, "atnotbefore") == 0 )
                              verify_at_notbefore=1;
                          else
                          {
                              fprintf( stderr, "Unknown option: %s\n", argv[i] );
                              print_usage();
                          }
                          break;
                case 'h': print_usage();
                          break;
		case 't': verify_at_notbefore=1;
			  break;
                case 'd': 
                          log_lvl = verify_get_log_level();
                          log_lvl++;
                          verify_set_log_level (log_lvl);
                          break;
                case 'q':
                          verify_set_log_level (0);
                          break;
                default:  fprintf( stderr, "Unknown option: %s\n", argv[i] );
                          print_usage();
            }
        }
        else
        {
            fileName = argv[i];
        }
    }


    /* First, find the trusted CA directory */
    CA_dir = getenv( "X509_CERT_DIR" );
    if ( CA_dir == NULL )
	CA_dir = "/etc/grid-security/certificates/";

    verify_log ( L_DEBUG, "Testing CA directory %s", CA_dir );

    /* Check the file permissions on the proxy */
    verify_log( L_INFO, "Checking file permissions for %s", fileName );

    if ( stat( fileName, &my_stat ) )	{
	verify_error(PROGNAME, "Could not stat file %s: %s",
		fileName, strerror(errno));
	rc=1;
	goto finalize;
    }

    if ( my_stat.st_mode & (S_IRWXG | S_IRWXO ) )
    {
        verify_error(PROGNAME,
		"Wrong file permissions for %s: should be 0600, are currently %04o.",
		fileName, my_stat.st_mode & 0xFFF);
    }


    /* The fileName is the proxy file and shown (above tests) to be accessible, including the CA_dir.
     * The CA_dir is used for the CA, sub-CA and CRL files.
     * Note: must implement check that make use of the CA namespace files
     */
    
    if (verify_X509_init(&verify_data)!=0)  {
	rc=1;
	goto finalize;
    }

    result = verify_X509_setParameter (&verify_data, VERIFY_X509_OPTIONS_VERIFY_AT_NOTBEFORE, verify_at_notbefore);
    if (result != VER_R_X509_PARAMS_OK)	{
	rc=1;
	goto finalize;
    }

    result = verify_X509_setParameter (&verify_data, VERIFY_X509_CERTIFICATE_FILEPATH, fileName);
    if (result == VER_R_X509_PARAMS_ALREADY_SET)
        printf ("VERIFY_X509_CERTIFICATE_FILEPATH already set...\n");
    else if (result == VER_R_X509_PARAMS_ACCESS_FAILURE)
    {
	rc=1;
	goto finalize;
    }
    
    result = verify_X509_setParameter (&verify_data, VERIFY_X509_CA_PATH, CA_dir);
    if (result == VER_R_X509_PARAMS_ALREADY_SET)
        printf ("VERIFY_X509_CA_PATH already set...\n");
    else if (result == VER_R_X509_PARAMS_ACCESS_FAILURE)
    {
	rc=1;
	goto finalize;
    }
    
    errcode = verify_X509_verify(&verify_data);
    rc=( errcode ? 2 : 0);
   
finalize:
    verify_X509_term(&verify_data);

    while ( (err=ERR_get_error()) ) {
	reason=ERR_reason_error_string(err);
	if (reason)
	    verify_log(L_WARN,"  error queue: function %s (%s): %s",
		    ERR_func_error_string(err),
		    ERR_lib_error_string(err),
		    reason);
	else
	    verify_log(L_WARN,"  error queue: %s",
		    ERR_error_string(err,NULL));
    }
    OBJ_cleanup();
    X509V3_EXT_cleanup();
#if OPENSSL_VERSION_NUMBER < 0x010100000L
    /* Claim is it is deprecated since 1.0.0, but for 1.0.1i I still see memory
     * leak from ERR_get_state() */
    ERR_remove_state(0);
#endif
    ERR_free_strings();
    EVP_cleanup();
    CRYPTO_cleanup_all_ex_data();

    return rc;
}

