/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_ENVIRONMENTS_H
#define MEDIASCANNER_ENVIRONMENTS_H

// Google Tests
#include <gtest/gtest.h>

// Boost C++
#include <boost/filesystem.hpp>

// C++ Standard Library
#include <map>
#include <string>

// Media Scanner Library
#include "mediascanner/glibutils.h"

namespace mediascanner {

class WritableMediaIndex;

class DBusEnvironment : public testing::Environment {
public:
    DBusEnvironment(const FileSystemPath &data_path,
                    const FileSystemPath &index_path);

    void SetUp();
    void TearDown();

private:
    FileSystemPath data_path_;
    FileSystemPath index_path_;

    pid_t dbus_daemon_pid_;
    GPid mediascanner_pid_;
};

class LuceneEnvironment : public testing::Environment {
public:
    explicit LuceneEnvironment(const FileSystemPath &index_path);
    void SetUp();

protected:
    virtual void FillMediaIndex(WritableMediaIndex *writer);

private:
    FileSystemPath index_path_;
};

class GriloPluginEnvironment : public testing::Environment {
public:
    typedef std::map<std::string, std::string> ConfigMap;
    typedef ConfigMap::value_type ConfigParam;

    GriloPluginEnvironment(const std::string &plugin_id,
                     const std::string &source_id,
                     const ConfigMap &config);

    void SetUp();
    void TearDown();

private:
    static void OnSourceDestroy(gpointer data, GObject *);

    std::string plugin_id_;
    std::string source_id_;
    ConfigMap config_params_;

    unsigned destroy_count_;

    GObject *object_;
};

class MediaScannerEnvironment : public GriloPluginEnvironment {
public:
    explicit MediaScannerEnvironment(const FileSystemPath &index_path);

protected:
    static ConfigMap MakeConfig(const FileSystemPath &index_path);
};

class TheMovieDbEnvironment : public GriloPluginEnvironment {
public:
    explicit TheMovieDbEnvironment(const std::string &mock_file);

    void SetUp();

protected:
    static ConfigMap MakeConfig();

private:
    std::string mock_file_;
};

class LastFmEnvironment : public GriloPluginEnvironment {
public:
    explicit LastFmEnvironment(const std::string &mock_file);

    void SetUp();

private:
    std::string mock_file_;
};

class FakeUDevEnvironment : public testing::Environment {
public:
    void SetUp();
};

} // namespace mediascanner

#endif // MEDIASCANNER_ENVIRONMENTS_H
