/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2015 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2015 Sun Microsystems, Inc.
 */

package org.netbeans.modules.cnd.mixeddev.wizard;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.platform.JavaPlatform;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ui.OpenProjects;
import org.netbeans.modules.cnd.api.model.CsmFile;
import org.netbeans.modules.cnd.api.model.CsmFunction;
import org.netbeans.modules.cnd.api.model.CsmModel;
import org.netbeans.modules.cnd.api.model.CsmModelAccessor;
import org.netbeans.modules.cnd.api.model.CsmOffsetableDeclaration;
import org.netbeans.modules.cnd.api.model.CsmProject;
import org.netbeans.modules.cnd.api.model.util.CsmKindUtilities;
import org.netbeans.modules.cnd.api.project.NativeProject;
import org.netbeans.modules.cnd.api.remote.RemoteFileUtil;
import org.netbeans.modules.cnd.api.toolchain.CompilerSet;
import org.netbeans.modules.cnd.makeproject.api.ProjectGenerator;
import org.netbeans.modules.cnd.makeproject.api.configurations.BasicCompilerConfiguration;
import org.netbeans.modules.cnd.makeproject.api.configurations.ConfigurationDescriptorProvider;
import org.netbeans.modules.cnd.makeproject.api.configurations.Folder;
import org.netbeans.modules.cnd.makeproject.api.configurations.Item;
import org.netbeans.modules.cnd.makeproject.api.configurations.LibraryItem;
import org.netbeans.modules.cnd.makeproject.api.configurations.MakeConfiguration;
import org.netbeans.modules.cnd.makeproject.api.configurations.MakeConfigurationDescriptor;
import org.netbeans.modules.cnd.makeproject.api.configurations.QmakeConfiguration;
import org.netbeans.modules.cnd.makeproject.api.wizards.WizardConstants;
import org.netbeans.modules.cnd.mixeddev.MixedDevUtils;
import org.netbeans.modules.cnd.mixeddev.java.JNISupport;
import org.netbeans.modules.cnd.modelutil.CsmUtilities;
import org.netbeans.modules.cnd.utils.CndPathUtilities;
import org.netbeans.modules.cnd.utils.FSPath;
import org.netbeans.modules.java.j2seproject.api.J2SEProjectPlatform;
import org.netbeans.modules.nativeexecution.api.ExecutionEnvironment;
import org.netbeans.modules.nativeexecution.api.ExecutionEnvironmentFactory;
import org.netbeans.modules.nativeexecution.api.NativeProcessBuilder;
import org.netbeans.modules.nativeexecution.api.util.ProcessUtils;
import org.openide.WizardDescriptor;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.Exceptions;
import org.openide.util.NbBundle;
import org.openide.util.Pair;
import org.openide.util.RequestProcessor;

/**
 *
 * @author alsimon
 */
public class Generator implements PropertyChangeListener {

    private static final Logger LOG = Logger.getLogger("org.netbeans.modules.mixeddev.wizard"); //NOI18N
    private static final RequestProcessor RP = new RequestProcessor(Generator.class.getName(), 2);
    private final FileObject fileObject;
    private final WizardDescriptor wiz;
    private volatile Project javaProject;
    private volatile Project makeProject;
    private volatile FileObject header;
    private volatile FileObject include;
    
    
    protected Generator(WizardDescriptor wiz, FileObject fileObject) {
        this.fileObject = fileObject;
        this.wiz = wiz;
    }

    protected void instantiate() throws IOException {
        try {
            if (generate()) {
                makeProject = instantiateImpl();
                if (makeProject != null) {
                    switchModel(false);
                    OpenProjects.getDefault().addPropertyChangeListener(this);
                    OpenProjects.getDefault().open(new Project[]{makeProject}, true);
                }
            }
        } catch (IOException ex) {
            Exceptions.printStackTrace(ex);
        }
    }

    protected String validate() {
        ClassPath sourceCP = ClassPath.getClassPath(fileObject, ClassPath.SOURCE);
        FileObject sr = sourceCP != null ? sourceCP.findOwnerRoot(fileObject) : null;
        javaProject = FileOwnerQuery.getOwner(sr);
        if (javaProject == null) {
            return NbBundle.getMessage(Generator.class, "Generator_NoOwner", sr.getPath()); // NOI18N
        }
        J2SEProjectPlatform pp = javaProject.getLookup().lookup(J2SEProjectPlatform.class);
        if (pp == null) {
            return NbBundle.getMessage(Generator.class, "Generator_NoJavaSE", javaProject.getClass()); // NOI18N
        }
        JavaPlatform jp = pp.getProjectPlatform();
        if (jp == null) {
            return NbBundle.getMessage(Generator.class, "Generator_NoJavaPlatform", pp.getClass()); //NOI18N
        }
        final FileObject binFO = jp.findTool("javah"); // NOI18N
        File javah = FileUtil.toFile(binFO); //NOI18N
        if (javah == null) {
            return NbBundle.getMessage(Generator.class, "Generator_NoJavah", jp.getClass()); //NOI18N
        }
        String classNameRelPath = FileUtil.getRelativePath(sr, fileObject);
        if (classNameRelPath.endsWith(".java")){ // NOI18N
            classNameRelPath = classNameRelPath.substring(0, classNameRelPath.length() - 5);
        }
        File workingDir = new File(FileUtil.toFile(sr.getParent()), "build/classes"); // NOI18N
        File classNamePath = new File(workingDir, classNameRelPath+".class"); // NOI18N
        if (!classNamePath.exists()) {
            return NbBundle.getMessage(Generator.class, "Generator_NoCompiled", fileObject.getPath()); //NOI18N
        }
        return null;
    }
    
    private boolean generate() {
        ClassPath sourceCP = ClassPath.getClassPath(fileObject, ClassPath.SOURCE);
        ClassPath compileCP = ClassPath.getClassPath(fileObject, ClassPath.COMPILE);
        FileObject sr = sourceCP != null ? sourceCP.findOwnerRoot(fileObject) : null;
        javaProject = FileOwnerQuery.getOwner(sr);
        J2SEProjectPlatform pp = javaProject.getLookup().lookup(J2SEProjectPlatform.class);
        JavaPlatform jp = pp.getProjectPlatform();
        final FileObject binFO = jp.findTool("javah"); // NOI18N
        final String headerName = fileObject.getName() + ".h"; // NOI18N
        header = JNISupport.generateJNIHeader(binFO, sr, fileObject, headerName, sourceCP, compileCP);
        if (header != null) {
            include = binFO.getParent().getParent().getFileObject("include"); // NOI18N
            return true;
        }
        return false;
    }
    
    private Project instantiateImpl() throws IOException {
        FSPath dirF = (FSPath) WizardConstants.PROPERTY_PROJECT_FOLDER.get(wiz);
        String hostUID = (String) WizardConstants.PROPERTY_HOST_UID.get(wiz);
        CompilerSet toolchain = (CompilerSet) WizardConstants.PROPERTY_TOOLCHAIN.get(wiz);
        boolean defaultToolchain = Boolean.TRUE.equals(WizardConstants.PROPERTY_TOOLCHAIN_DEFAULT.get(wiz));
        if (dirF != null) {
            ExecutionEnvironment ee = (ExecutionEnvironment) WizardConstants.PROPERTY_REMOTE_FILE_SYSTEM_ENV.get(wiz);
            if (ee == null) {
                ee = ExecutionEnvironmentFactory.getLocal();
            }
            dirF = new FSPath(dirF.getFileSystem(), RemoteFileUtil.normalizeAbsolutePath(dirF.getPath(), ee));
        }
        String projectName = (String) WizardConstants.PROPERTY_NAME.get(wiz);
        String makefileName = (String) WizardConstants.PROPERTY_GENERATED_MAKEFILE_NAME.get(wiz);
        int conftype = MakeConfiguration.TYPE_DYNAMIC_LIB;
        LibraryItem lib = new LibraryItem.OptionItem("-lstdc++"); // NOI18N
        List<LibraryItem> libs = Arrays.asList(lib);
        MakeConfiguration debug = MakeConfiguration.createConfiguration(dirF, "Debug", conftype, null, hostUID, toolchain, defaultToolchain); // NOI18N
        debug.getCCompilerConfiguration().getDevelopmentMode().setValue(BasicCompilerConfiguration.DEVELOPMENT_MODE_DEBUG);
        debug.getCCCompilerConfiguration().getDevelopmentMode().setValue(BasicCompilerConfiguration.DEVELOPMENT_MODE_DEBUG);
        debug.getFortranCompilerConfiguration().getDevelopmentMode().setValue(BasicCompilerConfiguration.DEVELOPMENT_MODE_DEBUG);
        debug.getAssemblerConfiguration().getDevelopmentMode().setValue(BasicCompilerConfiguration.DEVELOPMENT_MODE_DEBUG);
        debug.getQmakeConfiguration().getBuildMode().setValue(QmakeConfiguration.DEBUG_MODE);
        debug.getCCCompilerConfiguration().getIncludeDirectories().setValue(getIncludePaths(dirF.getPath()));
        debug.getLinkerConfiguration().getLibrariesConfiguration().setValue(libs);
        
        MakeConfiguration release = MakeConfiguration.createConfiguration(dirF, "Release", conftype, null, hostUID, toolchain, defaultToolchain); // NOI18N
        release.getCCompilerConfiguration().getDevelopmentMode().setValue(BasicCompilerConfiguration.DEVELOPMENT_MODE_RELEASE);
        release.getCCCompilerConfiguration().getDevelopmentMode().setValue(BasicCompilerConfiguration.DEVELOPMENT_MODE_RELEASE);
        release.getFortranCompilerConfiguration().getDevelopmentMode().setValue(BasicCompilerConfiguration.DEVELOPMENT_MODE_RELEASE);
        release.getAssemblerConfiguration().getDevelopmentMode().setValue(BasicCompilerConfiguration.DEVELOPMENT_MODE_RELEASE);
        release.getQmakeConfiguration().getBuildMode().setValue(QmakeConfiguration.RELEASE_MODE);
        release.getCCCompilerConfiguration().getIncludeDirectories().setValue(getIncludePaths(dirF.getPath()));
        release.getLinkerConfiguration().getLibrariesConfiguration().setValue(libs);
        
        MakeConfiguration[] confs = new MakeConfiguration[]{debug, release};
        ProjectGenerator.ProjectParameters prjParams = new ProjectGenerator.ProjectParameters(projectName, dirF);
        prjParams.setMakefileName(makefileName);
        prjParams.setConfigurations(confs);
        prjParams.setHostUID(hostUID);

        prjParams.setTemplateParams(new HashMap<String,Object>(wiz.getProperties()));
        Project createProject = ProjectGenerator.createProject(prjParams);
        return createProject;
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        if (evt.getPropertyName().equals(OpenProjects.PROPERTY_OPEN_PROJECTS)) {
            if (evt.getNewValue() instanceof Project[]) {
                final Project[] projects = (Project[])evt.getNewValue();
                if (projects.length == 0) {
                    return;
                }
                OpenProjects.getDefault().removePropertyChangeListener(this);
                RP.post(new Runnable() {

                    @Override
                    public void run() {
                        doWork();
                    }
                });
            }
        }
    }

    private void doWork() {
        try {
            ConfigurationDescriptorProvider pdp = makeProject.getLookup().lookup(ConfigurationDescriptorProvider.class);
            pdp.getConfigurationDescriptor();
            if (pdp.gotDescriptor()) {
                final MakeConfigurationDescriptor configurationDescriptor = pdp.getConfigurationDescriptor();
                if (header != null && header.isValid()) {
                    final FileObject newHeader = createHeader(configurationDescriptor);
                    final StringBuilder buf = new StringBuilder();
                    final FileObject newSource = createSource(configurationDescriptor, buf);
                    configurationDescriptor.save();
                    final CsmModel model = CsmModelAccessor.getModel();
                    if (model != null && makeProject != null) {
                        switchModel(true);
                        final NativeProject np = makeProject.getLookup().lookup(NativeProject.class);
                        final CsmProject p = model.getProject(np);
                        LOG.log(Level.FINE, "Generate stub, CsmProject: {0}", p); // NOI18N
                        if (p != null) {
                            p.waitParse();
                            try {
                                createStub(newHeader, newSource, buf);
                            } catch (IOException ex) {
                                Exceptions.printStackTrace(ex);
                            }
                            p.waitParse();
                            CsmUtilities.openSource(newSource, 0);
                            updateLibraryPath(configurationDescriptor);
                        }
                    }
                }
            }
        } catch (IOException ex) {
            Exceptions.printStackTrace(ex);
        }
    }
    
    private void updateLibraryPath(MakeConfigurationDescriptor configurationDescriptor) throws FileNotFoundException, IOException {
        FileObject propFile = javaProject.getProjectDirectory().getFileObject("nbproject/project.properties"); // NOI18N
        if (propFile != null && propFile.isValid() && configurationDescriptor.getActiveConfiguration() != null) {
            final InputStream inputStream = propFile.getInputStream();
            Properties prop = new Properties();
            prop.load(inputStream);
            inputStream.close();
            String args = prop.getProperty("run.jvmargs",""); // NOI18N
            String outputValue = configurationDescriptor.getActiveConfiguration().getAbsoluteOutputValue();
            if (!args.contains("-Djava.library.path=") ) { // NOI18N
                args+=" -Djava.library.path="+CndPathUtilities.getDirName(outputValue); // NOI18N
                prop.put("run.jvmargs", args); // NOI18N
                OutputStream outputStream = propFile.getOutputStream();
                prop.store(outputStream, args);
                outputStream.close();
            }
        }
    }
    
    private void switchModel(boolean state) {
        CsmModel model = CsmModelAccessor.getModel();
        if (model != null && makeProject != null) {
            NativeProject np = makeProject.getLookup().lookup(NativeProject.class);
            if (state) {
                model.enableProject(np);
            } else {
                model.disableProject(np);
            }
        }
    }

    public static void createStub(FileObject newHeader, FileObject newSource, StringBuilder buf) throws IOException {
        CsmFile includeFile = CsmUtilities.getCsmFile(newHeader, true, false);
        if (includeFile != null) {
            for(CsmOffsetableDeclaration declaration : includeFile.getDeclarations()) {
                if (CsmKindUtilities.isFunction(declaration)) {
                    CsmFunction f = (CsmFunction) declaration;
                    String declarationText = f.getText().toString();
                    int beg = declarationText.lastIndexOf('('); //NOI18N
                    int end = declarationText.lastIndexOf(')'); //NOI18N
                    if (beg > 0 && beg < end) {
                        buf.append('\n'); //NOI18N
                        buf.append(declarationText.substring(0, beg+1));
                        String[] params = declarationText.substring(beg+1,end).split(","); //NOI18N
                        for(int i = 0; i < params.length; i++) {
                            if (i > 0) {
                                buf.append(',').append(' '); //NOI18N
                            }
                            if (i == 0) {
                                buf.append(params[i].trim()).append(' ').append("env"); //NOI18N
                            } else if (i == 1) {
                                buf.append(params[i].trim()).append(' ').append("object"); //NOI18N
                            } else {
                                buf.append(params[i].trim()).append(' ').append("param").append(Integer.toString(i-1)); //NOI18N
                            }
                        }
                        buf.append(") {\n}\n"); //NOI18N
                    }
                }
            }
            Writer w = new OutputStreamWriter(newSource.getOutputStream());
            w.write(buf.toString());
            w.close();
        }
    }

    private FileObject createSource(final MakeConfigurationDescriptor configurationDescriptor, StringBuilder buf) throws IOException {
        FileObject newSource;
        Folder sourceFolder = getRootSource(configurationDescriptor);
        FileObject folder;
        if (sourceFolder.isDiskFolder()) {
            folder = RemoteFileUtil.getFileObject(sourceFolder.getAbsolutePath(), makeProject);
        } else {
            folder = configurationDescriptor.getBaseDirFileObject();
        }
        newSource = folder.createData(header.getName(), "cpp"); //NOI18N
        buf.append("// Native methods implementation of\n// ").append(fileObject.getPath()).append("\n\n"); //NOI18N
        buf.append("#include \"").append(header.getNameExt()).append("\"\n"); //NOI18N
        Writer w = new OutputStreamWriter(newSource.getOutputStream());
        w.write(buf.toString());
        w.close();
        Item item = Item.createInFileSystem(configurationDescriptor.getBaseDirFileSystem(),newSource.getPath());
        sourceFolder.addItemAction(item);
        return newSource;
    }

    private List<String> getIncludePaths(String baseDir) {
        List<String> includeDirectoriesVector = new ArrayList<String>();
        String includeDirectory = include.getPath();
        includeDirectory = CndPathUtilities.toRelativePath(baseDir, CndPathUtilities.naturalizeSlashes(includeDirectory));
        includeDirectory = CndPathUtilities.normalizeSlashes(includeDirectory);
        includeDirectoriesVector.add(includeDirectory);
        for(FileObject child : include.getChildren()) {
            if (child.isFolder()) {
                includeDirectory = child.getPath();
                includeDirectory = CndPathUtilities.toRelativePath(baseDir, CndPathUtilities.naturalizeSlashes(includeDirectory));
                includeDirectory = CndPathUtilities.normalizeSlashes(includeDirectory);
                includeDirectoriesVector.add(includeDirectory);
            }
        }
        return includeDirectoriesVector;
    }
    
    private FileObject createHeader(final MakeConfigurationDescriptor configurationDescriptor) throws IOException {
        Folder headersFolder = getRootHeader(configurationDescriptor);
        FileObject folder;
        if (headersFolder.isDiskFolder()) {
            folder = RemoteFileUtil.getFileObject(headersFolder.getAbsolutePath(), makeProject);
        } else {
            folder = configurationDescriptor.getBaseDirFileObject();
        }
        FileObject newHeader = FileUtil.copyFile(header, folder, header.getName());
        Item item = Item.createInFileSystem(configurationDescriptor.getBaseDirFileSystem(), newHeader.getPath());
        headersFolder.addItemAction(item);
        return newHeader;
    }
    
    public static Folder getRootHeader(MakeConfigurationDescriptor configurationDescriptor) {
        Folder folder = configurationDescriptor.getLogicalFolders();
        List<Folder> sources = folder.getFolders();
        for (Folder sub : sources){
            if (sub.isProjectFiles()) {
                if (MakeConfigurationDescriptor.HEADER_FILES_FOLDER.equals(sub.getName())) {
                    return sub;
                }
            }
        }
        return folder;
    }

    public static Folder getRootSource(MakeConfigurationDescriptor configurationDescriptor) {
        Folder folder = configurationDescriptor.getLogicalFolders();
        List<Folder> sources = folder.getFolders();
        for (Folder sub : sources){
            if (sub.isProjectFiles()) {
                if (MakeConfigurationDescriptor.SOURCE_FILES_FOLDER.equals(sub.getName())) {
                    return sub;
                }
            }
        }
        return folder;
    }
}
