/* payload.c
  
   Payload creation functions

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "payload.h"
#include "xmalloc.h"
#include "xstrdup.h"

/*
 * process standard C-style escape sequences in a string,
 * this can never lengthen the output, so cp and tp may overlap as long
 * as cp >= tp.
 *
 * This function comes from fetchmail's rcfile_l.l. It is released under
 * the GPL. See fetchmail's "COPYING" file for copyright information.
 *
 * Returns the number of bytes in the digested string (not including the
 * '\0' at the end.)
 */
static size_t
escapes(const char *cp /* source string with escapes */,
	char       *tp /* target buffer for digested string */)
{
    char *start = tp;

    while (*cp) {
	int cval = 0;

	/* we MUST check for NUL explicitly, as strchr(string, 0) will
	 * always succeed! */
	if (*cp == '\\' && cp[1] && strchr("0123456789xX", cp[1]) ) {
	    char *dp;
	    const char *hex = "00112233445566778899aAbBcCdDeEfF";
	    int dcount = 0;

	    if (*++cp == 'x' || *cp == 'X') {
		for (++cp;
		     *cp && (dp = strchr(hex, *cp) ) && (dcount++ < 2);
		     cp++)
		    cval = (cval * 16) + (dp - hex) / 2;
	    } else if (*cp == '0') {
		while (*cp && strchr("01234567", *cp) != NULL
		       && (dcount++ < 3) )
		    cval = (cval * 8) + (*cp++ - '0');
	    } else {
		while (*cp && (strchr("0123456789", *cp) != NULL)
		       &&(dcount++ < 3))
		    cval = (cval * 10) + (*cp++ - '0');
	    }
	} else if (*cp == '\\') {
	    /* C-style character escapes */
	    switch (*++cp) {
	    case '\n': cp++; continue;	/* backslash before LF to join lines */
	    case '\0': goto done;	   /* ignore backslash at file end */
	    case '\\': cval = '\\'; break;
	    case 'n': cval = '\n'; break;
	    case 't': cval = '\t'; break;
	    case 'b': cval = '\b'; break;
	    case 'r': cval = '\r'; break;
	    default: cval = *cp;
	    }
	    cp++;
	} else
	    cval = *cp++;

	*tp++ = cval;
    }

done:
    *tp++ = '\0';

    return tp - start - 1;
}

/*
 * Just reads a file into memory.
 */
static int
read_payload_from_file(char *fname, struct payload *p,
		       char errbuf[PAYLOAD_ERRBUF_SIZE])
{
    FILE *f;
    long fsize;

    f = fopen(fname, "r");
    if (f == NULL) {
	snprintf(errbuf, PAYLOAD_ERRBUF_SIZE,
		 "can't open file \"%s\": %s", fname, strerror(errno) );
	return -1;
    }

    fseek(f, 0, SEEK_END); /* Move file pointer to the end of the file */
    fsize = ftell(f);
    fseek(f, 0, SEEK_SET); /* Move file pointer to the beggining */

    p->data = xrealloc(p->data, p->len + fsize);

    fread(p->data + p->len, 1, fsize, f);
    fclose(f);

    p->len += fsize;

    return 0;
}

/*
 * Just reads a file from stdin, i.e. piped to us, into memory.
 */
static int
read_payload_from_stdin(struct payload *p, char errbuf[PAYLOAD_ERRBUF_SIZE])
{
    ssize_t bytes_read;
    char buffer[4096];

    while ( (bytes_read = fread(buffer, 1, sizeof(buffer), stdin) ) ) {
	if (ferror(stdin) ) {
	    snprintf(errbuf, PAYLOAD_ERRBUF_SIZE,
		     "fread(): %s", strerror(errno) );
	    return -1;
	}

	p->data = xrealloc(p->data, p->len + bytes_read);
	memcpy(p->data + p->len, buffer, bytes_read);
	p->len += bytes_read;
    }

    return 0;
}

static int
file_payload(struct payload *p, char *fname, char errbuf[PAYLOAD_ERRBUF_SIZE])
{
    int retval;

    if (!strcmp(fname, "-") )
	retval = read_payload_from_stdin(p, errbuf);
    else
	retval = read_payload_from_file(fname, p, errbuf);

    return retval;
}

static void
random_payload(struct payload *p, unsigned len)
{
    unsigned i;
    unsigned char *data;

    p->data = xrealloc(p->data, p->len + len);

    for (data = p->data + p->len, i = 0; i < len; i++)
	*data++ = rand();

    p->len += len;
}

static void
pattern_payload(struct payload *p, unsigned nrepetitions, char *pattern)
{
    struct payload payload;
    unsigned i;

    /*
     * This will give us more memory than we need since escape sequences
     * are represented by several bytes (like "\x12") and translate into
     * only one byte. We allocate an additional byte for the '\0' at the
     * end of the digested string.
     */
    payload.data = xmalloc(strlen(pattern) + 1);

    /*
     * Convert the pattern to actual data (because the pattern can
     * have escape codes or hex/octal numbers.)
     */
    payload.len = escapes(pattern, (char *) payload.data);

    /*
     * We allocated a bit more memory than we needed; we could give
     * back the excess by doing "payload.data = xrealloc(payload.data,
     * payload.len)" here, but it is probably not worth the trouble.
     */

    p->data = xrealloc(p->data, p->len + nrepetitions*payload.len);

    for (i = 0; i < nrepetitions; i++)
	memcpy(&p->data[p->len + i*payload.len], payload.data, payload.len);

    free(payload.data);

    p->len += payload.len*nrepetitions;
}

static void
string_payload(struct payload *p, char *string)
{
    struct payload payload;

    /*
     * This will give us more memory than we need since escape sequences
     * are represented by several bytes (like "\x12") and translate into
     * only one byte. We allocate an additional byte for the '\0' at the
     * end of the digested string.
     */
    payload.data = xmalloc(strlen(string) + 1);

    /*
     * Convert the string to actual data (because the string can
     * have escape codes or hex/octal numbers.)
     */
    payload.len = escapes(string, (char *) payload.data);

    /*
     * We allocated a bit more memory than we needed; we could give
     * back the excess by doing "payload.data = xrealloc(payload.data,
     * payload.len)" here, but it is probably not worth the trouble.
     */

    p->data = xrealloc(p->data, p->len + payload.len);

    memcpy(p->data + p->len, payload.data, payload.len);

    free(payload.data);

    p->len += payload.len;
}

/*

This is the stuff we implement in here:

-P "\x1\x2\03"
-P "file:/tmp/payload.dat" or -P "/tmp/payload.dat" or -P "./payload.dat"
   (file can be "-" for stdin)
-P "random:100" (100 bytes of random data)
-P "string:GET HTTP/1.0\n\n" or -P "GET HTTP/1.0\n\n"
-P "pattern:2:abcd" (2 repetitions of "abcd") or -P "pattern:2:\x0" (2 \0's)

Combination:

-P "file:/tmp/payload.dat:random:100:\x1\x2\03:GET / HTTP/1.0\n\n"

*/
int
create_payload(const char *payload_spec, struct payload *p,
	       char errbuf[PAYLOAD_ERRBUF_SIZE])
{
    char *spec, *ptrptr;
    char *s, *len_str, *pattern_str, *fname_str, *nreps_str, *string_str;
    unsigned len, nreps;

    spec = xstrdup(payload_spec);

    p->data = xmalloc(0); p->len = 0;

    for (s = strtok_r(spec, ":", &ptrptr);
	 s;
	 s = strtok_r(NULL, ":", &ptrptr) ) {
	if (strstr("random", s) ) {
	    len_str = strtok_r(NULL, ":", &ptrptr);
	    if (!len_str) {
		snprintf(errbuf, PAYLOAD_ERRBUF_SIZE,
			 "what is the length of the random-type payload?");
		goto error;
	    }

	    /*
	     * A random payload of 0 bytes in length is valid so we only check
	     * for errors, which are indicated by strtoul() setting errno to a
	     * negative number, per the following note from strtoul()'s man
	     * page:
	     *
	     * "Since  strtoul()  can legitimately return 0 or LONG_MAX
	     * (LLONG_MAX for strtoull()) on both success and failure, the
	     * calling program should set errno to 0 before the call, and then
	     * determine if an error occurred by checking whether errno has a
	     * non-zero value after the call."
	     */

	    errno = 0;
	    len = strtoul(len_str, NULL, 0);
	    if (errno < 0) {
		snprintf(errbuf, PAYLOAD_ERRBUF_SIZE,
			 "invalid length of random-type payload");
		goto error;
	    }

	    if (len > 0) {
		/*
		 * Fill len bytes starting at payload_data with random data.
		 */
		random_payload(p, len);
	    }
	} else if (strstr("string", s) ) {
	    string_str = strtok_r(NULL, ":", &ptrptr);
	    if (!string_str) {
		snprintf(errbuf, PAYLOAD_ERRBUF_SIZE,
			 "what is the string for the string-type payload?");
		goto error;
	    }

	    string_payload(p, string_str);
	} else if (strstr("pattern", s) ) {
	    nreps_str = strtok_r(NULL, ":", &ptrptr);
	    if (!nreps_str) {
		snprintf(errbuf, PAYLOAD_ERRBUF_SIZE,
			 "what is the number of repetitions of the "
			 "pattern-type payload?");
		goto error;
	    }

	    pattern_str = strtok_r(NULL, ":", &ptrptr);
	    if (!pattern_str) {
		snprintf(errbuf, PAYLOAD_ERRBUF_SIZE,
			 "what is the pattern for the pattern-type payload?");
		goto error;
	    }

	    nreps = strtoul(nreps_str, NULL, 0);
	    if (nreps == 0) {
		snprintf(errbuf, PAYLOAD_ERRBUF_SIZE,
			 "invalid number of repetitions of pattern-type "
			 "payload");
		goto error;
	    }

	    /* Fill len bytes starting at payload_data with random data */
	    pattern_payload(p, nreps, pattern_str);
	} else if (strstr("file", s) ) {
	    fname_str = strtok_r(NULL, ":", &ptrptr);
	    if (!fname_str) {
		snprintf(errbuf, PAYLOAD_ERRBUF_SIZE,
			 "what is the file path for the file-type payload?");
		goto error;
	    }

	    if (file_payload(p, fname_str, errbuf) == -1)
		goto error;
	} else
	    /* If we can't identify the thing we treat it as a string. */
	    string_payload(p, s);
    }

    free(spec);
    return 0;

error:
    free(spec);
    free(p->data);
    p->data = NULL;
    p->len = 0;
    return -1;
}
