var tape = require("tape"),
    topojson = require("../");

tape("filter null top-level geometry objects are preserved", function(test) {
  var topology = topojson.topology({
    feature: {type: "Feature", geometry: null},
    geometry: null
  });
  topojson.filter(topology, {"coordinate-system": "spherical"});
  test.deepEqual(topology.objects.feature, {type: null});
  test.deepEqual(topology.objects.geometry, {type: null});
  test.end();
});

tape("filter empty top-level feature collections are converted to null", function(test) {
  var topology = topojson.topology({
    collection: {type: "FeatureCollection", features: [
      {type: "Feature", properties: {}, geometry: null},
      {type: "Feature", properties: {}, geometry: null}
    ]}
  });
  topojson.filter(topology, {"coordinate-system": "spherical"});
  test.deepEqual(topology.objects.collection, {type: null});
  test.end();
});

tape("filter null inner features are removed", function(test) {
  var topology = topojson.topology({
    collection: {type: "FeatureCollection", features: [
      {type: "Feature", properties: {}, geometry: null},
      {type: "Feature", properties: {}, geometry: {type: "Point", coordinates: [0, 0]}},
      {type: "Feature", properties: {}, geometry: null}
    ]}
  });
  topojson.filter(topology, {"coordinate-system": "spherical"});
  test.deepEqual(topology.objects.collection, {type: "GeometryCollection", geometries: [{type: "Point", coordinates: [0, 0]}]});
  test.end();
});

tape("filter empty polygons are removed", function(test) {
  var topology = topojson.topology({
    collection: {type: "FeatureCollection", features: [
      {type: "Feature", properties: {}, geometry: null},
      {type: "Feature", properties: {}, geometry: {type: "Point", coordinates: [0, 0]}},
      {type: "Feature", properties: {}, geometry: {type: "Polygon", coordinates: [[[0, 0], [1, 1], [1, 1], [0, 0]]]}},
      {type: "Feature", properties: {}, geometry: null}
    ]}
  });
  topojson.filter(topology, {"coordinate-system": "spherical"});
  test.deepEqual(topology.objects.collection, {type: "GeometryCollection", geometries: [{type: "Point", coordinates: [0, 0]}]});
  test.end();
});

tape("filter empty rings are removed", function(test) {
  var topology = topojson.topology({
    collection: {type: "GeometryCollection", geometries: [
      {type: "Polygon", coordinates: [[[0, 0], [1, 1], [1, 1], [0, 0]], [[0, 0], [1, 0], [1, 1], [0, 1], [0, 0]]]},
      {type: "Polygon", coordinates: [[[0, 0], [0, 1], [1, 1], [1, 0], [0, 0]], [[0, 0], [1, 1], [1, 1], [0, 0]]]}
    ]}
  });
  topojson.filter(topology, {"coordinate-system": "spherical", "preserve-attached": false});
  test.deepEqual(topology.objects.collection, {type: "GeometryCollection", geometries: [
    {type: "Polygon", arcs: [[~0, ~1]]},
    {type: "Polygon", arcs: [[~0, ~1]]}
  ]});
  test.end();
});

tape("filter large interior rings (holes) are preserved", function(test) {
  var topology = topojson.topology({
    collection: {type: "GeometryCollection", geometries: [
      {type: "Polygon", coordinates: [
        [[0, 0], [0, 20], [20, 20], [20, 0], [0, 0]], // area ~0.12
        [[5, 5], [15, 5], [15, 15], [5, 15], [5, 5]] // area ~0.03
      ]}
    ]}
  });
  topojson.filter(topology, {"coordinate-system": "spherical", "minimum-area": 0.01});
  test.deepEqual(topology.objects.collection, {type: "GeometryCollection", geometries: [
    {type: "Polygon", arcs: [[0], [1]]}
  ]});
  test.end();
});

tape("filter small interior rings (holes) are removed", function(test) {
  var topology = topojson.topology({
    collection: {type: "GeometryCollection", geometries: [
      {type: "Polygon", coordinates: [
        [[0, 0], [0, 20], [20, 20], [20, 0], [0, 0]], // area ~0.12
        [[5, 5], [15, 5], [15, 15], [5, 15], [5, 5]] // area ~0.03
      ]}
    ]}
  });
  topojson.filter(topology, {"coordinate-system": "spherical", "minimum-area": 0.1});
  test.deepEqual(topology.objects.collection, {type: "GeometryCollection", geometries: [
    {type: "Polygon", arcs: [[0]]}
  ]});
  test.end();
});

tape("filter small exterior rings with large interior rings are removed", function(test) {
  var topology = topojson.topology({
    collection: {type: "GeometryCollection", geometries: [
      {type: "Polygon", coordinates: [
        [[0, 0], [0, 20], [20, 20], [20, 0], [0, 0]], // area ~0.12
        [[5, 5], [5, 15], [15, 15], [15, 5], [5, 5]] // clockwise! area ~12.54
      ]}
    ]}
  });
  topojson.filter(topology, {"coordinate-system": "spherical", "minimum-area": 0.2});
  test.deepEqual(topology.objects.collection, {type: null});
  test.end();
});

tape("filter very large exterior rings are preserved", function(test) {
  var topology = topojson.topology({
    collection: {type: "GeometryCollection", geometries: [
      {type: "Polygon", coordinates: [
        [[0, 0], [20, 0], [20, 20], [0, 20], [0, 0]] // area ~12.45
      ]}
    ]}
  });
  topojson.filter(topology, {"coordinate-system": "spherical", "minimum-area": 0.01, "force-clockwise": false});
  test.deepEqual(topology.objects.collection, {type: "GeometryCollection", geometries: [
    {type: "Polygon", arcs: [[0]]}
  ]});
  test.end();
});

tape("filter small exterior rings attached to other rings are preserved", function(test) {
  var topology = topojson.topology({
    collection: {type: "GeometryCollection", geometries: [
      {type: "MultiPolygon", coordinates: [
        [[[2, 0], [2, 1], [3, 1], [3, 0], [2, 0]]], // area 1, detached
        [[[4, 0], [4, 1], [5, 1], [5, 0], [4, 0]]], // area 1, detached
        [[[0, 0], [0, 1], [1, 1], [1, 0], [0, 0]]], // area 1, attached
        [[[0, 1], [0, 2], [1, 2], [1, 1], [0, 1]]] // area 1, attached
      ]}
    ]}
  });
  topojson.filter(topology, {"coordinate-system": "cartesian", "minimum-area": 2});
  test.deepEqual(topology.objects.collection, {type: "GeometryCollection", geometries: [
    {type: "MultiPolygon", arcs: [
      [[-1, -2]],
      [[1, -3]]
    ]}
  ]});
  test.end();
});

tape("filter zero-area exterior rings coincident with other rings are removed", function(test) {
  var topology = topojson.topology({
    collection: {type: "GeometryCollection", geometries: [
      {type: "MultiPolygon", coordinates: [
        [[[0, 0], [0, 1], [1, 1], [1, 0], [0, 0]]], // area 1
        [[[0, 0], [0, 0], [0, 0], [0, 0]]], // area 0
        [[[1, 0], [1, 0], [1, 0], [1, 0]]], // area 0
        [[[0, 1], [0, 1], [0, 1], [0, 1]]], // area 0
        [[[1, 1], [1, 1], [1, 1], [1, 1]]] // area 0
      ]}
    ]}
  });
  topojson.filter(topology, {"coordinate-system": "cartesian", "minimum-area": .5});
  test.deepEqual(topology.objects.collection, {type: "GeometryCollection", geometries: [
    {type: "MultiPolygon", arcs: [
      [[-1, -2, -3, -4]]
    ]}
  ]});
  test.end();
});

tape("filter polygons with no rings are removed", function(test) {
  var topology = {
    objects: {
      collection: {type: "GeometryCollection", geometries: [
        {type: "Polygon", arcs: []}
      ]}
    },
    arcs: []
  };
  topojson.filter(topology, {"coordinate-system": "spherical"});
  test.deepEqual(topology.objects.collection, {type: null});
  test.end();
});

tape("filter empty top-level geometry objects are converted to null", function(test) {
  var topology = topojson.topology({line: {type: "Polygon", coordinates: [[[0, 0], [1, 1], [1, 1], [0, 0]]]}});
  topojson.filter(topology, {"coordinate-system": "spherical"});
  test.deepEqual(topology.objects.line, {type: null});
  test.end();
});

tape("filter non-empty top-level geometry objects are preserved", function(test) {
  var topology = topojson.topology({polygon: {type: "Polygon", coordinates: [[[0, 0], [0, 1], [1, 1], [1, 0], [0, 0]]]}});
  topojson.filter(topology, {"coordinate-system": "spherical"});
  test.deepEqual(topology.objects.polygon, {type: "Polygon", arcs: [[0]]});
  test.end();
});
