[comment {-*- tcl -*- nx::Class manpage}]

[include version.inc]
[manpage_begin nx::Class 3 [vset VERSION]]

[comment {For the time being, we do not render keywords & terms; and
the corresponding reverse index}] 
[proc keywords args {}]
[proc term v {return $v}]


[keywords "base metaclass"]
[keywords NX]
[keywords "mixin class"]
[keywords "re-classification"]
[keywords "submethod"]
[keywords "method ensemble"]
[keywords "linearisation"]
[keywords "filter specification"]
[keywords "metaclass"]

[vset SCOPE "class"]
[vset CMD "cls"]
[vset MODIFIER ""]

[copyright {2014-2016 Stefan Sobernig <stefan.sobernig@wu.ac.at>, Gustaf Neumann <gustaf.neumann@wu.ac.at>; available under the Creative Commons Attribution 3.0 Austria license (CC BY 3.0 AT).}]
[titledesc {API reference of the base-metaclass of the NX objectsystem}]
[moddesc {NX API}]

[description]
[para]

[cmd nx::Class] is the [term "base metaclass"] of the [term NX] object
system. All classes (e.g. [emph cls]) are (direct or indirect)
instances of [cmd nx::Class]. Therefore, the methods provided by [cmd nx::Class] are
available to all classes. A class [emph cls] which does
not have [cmd nx::Class] as its direct or indirect superclass is
referred to as an [emph "application class"]. By default, when
instantiating a new class from [cmd nx::Class], it becomes an
application class with [cmd nx::Object] being set as its superclass. A
class [emph cls] which is explicitly declared as a (direct or
indirect) subclass of [cmd nx::Class] is referred to as a [emph metaclass], that
is, its instances will become classes as well. In other words, a
metaclass instantiates and subclasses [cmd nx::Class] at the same
time.

[example {
+---------+
| ::nx::* |
+---------+--------------------------------------Y
|                                                |
|  instance of                                   |
|   .-------.                                    |
|  +--------'+     instance of     +----------+  |
|  |         |<....................|          |  |
|  |  Class  |                     |  Object  |  |
|  |         |....................>|          |  |
|  +---------+     subclass of     +-----+----+  |
|   ^   ^                                ^       |
\...|...|................................|......./
    |   |                                |
    |   |subclass.....(xor)......subclass|
    |   |of       +-----------+        of|
    |   |.........|           |..........|
    | (metaclass) |   /cls/   | (application class)
    |.............|           |
    instance of   +-----------+
}]

Classes can be created in the following ways:

[list_begin definitions]

[call [cmd nx::Class] [method create] [arg cls] [opt "[option -superclasses] [arg superClassNames]"] [opt "[option -mixins] [arg mixinSpec]"] [opt "[option -filters] [arg filterSpec]"] [opt "[arg option] [arg value] ..."] [opt [arg initBlock]]]

To create a class having the explicit name [arg cls], use [method create].

  
  [call [cmd nx::Class] [method new] [opt "[option -superclasses] [arg superClassNames]"] [opt "[option -mixins] [arg mixinSpec]"] [opt "[option -filters] [arg filterSpec]"] [opt [arg initBlock]]]

To create a class having an automatically assigned, implicit name, use [method new].

[list_end]

The configuration options for direct and indirect instances of [cmd nx::Class],
which can be passed when calling [method create] and [method new], are
documented in the subsequent section.

[section {Configuration Options for Instances of nx::Class}]

[para] Configuration options can be used for configuring objects during
their creation by passing the options as non-positional arguments into calls
of [method new] and [method create] (see [cmd nx::Class]). An
existing object can be queried for its current configuration using
[method cget] and it can be re-configured using [method configure].

[list_begin options]

[opt_def -superclasses [opt [arg superClassNames]]]

If [arg superClassNames] is not specified, returns the superclasses of
the class. If provided, the class becomes the subclass of [arg superClassNames].

[opt_def -filters [opt [arg filterSpecs]]]

Retrieves the list of filter methods currently active on instances of
the class, if [arg filterSpecs] is not set. Otherwise, activates a
list of filter methods for the instances of the class. Filters are
returned or set in terms of a list of [term "filter specification"]s.

[opt_def -mixins [opt [arg mixinSpecs]]]

Returns the list of [term "mixin class"]es currently active on
instances of the class, if [arg mixinSpecs] is not specified. Otherwise, the class
is extended by the list of [term "mixin class"]es provided by [arg mixinSpecs]. 
[term "mixin class"]es are returned or set in terms of a list of [term "mixin specification"]s.

[list_end]

The configuration options provided by [cmd nx::Object] are equally
available because an application class [arg cls] is an indirect
instance of [cmd nx::Object].

[section {Methods for Instances of nx::Class}]

[list_begin commands]

[cmd_def alias]
[list_begin definitions]

[include alias.man.inc]

[list_end]

[cmd_def __class_configureparameter]

[list_begin definitions]

[def "[arg cls] [method "__class_configureparameter"]"]

Computes and returns the configuration options available for [arg cls] instances, to be consumed as method-parameter specification by [method configure].

[list_end]

[cmd_def create]

[list_begin definitions]

[call [arg cls] [method create] [arg instanceName] [opt "[arg option] [arg value] [arg option] [arg value] ..."]] 

This factory method creates an instance [arg instanceName] of [arg cls]
and returns [arg instanceName].

[example {
% nx::Class create AClass {
    :method init args {
      next
    }; # initialization method for instances of 'AClass'
  }; # defines a class 'AClass' being an instance of 'nx::Class'
::AClass
% ::AClass create anInstance; # defines an object 'anInstance' being an instance of 'AClass'
::anInstance
% ::anInstance info class
::AClass
% ::AClass info class
::nx::Class
}]

[method create] accepts the configuration options [arg option]
available for this instance, such as those defined by properties of
[arg cls] (see [method "property"]).

[para]

Note that [method create] is called internally when defining an
instance of [arg cls] using [method new].

[para]

By calling [method create] on [cmd nx::Class] itself, the created
instance will become a new application class [arg instanceName] on
which [method create] can also be applied (i.e., it can be
instantiated). If the so-created class has [cmd ::nx::Class] its
direct or indirect superclass, [arg instanceName] is referred to as a
[term "metaclass"]; that is, a class whose instances are again
classes.

[list_end]

[cmd_def delete]

[list_begin definitions]

[include delete.man.inc]

[list_end]

[cmd_def filters]

[list_begin definitions]

[include filter.man.inc]

[list_end]

[cmd_def forward]

[list_begin definitions]

[include forward.man.inc]

[list_end]

[cmd_def info]

A collection of introspection submethods on the structural features (e.g.
configuration options, superclasses) and the behavioral features (e.g.
methods, [term "filter"]s) provided by [arg cls] to its instances.

[list_begin definitions]

[call [arg cls] [method "info heritage"] [opt [arg pattern]]]

If [arg pattern] is omitted, returns the list of object names of all the direct and indirect
superclasses and [emph per-class] [term "mixin class"]es of [arg cls], in
their order of precedence, which are active for instances of [arg cls]. If
[arg pattern] is specified, only superclasses and [term "mixin class"]es whose names
match [arg pattern] will be listed (see [cmd "string match"]).

[call [arg cls] [method "info instances"] [opt [option -closure]] [opt [arg pattern]]]

If [arg pattern] is not specified, returns a list of the object names
of all the direct instances of [arg cls]. If the [term "switch"]
[option -closure] is set, indirect instances are also returned. A
direct instance is created by using [method create] or [method new] on
[arg cls], an indirect instance was created from a direct or indirect
subclass of [arg cls]. If [arg pattern] is specified, only instances
whose names match [arg pattern] will be listed (see [cmd "string match"]).

[call [arg cls] [method "info mixinof"] [opt [option -closure]] [opt "[option -scope] [arg option]"] [opt [arg pattern]]]

If [arg pattern] is not specified, returns a list of the object names
of all the objects for which [arg cls] is active as a
direct [term "mixin class"]. If the [term "switch"]
[option -closure] is set, objects which have [arg cls] as an indirect
[term "mixin class"] are also returned. If [arg pattern] is
specified, only objects whose names match [arg pattern] will
be listed (see [cmd "string match"]). Valid values of [arg option] are
[const all], [const object], and [const class]. Passing [const object]
will have only objects returned which have [arg cls] as [emph per-object]
[term "mixin class"]. Passing [const class] will have only classes
returned which have [arg cls] as [emph per-class] [term "mixin class"]. [const all] (the
default) will have contained both in the returned list.


[call [arg cls] [method "info subclasses"] [opt [option -closure]] [opt [option -dependent]] [opt [arg pattern]]]

If [arg pattern] is not specified, returns a list of the object names
of the direct subclasses of [arg cls]. If the [term "switch"] [option -closure] is
set, indirect subclasses are also returned. If the [term "switch"] [option -dependent] is on, indirect subclasses introduced by [term "mixin class"] relations of subclasses of [arg cls] are also reported. [option -closure] and [option -dependent] are mutually exclusive. If [arg pattern] is specified, only subclasses whose names match [arg pattern] will be listed (see [cmd "string match"]).

[call [arg cls] [method "info superclasses"] [opt [option -closure]] [opt [arg pattern]]]

If [arg pattern] is not specified, returns a list of the object names
of all direct superclasses of [arg cls]. If the [term "switch"] [option -closure] is
set, indirect superclasses will also be returned. If [arg pattern] is specified, only superclasses whose names match [arg pattern] will be listed (see [cmd "string match"]).

[include info.man.inc]

[list_end]


[cmd_def method]

[list_begin definitions]

[include method.man.inc]

[list_end]

[cmd_def mixins]
[list_begin definitions]

[include mixin.man.inc]

[list_end]

[cmd_def new]

[list_begin definitions]

[call [arg cls] [method new] [opt "[option "-childof"] [arg parentName]"] [opt "[arg option] [arg value] [arg option] [arg value] ..."]]

A factory method to create autonamed instances of [arg cls]. It
returns the name of the newly created instance. For example:

[example {
% nx::Class create AClass; # defines a class 'AClass' being an instance of 'nx::Class'
::AClass
% set inst [::AClass new]; # defines an autonamed object being an instance of 'AClass'
::nsf::__#0
% $inst info class
::AClass
}]

The factory method will provide computed object names of the form,
e.g. [const ::nsf::__#0]. The uniqueness of generated object names is
guaranteed for the scope of the current Tcl interpreter only.

[para]

It is a frontend to [method create] which will be called by [method new] once
the name of the instance has been computed, passing along the
arguments [arg option] to [method new] as the configuration options
(see [method create]).

[para]
If [option -childof] is provided, the new object will be created as a
nested object of [arg parentName]. [arg parentName] can be the name of
either an existing NX object or an existing Tcl namespace. If
non-existing, a Tcl namespace [arg parentName] will be created on the
fly.

[list_end]

[cmd_def property]

[list_begin definitions]

[call [arg cls] [method property] [opt "[option -accessor] [const public] | [const protected] | [const private]"] [opt "[option -class] [arg className]"] [opt "[option -configurable] [arg trueFalse]"] [opt [option -incremental]] [opt "[option -trace] [const set] | [const get] | [const default]"] [arg spec] [opt [arg initBlock]]]

[include property.man.inc]

[list_end]

[cmd_def require]

[list_begin definitions]

[include require.man.inc]

[list_end]

[cmd_def variable]

[list_begin definitions]

[comment {::nx::Object variable ?-accessor /value/? ?-incremental? ?-class /value/? ?-configurable /boolean/? ?-initblock /value/? ?-nocomplain? /spec/ ?/defaultValue/?}]

[call [arg cls] [method variable] [opt "[option -accessor] [const public] | [const protected] | [const private]"] [opt [option -incremental]] [opt "[option -class] [arg className]"] [opt "[option -configurable] [arg trueFalse]"] [opt "[option -initblock] [opt "[option -trace] [const set] | [const get] | [const default]"] [arg script]"] [arg spec] [opt [arg defaultValue]]]

[include variable.man.inc]

[list_end]

[list_end]

[section {Object Life Cycle}]

[cmd nx::Class] provides means to control important stages through
which an NX object passes between and including its creation and its
destruction: allocation, recreation, deallocation.

[example {
        /cls/->create(/instance/)
                  .---------------.   exists?    [false]   .----------------.                   .-------------------.
             ---->|Class::create()|----><>---------------->|Class::__alloc()|-----------><>---->|Object::configure()|
                  `---------------'      |      (1)        `----------------'             ^ (3) `---------+---------'
                                  [true] |                                                |               | (4)
                                         |  .-------------------.                         |      .------------------.
                                         `->|Class::__recreate()|-------------------------'      |/instance/->init()|
                                      (2)   `-------------------'                                `------------------'
    /instance/->destroy()
        .-----------------.     .------------------.
   ---->|Object::destroy()|---->|Class::__dealloc()|
        `-----------------' (5) `------------------'
}]

Object creation is controlled by the factory method [method create], provided by [cmd nx::Class] to its
instance [arg cls]. [method create] produces a new object [arg instance] as an
instance of [arg cls] in a number of steps.

[list_begin enumerated]

[enum] If [arg instance] does not represent an existing object, an
internal call to [method "__alloc"], provided by [cmd nx::Class], runs
the [emph {allocation}] procedure for a fresh [arg instance] of [arg cls].

[enum] If [arg instance] corresponds to an existing object, the
[emph {recreation}] procedure is triggered by calling [method "__recreate"]
defined by [cmd nx::Class].
  
[enum] The newly allocated or recreated object [arg instance] is then
configured by dispatching [method configure], provided by [cmd nx::Object], which
consumes the configuration options passed into [method create]. This
will establish the instance's initial state, e.g. by setting object
variables and object relations according to the configuration options
and corresponding default values.
  
[enum] Finally, the initialization method [method init] is dispatched, if
available for [arg instance]. [method "init"] can be defined by [arg cls] on
behalf of its instance [arg instance], e.g. to lay out a
class-specific initialisation behaviour.

[example_begin]
% nx::Class create Foo {:property x}
% Foo method init {} {set :y [lb]expr {${:x} + 1}[rb]}
% Foo public method bar {} {return ${:y}}
% Foo create f1 -x 101
% f1 cget -x
101
% f1 bar
102
[example_end]

Alternatively, the object [arg instance] may define an per-object
[method init] on its own. A per-object [method init] can be chained to
a class-level [method init] using [cmd nx::next], just like a regular
method.

[para]

Note that the definition of an [method init] method must contain an
empty parameter specification, since [method init] is always called
with an empty argument list.

[list_end]

Object destruction, such as triggered by an application-level
[method destroy] call (5), is finalized by [method {__dealloc}] offerd by
[cmd nx::Class]. 

[para]

In the following, the three built-in procedures ---
allocation, recreation, and deallocation --- are explained:

[list_begin itemized]

[item] [emph {Allocation}]: [method "__alloc"] creates a blank object [arg instance] as an instance of [arg cls] and returns the fully-qualified [arg instance]. [method __alloc] is primarily used internally by [method create] to allocate a Tcl memory storage for [arg instance] and to register [arg instance] with the Tcl
interpreter as a new command.

[item] [emph {Recreation}]:

Recreation is the NX scheme for resolving naming conflicts between
objects: An object is requested to be created using [method create] or
[method new] while an object of an identical object name, e.g. [arg instance], already
exists:

[example {
% Object create Bar
::Bar
% Object create Bar; # calls Object->__recreate(::Bar, ...)
::Bar
}]

In such a situation, the built-in [method "__recreate"] first unsets
the object state (i.e., Tcl variables held by the object) and removes
relations of the object under recreation with other objects. Then,
second, standard object initialization is performed by calling [method configure] and
[method init], if any.


[para]

Alternatively, recreation will be performed as a sequence of [method destroy] and
[method create] calls in the following recreation scenarios:

[list_begin itemized]
[item] An existing class is requested to be recreated as an object.
[item] An existing object is requested to be recreated as a class.
[example {
  % Object create Bar
  ::Bar
  % Class create Bar; # calls Bar->destroy() & Class::create(::Bar, ...)
}]
[item] An object of an object system other than NX (e.g. XOTcl2) is asked to be recreated.
[list_end]



[item] [emph {Deallocation}]: [method __dealloc] marks an instance [arg instance] of [arg cls] for
deletion by returning its Tcl memory representation to the Tcl memory pool and by 
unregistering the corresponding Tcl command with the Tcl interpreter.

[para]

Beware that [method __dealloc] does not necessarily
cause the object to be deleted immediately. Depending on the lifecycle
of the object's environment (e.g. the Tcl interp interpreter, the containing
namespace) and on call references down the callstack, the actual
memory freeing/returning operation may occur at a later point.

[list_end]

The three methods [method __alloc], [method __recreate], and [method __dealloc] are
internally provided and internally called. By default, they are not part of
the method interface of [arg cls] and cannot be called directly by clients of [arg cls].
In addition, [method __alloc], [method __recreate], and [method __dealloc] are protected from
redefinition by a script.

[para]

To extend or to replace the built-in allocation, recreation, and
deallocation procedure, the methods [method __alloc], [method __recreate], and
[method __dealloc] can be refined by providing a custom method
implementation:
[list_begin itemized]
[item] as a per-object method of [arg cls];
[item] as a method of a per-object [term "mixin class"] extending [arg cls];
[item] as a method of a per-class [term "mixin class"] extending [cmd nx::Class];
[item] as a method of a subclass specializing [cmd nx::Class], from which [arg cls] is to be instantiated.
[list_end]

This custom implementation can redirect to the built-in [method __alloc], [method __recreate], and
[method __dealloc], respectively, by using [cmd nx::next]. By
providing such a custom implementation, [method __alloc], [method __recreate], and
[method __dealloc], respectively, become available as callable methods
of [arg cls]:

[list_begin definitions]
[def "[arg cls] [method "__alloc"] [arg instance]"]
[def "[arg cls] [method "__recreate"] [arg instance] [opt "[arg arg] ..."]"]
[def "[arg cls] [method "__dealloc"] [arg instance]"]
[list_end]


[manpage_end]
