///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/BooleanControllerUI.h>
#include <core/undo/UndoManager.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/AnimManager.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(BooleanControllerUI, ReferenceParameterUI)

/******************************************************************************
* The constructor.
******************************************************************************/
BooleanControllerUI::BooleanControllerUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField) 
	: ReferenceParameterUI(parentEditor, refField)
{
	_checkBox = new QCheckBox(referenceField().displayName());
	connect(_checkBox, SIGNAL(clicked(bool)), this, SLOT(updateParameterValue()));
	
	// Update the displayed value when the animation time has changed.
	connect(&ANIM_MANAGER, SIGNAL(timeChanged(TimeTicks)), this, SLOT(updateUI()));	
}

/******************************************************************************
* Destructor, that releases all GUI controls.
******************************************************************************/
BooleanControllerUI::~BooleanControllerUI()
{
	// Release GUI controls. 
	delete checkBox(); 
}

/******************************************************************************
* This method is called when parameter object has been assigned to the reference field of the editable object 
* this parameter UI is bound to. It is also called when the editable object itself has
* been replaced in the editor. This implementation of the method automatically
* disables the UI widgets when the reference field of the edited object does not contain a
* controller. 
******************************************************************************/
void BooleanControllerUI::resetUI()
{
	ReferenceParameterUI::resetUI();	
	
	if(checkBox()) 
		checkBox()->setEnabled(parameterObject() != NULL && isEnabled());
}

/******************************************************************************
* Sets the enabled state of the UI.
******************************************************************************/
void BooleanControllerUI::setEnabled(bool enabled)
{
	if(enabled == isEnabled()) return;
	ReferenceParameterUI::setEnabled(enabled);
	if(checkBox()) checkBox()->setEnabled(parameterObject() != NULL && isEnabled());
}

/******************************************************************************
* This method updates the displayed value of the parameter UI.
******************************************************************************/
void BooleanControllerUI::updateUI()
{
	BooleanController* ctrl = dynamic_object_cast<BooleanController>(parameterObject());
	if(ctrl != NULL && checkBox()) {
		bool val = ctrl->getCurrentValue();
		checkBox()->setChecked(val);
	}
}

/******************************************************************************
* Takes the value entered by the user and stores it in the parameter object
* this parameter UI is bound to.
******************************************************************************/
void BooleanControllerUI::updateParameterValue()
{
	BooleanController* ctrl = dynamic_object_cast<BooleanController>(parameterObject());
	if(ctrl != NULL && checkBox()) {
		UNDO_MANAGER.beginCompoundOperation(tr("Change Parameter"));
		ctrl->setCurrentValue(checkBox()->isChecked());	
		UNDO_MANAGER.endCompoundOperation();
		updateUI();
	}
}

};

