/*	JPEG2000_Info

HiRISE CVS ID: JPEG2000_Info.java,v 1.25 2012/04/16 06:10:20 castalia Exp

Copyright (C) 2006-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Image_Tools;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.EOFException;
import java.lang.ArithmeticException;
import java.net.URL;
import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.net.ProtocolException;

import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.FileImageInputStream;

import PIRL.PVL.*;
import PIRL.Strings.String_Buffer;


/**	<i>JPEG2000_Info</i> is a Parameter Aggregate describing a file
	of JPEG2000 information.
<p>
	This class is intended to be the base class for <i>JP2_Info</i> and
	<i>JPEG2000_Codestream_Info</i> Parameter sets. The JPEG2000_Info methods
	know how to manage the assigned ImageInputStream to obtain data
	values and keep track of the stream position, but it expects its
	subclasses to know how to use the data values to generate the
	appropriate Parameters.
<p>
	@author		Bradford Castalia UA/PIRL
	@version	1.25
	@see		PIRL.Image_Tools.JP2_Info
	@see		PIRL.Image_Tools.JPEG2000_Codestream_Info
*/
public class JPEG2000_Info
	extends Parameter
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Image_Tools.JPEG2000_Info (1.25 2012/04/16 06:10:20)";

/**	Parameter name for specifying the position of data in the Source file
	(byte offset from the beginning of the file).
*/
public static final String
	DATA_POSITION_PARAMETER					= "^Data_Position";

/**	Parameter name for specifying a data block offset in an enclosing
	object.
<p>
	For JP2 boxes, the offset is relative to the byte immediately
	following the box header's length, type and extended length (if
	present) fields.
<p>
	For codestream segements, the offset is relative to the byte
	immediately following the segment marker and length fields.
*/
public static final String
	DATA_OFFSET_PARAMETER					= "Data_Offset";

/**	Parameter name for specifying the length (bytes) of a data block.
*/
public static final String
	DATA_LENGTH_PARAMETER					= "Data_Length";

/**	Parameter name for specifying the bit precision of a value.
*/
public static final String
	VALUE_BITS_PARAMETER					= "Value_Bits";

/**	Value of a bit precision Parameter value when the number of bits is
	variable.
*/
public static final int
	BITS_VARIABLE							= 0;

/**	Generic parameter names used by both JP2_Info and
	JPEG20000_Codestream_Info
*/
public static final String
	LENGTH_PARAMETER						= "Length",
	POSITION_PARAMETER						= "^Position",
	TOTAL_COMPONENTS_PARAMETER				= "Total_Components";

//!	Warning message parameter name.
public static final String
	WARNING_PARAMETER						= "WARNING";

//!	Name for an unrecognized box type or segment marker.
public static final String
	UNKNOWN									= "Unknown";

/**	The ImageInputStream Source from which the JPEG2000 information is obtained.
<p>
	@see	#Source()
*/
protected ImageInputStream
	Image_Input_Stream						= null;

/**	The current read position of the Source file.
<p>
	@see	#Stream_Position()
*/
protected long
	Stream_Position							= -1;

/**	Default flag for skipping tile segments.
<p>
	@see	#Skip_Tiles(boolean)
*/
public static boolean
	Skip_Tiles_Default						= true;

/**	Flag for skipping tile segments.
<p>
	@see	#Skip_Tiles(boolean)
*/
protected boolean
	Skip_Tiles								= Skip_Tiles_Default;

/**	Default flag for using {@link #Data_Position_Parameters(Parameter,
	long) data position parameters} in preference to {@link
	#Data_Offset_Parameters(Parameter, long, long) data offset
	parameters}.
<p>
	@see	#Use_Data_Position(boolean)
*/
public static boolean
	Use_Data_Position_Default				= true;

/**	Flag for using {@link #Data_Position_Parameters(Parameter, long) data
	position parameters} in preference to {@link
	#Data_Offset_Parameters(Parameter, long, long) data offset
	parameters}.
<p>
	@see	#Use_Data_Position(boolean)
*/
protected boolean
	Use_Data_Position						= Use_Data_Position_Default;

/**	Default flag for throwing an IllegalStateException when the Source is
	found to contain an invalid box or segment organization or contents.
<p.
	@see	#Throw_Illegal_State(boolean)
*/
public static boolean
	Throw_Illegal_State_Default				= true;

/**	Flag for throwing an IllegalStateException when the Source
	is found to contain an invalid box or segment organization or contents.
<p.
	@see	#Throw_Illegal_State(boolean)
*/
protected boolean
	Throw_Illegal_State						= Throw_Illegal_State_Default;


//  DEBUG control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTORS		= 1 << 0,
	DEBUG_ACCESSORS			= 1 << 1,
	DEBUG_PARAMETERS		= 1 << 2,
	DEBUG_HELPERS			= 1 << 3,
	DEBUG_MAIN				= 1 << 4,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a JPEG2000_Info for a named source.
<p>
	@param	source	The source from which to obtain the JPEG2000_Info.
		This may be a local filename or a remote URL reference.
	@throws	UnknownHostException If the source is a URL but the
		hostname is specifies can not be resolved.
	@throws IllegalArgumentException If the source is a URL but does
		not specify the http protocol or a source pathname.
	@throws	IOException	If there was a problem reading the file.
*/
public JPEG2000_Info
	(
	String	source
	)
	throws IOException, UnknownHostException, IllegalArgumentException
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> JPEG2000_Info: " + source);
Classification (AGGREGATE);
Source (source);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< JPEG2000_Info");
}

/**	Construct a JPEG2000_Info for a File.
<p>
	@param	file	The File from which to obtain the JPEG2000_Info.
	@throws	IOException	If there was a problem reading the file.
*/
public JPEG2000_Info
	(
	File	file
	)
	throws IOException
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> JPEG2000_Info: " + file);
Classification (AGGREGATE);
Source (file);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< JPEG2000_Info");
}

/**	Construct a JPEG2000_Info for an ImageInputStream.
<p>
	@param	image_input_stream	The ImageInputStream from which to obtain
		the JPEG2000_Info.
	@throws	IOException	If there was a problem reading the file.
*/
public JPEG2000_Info
	(
	ImageInputStream	image_input_stream
	)
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> JPEG2000_Info: " + image_input_stream);
Classification (AGGREGATE);
Source (image_input_stream);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< JPEG2000_Info");
}

/**	Construct a an empty JPEG2000_Info Object.
<p>
	Use one of the Source methods to assign a source file.
*/
public JPEG2000_Info ()
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">-< JPEG2000_Info");
Classification (AGGREGATE);
}

/*==============================================================================
	Accessors
*/
/**	Assign a named file as the information source.
<p>
	An attempt is made to form a URL from the source. If this succeeds an
	HTTP_ImageInput_Stream is constructed from the URL and then delegated
	to the {@link #Source(ImageInputStream)} method.
<p>
	If a URL can not be formed from the source a File is constructed from
	the source and then delegated to the {@link #Source(File)} method.
<p>
	@param	source	The name of the source to use as the information source.
	@return	This JPEG2000_Info.
	@throws	UnknownHostException If the source is a URL but the
		hostname is specifies can not be resolved.
	@throws IllegalArgumentException If the source is a URL but does
		not specify the http protocol or a source pathname.
	@throws	FileNotFoundException	If the file could not be accessed for
		any reason.
	@throws	ProtocolException	If the source is a valid URL but an HTTP
		{@link HTTP_ImageInputStream#Status() Status} protocol error was
		returned by the server.
	@throws	IOException	If there was a problem reading the source.
*/
public JPEG2000_Info Source
	(
	String	source
	)
	throws IOException, UnknownHostException, IllegalArgumentException
{
if (source == null)
	return this;
try
	{
	URL
		url = new URL (source);
	HTTP_ImageInputStream
		image_input_stream = null;
	try {image_input_stream = new HTTP_ImageInputStream (url);}
	catch (IllegalArgumentException exception)
		{
		throw new IllegalArgumentException (ID + '\n'
			+ "Inaccessble URL: " + url + '\n'
			+ exception.getMessage ());
		}
	catch (UnknownHostException exception)
		{
		throw new UnknownHostException (ID + '\n'
			+ "Inaccessble URL: " + url + '\n'
			+ exception.getMessage ());
		}
	catch (FileNotFoundException exception)
		{
		throw new FileNotFoundException (ID + '\n'
			+ "Inaccessble URL: " + url + '\n'
			+ exception.getMessage ());
		}
	catch (ProtocolException exception)
		{
		throw new ProtocolException (ID + '\n'
			+ "Inaccessble URL: " + url + '\n'
			+ exception.getMessage ());
		}
	catch (IOException exception)
		{
		throw new IOException (ID + '\n'
			+ "Inaccessble URL: " + url + '\n'
			+ exception.getMessage ());
		}
	if (DEBUG != DEBUG_OFF)
		image_input_stream.Logging (true);
	Name (image_input_stream.Source_URL ().toString ());
	return Source (image_input_stream);
	}
catch (MalformedURLException exception) {}
return Source (new File (source));
}

/**	Assign a File as the information source.
<p>
	The name of this JPEG2000_Info is set to the canonical pathname of
	the File.
<p>
	An ImageInputStream is constructed from the file and then delegated
	to the {@link #Source(ImageInputStream)} method.
<p>
	@param	file	The File to use as the information source.
	@return	This JPEG2000_Info.
	@throws	FileNotFoundException	If the file could not be accessed for
		any reason.
	@throws	IOException	If there was a problem reading the file.
*/
public JPEG2000_Info Source
	(
	File	file
	)
	throws IOException
{
if (file == null)
	return this;
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> JPEG2000_Info.Source: File " + file.getCanonicalPath ());
if (! file.exists ())
	throw new FileNotFoundException (ID + '\n'
		+"Unable to obtain JP2 Info for Source file "
			+ file.getCanonicalPath () + '\n'
		+"  because no file exists at that pathname.");
if (! file.isFile ())
	throw new FileNotFoundException (ID + '\n'
		+"Unable to obtain JP2 Info for Source file "
			+ file.getCanonicalPath () + '\n'
		+"  because the file is not a regular file.");
if (! file.canRead ())
	throw IO_Error
		("Unable to obtain JP2 Info for Source file "
			+ file.getCanonicalPath () + '\n'
		+"  because the file can not be read.");

ImageInputStream
	image_input_stream = null;
try {image_input_stream = new FileImageInputStream (file);}
catch (IOException exception)
	{
	throw IO_Error
		("Unable to obtain JP2 Info for Source file "
			+ file.getCanonicalPath () + '\n'
		+"  because an ImageInputStream could not be created for it.",
		exception);
	}

Name (file.getCanonicalPath ());

Source (image_input_stream);
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< JPEG2000_Info.Source: File " + file.getCanonicalPath ());
return this;
}

/**	Assign an ImageInputStream as the information source.
<p>
	If the image_input_stream is null or identical to the currently
	assigned ImageInputStream nothing is done.
<p>
	If this JPEG2000_Info does not have a name it is set to
	"ImageInputStream". All Parameters are removed. The current {@link
	#Stream_Position() stream position} is determined.
<p>
	@param	image_input_stream	The ImageInputStream to use as the
		information source.
	@return	This JPEG2000_Info.
*/
public JPEG2000_Info Source
	(
	ImageInputStream	image_input_stream
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> JPEG2000_Info.Source: " + image_input_stream + '\n'
		+"    Current Image_Input_Stream: " + Image_Input_Stream);
if (image_input_stream != null &&
	image_input_stream != Image_Input_Stream)
	{
	Image_Input_Stream = image_input_stream;
	try {Stream_Position = Image_Input_Stream.getStreamPosition ();}
	catch (IOException exception)
		{
		/*	Shouldn't happen.
			Just assume the beginning of the stream.
		*/
		Stream_Position = 0;
		}
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("    Image_Input_Stream reset to " + Image_Input_Stream);
	if (Name ().length () == 0)
		Name ("ImageInputStream");

	//	Remove any and all current parameters.
	Remove_All ();
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< JPEG2000_Info.Source");
return this;
}

/**	Get the currently assigned ImageInputStream information source.
<p>
	@return	The currently assigned ImageInputStream. This will be null
		if no source has been assigned.
*/
public ImageInputStream Source ()
{return Image_Input_Stream;}

/**	Enable of disable skipping of tile segments.
<p>
	When a codestream is being scanned for segment markers the tile-part
	segments - which begin with a Start-Of_Tile (SOT) marker - may be
	skipped. When there are many tiles in the codestream this will
	significantly reduce the amount of time to scan the Source and the
	number of Parameter groups generated.
<p>
	@param	enabled	If true, tile-part segments will be skipped; otherwise
		all tile-part segments will be examined and Parameter groups
		assembled for each one.
	@return	This JPEG2000_Info.
*/
public JPEG2000_Info Skip_Tiles
	(
	boolean	enabled
	)
{Skip_Tiles = enabled; return this;}

/**	Test if tile segments will be skipped.
<p>
	@return	true if tile segments will be skipped; false otherwise.
	@see	#Skip_Tiles(boolean)
*/
public boolean Skip_Tiles ()
{return Skip_Tiles;}

/**	Enable or disable the use of Source data position parameters.
<p>
	@param	enabled	If true, Source data position parameters are included
		for all JP2 boxes and codestream segments, and used elsewhere.
		Otherwise position parameters are not included and data offset
		parameters are used elsewhere instead of position parameters.
	@return	This JPEG2000_Info.
*/
public JPEG2000_Info Use_Data_Position
	(
	boolean	enabled
	)
{Use_Data_Position = enabled; return this;}

/**	Test if Source data position parameters will be used.
<p>
	@return	true if Source data position parameters will be used;
		false otherwise.
	@see	#Use_Data_Position(boolean)
*/
public boolean Use_Data_Position ()
{return Use_Data_Position;}

/**	Enable or disable throwing of IllegalStateException when an invalid
	JP2 box or codestream segment structure or contents is encountered.
<p>
	@param	enabled	If true an IllegalStateException will be thrown at
		the point where the JP2 box or codestream segment structure is
		determined to be invalid.
	@return	This JPEG2000_Info.
*/
public JPEG2000_Info Throw_Illegal_State
	(
	boolean	enabled
	)
{Throw_Illegal_State = enabled; return this;}

/**	Test if IllegalStateException will be thrown when an invalid
	JP2 box or codestream segment structure or contents is encountered.
<p>
	@return	true if IllegalStateException will be thrown when an invalid
		JP2 box or codestream segment structure or contents is encountered;
		false otherwise.
	@see	#Throw_Illegal_State(boolean)
*/
public boolean Throw_Illegal_State ()
{return Throw_Illegal_State;}

/*==============================================================================
	Manipulators
*/
/**	Get the next long data value from the Source.
<p>
	The current {@link #Stream_Position() stream position} is updated.
<p>
	@param	description	A String describing the value being obtained.
	@return	The next long data value from the Source.
	@throws	IOException	If there was a problem reading from the Source.
*/
protected long Read_long
	(
	String	description
	)
	throws IOException
{
try
	{
	long
		datum = Image_Input_Stream.readLong ();
	Stream_Position += 8;
	return datum;
	}
catch (IOException exception)
	{throw IO_Error ("Unable to read " + description + " value.",
		exception);}
}

/**	Get the next int data value from the Source.
<p>
	The current {@link #Stream_Position() stream position} is updated.
<p>
	@param	description	A String describing the value being obtained.
	@return	The next int data value from the Source.
	@throws	IOException	If there was a problem reading from the Source.
*/
protected int Read_int
	(
	String	description
	)
	throws IOException
{
try
	{
	int
		datum = Image_Input_Stream.readInt ();
	Stream_Position += 4;
	return datum;
	}
catch (IOException exception)
	{throw IO_Error ("Unable to read " + description + " value.",
		exception);}
}

/**	Get the next int data value from the Source as an unsigned integer.
<p>
	The next {@link #Read_int(String) int value read from the Source} is
	cast as a long value but without sign extension.
<p>
	The current {@link #Stream_Position() stream position} is updated.
<p>
	@param	description	A String describing the value being obtained.
	@return	A long value of the next int from the Source as an unsigned
		integer value.
	@throws	IOException	If there was a problem reading from the Source.
*/
protected long Read_Unsigned_int
	(
	String	description
	)
	throws IOException
{return (long)Read_int (description) & 0xFFFFFFFF;}

/**	Get the next short data value from the Source.
<p>
	The current {@link #Stream_Position() stream position} is updated.
<p>
	@param	description	A String describing the value being obtained.
	@return	An int value of the next short from the Source as a signed
		integer value.
	@throws	IOException	If there was a problem reading from the Source.
*/
protected int Read_short
	(
	String	description
	)
	throws IOException
{
try
	{
	int
		datum = Image_Input_Stream.readShort ();
	Stream_Position += 2;
	return datum;
	}
catch (IOException exception)
	{throw IO_Error ("Unable to read " + description + " value.",
		exception);}
}

/**	Get the next short data value from the Source as an unsigned integer.
<p>
	The next {@link #Read_short(String) short value read from the Source} is
	cast as an int value but without sign extension.
<p>
	The current {@link #Stream_Position() stream position} is updated.
<p>
	@param	description	A String describing the value being obtained.
	@return	An int value of the next short from the Source as an unsigned
		integer value.
	@throws	IOException	If there was a problem reading from the Source.
*/
protected int Read_Unsigned_short
	(
	String	description
	)
	throws IOException
{return (int)Read_short (description) & 0xFFFF;}

/**	Get the next byte data value from the Source.
<p>
	The current {@link #Stream_Position() stream position} is updated.
<p>
	@param	description	A String describing the value being obtained.
	@return	The next byte data value, as an int with sign extension from
		the byte, from the Source.
	@throws	IOException	If there was a problem reading from the Source.
*/
protected int Read_byte
	(
	String	description
	)
	throws IOException
{
try
	{
	int
		datum = Image_Input_Stream.readByte ();
	++Stream_Position;
	return datum;
	}
catch (IOException exception)
	{throw IO_Error ("Unable to read " + description + " value.",
		exception);}
}

/**	Get the next byte data value from the Source as an unsigned integer.
<p>
	The current {@link #Stream_Position() stream position} is updated.
<p>
	@param	description	A String describing the value being obtained.
	@return	An int value of the next byte from the Source as an unsigned
		integer value.
	@throws	IOException	If there was a problem reading from the Source.
*/
protected int Read_Unsigned_byte
	(
	String	description
	)
	throws IOException
{return Read_byte (description) & 0xFF;}

/**	Get a value of some bits precision from the Source.
<p>
	If the number of bits is negative the most significant bit of the
	value is treated as a sign bit (extended in the returned value) and
	the actual precision of the value is abs (bits). The Source stream is
	read starting at the current {@link #Stream_Position() stream
	position}. Bytes are read from the stream starting with the most
	significant byte of the value with the most significant bits right
	justified in this first byte; the least significant bit is presumed
	to be the least significant bit of the last byte read. The bytes read
	are concatenated to form the value that is returned.
<p>
	The current {@link #Stream_Position() stream position} is updated.
<p>
	@param	description	A String describing the value being obtained.
	@param	bits	The precision of the value is abs (bits). Negative
		bits means the value is signed; i.e. the most significant bit is
		the sign bit.
	@return	The value obtained as a long. Thus a maximum of 64 bits may
		be obtained. As a special case, if bits is 0, 0 is returned.
	@throws	IOException	If there was a problem reading the value bytes from
		the Source stream.
	@throws ArithmeticException	If abs (bits) > 64.
*/
protected long Read_Value
	(
	String	description,
	int		bits
	)
	throws IOException
{
long
	value = 0;
if (bits != 0)
	{
	boolean
		signed_value = false;
	if (bits < 0)
		{
		bits = -bits;
		if (bits < 64 &&	//	64-bit value does not need sign extension.
			bits > 1)		//	1-bit value is always unsigned.
			signed_value = true;
		}
	if (bits > 64)
		throw new ArithmeticException (ID + '\n'
			+ "Unable to Read Value of " + bits + " bits.");

	//	Convert bits to bytes.
	int
		bytes = bits >> 3;
	if ((bits % 8) != 0)
		bytes++;

	//	Move the data bytes into the local variable.
	while (bytes != 0)
		{
		value <<= 8;
		value |= Read_Unsigned_byte (description + " byte " + bytes);
		--bytes;
		}

	if (signed_value &&
		(value & (1 << (bits - 1))) != 0)
		{
		//	Sign extension.
		long
			mask = 0;
		while (bits != 0)
			{
			mask <<= 1;
			mask  |= 1;
			--bits;
			}
		value |= ~mask;
		}
	}
return value;
}

/**	Get a String from the Source.
<p>
	Starting at the current {@link #Stream_Position() stream position}
	bytes are read up to, but not including, the end position. Each byte
	read is concatenated as a character in the resultant String with
	unprintable characters being expanded into printable escape
	sequences. However, if the last character is null it is not
	included in the String.
<p>
	The current {@link #Stream_Position() stream position} is updated.
<p>
	@param	description	A String describing the value being obtained.
	@param	end_position	The end position in the Source after the
		last byte to be read.
	@return	A String representing the bytes read from the source.
	@throws	IOException	If there was a problem reading from the Source.
*/
protected String Read_String
	(
	String	description,
	long	end_position
	)
	throws IOException
{
String
	describe = description + " string";
char
	character;
String_Buffer
	string = new String_Buffer ();
while (Stream_Position < end_position)
	{
	if ((character = (char)Read_Unsigned_byte (describe)) == 0)
		break;
	if (Stream_Position < end_position ||
		character != '\0')
		string.append (character);
	}
Stream_Position (end_position);
//	Escape any non-printable characters.
string.special_to_escape ();
return string.toString ();
}

/**	Set the stream position of the information Source.
<p>
	If the specified position is the same as the current position of
	the Source, nothing is done. Otherwise the current position of
	the Source is changed to the specified position.
<p>
	@param	position	The position, as a byte offset from the beginning
		of the file, to which to reposition the image Source stream. This
		value must be a non-negative value less than the size of the
		Source file, otherwise an IOException will be thrown.
	@throws	IOException	If the Source stream could not be repositioned. An
		attempt will be made to reset the Source stream to return it to the
		last marked position before the exception is thrown.
*/
protected void Stream_Position
	(
	long	position
	)
	throws IOException
{
if (Stream_Position != position)
	{
	Exception
		exception = null;
	try
		{
		Image_Input_Stream.seek (position);
		Stream_Position = position;
		}
	catch (IndexOutOfBoundsException except) {exception = except;}
	catch (IOException except) {exception = except;}
	if (exception != null)
		{
		String
			message =
				"Could not reposition the Source to position " + position + '.';
		try {Image_Input_Stream.reset ();}
		catch (IOException e)
			{
			message += "\n"
				+ exception.getMessage () + '\n'
				+ "Could not reset the previous position.";
			exception = e;
			}
		throw IO_Error (message, exception);
		}
	}
}

/**	Get the current source stream position.
<p>
	@return	The current source stream position as a byte offset from the
		beginning of the file.
*/
public long Stream_Position ()
{return Stream_Position;}

/**	Generate an IOException containing a message.
<p>
	If the provided message does not already contain the class {@link
	#ID} the ID and a NL character is prepended to the message (or the ID
	becomes the message if it is null). If a non-null exception is
	provided, the message from the exception is appended, after a NL
	character, to the message used in constructing the IOException.
<p>
	If the exception argument is an instance of EOFException the returned
	IOException will also be an EOFException. If the exception argument
	is non-null and contains a cause, an attempt is made to apply the
	cause to the new IOException (failure to apply the cause is ignored).
<p>
	@param	message	The message String to be included in the IOException.
	@return	An IOException.
	@see	#IOException(String, Exception)
*/
private IOException IO_Error
	(
	String		message,
	Exception	exception
	)
{
if (message == null)
	message = ID;
else if (message.indexOf (ID) < 0)
	message = ID + '\n' + message;
if (exception != null)
	message += "\n" + exception.getMessage ();
IOException
	except;
if (exception instanceof EOFException)
	except = new EOFException (message);
else
	except = new IOException (message);
if (exception != null)
	{
	Throwable
		cause = exception.getCause ();
	if (cause != null)
		{
		try {except.initCause (cause);}
		catch (IllegalStateException e) {}
		}
	}
return except;
}

/**	Generate an IOException containing a message.
<p>
	@param	message	The message String to be included in the IOException.
	@return	An IOException.
	@see	#IOException(String, Exception)
*/
private IOException IO_Error
	(
	String		message
	)
{return IO_Error (message, null);}

/*==============================================================================
	Parameters
*/
/**	Add data position Parameters to a Parameter Group.
<p>
	A {@link #DATA_POSITION_PARAMETER} is added to the group with the
	current {@link #Stream_Position() stream position} as its value. A
	{@link #DATA_LENGTH_PARAMETER} is also added to the group with the
	number of bytes between the current stream position (inclusive) and
	the end position (exclusive) as its value.
<p>
	<b>N.B.</b>: If {@link #Use_Data_Position(boolean) using data
	position parameters} is disabled {@link
	#Data_Offset_Parameters(Parameter, long, long) data offset
	parameters} will be used instead for the same values.
<p>
	@param	group	The Parameter Group (Aggregate) to receive the data
		location parameters.
	@param	end_position	The end position in the Source of the data as
		a byte offset from the beginning of the file to the byte
		immediately following the last byte of the data segement. If this
		value is not greater than zero no {@link #DATA_LENGTH_PARAMETER}
		is included.
	@throws	IllegalArgumentException	If the current {@link
		#Stream_Position() stream position} is greater than the data end
		position.
	@see	#Data_Offset_Parameters(Parameter, long, long)
*/
protected void Data_Position_Parameters
	(
	Parameter	group,
	long		end_position
	)
	throws IllegalArgumentException
{
if (end_position > 0 &&
	Stream_Position > end_position)
	throw new IllegalArgumentException (ID + '\n'
		+ "Invalid " + group.Name () + " data position parameter values -\n"
		+ "  start = " + Stream_Position + '\n'
		+ "    end = " + end_position);
if (Use_Data_Position)
	{
	try
		{
		group
			.Add (new Parameter (DATA_POSITION_PARAMETER)
				.Value (new Value (Stream_Position)
				.Units ("byte offset")));
		if (end_position > 0)
			group
				.Add (new Parameter (DATA_LENGTH_PARAMETER)
					.Value (new Value (end_position - Stream_Position)
					.Units ("bytes")));
		}
	catch (PVL_Exception exception) {}
	}
else
	Data_Offset_Parameters (group, Stream_Position, end_position);
}

/**	Add data offset Parameters to a Parameter Group.
<p>
	A {@link #DATA_OFFSET_PARAMETER} is added to the group followed by a
	a {@link #DATA_LENGTH_PARAMETER}.
<p>
	@param	group	The Parameter Group (Aggregate) to receive the data
		offset parameters.
	@param	start_offset	The offset (in bytes relative to 0) of the
		data location - excluding segment marker and length elements - in
		the enclosing object.
	@param	end_offset	The offset of the end (byte immediately following
		the last data byte) of the data block. If this value is not greater
		than zero no {@link #DATA_LENGTH_PARAMETER} is included.
	@throws	IllegalArgumentException	If the data start offser is
		greater than the end offset. #Stream_Position() stream position}
		is greater than the data end position.
	@see	#Data_Position_Parameters(Parameter, long)
*/
protected void Data_Offset_Parameters
	(
	Parameter	group,
	long		start_offset,
	long		end_offset
	)
	throws IllegalArgumentException
{
if (end_offset > 0 &&
	start_offset > end_offset)
	throw new IllegalArgumentException (ID + '\n'
		+ "Invalild " + group.Name () + " data offset parameter values -\n"
		+ "  start = " + start_offset + '\n'
		+ "    end = " + end_offset);
try
	{
	group
		.Add (new Parameter (DATA_OFFSET_PARAMETER)
			.Value (new Value (start_offset).Units ("byte offset")));
	if (end_offset > 0)
		group
			.Add (new Parameter (DATA_LENGTH_PARAMETER)
				.Value (new Value (end_offset - start_offset).Units ("bytes")));
	}
catch (PVL_Exception exception) {}
}

/**	Generate a {@link #VALUE_BITS_PARAMETER} from the source.
<p>
	Signed byte values are read from the Source starting at the current
	{@link #Stream_Position() stream position} up to, but not including,
	the end position. The value is incremented by 1. If the byte value
	read is negative only the least significant 7 bits are valid; the
	resultant value is the negative of the value of these bits. A
	negative value signals that the values whose bit precision is the
	absolute of the value is a signed value. A value of zero signals that
	the actual precision values are indictated elsewhere.
<p>
	Each value read is added to an Array Value that is assigned to a
	Parameter named {@link #VALUE_BITS_PARAMETER} which is what is
	returned.
<p>
	@param	description	A String describing the values being obtained.
	@param	end_position	The end position in the Source of the values
		as a byte offset from the beginning of the file to the byte
		immediately following the last byte to be read.
	@return	An Assignment Parameter named {@link #VALUE_BITS_PARAMETER}
		that has an Array Value with the adjust values read from the Source.
	@throws	IOException	If there was a problem reading from the Source.
*/
protected Parameter Value_Bits_Parameter
	(
	String		description,
	long		end_position
	)
	throws IOException
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> JPEG2000_Info.Value_Bits_Parameter: " + description + '\n'
		+"     From stream position " + Stream_Position + '\n'
		+"    Until stream position " + end_position);
String
	describe = description + ' ' + VALUE_BITS_PARAMETER;
Parameter
	parameter = null;
try {
Value
	list = new Value ().Type (Value.SEQUENCE);
int
	datum;
while (Stream_Position < end_position)
	{
	datum = Read_byte (describe) + 1;
	if (datum < 0)
		datum = -(datum & 0x7F);
	list.Add (new Value (datum));
	}
parameter = new Parameter (VALUE_BITS_PARAMETER)
	.Comments
		("\nNegative bits indicate signed values of abs (bits);\n"
		+"  Zero bits indicate variable number of bits.") 
	.Value (list);
}
catch (PVL_Exception exception) {}
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("<<< JPEG2000_Info.Value_Bits_Parameter: "
			+ parameter.Description ());
return parameter;
}

/**	Generate a {@link #VALUE_BITS_PARAMETER} from an Array of raw Values.
<p>
	The datum of each Value in the array is assumed to be a raw bit
	precision value. This value is incremented by 1. If the value is
	negative the least significant 7 bits are taken and then negated to
	signal that the image values are signed. The image value precision of
	each component is the absolute value of the corresponding entry in
	the array.
<p>
	The resultant Array is assigned to a Parameter named {@link
	#VALUE_BITS_PARAMETER} which is what is returned.
<p>
	@param	array	A Array Value containing raw bit precision values.
	@return	An Assignment Parameter named {@link #VALUE_BITS_PARAMETER}
		that has a Value that is an Array of the adjusted array values.
*/
protected static Parameter Value_Bits_Parameter
	(
	Value		array
	)
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> JPEG2000_Info.Value_Bits_Parameter:\n"
		+"     From Value " + array.Description ());
Parameter
	parameter = null;
try {
Value
	value;
long
	datum;
for (int
		index = 0;
		index < array.getChildCount ();
		index++)
	{
	value = array.Get (index);
	datum = value.long_Data () + 1;
	if (datum < 0)
		datum = -(datum & 0x7F);
	value.Data (datum);
	}
parameter = new Parameter (VALUE_BITS_PARAMETER)
	.Comments
		("\nNegative bits indicate signed values of abs (bits);\n"
		+"  Zero bits indicate variable number of bits.") 
	.Value (array);
}
catch (PVL_Exception exception) {}

if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("<<< JPEG2000_Info.Value_Bits_Parameter: "
			+ parameter.Description ());
return parameter;
}

/**	Get the Array Value of a Parameter as an int array.
<p>
	@param	parameter_name	The name of the parameter to use. The first
		Parameter found with this name, which may be a pathname, is used.
	@return	An int array.
	@see	#int_Array(Parameter)
*/
public int[] int_Array
	(
	String	parameter_name
	)
{return int_Array (Find (parameter_name));}

/**	Get the Array Value of a Parameter as an int array.
<p>
	@param	parameter	The Parameter from which to obtain Array Values.
	@return	An int array.
	@see	#int_Array(Value)
*/
public static int[] int_Array
	(
	Parameter	parameter
	)
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> JP2_Info.int_Array (Parameter): " +
			((parameter == null) ? "null" : parameter.Path_Name ()));
int[]
	values = new int[0];
if (parameter != null &&
	parameter.Is_Assignment ())
	{
	try {values = int_Array (parameter.Value ());}
	catch (PVL_Exception exception) {}
	}
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("<<< JP2_Info.int_Array (Parameter)");
return values;
}

/**	Convert an Array Value to an int array.
<p>
	Each Integer Value in the Array Value is used to set the respective
	value of an int array. Any Value in the Array that is not an Integer
	is ignored; Sub-Array Values are not entered.
<p>
	@param	value	The Array Value to convert.
	@return	An int array. <b>N.B.</b>: The number of elements in the
		array may be less than the number of Values in the Parameter's
		Array if it contains non-Integer Values. If the Parameter is not
		an Assignment or its Value is not an Array an empty array will be
		returned.
*/
public static int[] int_Array
	(
	Value	value
	)
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> JP2_Info.int_Array (Value)");
int[]
	values = null;
try {
Value
	list;
if (value != null &&
	value.Is_Array ())
	{
	int
		index,
		count = 0,
		total_entries = value.getChildCount ();
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println
			("    total_entries = " + total_entries);
	values = new int[total_entries];
	Value
		entry;
	for (index = 0;
		 index < total_entries;
		 index++)
		{
		entry = value.Get (index);
		if (entry.Is_Integer ())
			values[count++] = (int)entry.long_Data ();
		}
	if (count < index)
		{
		//	Contract the int array.
		int[]
			int_array = new int[count];
		for (index = 0;
			 index < count;
			 index++)
			int_array[index] = values[index];
		values = int_array;
		}
	}
else
	values = new int[0];
}
catch (PVL_Exception exception) {}
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("<<< JP2_Info.int_Array: " + values);
return values;
}


}
