/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.aspect.advice;

import static jp.sourceforge.mergedoc.pleiades.resource.FileNames.*;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import jp.sourceforge.mergedoc.pleiades.aspect.Analyses;
import jp.sourceforge.mergedoc.pleiades.aspect.advice.JointPoint.EditPoint;
import jp.sourceforge.mergedoc.pleiades.log.Logger;
import jp.sourceforge.mergedoc.pleiades.resource.Files;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;

/**
 * Pleiades 構成クラスです。<br>
 * ジョイント・ポイントとポイント・カットの関連付けなどを保持します。
 * <p>
 * @author cypher256
 */
public class PleiadesConfig {

	/** ロガー */
	private static final Logger log = Logger.getLogger(PleiadesConfig.class);

	/** このクラスのインスタンス */
	private static final PleiadesConfig singleton = new PleiadesConfig();

	/**
	 * Pleiades 構成を取得します。
	 * <p>
	 * @return Pleiades 構成
	 */
	public static PleiadesConfig getInstance() {
		return singleton;
	}

	//-------------------------------------------------------------------------

	/** Pleiades 構成ファイルの内容を取得・構築するアセンブラ */
    private PleiadesConfigAssembler pleiadesConfigAssembler = new PleiadesConfigAssembler();

	/** call ウィービング対象となるクラス・メソッドセット */
	private Set<String> callNameSet = new HashSet<String>();

	/** call ウィービング対象となるクラス指定なし前方一致セット */
	private Set<String> callClassNameStartsSet = new HashSet<String>();

	/** call ウィービング対象となるクラス指定のみ後方一致セット */
	private Set<String> callClassNameEndsSet = new HashSet<String>();

	/**
	 * アスペクト・マッピングを構築します。
	 */
	private PleiadesConfig() {

		long start = System.nanoTime();

	    File configFile = Files.getFile(PLEIADES_CONFIG_XML);
		try {
		    if (!configFile.exists()) {
		    	throw new FileNotFoundException(configFile.getPath());
		    }
		    SAXParser parser = SAXParserFactory.newInstance().newSAXParser();
		    parser.parse(configFile, pleiadesConfigAssembler);
		    Map<JointPoint, PointCut> jointMap = pleiadesConfigAssembler.jointMap;

		    for (JointPoint jointPoint : jointMap.keySet()) {
				if (jointPoint.getEditPoint() == EditPoint.CALL) {

					String className = jointPoint.getClassName();
					String methodName = jointPoint.getMethodName();

					callNameSet.add(className + "#" + methodName);

					if (methodName == null) {

						if (className.equals(className.toLowerCase())) {
							callClassNameStartsSet.add(className);
						} else if (!className.contains(".")) {
							callClassNameEndsSet.add(className);
						}
					}
				}
			}
			log.info("Pleiades 構成ファイルをロードしました。");

		} catch (Exception e) {

			String msg = "Pleiades 構成ファイルのロードに失敗しました。";
			log.fatal(msg);
			throw new IllegalStateException(msg, e);
		}

		Analyses.end(PleiadesConfig.class, "<init>", start);
	}

	/**
	 * 呼び出し先クラスとして AOP が定義されているか判定します。
	 * <p>
	 * @param className 呼び出し先クラス名
	 * @param methodName 呼び出し先メソッド名
	 * @return 定義されている場合は true
	 */
	public boolean containsCall(String className, String methodName) {

		long start = System.nanoTime();
		try {

			if (callNameSet.contains(className + "#" + methodName)) {
				return true;
			}
			if (getSubstringMatch(className) != null) {
				return true;
			}
			return false;

		} finally {
			Analyses.end(PleiadesConfig.class, "containsCall", start);
		}
	}

	/**
	 * 部分一致クラス名文字列を取得します。
	 * <p>
	 * @param className クラス名
	 * @return 部分一致クラス名文字列
	 */
	private String getSubstringMatch(String className) {

		long start = System.nanoTime();
		try {

			// クラス名前方一致 (パッケージ名前方部分)
			for (String starts : callClassNameStartsSet) {
				if (className.startsWith(starts)) {
					return starts;
				}
			}
			// クラス名後方一致 (クラス名後方部分)
			for (String ends : callClassNameEndsSet) {
				if (className.startsWith(ends)) {
					return ends;
				}
			}
			return null;

		} finally {
			Analyses.end(PleiadesConfig.class, "getSubstringMatch", start);
		}
	}

	/**
	 * ポイント・カットを取得します。
	 * <p>
	 * @param jointPoint ジョイント・ポイント
	 * @return ポイント・カット
	 */
	public PointCut getPointCut(JointPoint jointPoint) {

	    Map<JointPoint, PointCut> jointMap = pleiadesConfigAssembler.jointMap;
		PointCut pointCut = jointMap.get(jointPoint);
		if (pointCut != null) {
			return pointCut;
		}

		JointPoint key = new JointPoint(jointPoint);
		key.setDescriptor(null);
		pointCut = jointMap.get(key);
		if (pointCut != null) {
			return pointCut;
		}

		key.setMethodName(null);
		pointCut = jointMap.get(key);
		if (pointCut != null) {
			return pointCut;
		}

		if (key.getEditPoint() == EditPoint.CALL) {

			String classNameParts = getSubstringMatch(key.getClassName());
			if (classNameParts != null) {

				key.setClassName(classNameParts);
				pointCut = jointMap.get(key);
				if (pointCut == null) {
					throw new IllegalStateException(
						"ロジック不正。ポイント・カットが見つかりません。");
				}
				return pointCut;
			}
		}
		return null;
	}

	/**
	 * プロパティーを取得します。
	 * <p>
	 * @param name プロパティー名
	 * @return プロパティー
	 */
	public String getProperty(String name) {
		return pleiadesConfigAssembler.propertyMap.get(name);
	}
	
	/**
	 * AOP 除外クラスか判定します。
	 * @param className クラス名
	 * @return 除外する場合は true
	 */
	public boolean isExcludePackage(String className) {
		for (String packageName : pleiadesConfigAssembler.excludePackageSet) {
			if (className.startsWith(packageName)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * このオブジェクトの文字列表現を取得します。
	 */
	public String toString() {

		long start = System.nanoTime();
	    Map<JointPoint, PointCut> jointMap = pleiadesConfigAssembler.jointMap;
		try {
			return new ToStringBuilder(this, ToStringStyle.MULTI_LINE_STYLE)
				.append(jointMap)
				.toString();

		} finally {
			Analyses.end(PleiadesConfig.class, "toString", start);
		}
	}
}
