##############################################################################
#    Copyright (c) 2012-2016 Russell V. Lenth                                #
#                                                                            #
#    This file is part of the lsmeans package for R (*lsmeans*)              #
#                                                                            #
#    *lsmeans* is free software: you can redistribute it and/or modify       #
#    it under the terms of the GNU General Public License as published by    #
#    the Free Software Foundation, either version 2 of the License, or       #
#    (at your option) any later version.                                     #
#                                                                            #
#    *lsmeans* is distributed in the hope that it will be useful,            #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of          #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           #
#    GNU General Public License for more details.                            #
#                                                                            #
#    You should have received a copy of the GNU General Public License       #
#    along with R and *lsmeans*.  If not, see                                #
#    <https://www.r-project.org/Licenses/> and/or                            #
#    <http://www.gnu.org/licenses/>.                                         #
##############################################################################

# Runs the function multicompLetters from the multcompView package
# returns an error if not installed
.mcletters = function(..., Letters=c("1234567890",LETTERS,letters), reversed=FALSE) {
    if(!requireNamespace("multcompView", quietly = TRUE)) {
        stop("The 'multcompView' package must be installed to use cld methods")
        return (list(monospacedLetters = "?"))
    }
    
    # Expand strings to individual letters
    Letters = as.character(unlist(sapply(Letters, function(stg) {
        sapply(seq_len(nchar(stg)), function(i) substr(stg, i, i))
    })))
    
    result = multcompView::multcompLetters(..., Letters=Letters, reversed=reversed)
    if (is.null(result$monospacedLetters))
        result$monospacedLetters = result$Letters
    result
}

# S3 method for ref.grid
cld.ref.grid = function(object, details=FALSE, sort=TRUE, 
                    by, alpha=.05, 
                    Letters = c("1234567890",LETTERS,letters), 
                    reversed=FALSE, ...) {
    lsmtbl = summary(object, ...)
    if(missing(by)) 
        by = object@misc$by.vars
    if (sort) {
        args = list()
        for (nm in by) args[[nm]] = lsmtbl[[nm]]
        args$.lsm. = lsmtbl[[attr(lsmtbl, "estName")]]
        ord = do.call("order", args)
        lsmtbl = lsmtbl[ord, ]
        object@grid = object@grid[ord, , drop=FALSE]
        object@linfct = object@linfct[ord, , drop = FALSE]
    }
    attr(lsmtbl, "by.vars") = by
    object@misc$by.vars = by
    
    prwise = contrast(object, "revpairwise", by=by)    
    pwtbl = test(prwise, ...)
    
    p.boo = (pwtbl$p.value < alpha)
    if(is.null(by)) {
        by.rows = list(seq_len(nrow(pwtbl)))
        by.out = list(seq_len(nrow(lsmtbl)))
    }
    else {
        by.rows = .find.by.rows(pwtbl, by)
        by.out = .find.by.rows(lsmtbl, by)
    }
    # Create comps matrix reflecting order generated by pairwise.lsmc
    icol = jcol = numeric(0)
    # create fake row indexes in revpairwise order for use by .mcletters
    k = length(by.out[[1]])
    for (i in 2:k) {
        icol = c(icol, seq_len(i-1))
        jcol = c(jcol, rep(i, i-1))
    }
    na.p = which(is.na(p.boo))
    # Take care of non-est cases. This is surprisingly complicated,
    # because it's possible we have some lsmeans that are non-est
    # but comparisons are est'ble. So cases to exclude must be missing in
    # the table of means, AND appar somewhere in the indexes of NA p values
    # All that said, it still messes up because I didn't track the indexes correctly
    # excl.rows = intersect(which(is.na(lsmtbl$SE)), union(icol[na.p], jcol[na.p]))
    # So I'll just go with which est's are missing
    excl.rows = which(is.na(lsmtbl$SE))
    p.boo[na.p] = FALSE
    
    labs = paste(icol,jcol,sep="-")
    ltrs = rep("", nrow(lsmtbl))
    for (i in seq_len(length(by.rows))) {
        pb = p.boo[by.rows[[i]]]
        names(pb) = labs
        mcl = .mcletters(pb, Letters = Letters, reversed = reversed)$monospacedLetters
        ltrs[by.out[[i]]] = paste(" ", mcl, sep="")
    }
    # any missing estimates get blanks...
    ltrs[excl.rows] = ""
    
    lsmtbl[[".group"]] = ltrs
    if(sort && reversed) for (i in seq_len(length(by.out))) {
        r = by.out[[i]]
        lsmtbl[r, ] = lsmtbl[rev(r), ]
    }
    
    attr(lsmtbl, "mesg") = c(attr(lsmtbl,"mesg"), attr(pwtbl, "mesg"), 
                             paste("significance level used: alpha =", alpha))
        
    if (details)
        list(lsmeans = lsmtbl, comparisons = pwtbl)
    else
        lsmtbl
}
