% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cluster_meta.R
\name{cluster_meta}
\alias{cluster_meta}
\title{Metaclustering}
\usage{
cluster_meta(list_of_clusters, rownames = NULL, ...)
}
\arguments{
\item{list_of_clusters}{A list of vectors with the clustering assignments from various methods.}

\item{rownames}{An optional vector of row.names for the matrix.}

\item{...}{Currently not used.}
}
\value{
A matrix containing all the pairwise (between each observation)
probabilities of being clustered together by the methods.
}
\description{
One of the core "issue" of statistical clustering is that, in many cases,
different methods will give different results. The \strong{metaclustering} approach
proposed by \emph{easystats} (that finds echoes in \emph{consensus clustering}; see Monti
et al., 2003) consists of treating the unique clustering solutions as a ensemble,
from which we can derive a probability matrix. This matrix contains, for each
pair of observations, the probability of being in the same cluster. For instance,
if the 6th and the 9th row of a dataframe has been assigned to a similar cluster
by 5 our of 10 clustering methods, then its probability of being grouped together
is 0.5.
}
\details{
Metaclustering is based on the hypothesis that, as each clustering algorithm
embodies a different prism by which it sees the data, running an infinite
amount of algorithms would result in the emergence of the "true" clusters.
As the number of algorithms and parameters is finite, the probabilistic
perspective is a useful proxy. This method is interesting where there is no
obvious reasons to prefer one over another clustering method, as well as to
investigate how robust some clusters are under different algorithms.

This metaclustering probability matrix can be transformed into a dissimilarity
matrix (such as the one produced by \code{dist()}) and submitted for instance to
hierarchical clustering (\code{hclust()}). See the example below.
}
\examples{
\donttest{
data <- iris[1:4]

rez1 <- cluster_analysis(data, n = 2, method = "kmeans")
rez2 <- cluster_analysis(data, n = 3, method = "kmeans")
rez3 <- cluster_analysis(data, n = 6, method = "kmeans")

list_of_clusters <- list(rez1, rez2, rez3)

m <- cluster_meta(list_of_clusters)

# Visualize matrix without reordering
heatmap(m, Rowv = NA, Colv = NA, scale = "none") # Without reordering
# Reordered heatmap
heatmap(m, scale = "none")

# Extract 3 clusters
predict(m, n = 3)

# Convert to dissimilarity
d <- as.dist(abs(m - 1))
model <- hclust(d)
plot(model, hang = -1)
}
}
