-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

--------------------------------------------------------------------------------
--  Directory_Operations
--
--  Purpose:
--   Provides a SPARK-friendly binding onto the facilities of
--   GNAT.Directory_Operations
--
--  Clients:
--   SPARKMake - used to scan directories and match files against
--   regular expressions.
--------------------------------------------------------------------------------

with E_Strings;
with RegularExpression;
with StringList;

--# inherit E_Strings,
--#         RegularExpression,
--#         StringList;

package Directory_Operations is

   function Directory_Separator return Character;

   function Current_Directory return  E_Strings.T;

   -- Looks for files InDirectory Matching the filter.
   -- Subdirectories are searched in Recursively is true.
   -- Setting Matching to TrueFilter will return all files
   --
   -- The StringList returned is in lexicographic order - the
   -- files are sorted according to the ordering defined by
   -- Standard."<=" for type String.  This ensures predictable
   -- behaviour on platforms where the order returned by
   -- GNAT.Directory_Operations.Read is not predictable.
   function Find_Files
     (Matching     : RegularExpression.Object;
      In_Directory : E_Strings.T;
      Recursively  : Boolean)
     return         StringList.Object;

   function File_Extension (Path : E_Strings.T) return E_Strings.T;
   --
   --  Return the file extension. This is the string after the last dot
   --  character in File_Name (Path). It returns the empty string if no
   --  extension is found. The returned value does contains the file
   --  extension separator (dot character).

   procedure Set_Extension (Path : in out E_Strings.T;
                            Ext  : in     E_Strings.T);
   --# derives Path from *,
   --#                   Ext;
   --
   -- Sets the file extension to Ext. If the file has and extension then it is replaced.
   -- If the file has no extension then it is added.

   function Filename (Path : E_Strings.T) return E_Strings.T;
   --  Returns the file name and the file extension if present. It removes all
   --  path information.

   function Is_File (Path : E_Strings.T) return Boolean;
   --
   -- Returns true if the given path is a unique file

   function Is_Directory (Path : E_Strings.T) return Boolean;
   --
   -- Returns true if the given path is a unique directory

   function Normalize_Path_Name (Name      : E_Strings.T;
                                 Directory : E_Strings.T) return E_Strings.T;
   --
   -- Resolves relative files names relative to the given directory.

   function Relative_Name (Of_This_File_Or_Dir : E_Strings.T;
                           To_This_Dir         : E_Strings.T) return E_Strings.T;
   --
   -- Returns the relative file or directory name.

end Directory_Operations;
