/*
	SRG - Squid Report Generator
	Report by user
	Copyright 2005 University of Waikato

	This file is part of SRG.

	SRG is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	SRG is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with SRG; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
	USA

	$Id: UserReport.cc 218 2005-05-11 04:05:17Z matt $
	
*/

#include "UserReport.h"
#include "srg.h"
#include "prototypes.h"

UserReport::~UserReport() {

	list<SiteReport*>::const_iterator iter;
	for (iter=sites.begin(); iter != sites.end(); iter++) {
		delete (*iter);
	}

}

void UserReport::process_line(const log_line *line) {

	if (srg.debug)
		fprintf(stderr, "In UserReport::process_line for user '%s'\n",
				mName);
	
	/* Work out what site this line is for */
	char *site = strdup(line->request->site);

	SiteReport *theSite = (SiteReport *)findSite(site);
	if (theSite==NULL) {
		if (srg.debug)
			fprintf(stderr, "Site not found in %s - creating %s\n",
				mName, site);
		/* The site does not exist. Create it. */
		theSite = new SiteReport(site);
		sites.push_back(theSite);
	}
    	
	/* Process the line */
	theSite->process_line(line);

	free(site);
}

void UserReport::updateStats() {
    
    /* Otherwise continue on and update our statistics */
	zeroStats();

	list<SiteReport*>::const_iterator iter;

	for (iter=sites.begin(); iter != sites.end(); iter++) {
		summary_info sitestats = (*iter)->getStats();
		if (srg.minimumConnects>0 && sitestats.connects<srg.minimumConnects)
			continue;
		stats.connects += sitestats.connects;
		stats.bytesTransferred += sitestats.bytesTransferred;
		stats.timeSpent += sitestats.timeSpent;
		stats.hits += sitestats.hits;
		stats.bytesHit += sitestats.bytesHit;
		stats.misses += sitestats.misses;
		stats.bytesMissed += sitestats.bytesMissed;
		stats.deniedHits += sitestats.deniedHits;
	}
}

void UserReport::generate_report(const char *basename) {

	char *t = NULL;
	char *basepath=NULL;
	char *filename=NULL;
    char *rootpath=NULL;
	FILE *outfile=NULL;
	float percentin = 0;
	float percentout = 0;
	
	/* Create the output filename */
	if (!singleUserMode) {
		t = md5this(mName);
		asprintf(&basepath, "%s/%s", basename, t);
		free(t);
		asprintf(&t, "%s/%s", srg.outputDir, basepath);
		if (mkdir(t, 0755) == -1 && errno != EEXIST) {
			fprintf(stderr, "Error (%s) creating directory in " 
					"user report for %s: %s\n", strerror(errno), 
					mName, t);
			exit(1);
		}
		free(t);
		rootpath = strdup("../../");
	} else {
		basepath = strdup(basename);
		rootpath = strdup("../");
	}
	
	if (srg.debug)
		fprintf(stderr, "Creating user report in %s\n", t);
		
	/* Create the index file */
	asprintf(&filename, "%s/%s/%s", srg.outputDir, basepath, 
			srg.indexfname);
	outfile = fopen(filename, "w");
	if(outfile==NULL) {
		fprintf(stderr,"SiteReport: Cannot open output file: %s\n",
				filename);
		free(rootpath);
		exit(1);
	}
	free(filename);

	/* Header & Title */
	html_header(outfile, rootpath);

	fprintf(outfile, "<!-- SRG %s (%s) Generated UserReport -->\n", 
			version, HOME_URL);

	/* misc stats */
	fprintf(outfile, "<center><table cellpadding=2 cellspacing=2>\n");
	fprintf(outfile, "<tr><td class=\"bodyText\">Period:</td><td class="
			"\"bodyText\">&nbsp;%d %s %d",
			localtime(&srg.startTime)->tm_mday, 
			month_names[localtime(&srg.startTime)->tm_mon], 
			localtime(&srg.startTime)->tm_year+1900);
	fprintf(outfile, " - %d %s %d</td></tr>\n", 
			localtime(&srg.endTime)->tm_mday, 
			month_names[localtime(&srg.endTime)->tm_mon], 
			localtime(&srg.endTime)->tm_year+1900);
	if (srg.groupBy > 0) {
		fprintf(outfile, "<tr><td class=\"bodyText\">Group:</td><td "
				"class=\"bodyText\">&nbsp;%s</td></tr>", mName);
	}
	fprintf(outfile, "</table></center>");
        
	/* Notices Row */
	fprintf(outfile, "<div align=\"center\" id=\"srg-message\">"
			"&nbsp;</div>\n");

	/* main table for location list */
	fprintf(outfile, "<center><table cellpadding=4 cellspacing=0 "
			"id=\"srgtable\">"
			"<thead><tr><th>LOCATION</th><th>REQUESTS</th><th>BYTES</th>"
			"<th>BYTES%%</th><th>HIT</th><th>MISS</th>");
	if (srg.showtimes) {
		fprintf(outfile, "<th>TIME%%</th><th>TIME(ms)</th>");
	}
	if (srg.showrates) {
		fprintf(outfile, "<th>RATE (kB/s)</th>");
	}
	fprintf(outfile, "</tr></thead>\n");
    
	/* Totals Row */
	summary_info thissitestats = getStats();
	fprintf(outfile, "<tfoot><tr><th class=\"cellText\">Totals:</th>");
	t = FormatOutput(thissitestats.connects);
	fprintf(outfile, "<th class=\"cellNum\">%s</th>", t);
	free(t);
	t = FormatOutput(thissitestats.bytesTransferred);
	fprintf(outfile, "<th class=\"cellNum\">%s</th>", t);
	free(t);
	fprintf(outfile, "<th class=\"cellNum\">100%%</th>");
	if (thissitestats.bytesTransferred > 0) {
        percentin = ((float)thissitestats.bytesHit/
                (float)thissitestats.bytesTransferred)*100.0;
        percentout = ((float)thissitestats.bytesMissed/
                (float)thissitestats.bytesTransferred)*100.0;    
		fprintf(outfile, "<th align=\"center\" class=\"cellNum\">"
				"%2.2f%%</th><th align=\"center\" class=\""
				"cellNum\">%2.2f%%</th>", percentin, 
				percentout);
	} else {
		fprintf(outfile, "<th align=\"center\" class=\"cellNum\">"
				"-</th><th align=\"center\" class=\""
				"cellNum\">-</th>");
	}
	if (srg.showtimes) {
		t = FormatOutput(thissitestats.timeSpent);
		fprintf(outfile, "<th class=\"cellNum\">100%%</th><th class=\""
				"cellNum\">%s</th>", t);
		free(t);
	}
	if (srg.showrates) {
		if (thissitestats.timeSpent == 0) {
			fprintf(outfile, "<th class=\"cellNum\">-</th>");
		} else {
			fprintf(outfile, "<th class=\"cellNum\">%2.0f</th>",
				(float)thissitestats.bytesTransferred/
				(float)thissitestats.timeSpent);
		}
	}
	fprintf(outfile, "</tr></tfoot>\n");

	summary_info sitestats;
	float timespent = 0;
	float bytespercent = 0;
	int rows = 0;

	/* Initialise the Summary Stats Structure */
	sitestats.connects = 0;
	sitestats.bytesTransferred = 0;
	sitestats.hits = 0;
	sitestats.misses = 0;
	sitestats.bytesHit = 0;
	sitestats.bytesMissed = 0;
	sitestats.timeSpent = 0;
	sitestats.deniedHits = 0;

	sites.sort(LessByBytesTransferred<SiteReport>());

	list<SiteReport*>::const_iterator iter;
	for (iter=sites.begin(); iter != sites.end(); iter++) {

		sitestats = (*iter)->getStats();

		if (srg.minimumConnects>0 && sitestats.connects<srg.minimumConnects)
			continue;

        if (srg.siteStats)
            (*iter)->generate_report(basepath, mName);

		if (sitestats.bytesTransferred>0 && sitestats.connects>0) {
			percentin = ((float)sitestats.bytesHit/
                    (float)sitestats.bytesTransferred)*100.0;
			percentout = ((float)sitestats.bytesMissed/
                    (float)sitestats.bytesTransferred)*100.0;
		} else {
            percentin = 0;
            percentout = 0;
        }
		if (thissitestats.timeSpent == 0) {
			timespent = 100;
		} else {
			timespent = (float)(100*sitestats.timeSpent) / 
					thissitestats.timeSpent;
		}
		if (thissitestats.bytesTransferred == 0) {
			bytespercent = 100;
		} else {
			bytespercent = (float)(100*sitestats.bytesTransferred)
					/ thissitestats.bytesTransferred;
		}
		t = md5this((*iter)->getName());
		fprintf(outfile, "<tr%s><td class=\"bodyText\">", 
                rows%2==0 ? " class=\"highlightRow\"" : "");
        if (srg.siteStats) {
            fprintf(outfile, "<a href=\"%s/%s\">", t, srg.indexfname);
        }
        fprintf(outfile, "%s", (*iter)->getName());
        if (srg.siteStats) {
            fprintf(outfile, "</a>");
        }
        fprintf(outfile, "</td>");
		free(t);
		t = FormatOutput(sitestats.connects);
		fprintf(outfile, "<td class=\"cellNum\">%s</td>", t);
		free(t);
		t = FormatOutput(sitestats.bytesTransferred);
		fprintf(outfile, "<td class=\"cellNum\">%s</td>", t);
		free(t);
		fprintf(outfile, "<td class=\"cellNum\">%2.2f%%</td>", 
				bytespercent);
		if (percentin != -1) {
			fprintf(outfile, "<td class=\"cellNum\">%2.2f%%</td>"
					"<td class=\"cellNum\">%2.2f%%</td>",
					percentin, percentout);
		} else {
			fprintf(outfile, "<td class=\"cellNum\">-</td>"
					"<td class=\"cellNum\">-</td>");
		}
		if (srg.showtimes) {
			t = FormatOutput(sitestats.timeSpent);
			fprintf(outfile, "<td class=\"cellNum\">%2.2f%%</td>"
					"<td class=\"cellNum\">%s</td>",
					timespent, t);
			free(t);
		}
		if (srg.showrates) {
			if (sitestats.timeSpent == 0) {
				fprintf(outfile, "<td class=\"cellNum\">"
						"-</td>");
			} else {
				fprintf(outfile, "<td class=\"cellNum\">"
						"%.2f</td>", 
						(float)sitestats.bytesTransferred/
						(float)sitestats.timeSpent);
			}
		}
		fprintf(outfile, "</tr>\n");
		rows++;
	}

    fprintf(outfile, "\t</table><br>");
      
	if (srg.authenticate) {
		fprintf(outfile, "\n\t<?php } else {\n report_error(\""
				"Could not authenticate user\");\n}?>");
	}


	if (srg.groupBy > 0) {
		fprintf(outfile, "\n\t<p><a href=\"../%s\">Back</a> to groups "
				"page</p></center>\n", srg.indexfname);
	} else {
		fprintf(outfile, "\n\t<p><a href=\"../%s\">Back</a> to dates "
				"page</p></center>\n", srg.indexfname);
	}
	
	/* Finish off the HTML */
	html_footer(outfile, rootpath);
	free(rootpath);
	fclose(outfile);

	/* If no rows were ouput, delete it all */
	if (rows == 0) {
		recurse_unlink(basepath);
	}
	free(basepath);

}

SiteReport *UserReport::findSite(char *site) {

	list<SiteReport*>::const_iterator iter;

	// Iterate through list and compare each element.
	for (iter=sites.begin(); iter != sites.end(); iter++)
	{
		if(strcasecmp((*iter)->getName(), site)==0)
			return (SiteReport *)(*iter);
	}

	return NULL;

}

// Get statistics about this report
summary_info UserReport::getStats() {

	summary_info returnInfo;

	updateStats();

	returnInfo = stats;

	return returnInfo;

}
