# Copyright (c) 2011 Canonical Ltd.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

import os
import shutil
import tempfile
import unittest

import tree
import xdeb


class TestGetControlLines(unittest.TestCase):
    def test_regular(self):
        control = [
            'Source: source\n',
            '\n',
            'Package: package\n',
            ]
        self.assertEqual(control, tree.get_control_lines(control))

    def test_strip_comments(self):
        expected = [
            'Source: source\n',
            '\n',
            'Package: package\n',
            ]

        control = [
            '# 1\n',
            'Source: source\n',
            '# 3\n',
            '\n',
            '# 5\n',
            'Package: package\n',
            '# 7\n',
            ]
        self.assertEqual(expected, tree.get_control_lines(control))

    def test_one_newline_between_each_stanza(self):
        expected = [
            'Source: source\n',
            '\n',
            'Package: package1\n',
            '\n',
            'Package: package2\n',
            ]
        control = [
            'Source: source\n',
            '\n',
            '\n',
            '\n',
            'Package: package1\n',
            '\n',
            '\n',
            '\n',
            'Package: package2\n',
            ]
        self.assertEqual(expected, tree.get_control_lines(control))

    def test_no_leading_blank_lines(self):
        expected = [
            'Source: source\n',
            '\n',
            'Package: package\n',
            ]
        control = [
            '\n',
            '\n',
            'Source: source\n',
            '\n',
            'Package: package\n',
            ]
        self.assertEqual(expected, tree.get_control_lines(control))


def clear_cache():
    tree.srcdir = None
    tree.dirsrc = None
    tree.pkgsrc = None
    tree.srcpkgs = None
    tree.srcrec = None
    tree.pkgrec = None
    tree.provides = None


class TestScanDir(unittest.TestCase):
    tmpdir = None

    def setUp(self):
        clear_cache()

        self.tmpdir = tempfile.mkdtemp()

    def tearDown(self):
        if self.tmpdir:
            shutil.rmtree(self.tmpdir)

    def test_empty_dir(self):
        self.assertFalse(tree.scan_dir(self.tmpdir))

    def test_empty_control(self):
        os.mkdir(os.path.join(self.tmpdir, 'debian'))
        with open(os.path.join(self.tmpdir, 'debian', 'control'), 'w'):
            pass
        self.assertFalse(tree.scan_dir(self.tmpdir))

    def test_no_source_stanza(self):
        os.mkdir(os.path.join(self.tmpdir, 'debian'))
        with open(os.path.join(self.tmpdir, 'debian', 'control'), 'w') as f:
            f.write(
                "Package: package1\n"
                "\n"
                "Package: package2\n")
        self.assertFalse(tree.scan_dir(self.tmpdir))

    def test_regular(self):
        os.mkdir(os.path.join(self.tmpdir, 'debian'))
        with open(os.path.join(self.tmpdir, 'debian', 'control'), 'w') as f:
            f.write(
                "Source: source\n"
                "Source-Field: source-field\n"
                "\n"
                "Package: package1\n"
                "Package-Field: package-field\n"
                "\n"
                "Package: package2\n"
                "Provides: provides\n")
        self.assertTrue(tree.scan_dir(self.tmpdir))
        self.assertEqual({'source': self.tmpdir}, tree.srcdir)
        self.assertEqual({self.tmpdir: 'source'}, tree.dirsrc)
        self.assertEqual(
            {'package1': 'source', 'package2': 'source'},
            tree.pkgsrc)
        self.assertEqual({'source': ['package1', 'package2']}, tree.srcpkgs)
        self.assertEqual(
            {'source': {'Source': 'source', 'Source-Field': 'source-field'}},
            tree.srcrec)
        self.assertEqual(
            {'package1':
                {'Package': 'package1',
                 'Package-Field': 'package-field'},
             'package2':
                {'Package': 'package2',
                 'Provides': 'provides'}},
            tree.pkgrec)
        self.assertEqual({'provides': ['package2']}, tree.provides)


class TestBuildCache(unittest.TestCase):
    options = None
    saved_scan_dir = None
    scan_dir_pathes = []
    tmpdir = None

    def setUp(self):
        clear_cache()

        self.tmpdir = tempfile.mkdtemp()

        _parser, self.options, _args = xdeb.parse_options(args=[])
        self.options.builddirs = [self.tmpdir]

        def mock_scan_dir(path):
            self.scan_dir_pathes.append(path)
            return False
        self.saved_scan_dir = tree.scan_dir
        tree.scan_dir = mock_scan_dir

    def tearDown(self):
        if self.saved_scan_dir:
            tree.scan_dir = self.saved_scan_dir

        if self.tmpdir:
            shutil.rmtree(self.tmpdir)

    def test_empty(self):
        tree.build_cache(self.options)
        self.assertEqual([], self.scan_dir_pathes)

    def test_hacks(self):
        dirs = [
            os.path.join(self.tmpdir, 'source'),
            os.path.join(self.tmpdir, 'source', 'files'),
            os.path.join(self.tmpdir, 'source', 'src'),
            ]
        for d in dirs:
            os.makedirs(d)
        tree.build_cache(self.options)
        self.assertEqual(dirs, self.scan_dir_pathes)
