/*
  Copyright (C) 2000-2007

  Code contributed by Greg Collecutt, Joseph Hope and the xmds-devel team

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmds_field.h 1604 2007-11-18 18:47:23Z paultcochrane $
*/

/*! @file xmds_field.h
  @brief 

  More detailed explanation...
*/

#ifndef XMDS_FIELD_H
#define XMDS_FIELD_H

#include <xmds_element.h>

// *****************************************************************************
// *****************************************************************************
//                              dimension
// *****************************************************************************
// *****************************************************************************

//! Structure containing dimension parameters
struct dimensionStruct {

  XMLString name;        //!< The name of the dimension
  unsigned long lattice; //!< The lattice of the dimension
  domainStruct domain;   //!< The structure of the domain's dimension
};

// *****************************************************************************
// *****************************************************************************
//                              xmdsFieldGeometry
// *****************************************************************************
// *****************************************************************************

//! xmds field geometry class
class xmdsFieldGeometry {

  public :

    //! Clears the dimensions list
    void clear();

  //! Returns the number of dimensions
  unsigned long nDims() const;

  //! Returns the dimension of the field geometry as a dimensionStruct
  const dimensionStruct* dimension(
                                   const unsigned long& index) const;

  //! Sets the dimension of the field geometry
  void setDimension(
                    const unsigned long& index,
                    const dimensionStruct& newDimension);

  //! Adds a dimension to the field geometry
  void addDimension(
                    const dimensionStruct& newDimension);

  //! Gets dimension number
  /*! Why in the crap is this called getDimNumber if it returns boolean?
    And if it actually does return the dimension number then why
    does it return boolean?
  */
  bool getDimNumber(
                    const XMLString& dimName,
                    unsigned long& dimNumber) const;

  //! Returns the full space
  unsigned long fullSpace() const;

  private :

    list<dimensionStruct> myDimensionsList;  //!< The list of the dimensions of the field
};

// *****************************************************************************
// *****************************************************************************
//                              xmdsField
// *****************************************************************************
// *****************************************************************************

//! Enumerator of the different kinds of type a vector can be
enum xmdsVectorType {
  COMPLEX       = 0,
  DOUBLE        = 1
};

//! The different kinds of parallelisation that we have
enum xmdsIsLoopParallelisable {
  PARALLELISE_LOOP = true,
  DO_NOT_PARALLELISE_LOOP = false
};

class xmdsVector;

class xmdsVectorElement;

//! xmds field class
class xmdsField : public xmdsElement {

  public :

    //! Constructor of xmdsField object
    xmdsField(
        const xmdsSimulation *const yourSimulation,
        const bool& yourVerboseMode);

  //! Destructor
  ~xmdsField();

  //! Processes the xmds element
  void processElement(
                      const Element *const yourElement);

  //! Processes the vectors in the vector names list
  void processVectors(
                      const list<XMLString>& vectorNamesList,
                      const unsigned long& space) const;

  //! The output sample count (but doesn't return anything... Why?)
  void outputSampleCount() const;

  //! Determines if needs FFTW plans
  bool needsFFTWPlans() const;

  //! Writes the initialisation calls to the C++ file
  virtual void writeInitialisationCalls(
                                        FILE *const outfile,
                                        const char *const indent) const;

  //! Writes the sample calls to the C++ file
  void writeSampleCalls(
                        FILE *const outfile,
                        const char *const indent) const;

  //! Assigns the active vector pointers
  void assignActiveVectorPointers(
                                  FILE *const outfile,
                                  const char *const tempVectorName) const;

  //! Returns the name of the field(?)
  const XMLString* name() const;

  //! Returns the field geometry
  const xmdsFieldGeometry* geometry() const;

  //! Gets the vector?
  /*!
    Why does this return boolean if it gets the vector?
  */
  bool getVector(
                 const XMLString& vectorName,
                 const xmdsVector*& theVector) const;

  //! Returns the vector names in a list
  void vectorNames(
                   list<XMLString>& vectorNamesList) const;

  //! Converts the vectors to the relevant space
  void vectors2space(
                     FILE *const outfile,
                     const unsigned long& space,
                     const list<XMLString>& vectorNamesList,
                     const char *const indent) const;

  //! Opens the loops
  void openLoops(
                 FILE *const outfile,
                 unsigned long space,
                 const list<XMLString>& vectorNamesList,
     xmdsIsLoopParallelisable isLoopParallelisable = DO_NOT_PARALLELISE_LOOP,
     const list<XMLString>& threadPrivateVariables = list<XMLString>()) const;

  //! Closes the loops
  void closeLoops(
                  FILE *const outfile,
                  unsigned long space,
                  const list<XMLString>& vectorNamesList) const;

  //! Writes the plan creation calls for fftw
  void writePlanCreationCalls(
                              FILE *const outfile,
                              const bool& useFFTWMeasure,
                              const bool& useWisdom) const;

  //! Writes the plan deletion calls for fftw
  void writePlanDeletionCalls(
                              FILE *const outfile) const;

  //! Creats an xmds vector
  xmdsVector* createxmdsVector();

 protected:

  //! Writes defines to outfile
  void writeDefines(
                    FILE *const outfile) const;

  //! Writes global variables to outfile
  void writeGlobals(
                    FILE *const outfile) const;

  //! Writes function prototypes to outfile
  void writePrototypes(
                       FILE *const outfile) const;

  //! Writes simulation routines to outfile
  void writeRoutines(
                     FILE *const outfile) const;

  //! Sets the name of the field
  void setName(
               const XMLString& yourName);

  //! Sets the geometry of the field
  void setGeometry(
                   const xmdsFieldGeometry& yourGeometry);

  private :

    XMLString                 myName;           //!< The name of the field
  xmdsFieldGeometry         myGeometry;       //!< The field geometry
  list<const xmdsVector*>   myVectorsList;    //!< The list of vectors in the field
  list<unsigned long>       mySamplesList;    //!< The list of samples of the field
  mutable bool              myNeedsFFTWPlans; //!< Whether or not the field needs the fftw plans

  //! Creates an xmds vector element
  xmdsVectorElement* createxmdsVectorElement();
};

#endif // XMDS_FIELD_H
