// Copyright (c) .NET Foundation. All rights reserved.
// Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

using System;
using FluentAssertions;
using NuGet.Protocol.Model;
using NuGet.Versioning;
using Xunit;

namespace NuGet.Protocol.Tests
{
    public class PackageVulnerabilityInfoTests
    {

        [Fact]
        public void Constructor_WithNullUri_Throws()
        {
            Assert.Throws<ArgumentNullException>(() => new PackageVulnerabilityInfo(null, PackageVulnerabilitySeverity.Low, VersionRange.Parse("1.0.0")));
        }

        [Fact]
        public void Constructor_WithNullVersionRange_Throws()
        {
            Assert.Throws<ArgumentNullException>(() => new PackageVulnerabilityInfo(new Uri("https://contoso.com/vulnerability1"), PackageVulnerabilitySeverity.Low, null));
        }

        [Fact]
        public void Equals_WithEquivalentObjects_ReturnsTrue()
        {
            var packageVulnerabilityInfo1 = new PackageVulnerabilityInfo(new Uri("https://contoso.com/vulnerability1"), PackageVulnerabilitySeverity.Low, VersionRange.Parse("1.0.0"));
            var packageVulnerabilityInfo2 = new PackageVulnerabilityInfo(new Uri("https://contoso.com/vulnerability1"), PackageVulnerabilitySeverity.Low, VersionRange.Parse("1.0.0"));
            packageVulnerabilityInfo1.Should().Be(packageVulnerabilityInfo2);
        }

        [Theory]
        [InlineData("https://contoso.com/vulnerability1", PackageVulnerabilitySeverity.Low, "1.0.0",
                    "https://contoso.com/vulnerability2", PackageVulnerabilitySeverity.Low, "1.0.0")]
        [InlineData("https://contoso.com/vulnerability1", PackageVulnerabilitySeverity.Low, "1.0.0",
                    "https://contoso.com/vulnerability1", PackageVulnerabilitySeverity.High, "1.0.0")]
        [InlineData("https://contoso.com/vulnerability1", PackageVulnerabilitySeverity.Low, "1.0.0",
                    "https://contoso.com/vulnerability1", PackageVulnerabilitySeverity.Low, "2.0.0")]

        public void Equals_WithVariousNotEquivalentInputs(string uri1, PackageVulnerabilitySeverity severity1, string version1,
                                                       string uri2, PackageVulnerabilitySeverity severity2, string version2)
        {
            var packageVulnerabilityInfo1 = new PackageVulnerabilityInfo(new Uri(uri1), severity1, VersionRange.Parse(version1));
            var packageVulnerabilityInfo2 = new PackageVulnerabilityInfo(new Uri(uri2), severity2, VersionRange.Parse(version2));
            packageVulnerabilityInfo1.Should().NotBe(packageVulnerabilityInfo2);

        }

        [Fact]
        public void HashCode_WithEquivalentObjects_ReturnsTrue()
        {
            var packageVulnerabilityInfo1 = new PackageVulnerabilityInfo(new Uri("https://contoso.com/vulnerability1"), PackageVulnerabilitySeverity.Low, VersionRange.Parse("1.0.0"));
            var packageVulnerabilityInfo2 = new PackageVulnerabilityInfo(new Uri("https://contoso.com/vulnerability1"), PackageVulnerabilitySeverity.Low, VersionRange.Parse("1.0.0"));
            packageVulnerabilityInfo1.GetHashCode().Should().Be(packageVulnerabilityInfo2.GetHashCode());
        }

        [Theory]
        [InlineData("https://contoso.com/vulnerability1", PackageVulnerabilitySeverity.Low, "1.0.0",
                    "https://contoso.com/vulnerability2", PackageVulnerabilitySeverity.Low, "1.0.0")]
        [InlineData("https://contoso.com/vulnerability1", PackageVulnerabilitySeverity.Low, "1.0.0",
                    "https://contoso.com/vulnerability1", PackageVulnerabilitySeverity.High, "1.0.0")]
        [InlineData("https://contoso.com/vulnerability1", PackageVulnerabilitySeverity.Low, "1.0.0",
                    "https://contoso.com/vulnerability1", PackageVulnerabilitySeverity.Low, "2.0.0")]

        public void HashCode_WithVariousNotEquivalentInputs(string uri1, PackageVulnerabilitySeverity severity1, string version1,
                                                       string uri2, PackageVulnerabilitySeverity severity2, string version2)
        {
            var packageVulnerabilityInfo1 = new PackageVulnerabilityInfo(new Uri(uri1), severity1, VersionRange.Parse(version1));
            var packageVulnerabilityInfo2 = new PackageVulnerabilityInfo(new Uri(uri2), severity2, VersionRange.Parse(version2));
            packageVulnerabilityInfo1.GetHashCode().Should().NotBe(packageVulnerabilityInfo2.GetHashCode());

        }
    }
}
