﻿// Copyright (c) .NET Foundation. All rights reserved.
// Licensed under the MIT license. See License.txt in the project root for license information.

#nullable disable

using System.Collections.Immutable;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Razor.Language;
using Microsoft.AspNetCore.Razor.Language.Components;
using Microsoft.AspNetCore.Razor.LanguageServer.Tooltip;
using Microsoft.AspNetCore.Razor.ProjectSystem;
using Microsoft.AspNetCore.Razor.Test.Common;
using Microsoft.AspNetCore.Razor.Test.Common.LanguageServer;
using Microsoft.CodeAnalysis.Razor.ProjectSystem;
using Roslyn.Test.Utilities;
using Xunit;
using Xunit.Abstractions;
using static Microsoft.AspNetCore.Razor.Language.CommonMetadata;

namespace Microsoft.AspNetCore.Razor.LanguageServer.Test.Completion;

public class TagHelperTooltipFactoryBaseTest(ITestOutputHelper testOutput) : LanguageServerTestBase(testOutput)
{
    [Fact]
    public void ReduceTypeName_Plain()
    {
        // Arrange
        var content = "Microsoft.AspNetCore.SomeTagHelpers.SomeTypeName";

        // Act
        var reduced = TagHelperTooltipFactoryBase.ReduceTypeName(content);

        // Assert
        Assert.Equal("SomeTypeName", reduced);
    }

    [Fact]
    public void ReduceTypeName_Generics()
    {
        // Arrange
        var content = "System.Collections.Generic.List<System.String>";

        // Act
        var reduced = TagHelperTooltipFactoryBase.ReduceTypeName(content);

        // Assert
        Assert.Equal("List<System.String>", reduced);
    }

    [Fact]
    public void ReduceTypeName_CrefGenerics()
    {
        // Arrange
        var content = "System.Collections.Generic.List{System.String}";

        // Act
        var reduced = TagHelperTooltipFactoryBase.ReduceTypeName(content);

        // Assert
        Assert.Equal("List{System.String}", reduced);
    }

    [Fact]
    public void ReduceTypeName_NestedGenerics()
    {
        // Arrange
        var content = "Microsoft.AspNetCore.SometTagHelpers.SomeType<Foo.Bar<Baz.Phi>>";

        // Act
        var reduced = TagHelperTooltipFactoryBase.ReduceTypeName(content);

        // Assert
        Assert.Equal("SomeType<Foo.Bar<Baz.Phi>>", reduced);
    }

    [Theory]
    [InlineData("Microsoft.AspNetCore.SometTagHelpers.SomeType.Foo.Bar<Baz.Phi>>")]
    [InlineData("Microsoft.AspNetCore.SometTagHelpers.SomeType.Foo.Bar{Baz.Phi}}")]
    public void ReduceTypeName_UnbalancedDocs_NotRecoverable_ReturnsOriginalContent(string content)
    {
        // Arrange

        // Act
        var reduced = TagHelperTooltipFactoryBase.ReduceTypeName(content);

        // Assert
        Assert.Equal(content, reduced);
    }

    [Fact]
    public void ReduceMemberName_Plain()
    {
        // Arrange
        var content = "Microsoft.AspNetCore.SometTagHelpers.SomeType.SomeProperty";

        // Act
        var reduced = TagHelperTooltipFactoryBase.ReduceMemberName(content);

        // Assert
        Assert.Equal("SomeType.SomeProperty", reduced);
    }

    [Fact]
    public void ReduceMemberName_Generics()
    {
        // Arrange
        var content = "Microsoft.AspNetCore.SometTagHelpers.SomeType<Foo.Bar>.SomeProperty<Foo.Bar>";

        // Act
        var reduced = TagHelperTooltipFactoryBase.ReduceMemberName(content);

        // Assert
        Assert.Equal("SomeType<Foo.Bar>.SomeProperty<Foo.Bar>", reduced);
    }

    [Fact]
    public void ReduceMemberName_CrefGenerics()
    {
        // Arrange
        var content = "Microsoft.AspNetCore.SometTagHelpers.SomeType{Foo.Bar}.SomeProperty{Foo.Bar}";

        // Act
        var reduced = TagHelperTooltipFactoryBase.ReduceMemberName(content);

        // Assert
        Assert.Equal("SomeType{Foo.Bar}.SomeProperty{Foo.Bar}", reduced);
    }

    [Fact]
    public void ReduceMemberName_NestedGenericsMethodsTypes()
    {
        // Arrange
        var content = "Microsoft.AspNetCore.SometTagHelpers.SomeType<Foo.Bar<Baz,Fi>>.SomeMethod(Foo.Bar<System.String>,Baz<Something>.Fi)";

        // Act
        var reduced = TagHelperTooltipFactoryBase.ReduceMemberName(content);

        // Assert
        Assert.Equal("SomeType<Foo.Bar<Baz,Fi>>.SomeMethod(Foo.Bar<System.String>,Baz<Something>.Fi)", reduced);
    }

    [Theory]
    [InlineData("Microsoft.AspNetCore.SometTagHelpers.SomeType.Foo.Bar<Baz.Phi>>")]
    [InlineData("Microsoft.AspNetCore.SometTagHelpers.SomeType.Foo.Bar{Baz.Phi}}")]
    [InlineData("Microsoft.AspNetCore.SometTagHelpers.SomeType.Foo.Bar(Baz.Phi))")]
    [InlineData("Microsoft.AspNetCore.SometTagHelpers.SomeType.Foo{.>")]
    public void ReduceMemberName_UnbalancedDocs_NotRecoverable_ReturnsOriginalContent(string content)
    {
        // Arrange

        // Act
        var reduced = TagHelperTooltipFactoryBase.ReduceMemberName(content);

        // Assert
        Assert.Equal(content, reduced);
    }

    [Fact]
    public void ReduceCrefValue_InvalidShortValue_ReturnsEmptyString()
    {
        // Arrange
        var content = "T:";

        // Act
        var value = TagHelperTooltipFactoryBase.ReduceCrefValue(content);

        // Assert
        Assert.Equal(string.Empty, value);
    }

    [Fact]
    public void ReduceCrefValue_InvalidUnknownIdentifierValue_ReturnsEmptyString()
    {
        // Arrange
        var content = "X:";

        // Act
        var value = TagHelperTooltipFactoryBase.ReduceCrefValue(content);

        // Assert
        Assert.Equal(string.Empty, value);
    }

    [Fact]
    public void ReduceCrefValue_Type()
    {
        // Arrange
        var content = "T:Microsoft.AspNetCore.SometTagHelpers.SomeType";

        // Act
        var value = TagHelperTooltipFactoryBase.ReduceCrefValue(content);

        // Assert
        Assert.Equal("SomeType", value);
    }

    [Fact]
    public void ReduceCrefValue_Property()
    {
        // Arrange
        var content = "P:Microsoft.AspNetCore.SometTagHelpers.SomeType.SomeProperty";

        // Act
        var value = TagHelperTooltipFactoryBase.ReduceCrefValue(content);

        // Assert
        Assert.Equal("SomeType.SomeProperty", value);
    }

    [Fact]
    public void ReduceCrefValue_Member()
    {
        // Arrange
        var content = "P:Microsoft.AspNetCore.SometTagHelpers.SomeType.SomeMember";

        // Act
        var value = TagHelperTooltipFactoryBase.ReduceCrefValue(content);

        // Assert
        Assert.Equal("SomeType.SomeMember", value);
    }

    [Fact]
    public void TryExtractSummary_Null_ReturnsFalse()
    {
        // Arrange & Act
        var result = TagHelperTooltipFactoryBase.TryExtractSummary(documentation: null, out var summary);

        // Assert
        Assert.False(result);
        Assert.Null(summary);
    }

    [Fact]
    public void TryExtractSummary_ExtractsSummary_ReturnsTrue()
    {
        // Arrange
        var expectedSummary = " Hello World ";
        var documentation = $@"
Prefixed invalid content


<summary>{expectedSummary}</summary>

Suffixed invalid content";

        // Act
        var result = TagHelperTooltipFactoryBase.TryExtractSummary(documentation, out var summary);

        // Assert
        Assert.True(result);
        Assert.Equal(expectedSummary, summary);
    }

    [Fact]
    public void TryExtractSummary_NoStartSummary_ReturnsFalse()
    {
        // Arrange
        var documentation = @"
Prefixed invalid content


</summary>

Suffixed invalid content";

        // Act
        var result = TagHelperTooltipFactoryBase.TryExtractSummary(documentation, out var summary);

        // Assert
        Assert.True(result);
        Assert.Equal(@"Prefixed invalid content


</summary>

Suffixed invalid content", summary);
    }

    [Fact]
    public void TryExtractSummary_NoEndSummary_ReturnsTrue()
    {
        // Arrange
        var documentation = @"
Prefixed invalid content


<summary>

Suffixed invalid content";

        // Act
        var result = TagHelperTooltipFactoryBase.TryExtractSummary(documentation, out var summary);

        // Assert
        Assert.True(result);
        Assert.Equal(@"Prefixed invalid content


<summary>

Suffixed invalid content", summary);
    }

    [Fact]
    public void TryExtractSummary_XMLButNoSummary_ReturnsFalse()
    {
        // Arrange
        var documentation = @"
<param type=""stuff"">param1</param>
<return>Result</return>
";

        // Act
        var result = TagHelperTooltipFactoryBase.TryExtractSummary(documentation, out var summary);

        // Assert
        Assert.False(result);
        Assert.Null(summary);
    }

    [Fact]
    public void TryExtractSummary_NoXml_ReturnsTrue()
    {
        // Arrange
        var documentation = @"
There is no xml, but I got you this < and the >.
";

        // Act
        var result = TagHelperTooltipFactoryBase.TryExtractSummary(documentation, out var summary);

        // Assert
        Assert.True(result);
        Assert.Equal("There is no xml, but I got you this < and the >.", summary);
    }

    [Fact]
    public async Task GetAvailableProjects_NoProjects_ReturnsNull()
    {
        var projectManager = CreateProjectSnapshotManager();
        var service = new TestTagHelperToolTipFactory(projectManager);

        var availability = await service.GetProjectAvailabilityAsync("file.razor", "MyTagHelper", CancellationToken.None);

        Assert.Null(availability);
    }

    [Fact]
    public async Task GetAvailableProjects_OneProject_ReturnsNull()
    {
        var builder = TagHelperDescriptorBuilder.Create(ComponentMetadata.Component.TagHelperKind, "TestTagHelper", "TestAssembly");
        builder.TagMatchingRule(rule => rule.TagName = "Test");
        var tagHelperTypeName = "TestNamespace.TestTagHelper";
        builder.Metadata(TypeName(tagHelperTypeName));
        var tagHelpers = ImmutableArray.Create(builder.Build());
        var projectWorkspaceState = ProjectWorkspaceState.Create(tagHelpers);

        var hostProject = new HostProject(
            "C:/path/to/project.csproj",
            "C:/path/to/obj/1",
            RazorConfiguration.Default,
            rootNamespace: null,
            displayName: "project");

        var hostDocument = new HostDocument(
            "C:/path/to/file.razor",
            "file.razor",
            FileKinds.Component);

        var projectManager = CreateProjectSnapshotManager();

        await projectManager.UpdateAsync(updater =>
        {
            updater.ProjectAdded(hostProject);
            updater.ProjectWorkspaceStateChanged(hostProject.Key, projectWorkspaceState);
            updater.DocumentAdded(hostProject.Key, hostDocument, TestMocks.CreateTextLoader(hostDocument.FilePath, text: ""));
        });

        var service = new TestTagHelperToolTipFactory(projectManager);

        var availability = await service.GetProjectAvailabilityAsync(hostDocument.FilePath, tagHelperTypeName, CancellationToken.None);

        Assert.Null(availability);
    }

    [Fact]
    public async Task GetAvailableProjects_AvailableInAllProjects_ReturnsNull()
    {
        var builder = TagHelperDescriptorBuilder.Create(ComponentMetadata.Component.TagHelperKind, "TestTagHelper", "TestAssembly");
        builder.TagMatchingRule(rule => rule.TagName = "Test");
        var tagHelperTypeName = "TestNamespace.TestTagHelper";
        builder.Metadata(TypeName(tagHelperTypeName));
        var tagHelpers = ImmutableArray.Create(builder.Build());
        var projectWorkspaceState = ProjectWorkspaceState.Create(tagHelpers);

        var hostProject1 = new HostProject(
            "C:/path/to/project.csproj",
            "C:/path/to/obj/1",
            RazorConfiguration.Default,
            rootNamespace: null,
            displayName: "project1");

        var hostProject2 = new HostProject(
            "C:/path/to/project.csproj",
            "C:/path/to/obj/2",
            RazorConfiguration.Default,
            rootNamespace: null,
            displayName: "project2");

        var hostDocument = new HostDocument(
            "C:/path/to/file.razor",
            "file.razor",
            FileKinds.Component);

        var projectManager = CreateProjectSnapshotManager();

        await projectManager.UpdateAsync(updater =>
        {
            updater.ProjectAdded(hostProject1);
            updater.ProjectWorkspaceStateChanged(hostProject1.Key, projectWorkspaceState);
            updater.DocumentAdded(hostProject1.Key, hostDocument, TestMocks.CreateTextLoader(hostDocument.FilePath, text: ""));

            updater.ProjectAdded(hostProject2);
            updater.ProjectWorkspaceStateChanged(hostProject2.Key, projectWorkspaceState);
            updater.DocumentAdded(hostProject2.Key, hostDocument, TestMocks.CreateTextLoader(hostDocument.FilePath, text: ""));
        });

        var service = new TestTagHelperToolTipFactory(projectManager);

        var availability = await service.GetProjectAvailabilityAsync(hostDocument.FilePath, tagHelperTypeName, CancellationToken.None);

        Assert.Null(availability);
    }

    [Fact]
    public async Task GetAvailableProjects_NotAvailableInAllProjects_ReturnsText()
    {
        var builder = TagHelperDescriptorBuilder.Create(ComponentMetadata.Component.TagHelperKind, "TestTagHelper", "TestAssembly");
        builder.TagMatchingRule(rule => rule.TagName = "Test");
        var tagHelperTypeName = "TestNamespace.TestTagHelper";
        builder.Metadata(TypeName(tagHelperTypeName));
        var tagHelpers = ImmutableArray.Create(builder.Build());
        var projectWorkspaceState = ProjectWorkspaceState.Create(tagHelpers);

        var hostProject1 = new HostProject(
            "C:/path/to/project.csproj",
            "C:/path/to/obj/1",
            RazorConfiguration.Default,
            rootNamespace: null,
            displayName: "project1");

        var hostProject2 = new HostProject(
            "C:/path/to/project.csproj",
            "C:/path/to/obj/2",
            RazorConfiguration.Default,
            rootNamespace: null,
            displayName: "project2");

        var hostDocument = new HostDocument(
            "C:/path/to/file.razor",
            "file.razor",
            FileKinds.Component);

        var projectManager = CreateProjectSnapshotManager();

        await projectManager.UpdateAsync(updater =>
        {
            updater.ProjectAdded(hostProject1);
            updater.ProjectWorkspaceStateChanged(hostProject1.Key, projectWorkspaceState);
            updater.DocumentAdded(hostProject1.Key, hostDocument, TestMocks.CreateTextLoader(hostDocument.FilePath, text: ""));

            updater.ProjectAdded(hostProject2);
            updater.DocumentAdded(hostProject2.Key, hostDocument, TestMocks.CreateTextLoader(hostDocument.FilePath, text: ""));
        });

        var service = new TestTagHelperToolTipFactory(projectManager);

        var availability = await service.GetProjectAvailabilityAsync(hostDocument.FilePath, tagHelperTypeName, CancellationToken.None);

        AssertEx.EqualOrDiff("""

            ⚠️ Not available in:
                project2
            """, availability);
    }

    [Fact]
    public async Task GetAvailableProjects_NotAvailableInAnyProject_ReturnsText()
    {
        var hostProject1 = new HostProject(
            "C:/path/to/project.csproj",
            "C:/path/to/obj/1",
            RazorConfiguration.Default,
            rootNamespace: null,
            displayName: "project1");

        var hostProject2 = new HostProject(
            "C:/path/to/project.csproj",
            "C:/path/to/obj/2",
            RazorConfiguration.Default,
            rootNamespace: null,
            displayName: "project2");

        var hostDocument = new HostDocument(
            "C:/path/to/file.razor",
            "file.razor",
            FileKinds.Component);

        var projectManager = CreateProjectSnapshotManager();

        await projectManager.UpdateAsync(updater =>
        {
            updater.ProjectAdded(hostProject1);
            updater.DocumentAdded(hostProject1.Key, hostDocument, TestMocks.CreateTextLoader(hostDocument.FilePath, text: ""));

            updater.ProjectAdded(hostProject2);
            updater.DocumentAdded(hostProject2.Key, hostDocument, TestMocks.CreateTextLoader(hostDocument.FilePath, text: ""));
        });

        var service = new TestTagHelperToolTipFactory(projectManager);

        var availability = await service.GetProjectAvailabilityAsync(hostDocument.FilePath, "MyTagHelper", CancellationToken.None);

        AssertEx.EqualOrDiff("""

            ⚠️ Not available in:
                project1
                project2
            """, availability);
    }
}

file class TestTagHelperToolTipFactory(IProjectSnapshotManager projectManager) : TagHelperTooltipFactoryBase(projectManager)
{
}
